////////////////////////////////////////////////////////////////////////////////
/// @brief Condition Variable
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "ConditionVariable.h"

#include <errno.h>

#include <Basics/Logger.h>

namespace triagens {
  namespace basics {

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    ConditionVariable::ConditionVariable () {
      pthread_mutex_init(&_mutex, 0);
      pthread_cond_init(&_condition, 0);
    }



    ConditionVariable::~ConditionVariable () {
      pthread_cond_destroy(&_condition);
      pthread_mutex_destroy(&_mutex);
    }

    // -----------------------------------------------------------------------------
    // public methods
    // -----------------------------------------------------------------------------

    bool ConditionVariable::lock () {
      int rc = pthread_mutex_lock(&_mutex);

      if (rc != 0) {
        LOGGER_ERROR << "could not lock the mutex: " << strerror(errno);
        return false;
      }

      return true;
    }



    bool ConditionVariable::unlock () {
      int rc = pthread_mutex_unlock(&_mutex);

      if (rc != 0) {
        LOGGER_ERROR << "could not release the mutex: " << strerror(errno);
        return false;
      }

      return true;
    }



    bool ConditionVariable::wait () {
      int rc = pthread_cond_wait(&_condition, &_mutex);

      if (rc != 0) {
        LOGGER_ERROR << "could not wait for condition: " << strerror(errno);
        return false;
      }

      return true;
    }



    bool ConditionVariable::broadcast () {
      int rc = pthread_cond_broadcast(&_condition);

      if (rc != 0) {
        LOGGER_ERROR << "could not send condition broadcast: " << strerror(errno);
        return false;
      }

      return true;
    }
  }
}
