////////////////////////////////////////////////////////////////////////////////
/// @brief application server scheduler implementation
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "ApplicationServerSchedulerImpl.h"

#include "config.h"

#include <errno.h>
#include <sys/time.h>
#include <sys/resource.h>

#include <Basics/Exceptions.h>
#include <Basics/Logger.h>
#include <Rest/PeriodicTask.h>
#include <Rest/SignalTask.h>

#include "Scheduler/SchedulerImpl.h"

using namespace std;
using namespace triagens::basics;
using namespace triagens::rest;

// -----------------------------------------------------------------------------
// helper classes and methods
// -----------------------------------------------------------------------------

namespace {

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief handles control-c
  ////////////////////////////////////////////////////////////////////////////////

  class ControlCTask : public SignalTask {
    public:
      ControlCTask (ApplicationServer* server)
        : Task("Control-C"), SignalTask(), server(server) {
        addSignal(SIGINT);
        addSignal(SIGTERM);
        addSignal(SIGQUIT);
      }

    public:
      bool handleSignal () {
        LOGGER_INFO << "control-c received, shutting down";

        server->beginShutdown();

        return true;
      }

    private:
      ApplicationServer* server;
  };

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief produces a scheduler status report
  ////////////////////////////////////////////////////////////////////////////////

  class SchedulerReporterTask : public PeriodicTask {
    public:
      SchedulerReporterTask (Scheduler* scheduler, double reportIntervall)
        : Task("Scheduler-Reporter"), PeriodicTask(reportIntervall * 0.1, reportIntervall), scheduler(scheduler) {
      }

    public:
      bool handlePeriod () {
        scheduler->reportStatus();
        return true;
      }

    public:
      Scheduler* scheduler;
  };
}

namespace triagens {
  namespace rest {

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    ApplicationServerSchedulerImpl::ApplicationServerSchedulerImpl (string const& description, string const& version)
      : ApplicationServerImpl(description, version),
        reportIntervall(60.0),
        multiSchedulerAllowed(false),
        nrSchedulerThreads(1),
        backend(0),
        reuseAddress(false),
        descriptorMinimum(0),

        schedulerValue(0),
        shutdownInProgress(false) {
    }



    ApplicationServerSchedulerImpl::~ApplicationServerSchedulerImpl () {

      // cleanup tasks and scheduler
      if (schedulerValue != 0) {
        for (vector<Task*>::iterator i = tasks.begin();  i != tasks.end();  ++i) {
          schedulerValue->destroyTask(*i);
        }

        delete schedulerValue;
      }
    }

    // -----------------------------------------------------------------------------
    // public methods
    // -----------------------------------------------------------------------------

    void ApplicationServerSchedulerImpl::buildScheduler () {
      if (schedulerValue != 0) {
        LOGGER_FATAL << "a scheduler has already been created";
        exit(EXIT_FAILURE);
      }

      schedulerValue = new SchedulerImpl(nrSchedulerThreads, backend);
      bool ok = schedulerValue->start(&schedulerCond);

      if (! ok) {
        LOGGER_FATAL << "the scheduler cannot be started";
        exit(EXIT_FAILURE);
      }
    }



    void ApplicationServerSchedulerImpl::buildSchedulerReporter () {
      if (0.0 < reportIntervall) {
        registerTask(new SchedulerReporterTask(schedulerValue, reportIntervall));
      }
    }



    void ApplicationServerSchedulerImpl::buildControlCHandler () {
      if (schedulerValue == 0) {
        LOGGER_FATAL << "no scheduler is known, cannot create control c handler";
        exit(1);
      }

      registerTask(new ControlCTask(this));
    }



    void ApplicationServerSchedulerImpl::start () {
      ApplicationServerImpl::start();
    }



    void ApplicationServerSchedulerImpl::wait () {
      ApplicationServerImpl::wait();

      if (schedulerValue != 0) {
        bool ok = schedulerCond.lock();

        if (! ok) {
          THROW_INTERNAL_ERROR("cannot lock scheduler condition");
        }

        while (schedulerValue->isRunning()) {
          LOGGER_TRACE << "waiting for scheduler to stop";

          ok = schedulerCond.wait();

          if (! ok) {
            THROW_INTERNAL_ERROR("cannot wait on scheduler condition");
          }
        }

        LOGGER_TRACE << "scheduler has stopped";

        ok = schedulerCond.unlock();

        if (! ok) {
          THROW_INTERNAL_ERROR("cannot unlock scheduler condition");
        }
      }
    }



    void ApplicationServerSchedulerImpl::beginShutdown () {
      ApplicationServerImpl::beginShutdown();

      if (! shutdownInProgress) {
        LOGGER_TRACE << "begin shutdown sequence of application server";

        if (schedulerValue != 0) {
          schedulerValue->beginShutdown();
        }

        shutdownInProgress = true;
      }
      else {
        LOGGER_TRACE << "shutdown sequence of application server already initiated";
      }
    }



    void ApplicationServerSchedulerImpl::shutdown () {
      ApplicationServerImpl::shutdown();

      if (schedulerValue != 0) {
        int count = 0;

        while (++count < 6 && schedulerValue->isRunning()) {
          LOGGER_TRACE << "waiting for scheduler to stop";
          sleep(1);
        }

        delete schedulerValue;
        schedulerValue = 0;
      }
    }

    // -----------------------------------------------------------------------------
    // protected methods
    // -----------------------------------------------------------------------------

    void ApplicationServerSchedulerImpl::registerTask (Task* task) {
      if (schedulerValue == 0) {
        LOGGER_FATAL << "no scheduler is known, cannot create tasks";
        exit(1);
      }

      schedulerValue->registerTask(task);
      tasks.push_back(task);
    }



    void ApplicationServerSchedulerImpl::setupOptions (section_e section, ProgramOptionsDescription& description) {
      ApplicationServerImpl::setupOptions(section, description);

      switch (section) {

        // -----------------------------------------------------------------------------
        // command line options
        // -----------------------------------------------------------------------------

        case OPTIONS_CMDLINE:
          description
            ("show-io-backends", "show available io backends")
          ;

          break;

        // -----------------------------------------------------------------------------
        // application server options
        // -----------------------------------------------------------------------------

        case OPTIONS_SERVER:
          description
            ("scheduler.backend", &backend, "1: select, 2: poll, 4: epoll")
            ("server.reuse-address", "try to reuse address")
            ("server.report", &reportIntervall, "report intervall")
#ifdef HAVE_GETRLIMIT
            ("server.descriptors-minimum", &descriptorMinimum, "minimum number of file descriptors needed to start")
#endif
          ;

          if (multiSchedulerAllowed) {
            description
              ("scheduler.threads", &nrSchedulerThreads, "number of scheduler threads")
            ;
          }

          break;

        default:
          break;
      }
    }



    bool ApplicationServerSchedulerImpl::parsePhase1 () {
      bool ok = ApplicationServerImpl::parsePhase1();

      if (! ok) {
        return false;
      }

      // show io backends
      if (options.has("show-io-backends")) {
        cout << "available io backends are: " << SchedulerImpl::availableBackends() << endl;
        exit(EXIT_SUCCESS);
      }

      return true;
    }



    bool ApplicationServerSchedulerImpl::parsePhase2 () {
      bool ok = ApplicationServerImpl::parsePhase2();

      if (! ok) {
        return false;
      }

      // check if want to reuse the address
      if (options.has("server.reuse-address")) {
        reuseAddress = true;
      }

      // adjust file descriptors
      adjustFileDescriptors();

      return true;
    }

    // -----------------------------------------------------------------------------
    // private methods
    // -----------------------------------------------------------------------------

    void ApplicationServerSchedulerImpl::adjustFileDescriptors () {
#ifdef HAVE_GETRLIMIT

      if (0 < descriptorMinimum) {
        struct rlimit rlim;
        int res = getrlimit(RLIMIT_NOFILE, &rlim);

        if (res != 0) {
          LOGGER_FATAL << "cannot get the file descriptor limit: " << strerror(errno) << "'";
          exit(1);
        }

        LOGGER_DEBUG << "hard limit is " << rlim.rlim_max << ", soft limit is " << rlim.rlim_cur;

        bool changed = false;

        if (rlim.rlim_max < descriptorMinimum) {
          LOGGER_DEBUG << "hard limit " << rlim.rlim_cur << " is too small, trying to raise";

          rlim.rlim_max = descriptorMinimum;
          rlim.rlim_cur = descriptorMinimum;

          res = setrlimit(RLIMIT_NOFILE, &rlim);

          if (res < 0) {
            LOGGER_FATAL << "cannot raise the file descriptor limit to '" << descriptorMinimum << "', got " << strerror(errno);
            exit(1);
          }

          changed = true;
        }
        else if (rlim.rlim_cur < descriptorMinimum) {
          LOGGER_DEBUG << "soft limit " << rlim.rlim_cur << " is too small, trying to raise";

          rlim.rlim_cur = descriptorMinimum;

          res = setrlimit(RLIMIT_NOFILE, &rlim);

          if (res < 0) {
            LOGGER_FATAL << "cannot raise the file descriptor limit to '" << descriptorMinimum << "', got " << strerror(errno);
            exit(1);
          }

          changed = true;
        }

        if (changed) {
          res = getrlimit(RLIMIT_NOFILE, &rlim);

          if (res != 0) {
            LOGGER_FATAL << "cannot get the file descriptor limit: " << strerror(errno) << "'";
            exit(1);
          }

          LOGGER_DEBUG << "new hard limit is " << rlim.rlim_max << ", new soft limit is " << rlim.rlim_cur;
        }

        // the select backend has more restrictions
        if (backend == 1) {
          if (FD_SETSIZE < descriptorMinimum) {
            LOGGER_FATAL << "i/o backend 'select' has been selected, which supports only " << FD_SETSIZE
                         << " descriptors, but " << descriptorMinimum << " are required";
            exit(1);
          }
        }
      }

#endif
    }
  }
}
