////////////////////////////////////////////////////////////////////////////////
/// @brief application server line server implementation
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "ApplicationLineServerImpl.h"

#include <Basics/ProgramOptions.h>
#include <Basics/ProgramOptionsDescription.h>
#include <Basics/DeleteObject.h>
#include <Rest/LineRequest.h>


#include "LineServer/LineCommTask.h"
#include "LineServer/LineServer.h"

using namespace std;
using namespace triagens::basics;
using namespace triagens::rest;

namespace triagens {
  namespace rest {

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    ApplicationLineServerImpl::ApplicationLineServerImpl (ApplicationServer* applicationServer)
      : applicationServer(applicationServer),
        showPort(true) {
    }



    ApplicationLineServerImpl::~ApplicationLineServerImpl () {
      for_each(lineServers.begin(), lineServers.end(), DeleteObject());
    }

    // -----------------------------------------------------------------------------
    // public methods
    // -----------------------------------------------------------------------------

    void ApplicationLineServerImpl::setupOptions (ApplicationServer::section_e section, ProgramOptionsDescription& description) {
      switch (section) {
        case ApplicationServer::OPTIONS_SERVER:
          if (showPort) {
            description
              ("server.line-port", &linePorts, "listen port or address:port")
            ;
          }

          break;

        default:
          break;
      }
    }



    bool ApplicationLineServerImpl::parsePhase2 (ProgramOptions&) {
      for (vector<string>::const_iterator i = linePorts.begin();  i != linePorts.end();  ++i) {
        addPort(*i);
      }

      return true;
    }



    AddressPort ApplicationLineServerImpl::addPort (string const& name) {
      AddressPort ap;

      if (! ap.split(name)) {
        LOGGER_ERROR << "unknown server:port definition '" << name << "'";
      }
      else {
        lineAddressPorts.push_back(ap);
      }

      return ap;
    }



    LineServer* ApplicationLineServerImpl::buildServer (LineHandlerFactory* lineHandlerFactory) {
      return buildServer(lineHandlerFactory, lineAddressPorts);
    }



    LineServer* ApplicationLineServerImpl::buildServer (LineHandlerFactory* lineHandlerFactory, vector<AddressPort> const& ports) {
      if (ports.empty()) {
        delete lineHandlerFactory;
        return 0;
      }

      Scheduler* scheduler = applicationServer->scheduler();

      if (scheduler == 0) {
        LOGGER_FATAL << "no scheduler is known, cannot create http server";
        exit(EXIT_FAILURE);
      }

      // create new server

      LineServer* lineServer = new LineServer(scheduler);


      lineServer->setHandlerFactory(lineHandlerFactory);

      // keep a list of active server
      lineServers.push_back(lineServer);

      // open http ports
      deque<AddressPort> addresses;
      addresses.insert(addresses.end(), ports.begin(), ports.end());

      // open ports
      while (! addresses.empty()) {
        AddressPort ap = addresses[0];
        addresses.pop_front();

        string bindAddress = ap.address;
        int port = ap.port;

        bool result;

        if (bindAddress.empty()) {
          LOGGER_TRACE << "trying to open port " << port << " for line requests";

          result = lineServer->addPort(port, applicationServer->addressReuseAllowed());
        }
        else {
          LOGGER_TRACE << "trying to open address " << bindAddress
                       << " on port " << port
                       << " for line requests";

          result = lineServer->addPort(bindAddress, port, applicationServer->addressReuseAllowed());
        }

        if (result) {
          LOGGER_DEBUG << "opened port " << port << " for " << (bindAddress.empty() ? "any" : bindAddress);
        }
        else {
          LOGGER_TRACE << "failed to open port " << port << " for " << (bindAddress.empty() ? "any" : bindAddress);
          addresses.push_back(ap);

          if (scheduler->isShutdownInProgress()) {
            addresses.clear();
          }
          else {
            sleep(1);
          }
        }
      }

      return lineServer;
    }
  }
}
