////////////////////////////////////////////////////////////////////////////////
/// @brief admin log request handler
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Achim Brandt
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "RestAdminLogHandler.h"

#include <Basics/Logger.h>
#include <Basics/StringUtils.h>
#include <Basics/VariantArray.h>
#include <Basics/VariantString.h>
#include <Basics/VariantUInt32.h>
#include <Basics/VariantUInt64.h>
#include <Basics/VariantVector.h>
#include <Rest/HttpRequest.h>

using namespace triagens::basics;
using namespace triagens::rest;

namespace {
  bool LidCompareAsc (Logger::buffer_t const& left, Logger::buffer_t const& right) {
    return left.lid < right.lid;
  }

  bool LidCompareDesc (Logger::buffer_t const& left, Logger::buffer_t const& right) {
    return left.lid > right.lid;
  }
}

namespace triagens {
  namespace admin {

    // -----------------------------------------------------------------------------
    // Handler methods
    // -----------------------------------------------------------------------------

    HttpHandler::status_e RestAdminLogHandler::execute () {

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // check the maximal log level to report
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      bool found;
      string upto = StringUtils::tolower(request->value("upto", found));
      LoggerData::level_e ul;

      if (found) {
        if (upto == "fatal" || upto == "0") {
          ul = LoggerData::LEVEL_FATAL;
        }
        else if (upto == "error" || upto == "1") {
          ul = LoggerData::LEVEL_ERROR;
        }
        else if (upto == "warning" || upto == "2") {
          ul = LoggerData::LEVEL_WARNING;
        }
        else if (upto == "info" || upto == "3") {
          ul = LoggerData::LEVEL_INFO;
        }
        else if (upto == "debug" || upto == "4") {
          ul = LoggerData::LEVEL_DEBUG;
        }
        else if (upto == "trace" || upto == "5") {
          ul = LoggerData::LEVEL_TRACE;
        }
        else {
          generateError(ERROR_ILLEGAL_PARAMETER, true, "unknown 'upto' log level: '" + upto + "'");
          return HANDLER_DONE;
        }
      }
      else {
        ul = LoggerData::LEVEL_INFO;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // check the starting position
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      uint64_t start = 0;

      string s = request->value("start", found);

      if (found) {
        start = StringUtils::uint64(s);
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // check the offset
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      uint64_t offset = 0;

      s = request->value("offset", found);

      if (found) {
        offset = StringUtils::uint64(s);
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // check the size
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      uint64_t size = (uint64_t) -1;

      s = request->value("size", found);

      if (found) {
        size = StringUtils::uint64(s);
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // check the sort direction
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      bool sortAscending = false;
      bool sortDescending = false;

      string sortdir = StringUtils::tolower(request->value("sort", found));

      if (found) {
        if (sortdir == "asc") {
          sortAscending = true;
        }
        else if (sortdir == "desc") {
          sortDescending = true;
        }
        else {
          LOGGER_DEBUG << "unknown sort direction '" << sortdir << "'";
        }
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // check the search criteria
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      bool search = false;
      string searchString = StringUtils::tolower(request->value("search", search));

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // generate result
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      VariantArray* result = new VariantArray();

      VariantVector* lid = new VariantVector();
      result->add("lid", lid);

      VariantVector* level = new VariantVector();
      result->add("level", level);

      VariantVector* timestamp = new VariantVector();
      result->add("timestamp", timestamp);

      VariantVector* text = new VariantVector();
      result->add("text", text);

      vector<Logger::buffer_t> logs = Logger::buffer(ul, start);
      vector<Logger::buffer_t> clean;

      for (vector<Logger::buffer_t>::iterator i = logs.begin();  i != logs.end();  ++i) {
        if (i->text.empty()) {
          continue;
        }

        if (search && i->text.find(searchString) == string::npos) {
          continue;
        }

        clean.push_back(*i);
      }

      result->add("totalAmount", new VariantUInt64(clean.size()));

      if (offset >= clean.size()) {
        clean.clear();
      }
      else if (offset > 0) {
        clean.erase(clean.begin(), clean.begin() + offset);
      }

      if (clean.size() > size) {
        clean.resize(size);
      }

      if (sortAscending) {
        sort(clean.begin(), clean.end(), LidCompareAsc);
      }
      else if (sortDescending) {
        sort(clean.begin(), clean.end(), LidCompareDesc);
      }

      for (vector<Logger::buffer_t>::iterator i = clean.begin();  i != clean.end();  ++i) {
        uint32_t l = 0;

        switch (i->level) {
          case LoggerData::LEVEL_FATAL:  l = 0; break;
          case LoggerData::LEVEL_ERROR:  l = 1; break;
          case LoggerData::LEVEL_WARNING:  l = 2; break;
          case LoggerData::LEVEL_INFO:  l = 3; break;
          case LoggerData::LEVEL_DEBUG:  l = 4; break;
          case LoggerData::LEVEL_TRACE:  l = 5; break;
        }

        lid->add(new VariantUInt64(i->lid));
        level->add(new VariantUInt32(l));
        timestamp->add(new VariantUInt32(i->timestamp));
        text->add(new VariantString(i->text));
      }

      generateResult(result);
      return HANDLER_DONE;
    }
  }
}
