#!/bin/sh
set -e

# Remove everything in directories which are only used for output.
# In most cases, we can remove the directories, too.
#
# (We don't remove all the directories themselves for a stupid technical
# reason: "gmake clean" in the src/runtime directory gets unhappy if the
# output/ directory doesn't exist, because it tries to build Depends
# before it cleans itself, and src/c-runtime/sbcl.h is a symlink into
# the output/ directory, and it gets the gcc dependency processing gets
# all confused trying to figure out a header file which is a symlink
# into a directory which doesn't exist. We'd like to be able to run
# this script (including "gmake clean" in the src/runtime directory)
# several times in a row without failure.. so we leave the output/
# directory in place.)
rm -rf obj/* output/* src/runtime/genesis/ src/runtime/sbcl.mk

# Ensure that we know GNUMAKE.
. ./find-gnumake.sh
find_gnumake

# Ask some other directories to clean themselves up.
original_pwd=`pwd`
for d in tools-for-build; do
    cd ./$d > /dev/null
    # I hope the -s option is standard. At least GNU make and BSD make
    # support it. It silences make, since otherwise the output from
    # this script is just the operations done by these make's, which
    # is misleading when this script does lotso other operations too.
    # -- WHN
    $GNUMAKE -I ../src/runtime -s clean
    cd "$original_pwd" > /dev/null
done
( cd ./doc && sh ./clean.sh )

# Within all directories, remove things which don't look like source
# files. Some explanations:
#   (symlinks)
#     are never in the sources, so must've been created
#   sbcl
#     the runtime environment, created by compiling C code
#   sbcl.h
#     information about Lisp code needed to build the runtime environment,
#     created by running GENESIS
#   Config, target
#     architecture-dependent or OS-dependent symlinks
#   core
#     probably a Unix core dump -- not part of the sources anyway
#   *.o, *.so, *.lib, *.nm, a.out
#     results of C-style linking, assembling, etc.
#   *.core, *.map
#     looks like SBCL SAVE-LISP-AND-DIE or GENESIS output, and
#     certainly not source
#   *~, #*#
#     common names for editor temporary files
#   TAGS, tags
#     files created by GNU etags and ctags
#   .#*, *.orig, .*.orig, *.rej
#     rubbish left behind by CVS updates
#   *.htm, *.html
#     The system doc sources are mostly texinfo, plus various odds
#     and ends like docstrings embedded in .lisp sources; any HTML is
#     automatically-generated output.
#   depend, *.d
#     made by "make depend" (or "gmake depend" or some such thing)
#   *.lisp-obj, *.fasl, *.x86f, *.axpf, *.lbytef, *.lib
#     typical extensions for fasl files (not just from SBCL, but
#     from other Lisp systems which might be used as xc hosts)
#   *.tmp, *.lisp-temp
#     conventional names for temporary files autogenerated in
#     building or testing
#   test-passed
#     generated by automatic directory-test-thyself procedure
find . \( \
        -name _darcs -o \
        -name '{arch}' -o \
        -name CVS -o \
        -name .hg -o \
        -name .git -o \
        -name .svn \) -type d -prune -o \
       \( \
        -type l -o \
        -name '*~' -o \
        -name '#*#' -o \
        -name '.#*' -o \
        -name '*.orig' -o \
        -name '.*.orig' -o \
        -name '*.rej' -o \
        -name '?*.x86f' -o \
        -name '?*.axpf' -o \
        -name '?*.lbytef' -o \
        -name '?*.fasl' -o \
        -name '?*.FASL' -o \
        -name 'core' -o \
        -name '?*.core' -o \
        -name '*.map' -o \
        -name '*.host-obj' -o \
        -name '*.lisp-obj' -o \
        -name '*.target-obj' -o \
        -name '*.lib' -o \
        -name '*.tmp' -o \
        -name '*.lisp-temp' -o \
        -name '*.o' -o \
        -name '*.so' -o \
        -name '*.d' -o \
        -name 'foo.c' -o \
        -name 'test-output' -o \
        -name 'test-lab' -o \
        -name 'encodings.texi-temp' -o \
        -name 'stack-alignment-offset' -o \
        -name 'test-status.lisp-expr' -o \
        -name 'last-random-state.lisp-expr' -o \
        -name 'test.log' -o \
        -name 'whitespace-stamp' -o \
        -name 'a.out' -o \
        -name 'sbcl' -o \
        -name 'sbcl.h' -o \
        -name 'ppc-linux-mcontext.h' -o \
        -name 'depend' -o \
        -name 'TAGS' -o \
        -name 'tags' -o \
        -name 'test-passed' -o \
        -name 'local-target-features.lisp-expr' \) -print | \
    if test -f .cleanignore; then
        # Because this file deletes all symlinks, it prevents building
        # in a tree of symlinks. Here's a low-tech workaround: have
        # whatever tool creates your tree of symlinks enumerate
        # relative paths to each one in a file called .cleanignore,
        # and this script won't delete them. This .cleanignore file
        # doesn't support any wildcards or comments.
        awk 'BEGIN{while(getline <".cleanignore"!=0){ign["./" $0]=1}} ign[$0]!=1';
    else
        cat;
    fi | xargs rm -fr
