#! /usr/bin/env python

'''
Python Interface to SAGA Tools Library
----------
- Category : Import/Export
- Name     : PDAL
- ID       : io_pdal

Description
----------
Tools that use the Point Data Abstraction Library (PDAL).

PDAL Version: 2.5.0 (git-version: Release)
'''

from PySAGA.helper import Tool_Wrapper

def Run_Import_Point_Cloud(POINTS=None, FILES=None, VARS=None, VAR_TIME=None, VAR_INTENSITY=None, VAR_SCANANGLE=None, VAR_RETURN=None, VAR_RETURNS=None, VAR_CLASSIFICATION=None, VAR_USERDATA=None, VAR_EDGE=None, VAR_DIRECTION=None, VAR_SOURCEID=None, VAR_COLOR_RED=None, VAR_COLOR_GREEN=None, VAR_COLOR_BLUE=None, VAR_COLOR=None, RGB_RANGE=None, Verbose=2):
    '''
    Import Point Cloud
    ----------
    [io_pdal.0]\n
    The tool allows one to import point cloud data from various file formats using the "Point Data Abstraction Library" (PDAL).\n
    By default, all supported attributes will be imported. Note that the list of attributes supported by the tool is currently based on the attributes defined in the ASPRS LAS specification.\n
    PDAL Version:2.5.0 (git-version: Release)\n
    The following point cloud formats are currently supported:\n
    <table border="1"><tr><th>Name</th><th>Extension</th><th>Description</th></tr><tr><td>readers.copc</td><td>copc</td><td>COPC Reader</td></tr><tr><td>readers.draco</td><td></td><td>Read data from a Draco array.</td></tr><tr><td>readers.e57</td><td></td><td>Reader for E57 files</td></tr><tr><td>readers.fbi</td><td>fbi</td><td>Fbi Reader</td></tr><tr><td>readers.hdf</td><td></td><td>HDF Reader</td></tr><tr><td>readers.i3s</td><td></td><td>I3S Reader</td></tr><tr><td>readers.icebridge</td><td></td><td>NASA HDF5-based IceBridge ATM reader.\n
    See http://nsidc.org/data/docs/daac/icebridge/ilatm1b/index.html\n
    for more information.</td></tr><tr><td>readers.ilvis2</td><td></td><td>ILVIS2 Reader</td></tr><tr><td>readers.las</td><td>las;laz</td><td>ASPRS LAS 1.0 - 1.4 read support. LASzip support is also\n
    enabled through this driver if LASzip was found during\n
    compilation.</td></tr><tr><td>readers.obj</td><td>obj</td><td>Obj Reader</td></tr><tr><td>readers.pcd</td><td>pcd</td><td>Read data in the Point Cloud Library (PCL) format.</td></tr><tr><td>readers.pgpointcloud</td><td></td><td>Read data from pgpointcloud format. "query" option needs to be a\n
    SQL statement selecting the data.</td></tr><tr><td>readers.ply</td><td>ply</td><td>Read ply files.</td></tr><tr><td>readers.pts</td><td>pts</td><td>Pts Reader</td></tr><tr><td>readers.ptx</td><td>ptx</td><td>Ptx Reader</td></tr><tr><td>readers.qfit</td><td>qi</td><td>QFIT Reader</td></tr><tr><td>readers.sbet</td><td>sbet</td><td>SBET Reader</td></tr><tr><td>readers.slpk</td><td></td><td>SLPK Reader</td></tr><tr><td>readers.smrmsg</td><td>smrmsg</td><td>SBET smrmsg Reader</td></tr><tr><td>readers.stac</td><td></td><td>STAC Reader</td></tr><tr><td>readers.terrasolid</td><td>bin</td><td>TerraSolid Reader</td></tr><tr><td>readers.text</td><td>csv;txt</td><td>Text Reader</td></tr><tr><td>readers.tindex</td><td>tindex</td><td>TileIndex Reader</td></tr></table>\n
    Arguments
    ----------
    - POINTS [`output point cloud list`] : Points
    - FILES [`file path`] : Files
    - VARS [`boolean`] : Import All Attributes. Default: 1 Check this to import all supported attributes, or select the attributes you want to become imported individually.
    - VAR_TIME [`boolean`] : GPS-Time. Default: 0
    - VAR_INTENSITY [`boolean`] : Intensity. Default: 0
    - VAR_SCANANGLE [`boolean`] : Scan Angle. Default: 0
    - VAR_RETURN [`boolean`] : Number of the Return. Default: 0
    - VAR_RETURNS [`boolean`] : Number of Returns of Given Pulse. Default: 0
    - VAR_CLASSIFICATION [`boolean`] : Classification. Default: 0
    - VAR_USERDATA [`boolean`] : User Data. Default: 0
    - VAR_EDGE [`boolean`] : Edge of Flight Line Flag. Default: 0
    - VAR_DIRECTION [`boolean`] : Direction of Scan Flag. Default: 0
    - VAR_SOURCEID [`boolean`] : Point Source ID. Default: 0
    - VAR_COLOR_RED [`boolean`] : Red Channel Color. Default: 0
    - VAR_COLOR_GREEN [`boolean`] : Green Channel Color. Default: 0
    - VAR_COLOR_BLUE [`boolean`] : Blue Channel Color. Default: 0
    - VAR_COLOR [`boolean`] : RGB-Coded Color. Default: 0 Import R,G,B values as SAGA RGB-coded value.
    - RGB_RANGE [`choice`] : RGB Value Range. Available Choices: [0] 8 bit [1] 16 bit Default: 1 The color depth of red, green, blue (and NIR) values in the LAS file.

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_pdal', '0', 'Import Point Cloud')
    if Tool.is_Okay():
        Tool.Set_Output('POINTS', POINTS)
        Tool.Set_Option('FILES', FILES)
        Tool.Set_Option('VARS', VARS)
        Tool.Set_Option('VAR_TIME', VAR_TIME)
        Tool.Set_Option('VAR_INTENSITY', VAR_INTENSITY)
        Tool.Set_Option('VAR_SCANANGLE', VAR_SCANANGLE)
        Tool.Set_Option('VAR_RETURN', VAR_RETURN)
        Tool.Set_Option('VAR_RETURNS', VAR_RETURNS)
        Tool.Set_Option('VAR_CLASSIFICATION', VAR_CLASSIFICATION)
        Tool.Set_Option('VAR_USERDATA', VAR_USERDATA)
        Tool.Set_Option('VAR_EDGE', VAR_EDGE)
        Tool.Set_Option('VAR_DIRECTION', VAR_DIRECTION)
        Tool.Set_Option('VAR_SOURCEID', VAR_SOURCEID)
        Tool.Set_Option('VAR_COLOR_RED', VAR_COLOR_RED)
        Tool.Set_Option('VAR_COLOR_GREEN', VAR_COLOR_GREEN)
        Tool.Set_Option('VAR_COLOR_BLUE', VAR_COLOR_BLUE)
        Tool.Set_Option('VAR_COLOR', VAR_COLOR)
        Tool.Set_Option('RGB_RANGE', RGB_RANGE)
        return Tool.Execute(Verbose)
    return False

def Run_Export_LASLAZ_File(POINTS=None, T=None, r=None, n=None, i=None, c=None, SCH=None, R=None, G=None, B=None, C=None, NIR=None, a=None, d=None, e=None, u=None, p=None, FILE=None, FILE_FORMAT=None, FORMAT=None, RGB_RANGE=None, OFF_X=None, OFF_Y=None, OFF_Z=None, SCALE_X=None, SCALE_Y=None, SCALE_Z=None, Verbose=2):
    '''
    Export LAS/LAZ File
    ----------
    [io_pdal.1]\n
    The tool allows one to export a point cloud as ASPRS LAS (or compressed LAZ) file using the "Point Data Abstraction Library" (PDAL).\n
    The file extension of the output file determines whether the file is written compressed (*.laz) or uncompressed (*.las).\n
    The number and type of attributes that can be exported depends on the chosen LAS file version and point data record format. Please have a look at the ASPRS LAS specification on how these formats are defined.\n
    PDAL Version:2.5.0 (git-version: Release)\n
    Arguments
    ----------
    - POINTS [`input point cloud`] : Point Cloud. The point cloud to export.
    - T [`table field`] : gps-time
    - r [`table field`] : number of the return
    - n [`table field`] : number of returns of given pulse
    - i [`table field`] : intensity
    - c [`table field`] : classification
    - SCH [`table field`] : scanner channel
    - R [`table field`] : red channel color
    - G [`table field`] : green channel color
    - B [`table field`] : blue channel color
    - C [`table field`] : SAGA RGB color
    - NIR [`table field`] : near infrared
    - a [`table field`] : scan angle
    - d [`table field`] : direction of scan flag
    - e [`table field`] : edge of flight line flag
    - u [`table field`] : user data
    - p [`table field`] : point source ID
    - FILE [`file path`] : Output File. The LAS/LAZ output file.
    - FILE_FORMAT [`choice`] : File Format. Available Choices: [0] LAS 1.2 [1] LAS 1.4 Default: 1 Choose the file format to write. The format determines which attributes can be written and in which data depth.
    - FORMAT [`choice`] : Point Data Record Format. Available Choices: [0] 0 [1] 1 [2] 2 [3] 3 [4] 6 [5] 7 [6] 8 Default: 3 Choose the point data record format to write. The format determines which attributes can be written.
    - RGB_RANGE [`choice`] : Input R,G,B (and NIR) Value Range. Available Choices: [0] 16 bit [1] 8 bit Default: 1 Color depth of the R,G,B (and NIR) values in the input point cloud. 8 bit values will be scaled to 16 bit.
    - OFF_X [`floating point number`] : Offset X. Default: 0.000000
    - OFF_Y [`floating point number`] : Offset Y. Default: 0.000000
    - OFF_Z [`floating point number`] : Offset Z. Default: 0.000000
    - SCALE_X [`floating point number`] : Scale X. Default: 0.001000
    - SCALE_Y [`floating point number`] : Scale Y. Default: 0.001000
    - SCALE_Z [`floating point number`] : Scale Z. Default: 0.001000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_pdal', '1', 'Export LAS/LAZ File')
    if Tool.is_Okay():
        Tool.Set_Input ('POINTS', POINTS)
        Tool.Set_Option('T', T)
        Tool.Set_Option('r', r)
        Tool.Set_Option('n', n)
        Tool.Set_Option('i', i)
        Tool.Set_Option('c', c)
        Tool.Set_Option('sCH', SCH)
        Tool.Set_Option('R', R)
        Tool.Set_Option('G', G)
        Tool.Set_Option('B', B)
        Tool.Set_Option('C', C)
        Tool.Set_Option('NIR', NIR)
        Tool.Set_Option('a', a)
        Tool.Set_Option('d', d)
        Tool.Set_Option('e', e)
        Tool.Set_Option('u', u)
        Tool.Set_Option('p', p)
        Tool.Set_Option('FILE', FILE)
        Tool.Set_Option('FILE_FORMAT', FILE_FORMAT)
        Tool.Set_Option('FORMAT', FORMAT)
        Tool.Set_Option('RGB_RANGE', RGB_RANGE)
        Tool.Set_Option('OFF_X', OFF_X)
        Tool.Set_Option('OFF_Y', OFF_Y)
        Tool.Set_Option('OFF_Z', OFF_Z)
        Tool.Set_Option('SCALE_X', SCALE_X)
        Tool.Set_Option('SCALE_Y', SCALE_Y)
        Tool.Set_Option('SCALE_Z', SCALE_Z)
        return Tool.Execute(Verbose)
    return False

