#! /usr/bin/env python

'''
Python Interface to SAGA Tools Library
----------
- Category : Import/Export
- Name     : GDAL/OGR
- ID       : io_gdal

Description
----------
Interface to Frank Warmerdam's Geospatial Data Abstraction Library (GDAL).
Version 3.6.2
Homepage: <a target="_blank" href="http://www.gdal.org/">www.gdal.org</a>

'''

from PySAGA.helper import Tool_Wrapper

def Run_Import_Raster(EXTENT_SHAPES=None, GRIDS=None, FILES=None, MULTIPLE=None, SUBSETS=None, SELECTION=None, TRANSFORM=None, RESAMPLING=None, EXTENT=None, EXTENT_XMIN=None, EXTENT_XMAX=None, EXTENT_YMIN=None, EXTENT_YMAX=None, EXTENT_GRID=None, EXTENT_BUFFER=None, Verbose=2):
    '''
    Import Raster
    ----------
    [io_gdal.0]\n
    The "GDAL Raster Import" tool imports grid data from various file formats using the "Geospatial Data Abstraction Library" (GDAL) by Frank Warmerdam.\n
    GDAL Version:3.6.2\n
    Following raster formats are currently supported:\n
    <table border="1"><tr><th>ID</th><th>Name</th><th>Extension</th></tr><tr><td>ECW</td><td>ERDAS Compressed Wavelets (SDK 5.5)</td><td>ecw</td></tr><tr><td>JP2ECW</td><td>ERDAS JPEG2000 (SDK 5.5)</td><td>jp2</td></tr><tr><td>FITS</td><td>Flexible Image Transport System</td><td></td></tr><tr><td>HDF4</td><td>Hierarchical Data Format Release 4</td><td>hdf</td></tr><tr><td>HDF4Image</td><td>HDF4 Dataset</td><td></td></tr><tr><td>HDF5</td><td>Hierarchical Data Format Release 5</td><td></td></tr><tr><td>HDF5Image</td><td>HDF5 Dataset</td><td></td></tr><tr><td>BAG</td><td>Bathymetry Attributed Grid</td><td>bag</td></tr><tr><td>KEA</td><td>KEA Image Format (.kea)</td><td>kea</td></tr><tr><td>netCDF</td><td>Network Common Data Format</td><td>nc</td></tr><tr><td>PDF</td><td>Geospatial PDF</td><td>pdf</td></tr><tr><td>VRT</td><td>Virtual Raster</td><td>vrt</td></tr><tr><td>DERIVED</td><td>Derived datasets using VRT pixel functions</td><td></td></tr><tr><td>GTiff</td><td>GeoTIFF</td><td>tif</td></tr><tr><td>NITF</td><td>National Imagery Transmission Format</td><td>ntf</td></tr><tr><td>RPFTOC</td><td>Raster Product Format TOC format</td><td>toc</td></tr><tr><td>ECRGTOC</td><td>ECRG TOC format</td><td>xml</td></tr><tr><td>HFA</td><td>Erdas Imagine Images (.img)</td><td>img</td></tr><tr><td>SAR_CEOS</td><td>CEOS SAR Image</td><td></td></tr><tr><td>CEOS</td><td>CEOS Image</td><td></td></tr><tr><td>JAXAPALSAR</td><td>JAXA PALSAR Product Reader (Level 1.1/1.5)</td><td></td></tr><tr><td>GFF</td><td>Ground-based SAR Applications Testbed File Format (.gff)</td><td>gff</td></tr><tr><td>ELAS</td><td>ELAS</td><td></td></tr><tr><td>ESRIC</td><td>Esri Compact Cache</td><td></td></tr><tr><td>AIG</td><td>Arc/Info Binary Grid</td><td></td></tr><tr><td>AAIGrid</td><td>Arc/Info ASCII Grid</td><td>asc</td></tr><tr><td>GRASSASCIIGrid</td><td>GRASS ASCII Grid</td><td></td></tr><tr><td>ISG</td><td>International Service for the Geoid</td><td>isg</td></tr><tr><td>SDTS</td><td>SDTS Raster</td><td>ddf</td></tr><tr><td>DTED</td><td>DTED Elevation Raster</td><td></td></tr><tr><td>PNG</td><td>Portable Network Graphics</td><td>png</td></tr><tr><td>JPEG</td><td>JPEG JFIF</td><td>jpg</td></tr><tr><td>MEM</td><td>In Memory Raster</td><td></td></tr><tr><td>JDEM</td><td>Japanese DEM (.mem)</td><td>mem</td></tr><tr><td>GIF</td><td>Graphics Interchange Format (.gif)</td><td>gif</td></tr><tr><td>BIGGIF</td><td>Graphics Interchange Format (.gif)</td><td>gif</td></tr><tr><td>ESAT</td><td>Envisat Image Format</td><td>n1</td></tr><tr><td>BSB</td><td>Maptech BSB Nautical Charts</td><td>kap</td></tr><tr><td>XPM</td><td>X11 PixMap Format</td><td>xpm</td></tr><tr><td>BMP</td><td>MS Windows Device Independent Bitmap</td><td>bmp</td></tr><tr><td>DIMAP</td><td>SPOT DIMAP</td><td></td></tr><tr><td>AirSAR</td><td>AirSAR Polarimetric Image</td><td></td></tr><tr><td>RS2</td><td>RadarSat 2 XML Product</td><td></td></tr><tr><td>SAFE</td><td>Sentinel-1 SAR SAFE Product</td><td></td></tr><tr><td>PCIDSK</td><td>PCIDSK Database File</td><td>pix</td></tr><tr><td>PCRaster</td><td>PCRaster Raster File</td><td>map</td></tr><tr><td>ILWIS</td><td>ILWIS Raster Map</td><td></td></tr><tr><td>SGI</td><td>SGI Image File Format 1.0</td><td>rgb</td></tr><tr><td>SRTMHGT</td><td>SRTMHGT File Format</td><td>hgt</td></tr><tr><td>Leveller</td><td>Leveller heightfield</td><td>ter</td></tr><tr><td>Terragen</td><td>Terragen heightfield</td><td>ter</td></tr><tr><td>ISIS3</td><td>USGS Astrogeology ISIS cube (Version 3)</td><td></td></tr><tr><td>ISIS2</td><td>USGS Astrogeology ISIS cube (Version 2)</td><td></td></tr><tr><td>PDS</td><td>NASA Planetary Data System</td><td></td></tr><tr><td>PDS4</td><td>NASA Planetary Data System 4</td><td>xml</td></tr><tr><td>VICAR</td><td>MIPL VICAR file</td><td></td></tr><tr><td>TIL</td><td>EarthWatch .TIL</td><td></td></tr><tr><td>ERS</td><td>ERMapper .ers Labelled</td><td>ers</td></tr><tr><td>JP2OpenJPEG</td><td>JPEG-2000 driver based on OpenJPEG library</td><td>jp2</td></tr><tr><td>L1B</td><td>NOAA Polar Orbiter Level 1b Data Set</td><td></td></tr><tr><td>FIT</td><td>FIT Image</td><td></td></tr><tr><td>GRIB</td><td>GRIdded Binary (.grb, .grb2)</td><td></td></tr><tr><td>RMF</td><td>Raster Matrix Format</td><td>rsw</td></tr><tr><td>WCS</td><td>OGC Web Coverage Service</td><td></td></tr><tr><td>WMS</td><td>OGC Web Map Service</td><td></td></tr><tr><td>MSGN</td><td>EUMETSAT Archive native (.nat)</td><td>nat</td></tr><tr><td>RST</td><td>Idrisi Raster A.1</td><td>rst</td></tr><tr><td>GSAG</td><td>Golden Software ASCII Grid (.grd)</td><td>grd</td></tr><tr><td>GSBG</td><td>Golden Software Binary Grid (.grd)</td><td>grd</td></tr><tr><td>GS7BG</td><td>Golden Software 7 Binary Grid (.grd)</td><td>grd</td></tr><tr><td>COSAR</td><td>COSAR Annotated Binary Matrix (TerraSAR-X)</td><td></td></tr><tr><td>TSX</td><td>TerraSAR-X Product</td><td></td></tr><tr><td>COASP</td><td>DRDC COASP SAR Processor Raster</td><td>hdr</td></tr><tr><td>R</td><td>R Object Data Store</td><td>rda</td></tr><tr><td>MAP</td><td>OziExplorer .MAP</td><td></td></tr><tr><td>KMLSUPEROVERLAY</td><td>Kml Super Overlay</td><td></td></tr><tr><td>WEBP</td><td>WEBP</td><td>webp</td></tr><tr><td>Rasterlite</td><td>Rasterlite</td><td>sqlite</td></tr><tr><td>MBTiles</td><td>MBTiles</td><td>mbtiles</td></tr><tr><td>PLMOSAIC</td><td>Planet Labs Mosaics API</td><td></td></tr><tr><td>CALS</td><td>CALS (Type 1)</td><td></td></tr><tr><td>WMTS</td><td>OGC Web Map Tile Service</td><td></td></tr><tr><td>SENTINEL2</td><td>Sentinel 2</td><td></td></tr><tr><td>MRF</td><td>Meta Raster Format</td><td>mrf</td></tr><tr><td>PNM</td><td>Portable Pixmap Format (netpbm)</td><td></td></tr><tr><td>DOQ1</td><td>USGS DOQ (Old Style)</td><td></td></tr><tr><td>DOQ2</td><td>USGS DOQ (New Style)</td><td></td></tr><tr><td>PAux</td><td>PCI .aux Labelled</td><td></td></tr><tr><td>MFF</td><td>Vexcel MFF Raster</td><td>hdr</td></tr><tr><td>MFF2</td><td>Vexcel MFF2 (HKV) Raster</td><td></td></tr><tr><td>GSC</td><td>GSC Geogrid</td><td></td></tr><tr><td>FAST</td><td>EOSAT FAST Format</td><td></td></tr><tr><td>BT</td><td>VTP .bt (Binary Terrain) 1.3 Format</td><td>bt</td></tr><tr><td>LAN</td><td>Erdas .LAN/.GIS</td><td></td></tr><tr><td>CPG</td><td>Convair PolGASP</td><td></td></tr><tr><td>NDF</td><td>NLAPS Data Format</td><td></td></tr><tr><td>EIR</td><td>Erdas Imagine Raw</td><td></td></tr><tr><td>DIPEx</td><td>DIPEx</td><td></td></tr><tr><td>LCP</td><td>FARSITE v.4 Landscape File (.lcp)</td><td>lcp</td></tr><tr><td>GTX</td><td>NOAA Vertical Datum .GTX</td><td>gtx</td></tr><tr><td>LOSLAS</td><td>NADCON .los/.las Datum Grid Shift</td><td></td></tr><tr><td>NTv2</td><td>NTv2 Datum Grid Shift</td><td></td></tr><tr><td>CTable2</td><td>CTable2 Datum Grid Shift</td><td></td></tr><tr><td>ACE2</td><td>ACE2</td><td>ACE2</td></tr><tr><td>SNODAS</td><td>Snow Data Assimilation System</td><td>hdr</td></tr><tr><td>KRO</td><td>KOLOR Raw</td><td>kro</td></tr><tr><td>ROI_PAC</td><td>ROI_PAC raster</td><td></td></tr><tr><td>RRASTER</td><td>R Raster</td><td>grd</td></tr><tr><td>BYN</td><td>Natural Resources Canada's Geoid</td><td></td></tr><tr><td>ARG</td><td>Azavea Raster Grid format</td><td></td></tr><tr><td>RIK</td><td>Swedish Grid RIK (.rik)</td><td>rik</td></tr><tr><td>USGSDEM</td><td>USGS Optional ASCII DEM (and CDED)</td><td>dem</td></tr><tr><td>GXF</td><td>GeoSoft Grid Exchange Format</td><td>gxf</td></tr><tr><td>NWT_GRD</td><td>Northwood Numeric Grid Format .grd/.tab</td><td>grd</td></tr><tr><td>NWT_GRC</td><td>Northwood Classified Grid Format .grc/.tab</td><td>grc</td></tr><tr><td>ADRG</td><td>ARC Digitized Raster Graphics</td><td>gen</td></tr><tr><td>SRP</td><td>Standard Raster Product (ASRP/USRP)</td><td>img</td></tr><tr><td>BLX</td><td>Magellan topo (.blx)</td><td>blx</td></tr><tr><td>PostGISRaster</td><td>PostGIS Raster driver</td><td></td></tr><tr><td>SAGA</td><td>SAGA GIS Binary Grid (.sdat, .sg-grd-z)</td><td></td></tr><tr><td>XYZ</td><td>ASCII Gridded XYZ</td><td>xyz</td></tr><tr><td>HF2</td><td>HF2/HFZ heightfield raster</td><td>hf2</td></tr><tr><td>OZI</td><td>OziExplorer Image File</td><td></td></tr><tr><td>CTG</td><td>USGS LULC Composite Theme Grid</td><td></td></tr><tr><td>ZMap</td><td>ZMap Plus Grid</td><td>dat</td></tr><tr><td>NGSGEOID</td><td>NOAA NGS Geoid Height Grids</td><td>bin</td></tr><tr><td>IRIS</td><td>IRIS data (.PPI, .CAPPi etc)</td><td>ppi</td></tr><tr><td>PRF</td><td>Racurs PHOTOMOD PRF</td><td>prf</td></tr><tr><td>EEDAI</td><td>Earth Engine Data API Image</td><td></td></tr><tr><td>DAAS</td><td>Airbus DS Intelligence Data As A Service driver</td><td></td></tr><tr><td>SIGDEM</td><td>Scaled Integer Gridded DEM .sigdem</td><td>sigdem</td></tr><tr><td>TGA</td><td>TGA/TARGA Image File Format</td><td>tga</td></tr><tr><td>OGCAPI</td><td>OGCAPI</td><td></td></tr><tr><td>STACTA</td><td>Spatio-Temporal Asset Catalog Tiled Assets</td><td>json</td></tr><tr><td>STACIT</td><td>Spatio-Temporal Asset Catalog Items</td><td></td></tr><tr><td>GPKG</td><td>GeoPackage</td><td>gpkg</td></tr><tr><td>CAD</td><td>AutoCAD Driver</td><td>dwg</td></tr><tr><td>PLSCENES</td><td>Planet Labs Scenes API</td><td></td></tr><tr><td>NGW</td><td>NextGIS Web</td><td></td></tr><tr><td>GenBin</td><td>Generic Binary (.hdr Labelled)</td><td></td></tr><tr><td>ENVI</td><td>ENVI .hdr Labelled</td><td></td></tr><tr><td>EHdr</td><td>ESRI .hdr Labelled</td><td>bil</td></tr><tr><td>ISCE</td><td>ISCE raster</td><td></td></tr><tr><td>Zarr</td><td>Zarr</td><td></td></tr><tr><td>HTTP</td><td>HTTP Fetching Wrapper</td><td></td></tr></table>\n
    Arguments
    ----------
    - EXTENT_SHAPES [`input shapes`] : Shapes Extent
    - GRIDS [`output grid list`] : Grids
    - FILES [`file path`] : Files
    - MULTIPLE [`choice`] : Multiple Bands Output. Available Choices: [0] single grids [1] grid collection [2] automatic Default: 2
    - SUBSETS [`text`] : Subsets. Semicolon separated list of subset names or indexes (zero-based). If empty (default) all subsets will be imported (if there are any).
    - SELECTION [`text`] : Select from Multiple Bands. Semicolon separated list of band indexes (zero-based). If empty (default) all bands will be imported.
    - TRANSFORM [`boolean`] : Transformation. Default: 1 align grid to coordinate system
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] Nearest Neighbour [1] Bilinear Interpolation [2] Bicubic Spline Interpolation [3] B-Spline Interpolation Default: 0 Resampling type to be used, if grid needs to be aligned to coordinate system.
    - EXTENT [`choice`] : Extent. Available Choices: [0] original [1] user defined [2] grid system [3] shapes extent Default: 0
    - EXTENT_XMIN [`floating point number`] : Left. Default: 0.000000
    - EXTENT_XMAX [`floating point number`] : Right. Default: 0.000000
    - EXTENT_YMIN [`floating point number`] : Bottom. Default: 0.000000
    - EXTENT_YMAX [`floating point number`] : Top. Default: 0.000000
    - EXTENT_GRID [`grid system`] : Grid System
    - EXTENT_BUFFER [`floating point number`] : Buffer. Minimum: 0.000000 Default: 0.000000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '0', 'Import Raster')
    if Tool.is_Okay():
        Tool.Set_Input ('EXTENT_SHAPES', EXTENT_SHAPES)
        Tool.Set_Output('GRIDS', GRIDS)
        Tool.Set_Option('FILES', FILES)
        Tool.Set_Option('MULTIPLE', MULTIPLE)
        Tool.Set_Option('SUBSETS', SUBSETS)
        Tool.Set_Option('SELECTION', SELECTION)
        Tool.Set_Option('TRANSFORM', TRANSFORM)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('EXTENT', EXTENT)
        Tool.Set_Option('EXTENT_XMIN', EXTENT_XMIN)
        Tool.Set_Option('EXTENT_XMAX', EXTENT_XMAX)
        Tool.Set_Option('EXTENT_YMIN', EXTENT_YMIN)
        Tool.Set_Option('EXTENT_YMAX', EXTENT_YMAX)
        Tool.Set_Option('EXTENT_GRID', EXTENT_GRID)
        Tool.Set_Option('EXTENT_BUFFER', EXTENT_BUFFER)
        return Tool.Execute(Verbose)
    return False

def Run_Export_Raster(GRIDS=None, MULTIPLE=None, FOLDER=None, EXTENSION=None, FILE=None, FORMAT=None, TYPE=None, SET_NODATA=None, NODATA=None, OPTIONS=None, Verbose=2):
    '''
    Export Raster
    ----------
    [io_gdal.1]\n
    The "GDAL Raster Export" tool exports one or more grids to various file formats using the "Geospatial Data Abstraction Library" (GDAL) by Frank Warmerdam.\n
    GDAL Version:3.6.2\n
    Following raster formats are currently supported:\n
    <table border="1"><tr><th>ID</th><th>Name</th><th>Extension</th></tr>\n
    <tr><td>ECW</td><td>ERDAS Compressed Wavelets (SDK 5.5)</td><td>ecw</td></tr><tr><td>JP2ECW</td><td>ERDAS JPEG2000 (SDK 5.5)</td><td>jp2</td></tr><tr><td>FITS</td><td>Flexible Image Transport System</td><td></td></tr><tr><td>HDF4Image</td><td>HDF4 Dataset</td><td></td></tr><tr><td>BAG</td><td>Bathymetry Attributed Grid</td><td>bag</td></tr><tr><td>KEA</td><td>KEA Image Format (.kea)</td><td>kea</td></tr><tr><td>netCDF</td><td>Network Common Data Format</td><td>nc</td></tr><tr><td>PDF</td><td>Geospatial PDF</td><td>pdf</td></tr><tr><td>VRT</td><td>Virtual Raster</td><td>vrt</td></tr><tr><td>GTiff</td><td>GeoTIFF</td><td>tif</td></tr><tr><td>NITF</td><td>National Imagery Transmission Format</td><td>ntf</td></tr><tr><td>HFA</td><td>Erdas Imagine Images (.img)</td><td>img</td></tr><tr><td>ELAS</td><td>ELAS</td><td></td></tr><tr><td>MEM</td><td>In Memory Raster</td><td></td></tr><tr><td>BMP</td><td>MS Windows Device Independent Bitmap</td><td>bmp</td></tr><tr><td>PCIDSK</td><td>PCIDSK Database File</td><td>pix</td></tr><tr><td>PCRaster</td><td>PCRaster Raster File</td><td>map</td></tr><tr><td>ILWIS</td><td>ILWIS Raster Map</td><td></td></tr><tr><td>SGI</td><td>SGI Image File Format 1.0</td><td>rgb</td></tr><tr><td>Leveller</td><td>Leveller heightfield</td><td>ter</td></tr><tr><td>Terragen</td><td>Terragen heightfield</td><td>ter</td></tr><tr><td>ISIS3</td><td>USGS Astrogeology ISIS cube (Version 3)</td><td></td></tr><tr><td>ISIS2</td><td>USGS Astrogeology ISIS cube (Version 2)</td><td></td></tr><tr><td>PDS4</td><td>NASA Planetary Data System 4</td><td>xml</td></tr><tr><td>VICAR</td><td>MIPL VICAR file</td><td></td></tr><tr><td>ERS</td><td>ERMapper .ers Labelled</td><td>ers</td></tr><tr><td>RMF</td><td>Raster Matrix Format</td><td>rsw</td></tr><tr><td>RST</td><td>Idrisi Raster A.1</td><td>rst</td></tr><tr><td>GSBG</td><td>Golden Software Binary Grid (.grd)</td><td>grd</td></tr><tr><td>GS7BG</td><td>Golden Software 7 Binary Grid (.grd)</td><td>grd</td></tr><tr><td>MBTiles</td><td>MBTiles</td><td>mbtiles</td></tr><tr><td>MRF</td><td>Meta Raster Format</td><td>mrf</td></tr><tr><td>PNM</td><td>Portable Pixmap Format (netpbm)</td><td></td></tr><tr><td>PAux</td><td>PCI .aux Labelled</td><td></td></tr><tr><td>MFF</td><td>Vexcel MFF Raster</td><td>hdr</td></tr><tr><td>MFF2</td><td>Vexcel MFF2 (HKV) Raster</td><td></td></tr><tr><td>BT</td><td>VTP .bt (Binary Terrain) 1.3 Format</td><td>bt</td></tr><tr><td>LAN</td><td>Erdas .LAN/.GIS</td><td></td></tr><tr><td>GTX</td><td>NOAA Vertical Datum .GTX</td><td>gtx</td></tr><tr><td>NTv2</td><td>NTv2 Datum Grid Shift</td><td></td></tr><tr><td>CTable2</td><td>CTable2 Datum Grid Shift</td><td></td></tr><tr><td>KRO</td><td>KOLOR Raw</td><td>kro</td></tr><tr><td>ROI_PAC</td><td>ROI_PAC raster</td><td></td></tr><tr><td>RRASTER</td><td>R Raster</td><td>grd</td></tr><tr><td>BYN</td><td>Natural Resources Canada's Geoid</td><td></td></tr><tr><td>NWT_GRD</td><td>Northwood Numeric Grid Format .grd/.tab</td><td>grd</td></tr><tr><td>ADRG</td><td>ARC Digitized Raster Graphics</td><td>gen</td></tr><tr><td>SAGA</td><td>SAGA GIS Binary Grid (.sdat, .sg-grd-z)</td><td></td></tr><tr><td>GPKG</td><td>GeoPackage</td><td>gpkg</td></tr><tr><td>NGW</td><td>NextGIS Web</td><td></td></tr><tr><td>ENVI</td><td>ENVI .hdr Labelled</td><td></td></tr><tr><td>EHdr</td><td>ESRI .hdr Labelled</td><td>bil</td></tr><tr><td>ISCE</td><td>ISCE raster</td><td></td></tr><tr><td>Zarr</td><td>Zarr</td><td></td></tr></table>\n
    Arguments
    ----------
    - GRIDS [`input grid list`] : Grid(s)
    - MULTIPLE [`choice`] : Multiple. Available Choices: [0] single files [1] one file Default: 1 If multiple grids are supplied: export each grid to a separate file or all grids to one multi-band file. Notice: Storing multiple bands in one file might not be supported by the selected format. For single files the grid name is used as file name.
    - FOLDER [`file path`] : Folder. The folder location to which single files will be stored.
    - EXTENSION [`text`] : Extension
    - FILE [`file path`] : File. The GDAL dataset to be created.
    - FORMAT [`choice`] : Format. Available Choices: [0] ERDAS Compressed Wavelets (SDK 5.5) [1] ERDAS JPEG2000 (SDK 5.5) [2] Flexible Image Transport System [3] HDF4 Dataset [4] Bathymetry Attributed Grid [5] KEA Image Format (.kea) [6] Network Common Data Format [7] Geospatial PDF [8] Virtual Raster [9] GeoTIFF [10] National Imagery Transmission Format [11] Erdas Imagine Images (.img) [12] ELAS [13] In Memory Raster [14] MS Windows Device Independent Bitmap [15] PCIDSK Database File [16] PCRaster Raster File [17] ILWIS Raster Map [18] SGI Image File Format 1.0 [19] Leveller heightfield [20] Terragen heightfield [21] USGS Astrogeology ISIS cube (Version 3) [22] USGS Astrogeology ISIS cube (Version 2) [23] NASA Planetary Data System 4 [24] MIPL VICAR file [25] ERMapper .ers Labelled [26] Raster Matrix Format [27] Idrisi Raster A.1 [28] Golden Software Binary Grid (.grd) [29] Golden Software 7 Binary Grid (.grd) [30] MBTiles [31] Meta Raster Format [32] Portable Pixmap Format (netpbm) [33] PCI .aux Labelled [34] Vexcel MFF Raster [35] Vexcel MFF2 (HKV) Raster [36] VTP .bt (Binary Terrain) 1.3 Format [37] Erdas .LAN/.GIS [38] NOAA Vertical Datum .GTX [39] NTv2 Datum Grid Shift [40] CTable2 Datum Grid Shift [41] KOLOR Raw [42] ROI_PAC raster [43] R Raster [44] Natural Resources Canada's Geoid [45] Northwood Numeric Grid Format .grd/.tab [46] ARC Digitized Raster Graphics [47] SAGA GIS Binary Grid (.sdat, .sg-grd-z) [48] GeoPackage [49] NextGIS Web [50] ENVI .hdr Labelled [51] ESRI .hdr Labelled [52] ISCE raster [53] Zarr Default: 0 The GDAL raster format (driver) to be used.
    - TYPE [`choice`] : Data Type. Available Choices: [0] match input data [1] 8 bit unsigned integer [2] 16 bit unsigned integer [3] 16 bit signed integer [4] 32 bit unsigned integer [5] 32 bit signed integer [6] 32 bit floating point [7] 64 bit floating point Default: 0 The GDAL datatype of the created dataset.
    - SET_NODATA [`boolean`] : Set Custom NoData. Default: 0
    - NODATA [`floating point number`] : NoData Value. Default: 0.000000
    - OPTIONS [`text`] : Creation Options. A space separated list of key-value pairs (K=V).

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '1', 'Export Raster')
    if Tool.is_Okay():
        Tool.Set_Input ('GRIDS', GRIDS)
        Tool.Set_Option('MULTIPLE', MULTIPLE)
        Tool.Set_Option('FOLDER', FOLDER)
        Tool.Set_Option('EXTENSION', EXTENSION)
        Tool.Set_Option('FILE', FILE)
        Tool.Set_Option('FORMAT', FORMAT)
        Tool.Set_Option('TYPE', TYPE)
        Tool.Set_Option('SET_NODATA', SET_NODATA)
        Tool.Set_Option('NODATA', NODATA)
        Tool.Set_Option('OPTIONS', OPTIONS)
        return Tool.Execute(Verbose)
    return False

def Run_Export_GeoTIFF(GRIDS=None, FILE=None, OPTIONS=None, Verbose=2):
    '''
    Export GeoTIFF
    ----------
    [io_gdal.2]\n
    The "GDAL GeoTIFF Export" tool exports one or more grids to a Geocoded Tagged Image File Format using the "Geospatial Data Abstraction Library" (GDAL) by Frank Warmerdam.\n
    Arguments
    ----------
    - GRIDS [`input grid list`] : Grid(s). The SAGA grids to be exported.
    - FILE [`file path`] : File. The GeoTIFF File to be created.
    - OPTIONS [`text`] : Creation Options. A space separated list of key-value pairs (K=V).

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '2', 'Export GeoTIFF')
    if Tool.is_Okay():
        Tool.Set_Input ('GRIDS', GRIDS)
        Tool.Set_Option('FILE', FILE)
        Tool.Set_Option('OPTIONS', OPTIONS)
        return Tool.Execute(Verbose)
    return False

def Run_Import_Shapes(SHAPES=None, FILES=None, GEOM_TYPE=None, Verbose=2):
    '''
    Import Shapes
    ----------
    [io_gdal.3]\n
    The "OGR Vector Data Import" tool imports vector data from various file/database formats using the "Geospatial Data Abstraction Library" (GDAL) by Frank Warmerdam.\n
    GDAL Version:3.6.2\n
    Following raster formats are currently supported:\n
    <table border="1"><tr><th>ID</th><th>Name</th><th>Extension</th></tr><tr><td>JP2ECW</td><td>ERDAS JPEG2000 (SDK 5.5)</td><td>jp2</td></tr><tr><td>FITS</td><td>Flexible Image Transport System</td><td></td></tr><tr><td>BAG</td><td>Bathymetry Attributed Grid</td><td>bag</td></tr><tr><td>netCDF</td><td>Network Common Data Format</td><td>nc</td></tr><tr><td>PDF</td><td>Geospatial PDF</td><td>pdf</td></tr><tr><td>FileGDB</td><td>ESRI FileGDB</td><td>gdb</td></tr><tr><td>ODBC</td><td></td><td></td></tr><tr><td>PCIDSK</td><td>PCIDSK Database File</td><td>pix</td></tr><tr><td>PDS4</td><td>NASA Planetary Data System 4</td><td>xml</td></tr><tr><td>VICAR</td><td>MIPL VICAR file</td><td></td></tr><tr><td>JP2OpenJPEG</td><td>JPEG-2000 driver based on OpenJPEG library</td><td>jp2</td></tr><tr><td>MBTiles</td><td>MBTiles</td><td>mbtiles</td></tr><tr><td>EEDA</td><td>Earth Engine Data API</td><td></td></tr><tr><td>OGCAPI</td><td>OGCAPI</td><td></td></tr><tr><td>ESRI Shapefile</td><td>ESRI Shapefile</td><td>shp</td></tr><tr><td>MapInfo File</td><td>MapInfo File</td><td></td></tr><tr><td>UK .NTF</td><td>UK .NTF</td><td></td></tr><tr><td>LVBAG</td><td>Kadaster LV BAG Extract 2.0</td><td>xml</td></tr><tr><td>OGR_SDTS</td><td>SDTS</td><td></td></tr><tr><td>S57</td><td>IHO S-57 (ENC)</td><td>000</td></tr><tr><td>DGN</td><td>Microstation DGN</td><td>dgn</td></tr><tr><td>OGR_VRT</td><td>VRT - Virtual Datasource</td><td>vrt</td></tr><tr><td>Memory</td><td>Memory</td><td></td></tr><tr><td>CSV</td><td>Comma Separated Value (.csv)</td><td>csv</td></tr><tr><td>NAS</td><td>NAS - ALKIS</td><td>xml</td></tr><tr><td>GML</td><td>Geography Markup Language (GML)</td><td>gml</td></tr><tr><td>GPX</td><td>GPX</td><td>gpx</td></tr><tr><td>LIBKML</td><td>Keyhole Markup Language (LIBKML)</td><td></td></tr><tr><td>KML</td><td>Keyhole Markup Language (KML)</td><td>kml</td></tr><tr><td>GeoJSON</td><td>GeoJSON</td><td></td></tr><tr><td>GeoJSONSeq</td><td>GeoJSON Sequence</td><td></td></tr><tr><td>ESRIJSON</td><td>ESRIJSON</td><td>json</td></tr><tr><td>TopoJSON</td><td>TopoJSON</td><td></td></tr><tr><td>Interlis 1</td><td>Interlis 1</td><td></td></tr><tr><td>Interlis 2</td><td>Interlis 2</td><td></td></tr><tr><td>OGR_GMT</td><td>GMT ASCII Vectors (.gmt)</td><td>gmt</td></tr><tr><td>GPKG</td><td>GeoPackage</td><td>gpkg</td></tr><tr><td>SQLite</td><td>SQLite / Spatialite</td><td></td></tr><tr><td>WAsP</td><td>WAsP .map format</td><td>map</td></tr><tr><td>PGeo</td><td>ESRI Personal GeoDatabase</td><td>mdb</td></tr><tr><td>OGR_OGDI</td><td>OGDI Vectors (VPF, VMAP, DCW)</td><td></td></tr><tr><td>PostgreSQL</td><td>PostgreSQL/PostGIS</td><td></td></tr><tr><td>MySQL</td><td>MySQL</td><td></td></tr><tr><td>OpenFileGDB</td><td>ESRI FileGDB</td><td>gdb</td></tr><tr><td>DXF</td><td>AutoCAD DXF</td><td>dxf</td></tr><tr><td>CAD</td><td>AutoCAD Driver</td><td>dwg</td></tr><tr><td>FlatGeobuf</td><td>FlatGeobuf</td><td>fgb</td></tr><tr><td>Geoconcept</td><td>Geoconcept</td><td></td></tr><tr><td>GeoRSS</td><td>GeoRSS</td><td></td></tr><tr><td>VFK</td><td>Czech Cadastral Exchange Data Format</td><td>vfk</td></tr><tr><td>PGDUMP</td><td>PostgreSQL SQL dump</td><td>sql</td></tr><tr><td>OSM</td><td>OpenStreetMap XML and PBF</td><td></td></tr><tr><td>GPSBabel</td><td>GPSBabel</td><td></td></tr><tr><td>OGR_PDS</td><td>Planetary Data Systems TABLE</td><td></td></tr><tr><td>WFS</td><td>OGC WFS (Web Feature Service)</td><td></td></tr><tr><td>OAPIF</td><td>OGC API - Features</td><td></td></tr><tr><td>EDIGEO</td><td>French EDIGEO exchange format</td><td>thf</td></tr><tr><td>SVG</td><td>Scalable Vector Graphics</td><td>svg</td></tr><tr><td>Idrisi</td><td>Idrisi Vector (.vct)</td><td>vct</td></tr><tr><td>XLS</td><td>MS Excel format</td><td>xls</td></tr><tr><td>ODS</td><td>Open Document/ LibreOffice / OpenOffice Spreadsheet </td><td>ods</td></tr><tr><td>XLSX</td><td>MS Office Open XML spreadsheet</td><td></td></tr><tr><td>Elasticsearch</td><td>Elastic Search</td><td></td></tr><tr><td>Carto</td><td>Carto</td><td></td></tr><tr><td>AmigoCloud</td><td>AmigoCloud</td><td></td></tr><tr><td>SXF</td><td>Storage and eXchange Format</td><td>sxf</td></tr><tr><td>Selafin</td><td>Selafin</td><td></td></tr><tr><td>JML</td><td>OpenJUMP JML</td><td>jml</td></tr><tr><td>PLSCENES</td><td>Planet Labs Scenes API</td><td></td></tr><tr><td>CSW</td><td>OGC CSW (Catalog  Service for the Web)</td><td></td></tr><tr><td>VDV</td><td>VDV-451/VDV-452/INTREST Data Format</td><td></td></tr><tr><td>GMLAS</td><td>Geography Markup Language (GML) driven by application schemas</td><td></td></tr><tr><td>MVT</td><td>Mapbox Vector Tiles</td><td></td></tr><tr><td>NGW</td><td>NextGIS Web</td><td></td></tr><tr><td>MapML</td><td>MapML</td><td></td></tr><tr><td>TIGER</td><td>U.S. Census TIGER/Line</td><td></td></tr><tr><td>AVCBin</td><td>Arc/Info Binary Coverage</td><td></td></tr><tr><td>AVCE00</td><td>Arc/Info E00 (ASCII) Coverage</td><td>e00</td></tr><tr><td>HTTP</td><td>HTTP Fetching Wrapper</td><td></td></tr></table>\n
    Arguments
    ----------
    - SHAPES [`output shapes list`] : Shapes
    - FILES [`file path`] : Files
    - GEOM_TYPE [`choice`] : Geometry Type. Available Choices: [0] Automatic [1] Point [2] Point (2.5D) [3] Multi-Point [4] Multi-Point (2.5D) [5] Line [6] Line (2.5D) [7] Polyline [8] Polyline (2.5D) [9] Polygon [10] Polygon (2.5D) [11] Multi-Polygon [12] Multi-Polygon (2.5D) Default: 0 Some OGR drivers are unable to determine the geometry type automatically, please choose the appropriate one in this case

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '3', 'Import Shapes')
    if Tool.is_Okay():
        Tool.Set_Output('SHAPES', SHAPES)
        Tool.Set_Option('FILES', FILES)
        Tool.Set_Option('GEOM_TYPE', GEOM_TYPE)
        return Tool.Execute(Verbose)
    return False

def Run_Export_Shapes(SHAPES=None, FILE=None, FORMAT=None, OPTIONS=None, LAYER_OPTIONS=None, Verbose=2):
    '''
    Export Shapes
    ----------
    [io_gdal.4]\n
    The "OGR Vector Data Export" tool exports vector data to various file formats using the "Geospatial Data Abstraction Library" (GDAL) by Frank Warmerdam.\n
    GDAL Version:3.6.2\n
    Following vector formats are currently supported:\n
    <table border="1"><tr><th>ID</th><th>Name</th><th>Extension</th></tr>\n
    <tr><td>JP2ECW</td><td>ERDAS JPEG2000 (SDK 5.5)</td><td>jp2</td></tr><tr><td>FITS</td><td>Flexible Image Transport System</td><td></td></tr><tr><td>BAG</td><td>Bathymetry Attributed Grid</td><td>bag</td></tr><tr><td>netCDF</td><td>Network Common Data Format</td><td>nc</td></tr><tr><td>PDF</td><td>Geospatial PDF</td><td>pdf</td></tr><tr><td>FileGDB</td><td>ESRI FileGDB</td><td>gdb</td></tr><tr><td>PCIDSK</td><td>PCIDSK Database File</td><td>pix</td></tr><tr><td>PDS4</td><td>NASA Planetary Data System 4</td><td>xml</td></tr><tr><td>VICAR</td><td>MIPL VICAR file</td><td></td></tr><tr><td>MBTiles</td><td>MBTiles</td><td>mbtiles</td></tr><tr><td>ESRI Shapefile</td><td>ESRI Shapefile</td><td>shp</td></tr><tr><td>MapInfo File</td><td>MapInfo File</td><td></td></tr><tr><td>S57</td><td>IHO S-57 (ENC)</td><td>000</td></tr><tr><td>DGN</td><td>Microstation DGN</td><td>dgn</td></tr><tr><td>Memory</td><td>Memory</td><td></td></tr><tr><td>CSV</td><td>Comma Separated Value (.csv)</td><td>csv</td></tr><tr><td>GML</td><td>Geography Markup Language (GML)</td><td>gml</td></tr><tr><td>GPX</td><td>GPX</td><td>gpx</td></tr><tr><td>LIBKML</td><td>Keyhole Markup Language (LIBKML)</td><td></td></tr><tr><td>KML</td><td>Keyhole Markup Language (KML)</td><td>kml</td></tr><tr><td>GeoJSON</td><td>GeoJSON</td><td></td></tr><tr><td>GeoJSONSeq</td><td>GeoJSON Sequence</td><td></td></tr><tr><td>Interlis 1</td><td>Interlis 1</td><td></td></tr><tr><td>Interlis 2</td><td>Interlis 2</td><td></td></tr><tr><td>OGR_GMT</td><td>GMT ASCII Vectors (.gmt)</td><td>gmt</td></tr><tr><td>GPKG</td><td>GeoPackage</td><td>gpkg</td></tr><tr><td>SQLite</td><td>SQLite / Spatialite</td><td></td></tr><tr><td>WAsP</td><td>WAsP .map format</td><td>map</td></tr><tr><td>PostgreSQL</td><td>PostgreSQL/PostGIS</td><td></td></tr><tr><td>MySQL</td><td>MySQL</td><td></td></tr><tr><td>OpenFileGDB</td><td>ESRI FileGDB</td><td>gdb</td></tr><tr><td>DXF</td><td>AutoCAD DXF</td><td>dxf</td></tr><tr><td>FlatGeobuf</td><td>FlatGeobuf</td><td>fgb</td></tr><tr><td>Geoconcept</td><td>Geoconcept</td><td></td></tr><tr><td>GeoRSS</td><td>GeoRSS</td><td></td></tr><tr><td>PGDUMP</td><td>PostgreSQL SQL dump</td><td>sql</td></tr><tr><td>GPSBabel</td><td>GPSBabel</td><td></td></tr><tr><td>ODS</td><td>Open Document/ LibreOffice / OpenOffice Spreadsheet </td><td>ods</td></tr><tr><td>XLSX</td><td>MS Office Open XML spreadsheet</td><td></td></tr><tr><td>Elasticsearch</td><td>Elastic Search</td><td></td></tr><tr><td>Carto</td><td>Carto</td><td></td></tr><tr><td>AmigoCloud</td><td>AmigoCloud</td><td></td></tr><tr><td>Selafin</td><td>Selafin</td><td></td></tr><tr><td>JML</td><td>OpenJUMP JML</td><td>jml</td></tr><tr><td>VDV</td><td>VDV-451/VDV-452/INTREST Data Format</td><td></td></tr><tr><td>MVT</td><td>Mapbox Vector Tiles</td><td></td></tr><tr><td>NGW</td><td>NextGIS Web</td><td></td></tr><tr><td>MapML</td><td>MapML</td><td></td></tr></table>\n
    Arguments
    ----------
    - SHAPES [`input shapes`] : Shapes
    - FILE [`file path`] : File
    - FORMAT [`choice`] : Format. Available Choices: [0] ERDAS JPEG2000 (SDK 5.5) [1] Flexible Image Transport System [2] Bathymetry Attributed Grid [3] Network Common Data Format [4] Geospatial PDF [5] ESRI FileGDB [6] PCIDSK Database File [7] NASA Planetary Data System 4 [8] MIPL VICAR file [9] MBTiles [10] ESRI Shapefile [11] MapInfo File [12] IHO S-57 (ENC) [13] Microstation DGN [14] Memory [15] Comma Separated Value (.csv) [16] Geography Markup Language (GML) [17] GPX [18] Keyhole Markup Language (LIBKML) [19] Keyhole Markup Language (KML) [20] GeoJSON [21] GeoJSON Sequence [22] Interlis 1 [23] Interlis 2 [24] GMT ASCII Vectors (.gmt) [25] GeoPackage [26] SQLite / Spatialite [27] WAsP .map format [28] PostgreSQL/PostGIS [29] MySQL [30] ESRI FileGDB [31] AutoCAD DXF [32] FlatGeobuf [33] Geoconcept [34] GeoRSS [35] PostgreSQL SQL dump [36] GPSBabel [37] Open Document/ LibreOffice / OpenOffice Spreadsheet  [38] MS Office Open XML spreadsheet [39] Elastic Search [40] Carto [41] AmigoCloud [42] Selafin [43] OpenJUMP JML [44] VDV-451/VDV-452/INTREST Data Format [45] Mapbox Vector Tiles [46] NextGIS Web [47] MapML Default: 0
    - OPTIONS [`text`] : Creation Options. The dataset creation options. A space separated list of key-value pairs (K=V).
    - LAYER_OPTIONS [`text`] : Layer Creation Options. The layer creation options. A space separated list of key-value pairs (K=V).

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '4', 'Export Shapes')
    if Tool.is_Okay():
        Tool.Set_Input ('SHAPES', SHAPES)
        Tool.Set_Option('FILE', FILE)
        Tool.Set_Option('FORMAT', FORMAT)
        Tool.Set_Option('OPTIONS', OPTIONS)
        Tool.Set_Option('LAYER_OPTIONS', LAYER_OPTIONS)
        return Tool.Execute(Verbose)
    return False

def Run_Export_Shapes_to_KML(SHAPES=None, FILE=None, Verbose=2):
    '''
    Export Shapes to KML
    ----------
    [io_gdal.5]\n
    This tool exports a vector layer to the Google Earth KML format using Frank Warmerdam's "Geospatial Data Abstraction Library" (GDAL/OGR). The output file is projected to geographic coordinates if necessary and possible.\n
    Arguments
    ----------
    - SHAPES [`input shapes`] : Shapes
    - FILE [`file path`] : File

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '5', 'Export Shapes to KML')
    if Tool.is_Okay():
        Tool.Set_Input ('SHAPES', SHAPES)
        Tool.Set_Option('FILE', FILE)
        return Tool.Execute(Verbose)
    return False

def Run_Import_NetCDF(GRIDS=None, FILE=None, SAVE_FILE=None, SAVE_PATH=None, TRANSFORM=None, RESAMPLING=None, Verbose=2):
    '''
    Import NetCDF
    ----------
    [io_gdal.6]\n
    This tool imports grids NetCDF Format using the "Geospatial Data Abstraction Library" (GDAL) by Frank Warmerdam.\n
    Arguments
    ----------
    - GRIDS [`output grid list`] : Grids
    - FILE [`file path`] : File
    - SAVE_FILE [`boolean`] : Save to File. Default: 0 save output to file instead of memory
    - SAVE_PATH [`file path`] : Save to Path
    - TRANSFORM [`boolean`] : Transformation. Default: 1 apply coordinate transformation if appropriate
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] Nearest Neighbour [1] Bilinear Interpolation [2] Bicubic Spline Interpolation [3] B-Spline Interpolation Default: 3 interpolation method to use if grid needs to be aligned to coordinate system

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '6', 'Import NetCDF')
    if Tool.is_Okay():
        Tool.Set_Output('GRIDS', GRIDS)
        Tool.Set_Option('FILE', FILE)
        Tool.Set_Option('SAVE_FILE', SAVE_FILE)
        Tool.Set_Option('SAVE_PATH', SAVE_PATH)
        Tool.Set_Option('TRANSFORM', TRANSFORM)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        return Tool.Execute(Verbose)
    return False

def Run_Create_Raster_Catalogue_from_Files(CATALOGUE=None, FILES=None, Verbose=2):
    '''
    Create Raster Catalogue from Files
    ----------
    [io_gdal.7]\n
    Create a raster catalogue from selected files. Output is a polygon layer that shows the extent for each valid raster file. Each extent is attributed with the original file path and raster system properties.\n
    The "GDAL Raster Import" tool imports grid data from various file formats using the "Geospatial Data Abstraction Library" (GDAL) by Frank Warmerdam.\n
    GDAL Version:3.6.2\n
    Following raster formats are currently supported:\n
    <table border="1"><tr><th>ID</th><th>Name</th></tr>\n
    <tr><td>ECW</td><td>ERDAS Compressed Wavelets (SDK 5.5)</td></tr>\n
    <tr><td>JP2ECW</td><td>ERDAS JPEG2000 (SDK 5.5)</td></tr>\n
    <tr><td>FITS</td><td>Flexible Image Transport System</td></tr>\n
    <tr><td>HDF4</td><td>Hierarchical Data Format Release 4</td></tr>\n
    <tr><td>HDF4Image</td><td>HDF4 Dataset</td></tr>\n
    <tr><td>HDF5</td><td>Hierarchical Data Format Release 5</td></tr>\n
    <tr><td>HDF5Image</td><td>HDF5 Dataset</td></tr>\n
    <tr><td>BAG</td><td>Bathymetry Attributed Grid</td></tr>\n
    <tr><td>KEA</td><td>KEA Image Format (.kea)</td></tr>\n
    <tr><td>netCDF</td><td>Network Common Data Format</td></tr>\n
    <tr><td>PDF</td><td>Geospatial PDF</td></tr>\n
    <tr><td>FileGDB</td><td>ESRI FileGDB</td></tr>\n
    <tr><td>ODBC</td><td></td></tr>\n
    <tr><td>VRT</td><td>Virtual Raster</td></tr>\n
    <tr><td>DERIVED</td><td>Derived datasets using VRT pixel functions</td></tr>\n
    <tr><td>GTiff</td><td>GeoTIFF</td></tr>\n
    <tr><td>COG</td><td>Cloud optimized GeoTIFF generator</td></tr>\n
    <tr><td>NITF</td><td>National Imagery Transmission Format</td></tr>\n
    <tr><td>RPFTOC</td><td>Raster Product Format TOC format</td></tr>\n
    <tr><td>ECRGTOC</td><td>ECRG TOC format</td></tr>\n
    <tr><td>HFA</td><td>Erdas Imagine Images (.img)</td></tr>\n
    <tr><td>SAR_CEOS</td><td>CEOS SAR Image</td></tr>\n
    <tr><td>CEOS</td><td>CEOS Image</td></tr>\n
    <tr><td>JAXAPALSAR</td><td>JAXA PALSAR Product Reader (Level 1.1/1.5)</td></tr>\n
    <tr><td>GFF</td><td>Ground-based SAR Applications Testbed File Format (.gff)</td></tr>\n
    <tr><td>ELAS</td><td>ELAS</td></tr>\n
    <tr><td>ESRIC</td><td>Esri Compact Cache</td></tr>\n
    <tr><td>AIG</td><td>Arc/Info Binary Grid</td></tr>\n
    <tr><td>AAIGrid</td><td>Arc/Info ASCII Grid</td></tr>\n
    <tr><td>GRASSASCIIGrid</td><td>GRASS ASCII Grid</td></tr>\n
    <tr><td>ISG</td><td>International Service for the Geoid</td></tr>\n
    <tr><td>SDTS</td><td>SDTS Raster</td></tr>\n
    <tr><td>DTED</td><td>DTED Elevation Raster</td></tr>\n
    <tr><td>PNG</td><td>Portable Network Graphics</td></tr>\n
    <tr><td>JPEG</td><td>JPEG JFIF</td></tr>\n
    <tr><td>MEM</td><td>In Memory Raster</td></tr>\n
    <tr><td>JDEM</td><td>Japanese DEM (.mem)</td></tr>\n
    <tr><td>GIF</td><td>Graphics Interchange Format (.gif)</td></tr>\n
    <tr><td>BIGGIF</td><td>Graphics Interchange Format (.gif)</td></tr>\n
    <tr><td>ESAT</td><td>Envisat Image Format</td></tr>\n
    <tr><td>BSB</td><td>Maptech BSB Nautical Charts</td></tr>\n
    <tr><td>XPM</td><td>X11 PixMap Format</td></tr>\n
    <tr><td>BMP</td><td>MS Windows Device Independent Bitmap</td></tr>\n
    <tr><td>DIMAP</td><td>SPOT DIMAP</td></tr>\n
    <tr><td>AirSAR</td><td>AirSAR Polarimetric Image</td></tr>\n
    <tr><td>RS2</td><td>RadarSat 2 XML Product</td></tr>\n
    <tr><td>SAFE</td><td>Sentinel-1 SAR SAFE Product</td></tr>\n
    <tr><td>PCIDSK</td><td>PCIDSK Database File</td></tr>\n
    <tr><td>PCRaster</td><td>PCRaster Raster File</td></tr>\n
    <tr><td>ILWIS</td><td>ILWIS Raster Map</td></tr>\n
    <tr><td>SGI</td><td>SGI Image File Format 1.0</td></tr>\n
    <tr><td>SRTMHGT</td><td>SRTMHGT File Format</td></tr>\n
    <tr><td>Leveller</td><td>Leveller heightfield</td></tr>\n
    <tr><td>Terragen</td><td>Terragen heightfield</td></tr>\n
    <tr><td>ISIS3</td><td>USGS Astrogeology ISIS cube (Version 3)</td></tr>\n
    <tr><td>ISIS2</td><td>USGS Astrogeology ISIS cube (Version 2)</td></tr>\n
    <tr><td>PDS</td><td>NASA Planetary Data System</td></tr>\n
    <tr><td>PDS4</td><td>NASA Planetary Data System 4</td></tr>\n
    <tr><td>VICAR</td><td>MIPL VICAR file</td></tr>\n
    <tr><td>TIL</td><td>EarthWatch .TIL</td></tr>\n
    <tr><td>ERS</td><td>ERMapper .ers Labelled</td></tr>\n
    <tr><td>JP2OpenJPEG</td><td>JPEG-2000 driver based on OpenJPEG library</td></tr>\n
    <tr><td>L1B</td><td>NOAA Polar Orbiter Level 1b Data Set</td></tr>\n
    <tr><td>FIT</td><td>FIT Image</td></tr>\n
    <tr><td>GRIB</td><td>GRIdded Binary (.grb, .grb2)</td></tr>\n
    <tr><td>RMF</td><td>Raster Matrix Format</td></tr>\n
    <tr><td>WCS</td><td>OGC Web Coverage Service</td></tr>\n
    <tr><td>WMS</td><td>OGC Web Map Service</td></tr>\n
    <tr><td>MSGN</td><td>EUMETSAT Archive native (.nat)</td></tr>\n
    <tr><td>RST</td><td>Idrisi Raster A.1</td></tr>\n
    <tr><td>GSAG</td><td>Golden Software ASCII Grid (.grd)</td></tr>\n
    <tr><td>GSBG</td><td>Golden Software Binary Grid (.grd)</td></tr>\n
    <tr><td>GS7BG</td><td>Golden Software 7 Binary Grid (.grd)</td></tr>\n
    <tr><td>COSAR</td><td>COSAR Annotated Binary Matrix (TerraSAR-X)</td></tr>\n
    <tr><td>TSX</td><td>TerraSAR-X Product</td></tr>\n
    <tr><td>COASP</td><td>DRDC COASP SAR Processor Raster</td></tr>\n
    <tr><td>R</td><td>R Object Data Store</td></tr>\n
    <tr><td>MAP</td><td>OziExplorer .MAP</td></tr>\n
    <tr><td>KMLSUPEROVERLAY</td><td>Kml Super Overlay</td></tr>\n
    <tr><td>WEBP</td><td>WEBP</td></tr>\n
    <tr><td>Rasterlite</td><td>Rasterlite</td></tr>\n
    <tr><td>MBTiles</td><td>MBTiles</td></tr>\n
    <tr><td>PLMOSAIC</td><td>Planet Labs Mosaics API</td></tr>\n
    <tr><td>CALS</td><td>CALS (Type 1)</td></tr>\n
    <tr><td>WMTS</td><td>OGC Web Map Tile Service</td></tr>\n
    <tr><td>SENTINEL2</td><td>Sentinel 2</td></tr>\n
    <tr><td>MRF</td><td>Meta Raster Format</td></tr>\n
    <tr><td>PNM</td><td>Portable Pixmap Format (netpbm)</td></tr>\n
    <tr><td>DOQ1</td><td>USGS DOQ (Old Style)</td></tr>\n
    <tr><td>DOQ2</td><td>USGS DOQ (New Style)</td></tr>\n
    <tr><td>PAux</td><td>PCI .aux Labelled</td></tr>\n
    <tr><td>MFF</td><td>Vexcel MFF Raster</td></tr>\n
    <tr><td>MFF2</td><td>Vexcel MFF2 (HKV) Raster</td></tr>\n
    <tr><td>GSC</td><td>GSC Geogrid</td></tr>\n
    <tr><td>FAST</td><td>EOSAT FAST Format</td></tr>\n
    <tr><td>BT</td><td>VTP .bt (Binary Terrain) 1.3 Format</td></tr>\n
    <tr><td>LAN</td><td>Erdas .LAN/.GIS</td></tr>\n
    <tr><td>CPG</td><td>Convair PolGASP</td></tr>\n
    <tr><td>NDF</td><td>NLAPS Data Format</td></tr>\n
    <tr><td>EIR</td><td>Erdas Imagine Raw</td></tr>\n
    <tr><td>DIPEx</td><td>DIPEx</td></tr>\n
    <tr><td>LCP</td><td>FARSITE v.4 Landscape File (.lcp)</td></tr>\n
    <tr><td>GTX</td><td>NOAA Vertical Datum .GTX</td></tr>\n
    <tr><td>LOSLAS</td><td>NADCON .los/.las Datum Grid Shift</td></tr>\n
    <tr><td>NTv2</td><td>NTv2 Datum Grid Shift</td></tr>\n
    <tr><td>CTable2</td><td>CTable2 Datum Grid Shift</td></tr>\n
    <tr><td>ACE2</td><td>ACE2</td></tr>\n
    <tr><td>SNODAS</td><td>Snow Data Assimilation System</td></tr>\n
    <tr><td>KRO</td><td>KOLOR Raw</td></tr>\n
    <tr><td>ROI_PAC</td><td>ROI_PAC raster</td></tr>\n
    <tr><td>RRASTER</td><td>R Raster</td></tr>\n
    <tr><td>BYN</td><td>Natural Resources Canada's Geoid</td></tr>\n
    <tr><td>ARG</td><td>Azavea Raster Grid format</td></tr>\n
    <tr><td>RIK</td><td>Swedish Grid RIK (.rik)</td></tr>\n
    <tr><td>USGSDEM</td><td>USGS Optional ASCII DEM (and CDED)</td></tr>\n
    <tr><td>GXF</td><td>GeoSoft Grid Exchange Format</td></tr>\n
    <tr><td>NWT_GRD</td><td>Northwood Numeric Grid Format .grd/.tab</td></tr>\n
    <tr><td>NWT_GRC</td><td>Northwood Classified Grid Format .grc/.tab</td></tr>\n
    <tr><td>ADRG</td><td>ARC Digitized Raster Graphics</td></tr>\n
    <tr><td>SRP</td><td>Standard Raster Product (ASRP/USRP)</td></tr>\n
    <tr><td>BLX</td><td>Magellan topo (.blx)</td></tr>\n
    <tr><td>PostGISRaster</td><td>PostGIS Raster driver</td></tr>\n
    <tr><td>SAGA</td><td>SAGA GIS Binary Grid (.sdat, .sg-grd-z)</td></tr>\n
    <tr><td>XYZ</td><td>ASCII Gridded XYZ</td></tr>\n
    <tr><td>HF2</td><td>HF2/HFZ heightfield raster</td></tr>\n
    <tr><td>OZI</td><td>OziExplorer Image File</td></tr>\n
    <tr><td>CTG</td><td>USGS LULC Composite Theme Grid</td></tr>\n
    <tr><td>ZMap</td><td>ZMap Plus Grid</td></tr>\n
    <tr><td>NGSGEOID</td><td>NOAA NGS Geoid Height Grids</td></tr>\n
    <tr><td>IRIS</td><td>IRIS data (.PPI, .CAPPi etc)</td></tr>\n
    <tr><td>PRF</td><td>Racurs PHOTOMOD PRF</td></tr>\n
    <tr><td>EEDAI</td><td>Earth Engine Data API Image</td></tr>\n
    <tr><td>EEDA</td><td>Earth Engine Data API</td></tr>\n
    <tr><td>DAAS</td><td>Airbus DS Intelligence Data As A Service driver</td></tr>\n
    <tr><td>SIGDEM</td><td>Scaled Integer Gridded DEM .sigdem</td></tr>\n
    <tr><td>TGA</td><td>TGA/TARGA Image File Format</td></tr>\n
    <tr><td>OGCAPI</td><td>OGCAPI</td></tr>\n
    <tr><td>STACTA</td><td>Spatio-Temporal Asset Catalog Tiled Assets</td></tr>\n
    <tr><td>STACIT</td><td>Spatio-Temporal Asset Catalog Items</td></tr>\n
    <tr><td>GNMFile</td><td>Geographic Network generic file based model</td></tr>\n
    <tr><td>GNMDatabase</td><td>Geographic Network generic DB based model</td></tr>\n
    <tr><td>ESRI Shapefile</td><td>ESRI Shapefile</td></tr>\n
    <tr><td>MapInfo File</td><td>MapInfo File</td></tr>\n
    <tr><td>UK .NTF</td><td>UK .NTF</td></tr>\n
    <tr><td>LVBAG</td><td>Kadaster LV BAG Extract 2.0</td></tr>\n
    <tr><td>OGR_SDTS</td><td>SDTS</td></tr>\n
    <tr><td>S57</td><td>IHO S-57 (ENC)</td></tr>\n
    <tr><td>DGN</td><td>Microstation DGN</td></tr>\n
    <tr><td>OGR_VRT</td><td>VRT - Virtual Datasource</td></tr>\n
    <tr><td>Memory</td><td>Memory</td></tr>\n
    <tr><td>CSV</td><td>Comma Separated Value (.csv)</td></tr>\n
    <tr><td>NAS</td><td>NAS - ALKIS</td></tr>\n
    <tr><td>GML</td><td>Geography Markup Language (GML)</td></tr>\n
    <tr><td>GPX</td><td>GPX</td></tr>\n
    <tr><td>LIBKML</td><td>Keyhole Markup Language (LIBKML)</td></tr>\n
    <tr><td>KML</td><td>Keyhole Markup Language (KML)</td></tr>\n
    <tr><td>GeoJSON</td><td>GeoJSON</td></tr>\n
    <tr><td>GeoJSONSeq</td><td>GeoJSON Sequence</td></tr>\n
    <tr><td>ESRIJSON</td><td>ESRIJSON</td></tr>\n
    <tr><td>TopoJSON</td><td>TopoJSON</td></tr>\n
    <tr><td>Interlis 1</td><td>Interlis 1</td></tr>\n
    <tr><td>Interlis 2</td><td>Interlis 2</td></tr>\n
    <tr><td>OGR_GMT</td><td>GMT ASCII Vectors (.gmt)</td></tr>\n
    <tr><td>GPKG</td><td>GeoPackage</td></tr>\n
    <tr><td>SQLite</td><td>SQLite / Spatialite</td></tr>\n
    <tr><td>WAsP</td><td>WAsP .map format</td></tr>\n
    <tr><td>PGeo</td><td>ESRI Personal GeoDatabase</td></tr>\n
    <tr><td>OGR_OGDI</td><td>OGDI Vectors (VPF, VMAP, DCW)</td></tr>\n
    <tr><td>PostgreSQL</td><td>PostgreSQL/PostGIS</td></tr>\n
    <tr><td>MySQL</td><td>MySQL</td></tr>\n
    <tr><td>OpenFileGDB</td><td>ESRI FileGDB</td></tr>\n
    <tr><td>DXF</td><td>AutoCAD DXF</td></tr>\n
    <tr><td>CAD</td><td>AutoCAD Driver</td></tr>\n
    <tr><td>FlatGeobuf</td><td>FlatGeobuf</td></tr>\n
    <tr><td>Geoconcept</td><td>Geoconcept</td></tr>\n
    <tr><td>GeoRSS</td><td>GeoRSS</td></tr>\n
    <tr><td>VFK</td><td>Czech Cadastral Exchange Data Format</td></tr>\n
    <tr><td>PGDUMP</td><td>PostgreSQL SQL dump</td></tr>\n
    <tr><td>OSM</td><td>OpenStreetMap XML and PBF</td></tr>\n
    <tr><td>GPSBabel</td><td>GPSBabel</td></tr>\n
    <tr><td>OGR_PDS</td><td>Planetary Data Systems TABLE</td></tr>\n
    <tr><td>WFS</td><td>OGC WFS (Web Feature Service)</td></tr>\n
    <tr><td>OAPIF</td><td>OGC API - Features</td></tr>\n
    <tr><td>EDIGEO</td><td>French EDIGEO exchange format</td></tr>\n
    <tr><td>SVG</td><td>Scalable Vector Graphics</td></tr>\n
    <tr><td>Idrisi</td><td>Idrisi Vector (.vct)</td></tr>\n
    <tr><td>XLS</td><td>MS Excel format</td></tr>\n
    <tr><td>ODS</td><td>Open Document/ LibreOffice / OpenOffice Spreadsheet </td></tr>\n
    <tr><td>XLSX</td><td>MS Office Open XML spreadsheet</td></tr>\n
    <tr><td>Elasticsearch</td><td>Elastic Search</td></tr>\n
    <tr><td>Carto</td><td>Carto</td></tr>\n
    <tr><td>AmigoCloud</td><td>AmigoCloud</td></tr>\n
    <tr><td>SXF</td><td>Storage and eXchange Format</td></tr>\n
    <tr><td>Selafin</td><td>Selafin</td></tr>\n
    <tr><td>JML</td><td>OpenJUMP JML</td></tr>\n
    <tr><td>PLSCENES</td><td>Planet Labs Scenes API</td></tr>\n
    <tr><td>CSW</td><td>OGC CSW (Catalog  Service for the Web)</td></tr>\n
    <tr><td>VDV</td><td>VDV-451/VDV-452/INTREST Data Format</td></tr>\n
    <tr><td>GMLAS</td><td>Geography Markup Language (GML) driven by application schemas</td></tr>\n
    <tr><td>MVT</td><td>Mapbox Vector Tiles</td></tr>\n
    <tr><td>NGW</td><td>NextGIS Web</td></tr>\n
    <tr><td>MapML</td><td>MapML</td></tr>\n
    <tr><td>TIGER</td><td>U.S. Census TIGER/Line</td></tr>\n
    <tr><td>AVCBin</td><td>Arc/Info Binary Coverage</td></tr>\n
    <tr><td>AVCE00</td><td>Arc/Info E00 (ASCII) Coverage</td></tr>\n
    <tr><td>GenBin</td><td>Generic Binary (.hdr Labelled)</td></tr>\n
    <tr><td>ENVI</td><td>ENVI .hdr Labelled</td></tr>\n
    <tr><td>EHdr</td><td>ESRI .hdr Labelled</td></tr>\n
    <tr><td>ISCE</td><td>ISCE raster</td></tr>\n
    <tr><td>Zarr</td><td>Zarr</td></tr>\n
    <tr><td>HTTP</td><td>HTTP Fetching Wrapper</td></tr>\n
    </table>\n
    Arguments
    ----------
    - CATALOGUE [`output shapes`] : Raster Catalogue
    - FILES [`file path`] : Files

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '7', 'Create Raster Catalogue from Files')
    if Tool.is_Okay():
        Tool.Set_Output('CATALOGUE', CATALOGUE)
        Tool.Set_Option('FILES', FILES)
        return Tool.Execute(Verbose)
    return False

def Run_Create_Raster_Catalogues_from_Directory(CATALOGUES=None, CATALOGUE_GCS=None, CATALOGUE_UKN=None, DIRECTORY=None, EXTENSIONS=None, OUTPUT=None, Verbose=2):
    '''
    Create Raster Catalogues from Directory
    ----------
    [io_gdal.8]\n
    Creates raster catalogues from all raster files found in the selected directory. Catalogues are split according to the coordinate reference system used by the different raster files. Output are polygon layers that show the extent for each raster file in it. Each extent is attributed with the original file path and raster system properties.\n
    The "GDAL Raster Import" tool imports grid data from various file formats using the "Geospatial Data Abstraction Library" (GDAL) by Frank Warmerdam. For more information have a look at the GDAL homepage:\n
    <a target="_blank" href="http://www.gdal.org/">  http://www.gdal.org</a>\n
    GDAL Version:3.6.2\n
    Following raster formats are currently supported:\n
    <table border="1"><tr><th>ID</th><th>Name</th></tr>\n
    <tr><td>ECW</td><td>ERDAS Compressed Wavelets (SDK 5.5)</td></tr>\n
    <tr><td>JP2ECW</td><td>ERDAS JPEG2000 (SDK 5.5)</td></tr>\n
    <tr><td>FITS</td><td>Flexible Image Transport System</td></tr>\n
    <tr><td>HDF4</td><td>Hierarchical Data Format Release 4</td></tr>\n
    <tr><td>HDF4Image</td><td>HDF4 Dataset</td></tr>\n
    <tr><td>HDF5</td><td>Hierarchical Data Format Release 5</td></tr>\n
    <tr><td>HDF5Image</td><td>HDF5 Dataset</td></tr>\n
    <tr><td>BAG</td><td>Bathymetry Attributed Grid</td></tr>\n
    <tr><td>KEA</td><td>KEA Image Format (.kea)</td></tr>\n
    <tr><td>netCDF</td><td>Network Common Data Format</td></tr>\n
    <tr><td>PDF</td><td>Geospatial PDF</td></tr>\n
    <tr><td>FileGDB</td><td>ESRI FileGDB</td></tr>\n
    <tr><td>ODBC</td><td></td></tr>\n
    <tr><td>VRT</td><td>Virtual Raster</td></tr>\n
    <tr><td>DERIVED</td><td>Derived datasets using VRT pixel functions</td></tr>\n
    <tr><td>GTiff</td><td>GeoTIFF</td></tr>\n
    <tr><td>COG</td><td>Cloud optimized GeoTIFF generator</td></tr>\n
    <tr><td>NITF</td><td>National Imagery Transmission Format</td></tr>\n
    <tr><td>RPFTOC</td><td>Raster Product Format TOC format</td></tr>\n
    <tr><td>ECRGTOC</td><td>ECRG TOC format</td></tr>\n
    <tr><td>HFA</td><td>Erdas Imagine Images (.img)</td></tr>\n
    <tr><td>SAR_CEOS</td><td>CEOS SAR Image</td></tr>\n
    <tr><td>CEOS</td><td>CEOS Image</td></tr>\n
    <tr><td>JAXAPALSAR</td><td>JAXA PALSAR Product Reader (Level 1.1/1.5)</td></tr>\n
    <tr><td>GFF</td><td>Ground-based SAR Applications Testbed File Format (.gff)</td></tr>\n
    <tr><td>ELAS</td><td>ELAS</td></tr>\n
    <tr><td>ESRIC</td><td>Esri Compact Cache</td></tr>\n
    <tr><td>AIG</td><td>Arc/Info Binary Grid</td></tr>\n
    <tr><td>AAIGrid</td><td>Arc/Info ASCII Grid</td></tr>\n
    <tr><td>GRASSASCIIGrid</td><td>GRASS ASCII Grid</td></tr>\n
    <tr><td>ISG</td><td>International Service for the Geoid</td></tr>\n
    <tr><td>SDTS</td><td>SDTS Raster</td></tr>\n
    <tr><td>DTED</td><td>DTED Elevation Raster</td></tr>\n
    <tr><td>PNG</td><td>Portable Network Graphics</td></tr>\n
    <tr><td>JPEG</td><td>JPEG JFIF</td></tr>\n
    <tr><td>MEM</td><td>In Memory Raster</td></tr>\n
    <tr><td>JDEM</td><td>Japanese DEM (.mem)</td></tr>\n
    <tr><td>GIF</td><td>Graphics Interchange Format (.gif)</td></tr>\n
    <tr><td>BIGGIF</td><td>Graphics Interchange Format (.gif)</td></tr>\n
    <tr><td>ESAT</td><td>Envisat Image Format</td></tr>\n
    <tr><td>BSB</td><td>Maptech BSB Nautical Charts</td></tr>\n
    <tr><td>XPM</td><td>X11 PixMap Format</td></tr>\n
    <tr><td>BMP</td><td>MS Windows Device Independent Bitmap</td></tr>\n
    <tr><td>DIMAP</td><td>SPOT DIMAP</td></tr>\n
    <tr><td>AirSAR</td><td>AirSAR Polarimetric Image</td></tr>\n
    <tr><td>RS2</td><td>RadarSat 2 XML Product</td></tr>\n
    <tr><td>SAFE</td><td>Sentinel-1 SAR SAFE Product</td></tr>\n
    <tr><td>PCIDSK</td><td>PCIDSK Database File</td></tr>\n
    <tr><td>PCRaster</td><td>PCRaster Raster File</td></tr>\n
    <tr><td>ILWIS</td><td>ILWIS Raster Map</td></tr>\n
    <tr><td>SGI</td><td>SGI Image File Format 1.0</td></tr>\n
    <tr><td>SRTMHGT</td><td>SRTMHGT File Format</td></tr>\n
    <tr><td>Leveller</td><td>Leveller heightfield</td></tr>\n
    <tr><td>Terragen</td><td>Terragen heightfield</td></tr>\n
    <tr><td>ISIS3</td><td>USGS Astrogeology ISIS cube (Version 3)</td></tr>\n
    <tr><td>ISIS2</td><td>USGS Astrogeology ISIS cube (Version 2)</td></tr>\n
    <tr><td>PDS</td><td>NASA Planetary Data System</td></tr>\n
    <tr><td>PDS4</td><td>NASA Planetary Data System 4</td></tr>\n
    <tr><td>VICAR</td><td>MIPL VICAR file</td></tr>\n
    <tr><td>TIL</td><td>EarthWatch .TIL</td></tr>\n
    <tr><td>ERS</td><td>ERMapper .ers Labelled</td></tr>\n
    <tr><td>JP2OpenJPEG</td><td>JPEG-2000 driver based on OpenJPEG library</td></tr>\n
    <tr><td>L1B</td><td>NOAA Polar Orbiter Level 1b Data Set</td></tr>\n
    <tr><td>FIT</td><td>FIT Image</td></tr>\n
    <tr><td>GRIB</td><td>GRIdded Binary (.grb, .grb2)</td></tr>\n
    <tr><td>RMF</td><td>Raster Matrix Format</td></tr>\n
    <tr><td>WCS</td><td>OGC Web Coverage Service</td></tr>\n
    <tr><td>WMS</td><td>OGC Web Map Service</td></tr>\n
    <tr><td>MSGN</td><td>EUMETSAT Archive native (.nat)</td></tr>\n
    <tr><td>RST</td><td>Idrisi Raster A.1</td></tr>\n
    <tr><td>GSAG</td><td>Golden Software ASCII Grid (.grd)</td></tr>\n
    <tr><td>GSBG</td><td>Golden Software Binary Grid (.grd)</td></tr>\n
    <tr><td>GS7BG</td><td>Golden Software 7 Binary Grid (.grd)</td></tr>\n
    <tr><td>COSAR</td><td>COSAR Annotated Binary Matrix (TerraSAR-X)</td></tr>\n
    <tr><td>TSX</td><td>TerraSAR-X Product</td></tr>\n
    <tr><td>COASP</td><td>DRDC COASP SAR Processor Raster</td></tr>\n
    <tr><td>R</td><td>R Object Data Store</td></tr>\n
    <tr><td>MAP</td><td>OziExplorer .MAP</td></tr>\n
    <tr><td>KMLSUPEROVERLAY</td><td>Kml Super Overlay</td></tr>\n
    <tr><td>WEBP</td><td>WEBP</td></tr>\n
    <tr><td>Rasterlite</td><td>Rasterlite</td></tr>\n
    <tr><td>MBTiles</td><td>MBTiles</td></tr>\n
    <tr><td>PLMOSAIC</td><td>Planet Labs Mosaics API</td></tr>\n
    <tr><td>CALS</td><td>CALS (Type 1)</td></tr>\n
    <tr><td>WMTS</td><td>OGC Web Map Tile Service</td></tr>\n
    <tr><td>SENTINEL2</td><td>Sentinel 2</td></tr>\n
    <tr><td>MRF</td><td>Meta Raster Format</td></tr>\n
    <tr><td>PNM</td><td>Portable Pixmap Format (netpbm)</td></tr>\n
    <tr><td>DOQ1</td><td>USGS DOQ (Old Style)</td></tr>\n
    <tr><td>DOQ2</td><td>USGS DOQ (New Style)</td></tr>\n
    <tr><td>PAux</td><td>PCI .aux Labelled</td></tr>\n
    <tr><td>MFF</td><td>Vexcel MFF Raster</td></tr>\n
    <tr><td>MFF2</td><td>Vexcel MFF2 (HKV) Raster</td></tr>\n
    <tr><td>GSC</td><td>GSC Geogrid</td></tr>\n
    <tr><td>FAST</td><td>EOSAT FAST Format</td></tr>\n
    <tr><td>BT</td><td>VTP .bt (Binary Terrain) 1.3 Format</td></tr>\n
    <tr><td>LAN</td><td>Erdas .LAN/.GIS</td></tr>\n
    <tr><td>CPG</td><td>Convair PolGASP</td></tr>\n
    <tr><td>NDF</td><td>NLAPS Data Format</td></tr>\n
    <tr><td>EIR</td><td>Erdas Imagine Raw</td></tr>\n
    <tr><td>DIPEx</td><td>DIPEx</td></tr>\n
    <tr><td>LCP</td><td>FARSITE v.4 Landscape File (.lcp)</td></tr>\n
    <tr><td>GTX</td><td>NOAA Vertical Datum .GTX</td></tr>\n
    <tr><td>LOSLAS</td><td>NADCON .los/.las Datum Grid Shift</td></tr>\n
    <tr><td>NTv2</td><td>NTv2 Datum Grid Shift</td></tr>\n
    <tr><td>CTable2</td><td>CTable2 Datum Grid Shift</td></tr>\n
    <tr><td>ACE2</td><td>ACE2</td></tr>\n
    <tr><td>SNODAS</td><td>Snow Data Assimilation System</td></tr>\n
    <tr><td>KRO</td><td>KOLOR Raw</td></tr>\n
    <tr><td>ROI_PAC</td><td>ROI_PAC raster</td></tr>\n
    <tr><td>RRASTER</td><td>R Raster</td></tr>\n
    <tr><td>BYN</td><td>Natural Resources Canada's Geoid</td></tr>\n
    <tr><td>ARG</td><td>Azavea Raster Grid format</td></tr>\n
    <tr><td>RIK</td><td>Swedish Grid RIK (.rik)</td></tr>\n
    <tr><td>USGSDEM</td><td>USGS Optional ASCII DEM (and CDED)</td></tr>\n
    <tr><td>GXF</td><td>GeoSoft Grid Exchange Format</td></tr>\n
    <tr><td>NWT_GRD</td><td>Northwood Numeric Grid Format .grd/.tab</td></tr>\n
    <tr><td>NWT_GRC</td><td>Northwood Classified Grid Format .grc/.tab</td></tr>\n
    <tr><td>ADRG</td><td>ARC Digitized Raster Graphics</td></tr>\n
    <tr><td>SRP</td><td>Standard Raster Product (ASRP/USRP)</td></tr>\n
    <tr><td>BLX</td><td>Magellan topo (.blx)</td></tr>\n
    <tr><td>PostGISRaster</td><td>PostGIS Raster driver</td></tr>\n
    <tr><td>SAGA</td><td>SAGA GIS Binary Grid (.sdat, .sg-grd-z)</td></tr>\n
    <tr><td>XYZ</td><td>ASCII Gridded XYZ</td></tr>\n
    <tr><td>HF2</td><td>HF2/HFZ heightfield raster</td></tr>\n
    <tr><td>OZI</td><td>OziExplorer Image File</td></tr>\n
    <tr><td>CTG</td><td>USGS LULC Composite Theme Grid</td></tr>\n
    <tr><td>ZMap</td><td>ZMap Plus Grid</td></tr>\n
    <tr><td>NGSGEOID</td><td>NOAA NGS Geoid Height Grids</td></tr>\n
    <tr><td>IRIS</td><td>IRIS data (.PPI, .CAPPi etc)</td></tr>\n
    <tr><td>PRF</td><td>Racurs PHOTOMOD PRF</td></tr>\n
    <tr><td>EEDAI</td><td>Earth Engine Data API Image</td></tr>\n
    <tr><td>EEDA</td><td>Earth Engine Data API</td></tr>\n
    <tr><td>DAAS</td><td>Airbus DS Intelligence Data As A Service driver</td></tr>\n
    <tr><td>SIGDEM</td><td>Scaled Integer Gridded DEM .sigdem</td></tr>\n
    <tr><td>TGA</td><td>TGA/TARGA Image File Format</td></tr>\n
    <tr><td>OGCAPI</td><td>OGCAPI</td></tr>\n
    <tr><td>STACTA</td><td>Spatio-Temporal Asset Catalog Tiled Assets</td></tr>\n
    <tr><td>STACIT</td><td>Spatio-Temporal Asset Catalog Items</td></tr>\n
    <tr><td>GNMFile</td><td>Geographic Network generic file based model</td></tr>\n
    <tr><td>GNMDatabase</td><td>Geographic Network generic DB based model</td></tr>\n
    <tr><td>ESRI Shapefile</td><td>ESRI Shapefile</td></tr>\n
    <tr><td>MapInfo File</td><td>MapInfo File</td></tr>\n
    <tr><td>UK .NTF</td><td>UK .NTF</td></tr>\n
    <tr><td>LVBAG</td><td>Kadaster LV BAG Extract 2.0</td></tr>\n
    <tr><td>OGR_SDTS</td><td>SDTS</td></tr>\n
    <tr><td>S57</td><td>IHO S-57 (ENC)</td></tr>\n
    <tr><td>DGN</td><td>Microstation DGN</td></tr>\n
    <tr><td>OGR_VRT</td><td>VRT - Virtual Datasource</td></tr>\n
    <tr><td>Memory</td><td>Memory</td></tr>\n
    <tr><td>CSV</td><td>Comma Separated Value (.csv)</td></tr>\n
    <tr><td>NAS</td><td>NAS - ALKIS</td></tr>\n
    <tr><td>GML</td><td>Geography Markup Language (GML)</td></tr>\n
    <tr><td>GPX</td><td>GPX</td></tr>\n
    <tr><td>LIBKML</td><td>Keyhole Markup Language (LIBKML)</td></tr>\n
    <tr><td>KML</td><td>Keyhole Markup Language (KML)</td></tr>\n
    <tr><td>GeoJSON</td><td>GeoJSON</td></tr>\n
    <tr><td>GeoJSONSeq</td><td>GeoJSON Sequence</td></tr>\n
    <tr><td>ESRIJSON</td><td>ESRIJSON</td></tr>\n
    <tr><td>TopoJSON</td><td>TopoJSON</td></tr>\n
    <tr><td>Interlis 1</td><td>Interlis 1</td></tr>\n
    <tr><td>Interlis 2</td><td>Interlis 2</td></tr>\n
    <tr><td>OGR_GMT</td><td>GMT ASCII Vectors (.gmt)</td></tr>\n
    <tr><td>GPKG</td><td>GeoPackage</td></tr>\n
    <tr><td>SQLite</td><td>SQLite / Spatialite</td></tr>\n
    <tr><td>WAsP</td><td>WAsP .map format</td></tr>\n
    <tr><td>PGeo</td><td>ESRI Personal GeoDatabase</td></tr>\n
    <tr><td>OGR_OGDI</td><td>OGDI Vectors (VPF, VMAP, DCW)</td></tr>\n
    <tr><td>PostgreSQL</td><td>PostgreSQL/PostGIS</td></tr>\n
    <tr><td>MySQL</td><td>MySQL</td></tr>\n
    <tr><td>OpenFileGDB</td><td>ESRI FileGDB</td></tr>\n
    <tr><td>DXF</td><td>AutoCAD DXF</td></tr>\n
    <tr><td>CAD</td><td>AutoCAD Driver</td></tr>\n
    <tr><td>FlatGeobuf</td><td>FlatGeobuf</td></tr>\n
    <tr><td>Geoconcept</td><td>Geoconcept</td></tr>\n
    <tr><td>GeoRSS</td><td>GeoRSS</td></tr>\n
    <tr><td>VFK</td><td>Czech Cadastral Exchange Data Format</td></tr>\n
    <tr><td>PGDUMP</td><td>PostgreSQL SQL dump</td></tr>\n
    <tr><td>OSM</td><td>OpenStreetMap XML and PBF</td></tr>\n
    <tr><td>GPSBabel</td><td>GPSBabel</td></tr>\n
    <tr><td>OGR_PDS</td><td>Planetary Data Systems TABLE</td></tr>\n
    <tr><td>WFS</td><td>OGC WFS (Web Feature Service)</td></tr>\n
    <tr><td>OAPIF</td><td>OGC API - Features</td></tr>\n
    <tr><td>EDIGEO</td><td>French EDIGEO exchange format</td></tr>\n
    <tr><td>SVG</td><td>Scalable Vector Graphics</td></tr>\n
    <tr><td>Idrisi</td><td>Idrisi Vector (.vct)</td></tr>\n
    <tr><td>XLS</td><td>MS Excel format</td></tr>\n
    <tr><td>ODS</td><td>Open Document/ LibreOffice / OpenOffice Spreadsheet </td></tr>\n
    <tr><td>XLSX</td><td>MS Office Open XML spreadsheet</td></tr>\n
    <tr><td>Elasticsearch</td><td>Elastic Search</td></tr>\n
    <tr><td>Carto</td><td>Carto</td></tr>\n
    <tr><td>AmigoCloud</td><td>AmigoCloud</td></tr>\n
    <tr><td>SXF</td><td>Storage and eXchange Format</td></tr>\n
    <tr><td>Selafin</td><td>Selafin</td></tr>\n
    <tr><td>JML</td><td>OpenJUMP JML</td></tr>\n
    <tr><td>PLSCENES</td><td>Planet Labs Scenes API</td></tr>\n
    <tr><td>CSW</td><td>OGC CSW (Catalog  Service for the Web)</td></tr>\n
    <tr><td>VDV</td><td>VDV-451/VDV-452/INTREST Data Format</td></tr>\n
    <tr><td>GMLAS</td><td>Geography Markup Language (GML) driven by application schemas</td></tr>\n
    <tr><td>MVT</td><td>Mapbox Vector Tiles</td></tr>\n
    <tr><td>NGW</td><td>NextGIS Web</td></tr>\n
    <tr><td>MapML</td><td>MapML</td></tr>\n
    <tr><td>TIGER</td><td>U.S. Census TIGER/Line</td></tr>\n
    <tr><td>AVCBin</td><td>Arc/Info Binary Coverage</td></tr>\n
    <tr><td>AVCE00</td><td>Arc/Info E00 (ASCII) Coverage</td></tr>\n
    <tr><td>GenBin</td><td>Generic Binary (.hdr Labelled)</td></tr>\n
    <tr><td>ENVI</td><td>ENVI .hdr Labelled</td></tr>\n
    <tr><td>EHdr</td><td>ESRI .hdr Labelled</td></tr>\n
    <tr><td>ISCE</td><td>ISCE raster</td></tr>\n
    <tr><td>Zarr</td><td>Zarr</td></tr>\n
    <tr><td>HTTP</td><td>HTTP Fetching Wrapper</td></tr>\n
    </table>\n
    Arguments
    ----------
    - CATALOGUES [`output shapes list`] : Raster Catalogues
    - CATALOGUE_GCS [`output shapes`] : Raster Catalogue
    - CATALOGUE_UKN [`output shapes`] : Raster Catalogue (unknown CRS)
    - DIRECTORY [`file path`] : Directory
    - EXTENSIONS [`text`] : Extensions. Default: sgrd; tif
    - OUTPUT [`choice`] : Output. Available Choices: [0] one catalogue for each coordinate system [1] one catalogue using geographic coordinates Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '8', 'Create Raster Catalogues from Directory')
    if Tool.is_Okay():
        Tool.Set_Output('CATALOGUES', CATALOGUES)
        Tool.Set_Output('CATALOGUE_GCS', CATALOGUE_GCS)
        Tool.Set_Output('CATALOGUE_UKN', CATALOGUE_UKN)
        Tool.Set_Option('DIRECTORY', DIRECTORY)
        Tool.Set_Option('EXTENSIONS', EXTENSIONS)
        Tool.Set_Option('OUTPUT', OUTPUT)
        return Tool.Execute(Verbose)
    return False

def Run_Import_TMS_Image(TARGET=None, TARGET_MAP=None, MAP=None, TARGET_GRIDSYSTEM=None, TARGET_MAP_GRIDSYSTEM=None, SERVER=None, SERVER_USER=None, SERVER_EPSG=None, BLOCKSIZE=None, CACHE=None, CACHE_DIR=None, GRAYSCALE=None, XMIN=None, YMIN=None, XMAX=None, YMAX=None, NX=None, NY=None, Verbose=2):
    '''
    Import TMS Image
    ----------
    [io_gdal.9]\n
    The "Import TMS Image" tool imports a map image from a Tile Mapping Service (TMS) using the "Geospatial Data Abstraction Library" (GDAL) by Frank Warmerdam.\n
    GDAL Version:3.6.2\n
    Arguments
    ----------
    - TARGET [`optional input grid`] : Target System
    - TARGET_MAP [`output grid`] : Target Map
    - MAP [`output data object`] : Map
    - TARGET_GRIDSYSTEM [`grid system`] : Grid system
    - TARGET_MAP_GRIDSYSTEM [`grid system`] : Grid system
    - SERVER [`choice`] : Server. Available Choices: [0] Open Street Map [1] Google Map [2] Google Satellite [3] Google Hybrid [4] Google Terrain [5] Google Terrain, Streets and Water [6] ArcGIS MapServer Tiles [7] TopPlusOpen [8] EMODnet Bathymetry WMTS service [9] user defined Default: 0
    - SERVER_USER [`text`] : Server. Default: tile.openstreetmap.org/${z}/${x}/${y}.png
    - SERVER_EPSG [`integer number`] : EPSG. Default: 3857
    - BLOCKSIZE [`integer number`] : Block Size. Minimum: 32 Default: 256
    - CACHE [`boolean`] : Cache. Default: 0 Enable local disk cache. Allows for offline operation.
    - CACHE_DIR [`file path`] : Cache Directory. If not specified the cache will be created in the current user's temporary directory.
    - GRAYSCALE [`boolean`] : Gray Scale Image. Default: 0
    - XMIN [`floating point number`] : West. Minimum: -20037508.340000 Maximum: 20037508.340000 Default: -20037508.340000
    - YMIN [`floating point number`] : South. Minimum: -20037508.340000 Maximum: 20037508.340000 Default: -20037508.340000
    - XMAX [`floating point number`] : East. Minimum: -20037508.340000 Maximum: 20037508.340000 Default: 20037508.340000
    - YMAX [`floating point number`] : North. Minimum: -20037508.340000 Maximum: 20037508.340000 Default: 20037508.340000
    - NX [`integer number`] : Columns. Minimum: 1 Default: 600
    - NY [`integer number`] : Rows. Minimum: 1 Default: 600

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '9', 'Import TMS Image')
    if Tool.is_Okay():
        Tool.Set_Input ('TARGET', TARGET)
        Tool.Set_Output('TARGET_MAP', TARGET_MAP)
        Tool.Set_Output('MAP', MAP)
        Tool.Set_Option('TARGET_GRIDSYSTEM', TARGET_GRIDSYSTEM)
        Tool.Set_Option('TARGET_MAP_GRIDSYSTEM', TARGET_MAP_GRIDSYSTEM)
        Tool.Set_Option('SERVER', SERVER)
        Tool.Set_Option('SERVER_USER', SERVER_USER)
        Tool.Set_Option('SERVER_EPSG', SERVER_EPSG)
        Tool.Set_Option('BLOCKSIZE', BLOCKSIZE)
        Tool.Set_Option('CACHE', CACHE)
        Tool.Set_Option('CACHE_DIR', CACHE_DIR)
        Tool.Set_Option('GRAYSCALE', GRAYSCALE)
        Tool.Set_Option('XMIN', XMIN)
        Tool.Set_Option('YMIN', YMIN)
        Tool.Set_Option('XMAX', XMAX)
        Tool.Set_Option('YMAX', YMAX)
        Tool.Set_Option('NX', NX)
        Tool.Set_Option('NY', NY)
        return Tool.Execute(Verbose)
    return False

def Run_GDAL_Formats(FORMATS=None, TYPE=None, ACCESS=None, RECOGNIZED=None, Verbose=2):
    '''
    GDAL Formats
    ----------
    [io_gdal.10]\n
    This tool lists all (file) formats supported by the currently loaded GDAL library.\n
    GDAL Version:3.6.2\n
    Arguments
    ----------
    - FORMATS [`output table`] : GDAL Formats
    - TYPE [`choice`] : Type. Available Choices: [0] raster [1] vector [2] all Default: 2
    - ACCESS [`choice`] : Access. Available Choices: [0] read [1] write [2] read or write Default: 2
    - RECOGNIZED [`boolean`] : All Recognized Files. Default: 1 Add an entry for all recognized files.

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '10', 'GDAL Formats')
    if Tool.is_Okay():
        Tool.Set_Output('FORMATS', FORMATS)
        Tool.Set_Option('TYPE', TYPE)
        Tool.Set_Option('ACCESS', ACCESS)
        Tool.Set_Option('RECOGNIZED', RECOGNIZED)
        return Tool.Execute(Verbose)
    return False

def Run_Import_ASTER_Scene(VNIR=None, SWIR=None, TIR=None, BANDS=None, METADATA=None, FILE=None, FORMAT=None, Verbose=2):
    '''
    Import ASTER Scene
    ----------
    [io_gdal.11]\n
    Import ASTER scene from Hierarchical Data Format version 4 (HDF4).\n
    Arguments
    ----------
    - VNIR [`output data object`] : Visible and Near Infrared
    - SWIR [`output data object`] : Short Wave Infrared
    - TIR [`output data object`] : Thermal Infrared
    - BANDS [`output grid list`] : Bands
    - METADATA [`output table`] : Metadata
    - FILE [`file path`] : File
    - FORMAT [`choice`] : Format. Available Choices: [0] single grids [1] grid collections Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '11', 'Import ASTER Scene')
    if Tool.is_Okay():
        Tool.Set_Output('VNIR', VNIR)
        Tool.Set_Output('SWIR', SWIR)
        Tool.Set_Output('TIR', TIR)
        Tool.Set_Output('BANDS', BANDS)
        Tool.Set_Output('METADATA', METADATA)
        Tool.Set_Option('FILE', FILE)
        Tool.Set_Option('FORMAT', FORMAT)
        return Tool.Execute(Verbose)
    return False

def Run_Create_Virtual_Raster_VRT(FILES=None, FILE_LIST=None, VRT_NAME=None, RESAMPLING=None, RESOLUTION=None, CELLSIZE=None, ALIGN=None, Verbose=2):
    '''
    Create Virtual Raster (VRT)
    ----------
    [io_gdal.12]\n
    The tool allows one to create a virtual dataset (VRT) which is a mosaic of the input raster datasets. Such a VRT can be used for seamless data access to a large number of raster tiles, a typical application is the clipping of raster tiles from such a VRT.\n
    GDAL Version:3.6.2\n
    Following raster formats are currently supported:\n
    <table border="1"><tr><th>ID</th><th>Name</th><th>Extension</th></tr><tr><td>ECW</td><td>ERDAS Compressed Wavelets (SDK 5.5)</td><td>ecw</td></tr><tr><td>JP2ECW</td><td>ERDAS JPEG2000 (SDK 5.5)</td><td>jp2</td></tr><tr><td>FITS</td><td>Flexible Image Transport System</td><td></td></tr><tr><td>HDF4</td><td>Hierarchical Data Format Release 4</td><td>hdf</td></tr><tr><td>HDF4Image</td><td>HDF4 Dataset</td><td></td></tr><tr><td>HDF5</td><td>Hierarchical Data Format Release 5</td><td></td></tr><tr><td>HDF5Image</td><td>HDF5 Dataset</td><td></td></tr><tr><td>BAG</td><td>Bathymetry Attributed Grid</td><td>bag</td></tr><tr><td>KEA</td><td>KEA Image Format (.kea)</td><td>kea</td></tr><tr><td>netCDF</td><td>Network Common Data Format</td><td>nc</td></tr><tr><td>PDF</td><td>Geospatial PDF</td><td>pdf</td></tr><tr><td>VRT</td><td>Virtual Raster</td><td>vrt</td></tr><tr><td>DERIVED</td><td>Derived datasets using VRT pixel functions</td><td></td></tr><tr><td>GTiff</td><td>GeoTIFF</td><td>tif</td></tr><tr><td>NITF</td><td>National Imagery Transmission Format</td><td>ntf</td></tr><tr><td>RPFTOC</td><td>Raster Product Format TOC format</td><td>toc</td></tr><tr><td>ECRGTOC</td><td>ECRG TOC format</td><td>xml</td></tr><tr><td>HFA</td><td>Erdas Imagine Images (.img)</td><td>img</td></tr><tr><td>SAR_CEOS</td><td>CEOS SAR Image</td><td></td></tr><tr><td>CEOS</td><td>CEOS Image</td><td></td></tr><tr><td>JAXAPALSAR</td><td>JAXA PALSAR Product Reader (Level 1.1/1.5)</td><td></td></tr><tr><td>GFF</td><td>Ground-based SAR Applications Testbed File Format (.gff)</td><td>gff</td></tr><tr><td>ELAS</td><td>ELAS</td><td></td></tr><tr><td>ESRIC</td><td>Esri Compact Cache</td><td></td></tr><tr><td>AIG</td><td>Arc/Info Binary Grid</td><td></td></tr><tr><td>AAIGrid</td><td>Arc/Info ASCII Grid</td><td>asc</td></tr><tr><td>GRASSASCIIGrid</td><td>GRASS ASCII Grid</td><td></td></tr><tr><td>ISG</td><td>International Service for the Geoid</td><td>isg</td></tr><tr><td>SDTS</td><td>SDTS Raster</td><td>ddf</td></tr><tr><td>DTED</td><td>DTED Elevation Raster</td><td></td></tr><tr><td>PNG</td><td>Portable Network Graphics</td><td>png</td></tr><tr><td>JPEG</td><td>JPEG JFIF</td><td>jpg</td></tr><tr><td>MEM</td><td>In Memory Raster</td><td></td></tr><tr><td>JDEM</td><td>Japanese DEM (.mem)</td><td>mem</td></tr><tr><td>GIF</td><td>Graphics Interchange Format (.gif)</td><td>gif</td></tr><tr><td>BIGGIF</td><td>Graphics Interchange Format (.gif)</td><td>gif</td></tr><tr><td>ESAT</td><td>Envisat Image Format</td><td>n1</td></tr><tr><td>BSB</td><td>Maptech BSB Nautical Charts</td><td>kap</td></tr><tr><td>XPM</td><td>X11 PixMap Format</td><td>xpm</td></tr><tr><td>BMP</td><td>MS Windows Device Independent Bitmap</td><td>bmp</td></tr><tr><td>DIMAP</td><td>SPOT DIMAP</td><td></td></tr><tr><td>AirSAR</td><td>AirSAR Polarimetric Image</td><td></td></tr><tr><td>RS2</td><td>RadarSat 2 XML Product</td><td></td></tr><tr><td>SAFE</td><td>Sentinel-1 SAR SAFE Product</td><td></td></tr><tr><td>PCIDSK</td><td>PCIDSK Database File</td><td>pix</td></tr><tr><td>PCRaster</td><td>PCRaster Raster File</td><td>map</td></tr><tr><td>ILWIS</td><td>ILWIS Raster Map</td><td></td></tr><tr><td>SGI</td><td>SGI Image File Format 1.0</td><td>rgb</td></tr><tr><td>SRTMHGT</td><td>SRTMHGT File Format</td><td>hgt</td></tr><tr><td>Leveller</td><td>Leveller heightfield</td><td>ter</td></tr><tr><td>Terragen</td><td>Terragen heightfield</td><td>ter</td></tr><tr><td>ISIS3</td><td>USGS Astrogeology ISIS cube (Version 3)</td><td></td></tr><tr><td>ISIS2</td><td>USGS Astrogeology ISIS cube (Version 2)</td><td></td></tr><tr><td>PDS</td><td>NASA Planetary Data System</td><td></td></tr><tr><td>PDS4</td><td>NASA Planetary Data System 4</td><td>xml</td></tr><tr><td>VICAR</td><td>MIPL VICAR file</td><td></td></tr><tr><td>TIL</td><td>EarthWatch .TIL</td><td></td></tr><tr><td>ERS</td><td>ERMapper .ers Labelled</td><td>ers</td></tr><tr><td>JP2OpenJPEG</td><td>JPEG-2000 driver based on OpenJPEG library</td><td>jp2</td></tr><tr><td>L1B</td><td>NOAA Polar Orbiter Level 1b Data Set</td><td></td></tr><tr><td>FIT</td><td>FIT Image</td><td></td></tr><tr><td>GRIB</td><td>GRIdded Binary (.grb, .grb2)</td><td></td></tr><tr><td>RMF</td><td>Raster Matrix Format</td><td>rsw</td></tr><tr><td>WCS</td><td>OGC Web Coverage Service</td><td></td></tr><tr><td>WMS</td><td>OGC Web Map Service</td><td></td></tr><tr><td>MSGN</td><td>EUMETSAT Archive native (.nat)</td><td>nat</td></tr><tr><td>RST</td><td>Idrisi Raster A.1</td><td>rst</td></tr><tr><td>GSAG</td><td>Golden Software ASCII Grid (.grd)</td><td>grd</td></tr><tr><td>GSBG</td><td>Golden Software Binary Grid (.grd)</td><td>grd</td></tr><tr><td>GS7BG</td><td>Golden Software 7 Binary Grid (.grd)</td><td>grd</td></tr><tr><td>COSAR</td><td>COSAR Annotated Binary Matrix (TerraSAR-X)</td><td></td></tr><tr><td>TSX</td><td>TerraSAR-X Product</td><td></td></tr><tr><td>COASP</td><td>DRDC COASP SAR Processor Raster</td><td>hdr</td></tr><tr><td>R</td><td>R Object Data Store</td><td>rda</td></tr><tr><td>MAP</td><td>OziExplorer .MAP</td><td></td></tr><tr><td>KMLSUPEROVERLAY</td><td>Kml Super Overlay</td><td></td></tr><tr><td>WEBP</td><td>WEBP</td><td>webp</td></tr><tr><td>Rasterlite</td><td>Rasterlite</td><td>sqlite</td></tr><tr><td>MBTiles</td><td>MBTiles</td><td>mbtiles</td></tr><tr><td>PLMOSAIC</td><td>Planet Labs Mosaics API</td><td></td></tr><tr><td>CALS</td><td>CALS (Type 1)</td><td></td></tr><tr><td>WMTS</td><td>OGC Web Map Tile Service</td><td></td></tr><tr><td>SENTINEL2</td><td>Sentinel 2</td><td></td></tr><tr><td>MRF</td><td>Meta Raster Format</td><td>mrf</td></tr><tr><td>PNM</td><td>Portable Pixmap Format (netpbm)</td><td></td></tr><tr><td>DOQ1</td><td>USGS DOQ (Old Style)</td><td></td></tr><tr><td>DOQ2</td><td>USGS DOQ (New Style)</td><td></td></tr><tr><td>PAux</td><td>PCI .aux Labelled</td><td></td></tr><tr><td>MFF</td><td>Vexcel MFF Raster</td><td>hdr</td></tr><tr><td>MFF2</td><td>Vexcel MFF2 (HKV) Raster</td><td></td></tr><tr><td>GSC</td><td>GSC Geogrid</td><td></td></tr><tr><td>FAST</td><td>EOSAT FAST Format</td><td></td></tr><tr><td>BT</td><td>VTP .bt (Binary Terrain) 1.3 Format</td><td>bt</td></tr><tr><td>LAN</td><td>Erdas .LAN/.GIS</td><td></td></tr><tr><td>CPG</td><td>Convair PolGASP</td><td></td></tr><tr><td>NDF</td><td>NLAPS Data Format</td><td></td></tr><tr><td>EIR</td><td>Erdas Imagine Raw</td><td></td></tr><tr><td>DIPEx</td><td>DIPEx</td><td></td></tr><tr><td>LCP</td><td>FARSITE v.4 Landscape File (.lcp)</td><td>lcp</td></tr><tr><td>GTX</td><td>NOAA Vertical Datum .GTX</td><td>gtx</td></tr><tr><td>LOSLAS</td><td>NADCON .los/.las Datum Grid Shift</td><td></td></tr><tr><td>NTv2</td><td>NTv2 Datum Grid Shift</td><td></td></tr><tr><td>CTable2</td><td>CTable2 Datum Grid Shift</td><td></td></tr><tr><td>ACE2</td><td>ACE2</td><td>ACE2</td></tr><tr><td>SNODAS</td><td>Snow Data Assimilation System</td><td>hdr</td></tr><tr><td>KRO</td><td>KOLOR Raw</td><td>kro</td></tr><tr><td>ROI_PAC</td><td>ROI_PAC raster</td><td></td></tr><tr><td>RRASTER</td><td>R Raster</td><td>grd</td></tr><tr><td>BYN</td><td>Natural Resources Canada's Geoid</td><td></td></tr><tr><td>ARG</td><td>Azavea Raster Grid format</td><td></td></tr><tr><td>RIK</td><td>Swedish Grid RIK (.rik)</td><td>rik</td></tr><tr><td>USGSDEM</td><td>USGS Optional ASCII DEM (and CDED)</td><td>dem</td></tr><tr><td>GXF</td><td>GeoSoft Grid Exchange Format</td><td>gxf</td></tr><tr><td>NWT_GRD</td><td>Northwood Numeric Grid Format .grd/.tab</td><td>grd</td></tr><tr><td>NWT_GRC</td><td>Northwood Classified Grid Format .grc/.tab</td><td>grc</td></tr><tr><td>ADRG</td><td>ARC Digitized Raster Graphics</td><td>gen</td></tr><tr><td>SRP</td><td>Standard Raster Product (ASRP/USRP)</td><td>img</td></tr><tr><td>BLX</td><td>Magellan topo (.blx)</td><td>blx</td></tr><tr><td>PostGISRaster</td><td>PostGIS Raster driver</td><td></td></tr><tr><td>SAGA</td><td>SAGA GIS Binary Grid (.sdat, .sg-grd-z)</td><td></td></tr><tr><td>XYZ</td><td>ASCII Gridded XYZ</td><td>xyz</td></tr><tr><td>HF2</td><td>HF2/HFZ heightfield raster</td><td>hf2</td></tr><tr><td>OZI</td><td>OziExplorer Image File</td><td></td></tr><tr><td>CTG</td><td>USGS LULC Composite Theme Grid</td><td></td></tr><tr><td>ZMap</td><td>ZMap Plus Grid</td><td>dat</td></tr><tr><td>NGSGEOID</td><td>NOAA NGS Geoid Height Grids</td><td>bin</td></tr><tr><td>IRIS</td><td>IRIS data (.PPI, .CAPPi etc)</td><td>ppi</td></tr><tr><td>PRF</td><td>Racurs PHOTOMOD PRF</td><td>prf</td></tr><tr><td>EEDAI</td><td>Earth Engine Data API Image</td><td></td></tr><tr><td>DAAS</td><td>Airbus DS Intelligence Data As A Service driver</td><td></td></tr><tr><td>SIGDEM</td><td>Scaled Integer Gridded DEM .sigdem</td><td>sigdem</td></tr><tr><td>TGA</td><td>TGA/TARGA Image File Format</td><td>tga</td></tr><tr><td>OGCAPI</td><td>OGCAPI</td><td></td></tr><tr><td>STACTA</td><td>Spatio-Temporal Asset Catalog Tiled Assets</td><td>json</td></tr><tr><td>STACIT</td><td>Spatio-Temporal Asset Catalog Items</td><td></td></tr><tr><td>GPKG</td><td>GeoPackage</td><td>gpkg</td></tr><tr><td>CAD</td><td>AutoCAD Driver</td><td>dwg</td></tr><tr><td>PLSCENES</td><td>Planet Labs Scenes API</td><td></td></tr><tr><td>NGW</td><td>NextGIS Web</td><td></td></tr><tr><td>GenBin</td><td>Generic Binary (.hdr Labelled)</td><td></td></tr><tr><td>ENVI</td><td>ENVI .hdr Labelled</td><td></td></tr><tr><td>EHdr</td><td>ESRI .hdr Labelled</td><td>bil</td></tr><tr><td>ISCE</td><td>ISCE raster</td><td></td></tr><tr><td>Zarr</td><td>Zarr</td><td></td></tr><tr><td>HTTP</td><td>HTTP Fetching Wrapper</td><td></td></tr></table>\n
    Arguments
    ----------
    - FILES [`file path`] : Files. The input files.
    - FILE_LIST [`file path`] : Input File List. A text file with the full path to an input grid on each line.
    - VRT_NAME [`file path`] : VRT Filename. The full path and name of the .vrt output file.
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] nearest [1] bilinear [2] cubic [3] cubic spline [4] lanczos [5] average [6] mode Default: 0 The resampling algorithm used when datasets are queried from the VRT.
    - RESOLUTION [`choice`] : Resolution. Available Choices: [0] highest [1] lowest [2] average [3] user Default: 0 The method how to compute the output resolution if the resolution of all input files is not the same.
    - CELLSIZE [`floating point number`] : Cellsize. Minimum: 0.000000 Default: 1.000000
    - ALIGN [`boolean`] : Align. Default: 1 Align the coordinates of the extent to the cellsize.

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '12', 'Create Virtual Raster (VRT)')
    if Tool.is_Okay():
        Tool.Set_Option('FILES', FILES)
        Tool.Set_Option('FILE_LIST', FILE_LIST)
        Tool.Set_Option('VRT_NAME', VRT_NAME)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('RESOLUTION', RESOLUTION)
        Tool.Set_Option('CELLSIZE', CELLSIZE)
        Tool.Set_Option('ALIGN', ALIGN)
        return Tool.Execute(Verbose)
    return False

def Run_Import_from_Virtual_Raster_VRT(SHAPES=None, GRIDS=None, VRT_NAME=None, EXTENT=None, GRIDSYSTEM=None, XMIN=None, XMAX=None, YMIN=None, YMAX=None, BUFFER=None, MULTIPLE=None, TRANSFORM=None, RESAMPLING=None, Verbose=2):
    '''
    Import from Virtual Raster (VRT)
    ----------
    [io_gdal.13]\n
    The tool allows one to clip / extract a raster subset from a virtual raster dataset (VRT). Such a VRT is actually an XML based description of a mosaic of raster datasets and can be created with the "Create Virtual Raster (VRT)" tool.\n
    Arguments
    ----------
    - SHAPES [`input shapes`] : Shapes Extent. The shapefile defining the clipping extent.
    - GRIDS [`output grid list`] : Grids
    - VRT_NAME [`file path`] : VRT Filename. The full path and name of the .vrt file.
    - EXTENT [`choice`] : Extent. Available Choices: [0] user defined [1] grid system [2] shapes extent Default: 0 Choose how to define the clipping extent.
    - GRIDSYSTEM [`grid system`] : Grid System. The target grid system.
    - XMIN [`floating point number`] : Left. Default: 0.000000 The minimum x-coordinate of the clipping extent.
    - XMAX [`floating point number`] : Right. Default: 0.000000 The maximum x-coordinate of the clipping extent.
    - YMIN [`floating point number`] : Bottom. Default: 0.000000 The minimum y-coordinate of the clipping extent.
    - YMAX [`floating point number`] : Top. Default: 0.000000 The maximum y-coordinate of the clipping extent.
    - BUFFER [`floating point number`] : Buffer. Minimum: 0.000000 Default: 0.000000 The optional buffer added to the clipping extent [map units].
    - MULTIPLE [`choice`] : Multiple Bands Output. Available Choices: [0] single grids [1] grid collection [2] automatic Default: 2 Choose how to handle output datasets with multiple bands.
    - TRANSFORM [`boolean`] : Transformation. Default: 1 Align grid to coordinate system.
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] Nearest Neighbour [1] Bilinear Interpolation [2] Bicubic Spline Interpolation [3] B-Spline Interpolation Default: 0 Resampling type to be used, if grid needs to be aligned to coordinate system.

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '13', 'Import from Virtual Raster (VRT)')
    if Tool.is_Okay():
        Tool.Set_Input ('SHAPES', SHAPES)
        Tool.Set_Output('GRIDS', GRIDS)
        Tool.Set_Option('VRT_NAME', VRT_NAME)
        Tool.Set_Option('EXTENT', EXTENT)
        Tool.Set_Option('GRIDSYSTEM', GRIDSYSTEM)
        Tool.Set_Option('XMIN', XMIN)
        Tool.Set_Option('XMAX', XMAX)
        Tool.Set_Option('YMIN', YMIN)
        Tool.Set_Option('YMAX', YMAX)
        Tool.Set_Option('BUFFER', BUFFER)
        Tool.Set_Option('MULTIPLE', MULTIPLE)
        Tool.Set_Option('TRANSFORM', TRANSFORM)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        return Tool.Execute(Verbose)
    return False

def Run_Create_Raster_Catalogue_from_Virtual_Raster_VRT(CATALOGUE=None, VRT_FILE=None, Verbose=2):
    '''
    Create Raster Catalogue from Virtual Raster (VRT)
    ----------
    [io_gdal.14]\n
    The tool allows one to create a polygon layer that shows the extent of each raster file referenced in the virtual raster. Each extent is attributed with the original file path, which can be used to load the dataset by 'CTRL + left-click' in the table field.\n
    Note: the tool only supports basic variants of the VRT format.\n
    Arguments
    ----------
    - CATALOGUE [`output shapes`] : Raster Catalogue. The polygon layer with dataset boundaries.
    - VRT_FILE [`file path`] : VRT File. The full path and name of the .vrt input file.

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('io_gdal', '14', 'Create Raster Catalogue from Virtual Raster (VRT)')
    if Tool.is_Okay():
        Tool.Set_Output('CATALOGUE', CATALOGUE)
        Tool.Set_Option('VRT_FILE', VRT_FILE)
        return Tool.Execute(Verbose)
    return False

