# acx_pthread.m4 - Configure pthreads for the host system. -*-Shell-script-*-
## http://www.gnu.org/software/ac-archive/Installed_Packages/acx_pthread.html
## Steven G. Johnson <stevenj@alum.mit.edu> and Alejandro Forero Cuervo
## <bachue@bachue.com>

AC_DEFUN([ACX_PTHREAD], [
AC_REQUIRE([AC_CANONICAL_HOST])
AC_LANG_SAVE
AC_LANG_C
acx_pthread_ok=no

# We used to check for pthread.h first, but this fails if pthread.h
# requires special compiler flags (e.g. on True64 or Sequent).
# It gets checked for in the link test anyway.

# First of all, check if the user has set any of the PTHREAD_LIBS,
# etcetera environment variables, and if threads linking works using
# them:
if test x"$PTHREAD_LIBS$PTHREAD_CFLAGS" != x; then
        save_CFLAGS="$CFLAGS"
        CFLAGS="$CFLAGS $PTHREAD_CFLAGS"
        save_LIBS="$LIBS"
        LIBS="$PTHREAD_LIBS $LIBS"
        AC_MSG_CHECKING([for pthread_join in LIBS=$PTHREAD_LIBS with CFLAGS=$PTHREAD_CFLAGS])
        AC_TRY_LINK_FUNC(pthread_join, acx_pthread_ok=yes)
        AC_MSG_RESULT($acx_pthread_ok)
        if test x"$acx_pthread_ok" = xno; then
                PTHREAD_LIBS=""
                PTHREAD_CFLAGS=""
        fi
        LIBS="$save_LIBS"
        CFLAGS="$save_CFLAGS"
fi

# We must check for the threads library under a number of different
# names; the ordering is very important because some systems
# (e.g. DEC) have both -lpthread and -lpthreads, where one of the
# libraries is broken (non-POSIX).

# Create a list of thread flags to try.  Items starting with a "-" are
# C compiler flags, and other items are library names, except for "none"
# which indicates that we try without any flags at all.

acx_pthread_flags="pthreads none -Kthread -kthread lthread -pthread -pthreads -mthreads pthread --thread-safe -mt"

# The ordering *is* (sometimes) important.  Some notes on the
# individual items follow:

# pthreads: AIX (must check this before -lpthread)
# none: in case threads are in libc; should be tried before -Kthread and
#       other compiler flags to prevent continual compiler warnings
# -Kthread: Sequent (threads in libc, but -Kthread needed for pthread.h)
# -kthread: FreeBSD kernel threads (preferred to -pthread since SMP-able)
# lthread: LinuxThreads port on FreeBSD (also preferred to -pthread)
# -pthread: Linux/gcc (kernel threads), BSD/gcc (userland threads)
# -pthreads: Solaris/gcc
# -mthreads: Mingw32/gcc, Lynx/gcc
# -mt: Sun Workshop C (may only link SunOS threads [-lthread], but it
#      doesn't hurt to check since this sometimes defines pthreads too;
#      also defines -D_REENTRANT)
# pthread: Linux, etcetera
# --thread-safe: KAI C++

case "${host_cpu}-${host_os}" in
        *solaris*)

        # On Solaris (at least, for some versions), libc contains stubbed
        # (non-functional) versions of the pthreads routines, so link-based
        # tests will erroneously succeed.  (We need to link with -pthread or
        # -lpthread.)  (The stubs are missing pthread_cleanup_push, or rather
        # a function called by this macro, so we could check for that, but
        # who knows whether they'll stub that too in a future libc.)  So,
        # we'll just look for -pthreads and -lpthread first:

        acx_pthread_flags="-pthread -pthreads pthread -mt $acx_pthread_flags"
        ;;
esac

if test x"$acx_pthread_ok" = xno; then
for flag in $acx_pthread_flags; do

        case $flag in
                none)
                AC_MSG_CHECKING([whether pthreads work without any flags])
                ;;

                -*)
                AC_MSG_CHECKING([whether pthreads work with $flag])
                PTHREAD_CFLAGS="$flag"
                ;;

                *)
                AC_MSG_CHECKING([for the pthreads library -l$flag])
                PTHREAD_LIBS="-l$flag"
                ;;
        esac

        save_LIBS="$LIBS"
        save_CFLAGS="$CFLAGS"
        LIBS="$PTHREAD_LIBS $LIBS"
        CFLAGS="$CFLAGS $PTHREAD_CFLAGS"

        # Check for various functions.  We must include pthread.h,
        # since some functions may be macros.  (On the Sequent, we
        # need a special flag -Kthread to make this header compile.)
        # We check for pthread_join because it is in -lpthread on IRIX
        # while pthread_create is in libc.  We check for pthread_attr_init
        # due to DEC craziness with -lpthreads.  We check for
        # pthread_cleanup_push because it is one of the few pthread
        # functions on Solaris that doesn't have a non-functional libc stub.
        # We try pthread_create on general principles.
        AC_TRY_LINK([#include <pthread.h>],
                    [pthread_t th; pthread_join(th, 0);
                     pthread_attr_init(0); pthread_cleanup_push(0, 0);
                     pthread_create(0,0,0,0); pthread_cleanup_pop(0); ],
                    [acx_pthread_ok=yes])

        LIBS="$save_LIBS"
        CFLAGS="$save_CFLAGS"

        AC_MSG_RESULT($acx_pthread_ok)
        if test "x$acx_pthread_ok" = xyes; then
                break;
        fi

        PTHREAD_LIBS=""
        PTHREAD_CFLAGS=""
done
fi

# Various other checks:
if test "x$acx_pthread_ok" = xyes; then
        save_LIBS="$LIBS"
        LIBS="$PTHREAD_LIBS $LIBS"
        save_CFLAGS="$CFLAGS"
        CFLAGS="$CFLAGS $PTHREAD_CFLAGS"

        # Detect AIX lossage: threads are created detached by default
        # and the JOINABLE attribute has a nonstandard name (UNDETACHED).
        AC_MSG_CHECKING([for joinable pthread attribute])
        AC_TRY_LINK([#include <pthread.h>],
                    [int attr=PTHREAD_CREATE_JOINABLE;],
                    ok=PTHREAD_CREATE_JOINABLE, ok=unknown)
        if test x"$ok" = xunknown; then
                AC_TRY_LINK([#include <pthread.h>],
                            [int attr=PTHREAD_CREATE_UNDETACHED;],
                            ok=PTHREAD_CREATE_UNDETACHED, ok=unknown)
        fi
        if test x"$ok" != xPTHREAD_CREATE_JOINABLE; then
                AC_DEFINE(PTHREAD_CREATE_JOINABLE, $ok,
                          [Define to the necessary symbol if this constant
                           uses a non-standard name on your system.])
        fi
        AC_MSG_RESULT(${ok})
        if test x"$ok" = xunknown; then
                AC_MSG_WARN([we do not know how to create joinable pthreads])
        fi

        AC_MSG_CHECKING([if more special flags are required for pthreads])
        flag=no
        case "${host_cpu}-${host_os}" in
                *-aix* | *-freebsd*)     flag="-D_THREAD_SAFE";;
                *solaris* | *-osf* | *-hpux*) flag="-D_REENTRANT";;
        esac
        AC_MSG_RESULT(${flag})
        if test "x$flag" != xno; then
                PTHREAD_CFLAGS="$flag $PTHREAD_CFLAGS"
        fi

        LIBS="$save_LIBS"
        CFLAGS="$save_CFLAGS"

        # More AIX lossage: must compile with cc_r
        AC_CHECK_PROG(PTHREAD_CC, cc_r, cc_r, ${CC})
else
        PTHREAD_CC="$CC"
fi

AC_SUBST(PTHREAD_LIBS)
AC_SUBST(PTHREAD_CFLAGS)
AC_SUBST(PTHREAD_CC)

# Finally, execute ACTION-IF-FOUND/ACTION-IF-NOT-FOUND:
if test x"$acx_pthread_ok" = xyes; then
        ifelse([$1],,AC_DEFINE(HAVE_PTHREAD,1,[Define if you have POSIX threads libraries and header files.]),[$1])
        :
else
        acx_pthread_ok=no
        $2
fi
AC_LANG_RESTORE
])dnl ACX_PTHREAD

dnl ACX_C99_INT_TYPE([symbol], [signed/unsigned], [size in bytes], [out file])
AC_DEFUN([ACX_C99_INT_TYPE], [
    if test "${ac_cv_sizeof_long_long}" = "$3"; then
        echo "typedef $2 long long $1;" >> $4
    elif test "${ac_cv_sizeof___int64}" = "$3"; then
        echo "typedef $2 __int64 $1;" >> $4
    elif test "${ac_cv_sizeof_long}" = "$3"; then
        echo "typedef $2 long $1;" >> $4
    elif test "${ac_cv_sizeof_int}" = "$3"; then
        echo "typedef $2 int $1;" >> $4
    elif test "${ac_cv_sizeof_short}" = "$3"; then
        echo "typedef $2 short $1;" >> $4
    else
        AC_MSG_ERROR([Unable to determine an appropriate type for $1!])
    fi
])

AC_DEFUN([ACX_C99_TYPES], [
    AC_REQUIRE([AC_CANONICAL_HOST])
    AC_LANG_PUSH(C)

    echo "/* This file is automatically generated by configure.  DO NOT EDIT! */" > safestr-c99.h
    echo "#ifndef SAFESTR_C99_H" >> safestr-c99.h
    echo "#define SAFESTR_C99_H" >> safestr-c99.h
    echo "" >> safestr-c99.h

    acx_c99_headers=no
    if test "${ac_cv_header_inttypes_h}" = "yes"; then
        echo '#include <inttypes.h>' >> safestr-c99.h
        acx_c99_headers=yes
    fi
    if test "${ac_cv_header_stdint_h}" = "yes"; then
        echo '#include <stdint.h>' >> safestr-c99.h
        acx_c99_headers=yes
    fi
    if test "${acx_c99_headers}" = "yes"; then
        echo "" >> safestr-c99.h
    fi

    acx_c99_typedefs=no
    if test "${ac_cv_type_int32_t}" != "yes"; then
        ACX_C99_INT_TYPE([int32_t], signed, 4, [safestr-c99.h])
        acx_c99_typedefs=yes
    fi
    if test "${ac_cv_type_u_int32_t}" != "yes"; then
        ACX_C99_INT_TYPE([u_int32_t], unsigned, 4, [safestr-c99.h])
        acx_c99_typedefs=yes
    fi
    if test "${ac_cv_type_int64_t}" != "yes"; then
        ACX_C99_INT_TYPE([int64_t], signed, 8, [safestr-c99.h])
        acx_c99_typedefs=yes
    fi
    if test "${ac_cv_type_u_int64_t}" != "yes"; then
        ACX_C99_INT_TYPE([u_int64_t], unsigned, 8, [safestr-c99.h])
        acx_c99_typedefs=yes
    fi

    if test "${ac_cv_type_intmax_t}" != "yes" -o "${ac_cv_type_uintmax_t}" != "yes"; then
        acx_c99_intmax=0
        acx_c99_intmax_t="none"
        acx_c99_typedefs=yes
        if test "${ac_cv_sizeof_short}" -gt "${acx_c99_intmax}"; then
            acx_c99_intmax="${ac_cv_sizeof_short}"
            acx_c99_intmax_t="short"
        fi
        if test "${ac_cv_sizeof_int}" -gt "${acx_c99_intmax}"; then
            acx_c99_intmax="${ac_cv_sizeof_int}"
            acx_c99_intmax_t="int"
        fi
        if test "${ac_cv_sizeof_long}" -gt "${acx_c99_intmax}"; then
            acx_c99_intmax="${ac_cv_sizeof_long}"
            acx_c99_intmax_t="long"
        fi
        if test "${ac_cv_sizeof___int64}" -gt "${acx_c99_intmax}"; then
            acx_c99_intmax="${ac_cv_sizeof___int64}"
            acx_c99_intmax_t="__int64"
        fi
        if test "${ac_cv_sizeof_long_long}" -gt "${acx_c99_intmax}"; then
            acx_c99_intmax="${ac_cv_sizeof_long_long}"
            acx_c99_intmax_t="long long"
        fi
        if test "${acx_c99_intmax}" = "0"; then
            AC_MSG_ERROR([Unable to determine maximum integer size!])
        fi
        if test "${ac_cv_type_intmax_t}" != "yes"; then
            echo "typedef signed ${acx_c99_intmax_t} intmax_t;" >> safestr-c99.h
        fi
        if test "${ac_cv_type_uintmax_t}" != "yes"; then
            echo "typedef unsigned ${acx_c99_intmax_t} uintmax_t;" >> safestr-c99.h
        fi
    fi

    if test "${ac_cv_type_ptrdiff_t}" != "yes"; then
        ACX_C99_INT_TYPE([ptrdiff_t], signed, [${ac_cv_sizeof_void_p}], [safestr-c99.h])
        acx_c99_typedefs=yes
    fi
    if test "${ac_cv_type_uptrdiff_t}" != "yes"; then
        ACX_C99_INT_TYPE([uptrdiff_t], unsigned, [${ac_cv_sizeof_void_p}], [safestr-c99.h])
        acx_c99_typedefs=yes
    fi

    if test "${ac_cv_type_size_t}" != "yes"; then
        echo "typedef unsigned int size_t;" >> safestr-c99.h
        acx_c99_typedefs=yes
    fi
    if test "${ac_cv_type_ssize_t}" != "yes"; then
        if test "${ac_cv_sizeof_size_t}" -gt "0"; then
            ACX_C99_INT_TYPE([ssize_t], signed, [${ac_cv_sizeof_size_t}], [safestr-c99.h])
        else
            echo "typedef signed int size_t;" >> safestr-c99.h
        fi
        acx_c99_typedefs=yes
    fi

    if test "${acx_c99_typedefs}" = "yes"; then
        echo "" >> safestr-c99.h
    fi
    echo "#endif  /* SAFESTR_C99_H */" >> safestr-c99.h

    AC_LANG_POP
])
