#include "config.h"

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <iostream>
#include <string>
#include <fstream>
#include <iomanip>
#include <string>
#include <cassert>
#include <algorithm>

#include "asserts.h"
#include "error.h"
#include "fs.h"
#include "rconfig.h"
#include "logger.h"
#include "vaulter.h"
#include "archiver.h"
#include "cataloger.h"
#include "reporter.h"
#include "test-fs-cwd.h"

#define ERR_OUT(e) std::cerr << e
// #define ERR_OUT(e)

bool make_dir(const std::string& path)
{
	bool thrown;
	bool value;

	thrown = false;
	try {
		mk_dir(path);
	}
	catch(...) {
		thrown = true;
	}

	value = (thrown == false);

	return(value);
}

void setup(void)
{
	std::ofstream out;

	assert(make_dir("./test-rvm.dir"));
	assert(make_dir("./test-rvm.dir/vault"));
	assert(make_dir("./test-rvm.dir/log.dir"));
	assert(make_dir("./test-rvm.dir/catalog.dir"));

	out.open("./test-rvm.dir/file.conf");
	assert(out.is_open());
	out << "link-catalog-dir ./test-rvm.dir/catalog.dir" << std::endl;
	out << "log-dir ./test-rvm.dir/log.dir" << std::endl;
	out << "logging-level rsync" << std::endl;
	out << "rsync-local-path " << LOCAL_RSYNC << std::endl;
	out << "rsync-parallel 1" << std::endl;
	out << "timestamp-resolution second" << std::endl;
	out << "vault ./test-rvm.dir/vault" << std::endl;
	out << "vault-overflow-behavior quit" << std::endl;
	out << "vault-overflow-blocks 0" << std::endl;
	out << "vault-overflow-inodes 0" << std::endl;
	out << "vault-selection-behavior round-robin" << std::endl;
	out << "<job>" << std::endl;
	out << "	jobname test-job" << std::endl;
	out << "	archive-path jobname/permutation" << std::endl;
	out << "	path " << check_cwd << "/" << std::endl;
	out << "	rsync-connection-type local" << std::endl;
	out << "	rsync-hardlink true" << std::endl;
	out << "	rsync-multi-hardlink true" << std::endl;
	out << "	<rsync-options>" << std::endl;
	out << "		-a -v --progress --stats" << std::endl;
	out << "		--exclude '/.svn/'" << std::endl;
	out << "		--exclude '/.deps/'" << std::endl;
	out << "		--exclude '/autom4te.cache/'" << std::endl;
	out << "		--exclude '/test-rvm.dir/'" << std::endl;
	out << "	</rsync-options>" << std::endl;
	out << "	rsync-retry-count 3" << std::endl;
	out << "</job>" << std::endl;
	out.close();
}

void cleanup(void)
{
	assert(system("rm -fr ./test-rvm.dir") == 0);
}

bool test_inodes(
	const std::string& path1,
	const std::string& path2,
	const std::string& file
	)
{
	estring p1, p2;
	filestatus f1, f2;
	bool r;

	p1 = path1;
	p1 += "/";
	p1 += file;
	p1 = reform_path(p1);
	f1.path(p1);

	p2 = path2;
	p2 += "/";
	p2 += file;
	p2 = reform_path(p2);
	f2.path(p2);

	r = (f1.inode() == f2.inode());

	return(r);
}

void test_rvm(void)
{
	std::string local_rsync;
	estring adir[4];
	int adirc = 0;

	local_rsync = LOCAL_RSYNC;
	if (local_rsync.size() == 0) {
		char const * text[] = {
			"",
			"NOTICE: Skipping this test",
			"",
			"This test cannot be run because no local rsync binary was found during",
			"configuration.  This also means that RVM will be compiled with no",
			"default value for the rsync-local-path command.",
			"",
			"To change this, run the configure script with --with-rsync=<path>,",
			"where <path> is the absolute path to the rsync binary.",
			"",
			0
		};
		int c;

		for (c = 0; text[c] != 0; ++c) {
			std::cerr << text[c] << std::endl;
		}
		return;
	}

	{
		char const * argv[256] = { 0 };
		int argc = 0;
		bool thrown = false;
		subdirectory subdir;
		estring dir;

		argv[argc++] = "<program>";
		argv[argc++] = "--archive";

		cataloger.clear();
		archiver.clear();
		reporter.clear();
		vaulter.clear();
		logger.clear();
		config.clear();

		config.default_file("./test-rvm.dir/file.conf");
		try {
			timer t;

			t.start();

			config.init(argc, argv);
			logger.init();
			vaulter.init();
			reporter.init();
			archiver.init();
			archiver.archive();
			cataloger.init();
			cataloger.catalog();

			t.stop();
			reporter.set_total_time(t);

			// reporter.print_report();
			reporter.file_report();
			logger.clear();

			// std::cerr << "vaulter.vault() == " << vaulter.vault() << std::endl;
			assert(vaulter.vault() == "./test-rvm.dir/vault");

			dir = vaulter.vault();
			dir += "/";
			dir += config.timestamp().str();
			dir += "/";
			dir += config.jobs()[0].generate_archive_path(check_cwd);
			adir[adirc] = dir;
			// std::cerr << "adir[" << adirc << "] = " << adir[adirc] << std::endl;
			adirc++;

			// std::cout << "dir = " << dir << std::endl;

			assert(
				exists(
					config.log_dir()
					+ static_cast<std::string>("/")
					+ config.timestamp().str()
					+ static_cast<std::string>(".log")
					)
				);
			assert(
				exists(
					config.log_dir()
					+ static_cast<std::string>("/")
					+ config.timestamp().str()
					+ static_cast<std::string>(".report")
					)
				);

			subdir.path(dir);
			assert(find(subdir.begin(), subdir.end(), "AUTHORS") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "COPYING") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "INSTALL") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "Makefile") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "Makefile.in") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "README") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "archiver.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "archiver.h") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "config.h") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "config.h.in") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "configure") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "error.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "error.h") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "estring.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "estring.h") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "exec.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "exec.h") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "fs.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "fs.h") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "help.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "help.h") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "logger.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "logger.h") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "make-test-fs-cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "rconfig.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "rconfig.h") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "rmath.h") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "rvm.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "rvm.1") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "rvm.1.in") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "test-rvm-001.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "test-error.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "test-estring.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "test-exec.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "test-fs-cwd.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "test-fs.cc") != subdir.end());
			// assert(find(subdir.begin(), subdir.end(), "test-job.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "test-logger.cc") != subdir.end());
			// assert(find(subdir.begin(), subdir.end(), "test-rconfig.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "test-rmath.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "test-strfmt.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "test-timer.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "test-tstamp.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "test-types.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "timer.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "timer.h") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "tstamp.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "tstamp.h") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "types.h") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "vaulter.cc") != subdir.end());
			assert(find(subdir.begin(), subdir.end(), "vaulter.h") != subdir.end());
		}
		catch(error e) {
			ERR_OUT(e);
			thrown = true;
		}
		catch(...) {
			ERR_OUT(err_unknown);
			assert(0);
		}
		assert(!thrown);
	}

	sleep(2);

	{
		char const * argv[256] = { 0 };
		int argc = 0;
		bool thrown = false;
		subdirectory subdir;
		estring dir;

		argv[argc++] = "<program>";
		argv[argc++] = "--archive";

		cataloger.clear();
		archiver.clear();
		reporter.clear();
		vaulter.clear();
		logger.clear();
		config.clear();

		config.default_file("./test-rvm.dir/file.conf");
		try {
			timer t;

			t.start();

			config.init(argc, argv);
			logger.init();
			vaulter.init();
			reporter.init();
			archiver.init();
			archiver.archive();
			estring path;
			filestatus fstat1, fstat2;
			cataloger.init();
			cataloger.catalog();

			t.stop();
			reporter.set_total_time(t);

			// reporter.print_report();
			reporter.file_report();
			logger.clear();

			// std::cerr << "vaulter.vault() == " << vaulter.vault() << std::endl;
			assert(vaulter.vault() == "./test-rvm.dir/vault");

			dir = vaulter.vault();
			dir += "/";
			dir += config.timestamp().str();
			dir += "/";
			dir += config.jobs()[0].generate_archive_path(check_cwd);
			adir[adirc] = dir;
			// std::cerr << "adir[" << adirc << "] = " << adir[adirc] << std::endl;
			adirc++;

			// std::cout << "dir = " << dir << std::endl;

			subdir.path(dir);
			if (subdir.size() == 0) {
				std::string filename;
				std::string line;
				std::ifstream in;
				bool found_usage_error = false;
				char line_cstr[2048] = { 0 };
				int c;

				filename = "./test-rvm.dir/log.dir/";
				filename += config.timestamp().str();
				filename += ".log";
				in.open(filename.c_str());
				assert(in.is_open());
				while (in.getline(line_cstr,2048)) {
					line = line_cstr;
					if (line.find("Rsync exit code: [1]") != std::string::npos)
						found_usage_error = true;
					for (c = 0; c < 2048; line_cstr[c++] = 0);
				}
				in.close();

				if (found_usage_error) {
					std::cerr 
						<< "*** NOTICE: One test has failed, the test that uses rsync with"
						<< std::endl
						<< "            the --hard-links and --link-dest command line"
						<< std::endl
						<< "            options.  Rsync reported a syntax or usage error,"
						<< std::endl
						<< "            which may indicate that this version of rsync is"
						<< std::endl
						<< "            old enough to not have these options."
						<< std::endl
						<< std::endl
						<< "            In order to be able to hard-link files that have"
						<< std::endl
						<< "            not changed from one archive to the next, you"
						<< std::endl
						<< "            will have to upgrade rsync."
						<< std::endl
						;
				}
				assert(found_usage_error);
			}
			else {
				assert(find(subdir.begin(), subdir.end(), "AUTHORS") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "COPYING") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "INSTALL") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "Makefile") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "Makefile.in") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "README") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "archiver.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "archiver.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "config.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "config.h.in") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "configure") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "error.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "error.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "estring.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "estring.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "exec.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "exec.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "fs.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "fs.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "help.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "help.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "logger.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "logger.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "make-test-fs-cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "rconfig.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "rconfig.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "rmath.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "rvm.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "rvm.1") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "rvm.1.in") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-rvm-001.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-error.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-estring.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-exec.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-fs-cwd.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-fs.cc") != subdir.end());
				// assert(find(subdir.begin(), subdir.end(), "test-job.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-logger.cc") != subdir.end());
				// assert(find(subdir.begin(), subdir.end(), "test-rconfig.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-rmath.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-strfmt.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-timer.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-tstamp.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-types.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "timer.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "timer.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "tstamp.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "tstamp.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "types.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "vaulter.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "vaulter.h") != subdir.end());

				assert(test_inodes(adir[0], adir[1], "AUTHORS"));
			}

			// TODO: Check output
			assert(
				exists(
					static_cast<std::string>("./test-rvm.dir/catalog.dir/")
					+ config.timestamp().str()
				)
			);
		}
		catch(error e) {
			ERR_OUT(e);
			thrown = true;
		}
		catch(...) {
			ERR_OUT(err_unknown);
			assert(0);
		}
		assert(!thrown);
	}

	sleep(2);

	{
		char const * argv[256] = { 0 };
		int argc = 0;
		bool thrown = false;
		subdirectory subdir;
		estring dir;

		argv[argc++] = "<program>";
		argv[argc++] = "--archive";

		cataloger.clear();
		archiver.clear();
		reporter.clear();
		vaulter.clear();
		logger.clear();
		config.clear();

		estring file;
		file = adir[1].c_str();
		file += "/";
		file += "COPYING";
		unlink(file.c_str());

		config.default_file("./test-rvm.dir/file.conf");
		try {
			timer t;

			t.start();

			config.init(argc, argv);
			logger.init();
			vaulter.init();
			reporter.init();
			archiver.init();
			archiver.archive();
			estring path;
			filestatus fstat1, fstat2;
			cataloger.init();
			cataloger.catalog();

			t.stop();
			reporter.set_total_time(t);

			// reporter.print_report();
			reporter.file_report();
			logger.clear();

			// std::cerr << "vaulter.vault() == " << vaulter.vault() << std::endl;
			assert(vaulter.vault() == "./test-rvm.dir/vault");

			dir = vaulter.vault();
			dir += "/";
			dir += config.timestamp().str();
			dir += "/";
			dir += config.jobs()[0].generate_archive_path(check_cwd);
			adir[adirc] = dir;
			// std::cerr << "adir[" << adirc << "] = " << adir[adirc] << std::endl;
			adirc++;

			// std::cout << "dir = " << dir << std::endl;

			subdir.path(dir);
			if (subdir.size() == 0) {
				std::string filename;
				std::string line;
				std::ifstream in;
				bool found_usage_error = false;
				char line_cstr[2048] = { 0 };
				int c;

				filename = "./test-rvm.dir/log.dir/";
				filename += config.timestamp().str();
				filename += ".log";
				in.open(filename.c_str());
				assert(in.is_open());
				while (in.getline(line_cstr,2048)) {
					line = line_cstr;
					if (line.find("Rsync exit code: [1]") != std::string::npos)
						found_usage_error = true;
					for (c = 0; c < 2048; line_cstr[c++] = 0);
				}
				in.close();

				if (found_usage_error) {
					std::cerr 
						<< "*** NOTICE: One test has failed, the test that uses rsync with"
						<< std::endl
						<< "            the --hard-links and --link-dest command line"
						<< std::endl
						<< "            options.  Rsync reported a syntax or usage error,"
						<< std::endl
						<< "            which may indicate that this version of rsync is"
						<< std::endl
						<< "            old enough to not have these options."
						<< std::endl
						<< std::endl
						<< "            In order to be able to hard-link files that have"
						<< std::endl
						<< "            not changed from one archive to the next, you"
						<< std::endl
						<< "            will have to upgrade rsync."
						<< std::endl
						;
				}
				assert(found_usage_error);
			}
			else {
				assert(find(subdir.begin(), subdir.end(), "AUTHORS") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "COPYING") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "INSTALL") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "Makefile") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "Makefile.in") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "README") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "archiver.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "archiver.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "config.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "config.h.in") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "configure") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "error.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "error.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "estring.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "estring.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "exec.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "exec.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "fs.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "fs.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "help.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "help.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "logger.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "logger.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "make-test-fs-cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "rconfig.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "rconfig.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "rmath.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "rvm.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "rvm.1") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "rvm.1.in") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-rvm-001.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-error.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-estring.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-exec.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-fs-cwd.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-fs.cc") != subdir.end());
				// assert(find(subdir.begin(), subdir.end(), "test-job.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-logger.cc") != subdir.end());
				// assert(find(subdir.begin(), subdir.end(), "test-rconfig.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-rmath.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-strfmt.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-timer.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-tstamp.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "test-types.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "timer.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "timer.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "tstamp.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "tstamp.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "types.h") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "vaulter.cc") != subdir.end());
				assert(find(subdir.begin(), subdir.end(), "vaulter.h") != subdir.end());

				assert(test_inodes(adir[0], adir[1], "AUTHORS"));
				assert(test_inodes(adir[0], adir[2], "AUTHORS"));
				assert(test_inodes(adir[0], adir[2], "COPYING"));
			}

			// TODO: Check output
			assert(
				exists(
					static_cast<std::string>("./test-rvm.dir/catalog.dir/")
					+ config.timestamp().str()
				)
			);
		}
		catch(error e) {
			ERR_OUT(e);
			thrown = true;
		}
		catch(...) {
			ERR_OUT(err_unknown);
			assert(0);
		}
		assert(!thrown);
	}
}

int main(int argc, char const * argv[])
{
	cleanup();
	setup();
	try {
		test_rvm();
	}
	catch(error e) {
		std::cerr << e;
		assert(0);
	}
	catch(...) {
		std::cerr << err_unknown;
		assert(0);
	}
	cleanup();
	return(0);
}

