#include "config.h"

#include <iostream>
#include <string>
#include <vector>

#include "asserts.h"
#include "error.h"
#include "estring.h"
#include "fs.h"
#include "rconfig.h"
#include "timer.h"
#include "logger.h"
#include "vaulter.h"

#include "cataloger.h"

//-----------------------------------------------------------------------------

/** C'tor */
catalog_manager::catalog_manager()
{
	if (this != &cataloger)
		throw(INTERNAL_ERROR(0,"Attempt to allocate multiple catalog managers"));

	clear();
}

/** Reset the catalog manager */
void catalog_manager::clear(void)
{
	m_initialized = false;
}

/** Initialize the catalog manager */
void catalog_manager::init(void)
{
	m_initialized = true;
}

/** Return the initialized status */
const bool catalog_manager::initialized(void) const
{
	return(m_initialized);
}

/** Erase the contents of the catalog */
void catalog_manager::erase(void)
{
	subdirectory subdir;
	subdirectory::const_iterator sdi;
	std::string path;
	estring lstr;

	if (!initialized())
		throw(INTERNAL_ERROR(0,"Catalog manager not initialized"));

	if (config.link_catalog_dir().size() == 0)
		return;
	subdir.path(config.link_catalog_dir());
	if (subdir.size() == 0)
		return;
	lstr = "Cataloger - Erasing old link catalog...\n";
	logger.write(lstr);
	for (sdi = subdir.begin(); sdi != subdir.end(); ++sdi) {
		TRY_nomem(path = config.link_catalog_dir());
		TRY_nomem(path += "/");
		TRY_nomem(path += *sdi);
		TRY_nomem(path = reform_path(path));
		rm_recursive(path);
	}
	lstr = "Cataloger - Finished erasing link catalog\n";
	logger.write(lstr);
}

/** Create or update the catalog */
void catalog_manager::catalog(void)
{
	estring cpath, vpath;
	subdirectory subdir;
	subdirectory::const_iterator si;
	configuration_manager::vaults_type::const_iterator vi;
	estring lstr;

	if (!initialized())
		throw(INTERNAL_ERROR(0,"Catalog manager not initialized"));

	if (config.link_catalog_dir().size() == 0)
		return;

	erase();
	lstr = "Cataloger - Building link catalog...\n";
	logger.write(lstr);
	for (
		vi = config.vaults().begin();
		vi != config.vaults().end();
		++vi)
	{
		subdir.path(*vi);
		if (subdir.size() == 0)
			continue;
		for (si = subdir.begin(); si != subdir.end(); ++si) {
			if (!is_timestamp(*si))
				continue;

			vpath = *vi;
			vpath += "/";
			vpath += *si;
			vpath = reform_path(vpath);

			cpath = config.link_catalog_dir();
			cpath += "/";
			cpath += *si;
			cpath = reform_path(cpath);

			lstr = "Linking: ";
			lstr += vpath;
			lstr += "\n";
			logger.write(lstr);
			lstr = "     To: ";
			lstr += cpath;
			lstr += "\n";
			logger.write(lstr);

			TRY_log(mk_relative_symlink(vpath, cpath),"Could not link");
		}
	}
	lstr = "Cataloger - Finished building link catalog\n";
	logger.write(lstr);
}

//-----------------------------------------------------------------------------

/** The global catalog manager */
catalog_manager cataloger;

