#include "config.h"

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <iostream>
#include <fstream>
#include <cassert>

#include "asserts.h"
#include "error.h"
#include "estring.h"
#include "table.h"

// #define ERR_OUT(s) std::cerr << s
#define ERR_OUT(s)

class check_output
{
public:
	bool open(const std::string& a_filename)
	{
		if (m_in.is_open())
			m_in.close();
		m_in.clear();
		m_in.open(a_filename.c_str());
		if (!m_in.is_open() || !m_in) {
			return(false);
		}
		return(true);
	}

	bool close(void)
	{
		m_in.close();
		return(true);
	}

	bool is_open(void)
	{
		bool value;

		value = m_in.is_open();
		return(value);
	}

	bool get_line(void)
	{
		char buffer[1024] = { 0 };

		if (!m_in.getline(buffer,1024)) {
			return(false);
		}
		try {
			m_str = buffer;
		}
		catch(...) {
			std::cerr 
				<< "*** ERROR: check_output::get_line() - Out of memory?"
				<< std::endl;
			return(false);
		}
		return(true);
	}

	const std::string& str(void) const
	{
		return(m_str);
	}

	bool remove(const std::string& a_filename)
	{
		if (unlink(a_filename.c_str()) != 0) {
			return(false);
		}
		return(true);
	}

private:
	std::string m_str;
	std::ifstream m_in;
};

check_output check;

void test1(void)
{
	table_cell_base tcb;

	assert(tcb.x == table_cell_base::npos);
	assert(tcb.y == table_cell_base::npos);

	table_cell_base tcb2;

	assert(tcb == tcb);
	assert(!(tcb == tcb2));

	std::ofstream out;
	out.open("./test-table.txt");
	assert(out.is_open());
	tcb.write(out,0,10);
	out.close();

	assert(check.open("./test-table.txt"));
	assert(!check.get_line());
	assert(check.close());
	assert(check.remove("./test-table.txt"));
}

void test2(void)
{
	table_cell_estring tce;
	estring estr;

	estr = "Hello World";
	tce.assign(estr);

	assert(tce.height() == 1);
	assert(tce.width() == estr.size());

	table_cell_estring tce2(estr);

	assert(tce2.height() == 1);
	assert(tce2.width() == estr.size());

	table_cell_estring tce3(tce);

	assert(tce3.height() == 1);
	assert(tce3.width() == estr.size());

	std::ofstream out;
	out.open("./test-table.txt");
	assert(out.is_open());
	tce.write(out,0,15);
	out << std::endl;
	tce.write(out,1,15);
	out << std::endl;
	out.close();

	assert(check.open("./test-table.txt"));
	assert(check.get_line() && check.str() == "Hello World    ");
	assert(check.get_line() && check.str() == "               ");
	assert(!check.get_line());
	assert(check.close());
	assert(check.remove("./test-table.txt"));
}

void test3(void)
{
	{
		table t;

		assert(t.ncols() == 0);
		assert(t.nrows() == 0);
		assert(t.width() == 0);
		assert(t.height() == 0);

		std::ofstream out;
		out.open("./test-table.txt");
		assert(out.is_open());
		out << t;
		out.close();

		assert(check.open("./test-table.txt"));
		assert(!check.get_line());
		assert(check.close());
		assert(check.remove("./test-table.txt"));
	}

	{
		table t;

		t.resize(1,1);

		assert(t.ncols() == 1);
		assert(t.nrows() == 1);
		assert(t.col_width(0) == 0);
		assert(t.row_width() == 0);
		assert(t.col_height() == 0);
		assert(t.row_height(0) == 0);
		assert(t.width() == 0);
		assert(t.height() == 0);

		std::ofstream out;
		out.open("./test-table.txt");
		assert(out.is_open());
		out << t;
		out.close();

		assert(check.open("./test-table.txt"));
		assert(!check.get_line());
		assert(check.close());
		assert(check.remove("./test-table.txt"));

		t.assign(0,0,"Hello World");

		assert(t.ncols() == 1);
		assert(t.nrows() == 1);
		assert(t.col_width(0) == 11);
		assert(t.row_width() == 11);
		assert(t.col_height() == 1);
		assert(t.row_height(0) == 1);
		assert(t.width() == 11);
		assert(t.height() == 1);

		out.open("./test-table.txt");
		assert(out.is_open());
		out << t;
		out.close();

		assert(check.open("./test-table.txt"));
		assert(check.get_line() && check.str() == "Hello World");
		assert(!check.get_line());
		assert(check.close());
		assert(check.remove("./test-table.txt"));

		t.resize(2,1);

		assert(t.ncols() == 2);
		assert(t.nrows() == 1);
		assert(t.col_width(0) == 11);
		assert(t.row_width() == 11);
		assert(t.col_height() == 1);
		assert(t.row_height(0) == 1);
		assert(t.width() == 11);
		assert(t.height() == 1);

		out.open("./test-table.txt");
		assert(out.is_open());
		out << t;
		out.close();

		assert(check.open("./test-table.txt"));
		assert(check.get_line() && check.str() == "Hello World");
		assert(!check.get_line());
		assert(check.close());
		assert(check.remove("./test-table.txt"));

		t.assign(0,0,"Hello ");
		t.assign(1,0,"World");

		assert(t.ncols() == 2);
		assert(t.nrows() == 1);
		assert(t.col_width(0) == 6);
		assert(t.col_width(1) == 5);
		assert(t.row_width() == 11);
		assert(t.col_height() == 1);
		assert(t.row_height(0) == 1);
		assert(t.width() == 11);
		assert(t.height() == 1);

		out.open("./test-table.txt");
		assert(out.is_open());
		out << t;
		out.close();

		assert(check.open("./test-table.txt"));
		assert(check.get_line() && check.str() == "Hello World");
		assert(!check.get_line());
		assert(check.close());
		assert(check.remove("./test-table.txt"));

		t.resize(2,2);

		assert(t.ncols() == 2);
		assert(t.nrows() == 2);
		assert(t.col_width(0) == 6);
		assert(t.col_width(1) == 5);
		assert(t.col_height() == 1);
		assert(t.row_height(0) == 1);
		assert(t.row_height(1) == 0);
		assert(t.width() == 11);
		assert(t.height() == 1);

		out.open("./test-table.txt");
		assert(out.is_open());
		out << t;
		out.close();

		assert(check.open("./test-table.txt"));
		assert(check.get_line() && check.str() == "Hello World");
		assert(!check.get_line());
		assert(check.close());
		assert(check.remove("./test-table.txt"));

		t.assign(0,1,"Hi ");
		t.assign(1,1,"There");

		assert(t.ncols() == 2);
		assert(t.nrows() == 2);
		assert(t.col_width(0) == 6);
		assert(t.col_width(1) == 5);
		assert(t.row_width() == 11);
		assert(t.col_height() == 2);
		assert(t.row_height(0) == 1);
		assert(t.row_height(1) == 1);
		assert(t.width() == 11);
		assert(t.height() == 2);

		out.open("./test-table.txt");
		assert(out.is_open());
		out << t;
		out.close();

		assert(check.open("./test-table.txt"));
		assert(check.get_line() && check.str() == "Hello World");
		assert(check.get_line() && check.str() == "Hi    There");
		assert(!check.get_line());
		assert(check.close());
		assert(check.remove("./test-table.txt"));

	}

	{
		table sub_t(2,2);

		sub_t.assign(0,0," A ");
		sub_t.assign(1,0," B ");
		sub_t.assign(1,1," D ");
		
		table t(2,2);

		t.assign(0,0,"Hello ");
		t.assign(1,0,"World");
		t.assign(0,1,"Hi");
		t.assign(1,1,sub_t);

		assert(t.ncols() == 2);
		assert(t.nrows() == 2);
		assert(t.col_width(0) == 6);
		assert(t.col_width(1) == 6);
		assert(t.row_width() == 12);
		assert(t.col_height() == 3);
		assert(t.row_height(0) == 1);
		assert(t.row_height(1) == 2);
		assert(t.width() == 12);
		assert(t.height() == 3);

		std::ofstream out;
		out.open("./test-table.txt");
		assert(out.is_open());
		out << t;
		out.close();

		assert(check.open("./test-table.txt"));
		assert(check.get_line() && check.str() == "Hello World ");
		assert(check.get_line() && check.str() == "Hi     A  B ");
		assert(check.get_line() && check.str() == "          D ");
		assert(!check.get_line());
		assert(check.close());
		assert(check.remove("./test-table.txt"));
	}

	{
		table t;

		assert(t.ncols() == 0);
		assert(t.nrows() == 0);

		t.insert_row(0);

		assert(t.ncols() == 0);
		assert(t.nrows() == 1);

		t.insert_col(0);

		assert(t.ncols() == 1);
		assert(t.nrows() == 1);

		t.assign(0,0,"Hello");

		t.insert_col(t.ncols());

		assert(t.ncols() == 2);
		assert(t.nrows() == 1);

		t.assign(1,0," ");
		
		t.insert_col(t.ncols());

		assert(t.ncols() == 3);
		assert(t.nrows() == 1);

		t.assign(2,0,"World");

		t.insert_row(t.nrows());

		assert(t.ncols() == 3);
		assert(t.nrows() == 2);

		t.assign(0,1,"Hi");
		t.assign(1,1," ");
		t.assign(2,1,"There");

		std::ofstream out;
		out.open("./test-table.txt");
		out << t;
		out.close();

		assert(check.open("./test-table.txt"));
		assert(check.get_line() && check.str() == "Hello World");
		assert(check.get_line() && check.str() == "Hi    There");
		assert(!check.get_line());
		assert(check.close());
		assert(check.remove("./test-table.txt"));
	}
}

void test4(void)
{
	table t, subt1, subt2;
	char const * array1[] = {
		"alces:/",
		"alces:/export/",
		"alces:/export/home/a1/",
		"alces:/export/home/a2/",
		"alces:/export/home/a3/",
		"alces:/export/home/a4/",
		"alces:/export/home/a5/",
		"alces:/export/home/b1/",
		"alces:/export/home/b2/",
		"alces:/export/home/b3/",
		"alces:/export/home/b4/",
		"alces:/export/home/b5/",
		"alces:/export/home/c1/",
		"alces:/export/home/c2/",
		"alces:/export/home/c3/",
		"alces:/export/home/c4/",
		"alces:/export/home/c5/",
		"alces:/export/home/d1/",
		"alces:/export/home/d2/",
		"alces:/export/home/d3/",
		"alces:/export/home/d4/",
		"alces:/export/home/d5/",
		"alces:/var/spool/",
		0
	};
	char const * array2[] = {
		"alces/",
		"alces/export/",
		"alces/export/home/a1/",
		"alces/export/home/a2/",
		"alces/export/home/a3/",
		"alces/export/home/a4/",
		"alces/export/home/a5/",
		"alces/export/home/b1/",
		"alces/export/home/b2/",
		"alces/export/home/b3/",
		"alces/export/home/b4/",
		"alces/export/home/b5/",
		"alces/export/home/c1/",
		"alces/export/home/c2/",
		"alces/export/home/c3/",
		"alces/export/home/c4/",
		"alces/export/home/c5/",
		"alces/export/home/d1/",
		"alces/export/home/d2/",
		"alces/export/home/d3/",
		"alces/export/home/d4/",
		"alces/export/home/d5/",
		"alces/var/spool/",
		0
	};
	int c;
	estring estr;

	for (c = 0; array1[c] != 0; ++c) {
		subt1 << "From:" << " " << array1[c] << " " << "To:" << " " << array2[c]
			<< table_endl;
	}

	estr = "Start Time:";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";

	estr = "2004 Mar 31 21:32:10";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";

	estr = "Total Files:";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";

	estr = "119,743";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";

	estr = "Total Size:";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";
	
	estr = "9.5GB";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << table_endl;



	estr = "Stop Time:";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";

	estr = "2004 Mar 31 23:16:31";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";

	estr = "Xferd Files:";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";

	estr = "1";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";

	estr = "Xferd Size:";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";
	
	estr = "6.0KB";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << table_endl;



	estr = "Duration:";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";

	estr = "01:34:21";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";

	estr = "Percent:";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";

	estr = "0.0%";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";

	estr = "Percent:";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << " ";
	
	estr = "0.0%";
	estr.align(estring::right);
	subt2 << estr;

	subt2 << table_endl;

	t << "Status" << " " << "Stats" << table_endl;

	estr = "";
	estr.fillchar('-');
	t << estr << " " << estr << table_endl;

	t << "Ok" << " " << subt1 << table_endl;
	t << table_skip << " " << table_skip << table_endl;
	t << table_skip << " " << subt2 << table_endl;

	std::ofstream out;
	out.open("./test-table.txt");
	out << t;
	out.close();

	assert(check.open("./test-table.txt"));
	assert(check.get_line() && check.str() == "Status Stats                                                                  ");
	assert(check.get_line() && check.str() == "------ -----------------------------------------------------------------------");
	assert(check.get_line() && check.str() == "Ok     From: alces:/                To: alces/                                ");
	assert(check.get_line() && check.str() == "       From: alces:/export/         To: alces/export/                         ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/a1/ To: alces/export/home/a1/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/a2/ To: alces/export/home/a2/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/a3/ To: alces/export/home/a3/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/a4/ To: alces/export/home/a4/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/a5/ To: alces/export/home/a5/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/b1/ To: alces/export/home/b1/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/b2/ To: alces/export/home/b2/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/b3/ To: alces/export/home/b3/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/b4/ To: alces/export/home/b4/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/b5/ To: alces/export/home/b5/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/c1/ To: alces/export/home/c1/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/c2/ To: alces/export/home/c2/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/c3/ To: alces/export/home/c3/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/c4/ To: alces/export/home/c4/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/c5/ To: alces/export/home/c5/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/d1/ To: alces/export/home/d1/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/d2/ To: alces/export/home/d2/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/d3/ To: alces/export/home/d3/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/d4/ To: alces/export/home/d4/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/export/home/d5/ To: alces/export/home/d5/                 ");
	assert(check.get_line() && check.str() == "       From: alces:/var/spool/      To: alces/var/spool/                      ");
	assert(check.get_line() && check.str() == "                                                                              ");
	assert(check.get_line() && check.str() == "       Start Time: 2004 Mar 31 21:32:10 Total Files: 119,743 Total Size: 9.5GB");
	assert(check.get_line() && check.str() == "        Stop Time: 2004 Mar 31 23:16:31 Xferd Files:       1 Xferd Size: 6.0KB");
	assert(check.get_line() && check.str() == "         Duration:             01:34:21     Percent:    0.0%    Percent:  0.0%");
	assert(!check.get_line());
	assert(check.close());
	assert(check.remove("./test-table.txt"));
}

int main(int argc, char const * argv[])
{
	try {
		test1();
		test2();
		test3();
		test4();
	}
	catch(error e) {
		std::cerr << e;
		assert(0);
	}
	catch(...) {
		std::cerr << err_unknown;
		assert(0);
	}
	return(0);
}

