#include "config.h"

#include <iostream>
#include <fstream>
#include <cstdio>
#include <cassert>

#include "asserts.h"
#include "error.h"
#include "fs.h"

#include "test-fs-cwd.h"

#define TRY_COMMAND(code) \
	thrown = false; \
	try { \
		code; \
	} \
	catch(...) { \
		thrown = true; \
	}

void create_file(const std::string& pathname, uint32 size)
{
	FILE *out = 0;
	char alpha[] = 
		"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
	int alpha_len = strlen(alpha);
	uint32 count;

	out = fopen(pathname.c_str(), "wb");
	if (out == 0) {
		std::string es;

		es = "Could not open file \"";
		es += pathname;
		es += "\"";

		throw(ERROR(0,es));
	}
	for (count = 0; count < size; count++) {
		if (fputc(alpha[count % alpha_len], out) == EOF) {
			std::string es;

			es = "Write error on file \"";
			es += pathname;
			es += "\"";

			throw(ERROR(0,es));
		}
	}
	if (fclose(out) != 0) {
		std::string es;

		es = "Close error on file \"";
		es += pathname;
		es += "\"";

		throw(ERROR(0,es));
	}
}

void cleanup(void)
{
	int r = 0;
	
	r = system("rm -fr ./test-fs.dir");
	assert(r == 0);
}

void test_cwd(void)
{
	if (cwd() != check_cwd) {
		std::cerr << "NOTICE: The following check failed: test_cwd()" << std::endl;
		std::cerr << "ceck_cwd = " << check_cwd << std::endl;
		std::cerr << "   cwd() = " << cwd() << std::endl;
	}
	assert(cwd() == check_cwd);
}

void test_reform_path(void)
{
	assert(reform_path("") == "");
	assert(reform_path("/") == "/");
	assert(reform_path("//") == "/");
	assert(reform_path("//var") == "/var");
	assert(reform_path("/var") == "/var");
	assert(reform_path("/var/rvm") == "/var/rvm");
	assert(reform_path("/var/rvm//") == "/var/rvm/");
	assert(reform_path("//var//rvm/////") == "/var/rvm/");
}

void test_permute_path(void)
{
	assert(permute_path("") == "root");
	assert(permute_path("/") == "root");
	assert(permute_path("//") == "root");
	assert(permute_path("//var") == "var");
	assert(permute_path("/var") == "var");
	assert(permute_path("/var/rvm") == "var-rvm");
	assert(permute_path("/var/rvm//") == "var-rvm");
	assert(permute_path("//var//rvm/////") == "var-rvm");
}

void test_exists(void)
{
	assert(exists("./INSTALL"));
	assert(!exists("./lkjsdflkjsdf.h"));
}

void test_readable(void)
{
	assert(readable("./INSTALL"));
}

void test_writable(void)
{
	assert(writable("./INSTALL"));
}

void test_executable(void)
{
	assert(!exists("./test-fs") || executable("./test-fs"));
	assert(!exists("./test-fs.exe") || executable("./test-fs.exe"));
}

void test_mk_dir(void)
{
	bool thrown;

	assert(!exists("./test-fs.dir"));
	TRY_COMMAND(mk_dir("./test-fs.dir"));
	assert(!thrown);
	assert(exists("./test-fs.dir"));
}

void test_rm_dir(void)
{
	bool thrown;

	assert(exists("./test-fs.dir"));
	TRY_COMMAND(rm_dir("./test-fs.dir"));
	assert(!thrown);
	assert(!exists("./test-fs.dir"));
}

void test_rm_file(void)
{
	bool thrown;

	if (!exists("./test-fs.dir"))
		mk_dir("./test-fs.dir");
	create_file("./test-fs.dir/file", 1024);
	assert(exists("./test-fs.dir/file"));
	TRY_COMMAND(rm_file("./test-fs.dir/file"));
	assert(!thrown);
	assert(!exists("./test-fs.dir/file"));
	TRY_COMMAND(rm_file("./test-fs.dir"));
	assert(thrown);
	TRY_COMMAND(rm_dir("./test-fs.dir"));
	assert(!thrown);
	assert(!exists("./test-fs.dir"));
}

void test_mk_dirhier(void)
{
	bool thrown;

	assert(!exists("./test-fs.dir"));
	TRY_COMMAND(mk_dirhier("./test-fs.dir/etc/rvm"));
	assert(!thrown);
	assert(exists("./test-fs.dir"));
	assert(exists("./test-fs.dir/etc"));
	assert(exists("./test-fs.dir/etc/rvm"));
	TRY_COMMAND(mk_dirhier("./test-fs.dir/var/rvm"));
	assert(!thrown);
	assert(exists("./test-fs.dir"));
	assert(exists("./test-fs.dir/var"));
	assert(exists("./test-fs.dir/var/rvm"));
}

void test_filestatus(void)
{
	bool thrown;
	filestatus f;

	assert(exists("./test-fs.dir"));
	assert(exists("./test-fs.dir/etc"));
	assert(exists("./test-fs.dir/etc/rvm"));
	assert(exists("./test-fs.dir/var"));
	assert(exists("./test-fs.dir/var/rvm"));
	create_file("./test-fs.dir/file1",1024);
	create_file("./test-fs.dir/file2",234);
	create_file("./test-fs.dir/etc/file1",988);
	create_file("./test-fs.dir/etc/rvm/file1",333);
	create_file("./test-fs.dir/var/rvm/file1",4484);

	TRY_COMMAND(f.path("./test-fs.dir"));
	assert(!thrown);
#ifdef S_ISDIR
	assert(f.is_directory());
#endif

	TRY_COMMAND(f.path("./test-fs.dir/file1"));
	assert(!thrown);
#ifdef S_ISREG
	assert(f.is_regular_file());
#endif
	assert(f.size() == 1024);

	TRY_COMMAND(f.path("./test-fs.dir/var/rvm/file1"));
	assert(!thrown);
#ifdef S_ISREG
	assert(f.is_regular_file());
#endif
	assert(f.size() == 4484);
}

void test_mk_symlink(void)
{
	bool thrown;
	filestatus f;

	assert(exists("./test-fs.dir/file2"));
	TRY_COMMAND(mk_symlink("file2", "./test-fs.dir/link-file2"));
	assert(!thrown);
	assert(exists("./test-fs.dir/link-file2"));
	TRY_COMMAND(f.path("./test-fs.dir/link-file2"));
	assert(!thrown);
#ifdef S_ISLNK
	assert(f.is_link());
	assert(f.link() == "file2");
#endif
}

void test_mk_relative_symlink(void)
{
	bool thrown;
	filestatus f;

	assert(exists("./test-fs.dir/etc/rvm"));
	assert(exists("./test-fs.dir/file1"));
	assert(exists("./test-fs.dir/var/rvm"));
	TRY_COMMAND(
		mk_relative_symlink("./test-fs.dir/file1","./test-fs.dir/link-file1"));
	assert(!thrown);
	assert(exists("./test-fs.dir/link-file1"));
	TRY_COMMAND(f.path("./test-fs.dir/link-file1"));
	assert(!thrown);
#ifdef S_ISLNK
	assert(f.is_link());
	assert(f.link() == "file1");
#endif
	TRY_COMMAND(
		mk_relative_symlink("./test-fs.dir/file1",
			"./test-fs.dir/etc/rvm/link-file1"));
	assert(!thrown);
	assert(exists("./test-fs.dir/etc/rvm/link-file1"));
	TRY_COMMAND(f.path("./test-fs.dir/etc/rvm/link-file1"));
	assert(!thrown);
#ifdef S_ISLNK
	assert(f.is_link());
	assert(f.link() == "../../file1");
#endif
	TRY_COMMAND(
		mk_relative_symlink("./test-fs.dir/var/rvm/file1",
			"./test-fs.dir/etc/rvm/link-rvm-file1"));
	assert(!thrown);
	assert(exists("./test-fs.dir/etc/rvm/link-rvm-file1"));
	TRY_COMMAND(f.path("./test-fs.dir/etc/rvm/link-rvm-file1"));
#ifdef S_INLNK
	assert(f.is_link());
	assert(f.link() == "../../var/rvm/file1");
#endif
}

const subdirectory test_subdirectory_sub(void)
{
	subdirectory subdir;

	subdir.path("./test-fs.dir/etc/rvm","l*1");
	return(subdir);
}

void test_subdirectory(void)
{
	bool thrown;
	subdirectory subdir;
	subdirectory subdir2;
	subdirectory subdir3;

	TRY_COMMAND(subdir.path("./test-fs.dir"));
	assert(!thrown);
	assert(subdir.size() == 6);
	assert(subdir[0] == "etc");
	assert(subdir[1] == "file1");
	assert(subdir[2] == "file2");
	assert(subdir[3] == "link-file1");
	assert(subdir[4] == "link-file2");
	assert(subdir[5] == "var");

	TRY_COMMAND(subdir.path("./test-fs.dir","l*"));
	assert(!thrown);
	assert(subdir.size() == 2);
	assert(subdir[0] == "link-file1");
	assert(subdir[1] == "link-file2");

	TRY_COMMAND(subdir.path("./test-fs.dir/etc/rvm","l*1"));
	assert(!thrown);
	assert(subdir.size() == 2);
	assert(subdir[0] == "link-file1");
	assert(subdir[1] == "link-rvm-file1");

	TRY_COMMAND(subdir2 = subdir);
	assert(!thrown);
	assert(subdir2.size() == 2);
	assert(subdir2[0] == "link-file1");
	assert(subdir2[1] == "link-rvm-file1");

	TRY_COMMAND(subdir3 = test_subdirectory_sub());
	assert(!thrown);
	assert(subdir3.size() == 2);
	assert(subdir3[0] == "link-file1");
	assert(subdir3[1] == "link-rvm-file1");
}

void test_directory(void)
{
	bool thrown;
	directory dir;

	TRY_COMMAND(dir.path("./test-fs.dir"));
	assert(!thrown);
	assert(dir.size() == 1);
	assert(dir[0] == "./test-fs.dir");
	
	TRY_COMMAND(dir.path("./test-fs.dir/*"));
	assert(!thrown);
	assert(dir.size() == 6);
	assert(dir[0] == "./test-fs.dir/etc");
	assert(dir[1] == "./test-fs.dir/file1");
	assert(dir[2] == "./test-fs.dir/file2");
	assert(dir[3] == "./test-fs.dir/link-file1");
	assert(dir[4] == "./test-fs.dir/link-file2");
	assert(dir[5] == "./test-fs.dir/var");

	TRY_COMMAND(dir.path("./test-fs.dir/etc/rvm/l*1"));
	assert(!thrown);
	assert(dir.size() == 2);
	assert(dir[0] == "./test-fs.dir/etc/rvm/link-file1");
	assert(dir[1] == "./test-fs.dir/etc/rvm/link-rvm-file1");

	TRY_COMMAND(dir.path("./test-fs.dir/*/*/*"));
	assert(!thrown);
	assert(dir.size() == 4);
	assert(dir[0] == "./test-fs.dir/etc/rvm/file1");
	assert(dir[1] == "./test-fs.dir/etc/rvm/link-file1");
	assert(dir[2] == "./test-fs.dir/etc/rvm/link-rvm-file1");
	assert(dir[3] == "./test-fs.dir/var/rvm/file1");
}

void test_rm_recursive(void)
{
	bool thrown;

	TRY_COMMAND(rm_recursive("./test-fs.dir"));
	assert(!thrown);
	assert(!exists("./test-fs.dir"));
}

void test_mk_relative_path(void)
{
	assert(
		mk_relative_path(
			"./test-fs.dir/dir1a/dir2a/dir3a/file.conf",
			"./test-fs.dir/dir1a/dir2a/dir3a"
			)
		== "file.conf"
	);
	assert(
		mk_relative_path(
			"./test-fs.dir/dir1a/dir2a/dir3a/file.conf",
			"./test-fs.dir/dir1a/dir2a/dir3b"
			)
		== "../dir3a/file.conf"
	);
	assert(
		mk_relative_path(
			"./test-fs.dir/dir1a/dir2a/dir3a/file.conf",
			"./test-fs.dir/dir1a/dir2b/dir3b"
			)
		== "../../dir2a/dir3a/file.conf"
	);
	assert(
		mk_relative_path(
			"./test-fs.dir/dir1a/dir2a/dir3a/file.conf",
			"./test-fs.dir/dir1b/dir2b/dir3b"
			)
		== "../../../dir1a/dir2a/dir3a/file.conf"
	);
	assert(
		mk_relative_path(
			"./test-fs.dir/dir1a/dir2a/dir3a/file.conf",
			"./test-fs.dir.b/dir1b/dir2b/dir3b"
			)
		== "../../../../test-fs.dir/dir1a/dir2a/dir3a/file.conf"
	);
}

void test_filesystem(void)
{
	filesystem fsys;

	fsys.path(cwd());
}

void test_simple_lock(void)
{
	simple_lock sl;
	bool thrown;

	assert(system("rm -fr ./test-fs.dir") == 0);
	mk_dir("./test-fs.dir");
	assert(exists("./test-fs.dir"));

	thrown = false;
	try {
		sl.lockfile("./test-fs.dir/lock");
	}
	catch(...) {
		thrown = true;
	}
	assert(!thrown);

	assert(sl.locked_by() == 0);
	assert(sl.is_locked() == false);

	thrown = false;
	std::ofstream out;
	out.open("./test-fs.dir/lock");
	assert(out.is_open());
	out << getpid()+1234 << std::endl;
	assert((out));
	out.close();
	assert(sl.locked_by() == getpid()+1234);
	assert(sl.is_locked() == false);

	thrown = false;
	try {
		pid_t pid;

		assert((pid = ::fork()) >= 0);
		if (pid > 0) {
			// Parent
			sleep(1);
			assert(sl.locked_by() == pid);
			assert(sl.is_locked());
			assert(system("rm -f ./test-fs.dir/lock") == 0);
			assert(!exists("./test-fs.dir/lock"));
		}
		else {
			// Child
			assert(sl.lock() == true);
			while (exists("./test-fs.dir/lock"));
			exit(0);
		}
	}
	catch(...) {
		thrown = true;
	}
	assert(!thrown);

	assert(sl.locked_by() == 0);
	assert(sl.is_locked() == false);
	assert(!exists("./test-fs.dir/lock"));

	thrown = false;
	try {
		assert(sl.lock() == true);
	}
	catch(...) {
		thrown = true;
	}
	assert(!thrown);

	assert(sl.locked_by() == getpid());
	assert(sl.is_locked() == true);
	assert(exists("./test-fs.dir/lock"));

	thrown = false;
	try {
		sl.unlock();
	}
	catch(...) {
		thrown = true;
	}
	assert(!thrown);

	assert(sl.locked_by() == 0);
	assert(sl.is_locked() == false);
	assert(!exists("./test-fs.dir/lock"));

	assert(system("rm -fr ./test-fs.dir") == 0);
	assert(!exists("./test-fs.dir"));
}

int main(int argc, char const * argv[])
{
	cleanup();
	try {
		// test_cwd();
		test_reform_path();
		test_permute_path();
		test_exists();
		test_readable();
		test_writable();
		test_executable();
		test_mk_dir();
		test_rm_dir();
		test_rm_file();
		test_mk_dirhier();
		test_filestatus();
		test_mk_symlink();
		test_mk_relative_symlink();
		test_subdirectory();
		test_directory();
		test_rm_recursive();
		test_mk_relative_path();
		test_filesystem();
		test_simple_lock();
	}
	catch(error e) {
		std::cerr << e;
		assert(0);
	}
	catch(...) {
		std::cerr << err_unknown;
		assert(0);
	}
	cleanup();
	return(0);
}
