#ifndef __strfmt_h__
#define __strfmt_h__

#include "asserts.h"
#include "types.h"
#include "estring.h"
#include "rmath.h"

/** Given an integral number, return a string with the number formatted as a
	human-readable machine-type size.  (Ex: 10240 becomes "10Kb") */
template<class T>
std::string size_to_string(
	T a_t,
	const std::string& a_base_size_unit = "b",
	uint16 a_width = 8,
	uint16 a_precision = 1, 
	uint16 a_kilo = 1024
	)
{
	enum {
		ones = 0,
		kilo,
		mega,
		giga,
		tera,
		peta,
		exa,
		zetta,
		yotta
	};
	const char* units[] = {
		" ",
		"K",
		"M",
		"G",
		"T",
		"P",
		"E",
		"Z",
		"Y"
	};
	std::string es;
	int resolution = ones;
	std::string str;
	safe_num<T> quotient = a_t;
	safe_num<T> remainder = 0;
	safe_num<T> rmul = 10;
	safe_num<T> tmp;

	TRY_nomem(es = "Could not generate size-string: \"");
	TRY_nomem(es += estring(a_t));
	TRY_nomem(es += "\"");

	for (tmp = 1; tmp < a_precision; ++tmp) {
		rmul *= 10;
	}
	while ((quotient >= a_kilo) && (resolution < yotta)) {
		TRY(
			tmp = quotient;
			tmp *= rmul;
			//
			// *** NOT SURE ABOUT THIS:
			// This involves a conversion from uint16 to T
			// for safe_num<T>(a_kilo)
			//
			remainder = (tmp / safe_num<T>(a_kilo)) % rmul;
			quotient /= a_kilo;
			,es);
			resolution++;
	}

	if (resolution > ones) {
		TRY(
			str += estring(quotient.value()).fmt_str(
				a_width-(a_precision+1)-2,estring::right,' ','x'),
			es);
		TRY_nomem(str += ".");
		TRY(
			str += estring(remainder.value()).fmt_str(a_precision,estring::right,'0','x'),
			es);
	}
	else {
		TRY(
			str += estring(quotient.value()).fmt_str(a_width-2,estring::right,' ','x'),
			es);
	}
	TRY_nomem(str += units[resolution]);
	TRY_nomem(str += a_base_size_unit);

	return(str);
}

/** Given an integral number, return a string with the number formatted in a
 generic, human-readable format.  (Ex: 10240 becomes "10,240") */
template<class T>
std::string num_to_string(T a_t, uint16 a_width = 0)
{
	std::string es;
	std::string tstr;
	estring str;
	std::string::size_type n;
	
	TRY_nomem(tstr = estring(a_t));
	TRY_nomem(es = "Could not generate num-string: \"");
	TRY_nomem(es += tstr);
	TRY_nomem(es += "\"");

	if (tstr.size() == 0) {
		return(tstr);
	}
	
	n = tstr.size();
	while (n > 0) {
		if (str.size() > 0) {
			TRY_nomem(str = "," + str);
		}
		str = tstr[(n--)-1] + str; if (n == 0) break;
		str = tstr[(n--)-1] + str; if (n == 0) break;
		str = tstr[(n--)-1] + str;
	}

	if (a_width == 0) {
		return(str);
	}
	str.width(a_width);
	str.align(estring::right);
	TRY_nomem(tstr = str.fmt_str());

	return(tstr);
}

/** Given an integral number, return a string with the number formated as a
 number of machine-type size per second.  (Ex: 10240 becomes "10Kb/s") */
template<class T>
std::string throughput_to_string(
	T a_t,
	const std::string& a_base_size_unit = "b",
	const std::string& a_base_time_unit = "s",
	uint16 a_width = 10,
	uint16 a_precision = 1,
	uint16 a_kilo = 1024
	)
{
	std::string str;

	TRY_nomem(
		str = size_to_string(a_t, a_base_size_unit, a_width-2, a_precision, a_kilo)
		);
	TRY_nomem(str += "/");
	TRY_nomem(str += a_base_time_unit);

	return(str);
}

/** Given a count complete and a count total, calculate a percentage */
template<typename T>
std::string percent_string(const T& a_complete, const T& a_total)
{
	safe_num<double> percent;
	estring es;
	estring estr;
	std::string str;

	es = "Could not calculate percentage";
	if (a_complete == static_cast<T>(0)) {
		percent = static_cast<T>(0);
	}
	else if (a_total == static_cast<T>(0)) {
		TRY_nomem(str = "??.?%");
		return(str);
	}
	else {
		TRY (
			percent = static_cast<double>(a_complete);
			percent *= 100;
			percent /= static_cast<double>(a_total);
			,es);
	}
	estr.width(5);
	estr.align(estring::right);
	estr.fillchar(' ');
	estr.precision(1);
	estr = percent.value();
	TRY_nomem(str = estr.fmt_str());
	TRY_nomem(str += "%");

	return(str);
}

#endif
