#include "config.h"

#include <iostream>

#include "asserts.h"
#include "error.h"
#include "help.h"
#include "rconfig.h"
#include "archiver.h"
#include "cataloger.h"
#include "reporter.h"
#include "estat.h"

/** \mainpage Rsync Vault Manager

Rvm is an archive manager that uses rsync to create and maintain a list of
archives that span one or more vaults.  A vault is a directory on a logical
partition that holds one or more archives.  An archive is a time-sensitive
subdirectory in a vault that holds the files and directories backed up from
one or more archive jobs.  Files that are identical to files in older archives
may be optionally replaced with hard links to the older version to save space.

*/
int main(int argc, char const * argv[])
{
	try {
		/*
		 * Parse command line arguments, read configuration files, and perform
		 * sanity checks on the configuration settings.
		 */
		config.init(argc, argv);
	}
	catch(error e) {
		std::cerr << e << std::endl;
		exit_manager.assign(exitstat::config_error);
	}
	catch(...) {
		std::cerr << err_unknown << std::endl;
		exit_manager.assign(exitstat::other_error);
	}

	if (config.initialized()) {
		try {
			/*
			 * Perform the requested action.
			 */
			if (config.action() == configuration_manager::action_help) {
				print_help();
			}
			else if (config.action() == configuration_manager::action_version) {
				print_version();
			}
			else if (config.action() == configuration_manager::action_archive) {
				timer t;

				t.start();

				logger.init();
				vaulter.init();
				reporter.init();
				archiver.init();
				cataloger.init();
				reporter.init();
				archiver.archive();
				cataloger.catalog();

				t.stop();

				reporter.set_total_time(t);

				reporter.print_report();
				reporter.file_report();
			}
			else if (config.action() == configuration_manager::action_relink) {
				logger.init();
				cataloger.init();
				cataloger.catalog();
			}
			else if (config.action() == configuration_manager::action_check_config) {
				// return(0);
			}
			else {
				// This should never happen
				throw(INTERNAL_ERROR(0,"Unknown action"));
			}
		}
		catch(error e) {
			std::cerr << e << std::endl;
			logger.write(e.str());
		}
		catch(...) {
			std::cerr << err_unknown << std::endl;
			logger.write(err_unknown.str());
			exit_manager.assign(exitstat::other_error);
		}
	}

	if (exit_manager.value() == exitstat::config_error)
		return(1);
	else if (exit_manager.value() == exitstat::vault_full)
		return(2);
	else if (exit_manager.value() == exitstat::job_failed)
		return(3);
	else if (exit_manager.value() == exitstat::other_error)
		return(4);
	return(0);
}

