#ifndef __archiver_h__
#define __archiver_h__

#include <string>
#include <map>

#include "asserts.h"
#include "estring.h"
#include "timer.h"
#include "exec.h"
#include "rconfig.h"
#include "logger.h"
#include "vaulter.h"
#include "reporter.h"

/** Map exit codes and signal numbers to verbose strings */
class rstat
{
public:
	rstat();
	const std::string& exit(const int a_int) const;
	const std::string& signal(const int a_int) const;

private:
	std::map<int,std::string> m_exit_str;
	std::map<int,std::string> m_signal_str;
	std::string m_unknown_exit;
	std::string m_unknown_signal;
};

extern class rstat rsync_estat_str;

class archiver;

/** Archive the paths associated with a single job */
class job_archiver {
public:
	enum archiving_status {
		status_pending,
		status_processing,
		status_reschedule,
		status_fatal_error,
		status_error,
		status_completed,
		status_done,
	};

	job_archiver(const job * a_job);
	const std::string prefix(void);
	const std::string id(void);
	void clear(void);
	void end(void);
	const archiving_status status(void);
	void start(void);
	void process(void);

	single_job_report report(void) const;

private:
	const job* m_job;
	archiving_status m_status;
	timer m_timer;
	timer m_io_timer;
	bool m_rsync_timeout_flag;
	execute m_exec;
	bool m_success;
	estring m_io_out;
	estring m_io_err;
	pid_t m_child_pid;
	job_path_report m_jpr;
	single_job_report m_jr;
	std::string m_error_msg;

	void mf_do_chores(void);
	void mf_process_report(const std::string& a_str);
	void mf_process_child_io(bool a_finalize);
	void mf_process_rsync_io(
		execute& a_exec, 
		uint16 a_timeout,
		uint64& a_files_total,
		uint64& a_files_xferd,
		uint64& a_size_total,
		uint64& a_size_xferd,
		bool& a_overflow_detected
		);
	void mf_parse_rsync_io(
		std::string a_str, 
		uint64& a_files_total,
		uint64& a_files_xferd,
		uint64& a_size_total,
		uint64& a_size_xferd
		);
	void mf_trim_string(std::string& a_str);
	void mf_parse_report(const std::string& a_str);
};

/** Create (or update an existing) archive in the selected vault */
class archive_manager {
public:
	archive_manager();

	void clear(void);
	void init(void);
	const bool initialized(void) const;

	void archive(void);

	const std::string archive_path(void) const;
	const std::string working_archive_path(void) const;

private:
	std::vector<job_archiver*> m_jobs;
	bool m_initialized;

	void mf_log_status(void);
};

extern archive_manager archiver;

#endif
