!*****************************************************************************************
!>
!  Helper application to generate floating point coefficients from the rational ones.
!  Uses the `mpfun2020` arbitrary precision Fortran package.

    program coefficients

    use mpmodule
    use iso_fortran_env, only: wp => real64, output_unit

    implicit none

    ! integer, parameter :: ndp1 = 500
    ! integer, parameter :: ndp2 = 1000
    integer, parameter :: ndp1 = 1000
    integer, parameter :: ndp2 = 2000
    integer, parameter :: neps1 = -ndp1
    integer, parameter :: neps2 = -ndp2
    integer, parameter :: nq1 = 11
    integer, parameter :: nq2 = 12 * 2 ** nq1
    integer, parameter :: nwds1 = int (ndp1 / mpdpw + 2)
    integer, parameter :: nwds2 = int (ndp2 / mpdpw + 2)

    integer,parameter :: ilen = 105 ! 105 ! for strings
    integer,parameter :: iprec = 85

    interface operator (.div.)
        procedure :: to_string
    end interface
    type (mp_real) :: r1,r2,r
    character(len=1),dimension(ilen) :: s

    if (ndp2 > mpipl) error stop  '("Increase default precision in module MPFUNF.")'

! On the Optimization of Some Nine-Stage Seventh-order Runge-Kutta Method, by M. Tanaka, S. Muramatsu and S. Yamashita,
! Information Processing Society of Japan, Vol. 33, No. 12 (1992) pages 1512-1526.
! see constructTanakaYamashitaEfficient7
    write(*,'(a,a)') &
    'real(wp),parameter :: c1       = ',&
    '36259' .div. '463869'
    write(*,'(a,a)') &
    'real(wp),parameter :: c2       = ',&
    '36259' .div. '309246'
    write(*,'(a,a)') &
    'real(wp),parameter :: c3       = ',&
    '36259' .div. '206164'
    write(*,'(a,a)') &
    'real(wp),parameter :: c4       = ',&
    '76401' .div. '153188'
    write(*,'(a,a)') &
    'real(wp),parameter :: c5       = ',&
    '5164663400901569152' .div. '6688924124988083687'
    write(*,'(a,a)') &
    'real(wp),parameter :: c6       = ',&
    '3486' .div. '3517'
    write(*,'(a,a)') &
    'real(wp),parameter :: c7       = ',&
    '44151' .div. '44173'
    write(*,'(a,a)') &
    'real(wp),parameter :: a21      = ',&
    '36259' .div. '463869'
    write(*,'(a,a)') &
    'real(wp),parameter :: a31      = ',&
    '36259' .div. '1236984'
    write(*,'(a,a)') &
    'real(wp),parameter :: a32      = ',&
    '36259' .div. '412328'
    write(*,'(a,a)') &
    'real(wp),parameter :: a41      = ',&
    '36259' .div. '824656'
    write(*,'(a,a)') &
    'real(wp),parameter :: a43      = ',&
    '108777' .div. '824656'
    write(*,'(a,a)') &
    'real(wp),parameter :: a51      = ',&
    '17751533712975975593187' .div. '24112920357813127230992'
    write(*,'(a,a)') &
    'real(wp),parameter :: a53      = ',&
    '-68331192803887602162951' .div. '24112920357813127230992'
    write(*,'(a,a)') &
    'real(wp),parameter :: a54      = ',&
    '7825717455900471140481' .div. '3014115044726640903874'
    write(*,'(a,a)') &
    'real(wp),parameter :: a61      = ',&
    '-2425518501234340256175806929031336393991001205323654593685210322030691047097621496102266496' .div. &
    '201073929944556265242953373967503382318096046559546854970564286270157897072030532387737241'
    write(*,'(a,a)') &
    'real(wp),parameter :: a63      = ',&
    '126875939114499086848675646731069753055308007638565564293214808307459627250976287910912' .div. &
    '2631823273838775215546306644775636213113650954300949659959480717139276934490785884841'
    write(*,'(a,a)') &
    'real(wp),parameter :: a64      = ',&
    '-18238165682427587123600563411903599919711680222699338744428834349094610403849667513626245575680' .div. &
    '479212348415302218688412787744011607018072627155280851781784515530195350673833210517995172867'
    write(*,'(a,a)') &
    'real(wp),parameter :: a65      = ',&
    '74777425357290689294313120787550134201356775453356604582280658347816977407509825814840320' .div. &
    '27848089034948481594251542168496834020714916243735255636715495143105044359669539013058107'
    write(*,'(a,a)') &
    'real(wp),parameter :: a71      = ',&
    '42210784012026021620512889337138957588173072058924928398799062235' .div. &
    '401168555464694196502745570125544252560955194769351196028554688'
    write(*,'(a,a)') &
    'real(wp),parameter :: a73      = ',&
    '-53537582181289418572806048482253962781541488' .div. &
    '128102133978061070595749084326726258918069'
    write(*,'(a,a)') &
    'real(wp),parameter :: a74      = ',&
    '6373437319382536771018620806214785516542915567996760353063349991182871200304' .div. &
    '19178871740288180724887392022898914045213833131528843480576173243533301485'
    write(*,'(a,a)') &
    'real(wp),parameter :: a75      = ',&
    '-836513109281956728811652083904588515347012294160401579661057793958992' .div. &
    '42189346226535262916910956145917457264775063492307360825161811325023'
    write(*,'(a,a)') &
    'real(wp),parameter :: a76      = ',&
    '10038768138260655813133796321688310283082351149893792474426644227234755871856831386997923013888351' .div. &
    '8279123943002224665888560854425725483235895533066047643118716510648226939201056966728652698557760'
    write(*,'(a,a)') &
    'real(wp),parameter :: a81      = ',&
    '1454976871505621321312348899226731229297985195430097820532172928754404221419640982320963761' .div. &
    '12687546780768188413911065021432924447284583965992535848754097389537051103097048673168256'
    write(*,'(a,a)') &
    'real(wp),parameter :: a83      = ',&
    '-1452249436938195913836212549773886207822959770792' .div. &
    '3187825000852340545619892931005470986913487349'
    write(*,'(a,a)') &
    'real(wp),parameter :: a84      = ',&
    '3193785703967379485471835519262043520640585789136428552340853315619929163223926155626278646291801931779256' .div. &
    '8816743814108800069900425523882492176796603795861854625575345408990649746129323017714575203134405597571'
    write(*,'(a,a)') &
    'real(wp),parameter :: a85      = ',&
    '-314398569508916946629277462588835135011587938712337655816458752800894863689255534896547161759213480' .div. &
    '14507196201560052990013371105817112064769849230048646555812475120383456376679192045076337148816813'
    write(*,'(a,a)') &
    'real(wp),parameter :: a86      = ',&
    '50216335168528704528035587946703411281334109782747532320001552406296886'//&
    '17274518068065484524425884625107263111090060721584249881611265924113' .div. &
    '38074025751923782871010537940160794177282662852784364394726589727558930'//&
    '33722804748992796724254152818232996309281540415603729279478920107136'
    write(*,'(a,a)') &
    'real(wp),parameter :: a87      = ',&
    '-894451839895008223904010765658125850176064186717638397881061173697811879745' .div. &
    '186244934020117483847289332768639722211239803963523669807238114327710091115676'
    write(*,'(a,a)') &
    'real(wp),parameter :: a91      = ',&
    '152015786770038627019906826956584678402371493198250158080970494807155603994339' .div. &
    '1319428594672311986480108760138089275639618425553698631283119461253421932416'
    write(*,'(a,a)') &
    'real(wp),parameter :: a93      = ',&
    '-19887569115365707672105043997835466942389220328' .div. &
    '43451712251082409470704235239058276887205131'
    write(*,'(a,a)') &
    'real(wp),parameter :: a94      = ',&
    '6298831527954572673520838478029639446424615570453903300371170696118960335541193275024146681623960' .div. &
    '17307483347318198085207889427954666589398911583434527253470846782562794571553580157056644256313'
    write(*,'(a,a)') &
    'real(wp),parameter :: a95      = ',&
    '-16267621644623777942279856217571823792451732234540266142050307930357537283432611648312520' .div. &
    '747020211145282116967827947968990352912884924402891384654470989583659988117513448655559'
    write(*,'(a,a)') &
    'real(wp),parameter :: a96      = ',&
    '49192051734527182139396013466558216354763286834791148749699566514605553857954527798'//&
    '3570189994492481977206720065882583432234119698425636137169515' .div. &
    '37124197069544150557837496529029600030926153008302661343833351539919857539481813742'//&
    '2626328203755084156959422247928840402063855870066548878130304'
    write(*,'(a,a)') &
    'real(wp),parameter :: a97      = ',&
    '-17535891839112183607157943692398769696531153719141528498448224128785868799210475' .div. &
    '3881175428498724209649715816699297677268154716152409333146177577349474565697791732'
    write(*,'(a,a)') &
    'real(wp),parameter :: a98      = ',&
    '-31140449219386755112730831706895080247696102690585728771850210691242594436100540310' .div. &
    '58531715707220748822628340615174217489020037018063169180406742622693159384762890406389'
    write(*,'(a,a)') &
    'real(wp),parameter :: a101     = ',&
    '24861126512935523838485032295435745281790804119672244200744512677831357181363' .div. &
    '215828469302253893975010055544246846578750854407392771457340001283636121600'
    write(*,'(a,a)') &
    'real(wp),parameter :: a103     = ',&
    '-76626859319946149305867456329803' .div. '167454524692981091214376557800'
    write(*,'(a,a)') &
    'real(wp),parameter :: a104     = ',&
    '257532657386915224604779230484778835596042580268896440943054087972106955277512448850995064336363' .div. &
    '707777528357579864776572552477247532276956780876653359042572831013312547307465249178438602200'
    write(*,'(a,a)') &
    'real(wp),parameter :: a105     = ',&
    '-103092665221253777021612043042409780416654274677686197534469014507504059634284484983141143' .div. &
    '4735075386204034224907103653335170134874540866215348781137359896717512695961598377363000'
    write(*,'(a,a)') &
    'real(wp),parameter :: a106     = ',&
    '1318945254307068672853031172410281620677291556423152759282406612372948205789241763'//&
    '483098989903852936890735513699395545618802215742952753372919' .div. &
    '9955201919272245091586606595196439163308470176111896180022560239287906654952760229'//&
    '49114110343406997764203331763292012060684160018593393766400'
    write(*,'(a,a)') &
    'real(wp),parameter :: a107     = ',&
    '-2175691361381933486174620849991740173349017185199505364607841' .div. &
    '482872625303278742130341621563226511344221688759361797916327450'
    write(*,'(a,a)') &
    'real(wp),parameter :: a108     = ',&
    '-11327601987184122343710458559595782081610122892585097' .div. &
    '21251874884678431935286330856983429378055579208005268000'
    write(*,'(a,a)') &
    'real(wp),parameter :: b1       = ',&
    '677260699094873524061210073954310211' .div. &
    '13212228177645157882237395248920447488'
    write(*,'(a,a)') &
    'real(wp),parameter :: b4       = ',&
    '5627843976805934592544586970647029617399366281651959837492864' .div. &
    '20448796992082885248862284273169726631726393791864145954479875'
    write(*,'(a,a)') &
    'real(wp),parameter :: b5       = ',&
    '1359735671458057021603668186882234273947181034928034734244224' .div. &
    '4035225037829041960922838374222759264846456609494840689395475'
    write(*,'(a,a)') &
    'real(wp),parameter :: b6       = ',&
    '357576437106384199404292036361576888838336978257996389606464243162619168059875079'//&
    '0399139608006651160426580137040859330533720256407' .div. &
    '188336182699563783260785721707598465094766175943007970622420965545070688380860624'//&
    '12372695473217373611870290738365243380652826304000'
    write(*,'(a,a)') &
    'real(wp),parameter :: b7       = ',&
    '14322850798205614664394883796805489119964080948503151' .div. &
    '1692788382425178679633337406927131793062126418747780'
    write(*,'(a,a)') &
    'real(wp),parameter :: b8       = ',&
    '-16735096417960349589058935251250023138290806176584545269411' .div. &
    '128573843052304513208482301684749747737236254208431871400'
    write(*,'(a,a)') &
    'real(wp),parameter :: b9       = ',&
    '33050288141543277444692395096256051' .div. '271248590133163812341791503489000'
    write(*,'(a,a)') &
    'real(wp),parameter :: bhat1    = ',&
    '962650826879437817605721930727384851' .div. '18874611682350225546053421784172067840'
    write(*,'(a,a)') &
    'real(wp),parameter :: bhat4    = ',&
    '99703652969826806275610089806158069716600653757297413344' .div. &
    '361062893830367886445877712954352019629670588714825566425'
    write(*,'(a,a)') &
    'real(wp),parameter :: bhat5    = ',&
    '17540887447270394964911517553576959050951784592644178144' .div. &
    '52550888012671962193116521992300249584518949945887203425'
    write(*,'(a,a)') &
    'real(wp),parameter :: bhat6    = ',&
    '101855668513773837712956593596043266148479443244790887636953159551191054134940671472736229702711787350735239179' .div. &
    '504322587935299170723833764883183242017770187561624249681119708768991642691172146267201689787026963930014131200'
    write(*,'(a,a)') &
    'real(wp),parameter :: bhat7    = ',&
    '179578338747395946570172802104016572846366090083599' .div. &
    '31203472487100067827342625012481692038011546889360'
    write(*,'(a,a)') &
    'real(wp),parameter :: bhat8    = ',&
    '-500374162579884236288722085953024481890963958534161489781' .div. &
    '5844265593286568782203740985670443078965284282201448700'
    write(*,'(a,a)') &
    'real(wp),parameter :: bhat10   = ',&
    '80' .div. '1'
    write(*,'(a,a)') &
    'real(wp),parameter :: btilde1  = ',&
    '-11350400930890172457349074817136051' .div. &
    '44040760592150526274124650829734824960'
    write(*,'(a,a)') &
    'real(wp),parameter :: btilde4  = ',&
    '18872409140206580874590465524732661000311743892579167244576' .div. &
    '20448796992082885248862284273169726631726393791864145954479875'
    write(*,'(a,a)') &
    'real(wp),parameter :: btilde5  = ',&
    '-4274515681501734477669162831906773100582117137555409033632' .div. &
    '1345075012609680653640946124740919754948818869831613563131825'
    write(*,'(a,a)') &
    'real(wp),parameter :: btilde6  = ',&
    '151982138295746861476872192192436808638630079892291260212523545728864842939'//&
    '698890632387350810275352451114165347163409431707619557' .div. &
    '125557455133042522173857147805065643396510783962005313748280643696713792253'//&
    '90708274915130315478249074580193825576828920435217536000'
    write(*,'(a,a)') &
    'real(wp),parameter :: btilde7  = ',&
    '-6107634561545846083950679043550120057398294081957207' .div. &
    '2257051176566904906177783209236175724082835224997040'
    write(*,'(a,a)') &
    'real(wp),parameter :: btilde8  = ',&
    '1908954947067632130235683120094494845563199696277664164743' .div. &
    '42857947684101504402827433894916582579078751402810623800'
    write(*,'(a,a)') &
    'real(wp),parameter :: btilde9  = ',&
    '-33050288141543277444692395096256051' .div. '271248590133163812341791503489000'
    write(*,'(a,a)') &
    'real(wp),parameter :: btilde10 = ',&
    '80' .div. '1'

    ! ! validate with these:
    ! ! http:/www.peterstone.name/Maplepgs/Maple/nmthds/RKcoeff/Runge_Kutta_schemes/RK8/RKcoeff8d_1.pdf

    ! call write('a[9,1]', mpreal ('7586331039021946882049083502441337664277676907617750536566352.0',nwds2) .div. &
    !                      mpreal('109794461601491217860220353338581031394059220336451160078730445.0',nwds2))
    !     ! 6.90957533591923006485645489845476785610413724468557040961859020532937914180177926127e-2   <---this code
    !     ! .6909575335919230064856454898454767856104137244685570409618590205329379141801779261271e-1  <---from site [SAME!]

    write(*,*) ''
    write(*,*) ''
    write(*,*) ''
    write(*,*) ''
    write(*,*) ''
    write(*,*) ' On Phase-Fitted modified Runge-Kutta Pairs of order 6(5), by Ch. Tsitouras and I. Th. Famelis'
    write(*,*) ''


    ! The coefficients are:


    write(*,'(a,a)') 'real(wp),parameter :: c2    = ', '17' .div. '183'
    write(*,'(a,a)') 'real(wp),parameter :: c3    = ', '12' .div. '83'
    write(*,'(a,a)') 'real(wp),parameter :: c4    = ', '18' .div. '83'
    write(*,'(a,a)') 'real(wp),parameter :: c5    = ', '71' .div. '125'
    write(*,'(a,a)') 'real(wp),parameter :: c6    = ', '42' .div. '59'
    write(*,'(a,a)') 'real(wp),parameter :: c7    = ', '199' .div. '200'
    write(*,'(a,a)') 'real(wp),parameter :: c8    = ', '1' .div. '1'
    write(*,'(a,a)') 'real(wp),parameter :: c9    = ', '1' .div. '1'
    write(*,'(a,a)') 'real(wp),parameter :: a21   = ', '17' .div. '183'
    write(*,'(a,a)') 'real(wp),parameter :: a31   = ', '3756' .div. '117113'
    write(*,'(a,a)') 'real(wp),parameter :: a32   = ', '13176' .div. '117113'
    write(*,'(a,a)') 'real(wp),parameter :: a41   = ', '9' .div. '166'
    write(*,'(a,a)') 'real(wp),parameter :: a42   = ', '0'.div. '1'
    write(*,'(a,a)') 'real(wp),parameter :: a43   = ', '27' .div. '166'
    write(*,'(a,a)') 'real(wp),parameter :: a51   = ', '207751751' .div. '316406250'
    write(*,'(a,a)') 'real(wp),parameter :: a52   = ', '0'.div. '1'
    write(*,'(a,a)') 'real(wp),parameter :: a53   = ', '-526769377' .div. '210937500'
    write(*,'(a,a)') 'real(wp),parameter :: a54   = ', '1524242129' .div. '632812500'
    write(*,'(a,a)') 'real(wp),parameter :: a61   = ', '-4970082682619223281' .div. '2887511529739311186'
    write(*,'(a,a)') 'real(wp),parameter :: a62   = ', '0'.div. '1'
    write(*,'(a,a)') 'real(wp),parameter :: a63   = ', '97919278033879057' .div. '13556392158400522'
    write(*,'(a,a)') 'real(wp),parameter :: a64   = ', '-407131674007930877068' .div. '74078904949579652469'
    write(*,'(a,a)') 'real(wp),parameter :: a65   = ', '1237601855204268750000' .div. '1753200750473385108433'
    write(*,'(a,a)') 'real(wp),parameter :: a71   = ', '176597685527535385020980411' .div. '42773485015591331328000000'
    write(*,'(a,a)') 'real(wp),parameter :: a72   = ', '0'.div. '1'
    write(*,'(a,a)') 'real(wp),parameter :: a73   = ', '-6793162515552646891859' .div. '401628967282547712000'
    write(*,'(a,a)') 'real(wp),parameter :: a74   = ', '12704926019361287204873446554247' .div. '886659402653054716778496000000'
    write(*,'(a,a)') 'real(wp),parameter :: a75   = ', '-50728836334509259632278125' .div. '32657591718008685915971584'
    write(*,'(a,a)') 'real(wp),parameter :: a76   = ', '51536223982796190703' .div. '51293749413888000000'
    write(*,'(a,a)') 'real(wp),parameter :: a81   = ', '299033520572337573523' .div. '66918720793812357519'
    write(*,'(a,a)') 'real(wp),parameter :: a82   = ', '0'.div. '1'
    write(*,'(a,a)') 'real(wp),parameter :: a83   = ', '-16550269823961899' .div. '902146153892364'
    write(*,'(a,a)') 'real(wp),parameter :: a84   = ', '49920346343238033627496282' .div. '3215735869387500624775563'
    write(*,'(a,a)') 'real(wp),parameter :: a85   = ', '-1686432488955761721093750' .div. '978844996793357447730403'
    write(*,'(a,a)') 'real(wp),parameter :: a86   = ', '161901609084039' .div. '149698803705724'
    write(*,'(a,a)') 'real(wp),parameter :: a87   = ', '-305146137600000' .div. '54760341991955873'
    write(*,'(a,a)') 'real(wp),parameter :: a91   = ', '24503' .div. '381483'
    write(*,'(a,a)') 'real(wp),parameter :: a92   = ', '0'.div. '1'
    write(*,'(a,a)') 'real(wp),parameter :: a93   = ', '0'.div. '1'
    write(*,'(a,a)') 'real(wp),parameter :: a94   = ', '1366847103121' .div. '4106349847584'
    write(*,'(a,a)') 'real(wp),parameter :: a95   = ', '20339599609375' .div. '75933913767768'
    write(*,'(a,a)') 'real(wp),parameter :: a96   = ', '35031290651' .div. '194765546144'
    write(*,'(a,a)') 'real(wp),parameter :: a97   = ', '16620160000000' .div. '11001207123543'
    write(*,'(a,a)') 'real(wp),parameter :: a98   = ', '-14933' .div. '11016'
    write(*,'(a,a)') 'real(wp),parameter :: b1    = ', '24503' .div. '381483'
    write(*,'(a,a)') 'real(wp),parameter :: b2    = ', '0'.div. '1'
    write(*,'(a,a)') 'real(wp),parameter :: b3    = ', '0'.div. '1'
    write(*,'(a,a)') 'real(wp),parameter :: b4    = ', '1366847103121' .div. '4106349847584'
    write(*,'(a,a)') 'real(wp),parameter :: b5    = ', '20339599609375' .div. '75933913767768'
    write(*,'(a,a)') 'real(wp),parameter :: b6    = ', '35031290651' .div. '194765546144'
    write(*,'(a,a)') 'real(wp),parameter :: b7    = ', '16620160000000' .div. '11001207123543'
    write(*,'(a,a)') 'real(wp),parameter :: b8    = ', '-14933' .div. '11016'
    write(*,'(a,a)') 'real(wp),parameter :: bhat1 = ', '61010485298317' .div. '979331468960880'
    write(*,'(a,a)') 'real(wp),parameter :: bhat2 = ', '0'.div. '1'
    write(*,'(a,a)') 'real(wp),parameter :: bhat3 = ', '0'.div. '1'
    write(*,'(a,a)') 'real(wp),parameter :: bhat4 = ', '320207313882553286621' .div. '941222813406992395200'
    write(*,'(a,a)') 'real(wp),parameter :: bhat5 = ', '6845867841119140625' .div. '29008216787127405534'
    write(*,'(a,a)') 'real(wp),parameter :: bhat6 = ', '124109197949158875473' .div. '562495660250110816320'
    write(*,'(a,a)') 'real(wp),parameter :: bhat7 = ', '19339714537078400000' .div. '16810691577722216811'
    write(*,'(a,a)') 'real(wp),parameter :: bhat8 = ', '-211029377951' .div. '210416202900'
    write(*,'(a,a)') 'real(wp),parameter :: bhat9 = ', '-1' .div. '150'

    write(*,*) ''
    write(*,*) ''
    write(*,*) ''
    write(*,*) ''
    write(*,*) ''
    write(*,*) 'http://users.uoa.gr/~tsitourasc/t87.m'

    write(*,*) 'T87bvec= '
    write(*,*) '959469921003535' .div. '20735873900418433'
    write(*,*) '0' .div. '1'
    write(*,*) '0' .div. '1'
    write(*,*) '0' .div. '1'
    write(*,*) '0' .div. '1'
    write(*,*) '83661087663817387' .div. '226096222469839182'
    write(*,*) '228743606234324881' .div. '883020026679163794'
    write(*,*) '3544120671195926375' .div. '8063503515187523'
    write(*,*) '164403934540876' .div. '64548125027903185'
    write(*,*) '1872154679941434671' .div. '50440600905843744'
    write(*,*) '-3908844507545666995' .div. '8324248434152054'
    write(*,*) '-402658040159189839' .div. '58491143516062232'
    write(*,*) '16491' .div. '120125'

    write(*,*) 'T87evec= '
    write(*,*) r2str((rr('10839870895445') / rr('185203278486104297')) / rr ('10'))
    write(*,*) r2str((rr('0') / rr('1')) / rr ('10'))
    write(*,*) r2str((rr('0') / rr('1')) / rr ('10'))
    write(*,*) r2str((rr('0') / rr('1')) / rr ('10'))
    write(*,*) r2str((rr('0') / rr('1')) / rr ('10'))
    write(*,*) r2str((rr('70876466420204') / rr('75470597442438275')) / rr ('10'))
    write(*,*) r2str((rr('-16496614726651') / rr('75468957694148719')) / rr ('10'))
    write(*,*) r2str((rr('54859577937538923405') / rr('14355606386435513')) / rr ('10'))
    write(*,*) r2str((rr('17895137500075704819') / rr('2362101251104030')) / rr ('10'))
    write(*,*) r2str((rr('-969063659258770673') / rr('19161965088242370')) / rr ('10'))
    write(*,*) r2str((rr('-39193463899416576680') / rr('3455993185375433')) / rr ('10'))
    write(*,*) r2str((rr('-973424986199410385') / rr('155592419305288032')) / rr ('10'))
    write(*,*) r2str((rr('16491') / rr('120125')) / rr ('10'))

    write(*,*) 'T87cvec='
    write(*,*) '3102' .div. '110773'
    write(*,*) '41448895555141' .div. '353624691619188'
    write(*,*) '41448895555141' .div. '235749794412792'
    write(*,*) '49442' .div. '119883'
    write(*,*) '51187' .div. '105369'
    write(*,*) '61011' .div. '376738'
    write(*,*) '77114' .div. '79499'
    write(*,*) '147909751614626799' .div. '152923788158104127'
    write(*,*) '74279' .div. '78046'
    write(*,*) '72043' .div. '74409'
    write(*,*) '1' .div. '1'
    write(*,*) '1' .div. '1'

    write(*,*) 'T87amat='

    write(*,*) '3102' .div. '110773'
    write(*,*) '-17033458900934993' .div. '132978864382888258'
    write(*,*) '17659313382611255' .div. '71989792689293837'
    write(*,*) '41448895555141' .div. '942999177651168'
    write(*,*) '0' .div. '1'
    write(*,*) '41448895555141' .div. '314333059217056'
    write(*,*) '33544131897542527' .div. '99303639017753176'
    write(*,*) '0' .div. '1'
    write(*,*) '-123806032279621065' .div. '100880451772826828'
    write(*,*) '80881552191452041' .div. '62126727673226683'
    write(*,*) '3901178494518027' .div. '70202052982346435'
    write(*,*) '0' .div. '1'
    write(*,*) '0' .div. '1'
    write(*,*) '12244602153330846' .div. '48744104078022083'
    write(*,*) '11363782051482252' .div. '63479278340035273'
    write(*,*) '7281184019796491' .div. '108906123149933189'
    write(*,*) '0' .div. '1'
    write(*,*) '0' .div. '1'
    write(*,*) '8912953764743186' .div. '75237479424494327'
    write(*,*) '-1193193435755019' .div. '24043824215671157'
    write(*,*) '3001381510813201' .div. '114340525306552991'
    write(*,*) '-297808918551351805' .div. '103302384399153762'
    write(*,*) '0' .div. '1'
    write(*,*) '0' .div. '1'
    write(*,*) '-2387409947307450796' .div. '38235137422988677'
    write(*,*) '-320655295147743895' .div. '172685972706995386'
    write(*,*) '266830735262229145' .div. '73369592821183637'
    write(*,*) '8174527' .div. '126711'
    write(*,*) '-312230898179118543' .div. '111335375555652709'
    write(*,*) '0' .div. '1'
    write(*,*) '0' .div. '1'
    write(*,*) '-5921685522031592717' .div. '97516557935639304'
    write(*,*) '-122516042059134140' .div. '66440638491697461'
    write(*,*) '143089054978597281' .div. '39930960285352934'
    write(*,*) '1966780853930863533' .div. '31340008936176199'
    write(*,*) '27204097600957' .div. '30119714219091834'
    write(*,*) '-497327926559154029' .div. '208366132906665209'
    write(*,*) '0' .div. '1'
    write(*,*) '0' .div. '1'
    write(*,*) '-2070519061247416919' .div. '40105304012179956'
    write(*,*) '-139926368413626755' .div. '79789745208684688'
    write(*,*) '436822604663916242' .div. '133157501626893287'
    write(*,*) '4951999978536596383' .div. '92678477827402881'
    write(*,*) '-1662171172972759' .div. '32043786293542537'
    write(*,*) '320510318790859' .div. '5467452906511140'
    write(*,*) '-267997292446794835' .div. '94625648159795289'
    write(*,*) '0' .div. '1'
    write(*,*) '0' .div. '1'
    write(*,*) '-1326916430444389167' .div. '21635054137957163'
    write(*,*) '-50510473210813287' .div. '27322222661367848'
    write(*,*) '680595213260915461' .div. '188925642391189177'
    write(*,*) '1090597603926315985' .div. '17207867085312708'
    write(*,*) '-818226826952911' .div. '56758278493554744'
    write(*,*) '794276136679319' .div. '44163223221855014'
    write(*,*) '-495594365453263' .div. '165024671142376612'
    write(*,*) '-286074472550848766' .div. '70568381571246193'
    write(*,*) '0' .div. '1'
    write(*,*) '0' .div. '1'
    write(*,*) '-2666282586603439301' .div. '29766446888618900'
    write(*,*) '-394981932622811234' .div. '181671027945865139'
    write(*,*) '354437914440687571' .div. '72293255173230666'
    write(*,*) '1737172167669457231' .div. '18855481952627537'
    write(*,*) '-1908527156826626453' .div. '17978177470082379'
    write(*,*) '14359180611877865064' .div. '20075894067162869'
    write(*,*) '-1863006586402493967' .div. '31715262582627044'
    write(*,*) '-5146117877451253921' .div. '9346764321565133'
    write(*,*) '-2286460617615599450' .div. '148215689608432541'
    write(*,*) '0' .div. '1'
    write(*,*) '0' .div. '1'
    write(*,*) '-21511651826330234931' .div. '52669819756106150'
    write(*,*) '-949790098629780736' .div. '69310896259636617'
    write(*,*) '2488552272190713800' .div. '64326656295428697'
    write(*,*) '14577683994864478388' .div. '35463253730030943'
    write(*,*) '-34626716477448076238' .div. '6579786536866391'
    write(*,*) '267076469802229885930' .div. '7436961774107587'
    write(*,*) '-15666088518007151408' .div. '5323429123670105'
    write(*,*) '-39614246945332388915' .div. '1429199330541022'
    write(*,*) '0' .div. '1'







    ! """
    ! Jim Verner's "Maple" (dverk78
    ! """

    ! write(*,*) '3102' .div. '110773'
    write(*,*) ''
    write(*,*) ''
    write(*,*) ''
    write(*,*) ''
    write(*,*) 'dverk78'

    write(*,*) 'a2  = ', '1' .div. '16'
    write(*,*) 'a3  = ', '112' .div. '1065'
    write(*,*) 'a4  = ', '56' .div. '355'
    write(*,*) 'a5  = ', '39' .div. '100'
    write(*,*) 'a6  = ', '7' .div. '15'
    write(*,*) 'a7  = ', '39' .div. '250'
    write(*,*) 'a8  = ', '24' .div. '25'
    write(*,*) 'a9  = ', '14435868' .div. '16178861'
    write(*,*) 'a10 = ', '11' .div. '12'
    write(*,*) 'a11 = ', '19' .div. '20'
    write(*,*) 'a12 = ', '1' .div. '1'
    write(*,*) 'a13 = ', '1' .div. '1'

    write(*,*) 'b21   = ', '1' .div. '16'
    write(*,*) 'b31   = ', '18928' .div. '1134225'
    write(*,*) 'b32   = ', '100352' .div. '1134225'
    write(*,*) 'b41   = ', '14' .div. '355'
    write(*,*) 'b42   = ', '0' .div. '0'
    write(*,*) 'b43   = ', '42' .div. '355'
    write(*,*) 'b51   = ', '94495479' .div. '250880000'
    write(*,*) 'b52   = ', '0' .div. '0'
    write(*,*) 'b53   = ', '-352806597' .div. '250880000'
    write(*,*) 'b54   = ', '178077159' .div. '125440000'
    write(*,*) 'b61   = ', '12089' .div. '252720'
    write(*,*) 'b62   = ', '0' .div. '0'
    write(*,*) 'b63   = ', '0' .div. '0'
    write(*,*) 'b64   = ', '2505377' .div. '10685520'
    write(*,*) 'b65   = ', '960400' .div. '5209191'
    write(*,*) 'b71   = ', '21400899' .div. '350000000'
    write(*,*) 'b72   = ', '0' .div. '0'
    write(*,*) 'b73   = ', '0' .div. '0'
    write(*,*) 'b74   = ', '3064329829899' .div. '27126050000000'
    write(*,*) 'b75   = ', '-21643947' .div. '592609375'
    write(*,*) 'b76   = ', '124391943' .div. '6756250000'
    write(*,*) 'b81   = ', '-15365458811' .div. '13609565775'
    write(*,*) 'b82   = ', '0' .div. '0'
    write(*,*) 'b83   = ', '0' .div. '0'
    write(*,*) 'b84   = ', '-7' .div. '5'
    write(*,*) 'b85   = ', '-8339128164608' .div. '939060038475'
    write(*,*) 'b86   = ', '341936800488' .div. '47951126225'
    write(*,*) 'b87   = ', '1993321838240' .div. '380523459069'
    write(*,*) 'b91   = ', '-1840911252282376584438157336464708426954728061551' .div. &
                           '2991923615171151921596253813483118262195533733898'
    write(*,*) 'b92   = ', '0' .div. '0'
    write(*,*) 'b93   = ', '0' .div. '0'
    write(*,*) 'b94   = ', '-14764960804048657303638372252908780219281424435' .div. &
                           '2981692102565021975611711269209606363661854518'
    write(*,*) 'b95   = ', '-875325048502130441118613421785266742862694404520560000' .div. &
                           '170212030428894418395571677575961339495435011888324169'
    write(*,*) 'b96   = ', '7632051964154290925661849798370645637589377834346780' .div. &
                           '1734087257418811583049800347581865260479233950396659'
    write(*,*) 'b97   = ', '7519834791971137517048532179652347729899303513750000' .div. &
                           '1045677303502317596597890707812349832637339039997351'
    write(*,*) 'b98   = ', '1366042683489166351293315549358278750' .div. &
                           '144631418224267718165055326464180836641'
    write(*,*) 'b101  = ', '-63077736705254280154824845013881' .div. &
                           '78369357853786633855112190394368'
    write(*,*) 'b102  = ', '0' .div. '0'
    write(*,*) 'b103  = ', '0' .div. '0'
    write(*,*) 'b104  = ', '-31948346510820970247215' .div. &
                           '6956009216960026632192'
    write(*,*) 'b105  = ', '-3378604805394255292453489375' .div. &
                           '517042670569824692230499952'
    write(*,*) 'b106  = ', '1001587844183325981198091450220795' .div. &
                           '184232684207722503701669953872896'
    write(*,*) 'b107  = ', '187023075231349900768014890274453125' .div. &
                           '25224698849808178010752575653374848'
    write(*,*) 'b108  = ', '1908158550070998850625' .div. &
                           '117087067039189929394176'
    write(*,*) 'b109  = ', '-52956818288156668227044990077324877908565' .div. &
                           '2912779959477433986349822224412353951940608'
    write(*,*) 'b111  = ', '-10116106591826909534781157993685116703' .div. &
                           '9562819945036894030442231411871744000'
    write(*,*) 'b112  = ', '0' .div. '0'
    write(*,*) 'b113  = ', '0' .div. '0'
    write(*,*) 'b114  = ', '-9623541317323077848129' .div. &
                           '3864449564977792573440'
    write(*,*) 'b115  = ', '-4823348333146829406881375' .div. &
                           '576413233634141239944816'
    write(*,*) 'b116  = ', '6566119246514996884067001154977284529' .div. &
                           '970305487021846325473990863582315520'
    write(*,*) 'b117  = ', '2226455130519213549256016892506730559375' .div. &
                           '364880443159675255577435648380047355776'
    write(*,*) 'b118  = ', '39747262782380466933662225' .div. &
                           '1756032802431424164410720256'
    write(*,*) 'b119  = ', '48175771419260955335244683805171548038966866545122229' .div. &
                           '1989786420513815146528880165952064118903852843612160000'
    write(*,*) 'b1110 = ', '-2378292068163246' .div. &
                           '47768728487211875'
    write(*,*) 'b121  = ', '-3218022174758599831659045535578571' .div. &
                           '1453396753634469525663775847094384'
    write(*,*) 'b122  = ', '0' .div. '0'
    write(*,*) 'b123  = ', '0' .div. '0'
    write(*,*) 'b124  = ', '26290092604284231996745' .div. &
                           '5760876126062860430544'
    write(*,*) 'b125  = ', '-697069297560926452045586710000' .div. &
                           '41107967755245430594036502319'
    write(*,*) 'b126  = ', '1827357820434213461438077550902273440' .div. &
                           '139381013914245317709567680839641697'
    write(*,*) 'b127  = ', '643504802814241550941949227194107500000' .div. &
                           '242124609118836550860494007545333945331'
    write(*,*) 'b128  = ', '162259938151380266113750' .div. &
                           '59091082835244183497007'
    write(*,*) 'b129  = ', '-23028251632873523818545414856857015616678575554130463402' .div. &
                           '20013169183191444503443905240405603349978424504151629055'
    write(*,*) 'b1210 = ', '7958341351371843889152' .div. &
                           '3284467988443203581305'
    write(*,*) 'b1211 = ', '-507974327957860843878400' .div. &
                           '121555654819179042718967'
    write(*,*) 'b131  = ', '-549080624436801105208519835138333' .div. &
                           '353987109028707139687100885600400'
    write(*,*) 'b132  = ', '0' .div. '0'
    write(*,*) 'b133  = ', '0' .div. '0'
    write(*,*) 'b134  = ', '29116675312186033956481' .div. &
                           '5331818957833865866320'
    write(*,*) 'b135  = ', '-91153092961177216058210567600' .div. &
                           '7609267653017028089793994539'
    write(*,*) 'b136  = ', '1540775569495234383390307262972464' .div. &
                           '164751248733597841985831445059895'
    write(*,*) 'b137  = ', '-2027488254536386321212021357622300000' .div. &
                           '7563800784313191306927092359781550321'
    write(*,*) 'b138  = ', '-707592954577756600025' .div. &
                           '2430671607007101253926'
    write(*,*) 'b139  = ', '-315869406877370103440389763510384832076674110731' .div. &
                           '550845901016714794114516357859531754926300500125'
    write(*,*) 'b1310 = ', '4305840920849725632512' .div. &
                           '5066413598663346018375'
    write(*,*) 'b1311 = ', '0' .div. '0'
    write(*,*) 'b1312 = ', '0' .div. '0'

    write(*,*) 'c1  = ', '4631674879841' .div. '103782082379976'
    write(*,*) 'c2  = ', '0' .div. '0'
    write(*,*) 'c3  = ', '0' .div. '0'
    write(*,*) 'c4  = ', '0' .div. '0'
    write(*,*) 'c5  = ', '0' .div. '0'
    write(*,*) 'c6  = ', '14327219974204125' .div. '40489566827933216'
    write(*,*) 'c7  = ', '2720762324010009765625000' .div. '10917367480696813922225349'
    write(*,*) 'c8  = ', '-498533005859375' .div. '95352091037424'
    write(*,*) 'c9  = ', '405932030463777247926705030596175437402459637909765779' .div. &
                         '78803919436321841083201886041201537229769115088303952'
    write(*,*) 'c10 = ', '-10290327637248' .div. '1082076946951'
    write(*,*) 'c11 = ', '863264105888000' .div. '85814662253313'
    write(*,*) 'c12 = ', '-29746300739' .div. '247142463456'
    write(*,*) 'c13 = ', '0' .div. '0'

    write(*,*) 'd1  = ', '325503096889' .div. '7282953149472'
    write(*,*) 'd2  = ', '0' .div. '0'
    write(*,*) 'd3  = ', '0' .div. '0'
    write(*,*) 'd4  = ', '0' .div. '0'
    write(*,*) 'd5  = ', '0' .div. '0'
    write(*,*) 'd6  = ', '495437430316125' .div. '1396191959583904'
    write(*,*) 'd7  = ', '760741072216796875000' .div. '3055518466469861159313'
    write(*,*) 'd8  = ', '1186143278515625' .div. '444976424841312'
    write(*,*) 'd9  = ', '13791579353894559147128282108092181066885426809' .div. &
                         '3515333032435874239702147176443897942760320040'
    write(*,*) 'd10 = ', '-30733299644928' .div. '5410384734755'
    write(*,*) 'd11 = ', '0' .div. '0'
    write(*,*) 'd12 = ', '0' .div. '0'
    write(*,*) 'd13 = ', '-45884771325' .div. '82380821152'



    write(*,*) ''
    write(*,*) ''
    write(*,*) ''
    write(*,*) 'PapakostasPapaGeorgiou5'

    write(*,*) 'a2 = ', '64' .div. '315'
    write(*,*) 'a3 = ', '115' .div. '381'
    write(*,*) 'a4 = ', '762' .div. '935'
    write(*,*) 'a5 = ', '25' .div. '28'
    write(*,*) 'a6 = ', '1'
    write(*,*) 'a7 = ', '1'
    write(*,*) 'b21 = ', '64' .div. '315'
    write(*,*) 'b31 = ', '480815' .div. '6193536'
    write(*,*) 'b32 = ', '462875' .div. '2064512'
    write(*,*) 'b41 = ', '344904825219069' .div. '345923838700000'
    write(*,*) 'b42 = ', '-2360077908867' .div. '601606676000'
    write(*,*) 'b43 = ', '40439332863108' .div. '10810119959375'
    write(*,*) 'b51 = ', '12078745127989699' .div. '5009699157786624'
    write(*,*) 'b52 = ', '-791781731775' .div. '81669668864'
    write(*,*) 'b53 = ', '39297175833216951' .div. '4694461413969824'
    write(*,*) 'b54 = ', '-10508413393960625' .div. '54097233987826176'
    write(*,*) 'b61 = ', '2251421737440863' .div. '828701767536000'
    write(*,*) 'b62 = ', '-39895842357' .div. '3782730880'
    write(*,*) 'b63 = ', '34564628685305112534' .div. '3916944972468643375'
    write(*,*) 'b64 = ', '12051135495733565' .div. '36492943960723917'
    write(*,*) 'b65 = ', '-26808346215168' .div. '82592376030125'
    write(*,*) 'b71 = ', '2405713' .div. '26289000'
    write(*,*) 'b72 = ', '0'
    write(*,*) 'b73 = ', '63896466577779' .div. '141024193000600'
    write(*,*) 'b74 = ', '454128848141375' .div. '589615117674696'
    write(*,*) 'b75 = ', '-1359311744' .div. '2892576375'
    write(*,*) 'b76 = ', '256979' .div. '1656648'
    write(*,*) 'c1 = ', '2405713' .div. '26289000'
    write(*,*) 'c2 = ', '0'
    write(*,*) 'c3 = ', '63896466577779' .div. '141024193000600'
    write(*,*) 'c4 = ', '454128848141375' .div. '589615117674696'
    write(*,*) 'c5 = ', '-1359311744' .div. '2892576375'
    write(*,*) 'c6 = ', '256979' .div. '1656648'
    write(*,*) 'd1 = ', '1818563883019' .div. '20194131951000'
    write(*,*) 'd2 = ', '0'
    write(*,*) 'd3 = ', '5513862498202899713' .div. '12036555896794210600'
    write(*,*) 'd4 = ', '324806515311046773125' .div. '452918159177876804664'
    write(*,*) 'd5 = ', '-126112324722496' .div. '317422653663375'
    write(*,*) 'd6 = ', '137695258717' .div. '1272569071032'
    write(*,*) 'd7 = ', '1' .div. '42'







    ! # """
    ! # The Relative Efficiency of Alternative Defect Control Schemes for High-Order Continuous Runge-Kutta Formulas
    ! #  W. H. Enright SIAM Journal on Numerical AnalysisVol. 30No. 5. (Oct.1993)pp. 1419-1445.
    ! # """

    write(*,*) ''
    write(*,*) ''
    write(*,*) ''
    write(*,*) ''
    write(*,*) ''
    write(*,*) '\n\n\n\n\n EnrightVerner8'

    write(*,*) 'c2 =',     '139' .div. &
              '2500'
    write(*,*) 'c3 =',     '473499' .div. &
              '4616000'
    write(*,*) 'c4 =',     '1420497' .div. &
              '9232000'
    write(*,*) 'c5 =',     '1923' .div. &
              '5000'
    write(*,*) 'c6 =',     '923' .div. &
              '2000'
    write(*,*) 'c7 =',     '769' .div. &
              '5000'
    write(*,*) 'c8 =',     '8571' .div. &
              '10000'
    write(*,*) 'c9 =',     '14280181565693441800' .div. &
              '15023510624554266963'
    write(*,*) 'c10 =',    '3611' .div. &
              '5000'
    write(*,*) 'c11 =',    '15' .div. &
              '16'
    write(*,*) 'c12 =',    '1'
    write(*,*) 'c13 =',    '1'
    write(*,*) 'A21 =',    '139' .div. &
              '2500'
    write(*,*) 'A31 =',    '94226774499' .div. &
              '11846945536000'
    write(*,*) 'A32 =',    '224201303001' .div. &
              '2369389107200'
    write(*,*) 'A41 =',    '1420497' .div. &
              '36928000'
    write(*,*) 'A42 =',    '0'
    write(*,*) 'A43 =',    '4261491' .div. &
              '36928000'
    write(*,*) 'A51 =',    '3590880889914277' .div. &
              '9341720958375000'
    write(*,*) 'A52 =',    '0'
    write(*,*) 'A53 =',    '-1122077307242443' .div. &
              '778476746531250'
    write(*,*) 'A54 =',    '1683359084698258' .div. &
              '1167715119796875'
    write(*,*) 'A61 =',    '3279216863' .div. &
              '71027928000'
    write(*,*) 'A62 =',    '0'
    write(*,*) 'A63 =',    '0'
    write(*,*) 'A64 =',    '1418466643672' .div. &
              '6146756958375'
    write(*,*) 'A65 =',    '725783021041' .div. &
              '3932390759616'
    write(*,*) 'A71 =',    '22628776819090891' .div. &
              '378192197956950000'
    write(*,*) 'A72 =',    '0'
    write(*,*) 'A73 =',    '0'
    write(*,*) 'A74 =',    '116326878837670912' .div. &
              '1047320101758291075'
    write(*,*) 'A75 =',    '-21022506263989' .div. &
              '614436056190000'
    write(*,*) 'A76 =',    '182198401' .div. &
              '10649112500'
    write(*,*) 'A81 =',    '-150702609045465151' .div. &
              '280142368857000000'
    write(*,*) 'A82 =',    '0'
    write(*,*) 'A83 =',    '0'
    write(*,*) 'A84 =',    '-47048174572430533795112' .div. &
              '6781573975400382234375'
    write(*,*) 'A85 =',    '-29670244019740727' .div. &
              '6363654238520000'
    write(*,*) 'A86 =',    '503350581600824913' .div. &
              '125990342179812500'
    write(*,*) 'A87 =',    '9'
    write(*,*) 'A91 =',    &
    '-16045918035544526531085666708200976799092954349059102554439418058855237034803138076235914485211856830024864166368323768200' &
    .div. &
    '9829483157728664844198544034921655975378257830080951893693520116322068439377446708493269493633244146421522730587295739613'
    write(*,*) 'A92 =',    '0'
    write(*,*) 'A93 =',    '0'
    write(*,*) 'A94 =',    &
'-2358539147881877873039445811751103289867403691705393271280994482043263320271984518766662589229245453919623760032009543680000000' &
    .div. &
    '217835526181955565558022800341429464086385172122906608031349543583863022015887117168566338470163090032498196064101837445053273'
    write(*,*) 'A95 =',    &
'-991136638972626168678903371416456100093900405535164924683058122802429707354033382826947398158683765324439618282500000000' .div. &
    '79848142008846002789298925227605775190331269194726743910364273272231784282184770467794155269096224513726772081370189773'
    write(*,*) 'A96 =',    &
'99411279821210413387149352497211785829547149358696952646781033905129048593757052549024957474512389085654050445280000000' .div. &
    '10219750071154355529199360565479179548497227516418867455143546824991883616697162903707552409044338358355837306818391433'
    write(*,*) 'A97 =',    &
    '194327599672380134095898291719912961363678073793023525007081328425098431574448809779310732532821200046895000000000' .div. &
    '11996011488227722649656673931136490891256463292620473601841875115170259043531987881275965965525693289412424400177'
    write(*,*) 'A98 =',    &
    '-4738143867684122189593816244199450540483384372163549951990525387550768038015218275414120082248510000000000' .div. &
    '45627381556119209828916528385326434273376137158228892503158792567081801761628344925618992749059885819540261'
    write(*,*) 'A101 =',   &
    '100509763879264306824096153463041174636629364248095333923106653001873' .div. &
    '229490324007644628042361756217436155832461488260089524115475000000000'
    write(*,*) 'A102 =',   '0'
    write(*,*) 'A103 =',   '0'
    write(*,*) 'A104 =',   '35076261889213578261995286390053983221937920015616' .div. &
              '8903662403052468890234321680409039895089390971875'
    write(*,*) 'A105 =',   '29877053472248545227782869189767925950557009' .div. &
              '10443709158645362958089905740134206606110000'
    write(*,*) 'A106 =',   '-72602025182798889442893966553844286012776770019588838776297031451' .div. &
              '40918439380405007926071673834718276106178863851548244800289187500'
    write(*,*) 'A107 =',   '-15322912063864370512130145988492098605486502994107816190' .div. &
              '3130069400645604876183669549605539076333579290524919889'
    write(*,*) 'A108 =',   '66085154677219418645471125072555541174985695924' .div. &
              '310222736648235062495097951962638800384203417327'
    write(*,*) 'A109 =',  &
    '-33475654618965607625490266678231230366345830527265525310030016230875755239420324728600957368877132012320553021' .div. &
    '563607181486505082775419237614873016043169125130359330269370345097328575740250457475506596993780051575000000000'
    write(*,*) 'A111 =',   '-5101097760197841615137571256611109219669965728737004453654940399435749927779' .div. &
              '3460254790065218498025394912113654440547987641021174968985010753183759952128'
    write(*,*) 'A112 =',   '0'
    write(*,*) 'A113 =',   '0'
    write(*,*) 'A114 =',   '-739282853758412257967453242147288028248514875000' .div. &
              '67244182875654829098323943713374207709631980757'
    write(*,*) 'A115 =',   '-304301954438407952266341438991435702455078125' .div. &
              '26817588459995310678515661957421952038871616'
    write(*,*) 'A116 =',   '7944893992399893116476631697520363855501654462308622798446100145004371780765625' .div. &
              '887005161602928659252827604035768657298435884939489754462273147402132735951354'
    write(*,*) 'A117 =',   '271466662889835128614810796916069488062485834784939879774912795880859375' .div. &
              '17080057868208807584911919572395871967414618088449127584294090356526117'
    write(*,*) 'A118 =',   '-547671639173493535187249719091874257724281347519101322819710937500' .div. &
              '5545897345777551056363997101744668576293320106258184122878184564689'
    write(*,*) 'A119 =',   &
        '2788050542294564730519147857701304010562775210713862748202538528260236025540'//&
        '35252878888463451556710568650135086751734733498388408614837563' .div. &
        '5703278975899384247167398970672581931599634186137224382398210398440764234613'//&
        '6141015055914541200747733442561186185728215790313661884647470336'
    write(*,*) 'A1110 =',  '-7641031089620713267943821483116886435546875' .div. &
              '1865115549729059107236140767953375513727732867'
    write(*,*) 'A121 =',   '-2173296165244568434534168496725754283210370856714048295955495704392191998074219' .div. &
              '826329709817197468253912814996276721297719271873047121496939265708086637102000'
    write(*,*) 'A122 =',   '0'
    write(*,*) 'A123 =',   '0'
    write(*,*) 'A124 =',   '-1176485889334472345397948024769479865991267657667189808793600000' .div. &
              '128238274125745784857395023683055391720808636521055761250085619'
    write(*,*) 'A125 =',   '-901641526273088667293677983960434306145731566007759375000' .div. &
              '47006051321531726477650870675115597779536141098588395119'
    write(*,*) 'A126 =',   '44619197709345843038609995810290222092713734357503573021289399188441075277893825000' .div. &
              '3047226829867759667805678237083584916028456629069014496162891378892137605913688013'
    write(*,*) 'A127 =',   '948339742665210716931560767459705347432692833888702670755668800833857753125000' .div. &
              '54100203068523816523920547203712158257383235486719126879290708372319555115953'
    write(*,*) 'A128 =',   '-469359986530784242367841163967997985280992052062479522694037718750000' .div. &
              '1262000068797740029604893636717124646814536106030908513651797310900389'
    write(*,*) 'A129 =',   &
      '-6641274810062608662454173531678791117254386081601341297887155553784655731178130017572'//&
      '38260693051901812594341608922016461968723754067215934799054852630446938063' .div. &
      '94740531367577932788646204486183484687947386153958347999510116829111616884404043875158'//&
      '4392785913224239894029843821464363224405440877235651675827150051218506000'
    write(*,*) 'A1210 =',  '124594688949431236718278718078688571245967961641584151713065625000' .div. &
              '2442266119995123914599926487344630873565880304003136401896150897137'
    write(*,*) 'A1211 =',  '561064742310110985699639624014768981793940948761680464947352' .div. &
              '671379391526329055316725343694451428305592235340194400714475'
    write(*,*) 'A131 =',   '2112537365439605317842214400294939600845864356126794907105288476157299972777' .div. &
              '9791129992576348954381966935712572673760150144277222568673000574989723295600'
    write(*,*) 'A132 =',   '0'
    write(*,*) 'A133 =',   '0'
    write(*,*) 'A134 =',   '63401732568650174882469655273037557867024014649473659852800000' .div. &
              '7597437179564699900056826682790095876501867794005624909986891'
    write(*,*) 'A135 =',   '6086761737797404637560639819223305802258863489403125000' .div. &
              '2784859079002788542258846760345051095481316496802932391'
    write(*,*) 'A136 =',   '-2559665601805637409198888850698428172565517218973577146475607855715703433900000' .div. &
              '1517075781450111169555356439342115045941218379260955323225394329122945220575603'
    write(*,*) 'A137 =',   '-3562962724414944757891560164749973139486260772587248231213362868690625000' .div. &
              '408976639178968899516222045518619849445955714595234573207901740578867341'
    write(*,*) 'A138 =',   '27274774093948890572327172846505411328624378015249298079900000000' .div. &
              '1115922486918625141794316350997246426030217325603407120793811703663'
    write(*,*) 'A139 =',   '1517649980273287794810711480928689762969941359446589355244683243470868'//&
                           '873980537462717945530891242892188019850453060256773054273287757818339' .div. &
              '17931096351403549517324822951619492690590259109554169692111771494162805088297821242'//&
              '848492102923447532215742446890559805801281057084758250800'
    write(*,*) 'A1310 =',  '36524642706569597276845612368244490741747970924877885921875000' .div. &
              '67204508345515429491838411146746729660269255184823759889942481'
    write(*,*) 'A1311 =',  '0'
    write(*,*) 'A1312 =',  '0'
    write(*,*) 'α1 =',     '55284231195707975647126708111723' .div. &
              '1258814250475014554517603612114000'
    write(*,*) 'α2 =',     '0'
    write(*,*) 'α3 =',     '0'
    write(*,*) 'α4 =',     '0'
    write(*,*) 'α5 =',     '0'
    write(*,*) 'α6 =',     '101652048214282205518610445783893750000000' .div. &
              '289586658278060310247144250081091360673509'
    write(*,*) 'α7 =',     '6219731958882093270433753490048828125000000' .div. &
              '25268792314562899182186819800834272764054341'
    write(*,*) 'α8 =',     '471525882014932587321673707929687500000000' .div. &
              '523728817391198795728810839649314044495553'
    write(*,*) 'α9 =',     '639984196590745029609794670270443805335134995621797161457881'//&
                           '53981258354882170183557294261050806789914954901252698438375102943237148428019253408419' .div. &
              '1406738387822415998067693599950032426772772394751630953646406168313006343'//&
              '7616452951326819824531063499969576146304645311203502485409773697867038000'
    write(*,*) 'α10 =',    '594562755257530592552224345703125000000' .div. &
              '123802720910327682301431417435953442122031'
    write(*,*) 'α11 =',    '-1886691133979705639959153870454656' .div. &
              '397947585885835383951563312487675'
    write(*,*) 'α12 =',    '-50061468875139778913910254637881' .div. &
              '141186641936036184819986782313781'
    write(*,*) 'α13 =',    '0'
    write(*,*) 'αEEst1 =', '3635543992294021475202312550589' .div. &
              '83920950031667636967840240807600'
    write(*,*) 'αEEst2 =', '0'
    write(*,*) 'αEEst3 =', '0'
    write(*,*) 'αEEst4 =', '0'
    write(*,*) 'αEEst5 =', '0'
    write(*,*) 'αEEst6 =', '91472553308336221233020750122000000000' .div. &
              '270389036674192633283981559366098375979'
    write(*,*) 'αEEst7 =', '31245710879106859500854236329453125000000' .div. &
              '125747467177230198813995913261775760852127'
    write(*,*) 'αEEst8 =', '7580382785455138868782239796250000000000' .div. &
              '33873008089978031564549954803189465564389'
    write(*,*) 'αEEst9 =', '-12646682075347720003894169090882958068230761859572980915113663'//&
                           '56638145411319432807599860924173256812108555319505561297378438513' .div. &
              '3145813368236387937635983732861131099009925923393634680707983641010013290201'//&
              '3789877583819562913435723874810037875576750158649200'
    write(*,*) 'αEEst10 =','2363202400377740499662213515625000000' .div. &
              '19167474982246118950523520271861502109'
    write(*,*) 'αEEst11 =','0'
    write(*,*) 'αEEst12 =','0'
    write(*,*) 'αEEst13 =','2965876353908674111604434604609' .div. &
              '47062213978678728273328927437927'



  write(*,*) ''
  write(*,*) ''
  write(*,*) ''
  write(*,*) 'Sharp9'

  write(*,*) 'c2  =', r2str( rr('1') / rr('50') )
  write(*,*) 'c3  =', r2str( rr('3837236') / rr('48429375') + rr('1031368') / rr('145288125') * sqrt(rr('6')) )
  write(*,*) 'c4  =', r2str( rr('1918618') / rr('16143125') + rr('515684') / rr('48429375') * sqrt(rr('6')) )
  write(*,*) 'c5  =', r2str( rr('14') / rr('45') )
  write(*,*) 'c6  =', r2str( rr('156') / rr('625') + rr('26') / rr('625') * sqrt(rr('6')) )
  write(*,*) 'c7  =', r2str( rr('156') / rr('625') - rr('26') / rr('625') * sqrt(rr('6')) )
  write(*,*) 'c8  =', r2str( rr('52') / rr('125') )
  write(*,*) 'c9  =', r2str( rr('39') / rr('125') )
  write(*,*) 'c10 =', r2str( rr('21') / rr('200') )
  write(*,*) 'c11 =', r2str( rr('280') / rr('477') )
  write(*,*) 'c12 =', r2str( rr('3658227035053715') / rr('5349704719299032') )
  write(*,*) 'c13 =', r2str( rr('247') / rr('281') )
  write(*,*) 'c14 =', r2str( rr('229') / rr('250') )
  write(*,*) 'c15 =', 1
  write(*,*) 'c16 =', 1
  write(*,*) 'A21 =', r2str( rr('1') / rr('50') )
  write(*,*) 'A31 =', r2str(rr('-24000387317036')/rr('281448523546875')-rr('5917264532296')/rr('281448523546875')*sqrt(rr('6')))
  write(*,*) 'A32 =', r2str(rr('46300580261936')/rr('281448523546875')+rr('7915204837696')/rr('281448523546875')*sqrt(rr('6')))
  write(*,*) 'A41 =', r2str(rr('959309')/rr('32286250')+rr('128921')/rr('48429375')*sqrt(rr('6')))
  write(*,*) 'A42 =', 0
  write(*,*) 'A43 =', r2str(rr('2877927')/rr('32286250')+rr('128921')/rr('16143125')*sqrt(rr('6')))
  write(*,*) 'A51 =', r2str(rr('2826523628723851')/rr('5953434698904030')-rr('68459492317475')/rr('595343469890403')*sqrt(rr('6')))
  write(*,*) 'A52 =', 0
  write(*,*) 'A53 =', r2str(rr('-704240024458145')/rr('396895646593602')+rr('91277530807085')/rr('198447823296801')*sqrt(rr('6')))
  write(*,*) 'A54 =', r2str(rr('958925642225180')/rr('595343469890403')-rr('205373100103780')/rr('595343469890403')*sqrt(rr('6')))
  write(*,*) 'A61 =', r2str(rr('376341108')/rr('9406484375')+rr('207933466')/rr('65845390625')*sqrt(rr('6')))
  write(*,*) 'A62 =', 0
  write(*,*) 'A63 =', 0
  write(*,*) 'A64 =', r2str(rr('4343545768844529')/rr('27892881885795625')+rr('469265141246109')/&
                            rr('27892881885795625')*sqrt(rr('6')))
  write(*,*) 'A65 =', r2str(rr('1559927818449')/rr('28957835234375')+rr('4382126882523')/rr('202704846640625')*sqrt(rr('6')))
  write(*,*) 'A71 =', r2str(rr('11781705468')/rr('235162109375')+rr('2328587014')/rr('1646134765625')*sqrt(rr('6')))
  write(*,*) 'A72 =', 0
  write(*,*) 'A73 =', 0
  write(*,*) 'A74 =', r2str(rr('23459106068523828440829')/rr('354298872323611753203125')+rr('7870375504052283205581')/&
                            rr('354298872323611753203125')*sqrt(rr('6')))
  write(*,*) 'A75 =', r2str(rr('146263465360621089')/rr('7558718942052734375')-rr('1881455818308499953')/&
                            rr('52911032594369140625')*sqrt(rr('6')))
  write(*,*) 'A76 =', r2str(rr('9444124356888')/rr('82889304453125')-rr('2459298027368')/rr('82889304453125')*sqrt(rr('6')))
  write(*,*) 'A81 =', r2str( rr('52') / rr('1125') )
  write(*,*) 'A82 =', 0
  write(*,*) 'A83 =', 0
  write(*,*) 'A84 =', 0
  write(*,*) 'A85 =', 0
  write(*,*) 'A86 =', r2str( rr('208') / rr('1125') - rr('13') / rr('1125') * sqrt(rr('6')) )
  write(*,*) 'A87 =', r2str( rr('208') / rr('1125') + rr('13') / rr('1125') * sqrt(rr('6')) )
  write(*,*) 'A91 =', r2str( rr('741') / rr('16000') )
  write(*,*) 'A92 =', 0
  write(*,*) 'A93 =', 0
  write(*,*) 'A94 =', 0
  write(*,*) 'A95 =', 0
  write(*,*) 'A96 =', r2str( rr('2301') / rr('16000') - rr('897') / rr('32000') * sqrt(rr('6')) )
  write(*,*) 'A97 =', r2str( rr('2301') / rr('16000') + rr('897') / rr('32000') * sqrt(rr('6')) )
  write(*,*) 'A98 =', r2str( rr('-351') / rr('16000') )
  write(*,*) 'A101 ='   , r2str( rr('35291978967') / rr('748709478400') )
  write(*,*) 'A102 ='   , 0
  write(*,*) 'A103 ='   , 0
  write(*,*) 'A104 ='   , 0
  write(*,*) 'A105 ='   , 0
  write(*,*) 'A106 ='   , r2str( rr('23154511989') / rr('149741895680') + rr('39398793') / rr('1772093440') * sqrt(rr('6')) )
  write(*,*) 'A107 ='   , r2str( rr('23154511989') / rr('149741895680') - rr('39398793') / rr('1772093440') * sqrt(rr('6')) )
  write(*,*) 'A108 ='   , r2str( rr('-6251205429') / rr('149741895680') )
  write(*,*) 'A109 ='   , r2str( rr('-981041103') / rr('4679434240') )
  write(*,*) 'A111 ='   , r2str( rr('1601589807329134144752443') / rr('16639785968494158002257920') )
  write(*,*) 'A112 ='   , 0
  write(*,*) 'A113 ='   , 0
  write(*,*) 'A114 ='   , 0
  write(*,*) 'A115 ='   , 0
  write(*,*) 'A116 ='   , r2str( rr('-1736562342312744743536201') / rr('1109319064566277200150528') - &
                                 rr('360257484908262597335743') / rr('511993414415204861607936') * sqrt(rr('6')) )
  write(*,*) 'A117 ='   , r2str( rr('-1736562342312744743536201') / rr('1109319064566277200150528') + &
                                 rr('360257484908262597335743') / rr('511993414415204861607936') * sqrt(rr('6')) )
  write(*,*) 'A118 ='   , r2str( rr('512032742176678555764127') / rr('369773021522092400050176') )
  write(*,*) 'A119 ='   , r2str( rr('248233526294563631278471') / rr('103998662303088487514112') )
  write(*,*) 'A1110 ='  , r2str( rr('-3') / rr('20') )
  write(*,*) 'A121 ='   , r2str( rr('-131987017608786696357225423387594635612719389206128606880670'//&
                                    '434178321331969627889057541436355642743061150672386594396559') / &
                                 rr('3187539260879955550151479262016120102404472282957894624867981'//&
                                    '16221476093939683123897279961564118214685494052121351290880') )
  write(*,*) 'A122 ='   , 0
  write(*,*) 'A123 ='   , 0
  write(*,*) 'A124 ='   , 0
  write(*,*) 'A125 ='   , 0
  write(*,*) 'A126 ='   , r2str( rr('-581038619225160876203856834629458675128926705143465'//&
                                    '192450716448466169075797359178616021045291080972121429188543592047') / &
                                 rr('1011917225676176365127453733973371461080784851732664'//&
                                    '960275549575306273314094232139356444322425772110112679530543972352') + &
                                 rr('7134827980789881496508823372973790665635109665954131'//&
                                    '1591981953238552744805776480028282602780942939988708855996996031') / &
                                 rr('5189319106031673667320275558837802364516845393500845'//&
                                    '9501310234631090939184319596890074067816706262057060488745844736') * sqrt(rr('6')) )
  write(*,*) 'A127 ='   , r2str( rr('-581038619225160876203856834629458675128926705143465'//&
                                    '192450716448466169075797359178616021045291080972121429188543592047') / &
                                 rr('1011917225676176365127453733973371461080784851732664'//&
                                    '960275549575306273314094232139356444322425772110112679530543972352') &
                                 - rr('71348279807898814965088233729737906656351096659541'//&
                                    '311591981953238552744805776480028282602780942939988708855996996031') / &
                                 rr('5189319106031673667320275558837802364516845393500845'//&
                                    '9501310234631090939184319596890074067816706262057060488745844736') * sqrt(rr('6')) )
  write(*,*) 'A128 ='   , r2str( rr('-189357008262607724321683086336517345228379250897103'//&
                                    '291049044350530935228180690663776657891613652665009511679250229667441') / &
                                 rr('1049020857284302831848793704219061747987080296296196'//&
                                    '00881898639306750333561102065113284728091471708748347777999725133824') )
  write(*,*) 'A129 ='   , r2str( rr('-161835099279281565399228415225411182739942653401484'//&
                                    '7245801101845172567304269189800544372100050869595166981551925667441') / &
                                 rr('1963751866077829758575464902492073991659898102893702'//&
                                    '9385347383945787366501641192454385997632075140011874187139618963456') )
  write(*,*) 'A1210 ='  , r2str( rr('6883437842714982754414155283530543027800010156600147'//&
                                    '069119889350771791431366439329656536871565378282089012991331513') / &
                                 rr('1827181489551794784669860898707808352423218653885642'//&
                                    '080180242252960011545073999200946066370836641132319880849653760') )
  write(*,*) 'A1211 ='  , r2str( rr('1155902714407169125662355662338897460971624798046364'//&
                                    '63234298604185457969653794053637008425503953091180886565') / &
                                 rr('3153613332490718363304114648792457541639644686569637'//&
                                    '67284078726004630975015921697525870414526347596936773632') )
  write(*,*) 'A131 ='   , r2str( rr('5215174783558918407997583468635543407988332719241764'//&
                                    '605769949554629') / &
                                 rr('2028313261321481206468509427515111171465117122753253'//&
                                    '3713038580121600') )
  write(*,*) 'A132 ='   , 0
  write(*,*) 'A133 ='   , 0
  write(*,*) 'A134 ='   , 0
  write(*,*) 'A135 ='   , 0
  write(*,*) 'A136 ='   , r2str( rr('2843598186227456480865065344408178581293412110128603') / &
                                 rr('792075053175002139265844335272820716293355019960320') - &
                                 rr('18227070890226867447840942666790512323422585544257') / &
                                 rr('121857700488461867579360666965049340968208464609280') * sqrt(rr('6')) )
  write(*,*) 'A137 ='   , r2str( rr('2843598186227456480865065344408178581293412110128603') / &
                                 rr('792075053175002139265844335272820716293355019960320') + &
                                 rr('18227070890226867447840942666790512323422585544257') / &
                                 rr('121857700488461867579360666965049340968208464609280') * sqrt(rr('6')) )
  write(*,*) 'A138 ='   , r2str( rr('9326829464422062118248457481351539504275339476759467047326605595685633') / &
                                 rr('4901901791858228863857691041029309678010355547895721285919177263022080') )
  write(*,*) 'A139 ='   , r2str( rr('-741604155090542466856213236072374206251235617068304762316465738169791') / &
                                 rr('141551673163321136844445993892555326037025917405403892742525852712960') )
  write(*,*) 'A1310 ='  , r2str( rr('-6058504866441219655595548618762485399974773685307046001179355536003') / &
                                 rr('2252275720815396172726400694965157641073696835574259179818290290400') )
  write(*,*) 'A1311 ='  , r2str( rr('-72917047186465183128180555150230405657138451692847535142343993') / &
                                 rr('44661747288016218276854771442831738093234145203222656783563600') )
  write(*,*) 'A1312 ='  , r2str( rr('273615392054092764377413314763529648694666091555825328598374202048'//&
                                    '8887296849241173151960647763453239551016003889152') / &
                                 rr('248567211069834101529026447046393920395586924961837540678716901800'//&
                                    '9688457749866177826801192710345262847046284166825') )
  write(*,*) 'A141 ='   , r2str( rr('1961431625890315687063141575818232405522545898155499982338718373117379429883') / &
                                 rr('480056647167077429990593568055406093586176318669944422673481728000000000000') )
  write(*,*) 'A142 ='   , 0
  write(*,*) 'A143 ='   , 0
  write(*,*) 'A144 ='   , 0
  write(*,*) 'A145 ='   , 0
  write(*,*) 'A146 ='   , r2str( rr('-8688525606146315530022414580346392155721271039') / &
                                 rr('22386738118754433181814607602481176248320000000') - &
                                 rr('10256190098435854298655077997613296122112139953') / &
                                 rr('1148037852243817086246902953973393653760000000') * sqrt(rr('6')) )
  write(*,*) 'A147 ='   , r2str( rr('-8688525606146315530022414580346392155721271039') / &
                                 rr('22386738118754433181814607602481176248320000000') + &
                                 rr('10256190098435854298655077997613296122112139953') / &
                                 rr('1148037852243817086246902953973393653760000000') * sqrt(rr('6')) )
  write(*,*) 'A148 ='   , r2str( rr('-108151392092290424953498836380059772609736403739434481043071361807712075869481') / &
                                 rr('8600735495194563448316261478331353993230137106756553600705167485829120000000') )
  write(*,*) 'A149 ='   , r2str( rr('683210554257935462600257975958139742203919396113084127371502375524416129719') / &
                                 rr('26895337200565243662247103690698994332502640106760065066162305761280000000') )
  write(*,*) 'A1410 ='  , r2str( rr('-125971034051203704183074450363446847441594334546885083244594242327104115033') / &
                                 rr('5066049934698363488698655054901069679084758735799331062593807151200000000') )
  write(*,*) 'A1411 ='  , r2str( rr('4322338495495152743252505005837177994220267688026960252214552638944423') / &
                                 rr('236867625787508422152958167179676757535142999000924357630500000000000') )
  write(*,*) 'A1412 ='  , r2str( rr('-8868241439418361942544164786624338811291728923916146394094449293049'//&
                                    '2112547171652363240146123589908870567811533658125375935101390832') / &
                                 rr('94051047762301760672023836894444996842386398230976500899543333833779'//&
                                    '58770491999076242219980453370248022825420814384818872314453125') )
  write(*,*) 'A1413 ='  , r2str( rr('26235475641986625187247554297838197168935151270802587') / &
                                 rr('31781620957198174033817415268740604591106877500000000') )
  write(*,*) 'A151 ='   , r2str( rr('-2933688768685553737193922190442902414638569907165819426999847151894747') / &
                                 rr('1423967854813137802350516795065201258930107696470226170813903745843200') )
  write(*,*) 'A152 ='   , 0
  write(*,*) 'A153 ='   , 0
  write(*,*) 'A154 ='   , 0
  write(*,*) 'A155 ='   , 0
  write(*,*) 'A156 ='   , r2str( rr('-279050827135618188106138704976571118076242172562777') / &
                                 rr('26980717750745660055932121988692169249262917386240') + &
                                 rr('59017804198407615229179283246229064921710388893173') / &
                                 rr('17987145167163773370621414659128112832841944924160') * sqrt(rr('6')) )
  write(*,*) 'A157 ='   , r2str( rr('-279050827135618188106138704976571118076242172562777') / &
                                 rr('26980717750745660055932121988692169249262917386240') - &
                                 rr('59017804198407615229179283246229064921710388893173') / &
                                 rr('17987145167163773370621414659128112832841944924160') * sqrt(rr('6')) )
  write(*,*) 'A158 ='   , r2str( rr('68240477823918559060550996013166770535743446467404475965020846786328901') / &
                                 rr('69628625900822775316247857716283393138506964199995817446989013471723520') )
  write(*,*) 'A159 ='   , r2str( rr('48531604865335743440838806675493568975092395234916265724406574203650554879') / &
                                 rr('7529075569049450724715447951105730327391304314424323559158913160761835520') )
  write(*,*) 'A1510 ='  , r2str( rr('7315898198049114373691779027237206235234893868747090317226910860963581499') / &
                                 rr('432400219379131684183655517400956805867024783959757582533074834771793600') )
  write(*,*) 'A1511 ='  , r2str( rr('-87035912584683752124645187592152267644073875904388006117245587111831') / &
                                 rr('41468532532723053663401983927439573730970639521941633396843682248800') )
  write(*,*) 'A1512 ='  , r2str( rr('7930060543280416510613601312564124744002530899095540053783327282'//&
                                    '14806108995212138291759017448087224471716436232175864384424753159293287828190208') / &
                                 rr('6006353055070484301705313231740079158134464998533481863014445357'//&
                                    '22932552444071789775577508083955280258070844780520371987041149053341617704219325') )
  write(*,*) 'A1513 ='  , r2str( rr('19013238692887784267164981427867630356262081870600946422701364458') / &
                                 rr('146516308633144198110735805762400905606463733191840985648075179899') )
  write(*,*) 'A1514 ='  , r2str( rr('368176545506575596342007241113258886329861009608750000000') / &
                                 rr('7515329389098801941975451526298754679007062667248055263091') )
  write(*,*) 'A161 ='   , r2str( rr('-36388658330162124762200023703074655379362961851837455245313588466117') / &
                                 rr('299291302137383314536268908335335078201218487321808786511993451315200') )
  write(*,*) 'A162 ='   , 0
  write(*,*) 'A163 ='   , 0
  write(*,*) 'A164 ='   , 0
  write(*,*) 'A165 ='   , 0
  write(*,*) 'A166 ='   , r2str( rr('-5017294099975580158862668031284197043053591531405121') / &
                                 rr('432874134925719951608236287739966681862399087083520') - &
                                 rr('115700422823857939498444446575144266776173664871303') / &
                                 rr('199788062273409208434570594341523083936491886346240') * sqrt(rr('6')) )
  write(*,*) 'A167 ='   , r2str( rr('-5017294099975580158862668031284197043053591531405121') / &
                                 rr('432874134925719951608236287739966681862399087083520') + &
                                 rr('115700422823857939498444446575144266776173664871303') / &
                                 rr('199788062273409208434570594341523083936491886346240') * sqrt(rr('6')) )
  write(*,*) 'A168 ='   , r2str( rr('-91869384706617020415871523809581333688744319256441669606452442156503951243') / &
                                 rr('10053990532496870785611049673131231012151168225596329651077879646797168640') )
  write(*,*) 'A169 ='   , r2str( rr('52862999381403119807509472978743982056878734540171187101666495333163485251') / &
                                 rr('2399905281474961058988767393522446655063434938667458594324573249770618880') )
  write(*,*) 'A1610 ='  , r2str( rr('11964965861294434337427534231330501089458731146841410298258149571218167') / &
                                 rr('1974017923973741961134058915802491373563600428420464580517659076868800') )
  write(*,*) 'A1611 ='  , r2str( rr('37167680872257703003686692191635149388479305578570534942584948859') / &
                                 rr('3913390524272247558198063815920084715858046504866006220243936800') )
  write(*,*) 'A1612 ='  , r2str( rr('-29443955867054347753341026121045589578978185460220369427665634428'//&
                                    '174791788280117223531690217195051250854008448256016995751289856') / &
                                 rr('613836440653194383209173469818173591961818242299772672817363925583'//&
                                    '9442923939727454711783369110119257425239678616468261908910975') )
  write(*,*) 'A1613 ='  , r2str( rr('407816748385172686498153181346812432791118177175769818363629863') / &
                                 rr('626162397882386095196201629768303759628250075752997120162580989') )
  write(*,*) 'A1614 ='  , 0
  write(*,*) 'A1615 ='  , 0
  write(*,*) 'α1 ='     , r2str( rr('30703843389361946002220520407') / rr('1036329015084155723633962896000') )
  write(*,*) 'α2 ='     , 0
  write(*,*) 'α3 ='     , 0
  write(*,*) 'α4 ='     , 0
  write(*,*) 'α5 ='     , 0
  write(*,*) 'α6 ='     , 0
  write(*,*) 'α7 ='     , 0
  write(*,*) 'α8 ='     , r2str( rr('1516681888913470906364013671875') / &
                                  rr('19423768214582439936604117641536') )
  write(*,*) 'α9 ='     , r2str( rr('1929922737998470573359614532470703125') / &
                                  rr('9295447834009061726737853188569292704') )
  write(*,*) 'α10 ='    , r2str( rr('27072397368129209968072433152000000000') / &
                                  rr('159540891067276798629433718421290211669') )
  write(*,*) 'α11 ='    , r2str( rr('3416676287738448149119878197304164096817920457') / &
                                  rr('22521752441211566270536786917243920830369456000') )
  write(*,*) 'α12 ='    , r2str( rr('90903490074941164563143999126052497791688659150254835'//&
                                    '5130330148829066896764151555292038222333366816993556860935646735988456500531298304') / &
                                  rr('6301978749188979317659380355882211371188146506066543'//&
                                     '226107226217493443986031316306450151922600620534579104501042337690306078523205079625') )
  write(*,*) 'α13 ='    , r2str( rr('9160897746149204383653282352747804858423571') / &
                                  rr('54934119002888850773584011583391921191449440') )
  write(*,*) 'α14 ='    , r2str( rr('3769686146953412690297035156250000') / &
                                  rr('195792979665408643382362918863397227') )
  write(*,*) 'α15 ='    , r2str( rr('50782110772148063247179059') / &
                                  rr('1538266148871578545201811280') )
  write(*,*) 'α16 ='    , 0
  write(*,*) 'αEEst1 =' , r2str( rr('135131455470598097879473933') / rr('4525454214341291369580624000') )
  write(*,*) 'αEEst2 =' , 0
  write(*,*) 'αEEst3 =' , 0
  write(*,*) 'αEEst4 =' , 0
  write(*,*) 'αEEst5 =' , 0
  write(*,*) 'αEEst6 =' , 0
  write(*,*) 'αEEst7 =' , 0
  write(*,*) 'αEEst8 =' , r2str( rr('2518169234679274570156341552734375') / &
                                 rr('38284247150941989115046715871467456') )
  write(*,*) 'αEEst9 =' , r2str( rr('13171020424136540706261627197265625') / &
                                 rr('61559257178867958455217570785227104') )
  write(*,*) 'αEEst10 =', r2str( rr('33191111003144264098986272000000000') / &
                                 rr('196721197370254992144801132455351679') )
  write(*,*) 'αEEst11 =', r2str( rr('98603841096694858013088556726735239713679') / &
                                 rr('574051243626833692823306576536179258032000') )
  write(*,*) 'αEEst12 =', r2str( rr('50936357685385761074157893008911450543340655548467600'//&
                                    '39946162068969260804962050605021707042060987973190001839145435136') / &
                                 rr('40589106364182299226269510371318864016087966429387805'//&
                                    '447847648739509460272079710913035195393529050583646417571607929125') )
  write(*,*) 'αEEst13 =', r2str( rr('108010721096523379193662759959856611609133') / &
                                 rr('570689193181223151553200582051397411377120') )
  write(*,*) 'αEEst14 =', 0
  write(*,*) 'αEEst15 =', 0
  write(*,*) 'αEEst16 =', r2str( rr('26859551018855966185191031') / rr('763900876650511794556001520') )




  write(*,*) ''
  write(*,*) ''
  write(*,*) ''
  write(*,*) 'PapakostasPapaGeorgiou52'

  write(*,*) 'c[2]     =',  '35' .div. '159'
  write(*,*) 'c[3]     =',  '42' .div. '131'
  write(*,*) 'c[4]     =',  '131' .div. '143'
  write(*,*) 'c[5]     =',  '21' .div. '22'
  write(*,*) 'c[6]     =',  '1'
  write(*,*) 'c[7]     =',  '1'
  write(*,*) 'A[2, 1]  =',  '35' .div. '159'
  write(*,*) 'A[3, 1]  =',  '7476' .div. '85805'
  write(*,*) 'A[3, 2]  =',  '20034' .div. '85805'
  write(*,*) 'A[4, 1]  =',  '2438549411' .div. '1983961980'
  write(*,*) 'A[4, 2]  =',  '-3707256508' .div. '716430715'
  write(*,*) 'A[4, 3]  =',  '25077455105' .div. '5158301148'
  write(*,*) 'A[5, 1]  =',  '105337889067' .div. '64388030080'
  write(*,*) 'A[5, 2]  =',  '-1698584121' .div. '245755840'
  write(*,*) 'A[5, 3]  =',  '6869523776931' .div. '1096562558080'
  write(*,*) 'A[5, 4]  =',  '-927215289' .div. '26981535520'
  write(*,*) 'A[6, 1]  =',  '67512025387' .div. '32454592380'
  write(*,*) 'A[6, 2]  =',  '-20051384' .div. '2293935'
  write(*,*) 'A[6, 3]  =',  '10587214001321' .div. '1373901639516'
  write(*,*) 'A[6, 4]  =',  '731293420' .div. '8209319229'
  write(*,*) 'A[6, 5]  =',  '-144610048' .div. '1077690663'
  write(*,*) 'A[7, 1]  =',  '669707' .div. '6932520'
  write(*,*) 'A[7, 2]  =',  '0'
  write(*,*) 'A[7, 3]  =',  '2215522905683' .div. '4570867891800'
  write(*,*) 'A[7, 4]  =',  '349043981' .div. '116904400'
  write(*,*) 'A[7, 5]  =',  '-2234144' .div. '575505'
  write(*,*) 'A[7, 6]  =',  '9363' .div. '7120'
  write(*,*) 'α[1]     =',  '669707' .div. '6932520'
  write(*,*) 'α[2]     =',  '0'
  write(*,*) 'α[3]     =',  '2215522905683' .div. '4570867891800'
  write(*,*) 'α[4]     =',  '349043981' .div. '116904400'
  write(*,*) 'α[5]     =',  '-2234144' .div. '575505'
  write(*,*) 'α[6]     =',  '9363' .div. '7120'
  write(*,*) 'αEEst[1] =',  '2243660497' .div. '23535905400'
  write(*,*) 'αEEst[2] =',  '0'
  write(*,*) 'αEEst[3] =',  '7589131232781673' .div. '15518096492661000'
  write(*,*) 'αEEst[4] =',  '1104461697911' .div. '396890438000'
  write(*,*) 'αEEst[5] =',  '-6925033984' .div. '1953839475'
  write(*,*) 'αEEst[6] =',  '3529851' .div. '3021550'
  write(*,*) 'αEEst[7] =',  '1' .div. '112'




  write(*,*) ''
  write(*,*) ''
  write(*,*) ''
  write(*,*) 'SharpSmart7'

  write(*,*) 'real(wp),parameter :: a2    =', '1' .div. '50'
  write(*,*) 'real(wp),parameter :: a3    =', '27' .div. '125'
  write(*,*) 'real(wp),parameter :: a4    =', '41' .div. '100'
  write(*,*) 'real(wp),parameter :: a5    =', '57' .div. '100'
  write(*,*) 'real(wp),parameter :: a6    =', '43' .div. '50'
  write(*,*) 'real(wp),parameter :: a7    =', '2272510' .div. '11977321'
  write(*,*) 'real(wp),parameter :: a8    =', '18' .div. '25'
  write(*,*) 'real(wp),parameter :: a9    =', '5' .div. '6'
  write(*,*) 'real(wp),parameter :: a10   =', '1'
  write(*,*) 'real(wp),parameter :: a11   =', '1'
  write(*,*) 'real(wp),parameter :: b21   =', '1' .div. '50'
  write(*,*) 'real(wp),parameter :: b31   =', '-594' .div. '625'
  write(*,*) 'real(wp),parameter :: b32   =', '729' .div. '625'
  write(*,*) 'real(wp),parameter :: b41   =', '451' .div. '21600'
  write(*,*) 'real(wp),parameter :: b42   =', '0'
  write(*,*) 'real(wp),parameter :: b43   =', '1681' .div. '4320'
  write(*,*) 'real(wp),parameter :: b51   =', '19' .div. '160'
  write(*,*) 'real(wp),parameter :: b52   =', '0'
  write(*,*) 'real(wp),parameter :: b53   =', '361' .div. '3104'
  write(*,*) 'real(wp),parameter :: b54   =', '3249' .div. '9700'
  write(*,*) 'real(wp),parameter :: b61   =', '-31' .div. '200'
  write(*,*) 'real(wp),parameter :: b62   =', '0'
  write(*,*) 'real(wp),parameter :: b63   =', '520921' .div. '412056'
  write(*,*) 'real(wp),parameter :: b64   =', '-17371' .div. '11640'
  write(*,*) 'real(wp),parameter :: b65   =', '132023' .div. '106200'
  write(*,*) 'real(wp),parameter :: b71   =', '25959766877768976976598957736980' .div. '487594514129628295945513157189933'
  write(*,*) 'real(wp),parameter :: b72   =', '0'
  write(*,*) 'real(wp),parameter :: b73   =', '347890318302644246405985993187156250' .div. '1321817402067092875750818220388519949'
  write(*,*) 'real(wp),parameter :: b74   =', '-1717046972617147709491116450178750' .div. '7467894926932728111586543618014237'
  write(*,*) 'real(wp),parameter :: b75   =', '29780304732725103577764751746216250' .div. '258912687002832625147067486467854423'
  write(*,*) 'real(wp),parameter :: b76   =', '-302662548054389051180423185000' .div. '25662869164717278733974376694207'
  write(*,*) 'real(wp),parameter :: b81   =', '42409705291266846' .div. '416462256407406875'
  write(*,*) 'real(wp),parameter :: b82   =', '0'
  write(*,*) 'real(wp),parameter :: b83   =', '3247095172038' .div. '883201854817'
  write(*,*) 'real(wp),parameter :: b84   =', '-518509279926' .div. '374238074075'
  write(*,*) 'real(wp),parameter :: b85   =', '435669225629732566638' .div. '393965828849029186615'
  write(*,*) 'real(wp),parameter :: b86   =', '-6468694559114760' .div. '61945939006089637'
  write(*,*) 'real(wp),parameter :: b87   =', '-8593750881095206170491007194502' .div. '3213504543545558150903880585625'
  write(*,*) 'real(wp),parameter :: b91   =', '-1401024812030113404025' .div. '19887564677841032175639'
  write(*,*) 'real(wp),parameter :: b92   =', '0'
  write(*,*) 'real(wp),parameter :: b93   =', '13281373111234375' .div. '5150833217292744'
  write(*,*) 'real(wp),parameter :: b94   =', '-50491693720625' .div. '29100752640072'
  write(*,*) 'real(wp),parameter :: b95   =', '8909776468783164583973193125' .div. '6271093223575470807674793192'
  write(*,*) 'real(wp),parameter :: b96   =', '-4792324941735635008750' .div. '159776107397443897190271'
  write(*,*) 'real(wp),parameter :: b97   =', '-1532806290465891141166096531902118541769245' .div. &
                                              '1203242011387872547807852011647420329982736'
  write(*,*) 'real(wp),parameter :: b98   =', '-7500029126894375' .div. '132689679447323376'
  write(*,*) 'real(wp),parameter :: b101  =', '36393032615434450612' .div. '324390586094889663425'
  write(*,*) 'real(wp),parameter :: b102  =', '0'
  write(*,*) 'real(wp),parameter :: b103  =', '-1462401427649331250' .div. '154787214582248211'
  write(*,*) 'real(wp),parameter :: b104  =', '4135780451822750' .div. '874504037187843'
  write(*,*) 'real(wp),parameter :: b105  =', '-2349378733647002895234008950' .div. '1090914599757106529355865311'
  write(*,*) 'real(wp),parameter :: b106  =', '-78686605908422443750' .div. '52446632451499515953'
  write(*,*) 'real(wp),parameter :: b107  =', '2315079813491204524435067899365885119542372444358703' .div. &
                                              '316169042039527157595235231573788308031260760584200'
  write(*,*) 'real(wp),parameter :: b108  =', '-33473047374792524975' .div. '32907430028856870472'
  write(*,*) 'real(wp),parameter :: b109  =', '5594658687556280397846' .div. '1893189870520997940175'
  write(*,*) 'real(wp),parameter :: b111  =', '2508607706701842363083' .div. '197875357745688550590720'
  write(*,*) 'real(wp),parameter :: b112  =', '0'
  write(*,*) 'real(wp),parameter :: b113  =', '-5122833329940625' .div. '508724268374592'
  write(*,*) 'real(wp),parameter :: b114  =', '13293920580875' .div. '2874148408896'
  write(*,*) 'real(wp),parameter :: b115  =', '-599188464780493707137440161875' .div. '277270064173229869784600732736'
  write(*,*) 'real(wp),parameter :: b116  =', '-3601465055348923762849875' .div. '2146128454918752594358208'
  write(*,*) 'real(wp),parameter :: b117  =', '606030238246181777051198920509497430523044409408159' .div. &
                                              '74752050141640998967813674460513197348653288024576'
  write(*,*) 'real(wp),parameter :: b118  =', '-1922750201834125' .div. '1941504226023936'
  write(*,*) 'real(wp),parameter :: b119  =', '12539348439579' .div. '3975412795840'
  write(*,*) 'real(wp),parameter :: b1110 =', '0'
  write(*,*) 'real(wp),parameter :: c1    =', '771570009067' .div. '14036203465200'
  write(*,*) 'real(wp),parameter :: c2    =', '0'
  write(*,*) 'real(wp),parameter :: c3    =', '0'
  write(*,*) 'real(wp),parameter :: c4    =', '0'
  write(*,*) 'real(wp),parameter :: c5    =', '28304779228000000' .div. '53707434325074117'
  write(*,*) 'real(wp),parameter :: c6    =', '-296881060859375' .div. '515060733835389'
  write(*,*) 'real(wp),parameter :: c7    =', '744858303758379680905615939985761920312207508379' .div. &
                                              '2487223884477764590764433396524922145673887618400'
  write(*,*) 'real(wp),parameter :: c8    =', '-5118512171875' .div. '11763620626464'
  write(*,*) 'real(wp),parameter :: c9    =', '136801854099' .div. '127885521925'
  write(*,*) 'real(wp),parameter :: c10   =', '103626500437' .div. '1717635089268'
  write(*,*) 'real(wp),parameter :: c11   =', '0'
  write(*,*) 'real(wp),parameter :: d1    =', '448234490819' .div. '8120946290580'
  write(*,*) 'real(wp),parameter :: d2    =', '0'
  write(*,*) 'real(wp),parameter :: d3    =', '0'
  write(*,*) 'real(wp),parameter :: d4    =', '0'
  write(*,*) 'real(wp),parameter :: d5    =', '7786773134600000' .div. '14452831163890377'
  write(*,*) 'real(wp),parameter :: d6    =', '-408698637296875' .div. '567617951573694'
  write(*,*) 'real(wp),parameter :: d7    =', '4426705150369152638325381078278067803359' .div. &
                                              '14828075230102658203818343670586143438076'
  write(*,*) 'real(wp),parameter :: d8    =', '-5004542378125' .div. '10330679593521'
  write(*,*) 'real(wp),parameter :: d9    =', '154806770859' .div. '124231649870'
  write(*,*) 'real(wp),parameter :: d10   =', '0'
  write(*,*) 'real(wp),parameter :: d11   =', '16' .div. '243'




  write(*,*) ''
  write(*,*) ''
  write(*,*) ''
  write(*,*) ''
  write(*,*) ''
  write(*,*) 'YamashitaStable7'

   ! http://www.peterstone.name/Maplepgs/Maple/nmthds/RKcoeff/Runge_Kutta_schemes/RK7/RKcoeff7d_3.pdf

  write(*,*) 'real(wp),parameter :: a2   =', '1288' .div. '14535'
  write(*,*) 'real(wp),parameter :: a3   =', '644' .div. '4845'
  write(*,*) 'real(wp),parameter :: a4   =', '322' .div. '1615'
  write(*,*) 'real(wp),parameter :: a5   =', '65' .div. '258'
  write(*,*) 'real(wp),parameter :: a6   =', '627862490' .div. '27724306937'
  write(*,*) 'real(wp),parameter :: a7   =', '78' .div. '115'
  write(*,*) 'real(wp),parameter :: a8   =', '95' .div. '124'
  write(*,*) 'real(wp),parameter :: a9   =', '1'
  write(*,*) 'real(wp),parameter :: a10  =', '1'
  write(*,*) 'real(wp),parameter :: b21  =', '1288' .div. '14535'
  write(*,*) 'real(wp),parameter :: b31  =', '161' .div. '4845'
  write(*,*) 'real(wp),parameter :: b32  =', '161' .div. '1615'
  write(*,*) 'real(wp),parameter :: b41  =', '161' .div. '3230'
  write(*,*) 'real(wp),parameter :: b42  =', '0'
  write(*,*) 'real(wp),parameter :: b43  =', '483' .div. '3230'
  write(*,*) 'real(wp),parameter :: b51  =', '196347867755' .div. '3561236836416'
  write(*,*) 'real(wp),parameter :: b52  =', '0'
  write(*,*) 'real(wp),parameter :: b53  =', '134004261625' .div. '1187078945472'
  write(*,*) 'real(wp),parameter :: b54  =', '149425089125' .div. '1780618418208'
  write(*,*) 'real(wp),parameter :: b61  =', '874723327324627172137139789673935509613630495' .div. &
                                             '56881344496107103495850556251109088355454531158'
  write(*,*) 'real(wp),parameter :: b62  =', '0'
  write(*,*) 'real(wp),parameter :: b63  =', '140089490273660861720564275306545765967660125' .div. &
                                             '4375488038162084884296196634700699104265733166'
  write(*,*) 'real(wp),parameter :: b64  =', '-2038049847879400647989164901369906650290192935250' .div. &
                                             '47909406273855748440601205051655304842157645301117'
  write(*,*) 'real(wp),parameter :: b65  =', '2992403630086592541124850354372857004595944160' .div. &
                                             '168193972876080132251638040959092347542006316207'
  write(*,*) 'real(wp),parameter :: b71  =', '-2001378790961964301303250341598299' .div. '131178829335937360185206084581250'
  write(*,*) 'real(wp),parameter :: b72  =', '0'
  write(*,*) 'real(wp),parameter :: b73  =', '-1366679891168526950613' .div. '3342867750190010177170'
  write(*,*) 'real(wp),parameter :: b74  =', '-197077954039191584877658472075693196650' .div. &
                                             '14482289235786224954374999272581521053'
  write(*,*) 'real(wp),parameter :: b75  =', '2928205733652489758138852423071126752' .div. &
                                             '289494157432907631631314224968221875'
  write(*,*) 'real(wp),parameter :: b76  =', '32572843800597493853254181634376441943013874856495312' .div. &
                                             '1642002429836009758962688168840520197029337863346875'
  write(*,*) 'real(wp),parameter :: b81  =', '226949925367094612475083609619198193642397605' .div. &
                                             '120406368918742115270494114142742317844627456'
  write(*,*) 'real(wp),parameter :: b82  =', '0'
  write(*,*) 'real(wp),parameter :: b83  =', '83451940525721530822125' .div. '1129677771714575730562048'
  write(*,*) 'real(wp),parameter :: b84  =', '7035716180093388934005544535766324331669337496890597125' .div. &
                                             '8744668672303692797949525789789288711635037496457428992'
  write(*,*) 'real(wp),parameter :: b85  =', '1783910495800307104322539337559667105512922125' .div. &
                                             '384741540181237650862158213355907627413029681664'
  write(*,*) 'real(wp),parameter :: b86  =', '-84153602056538973791098303633128803165153465256807063611390103929570560775' .div. &
                                             '37282901435832588263568764858094381410386083732352113498074163165795975168'
  write(*,*) 'real(wp),parameter :: b87  =', '529655154424978769932790603243342890625' .div. &
                                             '2074272966571578715335103162383459680256'
  write(*,*) 'real(wp),parameter :: b91  =', '-2220302447236283385210081868020072818509' .div. &
                                             '374126802552343922668161638021420098000'
  write(*,*) 'real(wp),parameter :: b92  =', '0'
  write(*,*) 'real(wp),parameter :: b93  =', '-580875348986851918117575' .div. '7422906155739208262352728'
  write(*,*) 'real(wp),parameter :: b94  =', '-10152884092399228192381460837845336141124812794348025' .div. &
                                             '1298474693887469810743537803700532058113206856602944'
  write(*,*) 'real(wp),parameter :: b95  =', '57628597675871150072147324302138021982593246488' .div. &
                                             '8694208064927983865022808524707850651868593125'
  write(*,*) 'real(wp),parameter :: b96  =', &
      '25006323928409346448859146781582297955220041834414805003311931721685248934905197' .div. &
      '3116797087097920659891402066923116953398124957660879445267880838924530153376000'
  write(*,*) 'real(wp),parameter :: b97  =', '-991935992163983524020354479671037652370649875' .div. &
                                             '1096924827756227471690652450958041839154828032'
  write(*,*) 'real(wp),parameter :: b98  =', '11028636941861502413824025771962099757599945728' .div. &
                                             '10166706010345110864067934052134974581343819375'
  write(*,*) 'real(wp),parameter :: b101 =', '2847557162233802909802913419338134005277' .div. &
                                             '175580852316165047798631596921711256000'
  write(*,*) 'real(wp),parameter :: b102 =', '0'
  write(*,*) 'real(wp),parameter :: b103 =', '1552914837310075' .div. '7167358071597822'
  write(*,*) 'real(wp),parameter :: b104 =', '7813795507395804332400817811705117266280297151179075' .div. &
                                             '609385085239477995902119898391644620983028518989568'
  write(*,*) 'real(wp),parameter :: b105 =', '-306358654025510315315806741256227901425369583' .div. &
                                             '37780230596405542839291492772111971777338125'
  write(*,*) 'real(wp),parameter :: b106 =', &
   '-30230616135053261889365940573714713926595600173797519397657905897494488117634591' .div. &
   '1462739064177653822206630548583223553011887775071057363338006334613651993472000'
  write(*,*) 'real(wp),parameter :: b107 =', '-3556025825918703192187464108779875' .div. &
                                             '11170666795578957880984290260063232'
  write(*,*) 'real(wp),parameter :: b108 =', '2140578935503723938488131712' .div. &
                                             '2556174768949326564363043125'
  write(*,*) 'real(wp),parameter :: b109 =', '0'
  write(*,*) 'real(wp),parameter :: c1   =', '-28836965799708194669' .div. '40897924114041540000'
  write(*,*) 'real(wp),parameter :: c2   =', '0'
  write(*,*) 'real(wp),parameter :: c3   =', '0'
  write(*,*) 'real(wp),parameter :: c4   =', '-5319231056637407390089058139231875' .div. '4078513870347642725257280732562048'
  write(*,*) 'real(wp),parameter :: c5   =', '2430832495624902882205404599808' .div. '1640677233246140147577399278125'
  write(*,*) 'real(wp),parameter :: c6   =', '58846832125102891510730576086257275195560457005949780449038492433452787' .div. &
                                             '55051022626529988904867724618756855605285697694381512902783594390080000'
  write(*,*) 'real(wp),parameter :: c7   =', '8765694250492187515737289375' .div. '142862058843931893355781359104'
  write(*,*) 'real(wp),parameter :: c8   =', '282726763309436004945812396032' .div. '864839130161188820942829590625'
  write(*,*) 'real(wp),parameter :: c9   =', '35795813026789129771' .div. '507885604115513709330'
  write(*,*) 'real(wp),parameter :: c10  =', '0'
  write(*,*) 'real(wp),parameter :: d1   =', '-4303806316703372599' .div. '11685121175440440000'
  write(*,*) 'real(wp),parameter :: d2   =', '0'
  write(*,*) 'real(wp),parameter :: d3   =', '0'
  write(*,*) 'real(wp),parameter :: d4   =', '-2188638181830974432849378205625' .div. '2703688346269567600435718085888'
  write(*,*) 'real(wp),parameter :: d5   =', '1313681506776569792299836438' .div. '1214416900996402773928496875'
  write(*,*) 'real(wp),parameter :: d6   =', '3205635250634133320066291736997892470430172563677735127041069' .div. &
                                             '5224293273794925935053617011014728265925425567271645535360000'
  write(*,*) 'real(wp),parameter :: d7   =', '153260086062341088187716875' .div. '1103181921574763655256998912'
  write(*,*) 'real(wp),parameter :: d8   =', '1144102534493369691260897984' .div. '4260291281582210940605071875'
  write(*,*) 'real(wp),parameter :: d9   =', '0'
  write(*,*) 'real(wp),parameter :: d10  =', '3' .div. '40'






    contains

      subroutine write(name, s1, s2)
        !! print s1/s2 for fortran parameter
        character(len=*),intent(in) :: name
        character(len=*),intent(in) :: s1, s2
        type (mp_real) :: r

        r = mpreal (trim(s1)//'.0',nwds2) / mpreal(trim(s2)//'.0',nwds2)
        write(output_unit,'(a)',advance='NO') 'real(wp),parameter :: '//trim(name)//' = '
        call mpwrite (output_unit,ilen,iprec,r)

      end subroutine write

      function rr(s) result(r)
        !! string to mp_real
        character(len=*),intent(in) :: s
        type (mp_real) :: r
        r = mpreal (s//'.0',nwds2)
      end function rr

      function r2str(r) result(str)
        !! to string
        type (mp_real),intent(in) :: r
        character(len=:),allocatable :: str
        character(len=1),dimension(ilen) :: s
        call mpeform (r,ilen,iprec,s)
        str = trim(transfer(s, repeat(' ', ilen)))//'_wp'
      end function r2str

      function to_string(s1,s2) result(str)
        !! 'int' / 'int' operator
        character(len=*),intent(in) :: s1, s2
        character(len=:),allocatable :: str
        type (mp_real) :: r,r1,r2
        character(len=1),dimension(ilen) :: s
        integer :: i

        s = ''
        r1 = mpreal (s1//'.0',nwds2)
        r2 = mpreal (s2//'.0',nwds2)
        r = r1 / r2
        call mpeform (r,ilen,iprec,s)
        ! write(*,*) 's=',s


        str = trim(transfer(s, repeat(' ', ilen)))//'_wp'
        return


        ! r = mpreal (trim(s1)//'.0',nwds2) / mpreal(trim(s2)//'.0',nwds2)

        ! s = ' '
        ! call mpeform (r,ilen,iprec,s)

        ! str = repeat(' ', ilen)
        ! do i = 1, ilen
        !     str(i:i) = s(i)
        ! end do
        ! str = trim(str)//'_wp'

      end function to_string

      function to_string2(s1,s2) result(str)
        !! 'int' / 'int' operator
        character(len=*),intent(in) :: s1, s2
        character(len=:),allocatable :: str
        character(len=1),dimension(ilen) :: s

        r1 = mpreal (s1//'.0',nwds2)
        r2 = mpreal (s2//'.0',nwds2)
        r = r1 / r2

        call mpeform (r,ilen,iprec,s)
        str = trim(transfer(s, repeat(' ', ilen)))//'_wp'

      end function to_string2

!*****************************************************************************************
    end program coefficients
!*****************************************************************************************