/******************************************************************************
 * unix.c (uname.c) - Ruby Extension for *nix (POSIX?) operating systems.
 *
 * Author(s):
 *    Daniel Berger (primary)
 *    Mike Hall (OS X patch)
 *****************************************************************************/
#include "ruby.h"
#include "version.h"
#include <sys/utsname.h>

/* Solaris */
#ifdef HAVE_SYS_SYSTEMINFO_H
#define BUFSIZE 257           /* Set as per the sysinfo(2) manpage */
#include <sys/systeminfo.h>
#endif

/* OS X - use Darwin's 'sysctl()' to get the model name */
#if defined(__MACH__) && defined(__APPLE__)
#include <sys/sysctl.h>
static int getmodel(char *buf, int n)
{
   size_t sz = n;
   int mib[2];

   mib[0] = CTL_HW;
   mib[1] = HW_MODEL;
   return sysctl(mib, 2, buf, &sz, NULL, 0);
}
#endif

#ifdef __cplusplus
extern "C" 
{
#endif

VALUE sUname;

/************************************************
 * Return all information within one big struct
 ************************************************/
static VALUE uname_uname_all()
{
   struct utsname u;
   uname(&u);

/* Extra brackets are for C89 compliance */
{
#ifdef HAVE_SYS_SYSTEMINFO_H
   char platform[BUFSIZE];
   char arch[BUFSIZE];
   sysinfo(SI_ARCHITECTURE,arch,BUFSIZE);
   sysinfo(SI_PLATFORM,platform,BUFSIZE);
#endif

#if defined(__MACH__) && defined(__APPLE__)
   char model[BUFSIZ];
   getmodel(model, sizeof(model));
#endif

   return rb_struct_new(sUname,
      rb_str_new2(u.sysname),
      rb_str_new2(u.nodename),
      rb_str_new2(u.machine),
      rb_str_new2(u.version),
      rb_str_new2(u.release)
#ifdef HAVE_SYS_SYSTEMINFO_H
      ,rb_str_new2(arch),
      rb_str_new2(platform)
#endif

#if defined(__MACH__) && defined(__APPLE__)
      ,rb_str_new2(model)
#endif

#if defined(__hpux)
      ,rb_str_new2(u.__idnumber)
#endif
   );
}
}

static VALUE uname_nodename()
{
   struct utsname u;
   uname(&u);
   return rb_str_new2(u.nodename);
}

static VALUE uname_machine()
{
   struct utsname u;
   uname(&u);
   return rb_str_new2(u.machine);
}

static VALUE uname_version()
{
   struct utsname u;
   uname(&u);
   return rb_str_new2(u.version);
}

static VALUE uname_release()
{
   struct utsname u;
   uname(&u);
   return rb_str_new2(u.release);
}

static VALUE uname_sysname()
{
   struct utsname u;
   uname(&u);
   return rb_str_new2(u.sysname);
}

#ifdef HAVE_SYS_SYSTEMINFO_H
static VALUE uname_architecture()
{
   char buf[BUFSIZE];
   sysinfo(SI_ARCHITECTURE,buf,BUFSIZE);
   return rb_str_new2(buf);
}

static VALUE uname_platform()
{
   char buf[BUFSIZE];
   sysinfo(SI_PLATFORM,buf,BUFSIZE);
   return rb_str_new2(buf);
}

/* Not defined in 2.5.x */
#ifdef SI_ISALIST
static VALUE uname_isalist()
{
   char buf[BUFSIZE];
   sysinfo(SI_ISALIST,buf,BUFSIZE);
   return rb_str_new2(buf);
}
#endif

static VALUE uname_hw_provider()
{
   char buf[BUFSIZE];
   sysinfo(SI_HW_PROVIDER,buf,BUFSIZE);
   return rb_str_new2(buf);
}

static VALUE uname_hw_serial()
{
   char buf[BUFSIZE];
   sysinfo(SI_HW_SERIAL,buf,BUFSIZE);
   return rb_Integer(rb_str_new2(buf));
}

static VALUE uname_srpc_domain()
{
   char buf[BUFSIZE];
   sysinfo(SI_SRPC_DOMAIN,buf,BUFSIZE);
   return rb_str_new2(buf);

}

/* Not defined in 2.5.x */
#ifdef SI_DHCP_CACHE
static VALUE uname_dhcp_cache()
{
   char buf[BUFSIZE];
   sysinfo(SI_DHCP_CACHE,buf,BUFSIZE);
   return rb_str_new2(buf);
}
#endif
#endif

#if defined(__MACH__) && defined(__APPLE__)
static VALUE uname_model()
{
   char model[BUFSIZ];
   getmodel(model, sizeof(model));
   return rb_str_new2(model);
}
#endif

#if defined(__hpux)
static VALUE uname_id()
{
   struct utsname u;
   uname(&u);
   return rb_str_new2(u.__idnumber);
}
#endif

void Init_uname()
{
   VALUE sys_mSys, cUname;

   sys_mSys = rb_define_module("Sys");
   cUname = rb_define_class_under(sys_mSys, "Uname", rb_cObject);

   rb_define_singleton_method(cUname, "sysname", uname_sysname, 0);
   rb_define_singleton_method(cUname, "nodename",uname_nodename,0);
   rb_define_singleton_method(cUname, "machine", uname_machine, 0);
   rb_define_singleton_method(cUname, "version", uname_version, 0);
   rb_define_singleton_method(cUname, "release", uname_release, 0);
   rb_define_singleton_method(cUname, "uname", uname_uname_all, 0);

#ifdef HAVE_SYS_SYSTEMINFO_H
   rb_define_singleton_method(cUname, "architecture", uname_architecture, 0);
   rb_define_singleton_method(cUname, "platform", uname_platform, 0);
   rb_define_singleton_method(cUname, "hw_provider", uname_hw_provider, 0);
   rb_define_singleton_method(cUname, "hw_serial_number", uname_hw_serial, 0);
   rb_define_singleton_method(cUname, "srpc_domain", uname_srpc_domain, 0);
#ifdef SI_ISALIST
   rb_define_singleton_method(cUname, "isa_list", uname_isalist, 0);
#endif
#ifdef SI_DHCP_CACHE
   rb_define_singleton_method(cUname, "dhcp_cache", uname_dhcp_cache, 0);
#endif
#endif

#if defined(__MACH__) && defined(__APPLE__)
   rb_define_singleton_method(cUname, "model", uname_model, 0);
#endif

#if defined(__hpux)
   rb_define_singleton_method(cUname, "id", uname_id, 0);
#endif

   sUname = rb_struct_define("UnameStruct","sysname","nodename",
      "machine","version","release",
#ifdef HAVE_SYS_SYSTEMINFO_H
      "architecture","platform",
#endif

#if defined(__MACH__) && defined(__APPLE__)
      "model",
#endif

#if defined(__hpux)
      "id",
#endif
   0);

   rb_define_const(cUname,"VERSION",rb_str_new2(SYS_UNAME_VERSION));
}

#ifdef __cplusplus
}
#endif
