/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include <squirrel.h>
#include <sqstdio.h>
#include <sqstdaux.h>

#include "../Globals.h"
#include "../Group.h"
#include "../RTS.h"
#include "../World.h"
#include "ScriptManager.h"
#include "SqExpose.h"
#include "SqWrapper.h"

#include <stdexcept>
#include <iterator>

using std::list;
using std::runtime_error;
using std::wstring;
using std::vector;

void ScriptManager::set_mission_script(const wstring& filename) {
	HSQUIRRELVM vm = setup_vm();
	vms.push_back(SquirrelVM(vm, 0));

	Scripting::register_mission_functions(vm);
	setup_vm_for_ai(vm, vms.back(), 0);

	run_script(vm, filename);
}

void ScriptManager::add_ai_script(const wstring& filename, int side) {
	HSQUIRRELVM vm = setup_vm();
	vms.push_back(SquirrelVM(vm, side));

	setup_vm_for_ai(vm, vms.back(), side);

	run_script(vm, filename);
}

void ScriptManager::setup_vm_for_ai(HSQUIRRELVM vm, SquirrelVM& vm_container, int side) {
	Scripting::register_ai_functions(vm);
	Scripting::expose_squads(vm, side, L"my_squads");
	vm_container.squad_arrays.push_back(SquadArray(L"my_squads", side));
	Scripting::expose_squads(vm, 0, L"player_squads");
	vm_container.squad_arrays.push_back(SquadArray(L"player_squads", 0));

	for (int i = 1; i != sides.size(); ++i) {
		if (i == side)
			continue;
		wchar_t array_name[20];
		swprintf(array_name, 20, L"side_%d_squads", i);
		Scripting::expose_squads(vm, i, array_name);
		vm_container.squad_arrays.push_back(SquadArray(array_name, i));
	}

	sq_pushroottable(vm);

	sqstd_register_iolib(vm);

	//push value of side as variable "my_side"
	sq_pushstring(vm, L"my_side", -1);
	sq_pushinteger(vm, side);
    sq_createslot(vm, 1);
}

HSQUIRRELVM ScriptManager::setup_vm() {
	//cannot do all this in SquirrelVM constructor because of silly way things get created multiple times during push_back
	HSQUIRRELVM vm = sq_open(1024);
	sqstd_seterrorhandlers(vm);
	return vm;
}

void ScriptManager::run_script(HSQUIRRELVM vm, const wstring& filename) {	
	if (!SQ_SUCCEEDED(sqstd_loadfile(vm, filename.c_str(), true))) {
		//assume squirrel has used printfunc to set the error string
		throw runtime_error(wstring_to_string(global_error_string));
	}

	sq_pushstring(vm, filename.c_str(), -1);
	sq_get(vm, -2);
	sq_pushroottable(vm);
	sq_call(vm, 1, false, true);
}

void ScriptManager::shutdown() {
	for (vector<SquirrelVM>::iterator iter = vms.begin(); iter != vms.end(); ++iter) {
		for (int i = 0; i != iter->squad_arrays.size(); ++i)
			Scripting::delete_script_squads(iter->vm, iter->squad_arrays[i].name);
		//FIXME could pop everything off stack but probably doesn't matter
		sq_close(iter->vm);
	}

	vms.clear();
}

void ScriptManager::run() {
	for (vector<SquirrelVM>::iterator iter = vms.begin(); iter != vms.end(); ++iter) {
		if (iter->wakeup_time != -1 && world.frame_counter >= iter->wakeup_time) {
			if (iter == vms.begin()) //mission script
				//important to do this before we wakeup the vm
				RTS::PortraitText::end_current_comm();
			sq_wakeupvm(iter->vm, false, false, true);
		}
	}

	SquirrelVM& mission_vm = vms[0];
	if (sq_getvmstate(mission_vm.vm) == SQ_VMSTATE_SUSPENDED) {
		switch (mission_vm.wait_event) {
		case WET_TIME:
			break;

		//PortraitText fires a wakeup event
		case WET_PORTRAIT_FINISHED:
			break;

		case WET_ALL_BIG_SELECTED: {
			bool all_big = true;
			for (vector<Squadron>::iterator squad = sides[0].squadrons.begin(); squad != sides[0].squadrons.end(); ++squad) {
				if (!squad->is_small() && !squad->get_selected()) {
					all_big = false;
					break;
				}
			}

			if (all_big) {
				mission_vm.wakeup_time = world.frame_counter;
				wakeup_reason = mission_vm.wait_event;
			}
			}
			break;

		case WET_NUMBER_HOTKEYED:
			for (int i = 0; i != sides[0].squadrons.size(); ++i) {
				if (sides[0].squadrons[i].get_hot_key() != 0) {
					mission_vm.wakeup_time = world.frame_counter;
					wakeup_reason = mission_vm.wait_event;
					break;
				}
			}
			break;

		case WET_SHIPS_MOVING_DOWN: {
			const AICommands* p_commands = sides[0].squadrons[0].get_the_commands();
			if (p_commands->aim_prop_y > 0) {
				mission_vm.wakeup_time = world.frame_counter;
				wakeup_reason = mission_vm.wait_event;
			}
			}
			break;

		//Missions fires a wakeup event
		case WET_MISSIONS_OPEN:
			break;

		case WET_TWO_FIGHTERS_SELECTED: {
			int selected_fighters = 0;
			for (list<int>::iterator which = sides[0].highlighted_squads.begin(); which != sides[0].highlighted_squads.end(); ++which) {
				if (sides[0].squadrons[*which].get_type() == UT_FIGHTER)
					++selected_fighters;
				else {
					selected_fighters = 0;
					break;
				}
			}

			if (selected_fighters == 2) {
				mission_vm.wakeup_time = world.frame_counter;
				wakeup_reason = mission_vm.wait_event;
			}
			}
			break;
		
		//TargetMove fires a wakeup event
		case WET_RECON_TARGET_OPEN:
			break;

		//Side fires a wakeup event
		case WET_RECON_LAUNCHED:
			break;

		//squirrel ai scripts can fire wakeup events
		case WET_CUSTOM_AI_TASK_0:
		case WET_CUSTOM_AI_TASK_1:
		case WET_CUSTOM_AI_TASK_2:
		case WET_CUSTOM_AI_TASK_3:
		case WET_CUSTOM_AI_TASK_4:
			break;
		}
	}
}

void ScriptManager::delete_squad(int side, int squad) {
	for (vector<SquirrelVM>::iterator iter = vms.begin(); iter != vms.end(); ++iter) {
		for (int i = 0; i != iter->squad_arrays.size(); ++i) {
			if (iter->squad_arrays[i].my_side == side) {
				Scripting::delete_squad(iter->vm, side, squad, iter->squad_arrays[i].name);
				break;
			}
		}
	}
}

void ScriptManager::set_wakeup_event(WakeEventType type, int frames) {
	vms[0].wait_event = type;
	if (frames == -1)
		vms[0].wakeup_time = -1;
	else
		vms[0].wakeup_time = world.frame_counter + frames;
}

void ScriptManager::fire_wakeup_event(WakeEventType type) {
	//wakeup fired when PortraitText closed, but on shutdown vm vector is cleared before window vector
	if (!vms.size())
		return;

	if (vms[0].wait_event == type) {
		vms[0].wakeup_time = world.frame_counter;
		wakeup_reason = type;
	} else if (type == WET_ALL_BIG_SHIPS_DEAD
	|| type == WET_CUSTOM_AI_TASK_0
	|| type == WET_CUSTOM_AI_TASK_1
	|| type == WET_CUSTOM_AI_TASK_2
	|| type == WET_CUSTOM_AI_TASK_3
	|| type == WET_CUSTOM_AI_TASK_4) {
		vms[0].wakeup_time = world.frame_counter;
		wakeup_reason = type;
	}
}

void ScriptManager::suspend_ai_script(HSQUIRRELVM vm) {
	for (vector<SquirrelVM>::iterator iter = vms.begin(); iter != vms.end(); ++iter) {
		if (iter->vm == vm) {
			iter->wakeup_time = world.frame_counter + stagger_frames;
			break;
		}
	}
}

