/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "DragWindow.h"
#include "Font.h"
#include "Globals.h"
#include "Group.h"
#include "Display.h"
#include "LoadGameData.h"
#include "Projectile.h"
#include "SettingsStruct.h"
#include "Side.h"
#include "Sound.h"
#include "Terrain.h"
#include "World.h"

using std::list;
using std::map;
using std::runtime_error;
using std::string;
using std::wstring;
using std::vector;

//various stuff probably already set by hovering mission select menu but it won't have been if playing with -mission
//so reload it all anyway
void World::load_mission(const string& mission_file) {
	free_title_images();
	sound.end_music();

	mission_folder_name = mission_file;
	mission_folder = "missions/" + mission_file + "/";
	FileReader the_file(world.mission_folder + "mission.dat");

	mission_name = string_to_wstring(the_file.get_string_after_colon());
	mission_number = the_file.get_int_after_colon();
	mission_briefing = string_to_wstring(the_file.get_string_after_colon());
	width = world.height = the_file.get_int_after_colon();
	viewx = the_file.get_int_after_colon();
	viewy = the_file.get_int();

	//music track list
	the_file.go_in_brackets();
	while (!the_file.on_close_bracket()) {
		string file = the_file.get_string_after_colon();
		if (gs_to != GST_MISSION_EDITOR)
			sound.load_song(file);
	}

	//background image
	string tmp_str = the_file.get_string_after_colon();
	string full_name = world.mission_folder + "images/" + tmp_str;
	if (!does_file_exist(full_name))
		full_name = "images/" + tmp_str;
	wstring full_name_w = string_to_wstring(full_name);

	terrain_pictures[full_name_w] = ScaledPic();
	terrain_pictures[full_name_w].load(full_name, true);
	background_pic = terrain_pictures[full_name_w];

	//foreground images
	the_file.go_in_brackets();
	while (!the_file.on_close_bracket()) {
		int x = the_file.get_int_after_colon();
		int y = the_file.get_int();
		string tmp_str = the_file.get_string();
		string full_name = world.mission_folder + "images/" + tmp_str;
		if (!does_file_exist(full_name))
			full_name = "images/" + tmp_str;

		wstring full_name_w = string_to_wstring(full_name);
		if (terrain_pictures.find(full_name_w) == terrain_pictures.end()) {
			terrain_pictures[full_name_w] = ScaledPic();
			terrain_pictures[full_name_w].load(full_name);
		}
		foreground_pics.push_back(ForegroundPic(x, y, terrain_pictures[full_name_w]));
	}
	
	//fleets
	the_file.go_in_brackets();	
	
	int n_sides = 0;
	while (!the_file.on_close_bracket()) {
		int flag = the_file.get_int_after_colon();
		string side_name = the_file.get_string();
		sides.push_back(Side(string_to_wstring(side_name), flag));
		sides.back().load_data();
		++n_sides;
	}

	if (n_sides > max_players)
		throw runtime_error("Missions can have a maximum of 6 sides");

	if (gs_to != GST_MISSION_EDITOR) {
		sound.set_music_volume(global_settings.music_volume);
		sound.start_music();
	}
}

void World::init() {
	frame_counter = 0;
	last_ai_time = now;
	last_scroll_time = now;
	view_side = -1;
	drag_box_start.x = -1;
	radar_sharing = true;
	ion_alpha = -1;

	paused = false;
	paused_by_script = false;
	cut_scene_on = false;
	script_anim = ANIM_NONE;
	mission_completed = false;
	last_alert_sound = 0;

	for (int i = 0; i != max_players; ++i)
		rem_side_dead[i] = false;

	//this includes setting up squadrons with their pointers, so must come before Side::init
	distribute_ai_frames();	

	for (int i = 0; i != sides.size(); ++i)
		sides[i].init();

	setup_terrain();
	set_starting_positions();

	if (gs_to != GST_MISSION_EDITOR) {
		script_manager.set_mission_script(string_to_wstring(mission_folder + "mission.nut"));
		//cannot happen in side::init because everything else must be inited before starting ai
		for (int i = 1; i != sides.size(); ++i)
			script_manager.add_ai_script(string_to_wstring(world.mission_folder) + sides[i].name + L".nut", i);
	}
}

void World::shutdown() {
	projectiles.clear();
	terrain_tiles.clear();
	update_interval = standard_interval;
	clear_sides();

	for (map<wstring, ScaledPic>::iterator iter = terrain_pictures.begin(); iter != terrain_pictures.end(); ++iter)
		glSDL_FreeSurface(iter->second.pic);
	terrain_pictures.clear();
	background_pic.pic = 0;
	foreground_pics.clear();

	old_selected_squads.clear();

	if (gs_current != GST_MISSION_EDITOR)
		script_manager.shutdown();
}

void World::mouse_d(Uint8 button, Uint16 x, Uint16 y) {
	if ((paused && gs_current != GST_MISSION_EDITOR) || cut_scene_on)
		return;

	if (button == SDL_BUTTON_LEFT) {
		drag_box_start.x = x;
		drag_box_start.y = y;
		//for single clicks we need to provide a box because mouse_m never called and even if it was w/h would be 0
		drag_box.x = x;
		drag_box.y = y;
		drag_box.w = 1;
		drag_box.h = 1;
	} else if (button == SDL_BUTTON_RIGHT) {
		CoordsFloat center = sides[0].work_out_selected_center();
		for (int i = 0; i != sides.size(); ++i) {
			for (int j = 0; j != sides[i].groups.size(); ++j) {
				if (gs_current == GST_BATTLE)
					sides[i].groups[j].mouse_d(x, y, center);
				else
					sides[i].groups[j].mouse_d_set_pos(x, y, center);
			}
		}
	}
}

void World::mouse_u(Uint8 button, Uint16 x, Uint16 y) {
	if (paused && gs_current != GST_MISSION_EDITOR || cut_scene_on)
		return;

	if (button == SDL_BUTTON_LEFT) {
		if (drag_box_start.x != -1) {
			//iterate through sides in reverse order so player side always gets selected last and hence it will
			//be computer selection unit that gets removed in case of conflict, not other way round
			for (int i = sides.size() - 1; i != -1; --i) {
				for (int j = 0; j != sides[i].squadrons.size(); ++j)
					sides[i].squadrons[j].drag_select(drag_box);
			}
		}
		drag_box_start.x = -1;
	}
}

void World::mouse_m(Uint8 state, Uint16 x, Uint16 y) {
	if (paused && gs_current != GST_MISSION_EDITOR || cut_scene_on)
		return;

	if (drag_box_start.x != -1) {
		drag_box.w = abs(static_cast<Uint16>(drag_box_start.x) - x);
		drag_box.h = abs(static_cast<Uint16>(drag_box_start.y) - y);
		drag_box.x = std::min(static_cast<Uint16>(drag_box_start.x), x);
		drag_box.y = std::min(static_cast<Uint16>(drag_box_start.y), y);
	}
}

void World::keyboard(SDL_keysym& keysym) {
	switch(keysym.sym) {
	case SDLK_p:
		if (!paused)
			pause(false);
		else
			unpause(false);
		break;
	
	case SDLK_o:
		if (!global_settings.debug)
			break;
		if (world.update_interval != standard_interval)
			set_update_interval(standard_interval);
		else
			set_update_interval(0);
		break;

	default:
		for (int i = 0; i != sides[0].squadrons.size(); ++i)
			sides[0].squadrons[i].keyboard(keysym);
		break;
	}
}

void World::play_sound(float x, float y, SoundEffect which) {
	int intx = static_cast<int>(x);
	int inty = static_cast<int>(y);

	if (viewx + global_settings.screen_width > intx
	&& viewx < intx
	&& viewy + global_settings.screen_height > inty
	&& viewy < inty)
		sound.play_sound(which);
}

void World::play_sound(SDL_Rect rect, SoundEffect which) {
	if (test_overlap(rect, viewable_screen_rect))
		sound.play_sound(which);
}

void World::pause(bool by_script) {
	paused = true;
	if (by_script)
		paused_by_script = true;
	else
		sound.pause_sound();
}

void World::unpause(bool by_script) {
	if (by_script)
		paused_by_script = false;
	else
		sound.resume_sound();

	if (!by_script && paused_by_script)
		return;

	if (update_interval < max_world_update_interval)
		paused = false;
}

/*
All ai is staggered over 10 frames.
In the first frame we run Squirrel AI, then group AI is distributed over the next 9 frames
*/
void World::distribute_ai_frames() {
	int num_groups = 0;

	for (int i = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			num_groups++;
	}

	int groups_per_frame = num_groups / (stagger_frames - 1);

	//ensure scripts all start within frames 1-9 (first frame is for squirrel)
	//also ensure that groups_per_frame is not 0 (i.e. if there are less than stagger_frames groups)
	while ((stagger_frames - 1) * groups_per_frame < num_groups)
		++groups_per_frame;

	//0 is for squirrel
	int current_open_frame = 1;

	vector<CoordsInt> done_groups;

	while (done_groups.size() != num_groups) {
		int ran_side = rand() % sides.size();
		int ran_group = rand() % sides[ran_side].groups.size();
		CoordsInt this_group(ran_side, ran_group);

		if (find(done_groups.begin(), done_groups.end(), this_group) != done_groups.end())
			continue;

		sides[ran_side].groups[ran_group].init(current_open_frame);
		done_groups.push_back(this_group);

		if (done_groups.size() % groups_per_frame == 0)
			++current_open_frame;
	}
}

//FIXME desperately try to optimise anywhere possible, this function takes up a huge amount of CPU time (though
//at least some of what I do is done automatically when you compile with optimization)
void World::setup_terrain() {
	//setup terrain tile vector
	int terrain_width = world.width / terrain_tile_dim;
	int terrain_height = world.height / terrain_tile_dim;

	terrain_tiles.resize(terrain_width, vector<TerrainTile>(terrain_height));
		
	float x = 0;
	float y = 0;

	//- use some cached values for end values rather than working out size on each loop
	//- use references to elements rather than constantly using operator []
	for (int i = 0; i != terrain_width; ++i) {
		vector<TerrainTile>& current_column =  world.terrain_tiles[i];
		for (int j = 0; j != terrain_height; ++j) {
			TerrainTile& current_tile = current_column[j];
			current_tile.x = x;
			current_tile.y = y;
			current_tile.int_x = static_cast<int>(x);
			current_tile.int_y = static_cast<int>(y);

			y += terrain_tile_dim;

			if (y == world.height) {
				y = 0;
				x += terrain_tile_dim;
			}
		}
	}
}

void World::set_starting_positions() {
	//start off by placing everyone outside world so we don't get false collisions with as yet unplaced groups
	for (int i = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
  			sides[i].groups[j].set_pos(static_cast<float>(world.width), static_cast<float>(world.height));
	}
	
	for (int i = 0; i != sides.size(); ++i) {
 		vector<bool> placed_groups(sides[i].groups.size(), false);
 		
		for (int j = 0; j != sides[i].groups.size(); ++j) {
	   	if (sides[i].groups[j].go_to_start_coords())
         	placed_groups[j] = true;
    	}

		if (gs_to != GST_MISSION_EDITOR) {
    		for (int j = 0; j != placed_groups.size(); ++j) {
      			if (placed_groups[j] == false) {
					string error_msg = wstring_to_string(sides[i].name) + " have illegally positioned units";
					throw runtime_error(error_msg.c_str());
				}
			}
		}
	}
}

void World::run() {
	int x, y;
	SDL_GetMouseState(&x, &y);
	scroll(x, y);
	if (!mouse_over_ui)
		update_cursor_type(x, y);

	if (paused_by_script)
		script_manager.run();
	else if (now - last_ai_time > update_interval && !paused) {
		script_manager.run();
		run_group_ai();
		run_move_commands();
		run_fire_commands();
		run_scanning();
		upkeep();

		last_ai_time = now;

		++frame_counter;
	}
}

bool World::update_squad_selection() {
	bool selected_squads_changed = false;
	bool side_one_selected = false;
	bool non_planet_selected = false;

	for (list<CoordsInt>::iterator iter = sides[0].selected_squads.begin(); iter != sides[0].selected_squads.end(); ++iter) {
		if (iter->x == 0)
			side_one_selected = true;
		if (sides[iter->x].squadrons[iter->y].get_type() != UT_PLANET)
			non_planet_selected = true;

		//efficiency
		if (side_one_selected && non_planet_selected)
			break;
	}
	
	//cannot select planet at same time as other groups
	if (non_planet_selected) {
		for (list<CoordsInt>::iterator iter = sides[0].selected_squads.begin(); iter != sides[0].selected_squads.end();) {
			if (sides[iter->x].squadrons[iter->y].get_type() == UT_PLANET)
				iter = sides[0].selected_squads.erase(iter);
			else
				++iter;
		}
	}

	//remove squads no longer selected
	for (list<CoordsInt>::iterator iter = old_selected_squads.begin(); iter != old_selected_squads.end(); ++iter) {
		list<CoordsInt>::iterator iter_two = find(sides[0].selected_squads.begin(), sides[0].selected_squads.end(), *iter);
		if (iter_two == sides[0].selected_squads.end()) {
			sides[iter->x].squadrons[iter->y].unselect();
			selected_squads_changed = true;
		}
	}

    //add newly selected squads
	for (list<CoordsInt>::iterator iter = sides[0].selected_squads.begin(); iter != sides[0].selected_squads.end(); ++iter) {
		list<CoordsInt>::iterator iter_two = find(old_selected_squads.begin(), old_selected_squads.end(), *iter);
		if (iter_two == old_selected_squads.end()) {
			sides[iter->x].squadrons[iter->y].select();
			selected_squads_changed = true;
		}
	}

	old_selected_squads = sides[0].selected_squads;
	return selected_squads_changed;
}

void World::update_cursor_type(int x, int y) {
	current_cursor_type = GENPIC_CURSOR;
	bool our_bigship_selected = false;

	for (int i  = 0; i != sides[0].squadrons.size(); ++i) {
		if (sides[0].squadrons[i].get_selected() && sides[0].squadrons[i].is_big_ship()) {
			our_bigship_selected = true;
			break;
		}
	}

	if (our_bigship_selected) {
		current_cursor_type = GENPIC_CURSORMOVE;

		for (int i  = 1; i != sides.size(); ++i) {
			if (sides[i].my_flag == sides[0].my_flag)
				continue;
			for (int j  = 0; j != sides[i].groups.size(); ++j) {
				if (sides[0].scanned_groups[i][j] && sides[i].groups[j].is_big_ship() && sides[i].groups[j].check_for_cursor(x, y)) {
					current_cursor_type = GENPIC_CURSORATTACK;
					break;
				}
			}

			if (current_cursor_type == GENPIC_CURSORATTACK)
				break;
		}
	}
}

void World::upkeep() {
	for (int i  = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].upkeep();
	}

	for (int i  = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].dead_upkeep();
	}

	for (int i  = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].squadrons.size(); ++j)
			sides[i].squadrons[j].upkeep();
	}

	if (!mission_completed) {
		for (int i  = 0; i != sides.size(); ++i) {
			if (!sides[i].get_start_big_ships())
				continue;
			int side_total_big = sides[i].get_total_big_ships();
			//if just one group may well be just a neutral planet
			if (!side_total_big) {
				if (i == 0) {
					my_windows.push_back(GenWindow(RTS_MISSION_FAILED, L"All ships lost"));
				} else if (rem_side_dead[i] == false) {
					bool side_dead = true;
					//don't wakeup if the fleet still has bombers in the air
					for (int j = 0; j != sides[i].squadrons.size(); ++j) {
						if (sides[i].squadrons[j].get_type() == UT_BOMBER
						&& sides[i].squadrons[j].get_alive()
						&& !sides[i].squadrons[j].both_in_hangar()) {
							side_dead = false;
							break;
						}
					}

					if (side_dead) {
						script_manager.fire_wakeup_event(WET_ALL_BIG_SHIPS_DEAD);
						script_manager.which_side_dead = i;
						rem_side_dead[i] = true;
					}
				}
			}
		}
	}
}

void World::draw() {
	follow_view_center();
	check_view_pos();

	draw_terrain();

	for (int i = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].set_screen_rect();
	}

	for (int i = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].draw_self_back_back();
	}

	for (int i = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].draw_self_back();
	}

	for (int i = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].draw_self_middle();
	}

	for (int i = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].draw_self_front();
	}

	for (list<Projectile>::iterator iter = projectiles.begin(); iter != projectiles.end(); ++iter)
		iter->draw_self();

	for (int i = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].draw_big_laser();
	}

	for (int i = 0; i != foreground_pics.size(); ++i)
		foreground_pics[i].draw_self();

	if (ion_alpha != -1)
		draw_ion_pulse();

	draw_fog_of_war();

	for (int i = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].draw_bound();
	}

	//drag box
	if (drag_box_start.x != -1)
		draw_rect_border(drag_box, standard_colors.white, 1);

	if (paused) {
		static wstring text = L"paused";
		static int text_width = normal_font.get_width(text);
		normal_font.render(global_settings.screen_width - text_width - 20, 20, text);
	} else if (cut_scene_on) {
		static wstring text = L"cut scene";
		static int text_width = normal_font.get_width(text);
		normal_font.render(global_settings.screen_width - text_width - 20, 20, text);
	} else if (update_interval < standard_interval) {
		static wstring text = L"time acceleration activated";
		static int text_width = normal_font.get_width(text);
		normal_font.render(global_settings.screen_width - text_width - 20, 20, text);
	}
}

void World::draw_terrain() {
	//work out top corner of background source rect (parallax scrolling)
	//src_rect.x = (world.viewx / (world.width - global_settings.screen_width)) * (pic->w - global_settings.screen_width);
	//src_rect.y = (world.viewy / (world.height - global_settings.screen_height)) * (pic->h - global_settings.screen_height);
	SDL_Rect src_rect;
	src_rect.x = world.viewx * (world.background_pic.w - global_settings.screen_width) / (world.width - global_settings.screen_width);
	src_rect.y = world.viewy * (world.background_pic.h - global_settings.screen_height) / (world.height - global_settings.screen_height);
	src_rect.x *= static_cast<float>(world.background_pic.pic->w) / world.background_pic.w;
	src_rect.y *= static_cast<float>(world.background_pic.pic->h) / world.background_pic.h;
	src_rect.w = world.background_pic.pic->w * global_settings.screen_width / world.background_pic.w;
	src_rect.h = world.background_pic.pic->h * global_settings.screen_height / world.background_pic.h;

	display.blt_part(world.background_pic.pic, src_rect, viewable_screen_rect, GL_EFFECT_SCALED);

	for (int i = 0; i != foreground_pics.size(); ++i)
		foreground_pics[i].draw_self();
}

void World::draw_fog_of_war() {
	int start_x = viewx / terrain_tile_dim;
	int start_y = viewy / terrain_tile_dim;

	int end_x = ((viewx + global_settings.screen_width) / terrain_tile_dim) + 1;
	if (end_x > world.terrain_tiles.size())
		end_x = world.terrain_tiles.size();
	int end_y = ((viewy + global_settings.screen_height) / terrain_tile_dim) + 1;
	if (end_y > world.terrain_tiles[0].size())
		end_y = world.terrain_tiles[0].size();

	for (int x = start_x; x != end_x; ++x) {
		for (int y = start_y; y != end_y; ++y) {
			if (terrain_tiles[x][y].num_seeing[0] < 1) {
				SDL_Rect dst = {terrain_tiles[x][y].int_x - world.viewx, terrain_tiles[x][y].int_y - world.viewy, terrain_tile_dim, terrain_tile_dim};
				//play with combining tiles together in a sort of RLE stylee to reduce number of calls to 
				//blt_fill, though doesn't seem to make any difference at all to speed
				while (y + 1 != end_y && terrain_tiles[x][y + 1].num_seeing[0] < 1) {
					dst.h += terrain_tile_dim;
					++y;
				}
				display.blt_fill(dst, standard_colors.black, 0.7f);
			}
		}
	}
}

void World::scroll(int x, int y) {
	//scrolling via mouse, also used to break out of follow view
	if (now - last_scroll_time > scroll_interval) {
		if (x > global_settings.screen_width - 8 || display.keyboard[SDLK_RIGHT]) {
			viewx += screen_move_speed;
			view_side = -1;
		}
		if (x < 8  || display.keyboard[SDLK_LEFT]) {
			viewx -= screen_move_speed;
			view_side = -1;
		}
		if (y > global_settings.screen_height - 8 || display.keyboard[SDLK_DOWN]) {
			viewy += screen_move_speed;
			view_side = -1;
		}
		if (y < 8 || display.keyboard[SDLK_UP]) {
			viewy -= screen_move_speed;
			view_side = -1;
		}

		last_scroll_time = now;
	}
}

void World::set_update_interval(int new_value) {
	update_interval = new_value;
	
	if (update_interval < 0)
		update_interval = 0;
		
	if (update_interval == max_world_update_interval)
		paused = true;
	else
		paused = false;
		
	if (update_interval > max_world_update_interval)
		update_interval = max_world_update_interval;

	if (update_interval > standard_interval - 10 && update_interval < standard_interval + 10)
		update_interval = standard_interval;
}

void World::run_scanning() {
	cap_found = false;
	frigate_found = false;
	bomber_found = false;

	//ai sides don't do scanning, they just treat everyone as always scanned
	sides[0].blank_scanned_groups();
	for (int i = 1; i != sides.size(); ++i)
		sides[i].check_if_scanned();

	if (frame_counter - last_alert_sound > min_alert_gap) {
		if (!rem_cap_found && cap_found) {
			sound.play_sound(SE_CAP_SPOTTED);
			last_alert_sound = frame_counter;
		} else if (!rem_frigate_found && frigate_found) {
			sound.play_sound(SE_FRIGATE_SPOTTED);
			last_alert_sound = frame_counter;
		} else if (!rem_bomber_found && bomber_found) {
			sound.play_sound(SE_ENEMY_BOMBERS);
			last_alert_sound = frame_counter;
		}
	}

	rem_cap_found = cap_found;
	rem_frigate_found = frigate_found;
	rem_bomber_found = bomber_found;
}

void World::run_group_ai() {
	for (int i  = 0; i != sides.size(); ++i)
		sides[i].run_ai();
}

void World::run_move_commands() {
	for (int i  = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].move();
	}

	for (list<Projectile>::iterator iter = projectiles.begin(); iter != projectiles.end();) {
		if (!iter->move())
			iter = projectiles.erase(iter);
		else
			++iter;
	}
}

void World::run_fire_commands() {
	for (int i  = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].run_fire_commands();
	}
}

void World::check_view_pos() {
	if (viewx > world.width - global_settings.screen_width)
		viewx = world.width - global_settings.screen_width;
	if (viewx < 0)
		viewx = 0;
	if (viewy > world.height - global_settings.screen_height)
		viewy = world.height - global_settings.screen_height;
	if (viewy < 0)
		viewy = 0;
}

void World::do_ion_pulse() {
	static const float start_ion_alpha = 0.2f;

	ion_alpha = start_ion_alpha;
	anim_timer = world.frame_counter;

	sound.play_sound(SE_DROP_SHIP_LAUNCH);
}

void World::draw_ion_pulse() {
	static const int frames_per_alpha = 1;

	int min_x = world.width;
	int max_x = 0;
	int min_y = world.height;
	int max_y = 0;

	for (int i = 0; i != foreground_pics.size(); ++i) {
		min_x = std::min(foreground_pics[i].rect.x, min_x);
		max_x = std::max(foreground_pics[i].rect.x + foreground_pics[i].rect.w, max_x);
		min_y = std::min(foreground_pics[i].rect.y, min_y);
		max_y = std::max(foreground_pics[i].rect.y + foreground_pics[i].rect.h, max_y);
	}

	static SDL_Rect ion_rect = {min_x, min_y, max_x - min_x, min_y + max_y - min_y};
	static SDL_Rect screen_rect;
	screen_rect.x = static_cast<Uint16>(ion_rect.x - world.viewx);
	screen_rect.y = static_cast<Uint16>(ion_rect.y - world.viewy);
	screen_rect.w = static_cast<Uint16>(ion_rect.w);
	screen_rect.h = static_cast<Uint16>(ion_rect.h);

	display.blt_fill(screen_rect, standard_colors.laser_blue, ion_alpha);

	if (world.frame_counter - anim_timer > frames_per_alpha) {
		ion_alpha -= 0.01f;
		anim_timer = world.frame_counter;
		if (!ion_alpha)
			ion_alpha = -1;
	}
}
