/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "Globals.h"
#include "SettingsStruct.h"
#include "Sound.h"
#include "World.h"

#include <string>

using std::runtime_error;
using std::string;
using std::wstring;
using std::map;

void Sound::init() {
	if (global_settings.disable_sound)
		return;

	if (SDL_InitSubSystem(SDL_INIT_AUDIO) == -1) {
		char error[120];
		sprintf(error, SDL_GetError());
		string error_str = error;
		error_str += ", continuing without initialising sound.";
		write_log(string_to_wstring(error_str));
		//set to false in SettingsStruct constructor
		global_settings.dont_write_sound = true;
		global_settings.disable_sound = true;
		return;
	}

	if (Mix_OpenAudio(MIX_DEFAULT_FREQUENCY, MIX_DEFAULT_FORMAT, 2, 1024) == -1) {
		char output[100];
		sprintf(output, Mix_GetError());
		throw runtime_error(output);
	}
	
	Mix_AllocateChannels(16);
	set_sound_volume(global_settings.sound_volume);
	set_music_volume(global_settings.music_volume);

	load_data();
}

void Sound::shutdown() {
	if (global_settings.disable_sound)
		return;

	end_music();
	end_sound();

	for (int i = 0; i != n_sounds; ++i)
		Mix_FreeChunk(sound_cache[i]);

	Mix_CloseAudio();
}

void Sound::end_sound() {
	Mix_HaltChannel(-1);
	Mix_FreeChunk(script_sound);
	script_sound = 0;
}


void Sound::update() {
	if (global_settings.disable_sound)
		return;

	if (b_script_sound && !Mix_Playing(15)) {
		b_script_sound = false;
		volume_reset_timer = now + VOLUME_RESET_TIME;
	}

	if (alert_sound_one != SE_NO_SOUND && !Mix_Playing(2))
		alert_sound_one = SE_NO_SOUND;
	if (alert_sound_two != SE_NO_SOUND && !Mix_Playing(3))
		alert_sound_two = SE_NO_SOUND;

	if (!volume_reset && !b_script_sound && now > volume_reset_timer) {
		Mix_VolumeMusic(global_settings.music_volume);
		volume_reset = true;
	}

	if (b_music_on && !Mix_PlayingMusic())
		next_music_track();
}

void Sound::play_sound(SoundEffect which) {
	if (global_settings.disable_sound)
		return;

	switch (which) {
	case SE_MENU_CLICK:
		if (!Mix_Playing(0))
			Mix_PlayChannel(0, sound_cache[which], 0);
		else if (!Mix_Playing(1))
			Mix_PlayChannel(1, sound_cache[which], 0);
		break;

	case SE_COORDINATES_SET:
	case SE_TARGET_ACQUIRED:
	case SE_BE_THERE:
	case SE_HEADING_FOR_HOME:
	case SE_MISSION_UNDER_WAY:
	case SE_CAP_SPOTTED:
	case SE_FRIGATE_SPOTTED:
	case SE_ENEMY_BOMBERS:
		//don't play more than one of the same alert sound simultaneously
		if (alert_sound_one == which || alert_sound_two == which)
			break;
		if (!Mix_Playing(2)) {
			Mix_PlayChannel(2, sound_cache[which], 0);
			alert_sound_one = which;
		} else if (!Mix_Playing(3)) {
			Mix_PlayChannel(3, sound_cache[which], 0);
			alert_sound_two = which;
		}
		break;

	case SE_SMALL_LASER:
	case SE_TWIN_LASER:
		if (!Mix_Playing(4))
			Mix_PlayChannel(4, sound_cache[which], 0);
		else if (!Mix_Playing(5))
			Mix_PlayChannel(5, sound_cache[which], 0);
		else if (!Mix_Playing(6))
			Mix_PlayChannel(6, sound_cache[which], 0);
		else if (!Mix_Playing(7))
			Mix_PlayChannel(7, sound_cache[which], 0);			
		break;

	case SE_MISSILE_LAUNCH:
	case SE_TORPEDO_EXPLOSION:
	case SE_MEDIUM_EXPLOSION:
	case SE_LARGE_EXPLOSION:
		if (!Mix_Playing(8))
			Mix_PlayChannel(8, sound_cache[which], 0);
		else if (!Mix_Playing(9))
			Mix_PlayChannel(9, sound_cache[which], 0);
		else if (!Mix_Playing(10))
			Mix_PlayChannel(10, sound_cache[which], 0);
		//we always want these sounds to be played - if no free channels at all, halt a channel
		else if (which == SE_MEDIUM_EXPLOSION || which == SE_LARGE_EXPLOSION) {
			Mix_HaltChannel(8);
			Mix_PlayChannel(8, sound_cache[which], 0);
		}
		break;
	
	case SE_LARGE_LASER:
	case SE_SS_EXPLOSION:
	case SE_NEUTRON_BOMB_EXPLOSION:
	case SE_DROP_SHIP_LAUNCH:
		if (!Mix_Playing(11))
			Mix_PlayChannel(11, sound_cache[which], 0);
		else if (!Mix_Playing(12))
			Mix_PlayChannel(12, sound_cache[which], 0);
		else if (!Mix_Playing(13))
			Mix_PlayChannel(13, sound_cache[which], 0);
		else if (!Mix_Playing(14))
			Mix_PlayChannel(14, sound_cache[which], 0);
		break;
	}
}

void Sound::play_script_sound(const wstring& filename) {
	if (global_settings.disable_sound)
		return;

	if (Mix_Playing(15))
		Mix_HaltChannel(15);

	Mix_FreeChunk(script_sound);
	string full_dir = world.mission_folder + "sound/" + wstring_to_string(filename);
	script_sound = Mix_LoadWAV(full_dir.c_str());

	if (!script_sound) {
		string error_str = Mix_GetError();
		error_str = "Could not load sound file '" + full_dir + "': " + error_str; 
		throw runtime_error(error_str.c_str());
	}
    
	Mix_VolumeMusic(global_settings.music_volume / 2);
	volume_reset = false;
	Mix_PlayChannel(15, script_sound, 0);
	b_script_sound = true;
}

void Sound::stop_script_sound() {
	if (global_settings.disable_sound)
		return;

	Mix_HaltChannel(15);
}

void Sound::pause_sound() {
	if (global_settings.disable_sound)
		return;

	Mix_Pause(-1);
}

void Sound::resume_sound() {
	if (global_settings.disable_sound)
		return;

	Mix_Resume(-1);
}

bool Sound::script_sound_playing() {
	return b_script_sound;
}

void Sound::set_sound_volume(int volume) {
	if (global_settings.disable_sound)
		return;

	//some sounds are too loud otherwise
	Mix_Volume(-1, volume / 4);
	Mix_Volume(8, volume / 2);
	Mix_Volume(9, volume / 2);
	Mix_Volume(10, volume / 2);

	Mix_Volume(0, volume);
	Mix_Volume(1, volume);
	Mix_Volume(2, volume);
	Mix_Volume(3, volume);
	Mix_Volume(11, volume);
	Mix_Volume(12, volume);
	Mix_Volume(13, volume);
	Mix_Volume(15, volume);
}

void Sound::load_song(const string& filename) {
	string full_name = world.mission_folder + "music/" + filename;
	if (gs_to != GST_BATTLE || !does_file_exist(full_name))
		full_name = "music/" + filename;

	music_cache.push_back(Mix_LoadMUS(full_name.c_str()));

	if (!music_cache.back()) {
		string error_str = Mix_GetError();
		error_str = "Could not load music file '" + filename + "': " + error_str; 
		throw runtime_error(error_str.c_str());
	}
}

void Sound::start_music() {
	if (global_settings.disable_sound || !b_music_on || !music_cache.size())
		return;

	play_music();
}

void Sound::end_music() {
	Mix_HaltMusic();

	for (int i = 0; i != music_cache.size(); ++i)
		Mix_FreeMusic(music_cache[i]);
	music_cache.clear();
	current_song = 0;
}

void Sound::next_music_track() {
	if (global_settings.disable_sound || !b_music_on  || !music_cache.size())
		return;

	++current_song;
	if (current_song == music_cache.size())
		current_song = 0;

	play_music();
}

void Sound::set_music_volume(int volume) {
	if (global_settings.disable_sound)
		return;
	
	if (volume) {
		b_music_on = true;
		Mix_VolumeMusic(volume);
	} else {
		b_music_on = false;
		Mix_HaltMusic();
	}
}

//private

void Sound::load_data() {
	sound_cache[SE_NO_SOUND] = 0;

	load_sound(L"sound/coordinates_set.ogg", SE_COORDINATES_SET);
	load_sound(L"sound/target_acquired.ogg", SE_TARGET_ACQUIRED);
	load_sound(L"sound/be_there.ogg", SE_BE_THERE);
	load_sound(L"sound/heading_for_home.ogg", SE_HEADING_FOR_HOME);
	load_sound(L"sound/mission_under_way.ogg", SE_MISSION_UNDER_WAY);

	load_sound(L"sound/capital_spotted.ogg", SE_CAP_SPOTTED);
	load_sound(L"sound/frigate_spotted.ogg", SE_FRIGATE_SPOTTED);
	load_sound(L"sound/enemy_bombers.ogg", SE_ENEMY_BOMBERS);

	load_sound(L"sound/5989_user_56891b_drum72.wav", SE_MENU_CLICK);
	load_sound(L"sound/18378_inferno_ftlas.ogg", SE_SMALL_LASER);
	load_sound(L"sound/18382_inferno_hvylas.ogg", SE_TWIN_LASER);
	load_sound(L"sound/18389_inferno_medlas.ogg", SE_LARGE_LASER);
	load_sound(L"sound/18380_inferno_hvrl.ogg", SE_MISSILE_LAUNCH);
	load_sound(L"sound/18400_inferno_waterxplo.ogg", SE_TORPEDO_EXPLOSION);
	load_sound(L"sound/21410_Heigh_hoo_blow_short.ogg", SE_SS_EXPLOSION);
	load_sound(L"sound/18401_inferno_xlaz.ogg", SE_NEUTRON_BOMB_EXPLOSION);
	load_sound(L"sound/18394_inferno_rlhvyx.ogg", SE_DROP_SHIP_LAUNCH);
	load_sound(L"sound/19105_Jace_Ship_damaged_explosions_sparks_short.ogg", SE_MEDIUM_EXPLOSION);
	load_sound(L"sound/19105_Jace_Ship_damaged_explosions_sparks.ogg", SE_LARGE_EXPLOSION);
}

void Sound::load_sound(const std::wstring& filename, SoundEffect which) {
	sound_cache[which] = Mix_LoadWAV(wstring_to_string(filename).c_str());

	if (!sound_cache[which]) {
		string error_str = Mix_GetError();
		error_str = "Could not load sound file '" + wstring_to_string(filename) + "': " + error_str; 
		throw runtime_error(error_str.c_str());
	}
}

void Sound::play_music() {
	if (Mix_PlayMusic(music_cache[current_song], 1) == -1) {
		char output[100];
		sprintf(output, Mix_GetError());
		throw runtime_error(output);
	}
}

