/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "RTSUnit_Base.h"
#include "Globals.h"
#include "Group.h"
#include "Projectile.h"
#include "World.h"
#include "Display.h"
#include "Side.h"

#include <stdexcept>

using std::runtime_error;
using std::wstring;
using std::vector;

//static member:
vector<SDL_Surface*> SmallShip::explode_pics;

SmallShip::SmallShip(const wstring& i_name, int i_my_side, int i_my_group):
RTSUnit_Base(i_name, i_my_side, i_my_group) {
	load_data();
}

void SmallShip::type_dep_stats(FileReader& the_file) {
	if (small_type == WT_NONE)
		small_number = 0;
	else
		small_number = 1;
}

void SmallShip::select_small_targets(AICommands& the_commands, bool fire) {
	if (!small_number)
		return;

	if (fire)
		small_targets[0] = the_commands.move_target;
	else
		small_targets[0].x = -1;
}

void SmallShip::select_big_targets(AICommands& the_commands, bool fire) {
	if (fire)
		big_target = the_commands.fire_target;
	else
		big_target.x = -1;
}

void SmallShip::move(float distx, float disty) {
	if (extra_move_frames) {
		distx += extra_move_x;
		disty += extra_move_y;
		--extra_move_frames;
	}
		
	RTSUnit_Base::move(distx, disty);
}

void SmallShip::fire(AICommands& the_commands) {
	fire_small(the_commands);
	fire_big(the_commands);
}

void SmallShip::fire_small(AICommands& the_commands) {
	if (!small_number)
		return;

	//only run this function if we are moving towards an enemy group
	if (small_stage[0] != W_RELOADING && (the_commands.move_command != MC_MOVE_GROUP || small_targets[0] != the_commands.move_target || the_commands.b_inverse == true)) {
		small_stage[0] = W_READY;
		small_targets[0].x = -1;
		return;
	}

	//else if aiming time is up and we still have a valid target...
	if (small_stage[0] == W_FIRING) {
		if (sides[my_side].groups[my_group].get_speed_plane_matches_target()) {
			small_stage[0] = W_RELOADING;
			small_timer[0] = world.frame_counter;
            CoordsFloat center = get_center();
			world.projectiles.push_back(Projectile(center.x, center.y, the_commands.move_target, small_type, sides[my_side].laser_color));
		}
		//else we wait, and with no reloading are allowed to fire as soon as we are properly aligned
	} else if (small_stage[0] == W_READY && small_targets[0].x != -1) {
		//make last minute checks for validity of target
		if (!sides[the_commands.move_target.x].groups[the_commands.move_target.y].get_in_hangar()
		&& sides[the_commands.move_target.x].groups[the_commands.move_target.y].get_alive()
		&& the_commands.move_target_dist <= weapon_lookup[small_type].range)
			setup_small_for_firing(0);
	}
}

void SmallShip::fire_big(AICommands& the_commands) {
	//if they are now out of range or we have decided not to fire any more, stop
	if (big_stage != W_RELOADING && (the_commands.move_command != MC_MOVE_GROUP || big_target != the_commands.move_target || the_commands.b_inverse == true || !the_commands.ordered_to_fire)) {
		big_stage = W_READY;
		big_target.x = -1;
	}

	//else if aiming time is up...
	if (big_stage == W_FIRING) {
		//only fire if we are moving in the right plane (we may well not be if dogfighting)
		if (sides[my_side].groups[my_group].get_speed_plane_matches_target()) {
			big_stage = W_RELOADING;
			big_timer = world.frame_counter;
            CoordsFloat center = get_center();
			world.projectiles.push_back(Projectile(center.x, center.y, big_target, big_type, sides[my_side].laser_color));
			
			//note that if there are more infinite ammo big weapons made, this needs changing
			--big_ammo;

			world.script_manager.fire_wakeup_event(WET_ENEMY_UNDER_ATTACK);
		}
		//else we wait, and with no reloading are allowed to fire as soon as we are properly aligned, assuming still within range	
	} else if (big_stage == W_READY && big_target.x != -1 && big_ammo > 0) {
		//make last minute checks for validity of target
		if (!sides[the_commands.move_target.x].groups[the_commands.move_target.y].get_in_hangar()
		&& sides[the_commands.move_target.x].groups[the_commands.move_target.y].get_alive()
		&& the_commands.fire_target_dist <= weapon_lookup[big_type].range)
			setup_big_for_firing();
	}
}

void SmallShip::upkeep() {
	RTSUnit_Base::upkeep();

	if (armour_current < 1) {
		world.play_sound(myx, myy, SE_SS_EXPLOSION);
		alive = false;
		anim_timer = world.frame_counter;
		finished_anim = false;
		return;
	}

	if (small_number) {
		//check for small reloading
		if (small_stage[0] == W_RELOADING && world.frame_counter - small_timer[0] > weapon_lookup[small_type].reload)
			small_stage[0] = W_READY;

		//check for small aiming (time decided when we choose
		//a target)
		if (small_stage[0] == W_AIMING && world.frame_counter - small_timer[0] > small_aiming[0])
			small_stage[0] = W_FIRING;
	}

	if (sides[my_side].groups[my_group].get_in_hangar()) {
		if (armour_current < armour_max) {
			armour_current += equip_lookup[armour_name].recharge;
			
			if (armour_current > armour_max)
				armour_current = armour_max;
		}

		if (big_ammo < big_ammo_max && world.frame_counter - ammo_recharge_timer > equip_lookup[L"Ammo recharge"].recharge) {
			big_ammo++;
			ammo_recharge_timer = world.frame_counter;
		}
	}
}

void SmallShip::just_docked() {
	ammo_recharge_timer = world.frame_counter;
}

void SmallShip::explode() {
	for (int i = 0; i != explode_pics.size(); ++i) {
		if (world.frame_counter - anim_timer < frames_per_anim_frame * (i + 1)) {
			display.blt(explode_pics[i], screen_rect);
			return;
		}
	}

	finished_anim = true;
}

