/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/


#ifndef GUARD_RTSUnit_Base
#define GUARD_RTSUnit_Base

#include "Globals.h"
#include "RTSStructs.h"
#include "Stuff.h"

#include <vector>
#include <string>
#include <iterator>

class Group;
class FileReader;

class RTSUnit_Base {
public:
	RTSUnit_Base(const std::wstring& i_name, int i_my_side, int i_my_group);
	virtual ~RTSUnit_Base() {}

	void set_pos(float ix, float iy);

	virtual void select_small_targets(AICommands& the_commands, bool fire) {}
	virtual void select_big_targets(AICommands& the_commands, bool fire) {}

	virtual void select_small_targets(const AICommands& the_commands) {}
	virtual void select_big_targets(const AICommands& the_commands) {}

	virtual void move(float distx, float disty);
	void add_extra_move_frames();
	virtual void fire(AICommands& the_commands) {}
	virtual void fire() {}

	void been_hit(int power);

	virtual void upkeep();
	virtual void dead_upkeep() {}
	virtual void just_docked() {}
	virtual void explode() = 0;

	void draw_self();
	virtual void draw_big_laser() {}
	virtual void draw_bound(bool b_draw_weapon_range) {}

	CoordsFloat get_center() const {CoordsFloat tmp(myx + width / 2, myy + height / 2); return tmp;}

	virtual int get_capacity() const = 0;

	//standard is just width / 2 height / 2
	virtual CoordsInt get_weak_spot() const;

	std::wstring name;
	UnitType my_type;
	std::wstring pic_name;
	std::wstring blurb;

	int armour_current;
	int armour_max;
	int shield_current;
	int shield_max;
	int shield_recharge;
	int shield_timer;

	int small_number;
	WeaponType small_type;
	WeaponType big_type;

	int big_ammo;
	int big_ammo_max;
	int ammo_recharge_timer;

	std::wstring engine_name;
	std::wstring shield_name;
	std::wstring armour_name;

	float myx;
    float myy;
	int width;
	int height;

	SDL_Rect screen_rect;

	int extra_move_frames;
	float extra_move_x;
	float extra_move_y;

	int my_side;
	int my_group;

	int alive;
	bool b_flip;

protected:
	void load_data();
	virtual void type_dep_stats(FileReader& the_file) = 0;
	void load_picture();
	void setup_small_for_firing(int which_small);
	void setup_big_for_firing();

	//sses
	virtual void fire_small(AICommands& the_commands) {};
	virtual void fire_big(AICommands& the_commands) {};

	//big ships
	virtual void fire_small() {};
	virtual void fire_big() {};

	void init_small();

	std::vector<WeaponStage> small_stage;
	std::vector<int> small_timer;
	std::vector<int> small_aiming;
	std::vector<CoordsInt> small_targets;

	WeaponStage big_stage;
	int big_timer;
	int big_aiming;
	CoordsInt big_target;
	int big_target_unit;
	CoordsInt target_weak_spot;
	bool we_hit_with_big;

	int anim_timer;
	bool finished_anim;

	SDL_Surface* pic;
};

class SmallShip: public RTSUnit_Base {
public:
	SmallShip(const std::wstring& i_name, int i_my_side, int i_my_group);

	void select_small_targets(AICommands& the_commands, bool fire);
	void select_big_targets(AICommands& the_commands, bool fire);
	void move(float distx, float disty);
	void fire(AICommands& the_commands);
	void upkeep();
	void just_docked();
	void explode();
	int get_capacity() const {return 0;}

	static std::vector<SDL_Surface*> explode_pics;
	
protected:
	void type_dep_stats(FileReader& the_file);
	void fire_small(AICommands& the_commands);
	void fire_big(AICommands& the_commands);
};

class BigUnit: public RTSUnit_Base {
public:
	BigUnit(const std::wstring& i_name, int i_my_side, int i_my_group, bool face_left);

	void select_small_targets(const AICommands& the_commands);
	void select_big_targets(const AICommands& the_commands);
	void fire();
	void upkeep();
	void dead_upkeep();
	void draw_big_laser();
	void draw_bound(bool b_draw_weapon_range);
	void explode();
	int get_capacity() const {return capacity;}

	static std::vector<SDL_Surface*> big_explode_pics;

protected:
	void type_dep_stats(FileReader& the_file);
	void find_in_range(std::vector<CoordsInt>& in_range, CoordsFloat weapon_coords, float range, bool including_small_targets = true);
	void fire_small();
	void fire_big();

	CoordsInt get_weak_spot() const;
	void random_small_explosions();
	void bombardment_explosions();
	void launch_drop_ships();

	std::vector<CoordsInt> small_positions;
	CoordsInt big_position;
	int capacity;

	WeaponStage missile_stage;
	int missile_timer;
	bool have_missiles;
};

#endif
