/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "Projectile.h"
#include "Globals.h"
#include "Group.h"
#include "Inlines.h"
#include "Display.h"
#include "SettingsStruct.h"
#include "Side.h"
#include "World.h"

#include <stdexcept>
#include <cmath>

using std::runtime_error;
using std::vector;

Projectile::Projectile(float ix, float iy, CoordsInt target, WeaponType i_type, const OpenGLColor& i_color) {
	if (i_type == WT_TWIN || i_type == WT_HEAVY_TWIN || i_type == WT_PLATFORM)
		h_proj = new TwinLaserBolt(ix, iy, target, i_type, i_color);

	else if (i_type == WT_SMALL)
		h_proj = new SmallLaserBolt(ix, iy, target, i_type, i_color);

	else if (i_type == WT_MISSILE)
		h_proj = new Missile(ix, iy, target, i_type);

	else if (i_type == WT_TORPEDO)
		h_proj = new Torpedo(ix, iy, target, i_type);

	else if (i_type == WT_DROP_SHIP)
		h_proj = new DropShip(ix, iy, target, i_type);

	else throw runtime_error("Projectile constructor didn't recognise i_type");
}

Projectile::Projectile(float ix, float iy) {
	h_proj = new LaserExplosion(ix, iy);
}

////

Projectile_Base::Projectile_Base(float ix, float iy, CoordsInt target, WeaponType i_type, const OpenGLColor& i_color):
x(ix), y(iy), color(i_color), timer(world.frame_counter), my_type(i_type) {
	targetSide = target.x;
	targetGroup = target.y;

	my_speed = weapon_lookup[my_type].speed;
	my_length = weapon_lookup[my_type].length;

	sides[targetSide].groups[targetGroup].get_unit_target_info(target_info);

	target_unit = target_info.which_unit;

	we_hit = check_to_hit(weapon_lookup[my_type].accuracy);
	finished_explode = false;
}

Projectile_Base::Projectile_Base(float ix, float iy):
x(ix), y(iy) {
	finished_explode = false;
}

void Projectile_Base::predict_target() {
	/*
	Another way to do this would be:
	mx = m.x + c, so x = c / (m. - m)
	with x being time
	 
	int time = distance / (laserSpeed - target_info.speed);
	 
	this must be done in each dimension, so:
	 
	int timex = dx / (laserSpeedx - target_info.speedx);
	int timey = dy / (laserSpeedy - target_info.speedy);
	 
	watching out for division by 0 and/or negative values
	 
	However, we do not know the proportions of laserSpeed
	in x and y, because that's what we're trying to work out!
	 
	For this to work, we'd also need to know the direction of
	laserSpeed in each dimension, so we could deal with them
	moving towards us
	

	what we actually do is two iterations of where they will
	be in the amount of time it takes us to reach them
	*/

	float predx = target_info.currentx + target_info.weak_spot.x;
	float predy = target_info.currenty + target_info.weak_spot.y;
	
	float dx = predx - x;
	float dy = predy - y;
	//use slow_dist here, makes things look noticeably better
	float distance = slow_dist(dx, dy);
		
	float time = distance / my_speed;
	
	predx += target_info.speedx * time;
	predy += target_info.speedy * time;
		
	dx = predx - x;
	dy = predy - y;
	props_to_speed_and_length(dx, dy);
	 
	float total_distance = fast_dist(dx, dy);
	set_duration(total_distance);
}

void Projectile_Base::props_to_speed_and_length(float dx, float dy) {
	float propx;
	float propy;

	get_move_props(propx, propy, dx, dy);

	speedx = propx * my_speed;
	speedy = propy * my_speed;

	//this isn't quite right but who cares, we're in 3 dimensions, right?
	//the real answer is:
	//length in x = ((speed in x) / (speed in x + speed in y)) * length
	lengthx = my_length * propx;
	lengthy = my_length * propy;
}

void Projectile_Base::draw_our_line() {
	float x0 = x - world.viewx;
	float y0 = y - world.viewy;
	float x1 = x + lengthx - world.viewx;
	float y1 = y + lengthy - world.viewy;
	display.draw_line(x0, y0, x1, y1, 1, color);
}

bool Projectile_Base::move() {
	if (world.frame_counter - timer > duration) {
		if (we_hit)
			sides[targetSide].groups[targetGroup].been_hit(target_unit, weapon_lookup[my_type].power);
		return false;
	}
	else {
		x += speedx;
		y += speedy;

		return true;
	}
}

void Projectile_Base::explode() {
	SDL_Rect tmp_rect = {static_cast<int>(x) - world.viewx, static_cast<int>(y) - world.viewy, 0, 0};

	for (int i = 0; i != SmallShip::explode_pics.size(); ++i) {
		if (world.frame_counter - explode_timer < frames_per_anim_frame * (i + 1)) {
			display.blt(SmallShip::explode_pics[i], tmp_rect);
			return;
		}
	}

	finished_explode = true;
}

void Projectile_Base::set_duration(float distance) {
	//+1 because move (and hence erasure) comes before drawing
	duration = static_cast<int>(((distance - my_length) / my_speed) + 1);

	//if we're right on top of them..
	if (duration < 1) {
		//right right on top of them, in which case we need
		//to make up arbitrary lengths and speeds
		if (speedx == 0 && speedy == 0) {
			speedx = my_speed / 2;
			speedy = my_speed / 2;
			lengthx = my_length / 4;
			lengthy = my_length / 4;
		}

		duration = 1;
	}
}

///

LaserBolt_Base::LaserBolt_Base(float ix, float iy, CoordsInt target, WeaponType i_type, const OpenGLColor& i_color):
Projectile_Base(ix, iy, target, i_type, i_color) {
	predict_target();
}

///

SmallLaserBolt::SmallLaserBolt(float ix, float iy, CoordsInt target, WeaponType i_type, const OpenGLColor& i_color):
LaserBolt_Base(ix, iy, target, i_type, i_color) {
	world.play_sound(ix, iy, SE_SMALL_LASER);
}

void SmallLaserBolt::draw_self() {
	draw_our_line();
}


////

TwinLaserBolt::TwinLaserBolt(float ix, float iy, CoordsInt target, WeaponType i_type, const OpenGLColor& i_color):
LaserBolt_Base(ix, iy, target, i_type, i_color) {
	world.play_sound(ix, iy, SE_TWIN_LASER);
}

void TwinLaserBolt::draw_self() {
	if (speedy > speedx) {
		GLfloat x0 = x - world.viewx - 3;
		GLfloat y0 = y - world.viewy;
		GLfloat x1 = x + lengthx - world.viewx - 3;
		GLfloat y1 = y + lengthy - world.viewy;

		display.draw_line(x0, y0, x1, y1, 1, color);

		x0+= 6;
		x1+= 6;

		display.draw_line(x0, y0, x1, y1, 1, color);
	}
	else {
		GLfloat x0 = x - world.viewx;
		GLfloat y0 = y - world.viewy - 3;
		GLfloat x1 = x + lengthx - world.viewx;
		GLfloat y1 = y + lengthy - world.viewy - 3;

		display.draw_line(x0, y0, x1, y1, 1, color);

		y0+= 6;
		y1+= 6;

		display.draw_line(x0, y0, x1, y1, 1, color);
	}
}

////
void Projectile_Base::draw_big_laser(float x0, float y0, float x1, float y1, const OpenGLColor& color) {
	x0-= world.viewx;
	y0-= world.viewy;
	x1-= world.viewx;
	y1-= world.viewy;

	display.draw_line(x0, y0, x1, y1, 3, color);
}

////

Missile::Missile(float ix, float iy, CoordsInt target, WeaponType i_type):
Projectile_Base(ix, iy, target, i_type, standard_colors.missile_grey), exploding(0), target_dead(0) {
	world.play_sound(ix, iy, SE_MISSILE_LAUNCH);
	//get it to sort out some props because it will be drawn before it first gets to move
	move();
	x-= speedx;
	y-= speedy;
}

bool Missile::move() {
	if (exploding) {
		if (we_hit)
			sides[targetSide].groups[targetGroup].been_hit(target_unit, weapon_lookup[my_type].power);
		return false;
	}

	if (!target_dead) {
		target_coords = sides[targetSide].groups[targetGroup].get_unit_center(target_unit);

		if (!sides[targetSide].groups[targetGroup].get_unit_alive(target_unit)) {
			target_coords.x += rand() % static_cast<int>(my_speed / 2 - my_speed);
			target_coords.y += rand() % static_cast<int>(my_speed / 2 - my_speed);
			target_dead = 1;
		}
	}

	float dx = target_coords.x - x;
	float dy = target_coords.y - y;
	float distance = fast_dist(dx, dy);

	if (distance < weapon_lookup[my_type].speed)
		exploding = true;

	else {
		props_to_speed_and_length(dx, dy);

		x += speedx;
		y += speedy;
	}

	return true;
}

void Missile::draw_self() {
	draw_our_line();
}

Torpedo::Torpedo(float ix, float iy, CoordsInt target, WeaponType i_type):
Projectile_Base(ix, iy, target, i_type, standard_colors.torpedo_blue), exploding(false) {
	predict_target();
}


void Torpedo::draw_self() {
	if (exploding) {
		explode();
		return;
	}

	float x0 = x - world.viewx;
	float y0 = y - world.viewy;
	float x1 = x + lengthx - world.viewx;
	float y1 = y + lengthy - world.viewy;

	display.draw_line(x0, y0, x1, y1, 2, color);
}

bool Torpedo::move() {
	if (exploding) {
		if (finished_explode)
			return false;
		return true;
	} else if (!Projectile_Base::move()) {
		world.play_sound(x, y, SE_TORPEDO_EXPLOSION);
		exploding = true;
		explode_timer = world.frame_counter;
	}

	return true;
}

bool Projectile_Base::check_to_hit(int accuracy) {
	if (rand() % 100 + accuracy > 99)
		return true;
	else
		return false;
}

LaserExplosion::LaserExplosion(float ix, float iy):
Projectile_Base(ix, iy) {
	explode_timer = world.frame_counter;
}

bool LaserExplosion::move() {
	if (finished_explode)
		return false;
	else
		return true;
}

void LaserExplosion::draw_self() {
	explode();
}

DropShip::DropShip(float ix, float iy, CoordsInt target, WeaponType i_type):
Projectile_Base(ix, iy, target, i_type, standard_colors.black) {
	predict_target();
	world.play_sound(ix, iy, SE_DROP_SHIP_LAUNCH);
}

void DropShip::draw_self() {
	SDL_Rect tmp_rect = {static_cast<int>(x) - world.viewx, static_cast<int>(y) - world.viewy, 0, 0};
	display.blt(gen_pictures[GENPIC_DROP_SHIP], tmp_rect);
}

