/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "BCCompiler.h"
#include "Display.h"
#include "FileUtils.h"
#include "Font.h"
#include "Globals.h"
#include "Inlines.h"
#include "LoadGameData.h"
#include "RTSUnit_Base.h"
#include "SettingsStruct.h"
#include "Sound.h"
#include "World.h"

#include <sstream>
#include <algorithm>

using std::istringstream;
using std::istringstream;
using std::find;
using std::getline;
using std::map;
using std::vector;
using std::runtime_error;
using std::string;
using std::wstring;

void load_standard_graphics() {
	//some will be unused, so they must be 0 when we free them
	for (int i = 0; i != max_gen_pictures; ++i)
		gen_pictures[i] = 0;

	load_title_images();
	load_title_music();

	//fonts
	normal_font.init("fonts/DejaVuSans.ttf", 14);
	bold_font.init("fonts/DejaVuSansMono-Bold.ttf", 14);

	//general pictures
	gen_pictures[GENPIC_CURSOR] = display.file_to_surface("images/cursor1.png");
	gen_pictures[GENPIC_CURSORMOVE] = display.file_to_surface("images/cursor2.png");
	gen_pictures[GENPIC_CURSORATTACK] = display.file_to_surface("images/cursor3.png");
	gen_pictures[GENPIC_CLOSEBOX] = display.file_to_surface("images/windowclose.png");

	gen_pictures[GENPIC_MISSIONS_ICON] = display.file_to_surface("images/icon_missions.png", standard_colors.black);
	gen_pictures[GENPIC_COMM_ICON] = display.file_to_surface("images/icon_comm.png", standard_colors.black);
	gen_pictures[GENPIC_BRIEFING_ICON] = display.file_to_surface("images/icon_briefing.png", standard_colors.black);
	gen_pictures[GENPIC_RANGE_ICON] = display.file_to_surface("images/icon_range.png", standard_colors.black);
	gen_pictures[GENPIC_OPTIONS_ICON] = display.file_to_surface("images/icon_options.png", standard_colors.black);

	gen_pictures[GENPIC_STATS_ICON] = display.file_to_surface("images/icon_stats.png", standard_colors.black);
	gen_pictures[GENPIC_STATUS_ICON] = display.file_to_surface("images/icon_status.png", standard_colors.black);
	gen_pictures[GENPIC_RETURN_ICON] = display.file_to_surface("images/icon_return.png", standard_colors.black);
	gen_pictures[GENPIC_RECALL_ICON] = display.file_to_surface("images/icon_recall.png", standard_colors.black);
	gen_pictures[GENPIC_PATROL_ICON] = display.file_to_surface("images/icon_patrol.png", standard_colors.black);
	gen_pictures[GENPIC_ATTACK_ICON] = display.file_to_surface("images/icon_attack.png", standard_colors.black);
	gen_pictures[GENPIC_RECON_ICON] = display.file_to_surface("images/icon_reccy.png", standard_colors.black);
	gen_pictures[GENPIC_SELECT_ICON] = display.file_to_surface("images/icon_select.png", standard_colors.black);
	gen_pictures[GENPIC_CANCEL_ICON] = display.file_to_surface("images/icon_cancel.png", standard_colors.black);

	gen_pictures[GENPIC_SMALL_EXPLODE1] = display.file_to_surface("images/ssexplode3.png", standard_colors.black);
	gen_pictures[GENPIC_SMALL_EXPLODE2] = display.file_to_surface("images/ssexplode4.png", standard_colors.black);
	gen_pictures[GENPIC_SMALL_EXPLODE3] = display.file_to_surface("images/ssexplode5.png", standard_colors.black);
	gen_pictures[GENPIC_SMALL_EXPLODE4] = display.file_to_surface("images/ssexplode6.png", standard_colors.black);
	gen_pictures[GENPIC_SMALL_EXPLODE5] = display.file_to_surface("images/ssexplode7.png", standard_colors.black);
	gen_pictures[GENPIC_SMALL_EXPLODE6] = display.file_to_surface("images/ssexplode8.png", standard_colors.black);

	///

	gen_pictures[GENPIC_LARGE_EXPLODE1] = display.file_to_surface("images/csexplode2.png", standard_colors.black);
	gen_pictures[GENPIC_LARGE_EXPLODE2] = display.file_to_surface("images/csexplode3.png", standard_colors.black);
	gen_pictures[GENPIC_LARGE_EXPLODE3] = display.file_to_surface("images/csexplode4.png", standard_colors.black);
	gen_pictures[GENPIC_LARGE_EXPLODE4] = display.file_to_surface("images/csexplode5.png", standard_colors.black);
	gen_pictures[GENPIC_LARGE_EXPLODE5] = display.file_to_surface("images/csexplode6.png", standard_colors.black);
	gen_pictures[GENPIC_LARGE_EXPLODE6] = display.file_to_surface("images/csexplode7.png", standard_colors.black);
	gen_pictures[GENPIC_LARGE_EXPLODE7] = display.file_to_surface("images/csexplode8.png", standard_colors.black);
	gen_pictures[GENPIC_LARGE_EXPLODE8] = display.file_to_surface("images/csexplode9.png", standard_colors.black);
	gen_pictures[GENPIC_LARGE_EXPLODE9] = display.file_to_surface("images/csexplode10.png", standard_colors.black);

	///

	gen_pictures[GENPIC_DROP_SHIP] = display.file_to_surface("unitpictures/blue/kasper-ss-2.png");

	///

	SmallShip::explode_pics.push_back(gen_pictures[GENPIC_SMALL_EXPLODE1]);
	SmallShip::explode_pics.push_back(gen_pictures[GENPIC_SMALL_EXPLODE2]);
	SmallShip::explode_pics.push_back(gen_pictures[GENPIC_SMALL_EXPLODE3]);
	SmallShip::explode_pics.push_back(gen_pictures[GENPIC_SMALL_EXPLODE4]);
	SmallShip::explode_pics.push_back(gen_pictures[GENPIC_SMALL_EXPLODE5]);
	SmallShip::explode_pics.push_back(gen_pictures[GENPIC_SMALL_EXPLODE6]);

	BigUnit::big_explode_pics.push_back(gen_pictures[GENPIC_LARGE_EXPLODE1]);
	BigUnit::big_explode_pics.push_back(gen_pictures[GENPIC_LARGE_EXPLODE2]);
	BigUnit::big_explode_pics.push_back(gen_pictures[GENPIC_LARGE_EXPLODE3]);
	BigUnit::big_explode_pics.push_back(gen_pictures[GENPIC_LARGE_EXPLODE4]);
	BigUnit::big_explode_pics.push_back(gen_pictures[GENPIC_LARGE_EXPLODE5]);
	BigUnit::big_explode_pics.push_back(gen_pictures[GENPIC_LARGE_EXPLODE6]);
	BigUnit::big_explode_pics.push_back(gen_pictures[GENPIC_LARGE_EXPLODE7]);
	BigUnit::big_explode_pics.push_back(gen_pictures[GENPIC_LARGE_EXPLODE8]);
	BigUnit::big_explode_pics.push_back(gen_pictures[GENPIC_LARGE_EXPLODE9]);
}

//we free the title pic even though it is not loaded in load_standard_graphics. This is OK because it is always set to 0 after
//being freed
void clear_standard_graphics() {
	normal_font.shutdown();
	bold_font.shutdown();

	for (int i = 0; i != max_gen_pictures; ++i) {
		if (gen_pictures[i]) {
			glSDL_FreeSurface(gen_pictures[i]);
			gen_pictures[i] = 0;
		}
	}

	SmallShip::explode_pics.clear();
	BigUnit::big_explode_pics.clear();
}

void reset_graphics() {
	clear_standard_graphics();
	display.reset_video();
	standard_colors.init();
	load_standard_graphics();
}

void load_title_images() {
	//free during rts to save memory
	gen_pictures[GENPIC_TITLE] = display.file_to_surface("images/title.png");
	gen_pictures[GENPIC_INTRO_NEBULA] = display.file_to_surface("images/title_background.jpg");
}

void free_title_images() {
	glSDL_FreeSurface(gen_pictures[GENPIC_TITLE]);
	gen_pictures[GENPIC_TITLE] = 0;
	glSDL_FreeSurface(gen_pictures[GENPIC_INTRO_NEBULA]);
	gen_pictures[GENPIC_INTRO_NEBULA] = 0;
}

void load_title_music() {
	sound.load_song("cdk_-_one_moment_(cdk_play_it_cool_mix).ogg");
}

void setup_lookup_tables() {
	//set up specifier lookup table
	token_lookup["#"] = TT_COMMENT;
	token_lookup["if"] = TT_IF;
	token_lookup["else"] = TT_ELSE;
	token_lookup["elif"] = TT_ELIF;
	//no TT_OpenB
	//no TT_CloseB
	token_lookup["@"] = TT_JUMP;
	token_lookup["return"] = TT_RETURN;
	
	token_lookup["starttimer"] = TT_START_TIMER;
	
	token_lookup["move"] = TT_MOVE;
	token_lookup["moveslow"] = TT_MOVE_SLOW;
	token_lookup["moveaway"] = TT_MOVE_AWAY;
	token_lookup["fire"] = TT_FIRE;
	token_lookup["patrol"] = TT_PATROL;
	token_lookup["dock"] = TT_DOCK;
	
	token_lookup["=="] = TT_Equal;
	token_lookup["!="] = TT_NotEqual;
	token_lookup[">"] = TT_GreaterThan;
	token_lookup["<"] = TT_LessThan;
	token_lookup[">="] = TT_GreaterThanEqual;
	token_lookup["<="] = TT_LessThanEqual;
	token_lookup["!"] = TT_Not;
	token_lookup["&&"] = TT_AND;
	token_lookup["||"] = TT_OR;
	
	token_lookup["+"] = TT_ADD;
	token_lookup["-"] = TT_MINUS;
	token_lookup["*"] = TT_MULTIPLY;
	token_lookup["/"] = TT_DIVIDE;
	token_lookup["%"] = TT_MODULO;

	token_lookup["="] = TT_Assign;
	token_lookup["++"] = TT_Increment;
	token_lookup["--"] = TT_Decrement;

	token_lookup["$"] = TT_SCRIPT_VAR;
	token_lookup["$g"] = TT_SaveGroup;
	token_lookup["$timer"] = TT_ScriptTimer;
	
	token_lookup["$global"] = TT_GLOBAL_SCRIPT_VAR;
	token_lookup["$globalg"] = TT_GSaveGroup;

	token_lookup["our"] = TT_Our;
	token_lookup["aenemy"] = TT_AnyEnemy;
	token_lookup["afriend"] = TT_AnyFriend;
	token_lookup["aally"] = TT_AnyAlly;
	token_lookup["numenemy"] = TT_NumEnemy;
	token_lookup["numfriend"] = TT_NumFriend;
	token_lookup["numally"] = TT_NumAlly;
	token_lookup["nenemy"] = TT_NearestEnemy;
	token_lookup["nfriend"] = TT_NearestFriend;
	token_lookup["nally"] = TT_NearestAlly;
	token_lookup["nedge"] = TT_NEAREST_WORLD_EDGE;
	token_lookup["waypoint"] = TT_WAYPOINT;

	token_lookup["health"] = TT_Health;
	token_lookup["shield"] = TT_Shield;
	token_lookup["armour"] = TT_Armour;
	token_lookup["healthmax"] = TT_HealthMax;
	token_lookup["shieldmax"] = TT_ShieldMax;
	token_lookup["armourmax"] = TT_ArmourMax;
	token_lookup["unitshieldmax"] = TT_UnitShieldMax;
	token_lookup["unitarmourmax"] = TT_UnitArmourMax;
	token_lookup["number"] = TT_Number;
	token_lookup["speed"] = TT_SPEED;
	token_lookup["fuel"] = TT_FUEL;
	token_lookup["distance"] = TT_DISTANCE;
	token_lookup["distancefrom"] = TT_DISTANCE_FROM;
	token_lookup["isalpha"] = TT_IS_ALPHA;

	token_lookup["smallrange"] = TT_SmallRange;
	token_lookup["smallpower"] = TT_SmallPower;
	token_lookup["bigrange"] = TT_BigRange;
	token_lookup["bigpower"] = TT_BigPower;
	token_lookup["bigammo"] = TT_BIG_AMMO;
	token_lookup["insmallrange"] = TT_IN_SMALL_RANGE;
	token_lookup["inbigrange"] = TT_InBigRange;
	token_lookup["ourinsmallrange"] = TT_OurInSmallRange;
	token_lookup["ourinbigrange"] = TT_OurInBigRange;
	token_lookup["left"] = TT_Left;
	token_lookup["misstarget"] = TT_MissTarget;
	token_lookup["torptarget"] = TT_TorpTarget;
	token_lookup["type"] = TT_GroupType;
	
	token_lookup["none"] = TT_NONE;

	token_lookup["fighter"] = TT_FIGHTER;
	token_lookup["bomber"] = TT_BOMBER;
	token_lookup["frigate"] = TT_FRIGATE;
	token_lookup["capital"] = TT_CAPITAL;
	token_lookup["freighter"] = TT_FREIGHTER;
	token_lookup["defencenode"] = TT_DEFENCE_NODE;
	token_lookup["planet"] = TT_PLANET;

	token_lookup["n"] = TT_N;
	token_lookup["ne"] = TT_NE;
	token_lookup["e"] = TT_E;
	token_lookup["se"] = TT_SE;
	token_lookup["s"] = TT_S;
	token_lookup["sw"] = TT_SW;
	token_lookup["w"] = TT_W;
	token_lookup["nw"] = TT_NW;

	token_lookup["northedge"] = TT_NorthEdge;
	token_lookup["eastedge"] = TT_EastEdge;
	token_lookup["southedge"] = TT_SouthEdge;
	token_lookup["westedge"] = TT_WestEdge;

	token_lookup["random"] = TT_RANDOM;
	token_lookup["numwaypoint"] = TT_NUM_WAYPOINT;

	///////////

	//weapon lookup table
	WeaponDesc tmp_weapon;

	tmp_weapon.start_ammo = 0;

	tmp_weapon.name = L"None";
	tmp_weapon.power = 0;
	tmp_weapon.range = 0;
	tmp_weapon.accuracy = 0;
	tmp_weapon.reload = 0;
	tmp_weapon.speed = 0;
	tmp_weapon.length = 0;

	weapon_lookup[WT_NONE] = tmp_weapon;
	weapon_load_lookup[weapon_lookup[WT_NONE].name] = WT_NONE;

	string input_str;
	file_to_string("unitdata/equip.dat", input_str);
	istringstream input(input_str);

	OptionSkipIter iter = input;
	OptionSkipIter file_end;

	skip_to_next_section(input, iter, file_end);

	WeaponType wi = WT_TWIN;
	while (wi != WT_DROP_SHIP + 1) {
		FillWeaponValues(wi, input, iter, file_end);
		wi = static_cast<WeaponType>(wi + 1);
		iter.equals_ss(input);
	}

	skip_to_next_section(input, iter, file_end);

	while (iter != file_end) {
		EquipDesc temp_equip;
		string temp_name;
		getline(input, temp_name, ',');
		iter.equals_ss(input);
		temp_equip.max = iter_to_int(iter, file_end);
		iter.plus_plus_ss();
		temp_equip.recharge = iter_to_int(iter, file_end);
		equip_lookup[string_to_wstring(temp_name)] = temp_equip;
		iter.equals_ss(input);
	}

	///

	int radius_in_squares = equip_lookup[L"Sensor array"].max / terrain_tile_dim;
	fog_line_lookup.resize(radius_in_squares);

	/*
	equation for a circle is x^2 + y^2 = r^2
	so to find y:
	y^2 = r^2 - x^2
	y = sqrt(r^2 - x^2)
	*/

	//this saves length of each vertical line in fog circle from the vertical center of each line
	for (int i = 0; i != radius_in_squares; ++i) {
		int length = static_cast<int>(std::sqrt(static_cast<float>(radius_in_squares * radius_in_squares - i * i)));
		/*if you don't do this you end up with a little point at the top and bottom of the circle
		I'm sure there is a logical explanation for this, but this is just a fix found with trial and error
		possibly something to do with fog_line_lookup having size equal to radius_in_squares which doesn't allow for
		a central horizontal line? i.e. issue was not the existence of a point but rather there should be 4 points
		But it looks better with no points so I will leave it like this
		*/
		if (length == radius_in_squares)
			fog_line_lookup[i] = length - 1;
		else
			fog_line_lookup[i] = length;
	}
	///

	unit_type_to_string[UT_FIGHTER] = L"Fighter";
	unit_type_to_string[UT_BOMBER] = L"Bomber";
	unit_type_to_string[UT_FRIGATE] = L"Frigate";
	unit_type_to_string[UT_CAPITAL] = L"Capital ship";
	unit_type_to_string[UT_FREIGHTER] = L"Freighter";
	unit_type_to_string[UT_DEFENCE_NODE] = L"Defence node";
	unit_type_to_string[UT_PLANET] = L"Planet";

	string_to_unit_type[L"Fighter"] = UT_FIGHTER;
	string_to_unit_type[L"Bomber"] = UT_BOMBER;
	string_to_unit_type[L"Frigate"] = UT_FRIGATE;
	string_to_unit_type[L"Capital ship"] = UT_CAPITAL;
	string_to_unit_type[L"Freighter"] = UT_FREIGHTER;
	string_to_unit_type[L"Defence node"] = UT_DEFENCE_NODE;
	string_to_unit_type[L"Planet"] = UT_PLANET;
}

void FillWeaponValues(WeaponType i, istringstream& input, OptionSkipIter& iter, const OptionSkipIter& file_end) {
	static WeaponDesc tmp_weapon;
	string tmp_str;

	getline(input, tmp_str, ',');
	tmp_weapon.name = string_to_wstring(tmp_str);
	iter.equals_ss(input);
	tmp_weapon.power = iter_to_int(iter, file_end);
	iter.plus_plus_ss();
	tmp_weapon.range = static_cast<float>(iter_to_int(iter, file_end));
	iter.plus_plus_ss();
	tmp_weapon.accuracy = iter_to_int(iter, file_end);
	iter.plus_plus_ss();
	tmp_weapon.reload = iter_to_int(iter, file_end);
	iter.plus_plus_ss();
	tmp_weapon.speed = static_cast<float>(iter_to_int(iter, file_end));
	iter.plus_plus_ss();
	tmp_weapon.length = static_cast<float>(iter_to_int(iter, file_end));

	if (i == WT_TWIN)
		tmp_weapon.start_ammo = max_ai_int;
	else if (i == WT_TORPEDO)
		tmp_weapon.start_ammo = 2;

	weapon_lookup[i] = tmp_weapon;
	weapon_load_lookup[weapon_lookup[i].name] = i;
}

void load_ai_scripts() {
	create_ai_script_vec(L"");
	create_ai_script_vec(L"recon");
	create_ai_script_vec(L"patrol");
	create_ai_script_vec(L"attack");
	create_ai_script_vec(L"return");
	create_ai_script_vec(L"recall");

	for (map<wstring, AIScript>::iterator iter = world.ai_scripts.begin(); iter != world.ai_scripts.end(); ++iter)
		make_function_lookup_table(iter->first);
}

void create_ai_script_vec(const wstring& script_name) {
	if (script_name == L"") {
		world.ai_scripts[script_name].script.push_back(str_to_ustr("main:"));
		return;
	}
		
	const string concat_string = "aiscripts/" + wstring_to_string(script_name) + ".ai";

	string ai_input_str;
	file_to_string(concat_string, ai_input_str);

	istringstream aiStream(ai_input_str);

	string temp_str;
	
	try {
		while (getline(aiStream, temp_str))
			world.ai_scripts[script_name].script.push_back(BCCompiler::convert_to_bc(temp_str));
	} catch (runtime_error e) {
		char error[200];
		printf(error, "Problem with AI script %s at line %d: %s", script_name.c_str(), world.ai_scripts[script_name].script.size() + 1, e.what());
		throw runtime_error(error);
	}

	check_over_ai(script_name);
	//don't create function lookup table here because we try to keep loading data seperate from setting up objects based on that data
}

void make_function_lookup_table(const wstring& script_name) {
	const vector<ustring>& the_text = world.ai_scripts[script_name].script;
	
	//check for main and set first_line
	int mainIndex = 0;
	while (mainIndex != the_text.size() && the_text[mainIndex][0] == TT_COMMENT)
		++mainIndex;

	if (the_text[mainIndex] == str_to_ustr("main:")) {
		if (the_text.size() > 1)
			world.ai_scripts[script_name].first_line = mainIndex + 1;
		else
			world.ai_scripts[script_name].first_line = mainIndex;
	} else {
		const string error = "AI script " + wstring_to_string(script_name) + " does not appear to begin with \"main:\"";
		throw runtime_error(error.c_str());
	}
	
	for (int i = 0; i != the_text.size(); ++i) {
		int bi = 0;
		while (the_text[i][bi] == TT_TAB)
			++bi;

		if (the_text[i][bi] == TT_JUMP) {
			ustring function_name = the_text[i].substr(bi+1, string::npos);

			if (world.ai_scripts[script_name].function_lookup.find(function_name) == world.ai_scripts[script_name].function_lookup.end()) {
				int function_index = 0;

				while (function_index != the_text.size()) {
					if (the_text[function_index].find(TT_COLON) == string::npos) {
						++function_index;
						continue;
					}
					
					bi = 0;
					while (the_text[function_index][bi] == TT_TAB)
						++bi;

					const ustring temp_str = the_text[function_index].substr(bi, the_text[function_index].find(TT_COLON));
					if (temp_str == function_name)
						break;

					++function_index;
				}

				if (function_index != the_text.size())
					world.ai_scripts[script_name].function_lookup[function_name] = function_index;
				else {
					const string output = "Couldn't find target of function call " + ustr_to_str(function_name) + " in script " + wstring_to_string(script_name);
					throw runtime_error(output);
				}
			}
		}
	}
}

void check_over_ai(const wstring& script_name) {
	const vector<ustring>& the_text = world.ai_scripts[script_name].script;
	char output[200];
	
	for (int i = 0; i != the_text.size(); ++i) {
		bool anIf = false;
		bool anElse = false;
		bool aNumX = false;
		bool anAssign = false;
		bool aScriptVar = false;
		bool anIncOrDec = false;
		
		for (int j = 0; j != the_text[i].size(); ++j) {
			switch(the_text[i][j]) {
			case TT_IF:
			case TT_ELIF:
				anIf = true;
				break;

			case TT_ELSE:
				anElse = true;
				break;

			case TT_NumEnemy:
			case TT_NumFriend:
			case TT_NumAlly:
				aNumX = true;
				break;

			case TT_Assign:
				if (!aScriptVar) {
					printf(output, "AI script %s attempts to assign to something other than a script variable on line %d.", script_name.c_str(), i + 1);
					throw runtime_error(output);
				}
				anAssign = true;
				break;

			case TT_SCRIPT_VAR:
			case TT_SaveGroup:
			case TT_ScriptTimer:
			case TT_GLOBAL_SCRIPT_VAR:
			case TT_GSaveGroup:
				aScriptVar = true;
				break;

			case TT_Increment:
			case TT_Decrement:
				if (!aScriptVar) {
					printf(output, "AI script %s attempts to increment or decrement before naming a variable on line %d.", script_name.c_str(), i + 1);
					throw runtime_error(output);
				}
				anIncOrDec = true;
				break;
			}
		}
		
		if (anIf && aNumX) {
			printf(output, "AI script %s uses a numx type specifier on the same line as an \"if\" on line %d.\nCurrently numx commands can only be assigned to variables.", script_name.c_str(), i + 1);
			throw runtime_error(output);
		}

		if (anIf && anElse) {
			printf(output, "AI script %s includes the phrase \"else if\" on line %d.\nIn Really Rather Good Battles In Space AI scripts you must write \"elif\" instead.", script_name.c_str(), i + 1);
			throw runtime_error(output);
		}
		
		if (anIf && anAssign) {
			printf(output, "AI script %s includes a single equals sign in an if statement on line %d.\nA single equals sign is used to assign values to script variables, you probably meant to use \"==\".", script_name.c_str(), i + 1);
			throw runtime_error(output);
		}
	}
	
	/*
		int openBCounter = 0;
		int closeBCounter = 0;

		for (int j = 0; j != the_text[i].size(); ++j) {
			if (the_text[i][j] == TT_OpenB)
				++openBCounter;
			else if (the_text[i][j] == TT_CloseB)
				++closeBCounter;
		}

		if (openBCounter > closeBCounter) {
			printf(output, "AI script %s has more open rounded brackets than close rounded brackets on line %d", script_name.c_str(), i + 1);
			throw runtime_error(output);
		}
		else if (openBCounter < closeBCounter) {
			printf(output, "AI script %s has less open rounded brackets than close rounded brackets on line %d", script_name.c_str(), i + 1);
			throw runtime_error(output);
		}
	*/
}

