/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/


#ifndef GUARD_Group_Base
#define GUARD_Group_Base

#include "AIInterpreter.h"
#include "Globals.h"
#include "RTSUnit.h"
#include "World.h"

class Group_Base
{
public:
	Group_Base(const std::wstring& i_data_filename, int i_my_side, int i_my_squad, int i_my_group, CoordsInt i_starting_coords);
	virtual ~Group_Base() {}

	void init(int i_ai_stagger);
	void toggle_draw_bound();
	void toggle_draw_number();
	void toggle_draw_weapon_range();
	void display_missions_fuel();
	void hide_missions_fuel();
	bool set_pos(float ix, float iy);
	virtual bool go_to_start_coords();

	virtual void mouse_d(Uint16 x, Uint16 y, CoordsFloat center) {}
	virtual void mouse_d_set_pos(Uint16 x, Uint16 y, CoordsFloat center) {}

	bool drag_select(SDL_Rect& drag_box);
	virtual bool check_for_cursor(Uint16 x, Uint16 y) const;
	
	void add_waypoint(CoordsFloat waypoint);
	virtual void launch(const std::wstring& i_ai_filename, int mission_slot, int launch_slot) {}
	void change_ai_script(const std::wstring& i_ai_filename);
	virtual void run_group_ai();
	virtual void run_fire_commands() = 0;
	virtual void move();
	virtual void check_if_scanned();
	virtual void upkeep();
	void dead_upkeep();
	void order_move(float x, float y);
	void order_halt();

	void get_unit_target_info(TargetDesc& target_info);
	void destroy();
	virtual void force_fill_fog(bool make_visible);

	void set_screen_rect();

	virtual void draw_self_back_back() {};
	virtual void draw_self_back() {};
	virtual void draw_self_middle() {};
	virtual void draw_self_front() {};
	void draw_big_laser();

	virtual void draw_bound();

	CoordsFloat get_closest_point(const CoordsFloat you) const;
	
	int get_health() const;
	int get_shield() const;
	int get_armour() const;
	int get_health_max() const;
	int get_shield_max() const;
	int get_armour_max() const;

	UnitType get_type() {return units[0].get_type();}
	virtual bool is_small() const = 0;
	bool is_big_ship() const {if (units[0].get_type() == UT_FRIGATE || units[0].get_type() == UT_CAPITAL || units[0].get_type() == UT_FREIGHTER || units[0].get_type() == UT_DEFENCE_NODE) return true; return false;}
	virtual bool get_in_hangar() const {return false;}
	virtual int get_launch_slot() {return 0;}
	
	int get_small_power() const;
	int get_big_power() const;
	int get_big_ammo() const;
	int get_big_ammo_max() const;
	const std::wstring get_current_mission() const;
	virtual int get_how_full() const = 0;
	CoordsFloat get_center() const {CoordsFloat tmp(myx + width / 2, myy + height / 2); return tmp;}
	void get_rect(Rect32& give_rect) const {give_rect.x = static_cast<int>(myx); give_rect.y = static_cast<int>(myy); give_rect.w =width; give_rect.h = height;}

	int find_distance_to(int n_side, int n_group) const;
	int find_point_distance_from_center(float x, float y) const;
	bool check_for_collision(int n_side, int n_group) const;
	void report_on_script_error(const char* error, int lineNumber = 0);
	CoordsFloat get_hangar_bay_coords(bool alpha_wing);

	int width;
	int height;
	float myx, myy;
	float rem_speed_x, rem_speed_y;

	float speed_cruise;
	float speed_max;
	int fuel_current;
	int fuel_max;
	int my_mission;

	std::vector<RTSUnit> units;
	CoordsFloat waypoints[n_ai_vars];

	std::wstring data_filename;
	CoordsInt starting_coords;

	AICommands the_commands;

	int my_side;
	int my_squad;
	int my_group;

	SDL_Rect screen_rect;
	bool on_screen;
	bool b_draw_bound;
	bool b_draw_weapon_range;
	bool b_draw_fuel_range;
	bool draw_number;

	bool alive;
	int units_left;
	int units_remembered;

	//-1 means none
	int my_parent_cap;

	bool speed_plane_matches_target;
	bool invert_patrol;

	AIInterpreter ai_interpreter;
	std::wstring ai_filename;

protected:	
	void set_initial_vars();
	void set_bounding_rect();

	void random_place_units();

	//if you don't care who hit
	bool check_for_big_collision() const;
	bool check_for_big_collision(CoordsInt& who_hit) const;

	virtual void small_ship_dont_sit_on_top_of_move_target() {}
	virtual void small_ship_dont_sit_on_top_of_move_point_target() {}
	void convert_patrol_target();
	//0 is normal, 1 means invert movement (i.e move away from)
	void convert_move_target(bool inverse = false);
	void check_for_going_off_screen();
	
	void convert_max_change(float& dx, float& dy);
	void pixel_shuffle(float& dx, float& dy);
	void relocate(float dx, float dy);
	void fill_fog(bool make_visible);
	void fill_fog_square(TerrainTile& tile, bool make_visible);

	virtual void select_targets() = 0;
	void update_manual_target_info();
	
	CoordsFloat get_dx_dy_close(int side, int group) const;
	CoordsFloat get_dx_dy_center(int side, int group) const;
	CoordsFloat get_dx_dy_center(float x, float y) const;
	
	float max_speed_change;
	
	//a hack so that if the whole group is dead when someone fires at us, they are given last
	//alive unit's last known position rather than the position of a unit which died ages ago
	int last_alive;

	bool already_ai_error;
	int ai_stagger;

	static const int standard_unit_spacing = 3;
};


///

class SmallShipGroup: public Group_Base
{
public:
	SmallShipGroup(int i_my_side, int i_my_squad, int i_my_group, const std::wstring& data_filename, int i_parent_cap, CoordsInt i_starting_coords);

	bool go_to_start_coords();
	void launch(const std::wstring& i_ai_filename, int mission_slot, int launch_slot);
	void run_group_ai() {if (!in_hangar) Group_Base::run_group_ai();}
	void run_fire_commands();
	void upkeep();

	void move();
	void force_fill_fog(bool make_visible);
	void check_if_scanned();

	void draw_self_back();
	void draw_self_front();
	void draw_bound();

	bool is_small() const {return true;}
	bool get_in_hangar() const {return in_hangar;}
	bool check_for_cursor(Uint16 x, Uint16 y) const;
	int get_how_full() const {return 0;}

protected:
	void select_targets();

	void small_ship_dont_sit_on_top_of_move_target();
	void small_ship_dont_sit_on_top_of_move_point_target();
	void set_pos_to_parent();

	int launch_time;
	bool in_hangar;
	float save_prop_x;
    float save_prop_y;
};

class BigUnitGroup: public Group_Base
{
public:
	BigUnitGroup(int i_my_side, int i_my_squad, int i_my_group, const std::wstring& data_filename, CoordsInt i_starting_coords, bool face_left);

	void mouse_d(Uint16 x, Uint16 y, CoordsFloat center);
	void mouse_d_set_pos(Uint16 x, Uint16 y, CoordsFloat center);

	void run_group_ai();
	void run_fire_commands();
	void draw_self_back_back();
	void draw_self_middle();
	bool is_small() const {return false;}
	int get_launch_slot();
	int get_how_full() const;

protected:
	void select_targets();

	int next_launch_time;
	static const int launch_time_per_group = 30;
};

#endif
