/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "Display.h"
#include "Font.h"
#include "GenWindow.h"
#include "Globals.h"
#include "Group.h"
#include "RTS.h"
#include "SettingsStruct.h"
#include "Side.h"

#include <string>
#include <stdexcept>
#include <list>
#include <algorithm>

using std::find;
using std::list;
using std::runtime_error;
using std::wstring;
using std::vector;

Group_Base::Group_Base(const wstring& i_data_filename, int i_my_side, int i_my_squad, int i_my_group, CoordsInt i_starting_coords):
data_filename(i_data_filename), my_side(i_my_side), my_squad(i_my_squad), my_group(i_my_group),
b_draw_bound(false), b_draw_weapon_range(false), b_draw_fuel_range(false), draw_number(false), alive(true), my_parent_cap(-1),
invert_patrol(false), rem_speed_x(0), rem_speed_y(0), last_alive(0), already_ai_error(false),
my_mission(-1), fuel_current(0) {
	starting_coords.x = i_starting_coords.x;
	starting_coords.y = i_starting_coords.y;

	//sometimes needed if we never run a script and hence never have this done by AI interpreter
	memset(&the_commands, 0, sizeof(AICommands));
	
	change_ai_script(L"");

	for (int i = 0; i != n_ai_vars; ++i)
		waypoints[i].x = -1;

	myx = world.width;
	myy = world.height;
}

void Group_Base::set_initial_vars() {
	speed_max = static_cast<float>(equip_lookup[units[0].get_engine_name()].max);
	max_speed_change = speed_max / 7;

	if (!is_small())
		the_commands.speed = speed_cruise = speed_max;

	units_left = units.size();
	units_remembered = units.size();

	set_bounding_rect();
}

bool Group_Base::set_pos(float ix, float iy) {
	myx = ix;
	myy = iy;

	random_place_units();
	if (check_for_big_collision())
		return false;
	else
		return true;
}

bool Group_Base::go_to_start_coords() {
	bool ret = set_pos(static_cast<float>(starting_coords.x), static_cast<float>(starting_coords.y));
	fill_fog(true);
	return ret;
}

void Group_Base::set_bounding_rect() {
	int temp_width, temp_height;

	units[0].get_dimensions(temp_width, temp_height);

	//if just one unit it has no extra space
	if (units.size() == 1) {
		width = temp_width;
		height = temp_height;
	} else {
		int spacing;

		spacing = standard_unit_spacing;

		int width_total = 0;
		int height_total = 0;

		for (int i = 0; i != units.size(); ++i) {
			width_total += temp_width;
			height_total += temp_height;
		}

		width = width_total * standard_unit_spacing;
		height = height_total * standard_unit_spacing;
	}
}

void Group_Base::random_place_units() {
	//not units.size() because if we only have one unit left then the random placement may divide by 0
	if (units_left == 1)
		units[0].set_pos(myx, myy);
	else {
		int temp_width, temp_height;

		for (int i = 0; i != units.size(); ++i) {
			units[i].get_dimensions(temp_width, temp_height);

			float x = static_cast<float>(rand() % (width - temp_width)) + myx;
			float y = static_cast<float>(rand() % (height - temp_height)) + myy;
			units[i].set_pos(x, y);
		}
	}
}

void Group_Base::set_screen_rect() {
	screen_rect.x = static_cast<int>(myx) - world.viewx;
	screen_rect.y = static_cast<int>(myy) - world.viewy;
	screen_rect.w = width;
	screen_rect.h = height;

	if (screen_rect.x < global_settings.screen_width
	&& screen_rect.y < global_settings.screen_height
	&& screen_rect.x + screen_rect.w > 0
	&& screen_rect.y + screen_rect.h > 0)
		on_screen = true;
	else
		on_screen = false;
}

void Group_Base::order_move(float x, float y) {
	the_commands.ordered_to_fire = false;
	the_commands.move_command = MC_MOVE_POINT;
	the_commands.point_target.x = x;
	the_commands.point_target.y = y;
}

void Group_Base::order_halt() {
	the_commands.ordered_to_fire = false;
	the_commands.move_command = MC_NO_MOVE;
}

bool Group_Base::drag_select(SDL_Rect& drag_box) {
	Rect32 rect_one(drag_box);
	Rect32 rect_two(screen_rect);
	if ((sides[0].scanned_groups[my_side][my_group] || debug_display_all_groups || units[0].get_type() == UT_PLANET)
	&& test_overlap(rect_one, rect_two) && !get_in_hangar())		
		return true;
	
	return false;
}

bool Group_Base::check_for_cursor(Uint16 x, Uint16 y) const {
	if (point_in_rect(x, y, screen_rect))
		return true;
	else
		return false;
}

void Group_Base::add_waypoint(CoordsFloat waypoint) {
	waypoint.x = waypoint.x - width / 2;
	waypoint.y = waypoint.y - height / 2;

	for (int i = 0; i != n_ai_vars; ++i) {
		if (waypoints[i].x == -1) {
			waypoints[i] = waypoint;
			return;
		}
	}
}

void Group_Base::change_ai_script(const std::wstring& i_ai_filename) {
	ai_filename = i_ai_filename;

	memset(ai_interpreter.script_vars, 0, sizeof(ai_interpreter.script_vars));
	memset(ai_interpreter.script_timers, 0, sizeof(ai_interpreter.script_timers));

	//$g0 is always our parent cap
	ai_interpreter.save_groups[0].x = my_side;
	ai_interpreter.save_groups[0].y = my_parent_cap;
	
	//$g1 is always the other wing
	ai_interpreter.save_groups[1].x = my_side;
	ai_interpreter.save_groups[1].y = sides[my_side].squadrons[my_squad].get_other_wing(my_group);

	//the rest start off invalid
	for (int i = 2; i != n_ai_vars; ++i) {
		ai_interpreter.save_groups[i].x = -1;
		ai_interpreter.save_groups[i].y = -1;
	}
}

bool Group_Base::check_for_collision(int n_side, int n_group) const {
	//technically I guess you could collide with someone you can't see, but we may as well rule this out
    if (!sides[n_side].groups[n_group].get_alive()
	|| !sides[my_side].scanned_groups[n_side][n_group]
	|| sides[n_side].groups[n_group].get_in_hangar())
        return false;
        
	Rect32 temp_rect;
	sides[n_side].groups[n_group].get_rect(temp_rect);

    Rect32 my_rect(static_cast<int>(myx), static_cast<int>(myy), width, height);
	if (test_overlap(my_rect, temp_rect))
		return true;

	return false;
}

bool Group_Base::check_for_big_collision() const {
	CoordsInt throw_away;
	return check_for_big_collision(throw_away);
}

bool Group_Base::check_for_big_collision(CoordsInt& who_hit) const {
	if (is_small() || units[0].get_type() == UT_PLANET)
		return false;

	//only concerned with collisions with our own side
	for (int i = 0; i != sides[my_side].groups.size(); ++i) {
		if (sides[my_side].groups[i].is_small() || sides[my_side].groups[i].get_type() == UT_PLANET || (i == my_group))
			continue;
		
		//if in battle, only concerned if we share the same move target
		if (gs_current == GST_BATTLE) {
			const AICommands* p_commands = sides[my_side].groups[i].get_the_commands();
			
			if (p_commands->move_target.x != the_commands.move_target.x
			|| p_commands->move_target.y != the_commands.move_target.y)
				continue;
		}
			
		if (check_for_collision(my_side, i)) {
			who_hit.x = my_side;
			who_hit.y = i;
			return true;
		}
	}
	
	return false;
}

//this gets done even if group is alive for capital explosions to work, so we check manually for alive in certain bits of the
//function
void Group_Base::check_if_scanned() {
	int start_x = static_cast<int>(myx) / terrain_tile_dim;
	int start_y = static_cast<int>(myy) / terrain_tile_dim;

	int end_x = ((static_cast<int>(myx) + width) / terrain_tile_dim) + 1;
	if (end_x > world.terrain_tiles.size())
		end_x = world.terrain_tiles.size();
	int end_y = ((static_cast<int>(myy) + height) / terrain_tile_dim) + 1;
	if (end_y > world.terrain_tiles[0].size())
		end_y = world.terrain_tiles[0].size();

	//only need to check against each set of allies scans, not against each side
	int max_flags = 0;
	for (int i = 0; i != sides.size(); ++i) {
		if (world.radar_sharing) {
			if (sides[i].my_flag > max_flags)
				//one past end
				max_flags = sides[i].my_flag + 1;
		} else
			++max_flags;

	}

	for (int flag = 0; flag != max_flags; ++flag) {
		if (flag == sides[my_side].my_flag && world.radar_sharing)
			continue;

		//for efficiency move on to next side when any square is discovered
		bool found  = false;

		for (int x = start_x; !found && x != end_x; ++x) {
			for (int y = start_y; !found && y != end_y; ++y) {				
				if (world.terrain_tiles[x][y].num_seeing[flag] > 0) {
					if (world.radar_sharing) {
						for (int side = 0; side != sides.size(); ++side) {
							if (sides[side].my_flag == flag)
								sides[side].scanned_groups[my_side][my_group] = true;
						}
					} else {
						sides[flag].scanned_groups[my_side][my_group] = true;
					}
					
					if (sides[my_side].my_flag != 0 && alive && world.radar_sharing) {
						switch (get_type()) {
						case UT_CAPITAL:
							world.cap_found = true;
							break;

						case UT_FRIGATE:
							world.frigate_found = true;
							break;

						case UT_BOMBER:
							//only say "bombers coming in" if we are actually on some sort of attack mission
							if (the_commands.move_command != MC_MOVE_GROUP || the_commands.move_target.x != my_side) {
								world.bomber_found = true;
								world.script_manager.fire_wakeup_event(WET_BOMBERS_COMING_IN);
							}
							break;

						case UT_FIGHTER:
							//removed
							//world.enemy_found = true;
							break;

						//no alert message unless in script
						case UT_FREIGHTER:
							world.script_manager.fire_wakeup_event(WET_FREIGHTER_FOUND);
							break;

						default:
							//no alert message for other ship types
							break;
						}
					}

					found = true;
				}
			}
		}
	}
}

int Group_Base::find_distance_to(int n_side, int n_group) const {
    CoordsFloat d = get_dx_dy_close(n_side, n_group);
	return static_cast<int>(slow_dist(d.x, d.y));
}

int Group_Base::find_point_distance_from_center(float x, float y) const {
    CoordsFloat d = get_dx_dy_center(x, y);
	return static_cast<int>(slow_dist(d.x, d.y));
}

void Group_Base::move() {
    //no processing, no shuffling
	if (!the_commands.aim_prop_x && !the_commands.aim_prop_y && !rem_speed_x && !rem_speed_y) {
		speed_plane_matches_target = true;
		return;
	}
	
	//due to pythagoras we move slightly faster diagonally, oh well
	float dx = the_commands.aim_prop_x * the_commands.speed;
	float dy = the_commands.aim_prop_y * the_commands.speed;
	
	convert_max_change(dx, dy);
	
	if (!is_small())
		pixel_shuffle(dx, dy);
	
	rem_speed_x = dx;
	rem_speed_y = dy;
	
	relocate(dx, dy);
   
	if (the_commands.move_command == MC_MOVE_GROUP) {
        CoordsFloat d_target = get_dx_dy_center(the_commands.move_target.x, the_commands.move_target.y);

		if (((dx >= 0 && d_target.x >= 0) || (dx <= 0 && d_target.x <= 0))
		        && ((dy >= 0 && d_target.y >= 0) || (dy <= 0 && d_target.y <= 0)))
			speed_plane_matches_target = true;
		else
			speed_plane_matches_target = false;
	}
}

void Group_Base::relocate(float dx, float dy) {
	fill_fog(false);

    //move rect first because units use it
	myx += dx;
	myy += dy;
	
	for (int i = 0; i != units.size(); ++i) {
		units[i].move(dx, dy);
		if (is_small())
			units[i].add_extra_move_frames();
	}

	fill_fog(true);
}

void Group_Base::fill_fog(bool make_visible) {
	if (sides[my_side].my_flag != 0 || (my_side != 0 && !world.radar_sharing))
		return;

	static int radius_in_squares = fog_line_lookup.size();
	
	CoordsFloat center = get_center();
	int center_x = static_cast<int>(center.x);
	int center_y = static_cast<int>(center.y);
	int center_x_tile = (center_x / terrain_tile_dim);
	int center_y_tile = (center_y / terrain_tile_dim);

	//draw vertical lines rather than horizontal lines, this way we can store a reference to
	//current_column = world.terrain_tiles[x] each iteration for efficiency
	for (int i = 0; i != radius_in_squares; ++i) {
		int start_y = center_y_tile - fog_line_lookup[i];
		if (start_y < 0)
			start_y = 0;
		int end_y = center_y_tile + fog_line_lookup[i];
		if (end_y > world.terrain_tiles[0].size())
			end_y = world.terrain_tiles[0].size();

		//fill fog_line_lookup[i] squares to top and bottom of center_y_tile - i
		int x = center_x_tile - i;
		if (x >= 0 && x < world.terrain_tiles.size()) {
			vector<TerrainTile>& current_column_left = world.terrain_tiles[x];
			for (int y = start_y; y != end_y; ++y)
				fill_fog_square(current_column_left[y], make_visible);
		}

		//minor bit of efficiency, don't draw center line twice
		if (i == 0)
			continue;

		//fill fog_line_lookup[i] squares to top and bottom of center_y_tile + i
		x = center_x_tile + i;
		if (x >= 0 && x < world.terrain_tiles.size()) {
			vector<TerrainTile>& current_column_right = world.terrain_tiles[x];			
			for (int y = start_y; y != end_y; ++y)
				fill_fog_square(current_column_right[y], make_visible);
		}
	}
}
void Group_Base::fill_fog_square(TerrainTile& tile, bool make_visible) {
	if (make_visible)
		tile.num_seeing[sides[my_side].my_flag]++;
	else
		tile.num_seeing[sides[my_side].my_flag]--;
}

void Group_Base::force_fill_fog(bool make_visible) {
	fill_fog(make_visible);
}

///

void Group_Base::run_group_ai() {
	if (!ai_stagger) {
		if (ai_filename != L"")
			ai_interpreter.get_commands();
		update_manual_target_info();
		select_targets();
		ai_stagger = stagger_frames;
	}
	--ai_stagger;

	if (the_commands.move_command == MC_MOVE_GROUP || the_commands.move_command == MC_PATROL)
		the_commands.move_target_dist = find_distance_to(the_commands.move_target.x, the_commands.move_target.y);
	else if (the_commands.move_command == MC_MOVE_POINT)
		the_commands.move_target_dist = find_point_distance_from_center(the_commands.point_target.x, the_commands.point_target.y);

	switch (the_commands.move_command) {
	case MC_NO_MOVE:
		the_commands.aim_prop_x = 0;
		the_commands.aim_prop_y = 0;
		break;

	case MC_MOVE_POINT: {
		if (is_small())
			small_ship_dont_sit_on_top_of_move_point_target();

		if (the_commands.move_target_dist > move_point_leeway) {
			CoordsFloat center = get_center();
			get_move_props(the_commands.aim_prop_x, the_commands.aim_prop_y, the_commands.point_target.x - center.x, the_commands.point_target.y - center.y);
		} else if (!is_small()) {
			the_commands.move_command = MC_NO_MOVE;
			the_commands.aim_prop_x = 0;
			the_commands.aim_prop_y = 0;
		}
		}
		break;

	case MC_MOVE_COMPASS:
		switch (the_commands.compass_target) {
		case CD_N:
			the_commands.aim_prop_x = 0;
			the_commands.aim_prop_y = -1;
			break;

		case CD_NE:
			the_commands.aim_prop_x = 0.5;
			the_commands.aim_prop_y = -0.5;
			break;

		case CD_E:
			the_commands.aim_prop_x = 1;
			the_commands.aim_prop_y = 0;
			break;

		case CD_SE:
			the_commands.aim_prop_x = 0.5;
			the_commands.aim_prop_y = 0.5;
			break;

		case CD_S:
			the_commands.aim_prop_x = 0;
			the_commands.aim_prop_y = 1;
			break;

		case CD_SW:
			the_commands.aim_prop_x = -0.5;
			the_commands.aim_prop_y = 0.5;
			break;

		case CD_W:
			the_commands.aim_prop_x = -1;
			the_commands.aim_prop_y = 0;
			break;

		case CD_NW:
			the_commands.aim_prop_x = -0.5;
			the_commands.aim_prop_y = -0.5;
			break;
		}

		if (the_commands.b_inverse) {
			the_commands.aim_prop_x = -the_commands.aim_prop_x;
			the_commands.aim_prop_y = -the_commands.aim_prop_y;
		}
		break;

		//we work out movement based on the position of a group
		//before they themselves move
		//this is good however, as
		//a) stops earlier groups in global vectors
		//gaining an advantage
		//b) makes groups gradually sweep round to targets
		//rather than immediately heading for them

	case MC_MOVE_GROUP:
		if (the_commands.b_inverse)
			convert_move_target(true);
		else if (is_small()) {
			convert_move_target();
			small_ship_dont_sit_on_top_of_move_target();
		} else if (
		(units[0].get_big_type() != WT_NONE && the_commands.move_target_dist > weapon_lookup[units[0].get_big_type()].range / 2)
		|| (units[0].get_big_type() == WT_NONE && the_commands.move_target_dist > weapon_lookup[units[0].get_small_type()].range / 2)) {
			convert_move_target();
		} else {
			the_commands.aim_prop_x = 0;
			the_commands.aim_prop_y = 0;
		}
		break;

	case MC_PATROL:
		convert_patrol_target();
		break;
	}

	check_for_going_off_screen();
	
	//always try to go round patrol circle clockwise when we first start patrolling
	if (the_commands.move_command != MC_PATROL)
		invert_patrol = false;
}

void Group_Base::convert_patrol_target() {
	int radius = the_commands.patrol_dist;

	//note correction combined with circular motion equation
	//means it is possible to exceed maximum speed,
	//oh well

	//if more than desired radius then move towards target
	//need leeway because we'll never get there exactly
	if (the_commands.move_target_dist - patrol_leeway > radius)
		convert_move_target();

	//if less than desired radius then move away from **centre**, even though we patrol closest point
	else if (the_commands.move_target_dist + patrol_leeway < radius)
		convert_move_target(true);

	else {
        CoordsFloat d = get_dx_dy_close(the_commands.move_target.x, the_commands.move_target.y);

		if (the_commands.patrol_dist) {
			the_commands.aim_prop_x = d.y / the_commands.patrol_dist;
			the_commands.aim_prop_y = -d.x / the_commands.patrol_dist;
		}
		
		if (invert_patrol) {
			the_commands.aim_prop_x = -the_commands.aim_prop_x;
			the_commands.aim_prop_y = -the_commands.aim_prop_y;
		}
	}
}

//inverse is used for two things: "moveaway", and for patrolling
void Group_Base::convert_move_target(bool inverse) {
	CoordsFloat d = get_dx_dy_center(the_commands.move_target.x, the_commands.move_target.y);

	float propx = 0;
	float propy = 0;

	//normally just get props, but if we want to move away and are currently directly 
	//on top of them, move away regardless of 0 props
	if (!get_move_props(propx, propy, d.x, d.y) && inverse == true) {
		propx = static_cast<float>(rand() % 100 + 1) / 100;
		propy = 1 - propx;
	}

	if (!inverse) {
		the_commands.aim_prop_x = propx;
		the_commands.aim_prop_y = propy;
	}
	else {
		the_commands.aim_prop_x = -propx;
		the_commands.aim_prop_y = -propy;
	}
}

///

CoordsFloat Group_Base::get_closest_point(const CoordsFloat you) const {
	CoordsFloat ret;
	CoordsFloat center = get_center();

    float xLeftDist = fabs(myx - you.x);
	float xMidDist = fabs(center.x - you.x);
	float xRightDist = fabs(myx + width - you.x);

	float yTopDist = fabs(myy - you.y);
	float yMidDist = fabs(center.y - you.y);
	float yBotDist = fabs(myy + height - you.y);

	float temp = std::min(xLeftDist, xMidDist);
	temp = std::min(temp, xRightDist);

	if (temp == xLeftDist)
		ret.x = myx;
	else if (temp == xMidDist)
		ret.x = center.x;
	else if (temp == xRightDist)
		ret.x = myx + width;

	temp = std::min(yTopDist, yMidDist);
	temp = std::min(temp, yBotDist);

	if (temp == yTopDist)
		ret.y = myy;
	else if (temp == yMidDist)
		ret.y = center.y;
	else if (temp == yBotDist)
		ret.y = myy + height;
		
	return ret;
}

int Group_Base::get_health() const {
	int total = 0;

	for (int i = 0; i != units.size(); ++i) {
		if (units[i].get_alive())
			total+= units[i].get_health();
	}

	return total;
}

int Group_Base::get_shield() const {
	int total = 0;

	for (int i = 0; i != units.size(); ++i) {
		if (units[i].get_alive())
			total += units[i].get_shield();
	}

	return total;
}

int Group_Base::get_armour() const {
	int total = 0;

	for (int i = 0; i != units.size(); ++i) {
		if (units[i].get_alive())
			total += units[i].get_armour();
	}

	return total;
}

int Group_Base::get_health_max() const {
	int per_unit = units[0].get_health_max();
	return per_unit * units.size();
}

int Group_Base::get_shield_max() const {
	int per_unit = units[0].get_shield_max();
	return per_unit * units.size();
}

int Group_Base::get_armour_max() const {
	int per_unit = units[0].get_armour_max();
	return per_unit * units.size();
}

int Group_Base::get_small_power() const {
	int powerTotal = 0;
	int powerPerUnit = units[0].get_small_number() * weapon_lookup[units[0].get_small_type()].power;

	for (int i = 0; i != units.size(); ++i) {
		if (units[i].get_alive())
			powerTotal+= powerPerUnit;

	}

	return powerTotal;
}

int Group_Base::get_big_power() const {
	int powerTotal = 0;
	int powerPerUnit = weapon_lookup[units[0].get_big_type()].power;

	for (int i = 0; i != units.size(); ++i) {
		if (units[i].get_alive())
			powerTotal+= powerPerUnit;
	}

	return powerTotal;
}

int Group_Base::get_big_ammo() const {
	int ammo_total = 0;

	for (int i = 0; i != units.size(); ++i) {
		if (units[i].get_alive())
			ammo_total+= units[0].get_big_ammo();
	}

	return ammo_total;
}

int Group_Base::get_big_ammo_max() const {
	int per_unit = units[0].get_big_ammo_max();
	return per_unit * units.size();
}

const wstring Group_Base::get_current_mission() const {
	if (ai_filename == L"recon")
		return L"Reconnaissance";
	else if (ai_filename == L"patrol")
		return L"Patrol";
	else if (ai_filename == L"attack" )
		return L"Attack";
	else if (ai_filename == L"return")
		return L"Return";
	else if (ai_filename == L"recall")
		return L"Defend";
	else
		return L"In hangar";
}

void Group_Base::get_unit_target_info(TargetDesc& target_info) {
	//select a random unit from the group, but not a dead one
	if (alive) {
		do {
			target_info.which_unit = rand() % units.size();
		} while ( !(units[target_info.which_unit].get_alive()) );
	}

	//if we are dead, give them the position of the last unit which was alive, rather than
	//one which died ages ago
	else
		target_info.which_unit = last_alive;

	CoordsFloat center = units[target_info.which_unit].get_center();

	target_info.currentx = center.x;
	target_info.currenty = center.y;

	target_info.speedx = rem_speed_x;
	target_info.speedy = rem_speed_y;
	  
	target_info.weak_spot = units[0].get_weak_spot();
}

void Group_Base::destroy() {
	for (int i = 0; i != units.size(); ++i)
		units[i].been_hit(1000);
}

void Group_Base::upkeep() {
	for (int i = 0; i != units.size(); ++i)
		units[i].upkeep();

	//how many units do we have left?
	units_left = 0;
	for (int i = 0; i != units.size(); ++i) {
		if (units[i].get_alive()) {
			++units_left;
			last_alive = i;
		}
	}

	if (units_left == 0) {
		alive = false;

		//SmallShip::upkeep() comes first so we hopefully can't fill_fog(false) twice
		if (!get_in_hangar())
			fill_fog(false);

		//don't want to set a non-existent bounding rect
		return;
	}

	units_remembered = units_left;
}

void Group_Base::dead_upkeep() {
	for (int i = 0; i != units.size(); ++i)
		units[i].dead_upkeep();
}

void Group_Base::draw_big_laser() {
	for (int i = 0; i != units.size(); ++i)
		units[i].draw_big_laser();
}

void Group_Base::draw_bound() {
	bool selected = sides[my_side].squadrons[my_squad].get_selected();

	if (on_screen) {
		if (selected || b_draw_bound) {
			draw_rect_border(screen_rect, sides[my_side].color, 1);
			
			if (units[0].get_type() != UT_PLANET) {
				SDL_Rect health_container;
				health_container.x = screen_rect.x + (screen_rect.w / 4);
				health_container.y = screen_rect.y + screen_rect.h - health_bar_height * 3;
				health_container.w = screen_rect.w / 2;
				health_container.h = health_bar_height;

				SDL_Rect health_bar;

				if (units[0].get_shield_max() > 0) {
					health_bar.x = health_container.x;
					health_bar.y = health_container.y;
					health_bar.w = get_shield() * health_container.w / get_shield_max();
					health_bar.h = health_bar_height;

					draw_rect_border(health_container, sides[my_side].color, 1);
					display.blt_fill(health_bar, standard_colors.shield_green);
				}

				//SDL clips draw rects and anyway won't be set if we didn't draw a shield bar
				health_container.x = screen_rect.x + (screen_rect.w / 4);
				health_container.y = screen_rect.y + screen_rect.h - health_bar_height * 2;
				health_container.w = screen_rect.w / 2;
				health_container.h = health_bar_height;
	            
				health_bar.x = health_container.x;
				health_bar.y = health_container.y;
				health_bar.w = get_armour() * health_container.w / get_armour_max();
				health_bar.h = health_bar_height;

				draw_rect_border(health_container, sides[my_side].color, 1);
				display.blt_fill(health_bar, standard_colors.armour_blue);
			}
		}

		int hot_key = sides[my_side].squadrons[my_squad].get_hot_key();
		if (hot_key != 0 && (selected || draw_number)) {
			int which_number = sdl_key_to_number(hot_key);
			wchar_t output[10];
			swprintf(output, 10, L"%d", which_number);
			normal_font.render(screen_rect.x + screen_rect.w, screen_rect.y + screen_rect.h, output);
		}
	}

	if (alive && selected && the_commands.move_command == MC_MOVE_POINT && my_side == 0) {
		SDL_Rect temp_rect;
		temp_rect.x = static_cast<int>(the_commands.point_target.x) - (width /2) - world.viewx;
		temp_rect.y = static_cast<int>(the_commands.point_target.y) - (height /2) - world.viewy;
		temp_rect.w = screen_rect.w;
		temp_rect.h = screen_rect.h;
		draw_rect_border(temp_rect, sides[my_side].color, 1);
	}

	if (b_draw_fuel_range) {
		CoordsFloat center = get_center();
		display.draw_circle(static_cast<int>(center.x) - world.viewx, static_cast<int>(center.y) - world.viewy, (equip_lookup[L"Fuel"].max - fuel_leeway) / 2, standard_colors.big_range_blue);
	}

	for (int i = 0; i != units.size(); ++i)
		units[i].draw_bound(b_draw_weapon_range);
}

void Group_Base::toggle_draw_bound() {
	b_draw_bound = !b_draw_bound;
}

void Group_Base::toggle_draw_number() {
	draw_number = !draw_number;
}

void Group_Base::toggle_draw_weapon_range() {
	b_draw_weapon_range = !b_draw_weapon_range;
}

void Group_Base::display_missions_fuel() {
	if (!is_small())
		b_draw_fuel_range = true;
}

void Group_Base::hide_missions_fuel() {
	if (!is_small())
		b_draw_fuel_range = false;
}

CoordsFloat Group_Base::get_dx_dy_close(int side, int group) const {
    CoordsFloat center = get_center();
	CoordsFloat target = sides[side].groups[group].get_closest_point(center);
	CoordsFloat my_coords = get_closest_point(target);

	CoordsFloat ret(target.x - my_coords.x, target.y - my_coords.y);
	return ret;
}

CoordsFloat Group_Base::get_dx_dy_center(int side, int group) const {
    CoordsFloat center = get_center();
	CoordsFloat target = sides[side].groups[group].get_center();

	CoordsFloat ret(target.x - center.x, target.y - center.y);
	return ret;
}

CoordsFloat Group_Base::get_dx_dy_center(float x, float y) const {
	CoordsFloat target(x, y);
	CoordsFloat my_coords = get_center();

	CoordsFloat ret(target.x - my_coords.x, target.y - my_coords.y);
	return ret;
}

//when moving non small ships, we move 1 pixel at a time, checking for collisions before each move and stopping moving if we see
//that we will collide
void Group_Base::pixel_shuffle(float& dx, float& dy) {	
	//if we are already overlapping then try to shuffle away from the person we are overlapping
	CoordsInt who_hit;
	if (check_for_big_collision(who_hit)) {		
		the_commands.move_target.x = who_hit.x;
		the_commands.move_target.y = who_hit.y;
		convert_move_target(true);
		dx = the_commands.aim_prop_x * the_commands.speed;
		dy = the_commands.aim_prop_y * the_commands.speed;
		check_for_going_off_screen();
		convert_max_change(dx, dy);
		return;
	}
		
	float xshuffle = 1;
	float yshuffle = 1;
	if (dx < 0)
		xshuffle = -1;
	if (dy < 0)
		yshuffle = -1;
	
	float actualx = myx;
	float actualy = myy;		
	
	while (dx) {
		myx += dx;
			
		if (check_for_big_collision())
			myx -= dx;
		else
			break;
		
		if (dx > -1 && dx < 1)
			dx = 0;
		else
			dx -= xshuffle;
	}
	
	while (dy) {		
		myy += dy;
			
		if (check_for_big_collision())
			myy -= dy;
		else
			break;
			
		if (dy > -1 && dy < 1)
			dy = 0;
		else
			dy -= yshuffle;
	}
		
	dx = myx - actualx;
	dy = myy - actualy;
	myx = actualx;
	myy = actualy;
}

void Group_Base::convert_max_change(float& dx, float& dy) {
	float dxsp = dx - rem_speed_x;
	float dysp = dy - rem_speed_y;
	
	if (dxsp > max_speed_change)
	   dx = rem_speed_x + max_speed_change;
	else if (dxsp < -max_speed_change)
	   dx = rem_speed_x - max_speed_change;
	   
	if (dysp > max_speed_change)
	   dy = rem_speed_y + max_speed_change;
	else if (dysp < -max_speed_change)
	   dy = rem_speed_y - max_speed_change;
}

void Group_Base::check_for_going_off_screen() {
	float possx = the_commands.aim_prop_x * the_commands.speed;
	float possy = the_commands.aim_prop_y * the_commands.speed;
	//if we are patrolling and hit the edge of the screen, turn back and go round the circle the other way the next frame
	if ((myx + possx < 0 && the_commands.aim_prop_x < 0)
	        || (myx + width + possx > world.width && the_commands.aim_prop_x > 0)) {
		the_commands.aim_prop_x = 0;
		invert_patrol = !invert_patrol;
	}

	if ((myy + possy < 0 && the_commands.aim_prop_y < 0)
	        || (myy + height + possy > world.height) && the_commands.aim_prop_y > 0) {
		the_commands.aim_prop_y = 0;
		invert_patrol = !invert_patrol;
	}
}

void Group_Base::init(int i_ai_stagger) {
	ai_stagger = i_ai_stagger;
	ai_interpreter.init(this, &the_commands);

	sides[my_side].squadrons[my_squad].init_pointer(this, my_group);
}

void Group_Base::report_on_script_error(const char* error, int lineNumber) {  
    if (already_ai_error)
        return;
        
    already_ai_error = true;
    
	wchar_t output[256];
	if (!sides[my_side].already_ai_error) {
		if (lineNumber != 0)	
			swprintf(output, 256, L"%ls for group %d-%d at line %d", error, my_side+1, my_group+1, lineNumber + 1);
		else
			swprintf(output, 256, L"%ls for group %d-%d", error, my_side+1, my_group+1);
		create_info_string(output, true);
	}

	write_log(output);
}

void Group_Base::update_manual_target_info() {
	//update status report window even if not in range
	//small ship groups also use these values in target selection calculations
	if (the_commands.ordered_to_fire) {
		the_commands.fire_target = the_commands.move_target;
		the_commands.fire_target_dist = the_commands.move_target_dist;
	}
}

CoordsFloat Group_Base::get_hangar_bay_coords(bool alpha_wing) {
	CoordsFloat center = get_center();

	if (units[0].get_type() == UT_PLANET) {
		if (alpha_wing)
			return CoordsFloat(center.x, center.y + 50);
		else
			return CoordsFloat(center.x, center.y - 50);
	}

	if (alpha_wing)
		return CoordsFloat(center.x - width / 4, center.y);
	else
		return CoordsFloat(center.x + width / 4, center.y);
}
