/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#ifndef GUARD_GenericHandle
#define GUARD_GenericHandle

#include "Inlines.h"

#include <cstddef>
#include <stdexcept>

template <class T> class GenericHandle {
public:
	GenericHandle(): object(0), ref(new size_t(1)) {}
	GenericHandle(T* init_object): object(init_object), ref(new size_t(1)) {}
	GenericHandle(const GenericHandle& const_gh);
	~GenericHandle();

	GenericHandle& operator=(const GenericHandle& const_gh);

	operator bool() const {return object;}
	T& operator*() const;
	T* operator->() const;

	void make_unique();

	void clear();
	T* get_pointer();

private:
	T* object;
	std::size_t* ref;
};

template <class T>
GenericHandle<T>::GenericHandle(const GenericHandle& const_gh):
object(const_gh.object), ref(const_gh.ref) {
	++*ref;
}

template <class T>
GenericHandle<T>::~GenericHandle() {
	if (--*ref == 0) {
		delete ref;
		delete object;
	}
}

template <class T>
GenericHandle<T>& GenericHandle<T>::operator=(const GenericHandle& const_gh) {
	++*const_gh.ref;

	if (--*ref == 0) {
		delete ref;
		delete object;
	}

	ref = const_gh.ref;
	object = const_gh.object;
	return *this;
}

template <class T>
T& GenericHandle<T>::operator*() const {
	if (object)
		return *object;
	throw std::runtime_error("Attempt to dereference unbound handle");
}

template <class T>
T* GenericHandle<T>::operator->() const {
	if (object)
		return object;
	throw std::runtime_error("Attempt to -> unbound handle");
}

template <class T>
void GenericHandle<T>::make_unique() {
	if (*ref != 1) {
		--*ref;
		ref = new size_t(1);
		object = object ? object->Clone() : 0;
	}
}

template <class T>
void GenericHandle<T>::clear() {
	if (--*ref == 0) {
		safe_delete(object);
		*ref = 1;
	}
}

template <class T>
T* GenericHandle<T>::get_pointer() {
	return object;
}

#endif


