/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "GenWindow_Base.h"
#include "GenWindow.h"
#include "Globals.h"
#include "Enums.h"
#include "SettingsStruct.h"
#include "Display.h"

using std::list;

GenWindow_Base::GenWindow_Base(int ix, int iy, int i_parent_id, int flags):
activate(true), closed(false), parentID(i_parent_id), border_color(standard_colors.light_blue),
transparent(flags& WFLAG_TRANSPARENT), b_static(flags & WFLAG_STATIC), cant_close(flags & WFLAG_CANTCLOSE), 
visible(!(flags & WFLAG_INVISIBLE)) {
	my_id = ++window_ids;
	
	if (flags & WFLAG_TILED) {
		rect.x = global_settings.screen_width;
		rect.y = global_settings.screen_height;
	} else {
		rect.x = ix;
		rect.y = iy;
	}

	for (list<GenWindow>::iterator iter = my_windows.begin(); iter != my_windows.end(); ++iter)
		iter->win_message(WC_IM_ACTIVE, 0, 0, window_id_all, my_id);
}

GenWindow_Base::~GenWindow_Base() {
	message_windows(WC_IM_CLOSING, 0, 0, window_id_all, my_id);
}

void GenWindow_Base::init_border() {
	border_rect.x = rect.x - small_border_size;
	border_rect.y = rect.y - small_border_size;
	border_rect.w = rect.w + (small_border_size * 2);
	border_rect.h = rect.h + (small_border_size * 2);
}

bool GenWindow_Base::mouse_d(Uint8 button, Uint16 x, Uint16 y) {
	//cursor inside window?
	if (point_in_rect(x, y, rect)) {
		if (button) {
			activate = true;
			message_windows(WC_IM_ACTIVE, 0, 0, window_id_all, my_id);
		}
		return true;
	}

	return false;
}

bool GenWindow_Base::mouse_m(Uint8 state, Uint16 x, Uint16 y) {
	//cursor inside window?
	if (point_in_rect(x, y, rect)) {
		mouse_over_ui = true;
		current_cursor_type = GENPIC_CURSOR;
		return true;
	}

	return false;
}

void GenWindow_Base::move(int distx, int disty) {
	rect.x += distx;
	rect.y += disty;

	border_rect.x += distx;
	border_rect.y += disty;

	message_windows(WC_IM_MOVING, distx, disty, window_id_all, my_id);
}

void GenWindow_Base::draw_self() {
	if (!transparent) {
		display.blt_fill(border_rect, border_color, 0.5);
		display.blt_fill(rect, standard_colors.dark_blue, 0.5);
	}
}

void GenWindow_Base::win_message(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id) {
	if (target_id == window_id_all || target_id == my_id) {
		switch (the_msg) {
		case WC_IM_ACTIVE:
			if (source_id == parentID) {
				activate = 1;
				message_windows(WC_IM_ACTIVE, 0, 0, window_id_all, my_id);
			}
			break;

		case WC_IM_CLOSING:
			if (source_id == parentID) {
				closed = true;
				message_windows(WC_IM_CLOSING, 0, 0, window_id_all, my_id);
			}
			break;

		case WC_IM_MOVING:
			if (source_id == parentID) {
				move(parem_one, parem_two);
				message_windows(WC_IM_MOVING, 0, 0, window_id_all, my_id);
			}
			break;

		case WC_YOU_CLOSE:
			closed = true;
			message_windows(WC_IM_CLOSING, 0, 0, window_id_all, my_id);
			break;
		}
	}
}

void GenWindow_Base::center_window() {
	rect.x = (global_settings.screen_width - rect.w) / 2;
	rect.y = (global_settings.screen_height - rect.h) / 2;

	init_border();
}

void GenWindow_Base::tile() {
	//shuffle away from edges if we have been created partially off screen
	if (rect.x < small_border_size)
		rect.x = small_border_size;
	if (rect.y < small_border_size)
		rect.y = small_border_size;
	if (rect.x + rect.w + small_border_size > global_settings.screen_width)
		rect.x = global_settings.screen_width - rect.w - small_border_size;
	if (rect.y + rect.h + small_border_size > global_settings.screen_height)
		rect.y = global_settings.screen_height - rect.h - small_border_size;
}

