/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

/*
This file based on glSDL 0.7 (c) David Olofson, 2001-2004
*/

#define	_GLSDL_NO_REDEFINES_
#include "GLSDL.h"

#define LEAK_TRACKING

#define	DBG(x)		/*error messages, warnings*/
#define	DBG2(x)		/*texture allocation*/
#define	DBG3(x)		/*chopping/tiling*/
#define	DBG4(x)		/*texture uploading*/
#define	DBG5(x) 	/*two-way chopping/tiling*/
#define	DBG6(x) 	/*OpenGL lib loading*/

/* Keep this on for now! Makes large surfaces faster. */
#define	FAKE_MAXTEXSIZE	256

#include <cstring>
#include <cstdlib>
#include <cmath>
#include <algorithm>
#include <stdexcept>

using std::runtime_error;

static void print_glerror(int point)
{
#if (DBG(1)+0 == 1)
	const char *err = "<unknown>";
	switch(glGetError())
	{
	  case GL_NO_ERROR:
		return;
	  case GL_INVALID_ENUM:
		err = "GL_INVALID_ENUM";
		break;
	  case GL_INVALID_VALUE:
		err = "GL_INVALID_VALUE";
		break;
	  case GL_INVALID_OPERATION:
		err = "GL_INVALID_OPERATION";
		break;
	  case GL_STACK_OVERFLOW:
		err = "GL_STACK_OVERFLOW";
		break;
	  case GL_STACK_UNDERFLOW:
		err = "GL_STACK_UNDERFLOW";
		break;
	  case GL_OUT_OF_MEMORY:
		err = "GL_OUT_OF_MEMORY";
		break;
	  default:
		err = "<unknown>";
		break;
	}
	fprintf(stderr,"OpenGL error \"%s\" at point %d.\n", err, point);
#endif
}

static struct
{
	int	do_blend;
	int	do_texture;
	GLint	texture;
	GLenum	sfactor, dfactor;
} glstate;

static void gl_reset(void)
{
	glstate.do_blend = -1;
	glstate.do_blend = -1;
	glstate.texture = -1;
	glstate.sfactor = 0xffffffff;
	glstate.dfactor = 0xffffffff;
}

static __inline__ void gl_do_blend(int on)
{
	if(glstate.do_blend == on)
		return;

	if(on)
		glEnable(GL_BLEND);
	else
		glDisable(GL_BLEND);
	glstate.do_blend = on;
}

static __inline__ void gl_do_texture(int on)
{
	if(glstate.do_texture == on)
		return;

	if(on)
		glEnable(GL_TEXTURE_2D);
	else
		glDisable(GL_TEXTURE_2D);
	glstate.do_texture = on;
}

static __inline__ void gl_blendfunc(GLenum sfactor, GLenum dfactor)
{
	if((sfactor == glstate.sfactor) && (dfactor == glstate.dfactor))
		return;

	glBlendFunc(sfactor, dfactor);

	glstate.sfactor = sfactor;
	glstate.dfactor = dfactor;
}

static __inline__ void gl_texture(GLuint tx)
{
	if(tx == glstate.texture)
		return;

	glBindTexture(GL_TEXTURE_2D, tx);
	glstate.texture = tx;
}


/*----------------------------------------------------------
	Global stuff
----------------------------------------------------------*/

#define	MAX_TEXINFOS	16384

static glSDL_TexInfo **texinfotab = NULL;
static GLint maxtexsize = 256;
static SDL_PixelFormat RGBfmt, RGBAfmt;

static void UnloadTexture(glSDL_TexInfo *txi);

static int scale = 1;

static SDL_Surface *fake_screen = NULL;


static int glSDL_BlitGL(SDL_Surface *src, SDL_Rect *srcrect,
			 SDL_Surface *dst, SDL_Rect *dstrect, int gl_effect = 0);


/* Get texinfo for a surface. */
static __inline__ glSDL_TexInfo *glSDL_GetTexInfo(SDL_Surface *surface)
{
	if(texinfotab)
		return texinfotab[surface->unused1];
	else
		return NULL;
}


/* Allocate a "blank" texinfo for a suface. */
glSDL_TexInfo *glSDL_AllocTexInfo(SDL_Surface *surface)
{
	int handle, i = 0;
	glSDL_TexInfo *txi;
	if(!surface)
		return NULL;

	txi = glSDL_GetTexInfo(surface);
	if(txi)
		return txi;		/* There already is one! --> */

	/* Find a free handle... */
	handle = -1;
	for(i = 1; i < MAX_TEXINFOS + 1; ++i)
		if(NULL == texinfotab[i])
		{
			handle = i;
			break;
		}

	if(handle < 0)
	{
		DBG(fprintf(stderr, "glSDL/wrapper: Out of handles!\n"));
		return NULL;
	}

	/* ...and hook a new texinfo struct up to it. */
	texinfotab[handle] = static_cast<glSDL_TexInfo*>(calloc(1, sizeof(glSDL_TexInfo)));
	if(!texinfotab[handle])
		return NULL;

	/* Connect the surface to the new TexInfo. */
	surface->unused1 = (Uint32)handle;

	DBG2(fprintf(stderr, "glSDL/wrapper: Allocated TexInfo %d.\n", handle));

	return texinfotab[handle];
}


static void FreeTexInfo(Uint32 handle)
{
	if(handle >= MAX_TEXINFOS)
		return;
	if(!texinfotab[handle])
		return;

	UnloadTexture(texinfotab[handle]);
	texinfotab[handle]->textures = 0;
	free(texinfotab[handle]->texture);
	texinfotab[handle]->texture = NULL;
	free(texinfotab[handle]);
	texinfotab[handle] = NULL;
	DBG2(fprintf(stderr, "glSDL/wrapper: Freed TexInfo %d.\n", handle));
}


/* Detach and free the texinfo of a surface. */
void glSDL_FreeTexInfo(SDL_Surface *surface)
{
	if(!texinfotab)
		return;

	if(!surface)
		return;

	if(!glSDL_GetTexInfo(surface))
		return;

	FreeTexInfo(surface->unused1);
	GLSDL_FIX_SURFACE(surface);
}


/*
 * Calculate chopping/tiling of a surface to
 * fit it into the smallest possible OpenGL
 * texture.
 */
static int CalcChop(SDL_Surface *s, glSDL_TexInfo *txi)
{
	int rows, vw, vh;
	int vertical = 0;
	int texsize;
	int lastw, lasth, minsize;

	vw = s->w;
	vh = s->h;

	DBG3(fprintf(stderr, "w=%d, h=%d ", vw, vh));
	if(vh > vw)
	{
		int t = vw;
		vw = vh;
		vh = t;
		vertical = 1;
		DBG3(fprintf(stderr, "(vertical) \t"));
	}

	/*
	 * Check whether this is a "huge" surface - at least one dimension
	 * must be <= than the maximum texture size, or we'll have to chop
	 * in both directions.
	 */
	if(vh > maxtexsize)
	{
		/*
		 * Very simple hack for now; we just tile
		 * both ways with maximum size textures.
		 */
		texsize = maxtexsize;

		txi->tilemode = GLSDL_TM_HUGE;
		txi->texsize = texsize;
		txi->tilew = texsize;
		txi->tileh = texsize;
		txi->tilespertex = 1;

		/* Calculate number of textures needed */
		txi->textures = (vw + texsize - 1) / texsize;
		txi->textures *= (vh + texsize - 1) / texsize;
		txi->texture = static_cast<int*>(malloc(txi->textures * sizeof(int)));
		memset(txi->texture, -1, txi->textures * sizeof(int));
		DBG5(fprintf(stderr, "two-way tiling; textures=%d\n", txi->textures));
		if(!txi->texture)
		{
			fprintf(stderr, "glSDL/wrapper: INTERNAL ERROR: Failed to allocate"
					" texture name table!\n");
			return -3;
		}
		return 0;
	}

	/* Calculate minimum size */
	rows = 1;
	lastw = vw;
	lasth = vh;
	minsize = lastw > lasth ? lastw : lasth;
	while(1)
	{
		int w, h, size;
		++rows;
		w = vw / rows;
		h = rows * vh;
		size = w > h ? w : h;
		if(size >= minsize)
		{
			--rows;
			break;
		}
		lastw = w;
		lasth = h;
		minsize = size;
	}
	if(minsize > maxtexsize)
	{
		/* Handle multiple textures for very wide/tall surfaces. */
		minsize = maxtexsize;
		rows = (vw + minsize-1) / minsize;
	}
	DBG3(fprintf(stderr, "==> minsize=%d ", minsize));
	DBG3(fprintf(stderr, "(rows=%d) \t", rows));

	/* Recalculate with nearest higher power-of-2 width. */
	for(texsize = 1; texsize < minsize; texsize <<= 1)
		;
	txi->texsize = texsize;
	rows = (vw + texsize-1) / texsize;
	DBG3(fprintf(stderr, "==> texsize=%d (rows=%d) \t", texsize, rows));

	/* Calculate number of tiles per texture */
	txi->tilespertex = txi->texsize / vh;
	DBG3(fprintf(stderr, "tilespertex=%d \t", txi->tilespertex));

	/* Calculate number of textures needed */
	txi->textures = (rows + txi->tilespertex-1) / txi->tilespertex;
	txi->texture = static_cast<int*>(malloc(txi->textures * sizeof(int)));
	memset(txi->texture, -1, txi->textures * sizeof(int));
	DBG3(fprintf(stderr, "textures=%d, ", txi->textures));
	if(!txi->texture)
	{
		fprintf(stderr, "glSDL/wrapper: INTERNAL ERROR: Failed to allocate"
				" texture name table!\n");
		return -2;
	}

	/* Set up tile size. (Only one axis supported here!) */
	if(1 == rows)
	{
		txi->tilemode = GLSDL_TM_SINGLE;
		if(vertical)
		{
			txi->tilew = vh;
			txi->tileh = vw;
		}
		else
		{
			txi->tilew = vw;
			txi->tileh = vh;
		}
	}
	else if(vertical)
	{
		txi->tilemode = GLSDL_TM_VERTICAL;
		txi->tilew = vh;
		txi->tileh = texsize;
	}
	else
	{
		txi->tilemode = GLSDL_TM_HORIZONTAL;
		txi->tilew = texsize;
		txi->tileh = vh;
	}

	DBG3(fprintf(stderr, "tilew=%d, tileh=%d\n", txi->tilew, txi->tileh));
	return 0;
}


/* Add a glSDL_TexInfo struct to an SDL_Surface */
static int glSDL_AddTexInfo(SDL_Surface *surface)
{
	glSDL_TexInfo *txi;

	if(!surface)
		return -1;
	if(IS_GLSDL_SURFACE(surface))
		return 0;	/* Do nothing */

	glSDL_AllocTexInfo(surface);
	txi = glSDL_GetTexInfo(surface);
	if(!txi)
		return -2;	/* Oops! Didn't get a texinfo... --> */

	if(CalcChop(surface, txi) < 0)
		return -3;

	SDL_SetClipRect(surface, NULL);

	return 0;
}


/* Create a surface of the prefered OpenGL RGB texture format */
static SDL_Surface *CreateRGBSurface(int w, int h)
{
	SDL_Surface *s;
	Uint32 rmask, gmask, bmask;
	int bits = 24;
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
	rmask = 0x00ff0000;
	gmask = 0x0000ff00;
	bmask = 0x000000ff;
#else
	rmask = 0x000000ff;
	gmask = 0x0000ff00;
	bmask = 0x00ff0000;
#endif
	s = SDL_CreateRGBSurface(SDL_SWSURFACE, w, h,
			bits, rmask, gmask, bmask, 0);
	if(s)
		GLSDL_FIX_SURFACE(s);

	glSDL_AddTexInfo(s);
	return s;
}


/* Create a surface of the prefered OpenGL RGBA texture format */
static SDL_Surface *CreateRGBASurface(int w, int h)
{
	SDL_Surface *s;
	Uint32 rmask, gmask, bmask, amask;
	int bits = 32;
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
	rmask = 0xff000000;
	gmask = 0x00ff0000;
	bmask = 0x0000ff00;
	amask = 0x000000ff;
#else
	rmask = 0x000000ff;
	gmask = 0x0000ff00;
	bmask = 0x00ff0000;
	amask = 0xff000000;
#endif
	s = SDL_CreateRGBSurface(SDL_SWSURFACE, w, h,
			bits, rmask, gmask, bmask, amask);
	if(s)
		GLSDL_FIX_SURFACE(s);

	glSDL_AddTexInfo(s);
	return s;
}


static void init_formats(void)
{
	SDL_Surface *s = CreateRGBSurface(1, 1);
	if(!s)
		return;
	RGBfmt = *(s->format);
	glSDL_FreeSurface(s);

	s = CreateRGBASurface(1, 1);
	if(!s)
		return;
	RGBAfmt = *(s->format);
	glSDL_FreeSurface(s);
}


static int FormatIsOk(SDL_Surface *surface)
{
	SDL_PixelFormat *pf;
	if(!surface)
		return 1;	/* Well, there ain't much we can do anyway... */

	pf = surface->format;

	/* Colorkeying requires an alpha channel! */
	if(surface->flags & SDL_SRCCOLORKEY)
		if(!pf->Amask)
			return 0;

	/* We need pitch == (width * BytesPerPixel) for glTex[Sub]Image2D() */
	if(surface->pitch != (surface->w * pf->BytesPerPixel))
		return 0;

	if(pf->Amask)
	{
		if(pf->BytesPerPixel != RGBAfmt.BytesPerPixel)
			return 0;
		if(pf->Rmask != RGBAfmt.Rmask)
			return 0;
		if(pf->Gmask != RGBAfmt.Gmask)
			return 0;
		if(pf->Bmask != RGBAfmt.Bmask)
			return 0;
		if(pf->Amask != RGBAfmt.Amask)
			return 0;
	}
	else
	{
		if(pf->BytesPerPixel != RGBfmt.BytesPerPixel)
			return 0;
		if(pf->Rmask != RGBfmt.Rmask)
			return 0;
		if(pf->Gmask != RGBfmt.Gmask)
			return 0;
		if(pf->Bmask != RGBfmt.Bmask)
			return 0;
	}
	return 1;
}



static void key2alpha(SDL_Surface *surface)
{
	int x, y;
#ifdef CKSTATS
	int transp = 0;
#endif
	Uint32 ckey = surface->format->colorkey;
	if(SDL_LockSurface(surface) < 0)
		return;

	for(y = 0; y < surface->h; ++y)
	{
		Uint32 *px = (Uint32 *)((char *)surface->pixels + y*surface->pitch);
		for(x = 0; x < surface->w; ++x)
			if(px[x] == ckey)
			{
				px[x] = 0;
#ifdef CKSTATS
				++transp;
#endif
			}
	}
#ifdef CKSTATS
	printf("glSDL/wrapper: key2alpha(); %dx%d surface, %d opaque pixels.\n",
			surface->w, surface->h,
			surface->w * surface->h - transp);
#endif
	SDL_UnlockSurface(surface);
}



/*----------------------------------------------------------
	SDL style API
----------------------------------------------------------*/

static void KillAllTextures(void)
{
	if(texinfotab)
	{
		unsigned i;
#ifdef LEAK_TRACKING
		int leaked = 0;
		for(i = 3; i < MAX_TEXINFOS + 1; ++i)
			if(texinfotab[i])
			{
				++leaked;
				fprintf(stderr, "glSDL/wrapper: Leaked TexInfo"
						" %d! (%d %dx%d textures)\n",
						i,
						texinfotab[i]->textures,
						texinfotab[i]->texsize,
						texinfotab[i]->texsize
						);
			}
		if(leaked)
			fprintf(stderr, "glSDL/wrapper: Leaked %d TexInfos!\n", leaked);
#endif
		for(i = 1; i < MAX_TEXINFOS + 1; ++i)
			FreeTexInfo(i);
		free(texinfotab);
		texinfotab = NULL;
	}
}

void glSDL_Quit(void)
{
	if(SDL_WasInit(SDL_INIT_VIDEO))
	{
		SDL_Surface *screen = SDL_GetVideoSurface();
		glSDL_FreeTexInfo(screen);
		SDL_QuitSubSystem(SDL_INIT_VIDEO);
		if(fake_screen)
		{
			glSDL_FreeTexInfo(fake_screen);
			SDL_FreeSurface(fake_screen);
			fake_screen = NULL;
		}
	}
#ifndef LEAK_TRACKING
	KillAllTextures();
#endif
}


void glSDL_FullQuit(void)
{
#ifdef LEAK_TRACKING
	KillAllTextures();
#endif
	glSDL_Quit();
	SDL_Quit();
}


void glSDL_QuitSubSystem(Uint32 flags)
{
	if(flags & SDL_INIT_VIDEO)
		glSDL_Quit();
	SDL_QuitSubSystem(flags);
}


SDL_Surface *glSDL_SetVideoMode(int width, int height, int bpp, Uint32 flags)
{
	SDL_Surface *screen;

/*
 * FIXME: Here's the place to insert proper handling of this call being
 *        used for resizing the window... For now, just make sure we
 *        don't end up with invalid texinfos and stuff no matter what.
 */
	KillAllTextures();

	//not in original glSDL?!
	if (fake_screen)	{
		SDL_FreeSurface(fake_screen);
		fake_screen = NULL;
	}

	texinfotab = static_cast<glSDL_TexInfo**>(calloc(MAX_TEXINFOS + 1, sizeof(glSDL_TexInfo *)));
	if(!texinfotab)
		return NULL;

	flags |= SDL_OPENGL;
	if(bpp == 15) {
		SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 5);
		SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 5);
		SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 5);
	}
	else if(bpp == 16) {
		SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 5);
		SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 6);
		SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 5);
	}
	else if(bpp >= 24) {
		SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 8);
		SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 8);
		SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 8);
	}

	SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);

	scale = 1;

	screen = SDL_SetVideoMode(width*scale, height*scale, bpp, flags);
	if(!screen)
	{
		KillAllTextures();
		return NULL;
	}

	GLSDL_FIX_SURFACE(screen);

#ifdef	FAKE_MAXTEXSIZE
	maxtexsize = FAKE_MAXTEXSIZE;
#else
	glGetIntegerv(GL_MAX_TEXTURE_SIZE, &maxtexsize);
#endif

	init_formats();
	gl_reset();

	if(glSDL_AddTexInfo(screen) < 0) {
		DBG(fprintf(stderr, "glSDL/wrapper: Failed to add info to screen surface!\n"));
		SDL_QuitSubSystem(SDL_INIT_VIDEO);
		return NULL;
	}

	glSDL_SetClipRect(screen, &screen->clip_rect);

	glViewport(0, 0, screen->w * scale, screen->h * scale);
	/*
	 * Note that this projection is upside down in
	 * relation to the OpenGL coordinate system.
	 */
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0, scale * (float)screen->w, scale * (float)screen->h, 0,
			-1.0, 1.0);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_CULL_FACE);

	/*
	 * Create a software shadow buffer of the requested size.
	 * This is used for blit-from-screen and simulation of
	 * direct software rendering. (Dog slow crap. It's only
	 * legitimate use is probably screen shots.)
	 */
	fake_screen = CreateRGBSurface(screen->w / scale, screen->h / scale);

	return fake_screen;
}


SDL_Surface *glSDL_GetVideoSurface(void)
{
	if(fake_screen)
		return fake_screen;
	else
		return SDL_GetVideoSurface();
}


void glSDL_UpdateRects(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
	if(IS_GLSDL_SURFACE(screen))
		glSDL_Flip(screen);
	else
		SDL_UpdateRects(screen, numrects, rects);
}


void glSDL_UpdateRect(SDL_Surface *screen, Sint32 x, Sint32 y, Uint32 w, Uint32 h)
{
	SDL_Rect r;
	r.x = x;
	r.y = y;
	r.w = w;
	r.h = h;
	glSDL_UpdateRects(screen, 1, &r);
}


int glSDL_Flip(SDL_Surface *screen)
{
	if(!IS_GLSDL_SURFACE(screen))
		return SDL_Flip(screen);

	SDL_GL_SwapBuffers();
	return 0;
}


void glSDL_FreeSurface(SDL_Surface *surface)
{
	if(!surface)
		return;
	glSDL_FreeTexInfo(surface);
	SDL_FreeSurface(surface);
}


int glSDL_LockSurface(SDL_Surface *surface)
{
	if(!surface)
		return 0;

	if(IS_GLSDL_SURFACE(surface))
	{
		if((surface == fake_screen) ||
				(SDL_GetVideoSurface() == surface))
		{
			if(scale > 1)
				return -1;

			glSDL_Invalidate(fake_screen, NULL);

			glPixelStorei(GL_UNPACK_ROW_LENGTH,
					fake_screen->pitch /
					fake_screen->format->BytesPerPixel);

			glReadPixels(0, 0, fake_screen->w, fake_screen->h,
					GL_RGB, GL_UNSIGNED_BYTE,
					fake_screen->pixels);
			return 0;
		}
		else
		{
			glSDL_Invalidate(surface, NULL);
			return SDL_LockSurface(surface);
		}
	}
	else
		return SDL_LockSurface(surface);
}


void glSDL_UnlockSurface(SDL_Surface *surface)
{
	if(!surface)
		return;

	if(IS_GLSDL_SURFACE(surface))
	{
		glSDL_UploadSurface(surface);
		if((surface == fake_screen) ||
				(SDL_GetVideoSurface() == surface))
			glSDL_BlitGL(fake_screen, NULL,
					SDL_GetVideoSurface(), NULL);
	}
	else
		SDL_UnlockSurface(surface);
}


int glSDL_SetColorKey(SDL_Surface *surface, Uint32 flag, Uint32 key)
{
	int res = SDL_SetColorKey(surface, flag, key);
	if(res < 0)
		return res;
	/*
	 * If an application does this *after* SDL_DisplayFormat,
	 * we're basically screwed, unless we want to do an
	 * in-place surface conversion hack here.
	 *
	 * What we do is just kill the glSDL texinfo... No big
	 * deal in most cases, as glSDL only converts once anyway,
	 * *unless* you keep modifying the surface.
	 */
	if(IS_GLSDL_SURFACE(surface))
		glSDL_FreeTexInfo(surface);
	return res;
}


int glSDL_SetAlpha(SDL_Surface *surface, Uint32 flag, Uint8 alpha)
{
	/*
	 * This is just parameters to OpenGL, so the actual
	 * "work" is done in glSDL_BlitSurface().
	 */
	return SDL_SetAlpha(surface, flag, alpha);
}


SDL_bool glSDL_SetClipRect(SDL_Surface *surface, SDL_Rect *rect)
{
	SDL_bool res;
	SDL_Surface *screen;
	SDL_Rect fsr;
	if(!surface)
		return SDL_FALSE;

	screen = SDL_GetVideoSurface();

	res = SDL_SetClipRect(surface, rect);
	if(!res)
		return SDL_FALSE;

	if(!rect)
	{
		fsr.x = 0;
		fsr.y = 0;
		fsr.w = screen->w;
		fsr.h = screen->h;
		rect = &fsr;
	}
	if(surface == fake_screen)
	{
		SDL_Rect r;
		r.x = rect->x;
		r.y = rect->y;
		r.w = rect->w;
		r.h = rect->h;
		surface = screen;
		SDL_SetClipRect(surface, rect);
		return SDL_TRUE;
	}
	return res;
}


static int glSDL_BlitFromGL(SDL_Rect *srcrect,
		SDL_Surface *dst, SDL_Rect *dstrect)
{
	int i, sy0, dy0;
	SDL_Rect sr, dr;

	if(scale > 1)
		return -1;
/*
FIXME: Some clipping, perhaps...? :-)
*/
	/* In case the destination has an OpenGL texture... */
	glSDL_Invalidate(dst, dstrect);

	/* Abuse the fake screen buffer a little. */
	glPixelStorei(GL_UNPACK_ROW_LENGTH, fake_screen->pitch /
			fake_screen->format->BytesPerPixel);
	if(srcrect)
		glReadPixels(srcrect->x, srcrect->y, srcrect->w, srcrect->h,
				GL_RGB, GL_UNSIGNED_BYTE, fake_screen->pixels);
	else
		glReadPixels(0, 0, fake_screen->w, fake_screen->h,
				GL_RGB, GL_UNSIGNED_BYTE, fake_screen->pixels);

	/* Blit to the actual target! (Vert. flip... Uuurgh!) */
	if(srcrect)
		sr = *srcrect;
	else
	{
		sr.x = sr.y = 0;
		sr.w = dst->w;
		srcrect = &sr;
	}

	if(dstrect)
		dr = *dstrect;
	else
	{
		dr.x = dr.y = 0;
		dstrect = &dr;
	}

	i = srcrect->h;
	sy0 = srcrect->y;
	dy0 = dstrect->y + dstrect->h - 1;
	while(i--)
	{
		sr.y = sy0 + i;
		dr.y = dy0 - i;
		sr.h = 1;
		if(SDL_BlitSurface(fake_screen, &sr, dst, &dr) < 0)
			return -1;
	}
	return 0;
}


static __inline__ void BlitGL_single(glSDL_TexInfo *txi,
		const SDL_Rect& src_rect, const SDL_Rect& dst_rect, unsigned char alpha, int gl_effect)
{
	if(!txi->textures)
		return;
	if(-1 == txi->texture[0])
		return;
	gl_texture(txi->texture[0]);

	float texscale = 1.0f / (float)txi->texsize;
	float sx2 = (src_rect.x + src_rect.w) * texscale;
	float sy2 = (src_rect.y + src_rect.h) * texscale;
	float sx1 = src_rect.x * texscale;
	float sy1 = src_rect.y * texscale;

	if (gl_effect & GL_EFFECT_HFLIPPED)
		std::swap(sx1, sx2);

	glBegin(GL_QUADS);
	glColor4ub(255, 255, 255, alpha);
	glTexCoord2f(sx1, sy1);
	glVertex2i(dst_rect.x, dst_rect.y);
	glTexCoord2f(sx2, sy1);
	glVertex2i(dst_rect.x + dst_rect.w, dst_rect.y);
	glTexCoord2f(sx2, sy2);
	glVertex2i(dst_rect.x + dst_rect.w, dst_rect.y + dst_rect.h);
	glTexCoord2f(sx1, sy2);
	glVertex2i(dst_rect.x, dst_rect.y + dst_rect.h);
	glEnd();
}


static void BlitGL_htile(glSDL_TexInfo *txi,
		const SDL_Rect& src_rect, const SDL_Rect& dst_rect, unsigned char alpha, int gl_effect)
{
	float texscale = 1.0f / (float)txi->texsize;
	float opengl_tile_height = (float)txi->tileh * texscale;
	float sx2 = (src_rect.x + src_rect.w) * texscale;
	float sy2 = (src_rect.y + src_rect.h) * texscale;
	float sx1 = src_rect.x * texscale;
	float sy1 = src_rect.y * texscale;
	float tile = floor(sx1);
	int which_texture = (int)tile / txi->tilespertex;
	float y_offset = ((int)tile % txi->tilespertex) * opengl_tile_height;

	if(which_texture >= txi->textures)
		return;
	if(-1 == txi->texture[which_texture])
		return;
	gl_texture(txi->texture[which_texture]);

	glBegin(GL_QUADS);
	while(tile < sx2)
	{
		int tdx1 = dst_rect.x;
		int tdx2 = dst_rect.x + dst_rect.w;
		float tsx1 = sx1 - tile;
		float tsx2 = sx2 - tile;

		/* Clip to current tile */
		if(tsx1 < 0.0f)
		{
			tdx1 -= tsx1 * txi->texsize * ((float)dst_rect.w / src_rect.w);
			tsx1 = 0.0;
		}
		if(tsx2 > 1.0f)
		{
			tdx2 -= (tsx2 - 1.0) * txi->texsize* ((float)dst_rect.w / src_rect.w);
			tsx2 = 1.0;
		}

		/* Maybe select next texture? */
		if(y_offset + opengl_tile_height > 1.0)
		{
			++which_texture;
			glEnd();
			if(which_texture >= txi->textures)
				return;
			if(-1 == txi->texture[which_texture])
				return;
			gl_texture(txi->texture[which_texture]);
			y_offset = 0.0;
			glBegin(GL_QUADS);
		}

		glColor4ub(255, 255, 255, alpha);
		glTexCoord2f(tsx1, y_offset + sy1);
		glVertex2i(tdx1, dst_rect.y);
		glTexCoord2f(tsx2, y_offset + sy1);
		glVertex2i(tdx2, dst_rect.y);
		glTexCoord2f(tsx2, y_offset + sy2);
		glVertex2i(tdx2, dst_rect.y + dst_rect.h);
		glTexCoord2f(tsx1, y_offset + sy2);
		glVertex2i(tdx1, dst_rect.y + dst_rect.h);

		tile += 1.0;
		y_offset += opengl_tile_height;
	}
	glEnd();
}

static void BlitGL_htile_hflipped(glSDL_TexInfo *txi,
		const SDL_Rect& src_rect, const SDL_Rect& dst_rect, unsigned char alpha, int gl_effect)
{
	float texscale = 1.0 / (float)txi->texsize;
	float opengl_tile_height = (float)txi->tileh * texscale;
	float sx2 = (src_rect.x + src_rect.w - 1) * texscale;
	float sy2 = (src_rect.y + src_rect.h) * texscale;
	float sx1 = (src_rect.x - 1) * texscale;
	float sy1 = src_rect.y * texscale;
	std::swap(sx1, sx2);
	float tile = floor(sx1);
	int which_texture = (int)tile / txi->tilespertex;
	float y_offset = ((int)tile % txi->tilespertex) * opengl_tile_height;

	if(which_texture >= txi->textures)
		return;
	if(-1 == txi->texture[which_texture])
		return;
	gl_texture(txi->texture[which_texture]);

	glBegin(GL_QUADS);
	while(tile > sx2)
	{
		int tdx1 = dst_rect.x;
		int tdx2 = dst_rect.x + dst_rect.w;
		float tsx1 = sx1 - tile;
		float tsx2 = sx2 - tile;

		/* Clip to current tile */
		if(tsx2 < 0.0)
		{
			tdx2 += tsx2 * txi->texsize * ((float)dst_rect.w / src_rect.w);
			tsx2 = 0.0;
		}
		if(tsx1 > 1.0)
		{
			tdx1 += (tsx1 - 1.0) * txi->texsize* ((float)dst_rect.w / src_rect.w);
			tsx1 = 1.0;
		}

		/* Maybe select next texture? */
		if(y_offset < 0.0)
		{
			--which_texture;
			glEnd();
			if(which_texture < 0)
				return;
			if(-1 == txi->texture[which_texture])
				return;
			gl_texture(txi->texture[which_texture]);
			y_offset = 0.0;
			glBegin(GL_QUADS);
		}

		glColor4ub(255, 255, 255, alpha);
		glTexCoord2f(tsx1, y_offset + sy1);
		glVertex2i(tdx1, dst_rect.y);
		glTexCoord2f(tsx2, y_offset + sy1);
		glVertex2i(tdx2, dst_rect.y);
		glTexCoord2f(tsx2, y_offset + sy2);
		glVertex2i(tdx2, dst_rect.y + dst_rect.h);
		glTexCoord2f(tsx1, y_offset + sy2);
		glVertex2i(tdx1, dst_rect.y + dst_rect.h);

		tile -= 1.0;
		y_offset -= opengl_tile_height;
	}
	glEnd();
}

static void BlitGL_vtile(glSDL_TexInfo *txi,
		const SDL_Rect& src_rect, const SDL_Rect& dst_rect, unsigned char alpha, int gl_effect)
{
	float texscale = 1.0 / (float)txi->texsize;
	float tilew = (float)txi->tilew * texscale;
	float opengl_scale_per_pixel = 1.0 / (float)txi->texsize;
	float opengl_tile_height = (float)txi->tileh * opengl_scale_per_pixel;
	float sx2 = (src_rect.x + src_rect.w) * texscale;
	float sy2 = (src_rect.y + src_rect.h) * texscale;
	float sx1 = src_rect.x * texscale;
	float sy1 = src_rect.y * texscale;
	float tile = floor(sy1);
	int tex = (int)tile / txi->tilespertex;
	float xo = ((int)tile % txi->tilespertex) * tilew;

	if(tex >= txi->textures)
		return;
	if(-1 == txi->texture[tex])
		return;
	gl_texture(txi->texture[tex]);

	glBegin(GL_QUADS);
	while(tile < sy2)
	{
		int tdy1 = dst_rect.y;
		int tdy2 = dst_rect.y + dst_rect.h;
		float tsy1 = sy1 - tile;
		float tsy2 = sy2 - tile;

		/* Clip to current tile */
		if(tsy1 < 0.0)
		{
			tdy1 -= tsy1 * txi->texsize * ((float)dst_rect.w / src_rect.w);
			tsy1 = 0.0;
		}
		if(tsy2 > 1.0)
		{
			tdy2 -= (tsy2 - 1.0) * txi->texsize * ((float)dst_rect.w / src_rect.w);
			tsy2 = 1.0;
		}

		/* Maybe select next texture? *//*
		if(xo + tilew > 1.0)
		{
			++tex;
			glEnd();
			if(tex >= txi->textures)
				return;
			if(-1 == txi->texture[tex])
				return;
			gl_texture(txi->texture[tex]);
			xo = 0.0;
			glBegin(GL_QUADS);
		}*/

		glColor4ub(255, 255, 255, alpha);
		glTexCoord2f(xo + sx1, tsy1);
		glVertex2i(dst_rect.x, tdy1);
		glTexCoord2f(xo + sx2, tsy1);
		glVertex2i(dst_rect.x + dst_rect.w, tdy1);
		glTexCoord2f(xo + sx2, tsy2);
		glVertex2i(dst_rect.x + dst_rect.w, tdy2);
		glTexCoord2f(xo + sx1, tsy2);
		glVertex2i(dst_rect.x, tdy2);
	//flip
		tile -= 1.0;
		
		xo -= tilew;
		//
	}
	glEnd();
}


static void BlitGL_hvtile(SDL_Surface *src, glSDL_TexInfo *txi,
		const SDL_Rect& src_rect, const SDL_Rect& dst_rect, unsigned char alpha, int gl_effect)
{
	int x, y, last_tex, tex;
	float texscale = 1.0 / (float)txi->texsize;
	int tilesperrow = (src->w + txi->tilew - 1) / txi->tilew;
	float sx2 = (src_rect.x + src_rect.w) * texscale;
	float sy2 = (src_rect.y + src_rect.h) * texscale;
	float sx1 = src_rect.x * texscale;
	float sy1 = src_rect.y * texscale;

	last_tex = tex = floor(sy1) * tilesperrow + floor(sx1);
	if(tex >= txi->textures)
		return;
	if(-1 == txi->texture[tex])
		return;
	gl_texture(txi->texture[tex]);

	glBegin(GL_QUADS);
	for(y = floor(sy1); y < sy2; ++y)
	{
		int tdy1 = dst_rect.y;
		int tdy2 = dst_rect.y + dst_rect.h;
		float tsy1 = sy1 - y;
		float tsy2 = sy2 - y;

		/* Clip to current tile */
		if(tsy1 < 0.0)
		{
			tdy1 -= tsy1 * txi->texsize * ((float)dst_rect.h / src_rect.h);
			tsy1 = 0.0;
		}
		if(tsy2 > 1.0)
		{
			tdy2 -= (tsy2 - 1.0) * txi->texsize * ((float)dst_rect.h / src_rect.h);
			tsy2 = 1.0;
		}
		for(x = floor(sx1); x < sx2; ++x)
		{
			int tdx1 = dst_rect.x;
			int tdx2 = dst_rect.x + dst_rect.w;
			float tsx1 = sx1 - x;
			float tsx2 = sx2 - x;

			/* Clip to current tile */
			if(tsx1 < 0.0)
			{
				tdx1 -= tsx1 * txi->texsize * ((float)dst_rect.w / src_rect.w);
				tsx1 = 0.0;
			}
			if(tsx2 > 1.0)
			{
				tdx2 -= (tsx2 - 1.0) * txi->texsize * ((float)dst_rect.w / src_rect.w);
				tsx2 = 1.0;
			}

			/* Select texture */
			tex = y * tilesperrow + x;
			if(tex != last_tex)
			{
				glEnd();
				if(tex >= txi->textures)
					return;
				if(-1 == txi->texture[tex])
					return;
				gl_texture(txi->texture[tex]);
				last_tex = tex;
				glBegin(GL_QUADS);
			}

			glColor4ub(255, 255, 255, alpha);
			glTexCoord2f(tsx1, tsy1);
			glVertex2i(tdx1, tdy1);
			glTexCoord2f(tsx2, tsy1);
			glVertex2i(tdx2, tdy1);
			glTexCoord2f(tsx2, tsy2);
			glVertex2i(tdx2, tdy2);
			glTexCoord2f(tsx1, tsy2);
			glVertex2i(tdx1, tdy2);
		}
	}
	glEnd();
}

static __inline__ bool blitclip(SDL_Rect& r, SDL_Rect& s, const SDL_Rect& clip)
{
	int sx1 = s.x;
	int sy1 = s.y;
	int sx2 = sx1 + s.w;
	int sy2 = sy1 + s.h;

	int dx1 = r.x;
	int dy1 = r.y;
	int dx2 = dx1 + r.w;
	int dy2 = dy1 + r.h;

	/* Clip to destination cliprect */
	if(dx1 < clip.x)
	{
		sx1 += (clip.x - dx1) * ((float)s.w / r.w);
		dx1 = clip.x;
	}
	if(dy1 < clip.y)
	{
		sy1 += (clip.y - dy1) * ((float)s.w / r.w);
		dy1 = clip.y;
	}
	if(dx2 > clip.x + clip.w) {
		sx2 += (clip.x + clip.w - dx2) * ((float)s.w / r.w);
		dx2 = clip.x + clip.w;
	}
	if(dy2 > clip.y + clip.h) {
		sy2 += (clip.y + clip.h - dy2) * ((float)s.w / r.w);
		dy2 = clip.y + clip.h;
	}

	/* Cull nop/off-screen blits */
	if(dx1 >= dx2 || dy1 >= dy2)
		return false;

	r.x = dx1;
	r.y = dy1;
	r.w = dx2 - dx1;
	r.h = dy2 - dy1;

	s.x = sx1;
	s.y = sy1;
	s.w = sx2 - sx1;
	s.h = sy2 - sy1;

	return true;
}

static bool clip_off_screen_only(const SDL_Rect& r, const SDL_Rect& clip) {
	if(r.x > clip.x + clip.w)
		return false;
	if(r.y > clip.y + clip.h)
		return false;
	if(r.x + r.w < clip.x)
		return false;
	if(r.y + r.h < clip.y)
		return false;
	return true;
}

static int glSDL_BlitGL(SDL_Surface *src, SDL_Rect *srcrect,
			 SDL_Surface *dst, SDL_Rect *dstrect, int gl_effect)
{
	glSDL_TexInfo *txi;
	SDL_Rect s, r;
	unsigned char alpha;
	if(!src || !dst)
		return -1;

	/* Get source and destination coordinates */
	if(srcrect)
		s = *srcrect;
	else
	{
		s.x = s.y = 0;
		s.w = src->w;
		s.h = src->h;
	}
	if(dstrect)
		r = *dstrect;
	else {
		r.x = r.y = 0;
		r.w = dst->w;
		r.h = dst->h;
	}

	if (!r.w || !r.h) {
		r.w = s.w;
		r.h = s.h;
	}

	//FIXME flipped v/hv tiled images is not currently supported so we cannot flip images more than 256 high,
	//plus anyway images are not cropped to the screen if flipped so flipping large images would cause
	//immense slowness - so add the check for 1024 width, too
	if (gl_effect & GL_EFFECT_HFLIPPED && (s.w > 1024 || s.h > 256))
		throw runtime_error("Attempted flipped blit for huge image.");

	//FIXME we divide by these for scaled images, and we don't want to divide by 0
	//though actually if s.w is non-zero r.w is always non zero because dest gets source dimensions if none specified
	if (!s.w || !r.w)
		throw runtime_error("0 for width of either source or dest, avoiding divide by 0");

	bool draw;
	if (gl_effect & GL_EFFECT_HFLIPPED)
		draw = clip_off_screen_only(r, dst->clip_rect);
	else
		draw = blitclip(r, s, dst->clip_rect);

	if (!draw) {
		if(dstrect)
			dstrect->w = dstrect->h = 0;
		return 0;
	}

	//would only work for non flipped blits anyway
	/* Write back the resulting cliprect */
	/*if(dstrect)
		*dstrect = r;
	*/	

	/* Make sure we have a source with a valid texture */
	glSDL_UploadSurface(src);
	txi = glSDL_GetTexInfo(src);
	if(!txi)
		return -1;

	/* Set up blending */
	if(src->flags & (SDL_SRCALPHA | SDL_SRCCOLORKEY))
	{
		gl_blendfunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		gl_do_blend(1);
	}
	else
		gl_do_blend(0);

	/* Enable texturing */
	gl_do_texture(1);

	/*
	 * Note that we actually *prevent* the use of "full surface alpha"
	 * and alpha channel in combination - to stay SDL 2D compatible.
	 */
	if((src->flags & SDL_SRCALPHA) &&
			(!src->format->Amask || (src->flags & SDL_SRCCOLORKEY)))
		alpha = src->format->alpha;
	else
		alpha = 255;

	/* Render! */
	switch(txi->tilemode)
	{
	  case GLSDL_TM_SINGLE:
		BlitGL_single(txi, s, r, alpha, gl_effect);
		break;
	  case GLSDL_TM_HORIZONTAL:
		if (gl_effect & GL_EFFECT_HFLIPPED)
			BlitGL_htile_hflipped(txi, s, r, alpha, gl_effect);
		else
			BlitGL_htile(txi, s, r, alpha, gl_effect);
		break;
	  case GLSDL_TM_VERTICAL:
		BlitGL_vtile(txi, s, r, alpha, gl_effect);
		break;
	  case GLSDL_TM_HUGE:
		BlitGL_hvtile(src, txi, s, r, alpha, gl_effect);
		break;
	}

	//these two lines not in original glSDL, but needed so we can draw lines etc elsewhere
	gl_do_texture(0);
	gl_do_blend(0);

	return 0;
}


int glSDL_BlitSurface(SDL_Surface *src, SDL_Rect *srcrect,
		SDL_Surface *dst, SDL_Rect *dstrect, int gl_effect)
{
	SDL_Surface *vs;
	if(!src || !dst)
		return -1;

	/*
	 * Figure out what to do:
	 *      Not using glSDL:        SDL_BlitSurface()
	 *      screen->screen:         _glSDL_BlitFromGL() + _glSDL_BlitGL()
	 *      surface->screen:        _glSDL_BlitGL()
	 *      screen->surface:        _glSDL_BlitFromGL()
	 *      surface->surface:       SDL_BlitSurface()
	 */

	vs = SDL_GetVideoSurface();
	if(src == fake_screen)
		src = vs;
	if(dst == fake_screen)
		dst = vs;
	if(src == vs)
	{
		if(dst == vs)
		{
			glSDL_BlitFromGL(srcrect, fake_screen, dstrect);
			return glSDL_BlitGL(fake_screen, srcrect,
					dst, dstrect, gl_effect);
		}
		else
		{
			return glSDL_BlitFromGL(srcrect, dst, dstrect);
		}
	}
	else
	{
		if(dst == vs)
		{
			return glSDL_BlitGL(src, srcrect,
					dst, dstrect, gl_effect);
		}
		else
		{
			glSDL_Invalidate(dst, dstrect);
			return SDL_BlitSurface(src, srcrect, dst, dstrect);
		}
	}
}


int glSDL_FillRect(SDL_Surface *dst, SDL_Rect *dstrect, const OpenGLColor& color, float alpha)
{
	SDL_Surface *vs = SDL_GetVideoSurface();
	SDL_PixelFormat *pf = dst->format;
	int dx1, dy1, dx2, dy2;

	if(dst == fake_screen)
		dst = vs;
	if(vs != dst)
		glSDL_Invalidate(dst, dstrect);

	if(dstrect)
	{
		dx1 = dstrect->x;
		dy1 = dstrect->y;
		dx2 = dx1 + dstrect->w;
		dy2 = dy1 + dstrect->h;
		if(dx1 < dst->clip_rect.x)
			dx1 = dst->clip_rect.x;
		if(dy1 < dst->clip_rect.y)
			dy1 = dst->clip_rect.y;
		if(dx2 > dst->clip_rect.x + dst->clip_rect.w)
			dx2 = dst->clip_rect.x + dst->clip_rect.w;
		if(dy2 > dst->clip_rect.y + dst->clip_rect.h)
			dy2 = dst->clip_rect.y + dst->clip_rect.h;
		dstrect->x = dx1;
		dstrect->y = dy1;
		dstrect->w = dx2 - dx1;
		dstrect->h = dy2 - dy1;
		if(!dstrect->w || !dstrect->h)
			return 0;
	}
	else
	{
		dx1 = dst->clip_rect.x;
		dy1 = dst->clip_rect.y;
		dx2 = dx1 + dst->clip_rect.w;
		dy2 = dy1 + dst->clip_rect.h;
	}

	gl_do_texture(0);

	if (alpha != -1) {
		gl_blendfunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		gl_do_blend(1);
	} else
		gl_do_blend(0);

	glColor4f(color.r, color.g, color.b, alpha);
	glBegin(GL_QUADS);
	glVertex2i(dx1, dy1);
	glVertex2i(dx2, dy1);
	glVertex2i(dx2, dy2);
	glVertex2i(dx1, dy2);
	glEnd();

	if (alpha != -1)
		gl_do_blend(0);

	return 0;
}


SDL_Surface *glSDL_DisplayFormat(SDL_Surface *surface)
{
	SDL_Surface *s, *tmp;

	int use_rgba = (surface->flags & SDL_SRCCOLORKEY) ||
			((surface->flags & SDL_SRCALPHA) &&
			surface->format->Amask);
	if(use_rgba)
		tmp = SDL_ConvertSurface(surface, &RGBAfmt, SDL_SWSURFACE);
	else
		tmp = SDL_ConvertSurface(surface, &RGBfmt, SDL_SWSURFACE);
	if(!tmp)
		return NULL;
	GLSDL_FIX_SURFACE(tmp);
	SDL_SetAlpha(tmp, 0, 0);

	if(surface->flags & SDL_SRCCOLORKEY)
	{
		/*
			* We drop colorkey data here, but we have to,
			* or we'll run into trouble when converting,
			* in particular from indexed color formats.
			*/
		SDL_SetColorKey(tmp, SDL_SRCCOLORKEY,
				surface->format->colorkey);
		key2alpha(tmp);
	}
	SDL_SetColorKey(tmp, 0, 0);

	if(use_rgba)
		s = CreateRGBASurface(surface->w, surface->h);
	else
		s = CreateRGBSurface(surface->w, surface->h);
	if(!s)
	{
		glSDL_FreeSurface(tmp);
		return NULL;
	}
	SDL_BlitSurface(tmp, NULL, s, NULL);
	glSDL_FreeSurface(tmp);

	if(surface->flags & SDL_SRCALPHA)
		SDL_SetAlpha(s, SDL_SRCALPHA,
				surface->format->alpha);
	return s;
}


SDL_Surface *glSDL_DisplayFormatAlpha(SDL_Surface *surface)
{
	SDL_Surface *s, *tmp;

	tmp = SDL_ConvertSurface(surface, &RGBAfmt, SDL_SWSURFACE);
	if(!tmp)
		return NULL;
	GLSDL_FIX_SURFACE(tmp);

	SDL_SetAlpha(tmp, 0, 0);
	SDL_SetColorKey(tmp, 0, 0);
	s = CreateRGBASurface(surface->w, surface->h);
	if(!s)
	{
		glSDL_FreeSurface(tmp);
		return NULL;
	}
	SDL_BlitSurface(tmp, NULL, s, NULL);
	glSDL_FreeSurface(tmp);

	if(surface->flags & SDL_SRCCOLORKEY)
	{
		SDL_SetColorKey(s, SDL_SRCCOLORKEY,
				surface->format->colorkey);
		key2alpha(s);
	}
	if(surface->flags & SDL_SRCALPHA)
		SDL_SetAlpha(s, SDL_SRCALPHA,
				surface->format->alpha);
	return s;
}


SDL_Surface *glSDL_ConvertSurface
			(SDL_Surface *src, SDL_PixelFormat *fmt, Uint32 flags)
{
	SDL_Surface *s = SDL_ConvertSurface(src, fmt, flags);
	if(s)
		GLSDL_FIX_SURFACE(s);
	return s;
}


SDL_Surface *glSDL_CreateRGBSurface
			(Uint32 flags, int width, int height, int depth, 
			Uint32 Rmask, Uint32 Gmask, Uint32 Bmask, Uint32 Amask)
{
	SDL_Surface *s = SDL_CreateRGBSurface(flags, width, height, depth, 
			Rmask, Gmask, Bmask, Amask);
	if(s)
		GLSDL_FIX_SURFACE(s);
	return s;
}


SDL_Surface *glSDL_CreateRGBSurfaceFrom(void *pixels,
			int width, int height, int depth, int pitch,
			Uint32 Rmask, Uint32 Gmask, Uint32 Bmask, Uint32 Amask)
{
	SDL_Surface *s = SDL_CreateRGBSurfaceFrom(pixels,
			width, height, depth, pitch,
			Rmask, Gmask, Bmask, Amask);
	if(s)
		GLSDL_FIX_SURFACE(s);
	return s;
}


SDL_Surface *glSDL_LoadBMP(const char *file)
{
	SDL_Surface *s = SDL_LoadBMP(file);
	if(s)
		GLSDL_FIX_SURFACE(s);
	return s;
}


int glSDL_SaveBMP(SDL_Surface *surface, const char *file)
{
	SDL_Rect r;
	SDL_Surface *buf;
	SDL_Surface *screen = SDL_GetVideoSurface();

	if((surface != screen) && (surface != fake_screen))
		return SDL_SaveBMP(surface, file);

	buf = CreateRGBSurface(fake_screen->w, fake_screen->h);

	r.x = 0;
	r.y = 0;
	r.w = fake_screen->w;
	r.h = fake_screen->h;
	if(glSDL_BlitFromGL(&r, buf, &r) < 0)
		return -1;
	
	return SDL_SaveBMP(buf, file);

	glSDL_FreeSurface(buf);
}




/*----------------------------------------------------------
	glSDL specific API extensions
----------------------------------------------------------*/

void glSDL_Invalidate(SDL_Surface *surface, SDL_Rect *area)
{
	glSDL_TexInfo *txi;
	if(!surface)
		return;
	txi = glSDL_GetTexInfo(surface);
	if(!txi)
		return;
	if(!area)
	{
		txi->invalid_area.x = 0;
		txi->invalid_area.y = 0;
		txi->invalid_area.w = surface->w;
		txi->invalid_area.h = surface->h;
		return;
	}
	txi->invalid_area = *area;
}


static int InitTexture(SDL_Surface *datasurf, glSDL_TexInfo *txi, int tex)
{
	glGenTextures(1, (unsigned int *)&txi->texture[tex]);
	glBindTexture(GL_TEXTURE_2D, txi->texture[tex]);
	glPixelStorei(GL_UNPACK_ROW_LENGTH, datasurf->pitch /
			datasurf->format->BytesPerPixel);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexImage2D(GL_TEXTURE_2D, 0,
			datasurf->format->Amask ? GL_RGBA8 : GL_RGB8,
			txi->texsize, txi->texsize, 0,
			datasurf->format->Amask ? GL_RGBA : GL_RGB,
			GL_UNSIGNED_BYTE, NULL);
	print_glerror(1);
	return 0;
}


/* Image tiled horizontally (wide surface), or not at all */
static int UploadHoriz(SDL_Surface *datasurf, glSDL_TexInfo *txi)
{
	int bpp = datasurf->format->BytesPerPixel;
	int res;
	int tex = 0;
	int fromx = 0;
	int toy = txi->texsize;	/* To init first texture */
	while(1)
	{
		int thistw = datasurf->w - fromx;
		if(thistw > txi->tilew)
			thistw = txi->tilew;
		else if(thistw <= 0)
			break;
		if(toy + txi->tileh > txi->texsize)
		{
			toy = 0;
			res = InitTexture(datasurf, txi, tex);
			if(res < 0)
				return res;
			++tex;
		}
		glTexSubImage2D(GL_TEXTURE_2D, 0, 0, toy,
				thistw, txi->tileh,
				datasurf->format->Amask ? GL_RGBA : GL_RGB,
				GL_UNSIGNED_BYTE,
				(char *)datasurf->pixels + bpp * fromx);
		print_glerror(2);
		fromx += txi->tilew;
		toy += txi->tileh;
		glFlush();
	}
	return 0;
}


/* Image tiled vertically (tall surface) */
static int UploadVert(SDL_Surface *datasurf, glSDL_TexInfo *txi)
{
	int res;
	int tex = 0;
	int fromy = 0;
	int tox = txi->texsize;	/* To init first texture */
	while(1)
	{
		int thisth = datasurf->h - fromy;
		if(thisth > txi->tileh)
			thisth = txi->tileh;
		else if(thisth <= 0)
			break;
		if(tox + txi->tilew > txi->texsize)
		{
			tox = 0;
			res = InitTexture(datasurf, txi, tex);
			if(res < 0)
				return res;
			++tex;
		}
		glTexSubImage2D(GL_TEXTURE_2D, 0, tox, 0,
				txi->tilew, thisth,
				datasurf->format->Amask ? GL_RGBA : GL_RGB,
				GL_UNSIGNED_BYTE,
				(char *)datasurf->pixels + datasurf->pitch * fromy);
		print_glerror(3);
		fromy += txi->tileh;
		tox += txi->tilew;
		glFlush();
	}
	return 0;
}


/* Image tiled two-way (huge surface) */
static int UploadHuge(SDL_Surface *datasurf, glSDL_TexInfo *txi)
{
	int bpp = datasurf->format->BytesPerPixel;
	int res;
	int tex = 0;
	int y = 0;
	while(y < datasurf->h)
	{
		int x;
		int thisth = datasurf->h - y;
		if(thisth > txi->tileh)
			thisth = txi->tileh;
		x = 0;
		while(x < datasurf->w)
		{
			int thistw = datasurf->w - x;
			if(thistw > txi->tilew)
				thistw = txi->tilew;
			res = InitTexture(datasurf, txi, tex++);
			if(res < 0)
				return res;
//			DBG5(printf("glTexSubImage(x = %d, y = %d, w = %d, h = %d)\n",
//					x, y, thistw, thisth);)
			glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0,
					thistw, thisth,
					datasurf->format->Amask ? GL_RGBA : GL_RGB,
					GL_UNSIGNED_BYTE,
					(char *)datasurf->pixels +
					datasurf->pitch * y + bpp * x);
			print_glerror(4);
			x += txi->tilew;
			glFlush();
		}
		y += txi->tileh;
	}
	return 0;
}


/* Upload all textures for a surface. */
static int UploadTextures(SDL_Surface *datasurf, glSDL_TexInfo *txi)
{
	switch(txi->tilemode)
	{
	  case GLSDL_TM_SINGLE:
	  case GLSDL_TM_HORIZONTAL:
		UploadHoriz(datasurf, txi);
		break;
	  case GLSDL_TM_VERTICAL:
		UploadVert(datasurf, txi);
		break;
	  case GLSDL_TM_HUGE:
		UploadHuge(datasurf, txi);
		break;
	}
	return 0;
}


int glSDL_UploadSurface(SDL_Surface *surface)
{
	SDL_Surface *datasurf = surface;
	glSDL_TexInfo *txi;
	int i;
	/* 
	 * For now, we just assume that *every* texture needs
	 * conversion before uploading.
	 */

	/* If there's no TexInfo, add one. */
	if(!IS_GLSDL_SURFACE(surface))
		glSDL_AddTexInfo(surface);

	txi = glSDL_GetTexInfo(surface);
	if(!txi)
		return -1;

	/* No partial updates implemented yet... */
	if(txi->invalid_area.w)
		glSDL_UnloadSurface(surface);
	else
	{
		int missing = 0;
		if(txi->textures)
		{
			for(i = 0; i < txi->textures; ++i)
				if(-1 == txi->texture[i])
				{
					missing = 1;
					break;
				}
			if(!missing)
				return 0;	/* They're already there! */
		}
	}

	if(txi->texsize > maxtexsize)
	{
		fprintf(stderr, "glSDL/wrapper: INTERNAL ERROR: Too large texture!\n");
		return -1;	/* This surface wasn't tiled properly... */
	}

	/*
	 * Kludge: Convert if not of preferred RGB or RGBA format.
	 *
	 *	Conversion should only be done when *really* needed.
	 *	That is, it should rarely have to be done with OpenGL
	 *	1.2+.
	 *
	 *	Besides, any surface that's been SDL_DisplayFormat()ed
	 *	should already be in the best known OpenGL format -
	 *	preferably one that makes DMA w/o conversion possible.
	 */
	if(FormatIsOk(surface))
		datasurf = surface;
	else
	{
		DBG(fprintf(stderr, "glSDL/wrapper: WARNING: On-the-fly conversion performed!\n"));
		if(surface->format->Amask)
			datasurf = glSDL_DisplayFormatAlpha(surface);
		else
			datasurf = glSDL_DisplayFormat(surface);
		if(!datasurf)
			return -2;
	}

	if(UploadTextures(datasurf, txi) < 0)
		return -3;

	if(datasurf != surface)
		glSDL_FreeSurface(datasurf);
	return 0;
}


static void UnloadTexture(glSDL_TexInfo *txi)
{
	int i;
	for(i = 0; i < txi->textures; ++i)
		glDeleteTextures(1, (unsigned int *)&txi->texture[i]);
	memset(&txi->invalid_area, 0, sizeof(txi->invalid_area));
}


void glSDL_UnloadSurface(SDL_Surface *surface)
{
	glSDL_TexInfo *txi;
	if(!IS_GLSDL_SURFACE(surface))
		return;

	txi = glSDL_GetTexInfo(surface);
	if(txi)
		UnloadTexture(txi);
}


SDL_Surface *glSDL_IMG_Load(const char *file)
{
	SDL_Surface *s;
	s = IMG_Load(file);
	if(s)
		GLSDL_FIX_SURFACE(s);
	return s;
}

