/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "DragWindow.h"
#include "Font.h"
#include "Globals.h"
#include "Display.h"
#include "SettingsStruct.h"
#include "Side.h"
#include "Sound.h"

using std::wstring;

DragWindow::DragWindow(int ix, int iy, int i_parent_id, int flags):
GenWindow_Base(ix, iy, i_parent_id, flags), b_drag(0) {}

void DragWindow::init_rects() {
	tile();
	init_border();
}

void DragWindow::draw_self() {
	if (visible) {
		GenWindow_Base::draw_self();

		if (!cant_close) {
			SDL_Rect close_rect;
			get_close_rect(close_rect);
			display.blt(gen_pictures[GENPIC_CLOSEBOX], close_rect);
		}
			
		int x = rect.x + small_border_size;
		int y = rect.y + small_border_size;
		
		for (int i = 0; i != the_text.size(); ++i) {
			if (the_text[i].bold)
				bold_font.render(x, y, the_text[i]);
			else
				normal_font.render(x, y, the_text[i]);
			y += normal_font.get_height();
		}
	}
}

bool DragWindow::mouse_d(Uint8 button, Uint16 x, Uint16 y) {
	if (!visible)
		return false;
		
	bool ret = GenWindow_Base::mouse_d(button, x, y);
	
	if (ret == true) {
		SDL_Rect close_rect;
		get_close_rect(close_rect);
		if (button == SDL_BUTTON_LEFT) {
			//close box?
			if (!cant_close && point_in_rect(x, y, close_rect)) {
				sound.play_sound(SE_MENU_CLICK);
				closed = true;
			} else if (!b_static) { //start drag
				b_drag = true;
				save_mouse_pos_x = x;
				save_mouse_pos_y = y;
			}
		}
	}

	return ret;
}
	
bool DragWindow::mouse_m(Uint8 state, Uint16 x, Uint16 y) {
	if (!visible)
		return false;
		
	bool ret = GenWindow_Base::mouse_m(state, x, y);

	//drag
	if (b_drag && state & SDL_BUTTON(1)) {
		int distx = x - save_mouse_pos_x;
		int disty = y - save_mouse_pos_y;

		move(distx, disty);

		b_drag = true;
		save_mouse_pos_x = x;
		save_mouse_pos_y = y;
		return true;
	}
	
	b_drag = false;
	return ret;
}

void DragWindow::get_close_rect(SDL_Rect& close_rect) {
	close_rect.x = rect.x + rect.w - gen_pictures[GENPIC_CLOSEBOX]->w;
	close_rect.y = rect.y;
	close_rect.w = gen_pictures[GENPIC_CLOSEBOX]->w;
	close_rect.h = gen_pictures[GENPIC_CLOSEBOX]->h;
}

WrappedString::WrappedString(int ix, int iy, int iw, int ih, const wstring& the_string, bool bold_title, int flags):
DragWindow(ix, iy, window_id_none, flags) {
	rect.w = iw;
	if (ih != -1)
		rect.h = ih;

	wstring cur_line;
	wstring cur_word;
	int newlines = 0;

	for (int i = 0; i != the_string.size(); ++i) {
		if (the_string[i] == L'\n') {
			cur_line += cur_word;
			the_text.push_back(cur_line);
			cur_line = L"";
			cur_word = L"";
			++newlines;
		} else if (normal_font.get_width(cur_word + L" ") + normal_font.get_width(cur_line) > rect.w) {
			cur_word += the_string[i];
			the_text.push_back(cur_line);
			cur_line = L"";
			++newlines;
		} else if (the_string[i] == ' ') {
			cur_line += cur_word + L' ';
			cur_word = L"";
		} else {
			cur_word += the_string[i];
		}
	}

	if (cur_word != L"") {
		cur_line += cur_word;
		the_text.push_back(cur_line);
		++newlines;
	}

	if (ih == -1)
		rect.h = (newlines + 1) * normal_font.get_height();

	if (bold_title && the_text.size())
		the_text[0].bold = true;

	if (flags & WFLAG_CENTER)
		center_window();
	init_rects();
}

InfoString::InfoString(const wstring& i_the_string):
DragWindow(0, 0, window_id_none, 0), the_string(i_the_string) {
	//width
	int longest = 0;
	int newlines = 0;

	int first_index = 0;
	int last_index = 0;
  wstring longest_data;

	for (; last_index != the_string.size(); ++last_index) {
		if (the_string[last_index] == '\n') {
			int current_length = normal_font.get_width(the_string.substr(first_index, last_index - first_index));
      if (current_length > longest) {
				longest = current_length;
        longest_data = the_string.substr(first_index, last_index - first_index);
      }
			
			first_index = last_index + 1;
			++newlines;
		}
	}

	//one last time
	int current_length = normal_font.get_width(longest_data);
  if (current_length > longest) {
		longest = current_length;
    longest_data = the_string.substr(first_index, last_index - first_index);
  }

	//overall dimensions
  rect.w = normal_font.get_width(longest_data + L" ");
	rect.h = (newlines + 1) * normal_font.get_height();

	//if just one line, make room for the close cross
	if (newlines == 0)
		rect.w += gen_pictures[GENPIC_CLOSEBOX]->w;

	center_window();

	init_rects();
	an_info_string = 1;
}

InfoString::~InfoString() {
	an_info_string = 0;
}

void InfoString::draw_self() {
	DragWindow::draw_self();

	int x = rect.x + small_border_size;
	int y = rect.y + small_border_size;


	normal_font.render(x, y, the_string);
}

///


Slider::Slider(int ix, int iy, int iSliderVar, int i_var_min, int i_var_max, int i_parent_id, int flags):
DragWindow(ix, iy, i_parent_id, flags), var_pointer(&slider_var), slider_var(iSliderVar) {
	init(i_var_min, i_var_max);
}

Slider::Slider(int ix, int iy, int* iVarPointer, int i_var_min, int i_var_max, int i_parent_id, int flags):
DragWindow(ix, iy, i_parent_id, flags), var_pointer(iVarPointer) {
	init(i_var_min, i_var_max);
}

void Slider::init(int i_var_min, int i_var_max) {
	shadow_var = static_cast<float>(*var_pointer);
	b_slider_drag = false;
	var_min = static_cast<float>(i_var_min);
	var_max = static_cast<float>(i_var_max);

	rect.w = slider_win_width;
	rect.h = slider_win_height;
	center_window();
	init_rects();

	rule_length = static_cast<float>(rect.w - (normal_font.get_height() * 2));

	middle = rect.y + (rect.h / 2) + (normal_font.get_height() / 2);
}

bool Slider::mouse_d(Uint8 button, Uint16 x, Uint16 y) {
	//note not DragWindow::mouse_d
	bool ret = GenWindow_Base::mouse_d(button, x, y);

	//slider start drag?
	if (button == SDL_BUTTON_LEFT && point_in_rect(x, y, slider_rect)) {
		b_slider_drag = true;
		save_mouse_pos_x = x;
	} else
		//check for the window start drag
		ret = DragWindow::mouse_d(button, x, y);
		
	return ret;
}

bool Slider::mouse_m(Uint8 state, Uint16 x, Uint16 y) {
	bool ret = false;
	
	//slider drag
	if (b_slider_drag && state & SDL_BUTTON(1)) {
		//not DragWindow mouse_m because we don't want to drag the window if slider dragging
		//and ignore result, we return true regardless
		GenWindow_Base::mouse_m(state, x, y);

		int d_pos = x - save_mouse_pos_x;
		slider_rect.x += d_pos;

		//add to the value
		shadow_var += d_pos * (var_max - var_min) / rule_length;

		if (shadow_var > var_max)
			shadow_var = var_max;
		else if (shadow_var < var_min)
			shadow_var = var_min;

		*var_pointer = static_cast<int>(shadow_var);

		b_slider_drag = true;
		save_mouse_pos_x = x;
		ret = true;
	} else {
		ret = DragWindow::mouse_m(state, x, y);
		b_slider_drag = false;
	}
		
	//work out middle /after/ any dragging of us or a parent
	middle = rect.y + (rect.h / 2) + (normal_font.get_height() / 2);
	
	return ret;
}

void Slider::draw_self() {
	DragWindow::draw_self();

	//draw a line down the centre
	SDL_Rect temp_rect;

	temp_rect.x = rect.x + normal_font.get_height();
	temp_rect.y = middle;
	temp_rect.w = static_cast<Uint16>(rule_length);
	temp_rect.h = small_border_size;

	display.blt_fill(temp_rect, standard_colors.light_blue);

	//we have to rework out position each time because something
	//other than us might have changed it
	//strange ordering to avoid casting, theVar/var_max * rule_length makes more sense
	int left_pos = rect.x + normal_font.get_height() + static_cast<int>(rule_length * (*var_pointer - var_min) / (var_max - var_min));
	slider_rect.x = left_pos;
	slider_rect.y = middle - (slider_height / 2);
	slider_rect.w = slider_width;
	slider_rect.h = slider_height;

	display.blt_fill(slider_rect, standard_colors.light_blue);
}


SliderWithUnits::SliderWithUnits(int ix, int iy, int iSliderVar, int i_var_min, int i_var_max, const wstring& iVarName, const wstring& iVarUnits, int i_parent_id, int flags):
Slider(ix, iy, iSliderVar, i_var_min, i_var_max, i_parent_id, flags), var_name(iVarName), var_units(iVarUnits) {}

SliderWithUnits::SliderWithUnits(int ix, int iy, int* iVarPointer, int i_var_min, int i_var_max, const wstring& iVarName, const wstring& iVarUnits, int i_parent_id, int flags):
Slider(ix, iy, iVarPointer, i_var_min, i_var_max, i_parent_id, flags), var_name(iVarName), var_units(iVarUnits) {}

void SliderWithUnits::draw_self() {
	Slider::draw_self();

	//in text
	int x = rect.x + small_border_size;
	int y = rect.y + small_border_size;
	wchar_t output[48];
	swprintf(output, 48, L"%ls: %d %ls", var_name.c_str(), *var_pointer, var_units.c_str());
	normal_font.render(x, y, output);
}

LargeBlankDW::LargeBlankDW(const wstring& iTitle):
DragWindow(0, 0, window_id_none, 0) {
	rect.w = 600;
	rect.h = normal_font.get_height() * 22;

	center_window();
	init_rects();
	
	the_text.push_back(WindowText(iTitle, true));
}

