/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "Display.h"
#include "GFX.h"
#include "Globals.h"
#include "Inlines.h"
#include "SettingsStruct.h"

#include <cstdlib>
#include <cmath>
#include <stdexcept>

using std::runtime_error;
using std::string;

void Display::init() {	
	if (SDL_Init(SDL_INIT_TIMER | SDL_INIT_VIDEO) == -1) {
		char output[120];
		printf(output, SDL_GetError());
		throw runtime_error(output);
	}
	
	reset_video();

	SDL_WM_SetCaption("Really Rather Good Battles In Space", 0);

	//keyboard
	SDL_EnableUNICODE(1);
	SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);
	keyboard = SDL_GetKeyState(0);

	//do cursor ourself via image blitting
	SDL_ShowCursor(SDL_DISABLE);
}

void Display::reset_video() {
	viewable_screen_rect.x = 0;
	viewable_screen_rect.y = 0;
	viewable_screen_rect.w = global_settings.screen_width;
	viewable_screen_rect.h = global_settings.screen_height;

	if (global_settings.full_screen)
		screen = glSDL_SetVideoMode(global_settings.screen_width, global_settings.screen_height, global_settings.bpp, SDL_FULLSCREEN);
	else
		screen = glSDL_SetVideoMode(global_settings.screen_width, global_settings.screen_height, global_settings.bpp, 0);

	if (screen == NULL) {
		glSDL_FullQuit();
		char output[120];
		printf(output, SDL_GetError());
		throw runtime_error(output);
	}
}

void Display::surface_color_convert(SDL_Surface* surface, const OpenGLColor& new_color) {
	int bytes_per_pixel = surface->format->BytesPerPixel;
	Uint32 pink = SDL_MapRGB(surface->format, 255, 0, 255);

	SDL_LockSurface(surface);

	Uint8* casted_buffer = reinterpret_cast<Uint8*>(surface->pixels);

	for (int i = 0; i != surface->pitch * surface->h / bytes_per_pixel; ++i) {
		if (pixel_color_matches(casted_buffer, pink, bytes_per_pixel))
			pixel_color(casted_buffer, new_color.sdl_color, bytes_per_pixel);
		casted_buffer += bytes_per_pixel;
	}

	SDL_UnlockSurface(surface);
}

bool Display::pixel_color_matches(Uint8* src, Uint32 color, int bytes_per_pixel) {
	switch (bytes_per_pixel) {
	case 1:
		if (*src == color)
			return true;
		break;
	case 2:
		if (*(Uint16 *) src == color)
			return true;
		break;
	case 3: {
		Uint32 tmp;
		Uint8* casted_tmp = reinterpret_cast<Uint8*>(&tmp);
	    if (SDL_BYTEORDER == SDL_BIG_ENDIAN) {			
			casted_tmp[0] = (color >> 16) & 0xff;
			casted_tmp[1] = (color >> 8) & 0xff;
			casted_tmp[2] = color & 0xff;
		} else {
			casted_tmp[0] = color & 0xff;
			casted_tmp[1] = (color >> 8) & 0xff;
			casted_tmp[2] = (color >> 16) & 0xff;
		}

		if (src[0] == casted_tmp[0]
		&& src[1] == casted_tmp[1]
		&& src[2] == casted_tmp[2])
			return true;
		}
	    break;
	case 4:
	    if (*(Uint32 *) src == color)
			return true;
	    break;
	}

	return false;
}

void Display::pixel_color(Uint8* dst, Uint32 color, int bytes_per_pixel) {
	switch (bytes_per_pixel) {
	case 1:
	    *dst = color;
	    break;
	case 2:
	    *(Uint16 *) dst = color;
	    break;
	case 3:
	case 4:
	    if (SDL_BYTEORDER == SDL_BIG_ENDIAN) {
			dst[0] = (color >> 16) & 0xff;
			dst[1] = (color >> 8) & 0xff;
			dst[2] = color & 0xff;
	    } else {
			dst[0] = color & 0xff;
			dst[1] = (color >> 8) & 0xff;
			dst[2] = (color >> 16) & 0xff;
	    }
	    break;	
	}
}

SDL_Surface* Display::file_to_surface(const string& filename, const OpenGLColor& transparency) {
	SDL_Surface* ret = glSDL_IMG_Load(filename.c_str());

	if (ret == NULL) {
		char output[100];
		printf(output, IMG_GetError());
		throw runtime_error(output);
	}

	if (transparency.sdl_color != 0xFFFFFFFF)
		glSDL_SetColorKey(ret, SDL_SRCCOLORKEY, transparency.sdl_color);

	//will be done automatically on the fly otherwise but may as well do it now
	SDL_Surface* tmp = ret;
	ret = glSDL_DisplayFormat(tmp);
	glSDL_FreeSurface(tmp);

	return ret;
}

SDL_Surface* Display::reduce_image_resolution(SDL_Surface* src, bool main_background) {
	if (src->w * src->h <= 1024 * 1024)
		return src;

	if (main_background && global_settings.texture_size)
		return src;

	double scalex, scaley;

	switch (global_settings.texture_size) {
		case 0:
			scalex = scaley = 0.25;
			break;
		case 1:
			scalex = scaley = 0.5;
			break;
		case 2:
			return src;
			break;
	}

	SDL_Surface* ret = gfx::zoomSurface(src, scalex, scaley, 1);
	glSDL_FreeSurface(src);

	//will be done automatically on the fly otherwise but may as well do it now
	SDL_Surface* tmp = ret;
	ret = glSDL_DisplayFormat(tmp);
	glSDL_FreeSurface(tmp);

	return ret;
}

void Display::take_screenshot() {
	Uint32 rmask, gmask, bmask;
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
	rmask = 0x00ff0000;
	gmask = 0x0000ff00;
	bmask = 0x000000ff;
#else
	rmask = 0x000000ff;
	gmask = 0x0000ff00;
	bmask = 0x00ff0000;
#endif
	SDL_Surface* tmp = glSDL_CreateRGBSurface(SDL_SWSURFACE, global_settings.screen_width, global_settings.screen_height, 24, rmask, gmask, bmask, 0); 
	glSDL_BlitSurface(screen, &viewable_screen_rect, tmp, 0);
	SDL_SaveBMP(tmp, "screen.bmp");
	glSDL_FreeSurface(tmp);
}

//////

void Display::blt_fill(SDL_Rect dest_rect, const OpenGLColor& color, float alpha) {
	if (skip_display_frame)
		return;

	glLoadIdentity();
	if (glSDL_FillRect(screen, &dest_rect, color, alpha) == -1)
		throw runtime_error("Failed to blt_fill");
}

void Display::blt(SDL_Surface* src, SDL_Rect dest_rect, int gl_effect) {
	if (skip_display_frame)
		return;

	glLoadIdentity();
	if (glSDL_BlitSurface(src, 0, screen, &dest_rect, gl_effect) == -1)
		throw runtime_error("Failed to blit");
}

void Display::blt_part(SDL_Surface* src, SDL_Rect& src_rect, SDL_Rect dest_rect, int gl_effect) {
	if (skip_display_frame)
		return;

	glLoadIdentity();
	if (glSDL_BlitSurface(src, &src_rect, screen, &dest_rect, gl_effect) == -1)
		throw runtime_error("blt_part fails");
}

void Display::flip() {
	//skip_display_frame checked for in Main.cpp, plus we want to force a flip in exception shutdown
	SDL_GL_SwapBuffers();
}

void Display::shutdown() {
	//this automatically frees the screen
	glSDL_FullQuit();
}

void Display::draw_line(float x1, float y1, float x2, float y2, float width, const OpenGLColor& color) {
	if (skip_display_frame)
		return;

	if (!clip_line(screen, &x1, &y1, &x2, &y2))
		return;

	glLoadIdentity();
	glLineWidth(width);

	glBegin(GL_LINES);
		glColor3f(color.r, color.g, color.b);
		glVertex2f(x1, y1);
		glVertex2f(x2, y2);
	glEnd();
}

void Display::draw_circle(Sint16 x, Sint16 y, Sint16 radius, const OpenGLColor& color) {
	if (skip_display_frame)
		return;

	static const float deg_2_rad = 3.14159f/180;

	glLoadIdentity();
	glTranslatef(x, y, 0);
	glLineWidth(1);

	glBegin(GL_LINE_LOOP);
		glColor3f(color.r, color.g, color.b);
		for (int i = 0; i < 360; i++) {
			float deg_in_rad = i * deg_2_rad;
			glVertex2f(cos(deg_in_rad) * radius, sin(deg_in_rad) * radius);
		}
	glEnd();
}

OpenGLColor Display::map_rgb(Uint32 r, Uint32 g, Uint32 b) {
	OpenGLColor ret;
	ret.r = static_cast<GLfloat>(r) / 255;
	ret.g = static_cast<GLfloat>(g) / 255;
	ret.b = static_cast<GLfloat>(b) / 255;
	ret.sdl_color = SDL_MapRGB(screen->format, r, g, b);

	return ret;
}

#define CLIP_LEFT_EDGE   0x1
#define CLIP_RIGHT_EDGE  0x2
#define CLIP_BOTTOM_EDGE 0x4
#define CLIP_TOP_EDGE    0x8
#define CLIP_INSIDE(a)   (!a)
#define CLIP_REJECT(a,b) (a&b)
#define CLIP_ACCEPT(a,b) (!(a|b))

bool Display::clip_line(SDL_Surface* dst, float* x1, float* y1, float* x2, float* y2)
{
    float left, right, top, bottom;
    int code1, code2;
    bool draw = false;
    float m;

    /*
     * Get clipping boundary 
     */
    left = static_cast<float>(dst->clip_rect.x);
    right = static_cast<float>(dst->clip_rect.x + dst->clip_rect.w - 1);
    top = static_cast<float>(dst->clip_rect.y);
    bottom = static_cast<float>(dst->clip_rect.y + dst->clip_rect.h - 1);

    while (1) {
	code1 = clip_encode(*x1, *y1, left, top, right, bottom);
	code2 = clip_encode(*x2, *y2, left, top, right, bottom);
	if (CLIP_ACCEPT(code1, code2)) {
	    draw = true;
	    break;
	} else if (CLIP_REJECT(code1, code2))
	    break;
	else {
	    if (CLIP_INSIDE(code1)) {
		float swaptmp = *x2;
		*x2 = *x1;
		*x1 = swaptmp;
		swaptmp = *y2;
		*y2 = *y1;
		*y1 = swaptmp;
		int swap_code = code2;
		code2 = code1;
		code1 = swap_code;
	    }
	    if (*x2 != *x1) {
		m = (*y2 - *y1) / (float) (*x2 - *x1);
	    } else {
		m = 1.0f;
	    }
	    if (code1 & CLIP_LEFT_EDGE) {
		*y1 += (Sint16) ((left - *x1) * m);
		*x1 = left;
	    } else if (code1 & CLIP_RIGHT_EDGE) {
		*y1 += (Sint16) ((right - *x1) * m);
		*x1 = right;
	    } else if (code1 & CLIP_BOTTOM_EDGE) {
		if (*x2 != *x1) {
		    *x1 += (Sint16) ((bottom - *y1) / m);
		}
		*y1 = bottom;
	    } else if (code1 & CLIP_TOP_EDGE) {
		if (*x2 != *x1) {
		    *x1 += (Sint16) ((top - *y1) / m);
		}
		*y1 = top;
	    }
	}
    }

    return draw;
}

int Display::clip_encode(float x, float y, float left, float top, float right, float bottom)
{
    int code = 0;

    if (x < left) {
	code |= CLIP_LEFT_EDGE;
    } else if (x > right) {
	code |= CLIP_RIGHT_EDGE;
    }
    if (y < top) {
	code |= CLIP_TOP_EDGE;
    } else if (y > bottom) {
	code |= CLIP_BOTTOM_EDGE;
    }
    return code;
}
