/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "Display.h"
#include "Globals.h"
#include "Group.h"
#include "Side.h"

#include <string>

using std::wstring;
using std::vector;

SmallShipGroup::SmallShipGroup(int i_my_side, int i_my_squad, int i_my_group, const wstring& i_data_filename, int i_parent_cap, CoordsInt i_starting_coords):
Group_Base(i_data_filename, i_my_side, i_my_squad, i_my_group, i_starting_coords), in_hangar(true), save_prop_x(0), save_prop_y(0), launch_time(-1) {
	my_parent_cap = i_parent_cap;
	speed_cruise = static_cast<float>(equip_lookup[L"Bomber engine"].max);
	fuel_current = fuel_max = equip_lookup[L"Fuel"].max;

	//FIXME could just use insert?
	for (int i = 0; i != ss_group_size; ++i)
		units.push_back(RTSUnit(my_side, my_group, data_filename));

	set_initial_vars();
}

bool SmallShipGroup::go_to_start_coords() {
	if (my_parent_cap == -1)
		return Group_Base::go_to_start_coords();
	else
		set_pos_to_parent();

	return true;
}

void SmallShipGroup::set_pos_to_parent() {
	CoordsFloat coords = sides[my_side].groups[my_parent_cap].get_hangar_bay_coords(sides[my_side].squadrons[my_squad].is_first(my_group));
	set_pos(coords.x - width / 2, coords.y - height / 2);
}

void SmallShipGroup::launch(const std::wstring& i_ai_filename, int mission_slot, int launch_slot) {
	launch_time = launch_slot;
	my_mission = mission_slot;
	change_ai_script(i_ai_filename);
}

void SmallShipGroup::move() {
	if (in_hangar)
		set_pos_to_parent();
	else
		Group_Base::move();
}

void SmallShipGroup::check_if_scanned() {
	if (in_hangar)
		return;
	else
		Group_Base::check_if_scanned();
}

void SmallShipGroup::small_ship_dont_sit_on_top_of_move_target() {
	//otherwise it is impossible to make it back to the launch site on the planet because we need to fly whilst
	//overlapping
	if (sides[the_commands.move_target.x].groups[the_commands.move_target.y].get_type() == UT_PLANET)
		return;

    if (check_for_collision(the_commands.move_target.x, the_commands.move_target.y)) {
        the_commands.aim_prop_x = save_prop_x;
        the_commands.aim_prop_y = save_prop_y;
        return;
    }
    save_prop_x = the_commands.aim_prop_x;
    save_prop_y = the_commands.aim_prop_y;
}

void SmallShipGroup::small_ship_dont_sit_on_top_of_move_point_target() {
    if (the_commands.move_target_dist <= move_point_leeway) {
        the_commands.aim_prop_x = save_prop_x;
        the_commands.aim_prop_y = save_prop_y;
        return;
    }
	//possible we got close enough that the get_move_props has told them to just stop
	if (!the_commands.aim_prop_x && !the_commands.aim_prop_y) {
		the_commands.aim_prop_x = 0.5;
		the_commands.aim_prop_y = 0.5;
	}
		
    save_prop_x = the_commands.aim_prop_x;
    save_prop_y = the_commands.aim_prop_y;
}

void SmallShipGroup::select_targets() {
	if (the_commands.move_command == MC_MOVE_GROUP
	&& !the_commands.b_inverse
	&& sides[the_commands.move_target.x].groups[the_commands.move_target.y].is_small()
	&& !sides[the_commands.move_target.x].groups[the_commands.move_target.y].get_in_hangar()
	&& the_commands.move_target_dist <= weapon_lookup[units[0].get_small_type()].range
	&& sides[the_commands.move_target.x].my_flag != sides[my_side].my_flag
	//if we are moving towards a saved group which has been killed then we
	//shouldn't automatically attack it
	&& sides[the_commands.move_target.x].groups[the_commands.move_target.y].get_alive()) {
	//loop through units
		for (int i = 0; i != units.size(); ++i)
			units[i].select_small_targets(the_commands, true);
	} else {
		for (int i = 0; i != units.size(); ++i)
			units[i].select_small_targets(the_commands, false);
	}

	if (the_commands.ordered_to_fire
	&& !sides[the_commands.move_target.x].groups[the_commands.move_target.y].get_in_hangar()
	&& the_commands.move_target_dist <= weapon_lookup[units[0].get_big_type()].range
	&& sides[the_commands.move_target.x].my_flag != sides[my_side].my_flag) {
	//let people fire at dead capital ships if they don't check their save groups health if they want) {		 
		if (the_commands.move_command == MC_MOVE_GROUP && the_commands.b_inverse == false) {			
			if (units[0].get_big_type() == WT_TORPEDO && !sides[the_commands.move_target.x].groups[the_commands.move_target.y].is_big_ship()) {
				the_commands.ordered_to_fire = false;
				report_on_script_error("Attempt to fire torpedos at non-capital ship");
				//FIXME blank to -1
			}

			//loop through units
			for (int i = 0; i != units.size(); ++i)
				units[i].select_big_targets(the_commands, true);
		} else {
			the_commands.ordered_to_fire = false;
			report_on_script_error("Attempt to fire big weapons when not moving towards a group");
			//FIXME blank to -1
		}
	} else {
		for (int i = 0; i != units.size(); ++i)
			units[i].select_big_targets(the_commands, false);
	}
}

void SmallShipGroup::run_fire_commands() {
	for (int i = 0; i != units.size(); ++i)
		units[i].fire(the_commands);
}

void SmallShipGroup::upkeep() {
	if (in_hangar && launch_time != -1 && world.frame_counter >= launch_time) {
		in_hangar = false;
		fill_fog(true);
	}

	//if our parent cap is dead, try to find a new home
	if (!in_hangar
	&& the_commands.ordered_to_dock
	&& !sides[my_side].groups[my_parent_cap].get_alive()) {
		bool found_new_parent = false;

		for (int i = 0; i != sides[my_side].groups.size(); ++i) {
			if (sides[my_side].groups[i].get_alive() && sides[my_side].groups[i].get_capacity() > sides[my_side].groups[i].get_how_full()) {
				my_parent_cap = i;
				ai_interpreter.save_groups[0].y = my_parent_cap;
				found_new_parent = true;
				break;
			}
		}
		//if not found a new parent and run out of fuel, explode
		if (!found_new_parent && !fuel_current)
			destroy();
	}

	if (!in_hangar
	&& the_commands.ordered_to_dock
	&& find_distance_to(my_side, my_parent_cap) < docking_leeway
	&& sides[my_side].groups[my_parent_cap].get_alive()) {
		in_hangar = true;
		launch_time = -1;
		change_ai_script(L"");

		for (int i = 0; i != n_ai_vars; ++i)
			waypoints[i].x = -1;

		fill_fog(false);

		for (int i = 0; i != units.size(); ++i)
			units[i].just_docked();

		my_mission = -1;
	}

	//only refuel if both wings are home (or one is home and other is dead)
	if (sides[my_side].squadrons[my_squad].both_in_hangar()) {
		if (fuel_current < fuel_max) {
			fuel_current += equip_lookup[L"Fuel"].recharge;
			
			if (fuel_current > fuel_max)
				fuel_current = fuel_max;
		}
	} else if (fuel_current > 0)
		fuel_current -= static_cast<int>(the_commands.speed);
		
	Group_Base::upkeep();
}

void SmallShipGroup::force_fill_fog(bool make_visible) {
	if (!in_hangar)
		Group_Base::force_fill_fog(make_visible);
}

void SmallShipGroup::draw_self_back() {
	if (on_screen
	&& !in_hangar
	&& world.frame_counter <= launch_time + ss_draw_at_back_time) {
		for (int i = 0; i != units.size(); ++i)
			units[i].draw_self();
	}
}

void SmallShipGroup::draw_self_front() {
	if (on_screen
	&& !in_hangar
	&& world.frame_counter > launch_time + ss_draw_at_back_time) {
		for (int i = 0; i != units.size(); ++i)
			units[i].draw_self();
	}

	if (alive && in_hangar) {
		for (int i = 0; i != n_ai_vars; ++i) {
			if (waypoints[i].x != -1) {
				SDL_Rect rect;
				rect.w = screen_rect.w;
				rect.h = screen_rect.h;
				rect.x = static_cast<int>(waypoints[i].x) - world.viewx;
				rect.y = static_cast<int>(waypoints[i].y) - world.viewy;
				draw_rect_border(rect, sides[my_side].color, 1);
			}
		}
	}
}

void SmallShipGroup::draw_bound() {
	if (b_draw_weapon_range && !in_hangar) {
		CoordsFloat center = get_center();
		if (units[0].get_small_type() != WT_NONE)
			display.draw_circle(static_cast<int>(center.x) - world.viewx, static_cast<int>(center.y) - world.viewy, static_cast<int>(weapon_lookup[units[0].get_small_type()].range), standard_colors.white);
		if (units[0].get_big_type() != WT_NONE)
			display.draw_circle(static_cast<int>(center.x) - world.viewx, static_cast<int>(center.y) - world.viewy, static_cast<int>(weapon_lookup[units[0].get_big_type()].range), standard_colors.big_range_blue);
	}

	Group_Base::draw_bound();
}

bool SmallShipGroup::check_for_cursor(Uint16 x, Uint16 y) const {
	if (in_hangar)
		return false;
	else
		return Group_Base::check_for_cursor(x, y);
}

////

BigUnitGroup::BigUnitGroup(int i_my_side, int i_my_squad, int i_my_group, const wstring& i_data_filename, CoordsInt i_starting_coords, bool face_left):
Group_Base(i_data_filename, i_my_side, i_my_squad, i_my_group, i_starting_coords), next_launch_time(0) {
	units.push_back(RTSUnit(my_side, my_group, data_filename, face_left));
	set_initial_vars();
}

void BigUnitGroup::mouse_d(Uint16 x, Uint16 y, CoordsFloat center) {
	if (!is_big_ship())
		return;

	if (sides[my_side].squadrons[my_squad].get_selected() && my_side == 0) {
		for (int i  = 1; i != sides.size(); ++i) {
			if (sides[i].my_flag == sides[my_side].my_flag)
				continue;
			for (int j  = 0; j != sides[i].groups.size(); ++j) {
				if (sides[my_side].scanned_groups[i][j] && sides[i].groups[j].is_big_ship() && sides[i].groups[j].check_for_cursor(x, y)) {
					the_commands.ordered_to_fire = true;
					the_commands.fire_target.x = i;
					the_commands.fire_target.y = j;

					the_commands.move_command = MC_MOVE_GROUP;
					the_commands.move_target = the_commands.fire_target;
					sound.play_sound(SE_TARGET_ACQUIRED);
					return;
				}
			}
		}
		//else
		CoordsFloat my_center = get_center();
		float dx = my_center.x - center.x;
		float dy = my_center.y - center.y;
		
		order_move(static_cast<float>(x + world.viewx) + dx, static_cast<float>(y + world.viewy) + dy);

		sound.play_sound(SE_COORDINATES_SET);
	}
}

void BigUnitGroup::mouse_d_set_pos(Uint16 x, Uint16 y, CoordsFloat center) {
	if (sides[my_side].squadrons[my_squad].get_selected()) {
		CoordsFloat my_center = get_center();
		float dx = my_center.x - center.x;
		float dy = my_center.y - center.y;
		relocate(static_cast<float>(x + world.viewx) + dx - my_center.x, static_cast<float>(y + world.viewy) + dy - my_center.y);
	}
}

void BigUnitGroup::run_group_ai() {
	//if we are moving towards or firing at a group which is now either out of scanning range or dead, stop
	//don't do this for small ships because we won't get a new move target until running ai script again, which could
	//be several frames away, and we don't want to just stop
	if (the_commands.move_command == MC_MOVE_GROUP &&
	(!sides[my_side].scanned_groups[the_commands.move_target.x][the_commands.move_target.y] || !sides[the_commands.move_target.x].groups[the_commands.move_target.y].get_alive())) {
		the_commands.move_command = MC_NO_MOVE;
		the_commands.ordered_to_fire = false;
	}

	Group_Base::run_group_ai();
}

void BigUnitGroup::select_targets() {
	for (int i = 0; i != units.size(); ++i) {
		units[i].select_small_targets(the_commands);
		units[i].select_big_targets(the_commands);
	}
}

void BigUnitGroup::run_fire_commands() {
	for (int i = 0; i != units.size(); ++i)
		units[i].fire();
}

void BigUnitGroup::draw_self_back_back() {
	if (units[0].get_type() != UT_PLANET)
		return;

	if (on_screen) {
		for (int i = 0; i != units.size(); ++i)
			units[i].draw_self();
	}
}

void BigUnitGroup::draw_self_middle() {
	if (units[0].get_type() == UT_PLANET)
		return;

	if (on_screen) {
		for (int i = 0; i != units.size(); ++i)
			units[i].draw_self();
	}
}

int BigUnitGroup::get_launch_slot() {
	if (next_launch_time + launch_time_per_group <= world.frame_counter)
		next_launch_time = world.frame_counter;
	else
		next_launch_time = next_launch_time + launch_time_per_group;

	return next_launch_time;
}

int BigUnitGroup::get_how_full() const {
	int total = 0;

	for (int i = 0; i != sides[my_side].squadrons.size(); ++i) {
		if (sides[my_side].squadrons[i].get_alive() && sides[my_side].squadrons[i].get_parent_cap() == my_group)
			++total;
	}

	return total;
}

	
