/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "BCCompiler.h"
#include "Globals.h"
#include "Inlines.h"

#include <stdexcept>
#include <string>

using std::runtime_error;
using std::string;

ustring BCCompiler::convert_to_bc(const string& the_line) {
	string::const_iterator l_iter = the_line.begin();
	string::const_iterator l_end =  the_line.end();
	
	string::const_iterator l_iterc = l_iter;
	if (!find_not_comment(l_iterc, l_end))
		return ustring(1, TT_COMMENT);
		
	ustring bc;

	while (*l_iter == '\t') {
		bc.push_back(TT_TAB);
		++l_iter;
	}

	if (std::isspace(*l_iter))
		throw runtime_error("Indentation should not include spaces, only tabs");

	if (the_line.find(':') != string::npos || the_line.find('@') != string::npos) {
		while (l_iter != l_end && not_space(*l_iter) && *l_iter != '#') {
			bc += *l_iter;
			++l_iter;
		}
		while (l_iter != l_end && *l_iter != '#') {
			if (not_space(*l_iter))
				throw runtime_error("Function label or function jump has an invalid character");
			++l_iter;
		}
		return bc;
	}
	
	while (l_iter != l_end) {
		if (std::isspace(*l_iter)) {
			++l_iter;
			continue;
		}

		const unsigned char tmp_token = what_is_this_token(l_iter, l_end);
		switch (tmp_token) {
		case TT_Integer:
			bc.push_back(TT_Integer);
			for (int i = 0; l_iter != l_end && isdigit(static_cast<int>(*l_iter)); ++i) {
				if (i == max_script_digits)
					throw runtime_error("Scripts only support numbers of up to 6 digits");
				bc.push_back(*l_iter);
				++l_iter;
			}
			break;
			
		case TT_SCRIPT_VAR:
		case TT_SaveGroup:
		case TT_ScriptTimer:
		case TT_GLOBAL_SCRIPT_VAR:
		case TT_GSaveGroup:
		case TT_WAYPOINT: {
			bc.push_back(tmp_token);
			int i = 0;
			for (; l_iter != l_end && isdigit(static_cast<int>(*l_iter)); ++i) {
				if (i == n_ai_vars)
					throw runtime_error("Scripts only allow variables 0 - 9 of each type");
				bc.push_back(*l_iter);
				++l_iter;
			}
			if (i == 0)
				throw runtime_error("Script variables must have a number from 0 to 9");
		}
			break;
			
		//comment on end of line
		case TT_COMMENT:
			return bc;
			break;

		default:
			bc.push_back(tmp_token);
			break;
		}
	}
	
	return bc;
}

//You have to give this an iter on the first letter of the command
//it leaves the iter one past the end
const unsigned char BCCompiler::what_is_this_token(string::const_iterator& l_iter, const string::const_iterator& l_end) {
	//if an integer, return as such and leave iterator where it is
	if (isdigit(*l_iter))
		return TT_Integer;

	//some tokens could be the first few letters of another token, so we start off with the whole string and gradually eat away at the end until it is recognisable as a token
	string token_str(l_iter, l_end);
	while (token_str.size()) {
		if (token_lookup.find(token_str) != token_lookup.end()) {
			l_iter += token_str.size();
			return token_lookup[token_str];
		}
		token_str = token_str.substr(0, token_str.size() - 1);
	}

	throw runtime_error("Unrecognised token");
}

