// Author: Sergey Linev <S.Linev@gsi.de>
// Date: 2019-10-14
// Warning: This is part of the ROOT 7 prototype! It will change without notice. It might trigger earthquakes. Feedback is welcome!

/*************************************************************************
 * Copyright (C) 1995-2020, Rene Brun and Fons Rademakers.               *
 * All rights reserved.                                                  *
 *                                                                       *
 * For the licensing terms see $ROOTSYS/LICENSE.                         *
 * For the list of contributors see $ROOTSYS/README/CREDITS.             *
 *************************************************************************/

#include <ROOT/RBrowserData.hxx>

#include <ROOT/Browsable/RGroup.hxx>
#include <ROOT/Browsable/RWrapper.hxx>
#include <ROOT/Browsable/RProvider.hxx>
#include <ROOT/Browsable/RLevelIter.hxx>
#include <ROOT/Browsable/TObjectHolder.hxx>
#include <ROOT/Browsable/RSysFile.hxx>

#include <ROOT/RLogger.hxx>

#include "TFolder.h"
#include "TROOT.h"
#include "TBufferJSON.h"
#include "TEnv.h"

#include <algorithm>
#include <regex>

using namespace ROOT::Experimental;
using namespace std::string_literals;

ROOT::Experimental::RLogChannel &ROOT::Experimental::BrowserLog() {
   static RLogChannel sLog("ROOT.Browser");
   return sLog;
}

namespace ROOT {
namespace Experimental {

class RBrowserDataCleanup : public TObject {

   RBrowserData &fData;

public:
   RBrowserDataCleanup(RBrowserData &_data) : fData(_data) {}

   void RecursiveRemove(TObject *obj) override
   {
      fData.RemoveFromCache(obj);
   }
};
}
}


/** \class ROOT::Experimental::RBrowserData
\ingroup rbrowser
\brief Way to browse (hopefully) everything in %ROOT
*/


/////////////////////////////////////////////////////////////////////
/// Default constructor

RBrowserData::RBrowserData()
{
   fCleanupHandle = std::make_unique<RBrowserDataCleanup>(*this);
   R__LOCKGUARD(gROOTMutex);
   gROOT->GetListOfCleanups()->Add(fCleanupHandle.get());
}

/////////////////////////////////////////////////////////////////////
/// Destructor

RBrowserData::~RBrowserData()
{
   // should be here because of fCleanupHandle destructor
   R__LOCKGUARD(gROOTMutex);
   gROOT->GetListOfCleanups()->Remove(fCleanupHandle.get());
}

/////////////////////////////////////////////////////////////////////
/// set top element for browsing

void RBrowserData::SetTopElement(std::shared_ptr<Browsable::RElement> elem)
{
   fTopElement = elem;

   SetWorkingPath({});
}

/////////////////////////////////////////////////////////////////////
/// set working directory relative to top element

void RBrowserData::SetWorkingPath(const Browsable::RElementPath_t &path)
{
   fWorkingPath = path;

   ResetLastRequestData(true);
}

/////////////////////////////////////////////////////////////////////
/// Create default elements shown in the RBrowser

void RBrowserData::CreateDefaultElements()
{
   auto comp = std::make_shared<Browsable::RGroup>("top","Root browser");

   auto seldir = Browsable::RSysFile::ProvideTopEntries(comp);

   std::unique_ptr<Browsable::RHolder> rootfold = std::make_unique<Browsable::TObjectHolder>(gROOT->GetRootFolder(), kFALSE);
   auto elem_root = Browsable::RProvider::Browse(rootfold);
   if (elem_root)
      comp->Add(std::make_shared<Browsable::RWrapper>("root", elem_root));

   std::unique_ptr<Browsable::RHolder> rootfiles = std::make_unique<Browsable::TObjectHolder>(gROOT->GetListOfFiles(), kFALSE);
   auto elem_files = Browsable::RProvider::Browse(rootfiles);
   if (elem_files) {
      auto files = std::make_shared<Browsable::RWrapper>("ROOT Files", elem_files);
      files->SetExpandByDefault(true);
      comp->Add(files);
      // if there are any open files, make them visible by default
      if (elem_files->GetNumChilds() > 0)
         seldir = {};
   }

   SetTopElement(comp);

   SetWorkingPath(seldir);
}

/////////////////////////////////////////////////////////////////////
/// Reset all data correspondent to last request

void RBrowserData::ResetLastRequestData(bool with_element)
{
   fLastAllChilds = false;
   fLastSortedItems.clear();
   fLastSortMethod.clear();
   fLastItems.clear();
   if (with_element) {
      fLastPath.clear();
      fLastElement.reset();
   }
}

/////////////////////////////////////////////////////////////////////////
/// Decompose path to elements
/// Returns array of names for each element in the path, first element either "/" or "."
/// If returned array empty - it is error

Browsable::RElementPath_t RBrowserData::DecomposePath(const std::string &strpath, bool relative_to_work_element)
{
   Browsable::RElementPath_t arr;
   if (relative_to_work_element) arr = fWorkingPath;

   if (strpath.empty())
      return arr;

   auto arr2 = Browsable::RElement::ParsePath(strpath);
   arr.insert(arr.end(), arr2.begin(), arr2.end());
   return arr;
}

/////////////////////////////////////////////////////////////////////////
/// Process browser request

bool RBrowserData::ProcessBrowserRequest(const RBrowserRequest &request, RBrowserReply &reply)
{
   auto path = fWorkingPath;
   path.insert(path.end(), request.path.begin(), request.path.end());

   if ((path != fLastPath) || !fLastElement) {

      auto elem = GetSubElement(path);
      if (!elem) return false;

      ResetLastRequestData(true);

      fLastPath = path;
      fLastElement = std::move(elem);

      fLastElement->cd(); // set element active
   } else if (request.reload) {
      // only reload items from element, not need to reset element itself
      ResetLastRequestData(false);
   }

   // when request childs, always try to make elements
   if (fLastItems.empty()) {

      auto iter = fLastElement->GetChildsIter();

      if (!iter) return false;
      int id = 0;
      fLastAllChilds = true;

      while (iter->Next() && fLastAllChilds) {
         fLastItems.emplace_back(iter->CreateItem());
         if (id++ > 10000)
            fLastAllChilds = false;
      }

      fLastSortedItems.clear();
      fLastSortMethod.clear();
   }

   // create sorted array
   if ((fLastSortedItems.size() != fLastItems.size()) ||
       (fLastSortMethod != request.sort) ||
       (fLastSortReverse != request.reverse)) {
      fLastSortedItems.resize(fLastItems.size(), nullptr);
      int id = 0;
      if (request.sort.empty()) {
         // no sorting, just move all folders up
         for (auto &item : fLastItems)
            if (item->IsFolder())
               fLastSortedItems[id++] = item.get();
         for (auto &item : fLastItems)
            if (!item->IsFolder())
               fLastSortedItems[id++] = item.get();
      } else {
         // copy items
         for (auto &item : fLastItems)
            fLastSortedItems[id++] = item.get();

         if (request.sort != "unsorted")
            std::sort(fLastSortedItems.begin(), fLastSortedItems.end(),
                      [request](const Browsable::RItem *a, const Browsable::RItem *b) { return a ? a->Compare(b, request.sort) : !b; });
      }

      if (request.reverse)
         std::reverse(fLastSortedItems.begin(), fLastSortedItems.end());

      fLastSortMethod = request.sort;
      fLastSortReverse = request.reverse;
   }

   const std::regex expr(request.regex);

   int id = 0;
   for (auto &item : fLastSortedItems) {

      // check if element is hidden
      if (!request.hidden && item->IsHidden())
         continue;

      if (!request.regex.empty() && !item->IsFolder() && !std::regex_match(item->GetName(), expr))
         continue;

      if ((id >= request.first) && ((request.number == 0) || (id < request.first + request.number)))
         reply.nodes.emplace_back(item);

      id++;
   }

   reply.first = request.first;
   reply.nchilds = id; // total number of childs

   return true;
}

/////////////////////////////////////////////////////////////////////////
/// Process browser request, returns string with JSON of RBrowserReply data

std::string RBrowserData::ProcessRequest(const RBrowserRequest &request)
{
   if (request.lastcycle < 0)
      gEnv->SetValue("WebGui.LastCycle", "no");
   else if (request.lastcycle > 0)
      gEnv->SetValue("WebGui.LastCycle", "yes");

   RBrowserReply reply;

   reply.path = request.path;
   reply.first = 0;
   reply.nchilds = 0;

   ProcessBrowserRequest(request, reply);

   return TBufferJSON::ToJSON(&reply, TBufferJSON::kSkipTypeInfo + TBufferJSON::kNoSpaces).Data();
}

/////////////////////////////////////////////////////////////////////////
/// Returns element with path, specified as string

std::shared_ptr<Browsable::RElement> RBrowserData::GetElement(const std::string &str)
{
   auto path = DecomposePath(str, true);

   return GetSubElement(path);
}

/////////////////////////////////////////////////////////////////////////
/// Returns element with path, specified as Browsable::RElementPath_t

std::shared_ptr<Browsable::RElement> RBrowserData::GetElementFromTop(const Browsable::RElementPath_t &path)
{
   return GetSubElement(path);
}

/////////////////////////////////////////////////////////////////////////
/// Returns sub-element starting from top, using cached data

std::shared_ptr<Browsable::RElement> RBrowserData::GetSubElement(const Browsable::RElementPath_t &path)
{
   if (path.empty())
      return fTopElement;

   // validate cache - removes no longer actual elements
   RemoveFromCache(nullptr);

   // first check direct match in cache
   for (auto &entry : fCache)
      if (entry.first == path)
         return entry.second;

   // find best possible entry in cache
   int pos = 0;
   auto elem = fTopElement;

   for (auto &entry : fCache) {
      if (entry.first.size() >= path.size())
         continue;

      auto comp = Browsable::RElement::ComparePaths(path, entry.first);

      if ((comp > pos) && (comp == (int) entry.first.size())) {
         pos = comp;
         elem = entry.second;
      }
   }

   while (pos < (int) path.size()) {
      std::string subname = path[pos];
      int indx = Browsable::RElement::ExtractItemIndex(subname);

      auto iter = elem->GetChildsIter();
      if (!iter)
         return nullptr;

      if (!iter->Find(subname, indx)) {
         if (indx < 0)
            return nullptr;
         iter = elem->GetChildsIter();
         if (!iter || !iter->Find(subname))
            return nullptr;
      }

      elem = iter->GetElement();

      if (!elem)
         return nullptr;

      auto subpath = path;
      subpath.resize(pos+1);
      fCache.emplace_back(subpath, elem);
      pos++; // switch to next element
   }

   return elem;
}

/////////////////////////////////////////////////////////////////////////
/// Clear internal objects cache

void RBrowserData::ClearCache()
{
   fCache.clear();
}

/////////////////////////////////////////////////////////////////////////
/// Remove object from cache
/// If nullptr specified - removes no-longer-valid elements
/// Returns true if any element was removed

bool RBrowserData::RemoveFromCache(void *obj)
{
   unsigned pos = 0;

   bool isany = false;

   while (pos < fCache.size()) {
      if (obj ? !fCache[pos].second->IsObject(obj) : fCache[pos].second->CheckValid()) {
         pos++;
         continue;
      }

      isany = true;
      auto path = fCache[pos].first;
      fCache.erase(fCache.begin() + pos);
      if (RemoveFromCache(path))
         pos = 0; // start scan from the beginning
   }

   return isany;
}

/////////////////////////////////////////////////////////////////////////
/// Remove path (and all sub-paths) from cache
/// Returns true if any element was removed

bool RBrowserData::RemoveFromCache(const Browsable::RElementPath_t &path)
{
   if (path.size() == 0)
      return false;

   bool isany = false;
   unsigned pos = 0;
   while (pos < fCache.size()) {
      if (Browsable::RElement::ComparePaths(path, fCache[pos].first) == (int) path.size()) {
         fCache.erase(fCache.begin() + pos);
         isany = true;
      } else {
         pos++;
      }
   }
   return isany;
}
