// @(#)root/mlp:$Id$
// Author: Christophe.Delaere@cern.ch   20/07/03

/*************************************************************************
 * Copyright (C) 1995-2003, Rene Brun and Fons Rademakers.               *
 * All rights reserved.                                                  *
 *                                                                       *
 * For the licensing terms see $ROOTSYS/LICENSE.                         *
 * For the list of contributors see $ROOTSYS/README/CREDITS.             *
 *************************************************************************/

/** \class TNeuron

This class describes an elementary neuron, which is the basic
element for a Neural Network.
A network is built connecting neurons by synapses.
There are different types of neurons: linear (a+bx),
sigmoid (1/(1+exp(-x)), tanh or gaussian.
An external function can also be used, together with its derivative.
In a Multi Layer Perceptron, the input layer is made of
inactive neurons (returning the normalized input) and output neurons
are linear. Hidden neurons may be anything, the default being sigmoids.

This implementation contains several methods to compute the value,
the derivative, the DeDw, ...
Values are stored in local buffers. The SetNewEvent() method is
there to inform buffered values are outdated.

*/

#include "TSynapse.h"
#include "TNeuron.h"
#include "TTree.h"
#include "TTreeFormula.h"
#include "TFormula.h"
#include "TBranch.h"
#include "TROOT.h"
#include "TH1D.h"
#include "TRegexp.h"
#include "TMath.h"

ClassImp(TNeuron);

////////////////////////////////////////////////////////////////////////////////
/// Usual constructor

TNeuron::TNeuron(TNeuron::ENeuronType type /*= kSigmoid*/,
                 const char* name /*= ""*/, const char* title /*= ""*/,
                 const char* extF /*= ""*/, const char* extD  /*= ""*/ )
   :TNamed(name, title)
{
   fpre.SetOwner(false);
   fpost.SetOwner(false);
   flayer.SetOwner(false);
   fWeight = 0.;
   fNorm[0] = 1.;
   fNorm[1] = 0.;
   fType = type;
   fNewInput = true;
   fNewValue = true;
   fNewDeriv = true;
   fNewDeDw = true;
   fFormula = 0;
   fDeDw  = 0;
   fDEDw  = 0;
   fValue = 0;
   fExtF  = 0;
   fExtD  = 0;
   fIndex = 0;
   fDerivative = 0;
   fInput = 0;
   if(fType==kExternal) {
      fExtF = (TFormula*)gROOT->FindObject(extF);
      fExtD = (TFormula*)gROOT->FindObject(extD);
   }
}

////////////////////////////////////////////////////////////////////////////////
/// The Sigmoid.
/// Fast computation of the values of the sigmoid function.
/// Uses values of the function up  to the seventh order
/// tabulated at 700 points.
/// Values were computed in long double precision (16 bytes,
/// precision to about 37 digits) on a hp computer.
/// Some values were checked with Mathematica.
/// Result should be correct to ~ 15 digits (about double
/// precision)
///
/// From the mlpfit package (J.Schwindling   20-Jul-1999)

Double_t TNeuron::Sigmoid(Double_t x) const
{
   static Double_t sigval[7000] = {
   -3.500000e+01, 6.30511676014698530e-16, 6.30511676014698130e-16, 3.15255838007348670e-16, 1.05085279335782620e-16, 2.62713198339455210e-17, 5.25426396678905190e-18, 8.75710661131491060e-19, 1.25101523018779380e-19, 1.56376903773461590e-20,
   -3.490000e+01, 6.96823167838580650e-16, 6.96823167838580160e-16, 3.48411583919289640e-16, 1.16137194639762880e-16, 2.90342986599405540e-17, 5.80685973198804710e-18, 9.67809955331319670e-19, 1.38258565047325200e-19, 1.72823206309141090e-20,
   -3.480000e+01, 7.70108700136546340e-16, 7.70108700136545750e-16, 3.85054350068272280e-16, 1.28351450022757030e-16, 3.20878625056890590e-17, 6.41757250113773230e-18, 1.06959541685626250e-18, 1.52799345265172810e-19, 1.90999181581447200e-20,
   -3.470000e+01, 8.51101739147945470e-16, 8.51101739147944780e-16, 4.25550869573971650e-16, 1.41850289857990070e-16, 3.54625724644972780e-17, 7.09251449289935940e-18, 1.18208574881652770e-18, 1.68869392688066160e-19, 2.11086740860059710e-20,
   -3.460000e+01, 9.40612890429916560e-16, 9.40612890429915770e-16, 4.70306445214957000e-16, 1.56768815071651720e-16, 3.91922037679126390e-17, 7.83844075358240950e-18, 1.30640679226369560e-18, 1.86629541751945290e-19, 2.33286927189903510e-20,
   -3.450000e+01, 1.03953801167022100e-15, 1.03953801167021980e-15, 5.19769005835108810e-16, 1.73256335278368870e-16, 4.33140838195918600e-17, 8.66281676391822790e-18, 1.44380279398632340e-18, 2.06257541998032470e-19, 2.57821927497506280e-20,
   -3.440000e+01, 1.14886717873211280e-15, 1.14886717873211140e-15, 5.74433589366054410e-16, 1.91477863122017240e-16, 4.78694657805038730e-17, 9.57389315610059860e-18, 1.59564885935004120e-18, 2.27949837049989100e-19, 2.84937296312444480e-20,
   -3.430000e+01, 1.26969459466634980e-15, 1.26969459466634820e-15, 6.34847297333172530e-16, 2.11615765777723080e-16, 5.29039414444302330e-17, 1.05807882888858320e-17, 1.76346471481423380e-18, 2.51923530687727210e-19, 3.14904413359607800e-20,
   -3.420000e+01, 1.40322954086308920e-15, 1.40322954086308720e-15, 7.01614770431541630e-16, 2.33871590143845880e-16, 5.84678975359608160e-17, 1.16935795071919010e-17, 1.94892991786522940e-18, 2.78418559695007730e-19, 3.48023199618697190e-20,
   -3.410000e+01, 1.55080847994653150e-15, 1.55080847994652920e-15, 7.75404239973262210e-16, 2.58468079991085760e-16, 6.46170199977706390e-17, 1.29234039995538080e-17, 2.15390066659219480e-18, 3.07700095227425790e-19, 3.84625119034205930e-20,
   -3.400000e+01, 1.71390843154201000e-15, 1.71390843154200710e-15, 8.56954215771000580e-16, 2.85651405256998220e-16, 7.14128513142485820e-17, 1.42825702628493230e-17, 2.38042837714142370e-18, 3.40061196734451790e-19, 4.25076495917971450e-20,
   -3.390000e+01, 1.89416175478487890e-15, 1.89416175478487530e-15, 9.47080877392433920e-16, 3.15693625797475610e-16, 7.89234064493677060e-17, 1.57846812898730630e-17, 2.63078021497868450e-18, 3.75825744996909330e-19, 4.69782181246022790e-20,
   -3.380000e+01, 2.09337248551938710e-15, 2.09337248551938240e-15, 1.04668624275968680e-15, 3.48895414253226000e-16, 8.72238535633050470e-17, 1.74447707126604240e-17, 2.90746178544320960e-18, 4.15351683634688560e-19, 5.19189604543221620e-20,
   -3.370000e+01, 2.31353439169574760e-15, 2.31353439169574210e-15, 1.15676719584786570e-15, 3.85589065282618330e-16, 9.63972663206527960e-17, 1.92794532641298470e-17, 3.21324221068806990e-18, 4.59034601526799130e-19, 5.73793251908329070e-20,
   -3.360000e+01, 2.55685092766997700e-15, 2.55685092766997070e-15, 1.27842546383497880e-15, 4.26141821278321860e-16, 1.06535455319578300e-16, 2.13070910639147860e-17, 3.55118184398550770e-18, 5.07311691997846610e-19, 6.34139614997100740e-20,
   -3.350000e+01, 2.82575728711560320e-15, 2.82575728711559530e-15, 1.41287864355778950e-15, 4.70959547852591190e-16, 1.17739886963145140e-16, 2.35479773926279630e-17, 3.92466289877097270e-18, 5.60666128395751820e-19, 7.00832660494436280e-20,
   -3.340000e+01, 3.12294477526050570e-15, 3.12294477526049620e-15, 1.56147238763023820e-15, 5.20490795876739570e-16, 1.30122698969181640e-16, 2.60245397938350280e-17, 4.33742329897207160e-18, 6.19631899853029210e-19, 7.74539874815976920e-20,
   -3.330000e+01, 3.45138774437420440e-15, 3.45138774437419260e-15, 1.72569387218708450e-15, 5.75231290729020130e-16, 1.43807822682251060e-16, 2.87615645364486300e-17, 4.79359408940757550e-18, 6.84799155629502340e-19, 8.55998944536499720e-20,
   -3.320000e+01, 3.81437336208505550e-15, 3.81437336208504050e-15, 1.90718668104250570e-15, 6.35728893680825560e-16, 1.58932223420201530e-16, 3.17864446840383680e-17, 5.29774078067241470e-18, 7.56820111524445930e-19, 9.46025139405095430e-20,
   -3.310000e+01, 4.21553451045883940e-15, 4.21553451045882130e-15, 2.10776725522939290e-15, 7.02589085076452470e-16, 1.75647271269107180e-16, 3.51294542538190670e-17, 5.85490904230238850e-18, 8.36415577471544070e-19, 1.04551947183886600e-19,
   -3.300000e+01, 4.65888614510337560e-15, 4.65888614510335430e-15, 2.32944307255165550e-15, 7.76481024183870600e-16, 1.94120256045960400e-16, 3.88240512091891920e-17, 6.47067520153056710e-18, 9.24382171646948210e-19, 1.15547771455799630e-19,
   -3.290000e+01, 5.14886547819381740e-15, 5.14886547819379060e-15, 2.57443273909686890e-15, 8.58144246365605200e-16, 2.14536061591392470e-16, 4.29072123182749640e-17, 7.15120205304464960e-18, 1.02160029329175600e-18, 1.27700036661385360e-19,
   -3.280000e+01, 5.69037638758347510e-15, 5.69037638758344200e-15, 2.84518819379168870e-15, 9.48396064597207920e-16, 2.37099016149291180e-16, 4.74198032298539250e-17, 7.90330053830754850e-18, 1.12904293404352430e-18, 1.41130366755337750e-19,
   -3.270000e+01, 6.28883849646157640e-15, 6.28883849646153690e-15, 3.14441924823072900e-15, 1.04813974941021660e-15, 2.62034937352540980e-16, 5.24069874705029230e-17, 8.73449791174873020e-18, 1.24778541596360200e-18, 1.55973176995324680e-19,
   -3.260000e+01, 6.95024141476392120e-15, 6.95024141476387310e-15, 3.47512070738188800e-15, 1.15837356912726370e-15, 2.89593392281799860e-16, 5.79186784563535330e-17, 9.65311307605677590e-18, 1.37901615372178300e-18, 1.72377019215069530e-19,
   -3.250000e+01, 7.68120468520203540e-15, 7.68120468520197700e-15, 3.84060234260092930e-15, 1.28020078086693710e-15, 3.20050195216714590e-16, 6.40100390433350520e-17, 1.06683398405532210e-17, 1.52404854864971080e-18, 1.90506068581026560e-19,
   -3.240000e+01, 8.48904403387170610e-15, 8.48904403387163350e-15, 4.24452201693574420e-15, 1.41484067231186670e-15, 3.53710168077942670e-16, 7.07420336155789230e-17, 1.17903389359266190e-17, 1.68433413370288750e-18, 2.10541766712632160e-19,
   -3.230000e+01, 9.38184458849859670e-15, 9.38184458849850840e-15, 4.69092229424916660e-15, 1.56364076474966340e-15, 3.90910191187386470e-16, 7.81820382374655680e-17, 1.30303397062403500e-17, 1.86147710089036070e-18, 2.32684637611015680e-19,
   -3.220000e+01, 1.03685417971139710e-14, 1.03685417971138640e-14, 5.18427089855682470e-15, 1.72809029951886980e-15, 4.32022574879681600e-16, 8.64045149759219900e-17, 1.44007524959822210e-17, 2.05725035656752340e-18, 2.57156294570599160e-19,
   -3.210000e+01, 1.14590108570221770e-14, 1.14590108570220460e-14, 5.72950542851089130e-15, 1.90983514283687630e-15, 4.77458785709175300e-16, 9.54917571418175490e-17, 1.59152928569637550e-17, 2.27361326527886920e-18, 2.84201658159441780e-19,
   -3.200000e+01, 1.26641655490940160e-14, 1.26641655490938570e-14, 6.33208277454676750e-15, 2.11069425818214890e-15, 5.27673564545483790e-16, 1.05534712909075370e-16, 1.75891188181721000e-17, 2.51273125973683490e-18, 3.14091407466595250e-19,
   -3.190000e+01, 1.39960674665542220e-14, 1.39960674665540250e-14, 6.99803373327681690e-15, 2.33267791109214150e-15, 5.83169477772970090e-16, 1.16633895554567910e-16, 1.94389825924192790e-17, 2.77699751320026640e-18, 3.47124689149411470e-19,
   -3.180000e+01, 1.54680467314603800e-14, 1.54680467314601400e-14, 7.73402336572983000e-15, 2.57800778857645030e-15, 6.44501947144032790e-16, 1.28900389428774660e-16, 2.14833982381184800e-17, 3.06905689115674430e-18, 3.83632111393833500e-19,
   -3.170000e+01, 1.70948354070450830e-14, 1.70948354070447900e-14, 8.54741770352210300e-15, 2.84913923450717270e-15, 7.12284808626695750e-16, 1.42456961725300190e-16, 2.37428269542037130e-17, 3.39183242202539070e-18, 4.23979052752246120e-19,
   -3.160000e+01, 1.88927149411560280e-14, 1.88927149411556710e-14, 9.44635747057747730e-15, 3.14878582352558780e-15, 7.87196455881277930e-16, 1.57439291176208010e-16, 2.62398818626854730e-17, 3.74855455180767740e-18, 4.68569318974826550e-19,
   -3.150000e+01, 2.08796791164589010e-14, 2.08796791164584650e-14, 1.04398395582287960e-14, 3.47994651940930800e-15, 8.69986629852181640e-16, 1.73997325970378200e-16, 2.89995543283769920e-17, 4.14279347547689150e-18, 5.17849184433227460e-19,
   -3.140000e+01, 2.30756141382617880e-14, 2.30756141382612580e-14, 1.15378070691300960e-14, 3.84593568970967660e-15, 9.61483922427241660e-16, 1.92296784485377340e-16, 3.20494640808725600e-17, 4.57849486868931840e-18, 5.72311858584474310e-19,
   -3.130000e+01, 2.55024976623420620e-14, 2.55024976623414120e-14, 1.27512488311700560e-14, 4.25041627705625120e-15, 1.06260406926384610e-15, 2.12520813852682500e-16, 3.54201356420848460e-17, 5.06001937743243370e-18, 6.32502422176989520e-19,
   -3.120000e+01, 2.81846187547126010e-14, 2.81846187547118060e-14, 1.40923093773551080e-14, 4.69743645911783960e-15, 1.17435911477919500e-15, 2.34871822955733110e-16, 3.91453038259202140e-17, 5.59218626083565740e-18, 6.99023282601935360e-19,
   -3.110000e+01, 3.11488209847576830e-14, 3.11488209847567110e-14, 1.55744104923773830e-14, 5.19147016412514760e-15, 1.29786754103096350e-15, 2.59573508206063350e-16, 4.32622513676341070e-17, 6.18032162393540810e-18, 7.72540202988845880e-19,
   -3.100000e+01, 3.44247710846985810e-14, 3.44247710846973950e-14, 1.72123855423475110e-14, 5.73746184744838010e-15, 1.43436546186170000e-15, 2.86873092372182000e-16, 4.78121820619776690e-17, 6.83031172312461810e-18, 8.53788965386815150e-19,
   -3.090000e+01, 3.80452558642202530e-14, 3.80452558642188080e-14, 1.90226279321079550e-14, 6.34087597736835340e-15, 1.58521899434160580e-15, 3.17043798868128200e-16, 5.28406331446237000e-17, 7.54866187778500530e-18, 9.43582734718530590e-19,
   -3.080000e+01, 4.20465103518829610e-14, 4.20465103518811940e-14, 2.10232551759388270e-14, 7.00775172531176340e-15, 1.75193793132735160e-15, 3.50387586265234630e-16, 5.83979310441271980e-17, 8.34256157771000690e-18, 1.04282019720813850e-18,
   -3.070000e+01, 4.64685804474675720e-14, 4.64685804474654130e-14, 2.32342902237305450e-14, 7.74476340790874180e-15, 1.93619085197646560e-15, 3.87238170395005280e-16, 6.45396950657382470e-17, 9.21995643793518570e-18, 1.15249455473504320e-18,
   -3.060000e+01, 5.13557237147994640e-14, 5.13557237147968260e-14, 2.56778618573957750e-14, 8.55928728579683310e-15, 2.13982182144832910e-15, 4.27964364289314180e-16, 7.13273940481018150e-17, 1.01896277211239110e-17, 1.27370346513211610e-18,
   -3.050000e+01, 5.67568523263239960e-14, 5.67568523263207780e-14, 2.83784261631571670e-14, 9.45947538771690840e-15, 2.36486884692815310e-15, 4.72973769385201140e-16, 7.88289615640570200e-17, 1.12612802233958120e-17, 1.40766002791425000e-18,
   -3.040000e+01, 6.27260225925671670e-14, 6.27260225925632420e-14, 3.13630112962776830e-14, 1.04543370987566040e-14, 2.61358427468783960e-15, 5.22716854937043290e-16, 8.71194758226656970e-17, 1.24456394031879920e-17, 1.55570492538600840e-18,
   -3.030000e+01, 6.93229759758606750e-14, 6.93229759758558660e-14, 3.46614879879231310e-14, 1.15538293293045040e-14, 2.88845733232452430e-15, 5.77691466464264150e-16, 9.62819110771637780e-17, 1.37545587252480860e-17, 1.71931984064075470e-18,
   -3.020000e+01, 7.66137370029775430e-14, 7.66137370029716740e-14, 3.83068685014799680e-14, 1.27689561671560770e-14, 3.19223904178706240e-15, 6.38447808356629930e-16, 1.06407968059177450e-16, 1.52011382940936700e-17, 1.90014228674307500e-18,
   -3.010000e+01, 8.46712740607860540e-14, 8.46712740607788850e-14, 4.23356370303822730e-14, 1.41118790101226440e-14, 3.52796975252827130e-15, 7.05593950504698390e-16, 1.17598991750464440e-16, 1.67998559642610260e-17, 2.09998199550986870e-18,
   -3.000000e+01, 9.35762296883929890e-14, 9.35762296883842290e-14, 4.67881148441833550e-14, 1.55960382813886140e-14, 3.89900957034423470e-15, 7.79801914067679430e-16, 1.29966985677557410e-16, 1.85667122395398620e-17, 2.32083902991468430e-18,
   -2.990000e+01, 1.03417727674778080e-13, 1.03417727674767390e-13, 5.17088638373729960e-14, 1.72362879457838680e-14, 4.30907198644240230e-15, 8.61814397287054390e-16, 1.43635732880700410e-16, 2.05193904113928150e-17, 2.56492380139014880e-18,
   -2.980000e+01, 1.14294265039630320e-13, 1.14294265039617240e-13, 5.71471325197955590e-14, 1.90490441732564790e-14, 4.76226104330976530e-15, 9.52452208660211380e-16, 1.58742034776121300e-16, 2.26774335392800200e-17, 2.83467919236853180e-18,
   -2.970000e+01, 1.26314697824628310e-13, 1.26314697824612330e-13, 6.31573489122902130e-14, 2.10524496374194340e-14, 5.26311240934954000e-15, 1.05262248186778070e-15, 1.75437080310587660e-16, 2.50624400441670580e-17, 3.13280500547022990e-18,
   -2.960000e+01, 1.39599330561290320e-13, 1.39599330561270840e-13, 6.97996652806159300e-14, 2.32665550935256480e-14, 5.81663877337491580e-15, 1.16332775467238480e-15, 1.93887959111198010e-16, 2.76982798727808210e-17, 3.46228498403573600e-18,
   -2.950000e+01, 1.54281120319164970e-13, 1.54281120319141160e-13, 7.71405601595467900e-14, 2.57135200531663900e-14, 6.42838001328366400e-15, 1.28567600265355900e-15, 2.14279333774535300e-16, 3.06113333960599290e-17, 3.82641667443192680e-18,
   -2.940000e+01, 1.70507007384868490e-13, 1.70507007384839410e-13, 8.52535036923906350e-14, 2.84178345641108290e-14, 7.10445864101801600e-15, 1.42089172819972690e-15, 2.36815288031995750e-16, 3.38307554327730670e-17, 4.22884442900433970e-18,
   -2.930000e+01, 1.88439385889862550e-13, 1.88439385889827030e-13, 9.42196929448780100e-14, 3.14065643149356600e-14, 7.85164107872207900e-15, 1.57032821573968130e-15, 2.61721369288368720e-16, 3.73887670407446150e-17, 4.67359587998034830e-18,
   -2.920000e+01, 2.08257729105506810e-13, 2.08257729105463440e-13, 1.04128864552688340e-13, 3.47096215175338660e-14, 8.67740537936900990e-15, 1.73548107586801910e-15, 2.89246845976075590e-16, 4.13209779960314800e-17, 5.16512224936624800e-18,
   -2.910000e+01, 2.30160385671876950e-13, 2.30160385671823990e-13, 1.15080192835859010e-13, 3.83600642785843490e-14, 9.59001606962842950e-15, 1.91800321391862300e-15, 3.19667202317416110e-16, 4.56667431875296190e-17, 5.70834289827303100e-18,
   -2.900000e+01, 2.54366564737627600e-13, 2.54366564737562880e-13, 1.27183282368716740e-13, 4.23944274561957760e-14, 1.05986068640273770e-14, 2.11972137279684860e-15, 3.53286895463265750e-16, 5.04695564939306360e-17, 6.30869456153592520e-18,
   -2.890000e+01, 2.81118529878825430e-13, 2.81118529878746370e-13, 1.40559264939294170e-13, 4.68530883130453670e-14, 1.17132720782349990e-14, 2.34265441563646270e-15, 3.90442402602564840e-16, 5.57774860850771650e-17, 6.97218576038376470e-18,
   -2.880000e+01, 3.10684023754248020e-13, 3.10684023754151490e-13, 1.55342011876979210e-13, 5.17806706255953880e-14, 1.29451676563666710e-14, 2.58903353126046440e-15, 4.31505588539120810e-16, 6.16436555043629750e-17, 7.70545693773894400e-18,
   -2.870000e+01, 3.43358947763907870e-13, 3.43358947763789930e-13, 1.71679473881777080e-13, 5.72264912938470960e-14, 1.43066228234224760e-14, 2.86132456466877550e-15, 4.76887427439556210e-16, 6.81267753470109370e-17, 8.51584691800209580e-18,
   -2.860000e+01, 3.79470323529711840e-13, 3.79470323529567850e-13, 1.89735161764639940e-13, 6.32450539214506360e-14, 1.58112634803146610e-14, 3.16225269604373280e-15, 5.27042116000888860e-16, 7.52917308554412920e-17, 9.41146635647302580e-18,
   -2.850000e+01, 4.19379565837778610e-13, 4.19379565837602710e-13, 2.09689782918625480e-13, 6.98965943060912380e-14, 1.74741485764641810e-14, 3.49482971526938620e-15, 5.82471619203747550e-16, 8.32102313125876900e-17, 1.04012789135151140e-17,
   -2.840000e+01, 4.63486099799084420e-13, 4.63486099798869650e-13, 2.31743049899220000e-13, 7.72476832995967810e-14, 1.93119208248275890e-14, 3.86238416493687560e-15, 6.43730694146598420e-16, 9.19615277325004720e-17, 1.14951909658805930e-17,
   -2.830000e+01, 5.12231358430229290e-13, 5.12231358429966960e-13, 2.56115679214721100e-13, 8.53718930713987660e-14, 2.13429732677622320e-14, 4.26859465351746230e-15, 7.11432442241249050e-16, 1.01633206031132330e-16, 1.27041507530585860e-17,
   -2.820000e+01, 5.66103200663442990e-13, 5.66103200663122500e-13, 2.83051600331240810e-13, 9.43505334435332780e-14, 2.35876333607764950e-14, 4.71752667211256930e-15, 7.86254445337851770e-16, 1.12322063615623600e-16, 1.40402579509355750e-17,
   -2.810000e+01, 6.25640794002741240e-13, 6.25640794002349870e-13, 3.12820397000783510e-13, 1.04273465666666880e-13, 2.60683664165362450e-14, 5.21367328325505900e-15, 8.68945547191779780e-16, 1.24135078165283750e-16, 1.55168847694178470e-17,
   -2.800000e+01, 6.91440010693542290e-13, 6.91440010693064180e-13, 3.45720005346053980e-13, 1.15240001781699280e-13, 2.88100004452654500e-14, 5.76200008898934550e-15, 9.60333348143642610e-16, 1.37190478300163680e-16, 1.71488097860027170e-17,
   -2.790000e+01, 7.64159391412363150e-13, 7.64159391411779220e-13, 3.82079695705305630e-13, 1.27359898568045920e-13, 3.18399746418168330e-14, 6.36799492828550810e-15, 1.06133248802163190e-15, 1.51618926852818000e-16, 1.89523658547484770e-17,
   -2.780000e+01, 8.44526736163259950e-13, 8.44526736162546770e-13, 4.22263368080560150e-13, 1.40754456026377880e-13, 3.51886140063567320e-14, 7.03772280117624980e-15, 1.17295380016434280e-15, 1.67564828585849290e-16, 2.09456035709669510e-17,
   -2.770000e+01, 9.33346388344898750e-13, 9.33346388344027540e-13, 4.66673194171142670e-13, 1.55557731389800110e-13, 3.88894328471596510e-14, 7.77788656931577800e-15, 1.29631442818057940e-15, 1.85187775443306440e-16, 2.31484719276477960e-17,
   -2.760000e+01, 1.03150728488961810e-12, 1.03150728488855400e-12, 5.15753642443213040e-13, 1.71917880813695000e-13, 4.29794702030690770e-14, 8.59589404047194890e-15, 1.43264900669803560e-15, 2.04664143800493880e-16, 2.55830179716839340e-17,
   -2.750000e+01, 1.13999185304305580e-12, 1.13999185304175630e-12, 5.69995926519578510e-13, 1.89998642172326450e-13, 4.74996605426484150e-14, 9.49993210835640690e-15, 1.58332201800164210e-15, 2.26188859698017680e-16, 2.82736074581265530e-17,
   -2.740000e+01, 1.25988584282620350e-12, 1.25988584282461610e-12, 6.29942921410720760e-13, 2.09980973802515370e-13, 5.24952434500997370e-14, 1.04990486898083070e-14, 1.74984144823083720e-15, 2.49977349727106100e-16, 3.12471687108491790e-17,
   -2.730000e+01, 1.39238919358655900e-12, 1.39238919358462020e-12, 6.96194596790371320e-13, 2.32064865595497920e-13, 5.80162163982282320e-14, 1.16032432793871470e-14, 1.93387387981169150e-15, 2.76267697091336890e-16, 3.45334621302623570e-17,
   -2.720000e+01, 1.53882804339444490e-12, 1.53882804339207690e-12, 7.69414021693670390e-13, 2.56471340562978150e-13, 6.41178351399552090e-14, 1.28235670276753080e-14, 2.13726117117397420e-15, 3.05323024423355150e-16, 3.81653780454019560e-17,
   -2.710000e+01, 1.70066800147851220e-12, 1.70066800147561990e-12, 8.50334000734917620e-13, 2.83444666909711020e-13, 7.08611667264636660e-14, 1.41722333449070960e-14, 2.36203889068930410e-15, 3.37434127204601900e-16, 4.21792658913934210e-17,
   -2.700000e+01, 1.87952881653555080e-12, 1.87952881653201790e-12, 9.39764408262476250e-13, 3.13254802751803660e-13, 7.83137006867733800e-14, 1.56627401368836570e-14, 2.61045668932360430e-15, 3.72922384144227540e-16, 4.66152980068137490e-17,
   -2.690000e+01, 2.07720058771982250e-12, 2.07720058771550760e-12, 1.03860029385343900e-12, 3.46200097948269810e-13, 8.65500244856292010e-14, 1.73100048965505380e-14, 2.88500081589998970e-15, 4.12142973645207850e-16, 5.15178716919533240e-17,
   -2.680000e+01, 2.29566168055709270e-12, 2.29566168055182270e-12, 1.14783084027064130e-12, 3.82610280086700350e-13, 9.56525700199184030e-14, 1.91305140032810070e-14, 3.18841900031260940e-15, 4.55488428549165600e-16, 5.69360535519153530e-17,
   -2.670000e+01, 2.53709852709174800e-12, 2.53709852708531110e-12, 1.26854926353621870e-12, 4.22849754507781650e-13, 1.05712438624799790e-13, 2.11424877241017100e-14, 3.52374795373086880e-15, 5.03392564736957500e-16, 6.29240705716851820e-17,
   -2.660000e+01, 2.80392750843360670e-12, 2.80392750842574480e-12, 1.40196375420501040e-12, 4.67321251396428790e-13, 1.16830312846486510e-13, 2.33660625682490380e-14, 3.89434376102541680e-15, 5.56334822903795940e-16, 6.95418528380157340e-17,
   -2.650000e+01, 3.09881913871222250e-12, 3.09881913870261990e-12, 1.54940956934170730e-12, 5.16469856440833900e-13, 1.29117464107007590e-13, 2.58234928201211620e-14, 4.30391546959340840e-15, 6.14845066962833770e-16, 7.68556333398695290e-17,
   -2.640000e+01, 3.42472479247986350e-12, 3.42472479246813470e-12, 1.71236239622233860e-12, 5.70787465399626990e-13, 1.42696866345997180e-13, 2.85393732676356060e-14, 4.75656221075132350e-15, 6.79508887101252680e-16, 8.49386108504224980e-17,
   -2.630000e+01, 3.78490624306004490e-12, 3.78490624304571950e-12, 1.89245312150853430e-12, 6.30817707159961000e-13, 1.57704426785215080e-13, 3.15408853551329510e-14, 5.25681422521880200e-15, 7.50973460563631870e-16, 9.38716825249761520e-17,
   -2.620000e+01, 4.18296830747124610e-12, 4.18296830745374850e-12, 2.09148415370937700e-12, 6.97161384558127530e-13, 1.74290346133699490e-13, 3.48580692244069370e-14, 5.80967820329016810e-15, 8.29954028819265610e-16, 1.03744253546861460e-16,
   -2.610000e+01, 4.62289492464729140e-12, 4.62289492462591960e-12, 2.31144746229158880e-12, 7.70482487416282180e-13, 1.92620621846946820e-13, 3.85241243665398760e-14, 6.42068739347348420e-15, 9.17241055939117900e-16, 1.14655131924544790e-16,
   -2.600000e+01, 5.10908902803722130e-12, 5.10908902801111900e-12, 2.55454451397945650e-12, 8.51514837975750260e-13, 2.12878709485236640e-13, 4.25757418935669590e-14, 7.09595698110103600e-15, 1.01370813982582680e-15, 1.26713517395362360e-16,
   -2.590000e+01, 5.64641661164308870e-12, 5.64641661161120670e-12, 2.82320830577372090e-12, 9.41069435236652320e-13, 2.35267358798535750e-13, 4.70534717554562100e-14, 7.84224529115905680e-15, 1.12032075547501430e-15, 1.40040094333164010e-16,
   -2.580000e+01, 6.24025543052348290e-12, 6.24025543048454240e-12, 3.12012771520333030e-12, 1.04004257170848290e-12, 2.60010642914140440e-13, 5.20021285776359840e-14, 8.66702142787529690e-15, 1.23814591777341320e-15, 1.54768239598055100e-16,
   -2.570000e+01, 6.89654882317364320e-12, 6.89654882312608120e-12, 3.44827441151547810e-12, 1.14942480380678440e-12, 2.87356200935841950e-13, 5.74712401808267460e-14, 9.57854002802390780e-15, 1.36836286054230570e-15, 1.71045357416796510e-16,
   -2.560000e+01, 7.62186519445479700e-12, 7.62186519439670370e-12, 3.81093259714025940e-12, 1.27031086567469110e-12, 3.17577716399308490e-13, 6.35155432721159960e-14, 1.05859238761040960e-14, 1.51227483870575550e-15, 1.89034354653797760e-16,
   -2.550000e+01, 8.42346375439769230e-12, 8.42346375432673890e-12, 4.21173187709241450e-12, 1.40391062565016820e-12, 3.50977656388890460e-13, 7.01955312683174650e-14, 1.16992552082327010e-14, 1.67132217170365880e-15, 2.08915271237704210e-16,
   -2.540000e+01, 9.30936717081640510e-12, 9.30936717072974020e-12, 4.65468358527820600e-12, 1.55156119503495900e-12, 3.87890298729851650e-13, 7.75780597344150870e-14, 1.29296766185507690e-14, 1.84709665869246760e-15, 2.30887082061433640e-16,
   -2.530000e+01, 1.02884418628644030e-11, 1.02884418627585500e-11, 5.14422093127342320e-12, 1.71474031035390620e-12, 4.28685077553192570e-13, 8.57370154965249040e-14, 1.42895025780496170e-14, 2.04135750980579250e-15, 2.55169688389685830e-16,
   -2.520000e+01, 1.13704867391374510e-11, 1.13704867390081620e-11, 5.68524336937479380e-12, 1.89508112303873930e-12, 4.73770280716588840e-13, 9.47540561260793620e-14, 1.57923426819337620e-14, 2.25604895292021420e-15, 2.82006118704588760e-16,
   -2.510000e+01, 1.25663312684444590e-11, 1.25663312682865470e-11, 6.28316563398536100e-12, 2.09438854455651170e-12, 5.23597136086490380e-13, 1.04719427196243050e-13, 1.74532378590221670e-14, 2.49331969214078360e-15, 3.11664961016287840e-16,
   -2.500000e+01, 1.38879438647711440e-11, 1.38879438645782690e-11, 6.94397193209625990e-12, 2.31465731057016970e-12, 5.78664327578250730e-13, 1.15732865489933500e-13, 1.92888109064166940e-14, 2.75554441275317880e-15, 3.44443050981845790e-16,
   -2.490000e+01, 1.53485516711897890e-11, 1.53485516709542100e-11, 7.67427583524152660e-12, 2.55809194492345680e-12, 6.39522986152338190e-13, 1.27904597199057240e-13, 2.13174328560394060e-14, 3.04534754787130510e-15, 3.80668442736046330e-16,
   -2.480000e+01, 1.69627729415529750e-11, 1.69627729412652390e-11, 8.48138647034488360e-12, 2.82712882325647090e-12, 7.06782205718205870e-13, 1.41356441105276420e-13, 2.35594068380911510e-14, 3.36562954464494980e-15, 4.20703692167172130e-16,
   -2.470000e+01, 1.87467633448913720e-11, 1.87467633445399300e-11, 9.37338167191852480e-12, 3.12446055707188060e-12, 7.81115139150823120e-13, 1.56223027783305790e-13, 2.60371712815980300e-14, 3.71959589290840220e-15, 4.64949485497864110e-16,
   -2.460000e+01, 2.07183776567796040e-11, 2.07183776563503520e-11, 1.03591888277459240e-11, 3.45306294229580730e-12, 8.63265735430868060e-13, 1.72653147028940140e-13, 2.87755244857455290e-14, 4.11078920679855240e-15, 5.13848649487116960e-16,
   -2.450000e+01, 2.28973484559312390e-11, 2.28973484554069520e-11, 1.14486742271791870e-11, 3.81622474204353670e-12, 9.54056185336121260e-13, 1.90811236997319100e-13, 3.18018728095848100e-14, 4.54312468042591100e-15, 5.67890583388830720e-16,
   -2.440000e+01, 2.53054836144786350e-11, 2.53054836138382680e-11, 1.26527418062787670e-11, 4.21758060166601060e-12, 1.05439515020304680e-12, 2.10879029955227020e-13, 3.51465049640770590e-14, 5.02092927245078680e-15, 6.27616157023435650e-16,
   -2.430000e+01, 2.79668845584872230e-11, 2.79668845577050790e-11, 1.39834422780703920e-11, 4.66114742550203310e-12, 1.16528685611479270e-12, 2.33057371118672340e-13, 3.88428951516833150e-14, 5.54898501173702400e-15, 6.93623123984122820e-16,
   -2.420000e+01, 3.09081874831280010e-11, 3.09081874821726850e-11, 1.54540937401310260e-11, 5.15136457940679800e-12, 1.28784114453326080e-12, 2.57568228779276710e-13, 4.29280380874209590e-14, 6.13257685750056810e-15, 7.66572104154821690e-16,
   -2.410000e+01, 3.41588299366717030e-11, 3.41588299355048740e-11, 1.70794149665856120e-11, 5.69313832141732000e-12, 1.42328457996538810e-12, 2.84656915837500910e-13, 4.74428192543912290e-14, 6.77754559295334210e-15, 8.47193195414959260e-16,
   -2.400000e+01, 3.77513454413658160e-11, 3.77513454399406490e-11, 1.88756727185451600e-11, 6.29189090523161070e-12, 1.57297272583284800e-12, 3.14594544976547720e-13, 5.24324240994173340e-14, 7.49034628181944040e-15, 9.36293280703099570e-16,
   -2.390000e+01, 4.17216890998595810e-11, 4.17216890981188770e-11, 2.08608445473187400e-11, 6.95361484794578040e-12, 1.73840371140621180e-12, 3.47680742049149160e-13, 5.79467902641604420e-14, 8.27811287277594410e-15, 1.03476410357096340e-15,
   -2.380000e+01, 4.61095974459562970e-11, 4.61095974438302030e-11, 2.30547987197890050e-11, 7.68493290517893790e-12, 1.92123322558603620e-12, 3.84246644833727960e-13, 6.40411073777948850e-14, 9.14872959840123680e-15, 1.14359119305064670e-15,
   -2.370000e+01, 5.09589861411988260e-11, 5.09589861386020090e-11, 2.54794930667041840e-11, 8.49316435383684880e-12, 2.12329108759360610e-12, 4.24658217172478840e-13, 7.07763694133323230e-14, 1.01109098832148610e-14, 1.26386372715799030e-15,
   -2.360000e+01, 5.63183894975709550e-11, 5.63183894943991960e-11, 2.81591947440278350e-11, 9.38639824589476990e-12, 2.34659956041643890e-12, 4.69319911660386350e-13, 7.82199851357639110e-14, 1.11742835505470840e-14, 1.39678543374930300e-15,
   -2.350000e+01, 6.22414462252038290e-11, 6.22414462213298320e-11, 3.11207231067909190e-11, 1.03735743663476410e-11, 2.59339359029557780e-12, 5.18678717542582610e-13, 8.64464527515860990e-14, 1.23494932010329630e-14, 1.54368663783071540e-15,
   -2.340000e+01, 6.87874362666143950e-11, 6.87874362618826760e-11, 3.43937181262096260e-11, 1.14645727055820680e-11, 2.86614317481827990e-12, 5.73228634332761170e-13, 9.55381055118285770e-14, 1.36483007273188580e-14, 1.70603757589355130e-15,
   -2.330000e+01, 7.60218740902944630e-11, 7.60218740845151330e-11, 3.80109370364782430e-11, 1.26703123416398630e-11, 3.16757808348352420e-12, 6.33515615926128080e-13, 1.05585935730829120e-13, 1.50837050310159020e-14, 1.88546311052992310e-15,
   -2.320000e+01, 8.40171643815300930e-11, 8.40171643744712180e-11, 4.20085821801767210e-11, 1.40028607220196500e-11, 3.50071517815195120e-12, 7.00143034689205770e-13, 1.16690505467806140e-13, 1.66700721200499690e-14, 2.08375899259709100e-15,
   -2.310000e+01, 9.28533266928275480e-11, 9.28533266842058060e-11, 4.64266633334811610e-11, 1.54755544387458930e-11, 3.86888860681256030e-12, 7.73777720212946620e-13, 1.28962952985636000e-13, 1.84232788884655840e-14, 2.30290983368759360e-15,
   -2.300000e+01, 1.02618796306488270e-10, 1.02618796295957660e-10, 5.13093981374482110e-11, 1.71031327054623260e-11, 4.27578317285537510e-12, 8.55156633166992820e-13, 1.42526105059804710e-13, 2.03608720176785480e-14, 2.54510896877928730e-15,
   -2.290000e+01, 1.13411309324635690e-10, 1.13411309311773570e-10, 5.67056546430246580e-11, 1.89018848724334690e-11, 4.72547121382099240e-12, 9.45094241049248420e-13, 1.57515706269891430e-13, 2.25022435895130590e-14, 2.81278040785698940e-15,
   -2.280000e+01, 1.25338880844974070e-10, 1.25338880829264260e-10, 6.26694403989222890e-11, 2.08898134558342040e-11, 5.22245335872193950e-12, 1.04449066964974320e-12, 1.74081777576742320e-13, 2.48688251686160770e-14, 3.10860309620451750e-15,
   -2.270000e+01, 1.38520886012187950e-10, 1.38520885992999890e-10, 6.92604429773119150e-11, 2.30868143129786130e-11, 5.77170357184864160e-12, 1.15434071181132360e-12, 1.92390117782419000e-13, 2.74843022966879750e-14, 3.43553772617159750e-15,
   -2.260000e+01, 1.53089254764511300e-10, 1.53089254741074970e-10, 7.65446273471011700e-11, 2.55148757667428400e-11, 6.37871893387360370e-12, 1.27574378364987810e-12, 2.12623962900032140e-13, 3.03748515452576710e-14, 3.79685636875619410e-15,
   -2.250000e+01, 1.69189792232887840e-10, 1.69189792204262660e-10, 8.45948960735061490e-11, 2.81982986720852560e-11, 7.04957465847958530e-12, 1.40991492787922570e-12, 2.34985820040973830e-13, 3.35694024995018370e-14, 4.19617522156412460e-15,
   -2.240000e+01, 1.86983638007721840e-10, 1.86983637972758960e-10, 9.34918189514166080e-11, 3.11639396271636140e-11, 7.79098489513660890e-12, 1.55819697436560440e-12, 2.59699494173694950e-13, 3.70999272951261880e-14, 4.63749080089750140e-15,
   -2.230000e+01, 2.06648878878055030e-10, 2.06648878835351260e-10, 1.03324439374971880e-10, 3.44414797631881180e-11, 8.61036992656244340e-12, 1.72207397961865410e-12, 2.87012328038497510e-13, 4.10017606060868320e-14, 5.12521994019336570e-15,
   -2.220000e+01, 2.28382331183999370e-10, 2.28382331131840880e-10, 1.14191165513761950e-10, 3.80637218031483190e-11, 9.51593043340091790e-12, 1.90318607972571820e-12, 3.17197677636131330e-13, 4.53139532856887410e-14, 5.66424399512858830e-15,
   -2.210000e+01, 2.52401510620814130e-10, 2.52401510557107620e-10, 1.26200755214847280e-10, 4.20669183624780740e-11, 1.05167295693840120e-11, 2.10334590538259950e-12, 3.50557648065698900e-13, 5.00796632004138460e-14, 6.25995769780880360e-15,
   -2.200000e+01, 2.78946809209081130e-10, 2.78946809131269810e-10, 1.39473404487823600e-10, 4.64911347774003090e-11, 1.16227836684129700e-11, 2.32455672330775130e-12, 3.87426117093010900e-13, 5.53465871680641350e-14, 6.91832314898794850e-15,
   -2.190000e+01, 3.08283901218829070e-10, 3.08283901123790100e-10, 1.54141950466856080e-10, 5.13806500922593780e-11, 1.28451624913851910e-11, 2.56903248560517660e-12, 4.28172076710242180e-13, 6.11674383231906210e-14, 7.64592948868783410e-15,
   -2.180000e+01, 3.40706402126909450e-10, 3.40706402010828600e-10, 1.70353200889333440e-10, 5.67844002190572430e-11, 1.41961000160706940e-11, 2.83921998773669170e-12, 4.73203326130299640e-13, 6.76004736874288230e-14, 8.45005884241796450e-15,
   -2.170000e+01, 3.76538807219354250e-10, 3.76538807077572750e-10, 1.88269403397004910e-10, 6.27564677044806550e-11, 1.56891168788596710e-11, 3.13782335686773810e-12, 5.22970553176557580e-13, 7.47100772248228560e-14, 9.33875920300294780e-15,
   -2.160000e+01, 4.16139739249242550e-10, 4.16139739076070280e-10, 2.08069869364862860e-10, 6.93566230061727600e-11, 1.73391556938190950e-11, 3.46783111567418200e-12, 5.77971844915817840e-13, 8.25674042175323030e-14, 1.03209249774382680e-14,
   -2.150000e+01, 4.59905537653718580e-10, 4.59905537442205500e-10, 2.29952768509589640e-10, 7.66509226955211350e-11, 1.91627306033759160e-11, 3.83254609247343680e-12, 6.38757672678323810e-13, 9.12510934110227360e-14, 1.14063860049076830e-14,
   -2.140000e+01, 5.08274225252250640e-10, 5.08274224993907970e-10, 2.54137112238611260e-10, 8.47123705739752980e-11, 2.11780925573795950e-11, 4.23561847703022720e-12, 7.05936401356474080e-13, 1.00848054056097130e-13, 1.26060059368766350e-14,
   -2.130000e+01, 5.61729892126191470e-10, 5.61729891810651010e-10, 2.80864945589785050e-10, 9.36216483195680240e-11, 2.34054119747118490e-11, 4.68108235287030770e-12, 7.80180378121030430e-13, 1.11454335724712680e-13, 1.39317909638733210e-14,
   -2.120000e+01, 6.20807540554960390e-10, 6.20807540169558430e-10, 3.10403769699377210e-10, 1.03467922976191060e-10, 2.58669806155804300e-11, 5.17339607172915330e-12, 8.62232661492547810e-13, 1.23176089604941790e-13, 1.53970099771193270e-14,
   -2.110000e+01, 6.86098439498613070e-10, 6.86098439027881960e-10, 3.43049219043209910e-10, 1.14349739367249250e-10, 2.85874346849019580e-11, 5.71748687421624950e-12, 9.52914458114660960e-13, 1.36130630895985690e-13, 1.70163273676139010e-14,
   -2.100000e+01, 7.58256042216238510e-10, 7.58256041641286250e-10, 3.79128020245690930e-10, 1.26376006365262140e-10, 3.15940013996647980e-11, 6.31880020327266280e-12, 1.05313334165867850e-12, 1.50447612935973550e-13, 1.88059497917515770e-14,
   -2.090000e+01, 8.38002526245700690e-10, 8.38002525543452470e-10, 4.19001262069478020e-10, 1.39667086888327180e-10, 3.49167714879990520e-11, 6.98335420396671280e-12, 1.16389233611675320e-12, 1.66270324813527010e-13, 2.07837883723314650e-14,
   -2.080000e+01, 9.26136021199050640e-10, 9.26136020341322760e-10, 4.63068009312933450e-10, 1.54356002532492510e-10, 3.85890003472138220e-11, 7.71779995507904090e-12, 1.28629995439193270e-12, 1.83757125449921650e-13, 2.29696379582944770e-14,
   -2.070000e+01, 1.02353859671178490e-09, 1.02353859566415370e-09, 5.11769296784445560e-10, 1.70589764896394350e-10, 4.26474408748881690e-11, 8.52948803529346740e-12, 1.42158129265418930e-12, 2.03083028504487520e-13, 2.53853752372475280e-14,
   -2.060000e+01, 1.13118509049205290e-09, 1.13118508921247310e-09, 5.65592543326656920e-10, 1.88530846922499150e-10, 4.71327113040982220e-11, 9.42654209020901760e-12, 1.57109029149796130e-12, 2.24441453965363630e-13, 2.80551776835127940e-14,
   -2.050000e+01, 1.25015286482386050e-09, 1.25015286326097840e-09, 6.25076430067607020e-10, 2.08358808980614190e-10, 5.20897017241928260e-11, 1.04179401364542760e-11, 1.73632328661428330e-12, 2.48046163955918000e-13, 3.10057655329592020e-14,
   -2.040000e+01, 1.38163258917063200e-09, 1.38163258726172350e-09, 6.90816291721953060e-10, 2.30272095968045280e-10, 5.75680233557084570e-11, 1.15136044166205460e-11, 1.91893398459637680e-12, 2.74133402130802270e-13, 3.42666692063232140e-14,
   -2.030000e+01, 1.52694015679506780e-09, 1.52694015446352150e-09, 7.63470074900214590e-10, 2.54490023412374030e-10, 6.36225050759114250e-11, 1.27245007043094580e-11, 2.12075001376063440e-12, 3.02964258073155130e-13, 3.78705248574106530e-14,
   -2.020000e+01, 1.68752985466077500e-09, 1.68752985181301800e-09, 8.43764923058751970e-10, 2.81254972454412640e-10, 7.03137421643508380e-11, 1.40627480531692380e-11, 2.34379121562789780e-12, 3.34827280356278600e-13, 4.18534010040368660e-14,
   -2.010000e+01, 1.86500891842450890e-09, 1.86500891494625040e-09, 9.32504453994867120e-10, 3.10834815679450180e-10, 7.77087027604431340e-11, 1.55417400883208640e-11, 2.59028986013089190e-12, 3.70041364421770160e-13, 4.62551595106321980e-14,
   -2.000000e+01, 2.06115361819020370e-09, 2.06115361394184920e-09, 1.03057680272257050e-09, 3.43525598075287300e-10, 8.58813981027037640e-11, 1.71762790540935310e-11, 2.86271298686651670e-12, 4.08958944176434980e-13, 5.11198545352164150e-14,
   -1.990000e+01, 2.27792703601641920e-09, 2.27792703082746770e-09, 1.13896351022478230e-09, 3.79654499948959710e-10, 9.49136232575894220e-11, 1.89827239596576850e-11, 3.16378709598955100e-12, 4.51969519249918920e-13, 5.64961734333789080e-14,
   -1.980000e+01, 2.51749871310048750e-09, 2.51749870676268750e-09, 1.25874934704354400e-09, 4.19583111455981510e-10, 1.04895775751395480e-10, 2.09791543052391430e-11, 3.49652543585987700e-12, 4.99503553214275300e-13, 6.24379240317867700e-14,
   -1.970000e+01, 2.78226636327486680e-09, 2.78226635553386090e-09, 1.39113317002592420e-09, 4.63711051514637360e-10, 1.15927760298324000e-10, 2.31855510275306740e-11, 3.86425816054374260e-12, 5.52036781779192830e-13, 6.90045731477787140e-14,
   -1.960000e+01, 3.07487987013171990e-09, 3.07487986067683370e-09, 1.53743992088353070e-09, 5.12479967324586100e-10, 1.28119988679517820e-10, 2.56239964752521050e-11, 4.27066565899153640e-12, 6.10094974079608510e-13, 7.62618417444422280e-14,
   -1.950000e+01, 3.39826780794684720e-09, 3.39826779639862330e-09, 1.69913388665108760e-09, 5.66377954518213150e-10, 1.41594484780145320e-10, 2.83188954162658940e-11, 4.71981538945660180e-12, 6.74259194706837880e-13, 8.42823626773298690e-14,
   -1.940000e+01, 3.75566675183327000e-09, 3.75566673772823660e-09, 1.87783335475908580e-09, 6.25944442183006810e-10, 1.56486105844074210e-10, 3.12972192881438640e-11, 5.21620258780033100e-12, 7.45171619145677150e-13, 9.31464076153333000e-14,
   -1.930000e+01, 4.15065367047031210e-09, 4.15065365324238660e-09, 2.07532680939326730e-09, 6.91775591645805250e-10, 1.72943892168809470e-10, 3.45887761367051850e-11, 5.76479525709864460e-12, 8.23541960818226570e-13, 1.02942690410457400e-13,
   -1.920000e+01, 4.58718172560528820e-09, 4.58718170456305170e-09, 2.29359083123928980e-09, 7.64530263051605940e-10, 1.91132558748822910e-10, 3.82265089441331980e-11, 6.37108388881175960e-12, 9.10154574055848990e-13, 1.13769254956241180e-13,
   -1.910000e+01, 5.06961983662124080e-09, 5.06961981092019570e-09, 2.53480987975905280e-09, 8.44936609452320900e-10, 2.11234143796065330e-10, 4.22468253324071740e-11, 7.04113641313259240e-12, 1.00587630408506760e-12, 1.25734456420026820e-13,
   -1.900000e+01, 5.60279640614594060e-09, 5.60279637475461250e-09, 2.80139815598597900e-09, 9.33799364401108190e-10, 2.33449830636501440e-10, 4.66899619417901430e-11, 7.78165892846167970e-12, 1.11166516258882510e-12, 1.38958045668612940e-13,
   -1.890000e+01, 6.19204764432258430e-09, 6.19204760598113080e-09, 3.09602376464911160e-09, 1.03200789598873480e-09, 2.58001961216699380e-10, 5.16003871311462670e-11, 8.60006281779322730e-12, 1.22857991566634560e-12, 1.53572367739415770e-13,
   -1.880000e+01, 6.84327097538763050e-09, 6.84327092855727240e-09, 3.42163541744827900e-09, 1.14054510792918840e-09, 2.85136261372178380e-10, 5.70272460303883370e-11, 9.50453892371568250e-12, 1.35779068014540250e-12, 1.69723686350405990e-13,
   -1.870000e+01, 7.56298406106641130e-09, 7.56298400386768260e-09, 3.78149194473511420e-09, 1.26049727677922010e-09, 3.15124300128563070e-10, 6.30248523992160290e-11, 1.05041395243705740e-11, 1.50059063429142890e-12, 1.87573647703209600e-13,
   -1.860000e+01, 8.35839003151195470e-09, 8.35838996164926950e-09, 4.17919491096195200e-09, 1.39306492374552880e-09, 3.48266207648821830e-10, 6.96532322147404860e-11, 1.16088689307822490e-11, 1.65840896010956840e-12, 2.07300898227457540e-13,
   -1.850000e+01, 9.23744957664011970e-09, 9.23744949130964510e-09, 4.61872466032434870e-09, 1.53957482988780090e-09, 3.84893679028459940e-10, 7.69787244282962100e-11, 1.28297836122509570e-11, 1.83282514676019000e-12, 2.29102872454708450e-13,
   -1.840000e+01, 1.02089606193747600e-08, 1.02089605151518820e-08, 5.10448015335306570e-09, 1.70149331496910550e-09, 4.25373294001319100e-10, 8.50746449038813890e-11, 1.41791028518529920e-11, 2.02558479822846130e-12, 2.53197768912390380e-13,
   -1.830000e+01, 1.12826463681984940e-08, 1.12826462409003860e-08, 5.64132299315208550e-09, 1.88044091285195820e-09, 4.70110185780288720e-10, 9.40220201829781080e-11, 1.56703310394701250e-11, 2.23861710344075750e-12, 2.79826733809260220e-13,
   -1.820000e+01, 1.24692526302687260e-08, 1.24692524747864650e-08, 6.23462608191097280e-09, 2.07820859031548680e-09, 5.19552095751454350e-10, 1.03910398419324820e-10, 1.73183928262325570e-11, 2.47405414365581420e-12, 3.09256274362694960e-13,
   -1.810000e+01, 1.37806553590393020e-08, 1.37806551691328390e-08, 6.89032739465996030e-09, 2.29677567161568300e-09, 5.74193854601770820e-10, 1.14838745599495390e-10, 1.91397824929635610e-11, 2.73425223034199750e-12, 3.41780925915359470e-13,
   -1.800000e+01, 1.52299795127603490e-08, 1.52299792808080730e-08, 7.61498940845176490e-09, 2.53832964818240970e-09, 6.34582334728182720e-10, 1.26916436018670240e-10, 2.11527290274570720e-11, 3.02181548706907280e-12, 3.77726199527563320e-13,
   -1.790000e+01, 1.68317304134304430e-08, 1.68317301301232970e-08, 8.41586478175450400e-09, 2.80528807171341020e-09, 7.01321923492643770e-10, 1.40264346924247440e-10, 2.33773785626152040e-11, 3.33962191139519260e-12, 4.17451839537112450e-13,
   -1.780000e+01, 1.86019389208833740e-08, 1.86019385748512410e-08, 9.30096894139349600e-09, 3.10032274977642130e-09, 7.75080572100071710e-10, 1.55016068282403760e-10, 2.58359960011985530e-11, 3.69085217754267710e-12, 4.61355423678788540e-13,
   -1.770000e+01, 2.05583218749598760e-08, 2.05583214523152770e-08, 1.02791603035130500e-08, 3.42638648607463180e-09, 8.56596480637138920e-10, 1.71319239774824100e-10, 2.85531878449380760e-11, 4.07902146807791210e-12, 5.09876341781742870e-13,
   -1.760000e+01, 2.27204594115193490e-08, 2.27204588953000720e-08, 1.13602289314307710e-08, 3.78674263299742590e-09, 9.46685486176283100e-10, 1.89337028406032540e-10, 3.15561484579333760e-11, 4.50801465311388360e-12, 5.63500192849081350e-13,
   -1.750000e+01, 2.51099909269281570e-08, 2.51099902964165110e-08, 1.25549945176966280e-08, 4.18499775222447230e-09, 1.04624922788559510e-09, 2.09249761508916910e-10, 3.48749322287556890e-11, 4.98212516904363310e-12, 6.22763644507803310e-13,
   -1.740000e+01, 2.77508316522988820e-08, 2.77508308821902250e-08, 1.38754146709864760e-08, 4.62513771025642260e-09, 1.15628417086125370e-09, 2.31256731491119580e-10, 3.85427543548143520e-11, 5.50609798582156390e-12, 6.88259803440489660e-13,
   -1.730000e+01, 3.06694120050226880e-08, 3.06694110644098580e-08, 1.53347045915921220e-08, 5.11156757012220070e-09, 1.27789157899298500e-09, 2.55578190383583880e-10, 4.25963232589327710e-11, 6.08517709270806810e-12, 7.60644150518970060e-13,
   -1.720000e+01, 3.38949421131021010e-08, 3.38949409642349980e-08, 1.69474693332504390e-08, 5.64915567850547670e-09, 1.41228853667073060e-09, 2.82457554151908200e-10, 4.70762079645808200e-11, 6.72515797759038280e-12, 8.40641100005629380e-13,
   -1.710000e+01, 3.74597041597231010e-08, 3.74597027564936650e-08, 1.87298499750174500e-08, 6.24328238951961330e-09, 1.56082012963684120e-09, 3.12163838830167140e-10, 5.20272441059726340e-11, 7.43244562493776210e-12, 9.29051248425967260e-13,
   -1.700000e+01, 4.13993754739433060e-08, 4.13993737600350160e-08, 2.06996851661092880e-08, 6.89989391276435460e-09, 1.72497290688843770e-09, 3.44994352856659150e-10, 5.74989826357830220e-11, 8.21411861264436620e-12, 1.02675938560909540e-12,
   -1.690000e+01, 4.57533856010857270e-08, 4.57533835077134340e-08, 2.28766896604845190e-08, 7.62556182458013600e-09, 1.90638975835442080e-09, 3.81277672554682090e-10, 6.35461857204012030e-11, 9.07799994900584020e-12, 1.13474334800918220e-12,
   -1.680000e+01, 5.05653109265044130e-08, 5.05653083696537430e-08, 2.52826516279763310e-08, 8.42754883809187850e-09, 2.10688635723961830e-09, 4.21376930534641180e-10, 7.02293747847084550e-11, 1.00327353584760360e-11, 1.25408380283645790e-12,
   -1.670000e+01, 5.58833108022380710e-08, 5.58833076792936500e-08, 2.79416507167025720e-08, 9.31388149027153060e-09, 2.32846933158668340e-09, 4.65693449924935570e-10, 7.76154361900615100e-11, 1.10878797993319670e-11, 1.38597506082503060e-12,
   -1.660000e+01, 6.17606095414307170e-08, 6.17606057270578270e-08, 3.08802990491562620e-08, 1.02934304734705510e-08, 2.57335634691038030e-09, 5.14670760799279210e-10, 8.57782906056672360e-11, 1.22539930786763320e-11, 1.53173702573831300e-12,
   -1.650000e+01, 6.82560291044628650e-08, 6.82560244455773610e-08, 3.41280075639034880e-08, 1.13759994153780190e-08, 2.84399830088317200e-09, 5.68799038992244420e-10, 9.47996327706489370e-11, 1.35427455211542820e-11, 1.69282840006403590e-12,
   -1.640000e+01, 7.54345778080666270e-08, 7.54345721176910950e-08, 3.77172803684704520e-08, 1.25724229959071790e-08, 3.14310385218563120e-09, 6.28620011720854130e-10, 1.04769749048148210e-10, 1.49670347482547110e-11, 1.87086127889146020e-12,
   -1.630000e+01, 8.33681009493868880e-08, 8.33680939991466200e-08, 4.16840400493336380e-08, 1.38946753829520070e-08, 3.47366652899216700e-09, 6.94732379100360320e-10, 1.15788420950832750e-10, 1.65411147361098870e-11, 2.06761727781813410e-12,
   -1.620000e+01, 9.21359998566180990e-08, 9.21359913675756180e-08, 4.60679871947461210e-08, 1.53559900722216970e-08, 3.83899468837583980e-09, 7.67797805803686010e-10, 1.27965923676963010e-10, 1.82807384424078510e-11, 2.28506535603969100e-12,
   -1.610000e+01, 1.01826026562659480e-07, 1.01826016194119800e-07, 5.09129977285212690e-08, 1.69709923304823920e-08, 4.24274462644237430e-09, 8.48547542817660610e-10, 1.41424129646242880e-10, 2.02033154286030170e-11, 2.52538151268420600e-12,
   -1.600000e+01, 1.12535162055094990e-07, 1.12535149390932300e-07, 5.62675620313048730e-08, 1.87558455676622010e-08, 4.68895717053024010e-09, 9.37789745552565470e-10, 1.56297728074587390e-10, 2.23280860534098560e-11, 2.79097055313068080e-12,
   -1.590000e+01, 1.24370586892242380e-07, 1.24370571424199500e-07, 6.21852702440587900e-08, 2.07284131026608780e-08, 5.18209811965397050e-09, 1.03641756152716040e-09, 1.72735572787081680e-10, 2.46763139790700080e-11, 3.08449014268218680e-12,
   -1.580000e+01, 1.37450753899427740e-07, 1.37450735006718000e-07, 6.87253486106518530e-08, 2.29084369417484450e-08, 5.72710293787130660e-09, 1.14541806854910830e-09, 1.90902171750384650e-10, 2.72714989147434240e-11, 3.40887738775180700e-12,
   -1.570000e+01, 1.51906573681285750e-07, 1.51906550605678630e-07, 7.59532522272356920e-08, 2.53177353586796520e-08, 6.32942614780642060e-09, 1.26588215281746470e-09, 2.10979333222426280e-10, 3.01396117234033410e-11, 3.76737820988778490e-12,
   -1.560000e+01, 1.67882724814952580e-07, 1.67882696630343290e-07, 8.39413201305670910e-08, 2.79804212537907200e-08, 6.99509591858540820e-09, 1.39901542577430220e-09, 2.33167984982521830e-10, 3.33093542418083850e-11, 4.16357980575917900e-12,
   -1.550000e+01, 1.85539101836833140e-07, 1.85539067412074840e-07, 9.27694992812854920e-08, 3.09231434772669350e-08, 7.73077439440740950e-09, 1.54615028892062660e-09, 2.57690184834589790e-10, 3.68124464096959630e-11, 4.60144651691731410e-12,
   -1.540000e+01, 2.05052415514691310e-07, 2.05052373468198210e-07, 1.02526144687614620e-07, 3.41753535315571700e-08, 8.54382436740523970e-09, 1.70875926729130720e-09, 2.84791342487244140e-10, 4.06839435767308960e-11, 5.08535946704142080e-12,
   -1.530000e+01, 2.26617961420859720e-07, 2.26617910065159300e-07, 1.13308903676890850e-07, 3.77696003218493840e-08, 9.44238296191396000e-09, 1.88846974496867490e-09, 3.14742675026025340e-10, 4.49625871566247750e-11, 5.62016036179708380e-12,
   -1.520000e+01, 2.50451574506755260e-07, 2.50451511780764100e-07, 1.25225693164406590e-07, 4.17418559041675970e-08, 1.04354430674030480e-08, 2.08708025003213910e-09, 3.47843920526067640e-10, 4.96911921302038970e-11, 6.21119988774926690e-12,
   -1.510000e+01, 2.76791789240365410e-07, 2.76791712626670810e-07, 1.38395779699662040e-07, 4.61318754907929540e-08, 1.15329433348336180e-08, 2.30657845183041800e-09, 3.84426336931072520e-10, 5.49170752662941200e-11, 6.86439119237452920e-12,
   -1.500000e+01, 3.05902226925624720e-07, 3.05902133349452300e-07, 1.52950973098582340e-07, 5.09835953154601910e-08, 1.27458676368528910e-08, 2.54916105057859740e-09, 4.24856016172213160e-10, 6.06925283347279390e-11, 7.58626897755046640e-12,
   -1.490000e+01, 3.38074234096247890e-07, 3.38074119802060130e-07, 1.69036945606880940e-07, 5.63455723395662010e-08, 1.40863549868901440e-08, 2.81725575819485410e-09, 4.69537546646778090e-10, 6.70753410335425690e-11, 8.38405479444791760e-12,
   -1.480000e+01, 3.73629798389248960e-07, 3.73629658790022730e-07, 1.86814689795837280e-07, 6.22714701992151870e-08, 1.55678210168109670e-08, 3.11354559018853250e-09, 5.18918060651937380e-10, 7.41293788468821130e-11, 9.26572918904053130e-12,
   -1.470000e+01, 4.12924771080390220e-07, 4.12924600573523660e-07, 2.06462129779965680e-07, 6.88205962555281830e-08, 1.72050922283713330e-08, 3.44099571150166380e-09, 5.73491707208584540e-10, 8.19252215961033100e-11, 1.02401114150638000e-11,
   -1.460000e+01, 4.56352428532766150e-07, 4.56352220275227090e-07, 2.28175901880169570e-07, 7.60584951218555640e-08, 1.90145543614346950e-08, 3.80288310471802740e-09, 6.33804594951417640e-10, 9.05408690493897220e-11, 1.13169475058357380e-11,
   -1.450000e+01, 5.04347408201451650e-07, 5.04347153835143500e-07, 2.52173322551391870e-07, 8.40576046064723250e-08, 2.10143163630518180e-08, 4.20282935724158890e-09, 7.00460254446204730e-10, 1.00062520620668450e-10, 1.25070075785966360e-11,
   -1.440000e+01, 5.57390058585609980e-07, 5.57389747901932570e-07, 2.78694563267462030e-07, 9.28979806336576940e-08, 2.32243915974628080e-08, 4.64483689518984350e-09, 7.74125674469697980e-10, 1.10585436923243300e-10, 1.38221933357032870e-11,
   -1.430000e+01, 6.16011246661965180e-07, 6.16010867192109130e-07, 3.08005054126432290e-07, 1.02668098395963010e-07, 2.56668981094088650e-08, 5.13332902615420870e-09, 8.55537972502441770e-10, 1.22214891754339240e-10, 1.52756568269570680e-11,
   -1.420000e+01, 6.80797670911850050e-07, 6.80797207426381290e-07, 3.40398140228037450e-07, 1.13465737752892570e-07, 2.83662799435665070e-08, 5.67319419099263970e-09, 9.45511765996215670e-10, 1.35067223981536400e-10, 1.68819316474784880e-11,
   -1.410000e+01, 7.52397733113647920e-07, 7.52397167011299190e-07, 3.76198017403726710e-07, 1.25398961733719590e-07, 3.13495517333213600e-08, 6.26983486681252740e-09, 1.04494731794734100e-09, 1.49270999789335810e-10, 1.86570778668549570e-11,
   -1.400000e+01, 8.31528027664132090e-07, 8.31527336225271330e-07, 4.15762976674349830e-07, 1.38587197933167670e-07, 3.46465690045819890e-08, 6.92922160969115450e-09, 1.15483953800280790e-09, 1.64968296833256300e-10, 2.06188421188755080e-11,
   -1.390000e+01, 9.18980513371998710e-07, 9.18979668846814780e-07, 4.59488989898999510e-07, 1.53162433617504030e-07, 3.82903268972101850e-08, 7.65795277692495110e-09, 1.27628792882308540e-09, 1.82316123050209980e-10, 2.27868344282111740e-11,
   -1.380000e+01, 1.01563043949625650e-06, 1.01562940799106700e-06, 5.07813672491391420e-07, 1.69270536495416780e-07, 4.23172902904497230e-08, 8.46332052519958820e-09, 1.41050757680552740e-09, 2.01487984198646820e-10, 2.51827235121776000e-11,
   -1.370000e+01, 1.12244510534991550e-06, 1.12244384546690100e-06, 5.61220662851850140e-07, 1.87072714364297240e-07, 4.67677586323085410e-08, 9.35338374359176250e-09, 1.55884129662682670e-09, 2.22675615662538020e-10, 2.78304524736295260e-11,
   -1.360000e+01, 1.24049354113058110e-06, 1.24049200230635550e-06, 6.20244462330861060e-07, 2.06747128230651470e-07, 5.16862691192767580e-08, 1.03370486493599110e-08, 1.72277305049100430e-09, 2.46090895663076440e-10, 3.07564769965724290e-11,
   -1.350000e+01, 1.37095720685784480e-06, 1.37095532733418170e-06, 6.85475784146004530e-07, 2.28490675037187380e-07, 5.71220422554890060e-08, 1.14241578507341960e-08, 1.90394277558092690e-09, 2.71967958795809030e-10, 3.39900283708562110e-11,
   -1.340000e+01, 1.51514181648504940e-06, 1.51513952083032530e-06, 7.57567464763916870e-07, 2.52520957823953280e-07, 6.31294742432541720e-08, 1.26255887651223760e-08, 2.10416276712683950e-09, 3.00565530764881780e-10, 3.75634039080737810e-11,
   -1.330000e+01, 1.67449040551145870e-06, 1.67448760159334050e-06, 8.37240996883247270e-07, 2.79078463023495490e-07, 6.97686811239351100e-08, 1.39533623741243090e-08, 2.32543577985620750e-09, 3.32169507337573860e-10, 4.15122875693582470e-11,
   -1.320000e+01, 1.85059777286345340e-06, 1.85059434815133640e-06, 9.25293749369889060e-07, 3.08428966659114640e-07, 7.71061001041077820e-08, 1.54207633994051810e-08, 2.56997502752138410e-09, 3.67095802910524850e-10, 4.58761039091410220e-11,
   -1.310000e+01, 2.04522644156374070e-06, 2.04522225861254330e-06, 1.02260694636362950e-06, 3.40866193501336790e-07, 8.52151540718139430e-08, 1.70424730968044440e-08, 2.84022627887282720e-09, 4.05693496684509510e-10, 5.06984087491535010e-11,
   -1.300000e+01, 2.26032429790357460e-06, 2.26031918883764270e-06, 1.13015448536443780e-06, 3.76714755763438350e-07, 9.41769859371668700e-08, 1.88347159911529010e-08, 3.13889226903028760e-09, 4.48348307309722420e-10, 5.60273203362117640e-11,
   -1.290000e+01, 2.49804408562679740e-06, 2.49803784540254350e-06, 1.24901268249260650e-06, 4.16333400707346860e-07, 1.04081270126766940e-07, 2.08154220123401690e-08, 3.46895966789294470e-09, 4.95486430012475780e-10, 6.19159951078826660e-11,
   -1.280000e+01, 2.76076495019305620e-06, 2.76075732836994600e-06, 1.38037104238290480e-06, 4.60118599613964810e-07, 1.15027109329104260e-07, 2.30044056455348990e-08, 3.83372887222821100e-09, 5.47578773672763900e-10, 6.84231525942886660e-11,
   -1.270000e+01, 3.05111624869765530e-06, 3.05110693938729200e-06, 1.52554416041168670e-06, 5.08508513977659770e-07, 1.27124025435933060e-07, 2.54235638765754010e-08, 4.23684691561622300e-09, 6.05145639118408730e-10, 7.56136544244531530e-11,
   -1.260000e+01, 3.37200386369079290e-06, 3.37199249328073580e-06, 1.68598487626865240e-06, 5.61987378546747840e-07, 1.40493054560707780e-07, 2.80970948990031270e-08, 4.68234382074759530e-09, 6.68761884064922050e-10, 8.35591428840215460e-11,
   -1.250000e+01, 3.72663928418656090e-06, 3.72662539634620680e-06, 1.86329881038450380e-06, 6.21090344987523090e-07, 1.55267957048707560e-07, 3.10517397537616080e-08, 5.17467275194488770e-09, 7.39062624692496680e-10, 9.23387449901975160e-11,
   -1.240000e+01, 4.11857174483263580e-06, 4.11855478219941820e-06, 2.05926042853635350e-06, 6.86408834539742390e-07, 1.71596554534477040e-07, 3.43170492981492790e-08, 5.71875436249466170e-09, 8.16749528846059030e-10, 1.02039848611429170e-10,
   -1.230000e+01, 4.55172374479989440e-06, 4.55170302661084600e-06, 2.27583079521067730e-06, 7.58596453101365150e-07, 1.89642207361637810e-07, 3.79256791492819520e-08, 6.32002577174944240e-09, 9.02597761305675350e-10, 1.12758957765585490e-10,
   -1.220000e+01, 5.03043030175513060e-06, 5.03040499652610920e-06, 2.51517719316133000e-06, 8.38375527779922470e-07, 2.09585447070191200e-07, 4.19137155214045760e-08, 6.98449465133025160e-09, 9.97463647539522230e-10, 1.24602634882612210e-10,
   -1.210000e+01, 5.55948233363450570e-06, 5.55945142579068760e-06, 2.77969480522335740e-06, 9.26544330131625550e-07, 2.31625780190364950e-07, 4.63210351783788810e-08, 7.71879894849249550e-09, 1.10229312885685420e-09, 1.37688538516806380e-10,
   -1.200000e+01, 6.14417460221471820e-06, 6.14413685133317550e-06, 3.07203067501699310e-06, 1.02398505813788140e-06, 2.55983681274537310e-07, 5.11917030553082150e-08, 8.53027282820451160e-09, 1.21813108896187020e-09, 1.52146565740089000e-10,
   -1.190000e+01, 6.79035869809513550e-06, 6.79031258912388690e-06, 3.39511018590379080e-06, 1.13167265650892300e-06, 2.82902794965879080e-07, 5.65744114695246260e-08, 9.42701947414730900e-09, 1.34613163960856930e-09, 1.68120109240107280e-10,
   -1.180000e+01, 7.50450159711002580e-06, 7.50444527956580460e-06, 3.75216632266131660e-06, 1.25068456322868000e-06, 3.12652368961594790e-07, 6.25229652442687930e-08, 1.04179914531163490e-08, 1.48756946140726180e-09, 1.85767439983337370e-10,
   -1.170000e+01, 8.29375037389160510e-06, 8.29368158759634180e-06, 4.14677200807340300e-06, 1.38221147944511360e-06, 3.45529941999465520e-07, 6.90968175118825750e-08, 1.15130794176650280e-08, 1.64385230486978280e-09, 2.05263227179602890e-10,
   -1.160000e+01, 9.16600371985334720e-06, 9.16591970422915550e-06, 4.58287583726047290e-06, 1.52756926995416800e-06, 3.81864313499767260e-07, 7.63616614509698770e-08, 1.27232099987598540e-08, 1.81653476653134410e-09, 2.26800208194066260e-10,
   -1.150000e+01, 1.01299909808739210e-05, 1.01298883641566480e-05, 5.06484156640055810e-06, 1.68821211272117110e-06, 4.22018824253730080e-07, 8.43900837478793370e-08, 1.40604538242312510e-08, 2.00733346547603950e-09, 2.50591021986417530e-10,
   -1.140000e+01, 1.11953595051133130e-05, 1.11952341690388640e-05, 5.59749174984816610e-06, 1.86574702823837760e-06, 4.66394980589781680e-07, 9.32622861614342590e-08, 1.55381446904248170e-08, 2.21814375683460740e-09, 2.76870220601442240e-10,
   -1.130000e+01, 1.23727711744276150e-05, 1.23726180889610800e-05, 6.18615596090809690e-06, 2.06194993314847120e-06, 5.15436457797198270e-07, 1.03066882215769650e-07, 1.71710110142080070e-08, 2.45105813081809670e-09, 3.05896474172172960e-10,
   -1.120000e+01, 1.36740090845997360e-05, 1.36738221060752910e-05, 6.83672407706994550e-06, 2.27878337760155970e-06, 5.69633522273677260e-07, 1.13901776754521430e-07, 1.89753208008991080e-08, 2.70838645858565140e-09, 3.37954985802327180e-10,
   -1.110000e+01, 1.51120954409751590e-05, 1.51118670655465430e-05, 7.55570516079588530e-06, 2.51841614239821710e-06, 6.29527915921014080e-07, 1.25875136865744330e-07, 2.09690414812555340e-08, 2.99267825968002520e-09, 3.73360133536598300e-10,
   -1.100000e+01, 1.67014218480951810e-05, 1.67011429106034340e-05, 8.35029252246862110e-06, 2.78324489025938510e-06, 6.95718250776955360e-07, 1.39106463536484480e-07, 2.31720160978619510e-08, 3.30674717983455920e-09, 4.12458357364119000e-10,
   -1.090000e+01, 1.84578932956670710e-05, 1.84575526018421570e-05, 9.22843561338465370e-06, 3.07591808572564290e-06, 7.68865966779621040e-07, 1.53727774324814250e-07, 2.56061574585067640e-08, 3.65369788255586900e-09, 4.55631309778500060e-10,
   -1.080000e+01, 2.03990872799214420e-05, 2.03986711571595860e-05, 1.01989194643064650e-05, 3.39936242040828650e-06, 8.49701910954561280e-07, 1.69884908351423830e-07, 2.82956620217400030e-08, 4.03695557286274760e-09, 5.03299288768604190e-10,
   -1.070000e+01, 2.25444296504454500e-05, 2.25439213991371880e-05, 1.12714524597185660e-05, 3.75681200479748040e-06, 9.39033602238241090e-07, 1.87738966018924850e-07, 3.12672454379727320e-08, 4.46029838670066780e-09, 5.55924972150874520e-10,
   -1.060000e+01, 2.49153889394299740e-05, 2.49147681628239680e-05, 1.24567633202728720e-05, 4.15184061479805410e-06, 1.03775325265255550e-06, 2.07467897071115790e-07, 3.45504018382092800e-08, 4.92789289455702600e-09, 6.14017471766447630e-10,
   -1.050000e+01, 2.75356911145834730e-05, 2.75349329002983130e-05, 1.37667082567419540e-05, 4.58839731085322790e-06, 1.14684662267372210e-06, 2.29268251912926360e-07, 3.81776891415831150e-08, 5.44433298228359790e-09, 6.78136725116404050e-10,
   -1.040000e+01, 3.04315569005653800e-05, 3.04306308209099890e-05, 1.52143893589816480e-05, 5.07084578019283920e-06, 1.26740279644938860e-06, 2.53357112530683230e-07, 4.21850428417587080e-08, 6.01468238557354170e-09, 7.48898240323386140e-10,
   -1.030000e+01, 3.36319640386712970e-05, 3.36308329296661930e-05, 1.68142853938694170e-05, 5.60400778868748910e-06, 1.40062497106620500e-06, 2.79974220887245140e-07, 4.66121209299954760e-08, 6.64452116626698870e-09, 8.26978207671318100e-10,
   -1.020000e+01, 3.71689371028894690e-05, 3.71675555730041110e-05, 1.85823963079666950e-05, 6.19321116831341250e-06, 1.54784236341643510e-06, 3.09384324322487890e-07, 5.15026828177811800e-08, 7.33999642781077380e-09, 9.13118987119026290e-10,
   -1.010000e+01, 4.10778677647634140e-05, 4.10761803735433150e-05, 2.05364028648659900e-05, 6.84434280966313920e-06, 1.71052334841719820e-06, 3.41879759270232850e-07, 5.69050053322321540e-08, 8.10787757266659710e-09, 1.00813497577477860e-09,
   -1.000000e+01, 4.53978687024343950e-05, 4.53958077359516670e-05, 2.26958429950565950e-05, 7.56390717663194450e-06, 1.89028995346304070e-06, 3.77783296510340980e-07, 6.28723390698916340e-08, 8.95561640484862610e-09, 1.11291885194540530e-09,
   -9.900000e+00, 5.01721646837642990e-05, 5.01696474376552460e-05, 2.50823065990142520e-05, 8.35909091275185550e-06, 2.08893384610552530e-06, 4.17451273209466730e-07, 6.94634086049880310e-08, 9.89141237430907690e-09, 1.22844818252732500e-09,
   -9.800000e+00, 5.54485247227951100e-05, 5.54454501839011790e-05, 2.77196507235353010e-05, 9.23783416603743410e-06, 2.30843396511463450e-06, 4.61277039162941680e-07, 7.67429602510735640e-08, 1.09242832443546240e-08, 1.35579236756070800e-09,
   -9.700000e+00, 6.12797396166024810e-05, 6.12759844101150000e-05, 3.06342372286881010e-05, 1.02089093220870710e-05, 2.55097595934783400e-06, 5.09694746927041420e-07, 8.47823612630035150e-08, 1.20641414358960640e-08, 1.49611987895293810e-09,
   -9.600000e+00, 6.77241496197703530e-05, 6.77195630593286340e-05, 3.38551952798434980e-05, 1.12820079040004910e-05, 2.81897361428865210e-06, 5.63183517913635560e-07, 9.36602545299377210e-08, 1.33218762596425380e-08, 1.65070572874813100e-09,
   -9.500000e+00, 7.48462275106112290e-05, 7.48406255528386500e-05, 3.74147112379321610e-05, 1.24678364729066360e-05, 3.11509246276708790e-06, 6.22272018989970910e-07, 1.03463272937304900e-07, 1.47094421850544260e-08, 1.82093907471919640e-09,
   -9.400000e+00, 8.27172228516665380e-05, 8.27103807127102440e-05, 4.13483487833615580e-05, 1.37782224450407310e-05, 3.44227579427708280e-06, 6.87543486658808110e-07, 1.14286817650954280e-07, 1.62399532076433140e-08, 2.00833083607347480e-09,
   -9.300000e+00, 9.14158738521617520e-05, 9.14075169901696020e-05, 4.56954023970424880e-05, 1.52262308308659580e-05, 3.80377329648256020e-06, 7.59641238451185740e-07, 1.26235904580311340e-07, 1.79277832574712820e-08, 2.21452114796532250e-09,
   -9.200000e+00, 1.01029193907772890e-04, 1.01018987009751250e-04, 5.04992876380486460e-05, 1.68262929992220640e-05, 4.20317258295547740e-06, 8.39274713695976610e-07, 1.39426083184529780e-07, 1.97886724333610590e-08, 2.44128642819716600e-09,
   -9.100000e+00, 1.11653340629562940e-04, 1.11640874161089200e-04, 5.58079720039937080e-05, 1.85943486753980790e-05, 4.64443388288645880e-06, 9.27226088269878110e-07, 1.53984431565228960e-07, 2.18398386394126290e-08, 2.69054576040562760e-09,
   -9.000000e+00, 1.23394575986231720e-04, 1.23379349764848910e-04, 6.16744505398747550e-05, 2.05480024968597510e-05, 5.13192819138484880e-06, 1.02435751019101200e-06, 1.70050631402632990e-07, 2.41000939307157230e-08, 2.96436621206719720e-09,
   -8.900000e+00, 1.36370327079497270e-04, 1.36351730213389510e-04, 6.81572707766476910e-05, 2.27066965745660630e-05, 5.67047920292362240e-06, 1.13161900488519120e-06, 1.87778125690478840e-07, 2.65899645286944180e-08, 3.26496659905703470e-09,
   -8.800000e+00, 1.50710358059757950e-04, 1.50687644447731440e-04, 7.53211120350158190e-05, 2.50919006417660350e-05, 6.26540937530157850e-06, 1.25005710047957070e-06, 2.07335361348967370e-07, 2.93318130250369810e-08, 3.59471907670552640e-09,
   -8.700000e+00, 1.66558064777336060e-04, 1.66530323188393680e-04, 8.32374246258398500e-05, 2.77273215161910480e-05, 6.92259049692900950e-06, 1.38082422437827180e-06, 2.28907117570330310e-07, 3.23499607343891410e-08, 3.95614777483460480e-09,
   -8.600000e+00, 1.84071904963424330e-04, 1.84038022497227440e-04, 9.19851350192269540e-05, 3.06391337558132180e-05, 7.64849915592083420e-06, 1.52518892241238990e-06, 2.52695919086896470e-07, 3.56708074528071020e-08, 4.35192349451371000e-09,
   -8.500000e+00, 2.03426978055206530e-04, 2.03385595519805850e-04, 1.01651423642826370e-04, 3.38562335528360290e-05, 8.45027753490249860e-06, 1.68454695073643540e-06, 2.78923531334687870e-07, 3.93229449984878610e-08, 4.78485323932654510e-09,
   -8.400000e+00, 2.24816770233295710e-04, 2.24766227653117570e-04, 1.12332582609200300e-04, 3.74105180850928480e-05, 9.33579997993108880e-06, 1.86043328799366020e-06, 3.07832531578976390e-07, 4.33372598136734670e-08, 5.25786305429556710e-09,
   -8.300000e+00, 2.48455081839335190e-04, 2.48393351911643400e-04, 1.24134961365244140e-04, 4.13371927279999960e-05, 1.03137458146274090e-05, 2.05453511063054410e-06, 3.39687946303440240e-07, 4.77470185436931870e-08, 5.77397227994798100e-09,
   -8.200000e+00, 2.74578156101332910e-04, 2.74502762937524930e-04, 1.37176009006270340e-04, 4.56751087227271500e-05, 1.13936788903748990e-05, 2.26870576703248610e-06, 3.74778940336350900e-07, 5.25879288228435630e-08, 6.33625688386963770e-09,
   -8.100000e+00, 3.03447030028919710e-04, 3.03354949928886350e-04, 1.51585422805842700e-04, 5.04671340958347000e-05, 1.25861343788083410e-05, 2.50497977566682690e-06, 4.13420537045402340e-07, 5.78981654174100510e-08, 6.94779899154798850e-09,
   -8.000000e+00, 3.35350130466478110e-04, 3.35237670756474240e-04, 1.67506413381611650e-04, 5.57605608301848050e-05, 1.39027133219421470e-05, 2.76558885777998050e-06, 4.55955341173887610e-07, 6.37183493199892450e-08, 7.61161908358632450e-09,
   -7.900000e+00, 3.70606140626397190e-04, 3.70468791714927220e-04, 1.85097097848343610e-04, 6.16075514935198190e-05, 1.53561854558388170e-05, 3.05297899532584760e-06, 5.02755226147294880e-07, 7.00914642564759430e-08, 8.33058653549461870e-09,
   -7.800000e+00, 4.09567164986051520e-04, 4.09399419723416790e-04, 2.04532033302025330e-04, 6.80656287356995920e-05, 1.69606008127534100e-05, 3.36982847838181250e-06, 5.54222935506777060e-07, 7.70626912387017960e-08, 9.10730322364060100e-09,
   -7.700000e+00, 4.52622223240535340e-04, 4.52417356363564130e-04, 2.26003904032112200e-04, 7.51982112629216880e-05, 1.87314105805526300e-05, 3.71906687172527860e-06, 6.10793532989770900e-07, 8.46791371308550300e-08, 9.94395377704568310e-09,
   -7.600000e+00, 5.00201107079565190e-04, 4.99950905932041610e-04, 2.49725376969388170e-04, 8.30752000803313180e-05, 2.06855976523322790e-05, 4.10389478553117670e-06, 6.72935617035876790e-07, 9.29894275301909560e-08, 1.08421146857547360e-08,
   -7.500000e+00, 5.52778636923599550e-04, 5.52473072702160420e-04, 2.75931141039014920e-04, 9.17736189542991000e-05, 2.28418172279072030e-05, 4.52780427792291040e-06, 7.41152192379701890e-07, 1.02043127397168810e-07, 1.18025128443360290e-08,
   -7.400000e+00, 6.10879359434402100e-04, 6.10506185842619100e-04, 3.04880147293571270e-04, 1.01378313170817730e-04, 2.52205477252676030e-05, 4.99459964447474200e-06, 8.15981062985805230e-07, 1.11889944579740490e-07, 1.28247221712008350e-08,
   -7.300000e+00, 6.75082730632839950e-04, 6.74626993939641310e-04, 3.36858067936593250e-04, 1.11982710742321510e-04, 2.78442521156263900e-05, 5.50841825822970310e-06, 8.97994575796464900e-07, 1.22578661398168580e-07, 1.39067846545080620e-08,
   -7.200000e+00, 7.46028833836697640e-04, 7.45472274815781850e-04, 3.72179993596052540e-04, 1.23689650223444630e-04, 3.07375495998707350e-05, 6.07375100916502470e-06, 9.87798502316530880e-07, 1.34155727489317820e-07, 1.50447394841792920e-08,
   -7.100000e+00, 8.24424686398296850e-04, 8.23745010334753960e-04, 4.11193389445559590e-04, 1.36612279213740930e-04, 3.39273972842915470e-05, 6.69546174926588850e-06, 1.08602979344121760e-06, 1.46663432811573240e-07, 1.62320407865954680e-08,
   -7.000000e+00, 9.11051194400645390e-04, 9.10221180121826650e-04, 4.54281331967594540e-04, 1.50875027423562060e-04, 3.74432811738753480e-05, 7.37880497244096980e-06, 1.19335288038025070e-06, 1.60137562702363960e-07, 1.74588408352964750e-08,
   -6.900000e+00, 1.00677082008563870e-03, 1.00575723260146270e-03, 5.01866049266858180e-04, 1.66614657822646980e-04, 4.13174153633909100e-05, 8.12944074027213180e-06, 1.31045411900320490e-06, 1.74604416741992440e-07, 1.87111114252563720e-08,
   -6.800000e+00, 1.11253603286032460e-03, 1.11129829643591200e-03, 5.54412788819914770e-04, 1.83981398835657290e-04, 4.55849477472550340e-05, 8.95344559655885660e-06, 1.43803388408029740e-06, 1.90077049463262680e-07, 1.99695713410905580e-08,
   -6.700000e+00, 1.22939862127742150e-03, 1.22788720030742250e-03, 6.12434037322569110e-04, 2.03140159741224950e-04, 5.02841698614197790e-05, 9.85731788582396570e-06, 1.57679571105668880e-06, 2.06550563194937730e-07, 2.12083825151639860e-08,
   -6.600000e+00, 1.35851995042896180e-03, 1.35667437397324820e-03, 6.76494117783345720e-04, 2.24271830305212320e-04, 5.54567275815385010e-05, 1.08479754917149330e-05, 1.72743175318997960e-06, 2.23996251086150850e-07, 2.23935715918402050e-08,
   -6.500000e+00, 1.50118225673699170e-03, 1.49892870856904960e-03, 7.47214189103107260e-04, 2.47574664154802470e-04, 6.11478282924288150e-05, 1.19327435270117320e-05, 1.89060366781194890e-06, 2.42354351223451110e-07, 2.34811272785029050e-08,
   -6.400000e+00, 1.65880108017442430e-03, 1.65604945915083640e-03, 8.25277672943756510e-04, 2.73265743380652250e-04, 6.74064387680519320e-05, 1.31193289221308410e-05, 2.06691786360689330e-06, 2.61525130420609410e-07, 2.44147172473252130e-08,
   -6.300000e+00, 1.83293894249281000e-03, 1.82957927732590340e-03, 9.11436131557163230e-04, 3.01582519222296830e-04, 7.42854663041938640e-05, 1.44157781558079750e-05, 2.25689382743799700e-06, 2.81357968618221200e-07, 2.51229617521266410e-08,
   -6.200000e+00, 2.02532038904988360e-03, 2.02121846637158240e-03, 1.00651561821512460e-03, 3.32784420306462190e-04, 8.18419135636992360e-05, 1.58304135300522110e-05, 2.46092400072711450e-06, 3.01638062064546510e-07, 2.55161950143962320e-08,
   -6.100000e+00, 2.23784852127633740e-03, 2.23284055527215880e-03, 1.11142351870121790e-03, 3.67154515600091720e-04, 9.01369950517954420e-05, 1.73717423893457960e-05, 2.67922338831253210e-06, 3.22070306219285790e-07, 2.54826405262093800e-08,
   -6.000000e+00, 2.47262315663477430e-03, 2.46650929136004800e-03, 1.22715589769015220e-03, 4.05001213808975830e-04, 9.92362000505793020e-05, 1.90483324974313430e-05, 2.91176675437529030e-06, 3.42259858984122120e-07, 2.48839237613873960e-08,
   -5.900000e+00, 2.73196076301106390e-03, 2.72449715340043190e-03, 1.35480535737819060e-03, 4.46659974161184900e-04, 1.09209283108594010e-04, 2.08686453886229530e-05, 3.15821088901944120e-06, 3.61688823780911090e-07, 2.35498468733291440e-08,
   -5.800000e+00, 3.01841632470843230e-03, 3.00930548759916620e-03, 1.49556940698977910e-03, 4.92494995082166520e-04, 1.20130158692275140e-04, 2.28408178788692900e-05, 3.41779901612002750e-06, 3.79688433955304150e-07, 2.12723568777358350e-08,
   -5.700000e+00, 3.33480730741334730e-03, 3.32368636763577010e-03, 1.65075933023154320e-03, 5.42900836868887150e-04, 1.32076671237157130e-04, 2.49723800306182560e-05, 3.68924396212468050e-06, 3.95406070736085820e-07, 1.77986544427549490e-08,
   -5.600000e+00, 3.68423989943599590e-03, 3.67066627579939980e-03, 1.82180952274888560e-03, 5.98303921724942260e-04, 1.45130205452989270e-04, 2.72698956944831410e-05, 3.97058622553928980e-06, 4.07766414969878420e-07, 1.28234186230420520e-08,
   -5.500000e+00, 4.07013771589612770e-03, 4.05357169486976740e-03, 2.01028725239550500e-03, 6.59163838992845190e-04, 1.59375094194757500e-04, 2.97385092867708560e-05, 4.25902259388678110e-06, 4.15426030123352990e-07, 5.98016885505317660e-09,
   -5.400000e+00, 4.49627316094118690e-03, 4.47605668860338690e-03, 2.21790277074587460e-03, 7.25974364620973280e-04, 1.74897772363820470e-04, 3.23813797045459900e-05, 4.55070047506631890e-06, 4.16720717412710230e-07, -3.16814392573550440e-09,
   -5.300000e+00, 4.96680165005697430e-03, 4.94213253142596590e-03, 2.44651967370109610e-03, 7.99264081279482000e-04, 1.91785615006919500e-04, 3.51989792480765250e-05, 4.84047268317016020e-06, 4.09605096400368040e-07, -1.51391212973012190e-08,
   -5.200000e+00, 5.48629889945040880e-03, 5.45619942383629860e-03, 2.69816537102397430e-03, 8.79596458486711450e-04, 2.10125385909201780e-04, 3.81882321625762130e-05, 5.12160710363575690e-06, 3.91584076958448300e-07, -3.05411323561957880e-08,
   -5.100000e+00, 6.05980149158412760e-03, 6.02308029746672240e-03, 2.97504147776284130e-03, 9.67569219974721750e-04, 2.30001209438043780e-04, 4.13414640178133730e-05, 5.38544554286534530e-06, 3.59636242105716150e-07, -5.00832087848598940e-08,
   -5.000000e+00, 6.69285092428485540e-03, 6.64805667079015550e-03, 3.27953388316128140e-03, 1.06381278763365500e-03, 2.51491963154407420e-04, 4.46451297657817080e-05, 5.62100625972984250e-06, 3.10129709169230980e-07, -7.45837869158400600e-08,
   -4.900000e+00, 7.39154134428198290e-03, 7.33690646083775320e-03, 3.61422218297446480e-03, 1.16898754705787600e-03, 2.74667971829370260e-04, 4.80782851857432350e-05, 5.81452533962047590e-06, 2.38731847095219420e-07, -1.04977086195805230e-07,
   -4.800000e+00, 8.16257115315991750e-03, 8.09594358532951880e-03, 3.98188807709753870e-03, 1.28377962835141500e-03, 2.99586865189517000e-04, 5.16107638858811270e-05, 5.94893341840662140e-06, 1.40315385838881670e-07, -1.42315417642175880e-07,
   -4.700000e+00, 9.01329865284783080e-03, 8.93205910024240210e-03, 4.38552223386582880e-03, 1.40889483693684380e-03, 3.26288442377169070e-04, 5.52010205716218210e-05, 6.00326657261755820e-06, 8.86506405300674040e-09, -1.87765077032700900e-07,
   -4.600000e+00, 9.95180186690434320e-03, 9.85276350650622290e-03, 4.82832900299489530e-03, 1.54505030236922960e-03, 3.54788366451460950e-04, 5.87936015930568350e-05, 5.95201382230377000e-06, -1.62608851827796740e-07, -2.42592671010629160e-07,
   -4.500000e+00, 1.09869426305931800e-02, 1.08662297222252340e-02, 5.31372821854368250e-03, 1.69296333866136780e-03, 3.85070493374473870e-04, 6.23162067713295410e-05, 5.76440909701650410e-06, -3.82140937580884310e-07, -3.08137729042538070e-07,
   -4.400000e+00, 1.21284349842742580e-02, 1.19813360491064900e-02, 5.84535316925691510e-03, 1.85333692799612950e-03, 4.17077623458166190e-04, 6.56763134025455460e-05, 5.40368325106171390e-06, -6.58851848354854590e-07, -3.85766258848212360e-07,
   -4.300000e+00, 1.33869178276648150e-02, 1.32077082587401650e-02, 6.42704362421855830e-03, 2.02684115234203410e-03, 4.50700451530005010e-04, 6.87573457418929530e-05, 4.82630245069899450e-06, -1.00294258400733050e-06, -4.76798514972935810e-07,
   -4.200000e+00, 1.47740316932730670e-02, 1.45557596807992310e-02, 7.06283258555582150e-03, 2.21408980691469110e-03, 4.85764488348583330e-04, 7.14143932084044670e-05, 3.98123376869764830e-06, -1.42561795102167810e-06, -5.82402709583483840e-07,
   -4.100000e+00, 1.63024993714409730e-02, 1.60367278856851400e-02, 7.75692519656621770e-03, 2.41561133966801180e-03, 5.22014734440237690e-04, 7.34695105422523110e-05, 2.80929796912428230e-06, -1.93890813171397730e-06, -7.03444793457825520e-07,
   -4.000000e+00, 1.79862099620915590e-02, 1.76627062132911180e-02, 8.51366796419456530e-03, 2.63181317810482020e-03, 5.59097914300470230e-04, 7.47066762395398010e-05, 1.24269413614214320e-06, -2.55534876234905630e-06, -8.40282944541781960e-07,
   -3.900000e+00, 1.98403057340775340e-02, 1.94466680024558660e-02, 9.33750616255010590e-03, 2.86293843734490310e-03, 5.96542131231844440e-04, 7.48665453514710110e-05, -7.95188183514866580e-07, -3.28747035475222720e-06, -9.92494306357329020e-07,
   -3.800000e+00, 2.18812709361305350e-02, 2.14024809183501860e-02, 1.02329269754953080e-02, 3.10901396359802030e-03, 6.33733890292699740e-04, 7.36412131670986460e-05, -3.38951570114013690e-06, -4.14703775227703650e-06, -1.15852126130839660e-06,
   -3.700000e+00, 2.41270214176692170e-02, 2.35449082551805470e-02, 1.12043856218404760e-02, 3.36978867111855560e-03, 6.69892570298141560e-04, 7.06693116302656430e-05, -6.63311834132034660e-06, -5.14397050514008850e-06, -1.33522575415530200e-06,
   -3.600000e+00, 2.65969935768659090e-02, 2.58895935095380630e-02, 1.22562114024881790e-02, 3.64466119950056150e-03, 7.04042619029963780e-04, 6.55318946370026160e-05, -1.06247064779467550e-05, -6.28486711652866830e-06, -1.51734380667807970e-06,
   -3.500000e+00, 2.93122307513563190e-02, 2.84530238797355630e-02, 1.33924903383311200e-02, 3.93259607872179120e-03, 7.34984014121930130e-04, 5.77497348688372690e-05, -1.54658929379206650e-05, -7.57105242822983820e-06, -1.69683959383878260e-06,
   -3.400000e+00, 3.22954646984505500e-02, 3.12524676583616910e-02, 1.46169208631807560e-02, 4.23202787499000420e-03, 7.61261892057007530e-04, 4.67828549658585340e-05, -2.12566585862240430e-05, -8.99607145360533280e-06, -1.86217080497492950e-06,
   -3.300000e+00, 3.55711892726362710e-02, 3.43058797663665580e-02, 1.59326389408495500e-02, 4.54075324118336040e-03, 7.81136715871748040e-04, 3.20333477669124930e-05, -2.80888515200180770e-05, -1.05425695254941560e-05, -1.99749627553839710e-06,
   -3.200000e+00, 3.91657227967643840e-02, 3.76317689545713970e-02, 1.73420090460590720e-02, 4.85581145777832200e-03, 7.92556942875540480e-04, 1.28527968678336550e-05, -3.60372788765848710e-05, -1.21785339929208390e-05, -2.08188494032338780e-06,
   -3.100000e+00, 4.31072549410862140e-02, 4.12490195125304120e-02, 1.88463777560687160e-02, 5.17335497467662050e-03, 7.93136875871795400e-04, -1.14441252533549670e-05, -4.51479527965798000e-05, -1.38529346180268090e-05, -2.08862370841833370e-06,
   -3.000000e+00, 4.74258731775667810e-02, 4.51766597309121300e-02, 2.04457873304717400e-02, 5.48851270404273740e-03, 7.80143233713328820e-04, -4.15580551386109190e-05, -5.54231098116953620e-05, -1.54908971165042420e-05, -1.98477178874566190e-06,
   -2.900000e+00, 5.21535630784178070e-02, 4.94335689366433020e-02, 2.21386477125931070e-02, 5.79525041829335050e-03, 7.50494941518227080e-04, -7.81733147369432390e-05, -6.68027637707623900e-05, -1.69886858896900930e-05, -1.73116951598277200e-06,
   -2.800000e+00, 5.73241758988689010e-02, 5.40381147563844390e-02, 2.39213669826539750e-02, 6.08623461297323230e-03, 7.00781674416256600e-04, -1.21914175348937420e-04, -7.91428101009804330e-05, -1.82089651359147730e-05, -1.28317708681257900e-06,
   -2.700000e+00, 6.29733560569967350e-02, 5.90077124839154530e-02, 2.57879425535992270e-02, 6.35270861473481020e-03, 6.27307713045423940e-04, -1.73287043781771690e-04, -9.21901110565383650e-05, -1.89770546921192300e-05, -5.92484434840283480e-07,
   -2.600000e+00, 6.91384203433467320e-02, 6.43582991757734310e-02, 2.77295184468892250e-02, 6.58439252316387910e-03, 5.26168559702410540e-04, -2.32607033498257920e-04, -1.05555593802384430e-04, -1.90791914477173410e-05, 3.89617153928183530e-07,
   -2.500000e+00, 7.58581800212435460e-02, 7.01037165451081630e-02, 2.97339179227170330e-02, 6.76942168408115490e-03, 3.93368339730086680e-04, -2.99908237158971970e-04, -1.18687212286738230e-04, -1.82641271644429910e-05, 1.70710132276501400e-06,
   -2.400000e+00, 8.31726964939224910e-02, 7.62549990518523320e-02, 3.17851656336421080e-02, 6.89434162824405490e-03, 2.24986027547160510e-04, -3.74838428925960460e-04, -1.30845663828534170e-04, -1.62496950231642210e-05, 3.39187260641409500e-06,
   -2.300000e+00, 9.11229610148563820e-02, 8.28195669907413470e-02, 3.38630193212060630e-02, 6.94418048858966380e-03, 1.73976795857779640e-05, -4.56541019292197210e-04, -1.41086975496148230e-04, -1.27361870108443370e-05, 5.45406812893943450e-06,
   -2.200000e+00, 9.97504891196855230e-02, 8.98003290400690300e-02, 3.59425377751789250e-02, 6.90262241097350580e-03, -2.32440237481267520e-04, -5.43529946386255560e-04, -1.48257384045381540e-04, -7.42839527943911650e-06, 7.87016077876329300e-06,
   -2.100000e+00, 1.09096821195612790e-01, 9.71947048006252810e-02, 3.79937190695184560e-02, 6.75230682549018880e-03, -5.26645053446529170e-04, -6.33566815394946450e-04, -1.51007140072426080e-04, -6.78431444195360410e-08, 1.05695507215487870e-05,
   -2.000000e+00, 1.19202922022117550e-01, 1.04993585403506520e-01, 3.99812505280765300e-02, 6.47527792470100220e-03, -8.66003964518882250e-04, -7.23553848759598890e-04, -1.47830679701737250e-04, 9.52409466015652970e-06, 1.34210561408648290e-05,
   -1.900000e+00, 1.30108474362998020e-01, 1.13180259261931110e-01, 4.18644187703871230e-02, 6.05360545705724200e-03, -1.24952420608354250e-03, -8.09460731799582210e-04, -1.37140621784980900e-04, 2.13929590802673880e-05, 1.62216037519927020e-05,
   -1.800000e+00, 1.41851064900488130e-01, 1.21729340287085640e-01, 4.35972335941858320e-02, 5.47019109445184840e-03, -1.67395968424672270e-03, -8.86307608006958500e-04, -1.17381772299136110e-04, 3.53908570690076410e-05, 1.86903606637620420e-05,
   -1.700000e+00, 1.54465265083535250e-01, 1.30605746966208440e-01, 4.51288221565357000e-02, 4.70976335376680800e-03, -2.13334834774821410e-03, -9.48229364517824110e-04, -8.71882334678011690e-05, 5.11149154480680300e-05, 2.04723258355813250e-05,
   -1.600000e+00, 1.67981614866075320e-01, 1.39763791933060850e-01, 4.64041484978087170e-02, 3.76004778666888180e-03, -2.61860738066120290e-03, -9.88646731697842430e-04, -4.55814033575266970e-05, 6.78538337828735570e-05, 2.11556871820008980e-05,
   -1.500000e+00, 1.82425523806356350e-01, 1.49146452070332860e-01, 4.73651063923763370e-02, 2.61307784655404570e-03, -3.11724537099207930e-03, -1.00056620297502250e-03, 7.80102976522912730e-06, 8.45551079739370080e-05, 2.03066444225203900e-05,
   -1.400000e+00, 1.97816111441418470e-01, 1.58684897495614760e-01, 4.79520193807447880e-02, 1.26658622274202760e-03, -3.61325966507915380e-03, -9.77022016894534080e-04, 7.24646867350921490e-05, 9.98277933086790340e-05, 1.75234957455372100e-05,
   -1.300000e+00, 2.14165016957441860e-01, 1.68298362469060490e-01, 4.81055595824342110e-02, -2.74611731590522550e-04, -4.08729027151198170e-03, -9.11658508830290490e-04, 1.46836905411713910e-04, 1.11995839548817760e-04, 1.25083357559633170e-05,
   -1.200000e+00, 2.31475216500983100e-01, 1.77894440646806100e-01, 4.77690661603623200e-02, -1.99735857190566940e-03, -4.51709579145440160e-03, -7.99430009776354920e-04, 2.28103804212779630e-04, 1.19213830328363190e-04, 5.14980858689414560e-06,
   -1.100000e+00, 2.49739894404882120e-01, 1.87369879547520460e-01, 4.68912058409069880e-02, -3.87915850373224750e-03, -4.87839910350648060e-03, -6.37369522318305510e-04, 3.12151769446088540e-04, 1.19649447598686270e-04, -4.39535348760823720e-06,
   -1.000000e+00, 2.68941421369995100e-01, 1.96611933241481850e-01, 4.54288738364742040e-02, -5.88759675270594200e-03, -5.14611922359971710e-03, -4.25349801037941650e-04, 3.93658206291141850e-04, 1.11724987283642060e-04, -1.56394991495607590e-05,
   -9.000000e-01, 2.89050497374996330e-01, 2.05500307342263570e-01, 4.33501876231358880e-02, -7.98032509405418970e-03, -5.29596124463789070e-03, -1.66736205775554510e-04, 4.66365777962912890e-04, 9.43948827803063110e-05, -2.77403009365558620e-05,
   -8.000000e-01, 3.10025518872388160e-01, 2.13909696520294650e-01, 4.06373836046078910e-02, -1.01057421786887830e-02, -5.30628176052314740e-03, 1.31183858107734580e-04, 5.23552542648552870e-04, 6.74201480805606460e-05, -3.95349355063028740e-05,
   -7.000000e-01, 3.31812227831834840e-01, 2.21712873293109370e-01, 3.72893942201707460e-02, -1.22044526350348090e-02, -5.16008921756595880e-03, 4.57108356135274710e-04, 5.58678296160800050e-04, 3.15880914084614440e-05, -4.96436994272190620e-05,
   -6.000000e-01, 3.54343693774204220e-01, 2.28784240456657210e-01, 3.33238673875909590e-02, -1.42115219385533420e-02, -4.84698674038245340e-03, 7.96055199473061920e-04, 5.66149170755054900e-04, -1.11785776363181220e-05, -5.66412336602477350e-05,
   -5.000000e-01, 3.77540668798145410e-01, 2.35003712201594490e-01, 2.87783974261603720e-02, -1.60594593815974360e-02, -4.36483044084713490e-03, 1.13016810992858680e-03, 5.42105497383336210e-04, -5.78708906527865280e-05, -5.92762197634034040e-05,
   -4.000000e-01, 4.01312339887548330e-01, 2.40260745741529220e-01, 2.37107708141042000e-02, -1.76817683206875460e-02, -3.72086991006114400e-03, 1.43997222858043460e-03, 4.85111608501311120e-04, -1.04626298216035280e-04, -5.67032531438075200e-05,
   -3.000000e-01, 4.25557483188341750e-01, 2.44458311690745970e-01, 1.81980919777879580e-02, -1.90168142062322460e-02, -2.93216717606728910e-03, 1.70598204579085260e-03, 3.96619989543374560e-04, -1.47135961074911260e-04, -4.86790802182835940e-05,
   -2.000000e-01, 4.50166002687523160e-01, 2.47516572711860060e-01, 1.23347402193163100e-02, -2.00116916483821620e-02, -2.02515760609995070e-03, 1.91049228965414760e-03, 2.81102185107734400e-04, -1.81178333294567830e-04, -3.56746364126751340e-05,
   -1.000000e-01, 4.75020812521059650e-01, 2.49376040192891950e-01, 6.22921086073401340e-03, -2.06257360568048690e-02, -1.03431503291523660e-03, 2.03933060032661290e-03, 1.45785422557720970e-04, -2.03196033102260650e-04, -1.88669980984517900e-05,
   0.000000e+00, 5.00000000000000000e-01, 2.50000000000000000e-01, 0.00000000000000000e+00, -2.08333333333333320e-02, 0.00000000000000000e+00, 2.08333333333333330e-03, 0.00000000000000000e+00, -2.10813492063492070e-04, 0.00000000000000000e+00,
   1.000000e-01, 5.24979187478940350e-01, 2.49376040192891950e-01, -6.22921086073400900e-03, -2.06257360568048690e-02, 1.03431503291523590e-03, 2.03933060032661290e-03, -1.45785422557720860e-04, -2.03196033102260650e-04, 1.88669980984517760e-05,
   2.000000e-01, 5.49833997312478620e-01, 2.47516572711859870e-01, -1.23347402193167460e-02, -2.00116916483821030e-02, 2.02515760610002010e-03, 1.91049228965413550e-03, -2.81102185107743620e-04, -1.81178333294565770e-04, 3.56746364126762310e-05,
   3.000000e-01, 5.74442516811660030e-01, 2.44458311690745720e-01, -1.81980919777883710e-02, -1.90168142062321630e-02, 2.93216717606735060e-03, 1.70598204579083550e-03, -3.96619989543381990e-04, -1.47135961074908440e-04, 4.86790802182843600e-05,
   4.000000e-01, 5.98687660112451670e-01, 2.40260745741529220e-01, -2.37107708141042070e-02, -1.76817683206875460e-02, 3.72086991006114480e-03, 1.43997222858043440e-03, -4.85111608501311170e-04, -1.04626298216035220e-04, 5.67032531438075270e-05,
   5.000000e-01, 6.22459331201854590e-01, 2.35003712201594490e-01, -2.87783974261603720e-02, -1.60594593815974360e-02, 4.36483044084713490e-03, 1.13016810992858680e-03, -5.42105497383336100e-04, -5.78708906527865420e-05, 5.92762197634034040e-05,
   6.000000e-01, 6.45656306225795840e-01, 2.28784240456657210e-01, -3.33238673875909590e-02, -1.42115219385533400e-02, 4.84698674038245340e-03, 7.96055199473061600e-04, -5.66149170755055010e-04, -1.11785776363180860e-05, 5.66412336602477280e-05,
   7.000000e-01, 6.68187772168166720e-01, 2.21712873293108850e-01, -3.72893942201710100e-02, -1.22044526350346600e-02, 5.16008921756597530e-03, 4.57108356135250480e-04, -5.58678296160798530e-04, 3.15880914084643170e-05, 4.96436994272184320e-05,
   8.000000e-01, 6.89974481127613390e-01, 2.13909696520294070e-01, -4.06373836046081060e-02, -1.01057421786886320e-02, 5.30628176052315180e-03, 1.31183858107712160e-04, -5.23552542648549510e-04, 6.74201480805629090e-05, 3.95349355063020810e-05,
   9.000000e-01, 7.10949502625003670e-01, 2.05500307342263570e-01, -4.33501876231358880e-02, -7.98032509405418970e-03, 5.29596124463789070e-03, -1.66736205775554540e-04, -4.66365777962912890e-04, 9.43948827803063110e-05, 2.77403009365558620e-05,
   1.000000e+00, 7.31058578630004900e-01, 1.96611933241481850e-01, -4.54288738364742040e-02, -5.88759675270594200e-03, 5.14611922359971710e-03, -4.25349801037941650e-04, -3.93658206291141790e-04, 1.11724987283642060e-04, 1.56394991495607550e-05,
   1.100000e+00, 7.50260105595117910e-01, 1.87369879547520460e-01, -4.68912058409069810e-02, -3.87915850373224920e-03, 4.87839910350648060e-03, -6.37369522318305400e-04, -3.12151769446088590e-04, 1.19649447598686270e-04, 4.39535348760824660e-06,
   1.200000e+00, 7.68524783499018090e-01, 1.77894440646805430e-01, -4.77690661603623610e-02, -1.99735857190554190e-03, 4.51709579145437390e-03, -7.99430009776364570e-04, -2.28103804212773750e-04, 1.19213830328362910e-04, -5.14980858689474360e-06,
   1.300000e+00, 7.85834983042559390e-01, 1.68298362469059830e-01, -4.81055595824342180e-02, -2.74611731590405520e-04, 4.08729027151194960e-03, -9.11658508830296780e-04, -1.46836905411708300e-04, 1.11995839548817050e-04, -1.25083357559637600e-05,
   1.400000e+00, 8.02183888558581470e-01, 1.58684897495614790e-01, -4.79520193807447880e-02, 1.26658622274202650e-03, 3.61325966507915430e-03, -9.77022016894534080e-04, -7.24646867350922030e-05, 9.98277933086790470e-05, -1.75234957455372060e-05,
   1.500000e+00, 8.17574476193643650e-01, 1.49146452070332860e-01, -4.73651063923763370e-02, 2.61307784655404700e-03, 3.11724537099207850e-03, -1.00056620297502270e-03, -7.80102976522905780e-06, 8.45551079739369940e-05, -2.03066444225203900e-05,
   1.600000e+00, 8.32018385133924680e-01, 1.39763791933060880e-01, -4.64041484978087170e-02, 3.76004778666888090e-03, 2.61860738066120330e-03, -9.88646731697842430e-04, 4.55814033575266490e-05, 6.78538337828735700e-05, -2.11556871820008980e-05,
   1.700000e+00, 8.45534734916465690e-01, 1.30605746966207800e-01, -4.51288221565356030e-02, 4.70976335376686790e-03, 2.13334834774818110e-03, -9.48229364517820420e-04, 8.71882334678036630e-05, 5.11149154480668850e-05, -2.04723258355812330e-05,
   1.800000e+00, 8.58148935099512730e-01, 1.21729340287085030e-01, -4.35972335941857140e-02, 5.47019109445189610e-03, 1.67395968424669120e-03, -8.86307608006953510e-04, 1.17381772299137870e-04, 3.53908570690065840e-05, -1.86903606637618860e-05,
   1.900000e+00, 8.69891525637002010e-01, 1.13180259261931090e-01, -4.18644187703871160e-02, 6.05360545705724280e-03, 1.24952420608354210e-03, -8.09460731799582210e-04, 1.37140621784980930e-04, 2.13929590802673740e-05, -1.62216037519927020e-05,
   2.000000e+00, 8.80797077977882430e-01, 1.04993585403506520e-01, -3.99812505280765300e-02, 6.47527792470100220e-03, 8.66003964518882580e-04, -7.23553848759599000e-04, 1.47830679701737250e-04, 9.52409466015653990e-06, -1.34210561408648320e-05,
   2.100000e+00, 8.90903178804387190e-01, 9.71947048006252950e-02, -3.79937190695184560e-02, 6.75230682549018880e-03, 5.26645053446529490e-04, -6.33566815394946450e-04, 1.51007140072426080e-04, -6.78431444195273990e-08, -1.05695507215487880e-05,
   2.200000e+00, 9.00249510880315130e-01, 8.98003290400685160e-02, -3.59425377751787790e-02, 6.90262241097351280e-03, 2.32440237481248250e-04, -5.43529946386249270e-04, 1.48257384045381190e-04, -7.42839527943956290e-06, -7.87016077876311010e-06,
   2.300000e+00, 9.08877038985144270e-01, 8.28195669907408480e-02, -3.38630193212059100e-02, 6.94418048858966300e-03, -1.73976795857944340e-05, -4.56541019292191090e-04, 1.41086975496147570e-04, -1.27361870108446520e-05, -5.45406812893927270e-06,
   2.400000e+00, 9.16827303506077550e-01, 7.62549990518523320e-02, -3.17851656336421080e-02, 6.89434162824405490e-03, -2.24986027547160530e-04, -3.74838428925960460e-04, 1.30845663828534150e-04, -1.62496950231642210e-05, -3.39187260641409450e-06,
   2.500000e+00, 9.24141819978756440e-01, 7.01037165451081630e-02, -2.97339179227170370e-02, 6.76942168408115490e-03, -3.93368339730086620e-04, -2.99908237158971970e-04, 1.18687212286738230e-04, -1.82641271644429910e-05, -1.70710132276501460e-06,
   2.600000e+00, 9.30861579656653280e-01, 6.43582991757734310e-02, -2.77295184468892250e-02, 6.58439252316387910e-03, -5.26168559702410540e-04, -2.32607033498257950e-04, 1.05555593802384450e-04, -1.90791914477173410e-05, -3.89617153928184060e-07,
   2.700000e+00, 9.37026643943003720e-01, 5.90077124839150850e-02, -2.57879425535990910e-02, 6.35270861473479200e-03, -6.27307713045430010e-04, -1.73287043781767810e-04, 9.21901110565374300e-05, -1.89770546921191960e-05, 5.92484434840341820e-07,
   2.800000e+00, 9.42675824101131490e-01, 5.40381147563840990e-02, -2.39213669826538470e-02, 6.08623461297321240e-03, -7.00781674416260940e-04, -1.21914175348934030e-04, 7.91428101009795250e-05, -1.82089651359146990e-05, 1.28317708681261880e-06,
   2.900000e+00, 9.47846436921582210e-01, 4.94335689366433020e-02, -2.21386477125931100e-02, 5.79525041829335050e-03, -7.50494941518227080e-04, -7.81733147369432530e-05, 6.68027637707623900e-05, -1.69886858896900930e-05, 1.73116951598277200e-06,
   3.000000e+00, 9.52574126822433250e-01, 4.51766597309121370e-02, -2.04457873304717400e-02, 5.48851270404273740e-03, -7.80143233713328820e-04, -4.15580551386109330e-05, 5.54231098116953620e-05, -1.54908971165042420e-05, 1.98477178874566190e-06,
   3.100000e+00, 9.56892745058913970e-01, 4.12490195125302800e-02, -1.88463777560686640e-02, 5.17335497467660920e-03, -7.93136875871795610e-04, -1.14441252533540270e-05, 4.51479527965794610e-05, -1.38529346180267520e-05, 2.08862370841833540e-06,
   3.200000e+00, 9.60834277203235780e-01, 3.76317689545712720e-02, -1.73420090460590200e-02, 4.85581145777831070e-03, -7.92556942875540260e-04, 1.28527968678344200e-05, 3.60372788765845660e-05, -1.21785339929207800e-05, 2.08188494032338610e-06,
   3.300000e+00, 9.64428810727363970e-01, 3.43058797663663290e-02, -1.59326389408494530e-02, 4.54075324118333870e-03, -7.81136715871746850e-04, 3.20333477669136790e-05, 2.80888515200175560e-05, -1.05425695254940430e-05, 1.99749627553838900e-06,
   3.400000e+00, 9.67704535301549650e-01, 3.12524676583614820e-02, -1.46169208631806680e-02, 4.23202787498998340e-03, -7.61261892057005900e-04, 4.67828549658594210e-05, 2.12566585862236030e-05, -8.99607145360522770e-06, 1.86217080497491890e-06,
   3.500000e+00, 9.70687769248643640e-01, 2.84530238797355600e-02, -1.33924903383311200e-02, 3.93259607872179120e-03, -7.34984014121930130e-04, 5.77497348688372760e-05, 1.54658929379206620e-05, -7.57105242822983730e-06, 1.69683959383878260e-06,
   3.600000e+00, 9.73403006423134150e-01, 2.58895935095379770e-02, -1.22562114024881400e-02, 3.64466119950055150e-03, -7.04042619029962590e-04, 6.55318946370028460e-05, 1.06247064779465970e-05, -6.28486711652862430e-06, 1.51734380667807310e-06,
   3.700000e+00, 9.75872978582330910e-01, 2.35449082551804700e-02, -1.12043856218404410e-02, 3.36978867111854600e-03, -6.69892570298140370e-04, 7.06693116302657790e-05, 6.63311834132022210e-06, -5.14397050514005120e-06, 1.33522575415529580e-06,
   3.800000e+00, 9.78118729063869650e-01, 2.14024809183500400e-02, -1.02329269754952420e-02, 3.10901396359800210e-03, -6.33733890292697130e-04, 7.36412131670987950e-05, 3.38951570113992980e-06, -4.14703775227697050e-06, 1.15852126130838430e-06,
   3.900000e+00, 9.80159694265922640e-01, 1.94466680024557300e-02, -9.33750616255004350e-03, 2.86293843734488620e-03, -5.96542131231841840e-04, 7.48665453514710520e-05, 7.95188183514702470e-07, -3.28747035475217040e-06, 9.92494306357317580e-07,
   4.000000e+00, 9.82013790037908450e-01, 1.76627062132911140e-02, -8.51366796419456350e-03, 2.63181317810481970e-03, -5.59097914300470120e-04, 7.47066762395398010e-05, -1.24269413614214560e-06, -2.55534876234905550e-06, 8.40282944541781740e-07,
   4.100000e+00, 9.83697500628559100e-01, 1.60367278856850850e-02, -7.75692519656619260e-03, 2.41561133966800450e-03, -5.22014734440236390e-04, 7.34695105422522440e-05, -2.80929796912433010e-06, -1.93890813171395740e-06, 7.03444793457820960e-07,
   4.200000e+00, 9.85225968306726930e-01, 1.45557596807991810e-02, -7.06283258555579810e-03, 2.21408980691468420e-03, -4.85764488348582080e-04, 7.14143932084043720e-05, -3.98123376869768300e-06, -1.42561795102166160e-06, 5.82402709583479820e-07,
   4.300000e+00, 9.86613082172335230e-01, 1.32077082587400730e-02, -6.42704362421851400e-03, 2.02684115234202150e-03, -4.50700451530002570e-04, 6.87573457418927490e-05, -4.82630245069904450e-06, -1.00294258400730340e-06, 4.76798514972928770e-07,
   4.400000e+00, 9.87871565015725820e-01, 1.19813360491064070e-02, -5.84535316925687610e-03, 1.85333692799611760e-03, -4.17077623458163860e-04, 6.56763134025453160e-05, -5.40368325106174610e-06, -6.58851848354832670e-07, 3.85766258848206380e-07,
   4.500000e+00, 9.89013057369406810e-01, 1.08662297222252340e-02, -5.31372821854368160e-03, 1.69296333866136780e-03, -3.85070493374473810e-04, 6.23162067713295410e-05, -5.76440909701650410e-06, -3.82140937580884150e-07, 3.08137729042538020e-07,
   4.600000e+00, 9.90048198133095750e-01, 9.85276350650618640e-03, -4.82832900299487880e-03, 1.54505030236922430e-03, -3.54788366451459870e-04, 5.87936015930567060e-05, -5.95201382230377430e-06, -1.62608851827789640e-07, 2.42592671010626930e-07,
   4.700000e+00, 9.90986701347152170e-01, 8.93205910024237080e-03, -4.38552223386581410e-03, 1.40889483693683920e-03, -3.26288442377168090e-04, 5.52010205716216920e-05, -6.00326657261755740e-06, 8.86506405301201780e-09, 1.87765077032699150e-07,
   4.800000e+00, 9.91837428846840120e-01, 8.09594358532946330e-03, -3.98188807709751180e-03, 1.28377962835140660e-03, -2.99586865189515150e-04, 5.16107638858808760e-05, -5.94893341840661460e-06, 1.40315385838889720e-07, 1.42315417642172960e-07,
   4.900000e+00, 9.92608458655718120e-01, 7.33690646083770200e-03, -3.61422218297444010e-03, 1.16898754705786820e-03, -2.74667971829368530e-04, 4.80782851857429910e-05, -5.81452533962046400e-06, 2.38731847095225350e-07, 1.04977086195802830e-07,
   5.000000e+00, 9.93307149075715160e-01, 6.64805667079015460e-03, -3.27953388316128140e-03, 1.06381278763365480e-03, -2.51491963154407420e-04, 4.46451297657817080e-05, -5.62100625972984250e-06, 3.10129709169231030e-07, 7.45837869158400470e-08,
   5.100000e+00, 9.93940198508415860e-01, 6.02308029746670070e-03, -2.97504147776283090e-03, 9.67569219974718500e-04, -2.30001209438043020e-04, 4.13414640178132570e-05, -5.38544554286533600e-06, 3.59636242105717580e-07, 5.00832087848591070e-08,
   5.200000e+00, 9.94513701100549600e-01, 5.45619942383627960e-03, -2.69816537102396480e-03, 8.79596458486708520e-04, -2.10125385909201100e-04, 3.81882321625761050e-05, -5.12160710363574680e-06, 3.91584076958449200e-07, 3.05411323561951730e-08,
   5.300000e+00, 9.95033198349943080e-01, 4.94213253142593120e-03, -2.44651967370107920e-03, 7.99264081279476690e-04, -1.91785615006918260e-04, 3.51989792480763220e-05, -4.84047268317013990e-06, 4.09605096400368890e-07, 1.51391212973002660e-08,
   5.400000e+00, 9.95503726839058860e-01, 4.47605668860335490e-03, -2.21790277074585940e-03, 7.25974364620968290e-04, -1.74897772363819330e-04, 3.23813797045458010e-05, -4.55070047506629860e-06, 4.16720717412710390e-07, 3.16814392573476410e-09,
   5.500000e+00, 9.95929862284103850e-01, 4.05357169486976650e-03, -2.01028725239550500e-03, 6.59163838992845080e-04, -1.59375094194757470e-04, 2.97385092867708520e-05, -4.25902259388678110e-06, 4.15426030123352990e-07, -5.98016885505318570e-09,
   5.600000e+00, 9.96315760100564000e-01, 3.67066627579938630e-03, -1.82180952274887910e-03, 5.98303921724940090e-04, -1.45130205452988760e-04, 2.72698956944830560e-05, -3.97058622553927970e-06, 4.07766414969878050e-07, -1.28234186230422650e-08,
   5.700000e+00, 9.96665192692586690e-01, 3.32368636763575840e-03, -1.65075933023153730e-03, 5.42900836868885190e-04, -1.32076671237156660e-04, 2.49723800306181780e-05, -3.68924396212467030e-06, 3.95406070736085290e-07, -1.77986544427550980e-08,
   5.800000e+00, 9.96981583675291550e-01, 3.00930548759914500e-03, -1.49556940698976870e-03, 4.92494995082163160e-04, -1.20130158692274330e-04, 2.28408178788691450e-05, -3.41779901612000880e-06, 3.79688433955302940e-07, -2.12723568777360370e-08,
   5.900000e+00, 9.97268039236988920e-01, 2.72449715340041280e-03, -1.35480535737818120e-03, 4.46659974161181810e-04, -1.09209283108593270e-04, 2.08686453886228180e-05, -3.15821088901942300e-06, 3.61688823780909770e-07, -2.35498468733292700e-08,
   6.000000e+00, 9.97527376843365230e-01, 2.46650929136004800e-03, -1.22715589769015220e-03, 4.05001213808975830e-04, -9.92362000505793020e-05, 1.90483324974313430e-05, -2.91176675437529030e-06, 3.42259858984122120e-07, -2.48839237613873960e-08,
   6.100000e+00, 9.97762151478723710e-01, 2.23284055527215100e-03, -1.11142351870121400e-03, 3.67154515600090470e-04, -9.01369950517951440e-05, 1.73717423893457380e-05, -2.67922338831252450e-06, 3.22070306219285100e-07, -2.54826405262093900e-08,
   6.200000e+00, 9.97974679610950100e-01, 2.02121846637157500e-03, -1.00651561821512090e-03, 3.32784420306461000e-04, -8.18419135636989510e-05, 1.58304135300521570e-05, -2.46092400072710690e-06, 3.01638062064545770e-07, -2.55161950143962260e-08,
   6.300000e+00, 9.98167061057507190e-01, 1.82957927732589060e-03, -9.11436131557156830e-04, 3.01582519222294770e-04, -7.42854663041933490e-05, 1.44157781558078800e-05, -2.25689382743798350e-06, 2.81357968618219770e-07, -2.51229617521266020e-08,
   6.400000e+00, 9.98341198919825530e-01, 1.65604945915082470e-03, -8.25277672943750660e-04, 2.73265743380650350e-04, -6.74064387680514710e-05, 1.31193289221307530e-05, -2.06691786360688020e-06, 2.61525130420608030e-07, -2.44147172473251530e-08,
   6.500000e+00, 9.98498817743262990e-01, 1.49892870856904940e-03, -7.47214189103107150e-04, 2.47574664154802410e-04, -6.11478282924288020e-05, 1.19327435270117310e-05, -1.89060366781194860e-06, 2.42354351223451110e-07, -2.34811272785029050e-08,
   6.600000e+00, 9.98641480049571100e-01, 1.35667437397324320e-03, -6.76494117783343330e-04, 2.24271830305211500e-04, -5.54567275815383040e-05, 1.08479754917148960e-05, -1.72743175318997390e-06, 2.23996251086150210e-07, -2.23935715918401650e-08,
   6.700000e+00, 9.98770601378722530e-01, 1.22788720030741820e-03, -6.12434037322566940e-04, 2.03140159741224220e-04, -5.02841698614196030e-05, 9.85731788582393180e-06, -1.57679571105668350e-06, 2.06550563194937130e-07, -2.12083825151639430e-08,
   6.800000e+00, 9.98887463967139680e-01, 1.11129829643590400e-03, -5.54412788819910760e-04, 1.83981398835655990e-04, -4.55849477472547160e-05, 8.95344559655879560e-06, -1.43803388408028780e-06, 1.90077049463261540e-07, -1.99695713410904680e-08,
   6.900000e+00, 9.98993229179914400e-01, 1.00575723260145560e-03, -5.01866049266854600e-04, 1.66614657822645790e-04, -4.13174153633906250e-05, 8.12944074027207590e-06, -1.31045411900319600e-06, 1.74604416741991380e-07, -1.87111114252562830e-08,
   7.000000e+00, 9.99088948805599400e-01, 9.10221180121826540e-04, -4.54281331967594540e-04, 1.50875027423562060e-04, -3.74432811738753480e-05, 7.37880497244096900e-06, -1.19335288038025050e-06, 1.60137562702363960e-07, -1.74588408352964750e-08,
   7.100000e+00, 9.99175575313601680e-01, 8.23745010334750930e-04, -4.11193389445558120e-04, 1.36612279213740410e-04, -3.39273972842914250e-05, 6.69546174926586470e-06, -1.08602979344121380e-06, 1.46663432811572760e-07, -1.62320407865954220e-08,
   7.200000e+00, 9.99253971166163320e-01, 7.45472274815779240e-04, -3.72179993596051240e-04, 1.23689650223444190e-04, -3.07375495998706260e-05, 6.07375100916500350e-06, -9.87798502316527490e-07, 1.34155727489317400e-07, -1.50447394841792490e-08,
   7.300000e+00, 9.99324917269367120e-01, 6.74626993939636430e-04, -3.36858067936590810e-04, 1.11982710742320710e-04, -2.78442521156261930e-05, 5.50841825822966410e-06, -8.97994575796458760e-07, 1.22578661398167780e-07, -1.39067846545079820e-08,
   7.400000e+00, 9.99389120640565620e-01, 6.10506185842614760e-04, -3.04880147293569100e-04, 1.01378313170817010e-04, -2.52205477252674230e-05, 4.99459964447470730e-06, -8.15981062985799730e-07, 1.11889944579739760e-07, -1.28247221712007600e-08,
   7.500000e+00, 9.99447221363076400e-01, 5.52473072702160420e-04, -2.75931141039014920e-04, 9.17736189542991000e-05, -2.28418172279072060e-05, 4.52780427792291130e-06, -7.41152192379702000e-07, 1.02043127397168810e-07, -1.18025128443360290e-08,
   7.600000e+00, 9.99499798892920400e-01, 4.99950905932039760e-04, -2.49725376969387250e-04, 8.30752000803310330e-05, -2.06855976523322050e-05, 4.10389478553116230e-06, -6.72935617035874460e-07, 9.29894275301906520e-08, -1.08421146857547030e-08,
   7.700000e+00, 9.99547377776759500e-01, 4.52417356363562560e-04, -2.26003904032111420e-04, 7.51982112629214300e-05, -1.87314105805525660e-05, 3.71906687172526550e-06, -6.10793532989768880e-07, 8.46791371308547520e-08, -9.94395377704565340e-09,
   7.800000e+00, 9.99590432835013920e-01, 4.09399419723413860e-04, -2.04532033302023900e-04, 6.80656287356991040e-05, -1.69606008127532920e-05, 3.36982847838178880e-06, -5.54222935506773140e-07, 7.70626912387012800e-08, -9.10730322364054310e-09,
   7.900000e+00, 9.99629393859373660e-01, 3.70468791714924570e-04, -1.85097097848342280e-04, 6.16075514935193710e-05, -1.53561854558387080e-05, 3.05297899532582600e-06, -5.02755226147291380e-07, 7.00914642564754660e-08, -8.33058653549456410e-09,
   8.000000e+00, 9.99664649869533520e-01, 3.35237670756474240e-04, -1.67506413381611650e-04, 5.57605608301848110e-05, -1.39027133219421480e-05, 2.76558885777998050e-06, -4.55955341173887610e-07, 6.37183493199892450e-08, -7.61161908358632620e-09,
   8.100000e+00, 9.99696552969971060e-01, 3.03354949928885320e-04, -1.51585422805842160e-04, 5.04671340958345240e-05, -1.25861343788082970e-05, 2.50497977566681800e-06, -4.13420537045400910e-07, 5.78981654174098590e-08, -6.94779899154796610e-09,
   8.200000e+00, 9.99725421843898680e-01, 2.74502762937523960e-04, -1.37176009006269880e-04, 4.56751087227269870e-05, -1.13936788903748580e-05, 2.26870576703247800e-06, -3.74778940336349630e-07, 5.25879288228433850e-08, -6.33625688386961700e-09,
   8.300000e+00, 9.99751544918160650e-01, 2.48393351911641610e-04, -1.24134961365243270e-04, 4.13371927279997050e-05, -1.03137458146273360e-05, 2.05453511063052970e-06, -3.39687946303437860e-07, 4.77470185436928630e-08, -5.77397227994794290e-09,
   8.400000e+00, 9.99775183229766660e-01, 2.24766227653115970e-04, -1.12332582609199500e-04, 3.74105180850925840e-05, -9.33579997993102270e-06, 1.86043328799364710e-06, -3.07832531578974220e-07, 4.33372598136731630e-08, -5.25786305429553240e-09,
   8.500000e+00, 9.99796573021944800e-01, 2.03385595519805850e-04, -1.01651423642826370e-04, 3.38562335528360220e-05, -8.45027753490249690e-06, 1.68454695073643520e-06, -2.78923531334687870e-07, 3.93229449984878610e-08, -4.78485323932654510e-09,
   8.600000e+00, 9.99815928095036610e-01, 1.84038022497226820e-04, -9.19851350192266420e-05, 3.06391337558131090e-05, -7.64849915592080710e-06, 1.52518892241238460e-06, -2.52695919086895630e-07, 3.56708074528069760e-08, -4.35192349451369590e-09,
   8.700000e+00, 9.99833441935222610e-01, 1.66530323188393110e-04, -8.32374246258395520e-05, 2.77273215161909540e-05, -6.92259049692898490e-06, 1.38082422437826690e-06, -2.28907117570329520e-07, 3.23499607343890290e-08, -3.95614777483459150e-09,
   8.800000e+00, 9.99849289641940200e-01, 1.50687644447730390e-04, -7.53211120350152910e-05, 2.50919006417658550e-05, -6.26540937530153360e-06, 1.25005710047956180e-06, -2.07335361348965880e-07, 2.93318130250367760e-08, -3.59471907670550150e-09,
   8.900000e+00, 9.99863629672920530e-01, 1.36351730213388530e-04, -6.81572707766472030e-05, 2.27066965745659000e-05, -5.67047920292358180e-06, 1.13161900488518310e-06, -1.87778125690477490e-07, 2.65899645286942290e-08, -3.26496659905701230e-09,
   9.000000e+00, 9.99876605424013800e-01, 1.23379349764848930e-04, -6.16744505398747690e-05, 2.05480024968597550e-05, -5.13192819138484970e-06, 1.02435751019101220e-06, -1.70050631402633010e-07, 2.41000939307157270e-08, -2.96436621206719760e-09,
   9.100000e+00, 9.99888346659370430e-01, 1.11640874161088810e-04, -5.58079720039935180e-05, 1.85943486753980140e-05, -4.64443388288644270e-06, 9.27226088269874940e-07, -1.53984431565228430e-07, 2.18398386394125530e-08, -2.69054576040561850e-09,
   9.200000e+00, 9.99898970806092250e-01, 1.01018987009750880e-04, -5.04992876380484690e-05, 1.68262929992220030e-05, -4.20317258295546220e-06, 8.39274713695973640e-07, -1.39426083184529280e-07, 1.97886724333609890e-08, -2.44128642819715730e-09,
   9.300000e+00, 9.99908584126147800e-01, 9.14075169901689510e-05, -4.56954023970421620e-05, 1.52262308308658500e-05, -3.80377329648253310e-06, 7.59641238451180340e-07, -1.26235904580310470e-07, 1.79277832574711560e-08, -2.21452114796530720e-09,
   9.400000e+00, 9.99917282777148310e-01, 8.27103807127096620e-05, -4.13483487833612670e-05, 1.37782224450406340e-05, -3.44227579427705870e-06, 6.87543486658803350e-07, -1.14286817650953480e-07, 1.62399532076432020e-08, -2.00833083607346070e-09,
   9.500000e+00, 9.99925153772489360e-01, 7.48406255528386630e-05, -3.74147112379321680e-05, 1.24678364729066370e-05, -3.11509246276708830e-06, 6.22272018989971020e-07, -1.03463272937304910e-07, 1.47094421850544280e-08, -1.82093907471919660e-09,
   9.600000e+00, 9.99932275850380250e-01, 6.77195630593283900e-05, -3.38551952798433820e-05, 1.12820079040004520e-05, -2.81897361428864240e-06, 5.63183517913633550e-07, -9.36602545299373900e-08, 1.33218762596424920e-08, -1.65070572874812520e-09,
   9.700000e+00, 9.99938720260383440e-01, 6.12759844101147830e-05, -3.06342372286879860e-05, 1.02089093220870340e-05, -2.55097595934782470e-06, 5.09694746927039620e-07, -8.47823612630032110e-08, 1.20641414358960210e-08, -1.49611987895293270e-09,
   9.800000e+00, 9.99944551475277170e-01, 5.54454501839007800e-05, -2.77196507235351010e-05, 9.23783416603736810e-06, -2.30843396511461800e-06, 4.61277039162938400e-07, -7.67429602510730080e-08, 1.09242832443545460e-08, -1.35579236756069850e-09,
   9.900000e+00, 9.99949827835316270e-01, 5.01696474376548870e-05, -2.50823065990140750e-05, 8.35909091275179450e-06, -2.08893384610551040e-06, 4.17451273209463760e-07, -6.94634086049875410e-08, 9.89141237430900580e-09, -1.22844818252731630e-09,
   1.000000e+01, 9.99954602131297610e-01, 4.53958077359516730e-05, -2.26958429950565990e-05, 7.56390717663194530e-06, -1.89028995346304090e-06, 3.77783296510341030e-07, -6.28723390698916470e-08, 8.95561640484862610e-09, -1.11291885194540550e-09,
   1.010000e+01, 9.99958922132235250e-01, 4.10761803735431730e-05, -2.05364028648659190e-05, 6.84434280966311460e-06, -1.71052334841719230e-06, 3.41879759270231690e-07, -5.69050053322319550e-08, 8.10787757266656900e-09, -1.00813497577477500e-09,
   1.020000e+01, 9.99962831062897070e-01, 3.71675555730039750e-05, -1.85823963079666270e-05, 6.19321116831338960e-06, -1.54784236341642940e-06, 3.09384324322486770e-07, -5.15026828177809950e-08, 7.33999642781074730e-09, -9.13118987119022980e-10,
   1.030000e+01, 9.99966368035961280e-01, 3.36308329296659620e-05, -1.68142853938692990e-05, 5.60400778868745020e-06, -1.40062497106619520e-06, 2.79974220887243180e-07, -4.66121209299951520e-08, 6.64452116626694230e-09, -8.26978207671312410e-10,
   1.040000e+01, 9.99969568443099480e-01, 3.04306308209097750e-05, -1.52143893589815410e-05, 5.07084578019280360e-06, -1.26740279644937950e-06, 2.53357112530681430e-07, -4.21850428417584110e-08, 6.01468238557349960e-09, -7.48898240323380870e-10,
   1.050000e+01, 9.99972464308885420e-01, 2.75349329002983130e-05, -1.37667082567419530e-05, 4.58839731085322790e-06, -1.14684662267372210e-06, 2.29268251912926360e-07, -3.81776891415831150e-08, 5.44433298228359790e-09, -6.78136725116404050e-10,
   1.060000e+01, 9.99975084611060550e-01, 2.49147681628238800e-05, -1.24567633202728270e-05, 4.15184061479803880e-06, -1.03775325265255170e-06, 2.07467897071115020e-07, -3.45504018382091540e-08, 4.92789289455700780e-09, -6.14017471766445350e-10,
   1.070000e+01, 9.99977455570349560e-01, 2.25439213991371070e-05, -1.12714524597185250e-05, 3.75681200479746720e-06, -9.39033602238237710e-07, 1.87738966018924190e-07, -3.12672454379726200e-08, 4.46029838670065200e-09, -5.55924972150872550e-10,
   1.080000e+01, 9.99979600912720090e-01, 2.03986711571594440e-05, -1.01989194643063940e-05, 3.39936242040826270e-06, -8.49701910954555350e-07, 1.69884908351422640e-07, -2.82956620217398040e-08, 4.03695557286271950e-09, -5.03299288768600680e-10,
   1.090000e+01, 9.99981542106704310e-01, 1.84575526018420250e-05, -9.22843561338458930e-06, 3.07591808572562090e-06, -7.68865966779615540e-07, 1.53727774324813170e-07, -2.56061574585065820e-08, 3.65369788255584290e-09, -4.55631309778496850e-10,
   1.100000e+01, 9.99983298578151940e-01, 1.67011429106034340e-05, -8.35029252246862110e-06, 2.78324489025938510e-06, -6.95718250776955250e-07, 1.39106463536484480e-07, -2.31720160978619510e-08, 3.30674717983455920e-09, -4.12458357364119000e-10,
   1.110000e+01, 9.99984887904558970e-01, 1.51118670655464890e-05, -7.55570516079585820e-06, 2.51841614239820780e-06, -6.29527915921011750e-07, 1.25875136865743850e-07, -2.09690414812554580e-08, 2.99267825968001440e-09, -3.73360133536596960e-10,
   1.120000e+01, 9.99986325990915410e-01, 1.36738221060752420e-05, -6.83672407706992100e-06, 2.27878337760155160e-06, -5.69633522273675250e-07, 1.13901776754521030e-07, -1.89753208008990390e-08, 2.70838645858564190e-09, -3.37954985802325990e-10,
   1.130000e+01, 9.99987627228825620e-01, 1.23726180889609920e-05, -6.18615596090805290e-06, 2.06194993314845680e-06, -5.15436457797194670e-07, 1.03066882215768930e-07, -1.71710110142078850e-08, 2.45105813081807930e-09, -3.05896474172170790e-10,
   1.140000e+01, 9.99988804640494890e-01, 1.11952341690387850e-05, -5.59749174984812630e-06, 1.86574702823836450e-06, -4.66394980589778340e-07, 9.32622861614335840e-08, -1.55381446904247040e-08, 2.21814375683459170e-09, -2.76870220601440280e-10,
   1.150000e+01, 9.99989870009019180e-01, 1.01298883641566480e-05, -5.06484156640055810e-06, 1.68821211272117090e-06, -4.22018824253730030e-07, 8.43900837478793370e-08, -1.40604538242312510e-08, 2.00733346547603950e-09, -2.50591021986417530e-10,
   1.160000e+01, 9.99990833996280190e-01, 9.16591970422912340e-06, -4.58287583726045680e-06, 1.52756926995416270e-06, -3.81864313499765880e-07, 7.63616614509696120e-08, -1.27232099987598100e-08, 1.81653476653133770e-09, -2.26800208194065460e-10,
   1.170000e+01, 9.99991706249626080e-01, 8.29368158759631130e-06, -4.14677200807338770e-06, 1.38221147944510870e-06, -3.45529941999464300e-07, 6.90968175118823230e-08, -1.15130794176649890e-08, 1.64385230486977710e-09, -2.05263227179602160e-10,
   1.180000e+01, 9.99992495498402860e-01, 7.50444527956575040e-06, -3.75216632266128950e-06, 1.25068456322867110e-06, -3.12652368961592510e-07, 6.25229652442683430e-08, -1.04179914531162750e-08, 1.48756946140725130e-09, -1.85767439983336050e-10,
   1.190000e+01, 9.99993209641301890e-01, 6.79031258912383860e-06, -3.39511018590376710e-06, 1.13167265650891500e-06, -2.82902794965877070e-07, 5.65744114695242290e-08, -9.42701947414724110e-09, 1.34613163960855980e-09, -1.68120109240106090e-10,
   1.200000e+01, 9.99993855825397790e-01, 6.14413685133317550e-06, -3.07203067501699310e-06, 1.02398505813788140e-06, -2.55983681274537310e-07, 5.11917030553082150e-08, -8.53027282820451160e-09, 1.21813108896187020e-09, -1.52146565740089000e-10,
   1.210000e+01, 9.99994440517666390e-01, 5.55945142579066820e-06, -2.77969480522334770e-06, 9.26544330131622270e-07, -2.31625780190364130e-07, 4.63210351783787160e-08, -7.71879894849246740e-09, 1.10229312885685030e-09, -1.37688538516805890e-10,
   1.220000e+01, 9.99994969569698240e-01, 5.03040499652609140e-06, -2.51517719316132110e-06, 8.38375527779919400e-07, -2.09585447070190430e-07, 4.19137155214044240e-08, -6.98449465133022680e-09, 9.97463647539518500e-10, -1.24602634882611740e-10,
   1.230000e+01, 9.99995448276255190e-01, 4.55170302661081300e-06, -2.27583079521066120e-06, 7.58596453101359750e-07, -1.89642207361636460e-07, 3.79256791492816810e-08, -6.32002577174939780e-09, 9.02597761305668940e-10, -1.12758957765584690e-10,
   1.240000e+01, 9.99995881428255130e-01, 4.11855478219938940e-06, -2.05926042853633910e-06, 6.86408834539737520e-07, -1.71596554534475850e-07, 3.43170492981490410e-08, -5.71875436249462110e-09, 8.16749528846053340e-10, -1.02039848611428460e-10,
   1.250000e+01, 9.99996273360715840e-01, 3.72662539634620680e-06, -1.86329881038450400e-06, 6.21090344987523090e-07, -1.55267957048707560e-07, 3.10517397537616080e-08, -5.17467275194488770e-09, 7.39062624692496680e-10, -9.23387449901975160e-11,
   1.260000e+01, 9.99996627996136310e-01, 3.37199249328072390e-06, -1.68598487626864630e-06, 5.61987378546745830e-07, -1.40493054560707270e-07, 2.80970948990030250e-08, -4.68234382074757800e-09, 6.68761884064919670e-10, -8.35591428840212360e-11,
   1.270000e+01, 9.99996948883751350e-01, 3.05110693938728140e-06, -1.52554416041168150e-06, 5.08508513977657970e-07, -1.27124025435932610e-07, 2.54235638765753120e-08, -4.23684691561620810e-09, 6.05145639118406560e-10, -7.56136544244528810e-11,
   1.280000e+01, 9.99997239235049800e-01, 2.76075732836992650e-06, -1.38037104238289500e-06, 4.60118599613961530e-07, -1.15027109329103450e-07, 2.30044056455347370e-08, -3.83372887222818370e-09, 5.47578773672759970e-10, -6.84231525942881740e-11,
   1.290000e+01, 9.99997501955914410e-01, 2.49803784540252620e-06, -1.24901268249259760e-06, 4.16333400707343890e-07, -1.04081270126766200e-07, 2.08154220123400230e-08, -3.46895966789292030e-09, 4.95486430012472260e-10, -6.19159951078822260e-11,
   1.300000e+01, 9.99997739675702050e-01, 2.26031918883764270e-06, -1.13015448536443780e-06, 3.76714755763438350e-07, -9.41769859371668570e-08, 1.88347159911528980e-08, -3.13889226903028720e-09, 4.48348307309722370e-10, -5.60273203362117570e-11,
   1.310000e+01, 9.99997954773558460e-01, 2.04522225861253610e-06, -1.02260694636362590e-06, 3.40866193501335580e-07, -8.52151540718136390e-08, 1.70424730968043850e-08, -2.84022627887281730e-09, 4.05693496684508060e-10, -5.06984087491533260e-11,
   1.320000e+01, 9.99998149402227090e-01, 1.85059434815133000e-06, -9.25293749369885780e-07, 3.08428966659113580e-07, -7.71061001041075170e-08, 1.54207633994051250e-08, -2.56997502752137550e-09, 3.67095802910523550e-10, -4.58761039091408600e-11,
   1.330000e+01, 9.99998325509594440e-01, 1.67448760159332860e-06, -8.37240996883241340e-07, 2.79078463023493540e-07, -6.97686811239346070e-08, 1.39533623741242100e-08, -2.32543577985619090e-09, 3.32169507337571490e-10, -4.15122875693579500e-11,
   1.340000e+01, 9.99998484858183550e-01, 1.51513952083031450e-06, -7.57567464763911470e-07, 2.52520957823951480e-07, -6.31294742432537220e-08, 1.26255887651222860e-08, -2.10416276712682460e-09, 3.00565530764879660e-10, -3.75634039080735170e-11,
   1.350000e+01, 9.99998629042793130e-01, 1.37095532733418150e-06, -6.85475784146004530e-07, 2.28490675037187380e-07, -5.71220422554890060e-08, 1.14241578507341960e-08, -1.90394277558092690e-09, 2.71967958795808980e-10, -3.39900283708562110e-11,
   1.360000e+01, 9.99998759506458890e-01, 1.24049200230635130e-06, -6.20244462330858940e-07, 2.06747128230650730e-07, -5.16862691192765790e-08, 1.03370486493598750e-08, -1.72277305049099810e-09, 2.46090895663075560e-10, -3.07564769965723200e-11,
   1.370000e+01, 9.99998877554894690e-01, 1.12244384546689700e-06, -5.61220662851848130e-07, 1.87072714364296580e-07, -4.67677586323083690e-08, 9.35338374359172940e-09, -1.55884129662682110e-09, 2.22675615662537220e-10, -2.78304524736294260e-11,
   1.380000e+01, 9.99998984369560470e-01, 1.01562940799105980e-06, -5.07813672491387920e-07, 1.69270536495415590e-07, -4.23172902904494250e-08, 8.46332052519952860e-09, -1.41050757680551750e-09, 2.01487984198645400e-10, -2.51827235121774230e-11,
   1.390000e+01, 9.99999081019486650e-01, 9.18979668846808110e-07, -4.59488989898996230e-07, 1.53162433617502940e-07, -3.82903268972099070e-08, 7.65795277692489650e-09, -1.27628792882307610e-09, 1.82316123050208660e-10, -2.27868344282110090e-11,
   1.400000e+01, 9.99999168471972340e-01, 8.31527336225271330e-07, -4.15762976674349830e-07, 1.38587197933167670e-07, -3.46465690045819890e-08, 6.92922160969115450e-09, -1.15483953800280790e-09, 1.64968296833256300e-10, -2.06188421188755080e-11,
   1.410000e+01, 9.99999247602266880e-01, 7.52397167011296440e-07, -3.76198017403725340e-07, 1.25398961733719140e-07, -3.13495517333212480e-08, 6.26983486681250500e-09, -1.04494731794733730e-09, 1.49270999789335270e-10, -1.86570778668548890e-11,
   1.420000e+01, 9.99999319202329100e-01, 6.80797207426378850e-07, -3.40398140228036230e-07, 1.13465737752892160e-07, -2.83662799435664040e-08, 5.67319419099261900e-09, -9.45511765996212160e-10, 1.35067223981535910e-10, -1.68819316474784270e-11,
   1.430000e+01, 9.99999383988753320e-01, 6.16010867192104680e-07, -3.08005054126430060e-07, 1.02668098395962280e-07, -2.56668981094086800e-08, 5.13332902615417150e-09, -8.55537972502435670e-10, 1.22214891754338360e-10, -1.52756568269569580e-11,
   1.440000e+01, 9.99999442609941450e-01, 5.57389747901928650e-07, -2.78694563267460070e-07, 9.28979806336570320e-08, -2.32243915974626420e-08, 4.64483689518981040e-09, -7.74125674469692400e-10, 1.10585436923242510e-10, -1.38221933357031860e-11,
   1.450000e+01, 9.99999495652591830e-01, 5.04347153835143390e-07, -2.52173322551391810e-07, 8.40576046064723110e-08, -2.10143163630518180e-08, 4.20282935724158810e-09, -7.00460254446204730e-10, 1.00062520620668450e-10, -1.25070075785966340e-11,
   1.460000e+01, 9.99999543647571490e-01, 4.56352220275225500e-07, -2.28175901880168750e-07, 7.60584951218552990e-08, -1.90145543614346250e-08, 3.80288310471801340e-09, -6.33804594951415360e-10, 9.05408690493893990e-11, -1.13169475058356990e-11,
   1.470000e+01, 9.99999587075228960e-01, 4.12924600573522180e-07, -2.06462129779964910e-07, 6.88205962555279310e-08, -1.72050922283712700e-08, 3.44099571150165140e-09, -5.73491707208582470e-10, 8.19252215961030130e-11, -1.02401114150637630e-11,
   1.480000e+01, 9.99999626370201570e-01, 3.73629658790020030e-07, -1.86814689795835930e-07, 6.22714701992147370e-08, -1.55678210168108540e-08, 3.11354559018851020e-09, -5.18918060651933660e-10, 7.41293788468815700e-11, -9.26572918904046500e-12,
   1.490000e+01, 9.99999661925765950e-01, 3.38074119802057690e-07, -1.69036945606879720e-07, 5.63455723395658040e-08, -1.40863549868900440e-08, 2.81725575819483420e-09, -4.69537546646774780e-10, 6.70753410335421040e-11, -8.38405479444785790e-12,
   1.500000e+01, 9.99999694097773120e-01, 3.05902133349452300e-07, -1.52950973098582340e-07, 5.09835953154601910e-08, -1.27458676368528910e-08, 2.54916105057859740e-09, -4.24856016172213160e-10, 6.06925283347279390e-11, -7.58626897755046640e-12,
   1.510000e+01, 9.99999723208210730e-01, 2.76791712626669860e-07, -1.38395779699661530e-07, 4.61318754907927890e-08, -1.15329433348335780e-08, 2.30657845183041010e-09, -3.84426336931071180e-10, 5.49170752662939320e-11, -6.86439119237450490e-12,
   1.520000e+01, 9.99999749548425520e-01, 2.50451511780763260e-07, -1.25225693164406170e-07, 4.17418559041674510e-08, -1.04354430674030120e-08, 2.08708025003213170e-09, -3.47843920526066450e-10, 4.96911921302037290e-11, -6.21119988774924510e-12,
   1.530000e+01, 9.99999773382038600e-01, 2.26617910065157680e-07, -1.13308903676890040e-07, 3.77696003218491190e-08, -9.44238296191389390e-09, 1.88846974496866160e-09, -3.14742675026023120e-10, 4.49625871566244580e-11, -5.62016036179704430e-12,
   1.540000e+01, 9.99999794947584510e-01, 2.05052373468196750e-07, -1.02526144687613890e-07, 3.41753535315569260e-08, -8.54382436740518010e-09, 1.70875926729129500e-09, -2.84791342487242120e-10, 4.06839435767306050e-11, -5.08535946704138450e-12,
   1.550000e+01, 9.99999814460898140e-01, 1.85539067412074810e-07, -9.27694992812854920e-08, 3.09231434772669350e-08, -7.73077439440740950e-09, 1.54615028892062660e-09, -2.57690184834589790e-10, 3.68124464096959630e-11, -4.60144651691731410e-12,
   1.560000e+01, 9.99999832117275170e-01, 1.67882696630342710e-07, -8.39413201305668000e-08, 2.79804212537906240e-08, -6.99509591858538340e-09, 1.39901542577429730e-09, -2.33167984982521050e-10, 3.33093542418082680e-11, -4.16357980575916520e-12,
   1.570000e+01, 9.99999848093426350e-01, 1.51906550605678100e-07, -7.59532522272354270e-08, 2.53177353586795660e-08, -6.32942614780639910e-09, 1.26588215281746030e-09, -2.10979333222425560e-10, 3.01396117234032380e-11, -3.76737820988777190e-12,
   1.580000e+01, 9.99999862549246110e-01, 1.37450735006717020e-07, -6.87253486106513630e-08, 2.29084369417482830e-08, -5.72710293787126690e-09, 1.14541806854910020e-09, -1.90902171750383300e-10, 2.72714989147432330e-11, -3.40887738775178280e-12,
   1.590000e+01, 9.99999875629413060e-01, 1.24370571424198600e-07, -6.21852702440583400e-08, 2.07284131026607290e-08, -5.18209811965393330e-09, 1.03641756152715300e-09, -1.72735572787080440e-10, 2.46763139790698300e-11, -3.08449014268216460e-12,
   1.600000e+01, 9.99999887464837920e-01, 1.12535149390932300e-07, -5.62675620313048730e-08, 1.87558455676622010e-08, -4.68895717053024010e-09, 9.37789745552565470e-10, -1.56297728074587390e-10, 2.23280860534098520e-11, -2.79097055313068040e-12,
   1.610000e+01, 9.99999898173973390e-01, 1.01826016194119430e-07, -5.09129977285210830e-08, 1.69709923304823300e-08, -4.24274462644235940e-09, 8.48547542817657510e-10, -1.41424129646242390e-10, 2.02033154286029420e-11, -2.52538151268419670e-12,
   1.620000e+01, 9.99999907864000170e-01, 9.21359913675753000e-08, -4.60679871947459620e-08, 1.53559900722216440e-08, -3.83899468837582570e-09, 7.67797805803683330e-10, -1.27965923676962550e-10, 1.82807384424077860e-11, -2.28506535603968290e-12,
   1.630000e+01, 9.99999916631899020e-01, 8.33680939991463290e-08, -4.16840400493334860e-08, 1.38946753829519570e-08, -3.47366652899215460e-09, 6.94732379100357940e-10, -1.15788420950832340e-10, 1.65411147361098290e-11, -2.06761727781812690e-12,
   1.640000e+01, 9.99999924565422190e-01, 7.54345721176905650e-08, -3.77172803684701810e-08, 1.25724229959070900e-08, -3.14310385218560890e-09, 6.28620011720849680e-10, -1.04769749048147460e-10, 1.49670347482546070e-11, -1.87086127889144730e-12,
   1.650000e+01, 9.99999931743970950e-01, 6.82560244455773610e-08, -3.41280075639034880e-08, 1.13759994153780210e-08, -2.84399830088317200e-09, 5.68799038992244420e-10, -9.47996327706489370e-11, 1.35427455211542820e-11, -1.69282840006403590e-12,
   1.660000e+01, 9.99999938239390480e-01, 6.17606057270576150e-08, -3.08802990491561490e-08, 1.02934304734705160e-08, -2.57335634691037160e-09, 5.14670760799277450e-10, -8.57782906056669390e-11, 1.22539930786762880e-11, -1.53173702573830780e-12,
   1.670000e+01, 9.99999944116689240e-01, 5.58833076792934510e-08, -2.79416507167024730e-08, 9.31388149027149750e-09, -2.32846933158667510e-09, 4.65693449924934020e-10, -7.76154361900612390e-11, 1.10878797993319290e-11, -1.38597506082502580e-12,
   1.680000e+01, 9.99999949434689060e-01, 5.05653083696535580e-08, -2.52826516279762380e-08, 8.42754883809184870e-09, -2.10688635723961050e-09, 4.21376930534639630e-10, -7.02293747847081970e-11, 1.00327353584759990e-11, -1.25408380283645340e-12,
   1.690000e+01, 9.99999954246614360e-01, 4.57533835077131030e-08, -2.28766896604843540e-08, 7.62556182458008140e-09, -1.90638975835440750e-09, 3.81277672554679400e-10, -6.35461857204007500e-11, 9.07799994900577560e-12, -1.13474334800917420e-12,
   1.700000e+01, 9.99999958600624520e-01, 4.13993737600350160e-08, -2.06996851661092910e-08, 6.89989391276435460e-09, -1.72497290688843790e-09, 3.44994352856659150e-10, -5.74989826357830220e-11, 8.21411861264436620e-12, -1.02675938560909540e-12,
   1.710000e+01, 9.99999962540295880e-01, 3.74597027564935260e-08, -1.87298499750173800e-08, 6.24328238951959010e-09, -1.56082012963683560e-09, 3.12163838830165950e-10, -5.20272441059724400e-11, 7.43244562493773470e-12, -9.29051248425964030e-13,
   1.720000e+01, 9.99999966105057920e-01, 3.38949409642348850e-08, -1.69474693332503790e-08, 5.64915567850545770e-09, -1.41228853667072560e-09, 2.82457554151907220e-10, -4.70762079645806580e-11, 6.72515797759035930e-12, -8.40641100005626450e-13,
   1.730000e+01, 9.99999969330587990e-01, 3.06694110644097520e-08, -1.53347045915920700e-08, 5.11156757012218340e-09, -1.27789157899298040e-09, 2.55578190383583000e-10, -4.25963232589326220e-11, 6.08517709270804790e-12, -7.60644150518967430e-13,
   1.740000e+01, 9.99999972249168370e-01, 2.77508308821900330e-08, -1.38754146709863800e-08, 4.62513771025639040e-09, -1.15628417086124560e-09, 2.31256731491117960e-10, -3.85427543548140800e-11, 5.50609798582152590e-12, -6.88259803440484810e-13,
   1.750000e+01, 9.99999974890009020e-01, 2.51099902964165140e-08, -1.25549945176966280e-08, 4.18499775222447320e-09, -1.04624922788559510e-09, 2.09249761508916940e-10, -3.48749322287556890e-11, 4.98212516904363390e-12, -6.22763644507803410e-13,
   1.760000e+01, 9.99999977279540600e-01, 2.27204588952999930e-08, -1.13602289314307310e-08, 3.78674263299741260e-09, -9.46685486176279800e-10, 1.89337028406031900e-10, -3.15561484579332660e-11, 4.50801465311386740e-12, -5.63500192849079330e-13,
   1.770000e+01, 9.99999979441678180e-01, 2.05583214523152080e-08, -1.02791603035130130e-08, 3.42638648607461990e-09, -8.56596480637136030e-10, 1.71319239774823510e-10, -2.85531878449379760e-11, 4.07902146807789830e-12, -5.09876341781741150e-13,
   1.780000e+01, 9.99999981398061120e-01, 1.86019385748511750e-08, -9.30096894139346290e-09, 3.10032274977641010e-09, -7.75080572100068920e-10, 1.55016068282403210e-10, -2.58359960011984590e-11, 3.69085217754266420e-12, -4.61355423678786930e-13,
   1.790000e+01, 9.99999983168269550e-01, 1.68317301301231750e-08, -8.41586478175444280e-09, 2.80528807171338990e-09, -7.01321923492638810e-10, 1.40264346924246440e-10, -2.33773785626150360e-11, 3.33962191139516880e-12, -4.17451839537109420e-13,
   1.800000e+01, 9.99999984770020500e-01, 1.52299792808080730e-08, -7.61498940845176490e-09, 2.53832964818240970e-09, -6.34582334728182720e-10, 1.26916436018670270e-10, -2.11527290274570720e-11, 3.02181548706907280e-12, -3.77726199527563320e-13,
   1.810000e+01, 9.99999986219344610e-01, 1.37806551691327910e-08, -6.89032739465993630e-09, 2.29677567161567480e-09, -5.74193854601768850e-10, 1.14838745599494990e-10, -1.91397824929634930e-11, 2.73425223034198820e-12, -3.41780925915358260e-13,
   1.820000e+01, 9.99999987530747370e-01, 1.24692524747864200e-08, -6.23462608191095050e-09, 2.07820859031547940e-09, -5.19552095751452490e-10, 1.03910398419324440e-10, -1.73183928262324930e-11, 2.47405414365580530e-12, -3.09256274362693850e-13,
   1.830000e+01, 9.99999988717353630e-01, 1.12826462409003450e-08, -5.64132299315206480e-09, 1.88044091285195110e-09, -4.70110185780287060e-10, 9.40220201829777590e-11, -1.56703310394700660e-11, 2.23861710344074940e-12, -2.79826733809259210e-13,
   1.840000e+01, 9.99999989791039430e-01, 1.02089605151518090e-08, -5.10448015335302930e-09, 1.70149331496909350e-09, -4.25373294001316050e-10, 8.50746449038807820e-11, -1.41791028518528920e-11, 2.02558479822844680e-12, -2.53197768912388570e-13,
   1.850000e+01, 9.99999990762550420e-01, 9.23744949130964340e-09, -4.61872466032434790e-09, 1.53957482988780090e-09, -3.84893679028459880e-10, 7.69787244282962100e-11, -1.28297836122509570e-11, 1.83282514676019000e-12, -2.29102872454708450e-13,
   1.860000e+01, 9.99999991641610020e-01, 8.35838996164923970e-09, -4.17919491096193710e-09, 1.39306492374552380e-09, -3.48266207648820590e-10, 6.96532322147402270e-11, -1.16088689307822070e-11, 1.65840896010956230e-12, -2.07300898227456780e-13,
   1.870000e+01, 9.99999992437015980e-01, 7.56298400386765620e-09, -3.78149194473510020e-09, 1.26049727677921560e-09, -3.15124300128561930e-10, 6.30248523992157960e-11, -1.05041395243705370e-11, 1.50059063429142370e-12, -1.87573647703208950e-13,
   1.880000e+01, 9.99999993156729050e-01, 6.84327092855724760e-09, -3.42163541744826620e-09, 1.14054510792918420e-09, -2.85136261372177340e-10, 5.70272460303881300e-11, -9.50453892371564860e-12, 1.35779068014539740e-12, -1.69723686350405390e-13,
   1.890000e+01, 9.99999993807952350e-01, 6.19204760598108620e-09, -3.09602376464908920e-09, 1.03200789598872740e-09, -2.58001961216697570e-10, 5.16003871311458980e-11, -8.60006281779316590e-12, 1.22857991566633670e-12, -1.53572367739414650e-13,
   1.900000e+01, 9.99999994397203550e-01, 5.60279637475461250e-09, -2.80139815598597900e-09, 9.33799364401108190e-10, -2.33449830636501440e-10, 4.66899619417901430e-11, -7.78165892846167970e-12, 1.11166516258882510e-12, -1.38958045668612940e-13,
   1.910000e+01, 9.99999994930380160e-01, 5.06961981092019570e-09, -2.53480987975905280e-09, 8.44936609452320900e-10, -2.11234143796065330e-10, 4.22468253324071740e-11, -7.04113641313259240e-12, 1.00587630408506780e-12, -1.25734456420026820e-13,
   1.920000e+01, 9.99999995412818250e-01, 4.58718170456303600e-09, -2.29359083123928190e-09, 7.64530263051603250e-10, -1.91132558748822240e-10, 3.82265089441330630e-11, -6.37108388881173700e-12, 9.10154574055845760e-13, -1.13769254956240810e-13,
   1.930000e+01, 9.99999995849346290e-01, 4.15065365324235680e-09, -2.07532680939325240e-09, 6.91775591645800390e-10, -1.72943892168808220e-10, 3.45887761367049390e-11, -5.76479525709860340e-12, 8.23541960818220710e-13, -1.02942690410456670e-13,
   1.940000e+01, 9.99999996244333220e-01, 3.75566673772821010e-09, -1.87783335475907260e-09, 6.25944442183002370e-10, -1.56486105844073100e-10, 3.12972192881436440e-11, -5.21620258780029460e-12, 7.45171619145671900e-13, -9.31464076153326440e-14,
   1.950000e+01, 9.99999996601732220e-01, 3.39826779639862290e-09, -1.69913388665108740e-09, 5.66377954518213050e-10, -1.41594484780145320e-10, 2.83188954162658910e-11, -4.71981538945660100e-12, 6.74259194706837880e-13, -8.42823626773298690e-14,
   1.960000e+01, 9.99999996925120090e-01, 3.07487986067683370e-09, -1.53743992088353070e-09, 5.12479967324586100e-10, -1.28119988679517820e-10, 2.56239964752521010e-11, -4.27066565899153560e-12, 6.10094974079608410e-13, -7.62618417444422280e-14,
   1.970000e+01, 9.99999997217733580e-01, 2.78226635553385050e-09, -1.39113317002591930e-09, 4.63711051514635650e-10, -1.15927760298323590e-10, 2.31855510275305900e-11, -3.86425816054372880e-12, 5.52036781779190810e-13, -6.90045731477784620e-14,
   1.980000e+01, 9.99999997482501230e-01, 2.51749870676266970e-09, -1.25874934704353510e-09, 4.19583111455978510e-10, -1.04895775751394730e-10, 2.09791543052389950e-11, -3.49652543585985230e-12, 4.99503553214271660e-13, -6.24379240317863160e-14,
   1.990000e+01, 9.99999997722072930e-01, 2.27792703082745150e-09, -1.13896351022477420e-09, 3.79654499948956970e-10, -9.49136232575887370e-11, 1.89827239596575490e-11, -3.16378709598952840e-12, 4.51969519249915690e-13, -5.64961734333785040e-14,
   2.000000e+01, 9.99999997938846420e-01, 2.06115361394184920e-09, -1.03057680272257050e-09, 3.43525598075287350e-10, -8.58813981027037640e-11, 1.71762790540935310e-11, -2.86271298686651670e-12, 4.08958944176434980e-13, -5.11198545352164150e-14,
   2.010000e+01, 9.99999998134991070e-01, 1.86500891494625040e-09, -9.32504453994866920e-10, 3.10834815679450130e-10, -7.77087027604431210e-11, 1.55417400883208640e-11, -2.59028986013089150e-12, 3.70041364421770110e-13, -4.62551595106321980e-14,
   2.020000e+01, 9.99999998312470110e-01, 1.68752985181301180e-09, -8.43764923058748870e-10, 2.81254972454411600e-10, -7.03137421643505800e-11, 1.40627480531691870e-11, -2.34379121562788890e-12, 3.34827280356277340e-13, -4.18534010040367140e-14,
   2.030000e+01, 9.99999998473059870e-01, 1.52694015446351060e-09, -7.63470074900209110e-10, 2.54490023412372220e-10, -6.36225050759109730e-11, 1.27245007043093660e-11, -2.12075001376061900e-12, 3.02964258073152960e-13, -3.78705248574103820e-14,
   2.040000e+01, 9.99999998618367410e-01, 1.38163258726171380e-09, -6.90816291721948200e-10, 2.30272095968043680e-10, -5.75680233557080490e-11, 1.15136044166204650e-11, -1.91893398459636310e-12, 2.74133402130800350e-13, -3.42666692063229750e-14,
   2.050000e+01, 9.99999998749847130e-01, 1.25015286326097820e-09, -6.25076430067606910e-10, 2.08358808980614170e-10, -5.20897017241928200e-11, 1.04179401364542740e-11, -1.73632328661428310e-12, 2.48046163955918000e-13, -3.10057655329591960e-14,
   2.060000e+01, 9.99999998868814850e-01, 1.13118508921247310e-09, -5.65592543326656920e-10, 1.88530846922499150e-10, -4.71327113040982220e-11, 9.42654209020901760e-12, -1.57109029149796130e-12, 2.24441453965363630e-13, -2.80551776835127940e-14,
   2.070000e+01, 9.99999998976461410e-01, 1.02353859566414990e-09, -5.11769296784443690e-10, 1.70589764896393730e-10, -4.26474408748880140e-11, 8.52948803529343670e-12, -1.42158129265418420e-12, 2.03083028504486790e-13, -2.53853752372474360e-14,
   2.080000e+01, 9.99999999073863940e-01, 9.26136020341316250e-10, -4.63068009312930200e-10, 1.54356002532491450e-10, -3.85890003472135510e-11, 7.71779995507898760e-12, -1.28629995439192380e-12, 1.83757125449920370e-13, -2.29696379582943160e-14,
   2.090000e+01, 9.99999999161997440e-01, 8.38002525543446580e-10, -4.19001262069475020e-10, 1.39667086888326170e-10, -3.49167714879988000e-11, 6.98335420396666360e-12, -1.16389233611674490e-12, 1.66270324813525820e-13, -2.07837883723313160e-14,
   2.100000e+01, 9.99999999241743990e-01, 7.58256041641286250e-10, -3.79128020245690870e-10, 1.26376006365262140e-10, -3.15940013996647920e-11, 6.31880020327266280e-12, -1.05313334165867850e-12, 1.50447612935973550e-13, -1.88059497917515770e-14,
   2.110000e+01, 9.99999999313901600e-01, 6.86098439027881960e-10, -3.43049219043209910e-10, 1.14349739367249250e-10, -2.85874346849019580e-11, 5.71748687421624950e-12, -9.52914458114660960e-13, 1.36130630895985690e-13, -1.70163273676139010e-14,
   2.120000e+01, 9.99999999379192480e-01, 6.20807540169556160e-10, -3.10403769699376070e-10, 1.03467922976190680e-10, -2.58669806155803370e-11, 5.17339607172913470e-12, -8.62232661492544680e-13, 1.23176089604941330e-13, -1.53970099771192710e-14,
   2.130000e+01, 9.99999999438270120e-01, 5.61729891810647090e-10, -2.80864945589783040e-10, 9.36216483195673650e-11, -2.34054119747116840e-11, 4.68108235287027450e-12, -7.80180378121024880e-13, 1.11454335724711900e-13, -1.39317909638732220e-14,
   2.140000e+01, 9.99999999491725800e-01, 5.08274224993904350e-10, -2.54137112238609450e-10, 8.47123705739746900e-11, -2.11780925573794440e-11, 4.23561847703019730e-12, -7.05936401356469130e-13, 1.00848054056096410e-13, -1.26060059368765470e-14,
   2.150000e+01, 9.99999999540094440e-01, 4.59905537442205450e-10, -2.29952768509589620e-10, 7.66509226955211350e-11, -1.91627306033759160e-11, 3.83254609247343600e-12, -6.38757672678323810e-13, 9.12510934110227360e-14, -1.14063860049076810e-14,
   2.160000e+01, 9.99999999583860210e-01, 4.16139739076070330e-10, -2.08069869364862890e-10, 6.93566230061727730e-11, -1.73391556938190990e-11, 3.46783111567418200e-12, -5.77971844915817840e-13, 8.25674042175323160e-14, -1.03209249774382680e-14,
   2.170000e+01, 9.99999999623461200e-01, 3.76538807077571410e-10, -1.88269403397004230e-10, 6.27564677044804220e-11, -1.56891168788596160e-11, 3.13782335686772680e-12, -5.22970553176555670e-13, 7.47100772248225910e-14, -9.33875920300291470e-15,
   2.180000e+01, 9.99999999659293650e-01, 3.40706402010826230e-10, -1.70353200889332250e-10, 5.67844002190568430e-11, -1.41961000160705930e-11, 2.83921998773667190e-12, -4.73203326130296410e-13, 6.76004736874283440e-14, -8.45005884241790450e-15,
   2.190000e+01, 9.99999999691716050e-01, 3.08283901123787880e-10, -1.54141950466854970e-10, 5.13806500922590160e-11, -1.28451624913850980e-11, 2.56903248560515810e-12, -4.28172076710239100e-13, 6.11674383231901790e-14, -7.64592948868777890e-15,
   2.200000e+01, 9.99999999721053250e-01, 2.78946809131269810e-10, -1.39473404487823600e-10, 4.64911347774003090e-11, -1.16227836684129700e-11, 2.32455672330775130e-12, -3.87426117093010900e-13, 5.53465871680641350e-14, -6.91832314898794850e-15,
   2.210000e+01, 9.99999999747598460e-01, 2.52401510557107620e-10, -1.26200755214847310e-10, 4.20669183624780810e-11, -1.05167295693840130e-11, 2.10334590538259950e-12, -3.50557648065698950e-13, 5.00796632004138530e-14, -6.25995769780880440e-15,
   2.220000e+01, 9.99999999771617690e-01, 2.28382331131840030e-10, -1.14191165513761530e-10, 3.80637218031481840e-11, -9.51593043340088230e-12, 1.90318607972571140e-12, -3.17197677636130170e-13, 4.53139532856885770e-14, -5.66424399512856770e-15,
   2.230000e+01, 9.99999999793351080e-01, 2.06648878835349790e-10, -1.03324439374971130e-10, 3.44414797631878660e-11, -8.61036992656238030e-12, 1.72207397961864170e-12, -2.87012328038495440e-13, 4.10017606060865360e-14, -5.12521994019332860e-15,
   2.240000e+01, 9.99999999813016350e-01, 1.86983637972757640e-10, -9.34918189514159360e-11, 3.11639396271633880e-11, -7.79098489513655400e-12, 1.55819697436559350e-12, -2.59699494173693080e-13, 3.70999272951259230e-14, -4.63749080089746830e-15,
   2.250000e+01, 9.99999999830810230e-01, 1.69189792204262680e-10, -8.45948960735061490e-11, 2.81982986720852590e-11, -7.04957465847958610e-12, 1.40991492787922590e-12, -2.34985820040973830e-13, 3.35694024995018370e-14, -4.19617522156412460e-15,
   2.260000e+01, 9.99999999846910790e-01, 1.53089254741075000e-10, -7.65446273471011700e-11, 2.55148757667428430e-11, -6.37871893387360370e-12, 1.27574378364987830e-12, -2.12623962900032170e-13, 3.03748515452576770e-14, -3.79685636875619410e-15,
   2.270000e+01, 9.99999999861479140e-01, 1.38520885992999430e-10, -6.92604429773116690e-11, 2.30868143129785320e-11, -5.77170357184862140e-12, 1.15434071181131950e-12, -1.92390117782418320e-13, 2.74843022966878780e-14, -3.43553772617158530e-15,
   2.280000e+01, 9.99999999874661150e-01, 1.25338880829263350e-10, -6.26694403989218490e-11, 2.08898134558340590e-11, -5.22245335872190240e-12, 1.04449066964973590e-12, -1.74081777576741110e-13, 2.48688251686159030e-14, -3.10860309620449580e-15,
   2.290000e+01, 9.99999999886588720e-01, 1.13411309311772790e-10, -5.67056546430242640e-11, 1.89018848724333370e-11, -4.72547121382095930e-12, 9.45094241049241960e-13, -1.57515706269890320e-13, 2.25022435895129010e-14, -2.81278040785696970e-15,
   2.300000e+01, 9.99999999897381200e-01, 1.02618796295957670e-10, -5.13093981374482180e-11, 1.71031327054623260e-11, -4.27578317285537590e-12, 8.55156633166992920e-13, -1.42526105059804710e-13, 2.03608720176785480e-14, -2.54510896877928730e-15,
   2.310000e+01, 9.99999999907146720e-01, 9.28533266842058060e-11, -4.64266633334811670e-11, 1.54755544387458930e-11, -3.86888860681256030e-12, 7.73777720212946720e-13, -1.28962952985636000e-13, 1.84232788884655840e-14, -2.30290983368759400e-15,
   2.320000e+01, 9.99999999915982870e-01, 8.40171643744709210e-11, -4.20085821801765730e-11, 1.40028607220196020e-11, -3.50071517815193910e-12, 7.00143034689203340e-13, -1.16690505467805710e-13, 1.66700721200499090e-14, -2.08375899259708350e-15,
   2.330000e+01, 9.99999999923978140e-01, 7.60218740845145900e-11, -3.80109370364779650e-11, 1.26703123416397720e-11, -3.16757808348350120e-12, 6.33515615926123530e-13, -1.05585935730828360e-13, 1.50837050310157920e-14, -1.88546311052990970e-15,
   2.340000e+01, 9.99999999931212580e-01, 6.87874362618821850e-11, -3.43937181262093800e-11, 1.14645727055819860e-11, -2.86614317481825930e-12, 5.73228634332757030e-13, -9.55381055118278960e-14, 1.36483007273187600e-14, -1.70603757589353910e-15,
   2.350000e+01, 9.99999999937758570e-01, 6.22414462213298320e-11, -3.11207231067909190e-11, 1.03735743663476410e-11, -2.59339359029557740e-12, 5.18678717542582510e-13, -8.64464527515860870e-14, 1.23494932010329620e-14, -1.54368663783071520e-15,
   2.360000e+01, 9.99999999943681610e-01, 5.63183894943992020e-11, -2.81591947440278380e-11, 9.38639824589477160e-12, -2.34659956041643930e-12, 4.69319911660386450e-13, -7.82199851357639110e-14, 1.11742835505470860e-14, -1.39678543374930320e-15,
   2.370000e+01, 9.99999999949040990e-01, 5.09589861386018280e-11, -2.54794930667040940e-11, 8.49316435383681970e-12, -2.12329108759359880e-12, 4.24658217172477320e-13, -7.07763694133320710e-14, 1.01109098832148270e-14, -1.26386372715798570e-15,
   2.380000e+01, 9.99999999953890440e-01, 4.61095974438298670e-11, -2.30547987197888400e-11, 7.68493290517888300e-12, -1.92123322558602250e-12, 3.84246644833725180e-13, -6.40411073777944180e-14, 9.14872959840117060e-15, -1.14359119305063860e-15,
   2.390000e+01, 9.99999999958278260e-01, 4.17216890981185860e-11, -2.08608445473185910e-11, 6.95361484794573110e-12, -1.73840371140619950e-12, 3.47680742049146680e-13, -5.79467902641600380e-14, 8.27811287277588580e-15, -1.03476410357095610e-15,
   2.400000e+01, 9.99999999962248640e-01, 3.77513454399406490e-11, -1.88756727185451600e-11, 6.29189090523161070e-12, -1.57297272583284800e-12, 3.14594544976547720e-13, -5.24324240994173280e-14, 7.49034628181943890e-15, -9.36293280703099570e-16,
   2.410000e+01, 9.99999999965841210e-01, 3.41588299355048740e-11, -1.70794149665856120e-11, 5.69313832141732000e-12, -1.42328457996538810e-12, 2.84656915837500860e-13, -4.74428192543912290e-14, 6.77754559295334130e-15, -8.47193195414959260e-16,
   2.420000e+01, 9.99999999969091840e-01, 3.09081874821725750e-11, -1.54540937401309710e-11, 5.15136457940677950e-12, -1.28784114453325620e-12, 2.57568228779275750e-13, -4.29280380874208080e-14, 6.13257685750054600e-15, -7.66572104154818930e-16,
   2.430000e+01, 9.99999999972033150e-01, 2.79668845577048780e-11, -1.39834422780702910e-11, 4.66114742550199920e-12, -1.16528685611478420e-12, 2.33057371118670670e-13, -3.88428951516830370e-14, 5.54898501173698450e-15, -6.93623123984117890e-16,
   2.440000e+01, 9.99999999974694460e-01, 2.53054836138380930e-11, -1.26527418062786780e-11, 4.21758060166598080e-12, -1.05439515020303940e-12, 2.10879029955225560e-13, -3.51465049640768190e-14, 5.02092927245075130e-15, -6.27616157023431210e-16,
   2.450000e+01, 9.99999999977102650e-01, 2.28973484554069520e-11, -1.14486742271791870e-11, 3.81622474204353670e-12, -9.54056185336121260e-13, 1.90811236997319130e-13, -3.18018728095848100e-14, 4.54312468042591100e-15, -5.67890583388830820e-16,
   2.460000e+01, 9.99999999979281570e-01, 2.07183776563503520e-11, -1.03591888277459260e-11, 3.45306294229580770e-12, -8.63265735430868160e-13, 1.72653147028940160e-13, -2.87755244857455290e-14, 4.11078920679855240e-15, -5.13848649487116960e-16,
   2.470000e+01, 9.99999999981253220e-01, 1.87467633445398620e-11, -9.37338167191849080e-12, 3.12446055707186930e-12, -7.81115139150820290e-13, 1.56223027783305240e-13, -2.60371712815979360e-14, 3.71959589290838880e-15, -4.64949485497862530e-16,
   2.480000e+01, 9.99999999983037240e-01, 1.69627729412651200e-11, -8.48138647034482390e-12, 2.82712882325645110e-12, -7.06782205718200820e-13, 1.41356441105275410e-13, -2.35594068380909870e-14, 3.36562954464492610e-15, -4.20703692167169170e-16,
   2.490000e+01, 9.99999999984651500e-01, 1.53485516709541000e-11, -7.67427583524147330e-12, 2.55809194492343870e-12, -6.39522986152333650e-13, 1.27904597199056330e-13, -2.13174328560392550e-14, 3.04534754787128340e-15, -3.80668442736043620e-16,
   2.500000e+01, 9.99999999986112110e-01, 1.38879438645782700e-11, -6.94397193209626070e-12, 2.31465731057017010e-12, -5.78664327578250840e-13, 1.15732865489933500e-13, -1.92888109064166970e-14, 2.75554441275317920e-15, -3.44443050981845840e-16,
   2.510000e+01, 9.99999999987433720e-01, 1.25663312682865470e-11, -6.28316563398536010e-12, 2.09438854455651170e-12, -5.23597136086490280e-13, 1.04719427196243050e-13, -1.74532378590221670e-14, 2.49331969214078320e-15, -3.11664961016287790e-16,
   2.520000e+01, 9.99999999988629540e-01, 1.13704867390081220e-11, -5.68524336937477280e-12, 1.89508112303873240e-12, -4.73770280716587120e-13, 9.47540561260790210e-14, -1.57923426819337060e-14, 2.25604895292020590e-15, -2.82006118704587770e-16,
   2.530000e+01, 9.99999999989711560e-01, 1.02884418627584790e-11, -5.14422093127338680e-12, 1.71474031035389420e-12, -4.28685077553189540e-13, 8.57370154965243110e-14, -1.42895025780495160e-14, 2.04135750980577790e-15, -2.55169688389684050e-16,
   2.540000e+01, 9.99999999990690670e-01, 9.30936717072967560e-12, -4.65468358527817370e-12, 1.55156119503494810e-12, -3.87890298729848920e-13, 7.75780597344145440e-14, -1.29296766185506780e-14, 1.84709665869245460e-15, -2.30887082061432010e-16,
   2.550000e+01, 9.99999999991576520e-01, 8.42346375432673730e-12, -4.21173187709241450e-12, 1.40391062565016820e-12, -3.50977656388890460e-13, 7.01955312683174650e-14, -1.16992552082327010e-14, 1.67132217170365880e-15, -2.08915271237704210e-16,
   2.560000e+01, 9.99999999992378100e-01, 7.62186519439670540e-12, -3.81093259714025940e-12, 1.27031086567469130e-12, -3.17577716399308540e-13, 6.35155432721159960e-14, -1.05859238761040960e-14, 1.51227483870575570e-15, -1.89034354653797760e-16,
   2.570000e+01, 9.99999999993103410e-01, 6.89654882312605530e-12, -3.44827441151546520e-12, 1.14942480380678020e-12, -2.87356200935840890e-13, 5.74712401808265320e-14, -9.57854002802387150e-15, 1.36836286054230050e-15, -1.71045357416795870e-16,
   2.580000e+01, 9.99999999993759770e-01, 6.24025543048449800e-12, -3.12012771520330850e-12, 1.04004257170847540e-12, -2.60010642914138620e-13, 5.20021285776356180e-14, -8.66702142787523540e-15, 1.23814591777340460e-15, -1.54768239598054020e-16,
   2.590000e+01, 9.99999999994353630e-01, 5.64641661161116710e-12, -2.82320830577370160e-12, 9.41069435236645860e-13, -2.35267358798534080e-13, 4.70534717554558880e-14, -7.84224529115900320e-15, 1.12032075547500640e-15, -1.40040094333163040e-16,
   2.600000e+01, 9.99999999994890860e-01, 5.10908902801111980e-12, -2.55454451397945690e-12, 8.51514837975750460e-13, -2.12878709485236670e-13, 4.25757418935669650e-14, -7.09595698110103680e-15, 1.01370813982582700e-15, -1.26713517395362360e-16,
   2.610000e+01, 9.99999999995377140e-01, 4.62289492462592040e-12, -2.31144746229158880e-12, 7.70482487416282180e-13, -1.92620621846946820e-13, 3.85241243665398760e-14, -6.42068739347348420e-15, 9.17241055939117900e-16, -1.14655131924544790e-16,
   2.620000e+01, 9.99999999995817010e-01, 4.18296830745373390e-12, -2.09148415370936970e-12, 6.97161384558125010e-13, -1.74290346133698860e-13, 3.48580692244068100e-14, -5.80967820329014760e-15, 8.29954028819262650e-16, -1.03744253546861090e-16,
   2.630000e+01, 9.99999999996215140e-01, 3.78490624304569280e-12, -1.89245312150852060e-12, 6.30817707159956550e-13, -1.57704426785213970e-13, 3.15408853551327240e-14, -5.25681422521876490e-15, 7.50973460563626540e-16, -9.38716825249754870e-17,
   2.640000e+01, 9.99999999996575300e-01, 3.42472479246811050e-12, -1.71236239622232650e-12, 5.70787465399622950e-13, -1.42696866345996170e-13, 2.85393732676354040e-14, -4.75656221075129030e-15, 6.79508887101247850e-16, -8.49386108504218940e-17,
   2.650000e+01, 9.99999999996901150e-01, 3.09881913870262030e-12, -1.54940956934170750e-12, 5.16469856440834000e-13, -1.29117464107007590e-13, 2.58234928201211660e-14, -4.30391546959340840e-15, 6.14845066962833870e-16, -7.68556333398695290e-17,
   2.660000e+01, 9.99999999997196020e-01, 2.80392750842574440e-12, -1.40196375420501020e-12, 4.67321251396428690e-13, -1.16830312846486510e-13, 2.33660625682490340e-14, -3.89434376102541680e-15, 5.56334822903795940e-16, -6.95418528380157220e-17,
   2.670000e+01, 9.99999999997462920e-01, 2.53709852708530230e-12, -1.26854926353621420e-12, 4.22849754507780140e-13, -1.05712438624799420e-13, 2.11424877241016340e-14, -3.52374795373085620e-15, 5.03392564736955720e-16, -6.29240705716849610e-17,
   2.680000e+01, 9.99999999997704390e-01, 2.29566168055180610e-12, -1.14783084027063300e-12, 3.82610280086697580e-13, -9.56525700199177090e-14, 1.91305140032808680e-14, -3.18841900031258660e-15, 4.55488428549162340e-16, -5.69360535519149340e-17,
   2.690000e+01, 9.99999999997922770e-01, 2.07720058771549270e-12, -1.03860029385343170e-12, 3.46200097948267330e-13, -8.65500244856285820e-14, 1.73100048965504150e-14, -2.88500081589996880e-15, 4.12142973645204890e-16, -5.15178716919529540e-17,
   2.700000e+01, 9.99999999998120500e-01, 1.87952881653201790e-12, -9.39764408262476450e-13, 3.13254802751803660e-13, -7.83137006867733800e-14, 1.56627401368836600e-14, -2.61045668932360430e-15, 3.72922384144227540e-16, -4.66152980068137490e-17,
   2.710000e+01, 9.99999999998299360e-01, 1.70066800147561990e-12, -8.50334000734917720e-13, 2.83444666909711020e-13, -7.08611667264636660e-14, 1.41722333449070980e-14, -2.36203889068930450e-15, 3.37434127204601900e-16, -4.21792658913934270e-17,
   2.720000e+01, 9.99999999998461120e-01, 1.53882804339207140e-12, -7.69414021693667660e-13, 2.56471340562977240e-13, -6.41178351399549690e-14, 1.28235670276752620e-14, -2.13726117117396640e-15, 3.05323024423354020e-16, -3.81653780454018200e-17,
   2.730000e+01, 9.99999999998607560e-01, 1.39238919358461520e-12, -6.96194596790368790e-13, 2.32064865595497110e-13, -5.80162163982280180e-14, 1.16032432793871060e-14, -1.93387387981168440e-15, 2.76267697091335900e-16, -3.45334621302622340e-17,
   2.740000e+01, 9.99999999998740120e-01, 1.25988584282460730e-12, -6.29942921410716320e-13, 2.09980973802513880e-13, -5.24952434500993650e-14, 1.04990486898082330e-14, -1.74984144823082500e-15, 2.49977349727104320e-16, -3.12471687108489570e-17,
   2.750000e+01, 9.99999999998860020e-01, 1.13999185304175630e-12, -5.69995926519578510e-13, 1.89998642172326450e-13, -4.74996605426484150e-14, 9.49993210835640690e-15, -1.58332201800164190e-15, 2.26188859698017630e-16, -2.82736074581265470e-17,
   2.760000e+01, 9.99999999998968490e-01, 1.03150728488855420e-12, -5.15753642443213040e-13, 1.71917880813695000e-13, -4.29794702030690830e-14, 8.59589404047194890e-15, -1.43264900669803580e-15, 2.04664143800493900e-16, -2.55830179716839340e-17,
   2.770000e+01, 9.99999999999066640e-01, 9.33346388344024310e-13, -4.66673194171140950e-13, 1.55557731389799560e-13, -3.88894328471595120e-14, 7.77788656931575120e-15, -1.29631442818057490e-15, 1.85187775443305800e-16, -2.31484719276477160e-17,
   2.780000e+01, 9.99999999999155450e-01, 8.44526736162543640e-13, -4.22263368080558590e-13, 1.40754456026377370e-13, -3.51886140063566000e-14, 7.03772280117622370e-15, -1.17295380016433840e-15, 1.67564828585848670e-16, -2.09456035709668770e-17,
   2.790000e+01, 9.99999999999235830e-01, 7.64159391411773770e-13, -3.82079695705302950e-13, 1.27359898568045010e-13, -3.18399746418166060e-14, 6.36799492828546310e-15, -1.06133248802162440e-15, 1.51618926852816940e-16, -1.89523658547483410e-17,
   2.800000e+01, 9.99999999999308550e-01, 6.91440010693064080e-13, -3.45720005346053980e-13, 1.15240001781699250e-13, -2.88100004452654500e-14, 5.76200008898934550e-15, -9.60333348143642610e-16, 1.37190478300163660e-16, -1.71488097860027170e-17,
   2.810000e+01, 9.99999999999374390e-01, 6.25640794002349870e-13, -3.12820397000783510e-13, 1.04273465666666880e-13, -2.60683664165362450e-14, 5.21367328325505900e-15, -8.68945547191779780e-16, 1.24135078165283750e-16, -1.55168847694178470e-17,
   2.820000e+01, 9.99999999999433900e-01, 5.66103200663120480e-13, -2.83051600331239800e-13, 9.43505334435329370e-14, -2.35876333607764100e-14, 4.71752667211255270e-15, -7.86254445337848910e-16, 1.12322063615623210e-16, -1.40402579509355260e-17,
   2.830000e+01, 9.99999999999487740e-01, 5.12231358429965140e-13, -2.56115679214720190e-13, 8.53718930713984750e-14, -2.13429732677621560e-14, 4.26859465351744730e-15, -7.11432442241246580e-16, 1.01633206031131970e-16, -1.27041507530585410e-17,
   2.840000e+01, 9.99999999999536480e-01, 4.63486099798866320e-13, -2.31743049899218340e-13, 7.72476832995962260e-14, -1.93119208248274500e-14, 3.86238416493684800e-15, -6.43730694146593790e-16, 9.19615277324998190e-17, -1.14951909658805120e-17,
   2.850000e+01, 9.99999999999580670e-01, 4.19379565837602710e-13, -2.09689782918625460e-13, 6.98965943060912380e-14, -1.74741485764641810e-14, 3.49482971526938580e-15, -5.82471619203747550e-16, 8.32102313125876770e-17, -1.04012789135151130e-17,
   2.860000e+01, 9.99999999999620530e-01, 3.79470323529567900e-13, -1.89735161764639940e-13, 6.32450539214506480e-14, -1.58112634803146610e-14, 3.16225269604373280e-15, -5.27042116000888960e-16, 7.52917308554413040e-17, -9.41146635647302730e-18,
   2.870000e+01, 9.99999999999656610e-01, 3.43358947763788770e-13, -1.71679473881776470e-13, 5.72264912938468950e-14, -1.43066228234224260e-14, 2.86132456466876570e-15, -4.76887427439554530e-16, 6.81267753470107030e-17, -8.51584691800206660e-18,
   2.880000e+01, 9.99999999999689360e-01, 3.10684023754150380e-13, -1.55342011876978680e-13, 5.17806706255952050e-14, -1.29451676563666270e-14, 2.58903353126045530e-15, -4.31505588539119280e-16, 6.16436555043627530e-17, -7.70545693773891780e-18,
   2.890000e+01, 9.99999999999718890e-01, 2.81118529878744400e-13, -1.40559264939293190e-13, 4.68530883130450390e-14, -1.17132720782349170e-14, 2.34265441563644660e-15, -3.90442402602562120e-16, 5.57774860850767700e-17, -6.97218576038371540e-18,
   2.900000e+01, 9.99999999999745650e-01, 2.54366564737562880e-13, -1.27183282368716740e-13, 4.23944274561957760e-14, -1.05986068640273770e-14, 2.11972137279684820e-15, -3.53286895463265750e-16, 5.04695564939306290e-17, -6.30869456153592520e-18,
   2.910000e+01, 9.99999999999769850e-01, 2.30160385671822320e-13, -1.15080192835858200e-13, 3.83600642785840840e-14, -9.59001606962836320e-15, 1.91800321391860960e-15, -3.19667202317413840e-16, 4.56667431875292930e-17, -5.70834289827299170e-18,
   2.920000e+01, 9.99999999999791720e-01, 2.08257729105462710e-13, -1.04128864552687990e-13, 3.47096215175337460e-14, -8.67740537936897830e-15, 1.73548107586801300e-15, -2.89246845976074600e-16, 4.13209779960313320e-17, -5.16512224936623100e-18,
   2.930000e+01, 9.99999999999811600e-01, 1.88439385889827740e-13, -9.42196929448783500e-14, 3.14065643149357740e-14, -7.85164107872210740e-15, 1.57032821573968700e-15, -2.61721369288369650e-16, 3.73887670407447510e-17, -4.67359587998036610e-18,
   2.940000e+01, 9.99999999999829470e-01, 1.70507007384838220e-13, -8.52535036923900290e-14, 2.84178345641106270e-14, -7.10445864101796640e-15, 1.42089172819971680e-15, -2.36815288031994070e-16, 3.38307554327728270e-17, -4.22884442900430890e-18,
   2.950000e+01, 9.99999999999845680e-01, 1.54281120319141160e-13, -7.71405601595467770e-14, 2.57135200531663900e-14, -6.42838001328366320e-15, 1.28567600265355900e-15, -2.14279333774535300e-16, 3.06113333960599290e-17, -3.82641667443192610e-18,
   2.960000e+01, 9.99999999999860440e-01, 1.39599330561269830e-13, -6.97996652806154250e-14, 2.32665550935254810e-14, -5.81663877337487480e-15, 1.16332775467237650e-15, -1.93887959111196630e-16, 2.76982798727806240e-17, -3.46228498403571140e-18,
   2.970000e+01, 9.99999999999873660e-01, 1.26314697824611900e-13, -6.31573489122899990e-14, 2.10524496374193610e-14, -5.26311240934952190e-15, 1.05262248186777700e-15, -1.75437080310587040e-16, 2.50624400441669680e-17, -3.13280500547021870e-18,
   2.980000e+01, 9.99999999999885760e-01, 1.14294265039617650e-13, -5.71471325197957610e-14, 1.90490441732565420e-14, -4.76226104330978190e-15, 9.52452208660214540e-16, -1.58742034776121860e-16, 2.26774335392800970e-17, -2.83467919236854260e-18,
   2.990000e+01, 9.99999999999896530e-01, 1.03417727674766650e-13, -5.17088638373726360e-14, 1.72362879457837450e-14, -4.30907198644237150e-15, 8.61814397287048370e-16, -1.43635732880699370e-16, 2.05193904113926700e-17, -2.56492380139013070e-18,
   3.000000e+01, 9.99999999999906410e-01, 9.35762296883842290e-14, -4.67881148441833610e-14, 1.55960382813886140e-14, -3.89900957034423470e-15, 7.79801914067679530e-16, -1.29966985677557410e-16, 1.85667122395398650e-17, -2.32083902991468430e-18,
   3.010000e+01, 9.99999999999915290e-01, 8.46712740607782790e-14, -4.23356370303819700e-14, 1.41118790101225430e-14, -3.52796975252824600e-15, 7.05593950504693360e-16, -1.17598991750463600e-16, 1.67998559642609060e-17, -2.09998199550985370e-18,
   3.020000e+01, 9.99999999999923390e-01, 7.66137370029714090e-14, -3.83068685014798350e-14, 1.27689561671560310e-14, -3.19223904178705130e-15, 6.38447808356627660e-16, -1.06407968059177070e-16, 1.52011382940936180e-17, -1.90014228674306770e-18,
   3.030000e+01, 9.99999999999930720e-01, 6.93229759758561190e-14, -3.46614879879232510e-14, 1.15538293293045450e-14, -2.88845733232453450e-15, 5.77691466464266220e-16, -9.62819110771641100e-17, 1.37545587252481340e-17, -1.71931984064076160e-18,
   3.040000e+01, 9.99999999999937270e-01, 6.27260225925628000e-14, -3.13630112962774620e-14, 1.04543370987565300e-14, -2.61358427468782110e-15, 5.22716854937039650e-16, -8.71194758226650810e-17, 1.24456394031879060e-17, -1.55570492538599900e-18,
   3.050000e+01, 9.99999999999943270e-01, 5.67568523263207840e-14, -2.83784261631571700e-14, 9.45947538771690840e-15, -2.36486884692815350e-15, 4.72973769385201240e-16, -7.88289615640570320e-17, 1.12612802233958130e-17, -1.40766002791424940e-18,
   3.060000e+01, 9.99999999999948600e-01, 5.13557237147964600e-14, -2.56778618573955920e-14, 8.55928728579677160e-15, -2.13982182144831410e-15, 4.27964364289311180e-16, -7.13273940481013100e-17, 1.01896277211238390e-17, -1.27370346513210690e-18,
   3.070000e+01, 9.99999999999953480e-01, 4.64685804474652490e-14, -2.32342902237304630e-14, 7.74476340790871500e-15, -1.93619085197645890e-15, 3.87238170395003900e-16, -6.45396950657380130e-17, 9.21995643793515330e-18, -1.15249455473503900e-18,
   3.080000e+01, 9.99999999999957920e-01, 4.20465103518813390e-14, -2.10232551759388990e-14, 7.00775172531178780e-15, -1.75193793132735770e-15, 3.50387586265235810e-16, -5.83979310441273950e-17, 8.34256157771003620e-18, -1.04282019720814140e-18,
   3.090000e+01, 9.99999999999961920e-01, 3.80452558642185360e-14, -1.90226279321078230e-14, 6.34087597736830850e-15, -1.58521899434159470e-15, 3.17043798868125930e-16, -5.28406331446233300e-17, 7.54866187778495290e-18, -9.43582734718524040e-19,
   3.100000e+01, 9.99999999999965580e-01, 3.44247710846973950e-14, -1.72123855423475140e-14, 5.73746184744838090e-15, -1.43436546186170020e-15, 2.86873092372182050e-16, -4.78121820619776750e-17, 6.83031172312461890e-18, -8.53788965386815540e-19,
   3.110000e+01, 9.99999999999968800e-01, 3.11488209847564900e-14, -1.55744104923772730e-14, 5.19147016412511060e-15, -1.29786754103095420e-15, 2.59573508206061480e-16, -4.32622513676337930e-17, 6.18032162393536420e-18, -7.72540202988840200e-19,
   3.120000e+01, 9.99999999999971800e-01, 2.81846187547117050e-14, -1.40923093773550570e-14, 4.69743645911782230e-15, -1.17435911477919090e-15, 2.34871822955732270e-16, -3.91453038259200720e-17, 5.59218626083563660e-18, -6.99023282601933050e-19,
   3.130000e+01, 9.99999999999974460e-01, 2.55024976623415030e-14, -1.27512488311701020e-14, 4.25041627705626620e-15, -1.06260406926384980e-15, 2.12520813852683260e-16, -3.54201356420849750e-17, 5.06001937743245150e-18, -6.32502422176991450e-19,
   3.140000e+01, 9.99999999999976910e-01, 2.30756141382610940e-14, -1.15378070691300150e-14, 3.84593568970964980e-15, -9.61483922427234960e-16, 1.92296784485376010e-16, -3.20494640808723390e-17, 4.57849486868928600e-18, -5.72311858584469210e-19,
   3.150000e+01, 9.99999999999979130e-01, 2.08796791164584650e-14, -1.04398395582287960e-14, 3.47994651940930800e-15, -8.69986629852181640e-16, 1.73997325970378200e-16, -2.89995543283769920e-17, 4.14279347547689220e-18, -5.17849184433227650e-19,
   3.160000e+01, 9.99999999999981130e-01, 1.88927149411555350e-14, -9.44635747057741110e-15, 3.14878582352556530e-15, -7.87196455881272310e-16, 1.57439291176206870e-16, -2.62398818626852850e-17, 3.74855455180765200e-18, -4.68569318974822510e-19,
   3.170000e+01, 9.99999999999982900e-01, 1.70948354070447300e-14, -8.54741770352207300e-15, 2.84913923450716240e-15, -7.12284808626693180e-16, 1.42456961725299670e-16, -2.37428269542036270e-17, 3.39183242202537880e-18, -4.23979052752244240e-19,
   3.180000e+01, 9.99999999999984570e-01, 1.54680467314601930e-14, -7.73402336572985840e-15, 2.57800778857645970e-15, -6.44501947144035160e-16, 1.28900389428775130e-16, -2.14833982381185570e-17, 3.06905689115675550e-18, -3.83632111393834950e-19,
   3.190000e+01, 9.99999999999986010e-01, 1.39960674665539270e-14, -6.99803373327676720e-15, 2.33267791109212490e-15, -5.83169477772965950e-16, 1.16633895554567090e-16, -1.94389825924191410e-17, 2.77699751320024640e-18, -3.47124689149409350e-19,
   3.200000e+01, 9.99999999999987340e-01, 1.26641655490938550e-14, -6.33208277454676670e-15, 2.11069425818214850e-15, -5.27673564545483690e-16, 1.05534712909075360e-16, -1.75891188181721000e-17, 2.51273125973683490e-18, -3.14091407466594670e-19,
   3.210000e+01, 9.99999999999988560e-01, 1.14590108570219640e-14, -5.72950542851085020e-15, 1.90983514283686250e-15, -4.77458785709171850e-16, 9.54917571418168590e-17, -1.59152928569636410e-17, 2.27361326527885370e-18, -2.84201658159439420e-19,
   3.220000e+01, 9.99999999999989670e-01, 1.03685417971138640e-14, -5.18427089855682390e-15, 1.72809029951886960e-15, -4.32022574879681600e-16, 8.64045149759219900e-17, -1.44007524959822210e-17, 2.05725035656752380e-18, -2.57156294570599110e-19,
   3.230000e+01, 9.99999999999990560e-01, 9.38184458849850840e-15, -4.69092229424916660e-15, 1.56364076474966340e-15, -3.90910191187386520e-16, 7.81820382374655680e-17, -1.30303397062403490e-17, 1.86147710089036070e-18, -2.32684637611014280e-19,
   3.240000e+01, 9.99999999999991560e-01, 8.48904403387157360e-15, -4.24452201693571420e-15, 1.41484067231185670e-15, -3.53710168077940160e-16, 7.07420336155784300e-17, -1.17903389359265360e-17, 1.68433413370287570e-18, -2.10541766712631340e-19,
   3.250000e+01, 9.99999999999992340e-01, 7.68120468520197700e-15, -3.84060234260092930e-15, 1.28020078086693710e-15, -3.20050195216714640e-16, 6.40100390433350650e-17, -1.06683398405532220e-17, 1.52404854864971060e-18, -1.90506068581026700e-19,
   3.260000e+01, 9.99999999999993010e-01, 6.95024141476382340e-15, -3.47512070738186320e-15, 1.15837356912725540e-15, -2.89593392281797740e-16, 5.79186784563531140e-17, -9.65311307605670500e-18, 1.37901615372177360e-18, -1.72377019215068900e-19,
   3.270000e+01, 9.99999999999993670e-01, 6.28883849646153690e-15, -3.14441924823072860e-15, 1.04813974941021660e-15, -2.62034937352540930e-16, 5.24069874705029170e-17, -8.73449791174872870e-18, 1.24778541596360150e-18, -1.55973176995325000e-19,
   3.280000e+01, 9.99999999999994340e-01, 5.69037638758344200e-15, -2.84518819379168870e-15, 9.48396064597207920e-16, -2.37099016149291180e-16, 4.74198032298539190e-17, -7.90330053830754850e-18, 1.12904293404352410e-18, -1.41130366755338760e-19,
   3.290000e+01, 9.99999999999994890e-01, 5.14886547819375430e-15, -2.57443273909685070e-15, 8.58144246365599090e-16, -2.14536061591390950e-16, 4.29072123182746550e-17, -7.15120205304459720e-18, 1.02160029329174860e-18, -1.27700036661385190e-19,
   3.300000e+01, 9.99999999999995340e-01, 4.65888614510335430e-15, -2.32944307255165550e-15, 7.76481024183870600e-16, -1.94120256045960430e-16, 3.88240512091891920e-17, -6.47067520153056780e-18, 9.24382171646948020e-19, -1.15547771455798300e-19,
   3.310000e+01, 9.99999999999995780e-01, 4.21553451045879130e-15, -2.10776725522937790e-15, 7.02589085076447440e-16, -1.75647271269105920e-16, 3.51294542538188150e-17, -5.85490904230234690e-18, 8.36415577471538480e-19, -1.04551947183885770e-19,
   3.320000e+01, 9.99999999999996230e-01, 3.81437336208504050e-15, -1.90718668104250570e-15, 6.35728893680825560e-16, -1.58932223420201530e-16, 3.17864446840383680e-17, -5.29774078067241470e-18, 7.56820111524446220e-19, -9.46025139405087250e-20,
   3.330000e+01, 9.99999999999996560e-01, 3.45138774437419260e-15, -1.72569387218708430e-15, 5.75231290729020130e-16, -1.43807822682251060e-16, 2.87615645364486240e-17, -4.79359408940757470e-18, 6.84799155629502910e-19, -8.55998944536504290e-20,
   3.340000e+01, 9.99999999999996890e-01, 3.12294477526047450e-15, -1.56147238763022740e-15, 5.20490795876735930e-16, -1.30122698969180730e-16, 2.60245397938348490e-17, -4.33742329897204160e-18, 6.19631899853024490e-19, -7.74539874815976800e-20,
   3.350000e+01, 9.99999999999997220e-01, 2.82575728711559530e-15, -1.41287864355778950e-15, 4.70959547852591190e-16, -1.17739886963145140e-16, 2.35479773926279630e-17, -3.92466289877097200e-18, 5.60666128395752110e-19, -7.00832660494431820e-20,
   3.360000e+01, 9.99999999999997450e-01, 2.55685092766995250e-15, -1.27842546383496980e-15, 4.26141821278318850e-16, -1.06535455319577530e-16, 2.13070910639146380e-17, -3.55118184398548230e-18, 5.07311691997843240e-19, -6.34139614997088460e-20,
   3.370000e+01, 9.99999999999997670e-01, 2.31353439169574210e-15, -1.15676719584786570e-15, 3.85589065282618280e-16, -9.63972663206527840e-17, 1.92794532641298440e-17, -3.21324221068806990e-18, 4.59034601526799610e-19, -5.73793251908333040e-20,
   3.380000e+01, 9.99999999999997890e-01, 2.09337248551938240e-15, -1.04668624275968680e-15, 3.48895414253225950e-16, -8.72238535633050340e-17, 1.74447707126604210e-17, -2.90746178544320920e-18, 4.15351683634688900e-19, -5.19189604543224990e-20,
   3.390000e+01, 9.99999999999998110e-01, 1.89416175478486190e-15, -9.47080877392427420e-16, 3.15693625797473390e-16, -7.89234064493671520e-17, 1.57846812898729520e-17, -2.63078021497866600e-18, 3.75825744996906480e-19, -4.69782181246030610e-20,
   3.400000e+01, 9.99999999999998330e-01, 1.71390843154200710e-15, -8.56954215771000580e-16, 2.85651405256998220e-16, -7.14128513142485820e-17, 1.42825702628493230e-17, -2.38042837714142440e-18, 3.40061196734451400e-19, -4.25076495917978010e-20,
   3.410000e+01, 9.99999999999998450e-01, 1.55080847994651830e-15, -7.75404239973256690e-16, 2.58468079991083940e-16, -6.46170199977701830e-17, 1.29234039995537170e-17, -2.15390066659217940e-18, 3.07700095227423530e-19, -3.84625119034199790e-20,
   3.420000e+01, 9.99999999999998560e-01, 1.40322954086308700e-15, -7.01614770431541530e-16, 2.33871590143845830e-16, -5.84678975359608040e-17, 1.16935795071918990e-17, -1.94892991786522900e-18, 2.78418559695007630e-19, -3.48023199618701640e-20,
   3.430000e+01, 9.99999999999998780e-01, 1.26969459466634840e-15, -6.34847297333172530e-16, 2.11615765777723100e-16, -5.29039414444302390e-17, 1.05807882888858340e-17, -1.76346471481423380e-18, 2.51923530687728270e-19, -3.14904413359617250e-20,
   3.440000e+01, 9.99999999999998890e-01, 1.14886717873210330e-15, -5.74433589366050270e-16, 1.91477863122015880e-16, -4.78694657805035270e-17, 9.57389315610053080e-18, -1.59564885935003000e-18, 2.27949837049987270e-19, -2.84937296312450620e-20,
   3.450000e+01, 9.99999999999999000e-01, 1.03953801167021980e-15, -5.19769005835108810e-16, 1.73256335278368870e-16, -4.33140838195918600e-17, 8.66281676391822790e-18, -1.44380279398632340e-18, 2.06257541998031890e-19, -2.57821927497502700e-20,
   3.460000e+01, 9.99999999999999110e-01, 9.40612890429909060e-16, -4.70306445214953640e-16, 1.56768815071650610e-16, -3.91922037679123620e-17, 7.83844075358235410e-18, -1.30640679226368620e-18, 1.86629541751943390e-19, -2.33286927189902040e-20,
   3.470000e+01, 9.99999999999999110e-01, 8.51101739147944780e-16, -4.25550869573971700e-16, 1.41850289857990070e-16, -3.54625724644972780e-17, 7.09251449289935940e-18, -1.18208574881652830e-18, 1.68869392688066110e-19, -2.11086740860057180e-20,
   3.480000e+01, 9.99999999999999220e-01, 7.70108700136545650e-16, -3.85054350068272230e-16, 1.28351450022757000e-16, -3.20878625056890530e-17, 6.41757250113773230e-18, -1.06959541685626230e-18, 1.52799345265172830e-19, -1.90999181581443530e-20,
   3.490000e+01, 9.99999999999999330e-01, 6.96823167838575130e-16, -3.48411583919287070e-16, 1.16137194639762020e-16, -2.90342986599403450e-17, 5.80685973198800470e-18, -9.67809955331312550e-19, 1.38258565047323660e-19, -1.72823206309143410e-20 };

   Int_t i = (Int_t)(10*x+350.5);
   if(i<0)   return TMath::Exp(x);
   if(i>699) return 1;
   Double_t* basePoint = sigval+(10*i);
   Double_t res =  x - basePoint[0];
   res =        basePoint[1] + res * ( basePoint[2] + res * ( basePoint[3]
      + res * ( basePoint[4] + res * ( basePoint[5] + res * ( basePoint[6]
      + res * ( basePoint[7] + res * ( basePoint[8] )))))));
   /*
   if(x<-15) return TMath::Exp(x);
   if(x> 15) return 1;
   return (1/(1+TMath::Exp(-x)));
   */
   return res;
}

////////////////////////////////////////////////////////////////////////////////
/// The Derivative of the Sigmoid.

Double_t TNeuron::DSigmoid(Double_t x) const
{
   //if (x>2.1972246) return x/(1+TMath::Exp(x));
   //Double_t a = Sigmoid(x);
   //return a-a*a;
   if (TMath::Abs(x) > 5)
      return TMath::Exp(-TMath::Abs(x));
   Double_t expmx = TMath::Exp(-x);
   return expmx / ((1 + expmx) * (1 + expmx));
}

////////////////////////////////////////////////////////////////////////////////
/// Adds a synapse to the neuron as an input
/// This method is used by the TSynapse while
/// connecting two neurons.

void TNeuron::AddPre(TSynapse * pre)
{
   fpre.AddLast(pre);
   if (fpre.GetEntriesFast() == fpre.GetSize())
      fpre.Expand(2 * fpre.GetSize());
}

////////////////////////////////////////////////////////////////////////////////
/// Adds a synapse to the neuron as an output
/// This method is used by the TSynapse while
/// connecting two neurons.

void TNeuron::AddPost(TSynapse * post)
{
   fpost.AddLast(post);
   if (fpost.GetEntriesFast() == fpost.GetSize())
      fpost.Expand(2 * fpost.GetSize());
}

////////////////////////////////////////////////////////////////////////////////
/// Tells a neuron which neurons form its layer (including itself).
/// This is needed for self-normalizing functions, like Softmax.

void TNeuron::AddInLayer(TNeuron * nearP)
{
   flayer.AddLast(nearP);
   if (flayer.GetEntriesFast() == flayer.GetSize())
      flayer.Expand(2 * flayer.GetSize());
}

////////////////////////////////////////////////////////////////////////////////
/// Returns the neuron type.

TNeuron::ENeuronType TNeuron::GetType() const
{
   return fType;
}

////////////////////////////////////////////////////////////////////////////////
/// Sets a formula that can be used to make the neuron an input.
/// The formula is automatically normalized to mean=0, RMS=1.
/// This normalisation is used by GetValue() (input neurons)
/// and GetError() (output neurons)

TTreeFormula* TNeuron::UseBranch(TTree* input, const char* formula)
{
   if (fFormula) delete fFormula;
   // Set the formula
   // One checks for {}: defined as instance in TMultiLayerPerceptron
   // so that EvalInstance can be used. This allows to use arrays as
   // input, also in complex calculation, without detailed string parsing.
   TRegexp re("{[0-9]+}$");
   TString f(formula);
   Ssiz_t len = f.Length();
   Ssiz_t pos = re.Index(f,&len);
   if(pos==-1 || len<3)
      fFormula = new TTreeFormula(Form("NF%zu",(size_t)this),formula,input);
   else {
      TString newformula(formula,pos);
      TString val = f(pos+1,len-2);
      fFormula = new TTreeFormula(Form("NF%zu",(size_t)this),
                                  (const char*) newformula,input);
      fIndex = val.Atoi();
      f = newformula;
   }
   // Computes the default normalization
   TH1D tmp("tmpb", "tmpb", 1, -FLT_MAX, FLT_MAX);
   input->Draw(Form("%s>>tmpb",(const char*)f),"","goff");
   fNorm[0] = tmp.GetRMS();
   if(fNorm[0]<1e-15) fNorm[0]=1.;
   fNorm[1] = tmp.GetMean();
   // Check the dimensionality
   if(fFormula->GetNdata()>1 && fIndex==0)
      Warning("TNeuron::UseBranch()","all indices in arrays must be specified, otherwise the first element will be assumed.");
   return fFormula;
}

////////////////////////////////////////////////////////////////////////////////
/// Returns the formula value.

Double_t TNeuron::GetBranch() const
{
   Double_t branch = fFormula->EvalInstance(fIndex);
   if (TMath::IsNaN(branch))
      branch = 0.;
   return branch;
}

////////////////////////////////////////////////////////////////////////////////
/// Returns neuron input

Double_t TNeuron::GetInput() const
{
   if (!fNewInput) {
      return fInput;
   }
   ((TNeuron*)this)->fNewInput = false;
   Double_t input = 0.0;
   Int_t nEntries = fpre.GetEntriesFast();
   if (nEntries) {
      input = fWeight;
      for (Int_t i = 0; i < nEntries; i++) {
         TSynapse *preSynapse = (TSynapse*)fpre.UncheckedAt(i);
         input += preSynapse->GetValue();
      }
   }
   return (((TNeuron*)this)->fInput = input);
}

////////////////////////////////////////////////////////////////////////////////
/// Computes the output using the appropriate function and all
/// the weighted inputs, or uses the branch as input.
/// In that case, the branch normalisation is also used.

Double_t TNeuron::GetValue() const
{
   if (!fNewValue) {
      return fValue;
   }
   ((TNeuron*)this)->fNewValue = false;
   Int_t nentries = fpre.GetEntriesFast();
   if (!nentries) {
      Double_t branch = GetBranch();
      return (((TNeuron*)this)->fValue = (branch - fNorm[1]) / fNorm[0]);
   } else {
      Double_t input = GetInput();
      Double_t value = 0;
      switch (fType) {
      case TNeuron::kOff: {
         value = 0;
         break;
      }
      case TNeuron::kLinear: {
         value = input;
         break;
      }
      case TNeuron::kSigmoid: {
         value =  Sigmoid(input);
         break;
      }
      case TNeuron::kTanh: {
         value = TMath::TanH(input);
         break;
      }
      case TNeuron::kGauss: {
         value = TMath::Exp(-input * input);
         break;
      }
      case TNeuron::kSoftmax: {
         Double_t normalization = 0.0;
         for (Int_t i = 0; i < flayer.GetEntriesFast(); i++) {
            // this is not very efficient, as the same normalization
            // is calculated once per neuron instead of just once
            normalization += TMath::Exp(((TNeuron*) flayer.UncheckedAt(i))->GetInput());
         }
         if (normalization > 0.0) {
            // exp(a_k) / sum_k'[ exp(a_k') ]
            value = TMath::Exp(input) / normalization;
         } else {
            // all neurons have output = 0 ?
            value = 1.0 / flayer.GetEntriesFast();
         }
         break;
      }
      case TNeuron::kExternal: {
         value = fExtF->Eval(input);
         break;
      }
      }
      return (((TNeuron*)this)->fValue = value);
   }
}

////////////////////////////////////////////////////////////////////////////////
/// computes the derivative for the appropriate function
/// at the working point

Double_t TNeuron::GetDerivative() const
{
   if (!fNewDeriv)
      return fDerivative;
   ((TNeuron*)this)->fNewDeriv = false;
   Int_t nentries = fpre.GetEntriesFast();
   Double_t input = fWeight;
   for (Int_t i=0;i<nentries;i++) {
      TSynapse *preSynapse = (TSynapse*)fpre.UncheckedAt(i);
      input += preSynapse->GetValue();
   }
   Double_t derivative = 0;
   switch (fType) {
      case TNeuron::kOff: {
         derivative = 0;
         break;
      }
      case TNeuron::kLinear: {
         derivative = 1;
         break;
      }
      case TNeuron::kSigmoid: {
         derivative = DSigmoid(input);
         break;
      }
      case TNeuron::kTanh: {
         derivative = (1 - (TMath::TanH(input) * TMath::TanH(input)));
         break;
      }
      case TNeuron::kGauss: {
         derivative = (-2) * input * TMath::Exp(-input * input);
         break;
      }
      case TNeuron::kSoftmax: {
         // actually this is not the derivative, as the Softmax function needs
         // exp(activation) of all the neurons in the layer for the normalization.
         // it will be smartly taken care of in GetDeDw()
         derivative = GetValue();
         break;
      }
      case TNeuron::kExternal: {
         derivative = fExtD->Eval(input);
         break;
      }
   }
   return (((TNeuron*)this)->fDerivative = derivative);
}

////////////////////////////////////////////////////////////////////////////////
/// Computes the error for output neurons.
/// Returns 0 for other neurons.

Double_t TNeuron::GetError() const
{
   if (!fpost.GetEntriesFast())
      return (GetValue() - (GetBranch() - fNorm[1]) / fNorm[0]);
   return 0;
}

////////////////////////////////////////////////////////////////////////////////
/// Computes the normalized target pattern for output neurons.
/// Returns 0 for other neurons.

Double_t TNeuron::GetTarget() const
{
   if (!fpost.GetEntriesFast())
      return ((GetBranch() - fNorm[1]) / fNorm[0]);
   return 0;
}

////////////////////////////////////////////////////////////////////////////////
/// Computes the derivative of the error wrt the neuron weight.

Double_t TNeuron::GetDeDw() const
{
   if (!fNewDeDw)
      return fDeDw;
   ((TNeuron*)this)->fNewDeDw = false;
   // Output units DeDw is just the error for these cases:
   //  - linear output,  with sum-of-squares error
   //  - sigmoid output, with binary cross-entropy
   //  - softmax output, with 1-of-c cross entropy
   Int_t nEntries = fpost.GetEntriesFast();
   if (nEntries == 0) {
      // output neuron
      ((TNeuron*)this)->fDeDw = GetError();
   } else {
      // hidden nuron
      ((TNeuron*)this)->fDeDw = 0.0;
      if (fType != TNeuron::kSoftmax) {
         // non-softmax
         for (Int_t i = 0; i < nEntries; i++) {
            TSynapse *postSynapse = (TSynapse*)fpost.UncheckedAt(i);
            ((TNeuron*)this)->fDeDw +=
              postSynapse->GetWeight() *
              postSynapse->GetPost()->GetDeDw();
         }
      } else {
         // softmax derivative can be taken care of correcting the forward weight
         for (Int_t i = 0; i < nEntries; i++) {
            TSynapse *postSynapse = (TSynapse*)fpost.UncheckedAt(i);
            ((TNeuron*)this)->fDeDw +=
              (postSynapse->GetWeight() - postSynapse->GetPost()->GetInput()) *
              postSynapse->GetPost()->GetDeDw();
         }
      }
      ((TNeuron*)this)->fDeDw *= GetDerivative();
   }
   return fDeDw;
}

////////////////////////////////////////////////////////////////////////////////
/// Uses the branch type to force an external value.

void TNeuron::ForceExternalValue(Double_t value)
{
   fNewValue = false;
   fValue = (value - fNorm[1]) / fNorm[0];
}

////////////////////////////////////////////////////////////////////////////////
/// Sets the normalization variables.
/// Any input neuron will return (branch-mean)/RMS.
/// When UseBranch is called, mean and RMS are automatically set
/// to the actual branch mean and RMS.

void TNeuron::SetNormalisation(Double_t mean, Double_t RMS)
{
   fNorm[0] = RMS;
   fNorm[1] = mean;
}

////////////////////////////////////////////////////////////////////////////////
/// Sets the neuron weight to w.
/// The neuron weight corresponds to the bias in the
/// linear combination of the inputs.

void TNeuron::SetWeight(Double_t w)
{
   fWeight = w;
}

////////////////////////////////////////////////////////////////////////////////
/// Inform the neuron that inputs of the network have changed,
/// so that the buffered values have to be recomputed.

void TNeuron::SetNewEvent() const
{
   ((TNeuron*)this)->fNewInput = true;
   ((TNeuron*)this)->fNewValue = true;
   ((TNeuron*)this)->fNewDeriv = true;
   ((TNeuron*)this)->fNewDeDw = true;
}

////////////////////////////////////////////////////////////////////////////////
/// Sets the derivative of the total error wrt the neuron weight.

void TNeuron::SetDEDw(Double_t in)
{
   fDEDw = in;
}
