/*-4 2002/05/13 11:52:50
 * $Id: rot13.c,v 1.18 2002/08/15 16:21:41 jonas Exp $
 *
 * See the file LICENSE for redistribution information. 
 * If you have not received a copy of the license, please contact CodeFactory
 * by email at info@codefactory.se, or on the web at http://www.codefactory.se/
 * You may also write to: CodeFactory AB, SE-903 47, Ume, Sweden.
 *
 * Copyright (c) 2002 Jonas Borgstrm <jonas@codefactory.se>
 * Copyright (c) 2002 Daniel Lundin   <daniel@codefactory.se>
 * Copyright (c) 2002 CodeFactory AB.  All rights reserved.
 */

#include <librr/rr.h>
#include "rot13.h"
#include "rot13-filter.h"

#include <stdio.h>
#include <string.h>

/*
 * Rot13 tuning profile semantics:
 *
 * initiator                                            listener
 * ------------------------ <begin /> ------------------------->
 *        *** listener enables rot13 read() filter ***
 * <---------------------- <proceed /> -------------------------
 *  *** initiator enables rot13 read()/write() filter ***
 * -------*rot13*--------- <party_on /> -------*rot13*--------->
 *        *** listener enables rot13 write() filter ***
 * <------*rot13*--------- <greeting /> -------*rot13*--------->
 */

static GObjectClass *parent_class = NULL;

static gboolean frame_available (RRChannel *channel, RRFrame *frame, 
				 GError **error);
static gboolean client_init (RRChannel *channel, GError **error);

static gchar party_on_msg[] = RR_BEEP_MIME_HEADER "<party_on />\r\n";
static gchar proceed_msg[]  = RR_BEEP_MIME_HEADER "<proceed />\r\n";



static void
rr_rot13_init (GObject *object)
{
}

static void
rr_rot13_class_init (GObjectClass *klass)
{
	RRChannelClass *channel_class = (RRChannelClass *)klass;

	channel_class->frame_available = frame_available;
	channel_class->client_init = client_init;

	parent_class = g_type_class_peek_parent (klass);
}

GType 
rr_rot13_get_type (void)
{
	static GType rr_type = 0;

	if (!rr_type) {
		static GTypeInfo type_info = {
			sizeof (RRRot13Class),
			NULL,
			NULL,
			(GClassInitFunc) rr_rot13_class_init,
			NULL,
			NULL,
			sizeof (RRRot13),
			16,
			(GInstanceInitFunc) rr_rot13_init
		};
		rr_type = g_type_register_static (RR_TYPE_CHANNEL, "RRRot13", 
						  &type_info, 0);

		rr_channel_set_uri (rr_type, RR_ROT13_URI);
	}
	return rr_type;
}

static void
do_enable_crypt (gpointer data, gpointer user_data)
{
	RRRot13Filter *filter = RR_ROT13_FILTER (data);
	filter->encrypt_write = TRUE;
}

static void
do_client_handshake (gpointer data, gpointer user_data)
{
	RRRot13Filter *filter;
	RRChannel *channel = RR_CHANNEL (data);
	RRConnection *conn = channel->connection;
	RRMessage *msg;

	filter = rr_rot13_filter_new ();
	filter->decrypt_read  = TRUE;
	filter->encrypt_write = TRUE;
	rr_filterstack_push (conn->filter_stack, RR_FILTER (filter));
	
	/* Send the last message before the tuning reset */
	msg = rr_message_static_new (RR_FRAME_TYPE_MSG, party_on_msg, 
				     sizeof (party_on_msg), FALSE);
	
	rr_channel_send_message (channel, msg, NULL);
	rr_connection_complete_tuning_reset (conn, channel);
}

static gboolean
frame_available (RRChannel *channel, RRFrame *frame, GError **error)
{
	RRRot13 *rot13 = RR_ROT13 (channel);
	RRMessage *msg;
	RRRot13Filter *filter;
	RRConnection *conn = channel->connection;

	g_return_val_if_fail (RR_IS_ROT13 (channel), FALSE);

	if (frame->type == RR_FRAME_TYPE_MSG) {
		if (strstr (frame->payload, "<begin")) {
			filter = rr_rot13_filter_new ();
			filter->decrypt_read  = TRUE;
			filter->encrypt_write = FALSE;
			rr_filterstack_push (channel->connection->filter_stack,
					     RR_FILTER (filter));

			msg = rr_message_static_new (RR_FRAME_TYPE_RPY,
						     proceed_msg, 
						     sizeof (proceed_msg), 
						     FALSE);

			if (!rr_connection_begin_tuning_reset (conn, error))
				return FALSE;
			if (!rr_channel_send_message (channel, msg, error))
				return FALSE;
			
			rr_connection_do_quiescence (conn, do_enable_crypt, 
						     filter, NULL);
		}
		else {
			rr_connection_complete_tuning_reset (conn, channel);
		}
	}
	else if (frame->type == RR_FRAME_TYPE_RPY) {
		
		/* Enable encryption */
		rr_connection_do_quiescence (conn, do_client_handshake, 
					     rot13, NULL);

	}

	return TRUE;
}

static gboolean
client_init (RRChannel *channel, GError **error)
{
/* 	g_print ("instance_config = %s\n", channel->instance_config); */
/* 	g_print ("global_config = %s\n", channel->global_config); */

	rr_connection_begin_tuning_reset (channel->connection, NULL);
	return TRUE;
}

gboolean
rr_rot13_start (RRConnection *connection, GError **error)
{
	RRMessage *msg;
	RRManager *manager;
	RRRot13 *rot13;
	char str[] = RR_BEEP_MIME_HEADER "<begin />\r\n";

	g_return_val_if_fail (RR_IS_CONNECTION (connection), FALSE);
	manager = rr_connection_get_manager (connection);
	g_return_val_if_fail (RR_IS_MANAGER (manager), FALSE);

	if ((rot13 = (RRRot13 *)rr_connection_start (connection, NULL, 
						  RR_TYPE_ROT13, 
						  NULL, error)) == NULL)
		return FALSE;

	msg = rr_message_static_new (RR_FRAME_TYPE_MSG, str, strlen(str), 
				     FALSE);

	if (!rr_channel_send_message (RR_CHANNEL (rot13), msg, error))
		return FALSE;
	if (!rr_manager_wait_for_greeting (manager, error))
		return FALSE;
	if (!rr_manager_wait_for_greeting_sent (manager, error))
		return FALSE;

	if (rot13->response_error) {

		g_propagate_error (error, rot13->response_error);
		rot13->response_error = NULL;
		return FALSE;
	}
	g_object_unref (G_OBJECT (rot13));
	return TRUE;
}
