/*-2 2002/08/19 18:05:43
 * $Id: rr-tcplistener.c,v 1.17 2003/01/06 19:29:48 jonas Exp $
 *
 * See the file LICENSE for redistribution information. 
 * If you have not received a copy of the license, please contact CodeFactory
 * by email at info@codefactory.se, or on the web at http://www.codefactory.se/
 * You may also write to: CodeFactory AB, SE-903 47, Ume, Sweden.
 *
 * Copyright (c) 2002 Jonas Borgstrm <jonas@codefactory.se>
 * Copyright (c) 2002 Daniel Lundin   <daniel@codefactory.se>
 * Copyright (c) 2002 CodeFactory AB.  All rights reserved.
 */

#include <librr/rr.h>

#ifdef G_PLATFORM_WIN32
#include "rr-win32.h"
#else
#include <unistd.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#endif

#include <stdio.h>
#include <string.h>

#define BACKLOG 5

static GObjectClass *parent_class = NULL;

static void finalize (GObject *object);
static gboolean tcp_shutdown (RRListener *listener, GError **error);

/* same as g_source_remove except that it uses rr_get_main_context */
static gboolean
source_remove (guint tag)
{
	GSource *source;
  
	g_return_val_if_fail (tag > 0, FALSE);

	source = g_main_context_find_source_by_id (rr_get_main_context (), tag);
	if (source)
		g_source_destroy (source);
	
	return source != NULL;
}

static guint
add_watch_full (GIOChannel *channel, gint priority, GIOCondition condition, 
		GIOFunc func, gpointer user_data, GDestroyNotify notify)
{
	GSource *source;
	guint id;

	source = g_io_create_watch (channel, condition);
	if (priority != G_PRIORITY_DEFAULT)
		g_source_set_priority (source, priority);
	g_source_set_callback (source, (GSourceFunc)func, user_data, notify);
	id = g_source_attach (source, rr_get_main_context ());
	g_source_unref (source);

	return id;
}

static void
rr_tcp_listener_init (GObject *object)
{
	RRTCPListener *tcpl = (RRTCPListener *)object;

	g_static_mutex_init (&tcpl->in_active);
	g_static_mutex_init (&tcpl->err_active);
}

static void
rr_tcp_listener_class_init (GObjectClass *klass)
{
	RRListenerClass *listener_class = (RRListenerClass *)klass;
	klass->finalize = finalize;

	listener_class->shutdown = tcp_shutdown;

	parent_class = g_type_class_peek_parent (klass);
}

GType 
rr_tcp_listener_get_type (void)
{
	static GType rr_type = 0;

	if (!rr_type) {
		static GTypeInfo type_info = {
			sizeof (RRTCPListenerClass),
			NULL,
			NULL,
			(GClassInitFunc) rr_tcp_listener_class_init,
			NULL,
			NULL,
			sizeof (RRTCPListener),
			16,
			(GInstanceInitFunc) rr_tcp_listener_init
		};
		rr_type = g_type_register_static (RR_TYPE_LISTENER, 
						  "RRTCPListener", 
						  &type_info, 0);
#ifdef G_PLATFORM_WIN32
	rr_win32_init_winsock ();
#endif
	}
	return rr_type;
}

static gboolean
in_event (GIOChannel *source, GIOCondition condition, gpointer data)
{
	RRListener *listener = RR_LISTENER (data);
	RRTCPConnection *tcpc;
        struct sockaddr_in inet_addr;
#ifdef G_PLATFORM_WIN32
	int addrlen;
#else
        socklen_t addrlen;
#endif
        gint fd, sock;

	g_return_val_if_fail (condition == G_IO_IN, FALSE);

	sock = g_io_channel_unix_get_fd (source);
        
        addrlen = sizeof(struct sockaddr_in);
        /* Accept the connection */
        if ((fd = accept(sock, (struct sockaddr *)&inet_addr, 
			 &addrlen)) < 0) {

		g_warning ("accept () failed\n");
                return TRUE;
        }
	
	/* FIXME: we should probably send an error message here */
	if (listener->max_connections != -1 &&
	    listener->num_connections >= listener->max_connections) {

		close (fd);
		return TRUE;
	}
	tcpc = rr_tcp_connection_new_unconnected (NULL);

	rr_listener_add_connection (listener, RR_CONNECTION (tcpc));

	rr_tcp_connection_connect_fd (tcpc, fd, RR_ROLE_LISTENER, NULL);
	g_object_unref (G_OBJECT (tcpc));

	return TRUE;
}

static gboolean
error_event (GIOChannel *source, GIOCondition condition, gpointer data)
{
	rr_debug2 ("listener::error_event: condition = %0x\n", condition);

	return FALSE;
}

static void
in_removed (gpointer data)
{
	RRTCPListener *tcpl = RR_TCP_LISTENER (data);

	g_static_mutex_unlock (&tcpl->in_active);
}

static void
err_removed (gpointer data)
{
	RRTCPListener *tcpl = RR_TCP_LISTENER (data);

	g_static_mutex_unlock (&tcpl->err_active);
}

RRListener *
rr_tcp_listener_new (RRProfileRegistry *profreg,
		     const gchar *hostname, 
		     gint port, GError **error)
{
	RRTCPListener *tcpl;
	RRListener *listener;
	struct hostent *he;
        struct in_addr *haddr;
        struct sockaddr_in saddr;
	gint fd;
	gint one = 1;


        he = gethostbyname(hostname);
        if (he == NULL) {
		g_set_error (error,
			     RR_ERROR,                /* error domain */
			     RR_ERROR_GETHOSTBYNAME,  /* error code */
			     "gethostbyname() failed"); /* error message */
		return NULL;
        }

        haddr = ((struct in_addr *) (he->h_addr_list)[0]);

        if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		g_set_error (error,
			     RR_ERROR,                /* error domain */
			     RR_ERROR_SOCKET,         /* error code */
			     "socket() failed");      /* error message */

                return NULL;
        }
        setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &one, sizeof (one));

	memset(&saddr, 0, sizeof(struct sockaddr_in));

	saddr.sin_family = AF_INET;
        saddr.sin_port = htons(port);
/* 	saddr.sin_addr.s_addr = INADDR_ANY; */
        memcpy(&saddr.sin_addr, haddr, sizeof(struct in_addr));

        if (bind(fd, (struct sockaddr *)&saddr, 
                 sizeof (struct sockaddr_in)) < 0) {
		
		g_set_error (error,
			     RR_ERROR,                /* error domain */
			     RR_ERROR_BIND,           /* error code */
			     "bind() failed");        /* error message */
		
                return NULL;
        }
 
	if (listen(fd, BACKLOG) < 0) {

		g_set_error (error,
			     RR_ERROR,                /* error domain */
			     RR_ERROR_LISTEN,         /* error code */
			     "listen() failed");      /* error message */
		
                return NULL;
        }

	tcpl = g_object_new (RR_TYPE_TCP_LISTENER, NULL);
	listener = RR_LISTENER (tcpl);

	if (profreg) {
		rr_listener_set_profile_registry (listener, profreg);
		g_object_unref (G_OBJECT (profreg));
	}

	tcpl->iochannel = g_io_channel_unix_new (fd);
	g_io_channel_set_close_on_unref (tcpl->iochannel, TRUE);
	g_io_channel_set_encoding (tcpl->iochannel, NULL, NULL);

	g_static_mutex_lock (&tcpl->err_active);
	tcpl->err_event = add_watch_full (tcpl->iochannel, 0,
					  G_IO_HUP | G_IO_ERR,
					  error_event, tcpl,
					  err_removed);

	g_static_mutex_lock (&tcpl->in_active);
	tcpl->in_event = add_watch_full (tcpl->iochannel, 0,
					 G_IO_IN, 
					 in_event, tcpl,
					 in_removed);

	rr_debug2 ("listener::listen ('%s', %d)", hostname, port);

	return listener;
}

gboolean
rr_tcp_listener_unlisten (RRTCPListener *tcpl, GError **error)
{
	if (tcpl->in_event) {
		source_remove (tcpl->in_event);
		tcpl->in_event = 0;
	}
	if (tcpl->err_event) {
		source_remove (tcpl->err_event);
		tcpl->err_event = 0;
	}
	if (tcpl->iochannel) {
		g_io_channel_unref (tcpl->iochannel);
		tcpl->iochannel = NULL;
	}
	return TRUE;
}

static gboolean
tcp_shutdown (RRListener *listener, GError **error)
{
	RRTCPListener *tcpl = RR_TCP_LISTENER (listener);

	if (!rr_tcp_listener_unlisten (tcpl, error))
		return FALSE;

	return RR_LISTENER_CLASS (parent_class)->shutdown (listener, error);
}

static void
finalize (GObject *object)
{
	RRTCPListener *tcpl = RR_TCP_LISTENER (object);

	rr_tcp_listener_unlisten (tcpl, NULL);

	g_static_mutex_free (&tcpl->in_active);
	g_static_mutex_free (&tcpl->err_active);

	parent_class->finalize (object);
}
