/*-
 * $Id: rr-connection.h,v 1.31 2002/11/21 17:04:47 jonas Exp $
 *
 * See the file LICENSE for redistribution information. 
 * If you have not received a copy of the license, please contact CodeFactory
 * by email at info@codefactory.se, or on the web at http://www.codefactory.se/
 * You may also write to: CodeFactory AB, SE-903 47, Ume, Sweden.
 *
 * Copyright (c) 2002 Jonas Borgstrm <jonas@codefactory.se>
 * Copyright (c) 2002 Daniel Lundin   <daniel@codefactory.se>
 * Copyright (c) 2002 CodeFactory AB.  All rights reserved.
 */

#ifndef __RR_CONNECTION_H__
#define __RR_CONNECTION_H__

typedef struct _RRConnection RRConnection;
typedef struct _RRConnectionClass RRConnectionClass;

#include <glib-object.h>
#include <librr/rr-channel.h>
#include <librr/rr-frame.h>
#include <librr/rr-profileregistry.h>
#include <librr/rr-listener.h>
#include <librr/rr-filterstack.h>
#include <librr/rr-callbacklist.h>

#include <librr/rr-manager.h>

G_BEGIN_DECLS

#define RR_TYPE_CONNECTION (rr_connection_get_type ())
#define RR_CONNECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST((obj), RR_TYPE_CONNECTION, RRConnection))
#define RR_CONNECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), RR_TYPE_CONNECTION, RRConnectionClass))
#define RR_IS_CONNECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE((obj), RR_TYPE_CONNECTION))
#define RR_IS_CONNECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), RR_TYPE_CONNECTION))
#define RR_CONNECTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), RR_TYPE_CONNECTION, RRConnectionClass))

typedef enum {
	RR_ROLE_LISTENER,
	RR_ROLE_INITIATOR
} RRRole;

struct _RRConnection {
	GObject parent_object;
	
	RRListener *listener;

	RRProfileRegistry *profreg;

	GStaticRWLock channel_lock;
	GHashTable *channel_by_id;
	RRRole role;
	RRManager *manager;

	RRFilterStack *filter_stack;

	GStaticRWLock languages_lock;
	GSList *languages; /* Requested languages after greeting */

	GMutex *out_queue_lock; /* protects out_queue and out_frames */
	GCond *out_queue_cond;
	GPtrArray *out_queue;  /* List of channels with active messages
				* in their out queue */
	GQueue *out_frames;    /* The frame queue has higher priority then the
				  out_queue */
	gint active_item;
	gboolean output_disabled;
	GSList *peer_profiles;
	char *server_name;
	RRCallbackList *quiescence_cb;
	gboolean connected;
};

struct _RRConnectionClass {
	GObjectClass parent_class;

	gboolean (* enable_input)   (RRConnection *connection);
	gboolean (* disable_input)  (RRConnection *connection);
	gboolean (* enable_output)  (RRConnection *connection);
	gboolean (* disable_output) (RRConnection *connection);
	gboolean (* disconnect)     (RRConnection *connection, GError **error);
};

GType rr_connection_get_type (void);

RRManager *rr_connection_get_manager (RRConnection *connection);
gboolean   rr_connection_disconnect (RRConnection *connection, GError **error);

RRChannel *rr_connection_start (RRConnection *connection, const gchar *server_name,
				GType profile_type, gpointer config_data, GError **error);

RRChannel *rr_connection_start_multi (RRConnection *connection, const gchar *server_name,
				      GError **error, ...);

void rr_connection_set_peer_profiles (RRConnection *connection, GSList *list);

const GSList *rr_connection_get_peer_profiles (RRConnection *connection);

gboolean rr_connection_peer_supports_profile (RRConnection *connection,
					      GType profile);

void rr_connection_do_quiescence (RRConnection *connection, GFunc callback,
				  gpointer data, gpointer user_data);
gboolean rr_connection_wait_quiescence (RRConnection *connection, 
					GError **error);

RRChannel *rr_connection_get_channel (RRConnection *connection, gint id);
RRChannel *rr_connection_get_channel_locked (RRConnection *connection, gint id);


void rr_connection_set_profile_registry (RRConnection *connection, 
					 RRProfileRegistry *profreg);

const gchar *rr_connection_get_server_name (RRConnection *connection);

void rr_connection_set_server_name (RRConnection *connection, 
				    const gchar *server_name);

gboolean rr_connection_send_frame (RRConnection *connection, 
				   RRFrame *frame, 
				   GError **error);

/* Tuning reset related functions */

gboolean rr_connection_begin_tuning_reset (RRConnection *connection, 
					   GError **error);

void rr_connection_complete_tuning_reset (RRConnection *connection,
					  RRChannel *channel);

void rr_connection_register_sender (RRConnection *connection, 
				    RRChannel *channel);

void rr_connection_add_channel (RRConnection *connection, RRChannel *channel);
void rr_connection_remove_channel (RRConnection *connection, RRChannel *channel);
gboolean rr_connection_enable_input   (RRConnection *connection);
gboolean rr_connection_disable_input  (RRConnection *connection);

gboolean rr_connection_enable_output  (RRConnection *connection);
gboolean rr_connection_disable_output (RRConnection *connection);

gboolean rr_connection_pending_transmissions_p (RRConnection *connection);

RRFrame *rr_connection_get_next_frame (RRConnection *connection, 
				       gsize buffer_size);

GSList * rr_connection_get_languages (RRConnection *connection);
gchar *rr_connection_get_languages_str (RRConnection *connection);

gboolean rr_connection_language_supported (RRConnection *connection, 
					   const gchar *lang);

void rr_connection_add_language (RRConnection *connection, 
				 const gchar *lang);

gboolean rr_connection_remove_language (RRConnection *connection, 
					const gchar *lang);


void rr_connection_close_all (RRConnection *conn);

G_END_DECLS

#endif /* __RR_CONNECTION_H__ */
