// SPDX-FileCopyrightText: 2023 Bastian Engel <bastian.engel00@gmail.com>
// SPDX-License-Identifier: LGPL-3.0-only

#include "rl78_maps.h"

/*
 * The 'User's Manual: Software' for the RL78 Family arranges instructions in
 * four instruction maps, each indexed by an unsigned byte. Thus, the entry at
 * index [0xXYZ] corresponds to entry YZ of instruction map (X + 1). Empty
 * positions are left out
 */

RL78Instr rl78_instr_maps[4 * 256] = {
	// map 0
	// 0x000-0x0f
	[0x000] = { .operation = RL78_OPERATION_NOP },
	[0x001] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_ADDW },
	[0x002] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_ADDW },
	[0x003] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		.operation = RL78_OPERATION_ADDW },
	[0x004] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_IMMEDIATE_16 },
		.operation = RL78_OPERATION_ADDW },
	[0x005] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_ADDW },
	[0x006] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_ADDW },
	[0x007] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_ADDW },
	[0x008] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_XCH },
	[0x009] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_MOV },
	[0x00A] = { { .type = RL78_OP_TYPE_SADDR },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_ADD },
	[0x00B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_ADD },
	[0x00C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{
			.type = RL78_OP_TYPE_IMMEDIATE_8,
		},
		.operation = RL78_OPERATION_ADD },
	[0x00D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_ADD },
	[0x00E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_ADD },
	[0x00F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_ADD },

	// 0x010-0x01f
	[0x010] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_CR_SP },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_ADDW },
	[0x012] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_MOVW },
	[0x013] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		.operation = RL78_OPERATION_MOVW },
	[0x014] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_DE },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_MOVW },
	[0x015] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_MOVW },
	[0x016] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_HL },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_MOVW },
	[0x017] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_MOVW },
	[0x018] = { { .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x019] = { { .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x01A] = { { .type = RL78_OP_TYPE_SADDR },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_ADDC },
	[0x01B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_ADDC },
	[0x01C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{
			.type = RL78_OP_TYPE_IMMEDIATE_8,
		},
		.operation = RL78_OPERATION_ADDC },
	[0x01D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_ADDC },
	[0x01E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_ADDC },
	[0x01F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_ADDC },

	// 0x020-0x02f
	[0x020] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_CR_SP },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_SUBW },
	[0x022] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_SUBW },
	[0x023] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		.operation = RL78_OPERATION_SUBW },
	[0x024] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_IMMEDIATE_16 },
		.operation = RL78_OPERATION_SUBW },
	[0x025] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_SUBW },
	[0x026] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_SUBW },
	[0x027] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_SUBW },
	[0x028] = { { .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x029] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_MOV },
	[0x02A] = { { .type = RL78_OP_TYPE_SADDR },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_SUB },
	[0x02B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_SUB },
	[0x02C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{
			.type = RL78_OP_TYPE_IMMEDIATE_8,
		},
		.operation = RL78_OPERATION_SUB },
	[0x02D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_SUB },
	[0x02E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_SUB },
	[0x02F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_SUB },

	// 0x030-0x03f
	[0x030] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_IMMEDIATE_16 },
		.operation = RL78_OPERATION_MOVW },
	[0x032] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_IMMEDIATE_16 },
		.operation = RL78_OPERATION_MOVW },
	[0x033] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		.operation = RL78_OPERATION_XCHW },
	[0x034] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_DE },
		{ .type = RL78_OP_TYPE_IMMEDIATE_16 },
		.operation = RL78_OPERATION_MOVW },
	[0x035] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_XCHW },
	[0x036] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_HL },
		{ .type = RL78_OP_TYPE_IMMEDIATE_16 },
		.operation = RL78_OPERATION_MOVW },
	[0x037] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_XCHW },
	[0x038] = { { .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x039] = { { .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x03A] = { { .type = RL78_OP_TYPE_SADDR },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_SUBC },
	[0x03B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_SUBC },
	[0x03C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{
			.type = RL78_OP_TYPE_IMMEDIATE_8,
		},
		.operation = RL78_OPERATION_SUBC },
	[0x03D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_SUBC },
	[0x03E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_SUBC },
	[0x03F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_SUBC },

	// 0x040-0x04f
	[0x040] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_CMP },
	[0x041] = { { .type = RL78_OP_TYPE_SFR, .v0 = RL78_SFR_ES },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x042] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_CMPW },
	[0x043] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		.operation = RL78_OPERATION_CMPW },
	[0x044] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_IMMEDIATE_16 },
		.operation = RL78_OPERATION_CMPW },
	[0x045] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_CMPW },
	[0x046] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_CMPW },
	[0x047] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_CMPW },
	[0x048] = { { .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x049] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_BC },
		.operation = RL78_OPERATION_MOV },
	[0x04A] = { { .type = RL78_OP_TYPE_SADDR },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_CMP },
	[0x04B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_CMP },
	[0x04C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{
			.type = RL78_OP_TYPE_IMMEDIATE_8,
		},
		.operation = RL78_OPERATION_CMP },
	[0x04D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_CMP },
	[0x04E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_CMP },
	[0x04F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_CMP },

	// 0x050-0x05f
	[0x050] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x051] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x052] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x053] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x054] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x055] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x056] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x057] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x058] = { { .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_MOVW },
	[0x059] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_MOVW },
	[0x05A] = { { .type = RL78_OP_TYPE_SADDR },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_AND },
	[0x05B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_AND },
	[0x05C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{
			.type = RL78_OP_TYPE_IMMEDIATE_8,
		},
		.operation = RL78_OPERATION_AND },
	[0x05D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_AND },
	[0x05E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_AND },
	[0x05F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_AND },

	// 0x060-0x06f
	[0x060] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_MOV },
	[0x062] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_MOV },
	[0x063] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_MOV },
	[0x064] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		.operation = RL78_OPERATION_MOV },
	[0x065] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		.operation = RL78_OPERATION_MOV },
	[0x066] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		.operation = RL78_OPERATION_MOV },
	[0x067] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		.operation = RL78_OPERATION_MOV },
	[0x068] = { { .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_MOVW },
	[0x069] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_MOVW },
	[0x06A] = { { .type = RL78_OP_TYPE_SADDR },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_OR },
	[0x06B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_OR },
	[0x06C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{
			.type = RL78_OP_TYPE_IMMEDIATE_8,
		},
		.operation = RL78_OPERATION_OR },
	[0x06D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_OR },
	[0x06E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_OR },
	[0x06F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_OR },

	// 0x070-0x07f
	[0x070] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x072] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x073] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x074] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x075] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x076] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x077] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x078] = { { .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_MOVW },
	[0x079] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_BASED_ADDR_16, .v0 = RL78_GPR_BC },
		.operation = RL78_OPERATION_MOVW },
	[0x07A] = { { .type = RL78_OP_TYPE_SADDR },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_XOR },
	[0x07B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_XOR },
	[0x07C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{
			.type = RL78_OP_TYPE_IMMEDIATE_8,
		},
		.operation = RL78_OPERATION_XOR },
	[0x07D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_XOR },
	[0x07E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_XOR },
	[0x07F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_XOR },

	// 0x080-0x08f
	[0x080] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_INC },
	[0x081] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_INC },
	[0x082] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_INC },
	[0x083] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_INC },
	[0x084] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		.operation = RL78_OPERATION_INC },
	[0x085] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		.operation = RL78_OPERATION_INC },
	[0x086] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		.operation = RL78_OPERATION_INC },
	[0x087] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		.operation = RL78_OPERATION_INC },
	[0x088] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_CR_SP },
		.operation = RL78_OPERATION_MOV },
	[0x089] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_MOV },
	[0x08A] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_MOV },
	[0x08B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_MOV },
	[0x08C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_MOV },
	[0x08D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_MOV },
	[0x08E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SFR },
		.operation = RL78_OPERATION_MOV },
	[0x08F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_MOV },

	// 0x090-0x09f
	[0x090] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_DEC },
	[0x091] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_DEC },
	[0x092] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_DEC },
	[0x093] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_DEC },
	[0x094] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		.operation = RL78_OPERATION_DEC },
	[0x095] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		.operation = RL78_OPERATION_DEC },
	[0x096] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		.operation = RL78_OPERATION_DEC },
	[0x097] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		.operation = RL78_OPERATION_DEC },
	[0x098] = { { .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_CR_SP },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x099] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_DE },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x09A] = { { .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_DE },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x09B] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_HL },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x09C] = { { .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x09D] = { { .type = RL78_OP_TYPE_SADDR },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x09E] = { { .type = RL78_OP_TYPE_SFR },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x09F] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },

	// 0x0A0-0x0af
	[0x0A0] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_INC },
	[0x0A1] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_INCW },
	[0x0A2] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_INCW },
	[0x0A3] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		.operation = RL78_OPERATION_INCW },
	[0x0A4] = { { .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_INC },
	[0x0A5] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_INCW },
	[0x0A6] = { { .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_INCW },
	[0x0A7] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_INCW },
	[0x0A8] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_CR_SP },
		.operation = RL78_OPERATION_MOVW },
	[0x0A9] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_MOVW },
	[0x0AA] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_MOVW },
	[0x0AB] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_MOVW },
	[0x0AC] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_MOVW },
	[0x0AD] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_MOVW },
	[0x0AE] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_SFR },
		.operation = RL78_OPERATION_MOVW },
	[0x0AF] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_MOVW },

	// 0x0B0-0x0bf
	[0x0B0] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_DEC },
	[0x0B1] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_DECW },
	[0x0B2] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_DECW },
	[0x0B3] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		.operation = RL78_OPERATION_DECW },
	[0x0B4] = { { .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_DEC },
	[0x0B5] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_DECW },
	[0x0B6] = { { .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_DECW },
	[0x0B7] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_DECW },
	[0x0B8] = { { .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_CR_SP },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_MOVW },
	[0x0B9] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_DE },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_MOVW },
	[0x0BA] = { { .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_DE },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_MOVW },
	[0x0BB] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_HL },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_MOVW },
	[0x0BC] = { { .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_MOVW },
	[0x0BD] = { { .type = RL78_OP_TYPE_SADDR },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_MOVW },
	[0x0BE] = { { .type = RL78_OP_TYPE_SFR },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_MOVW },
	[0x0BF] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_MOVW },

	// 0x0C0-0x0cf
	[0x0C0] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_POP },
	[0x0C1] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_PUSH },
	[0x0C2] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		.operation = RL78_OPERATION_POP },
	[0x0C3] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		.operation = RL78_OPERATION_PUSH },
	[0x0C4] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_POP },
	[0x0C5] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_PUSH },
	[0x0C6] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_POP },
	[0x0C7] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_PUSH },
	[0x0C8] = { { .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_CR_SP },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x0C9] = { { .type = RL78_OP_TYPE_SADDR },
		{ .type = RL78_OP_TYPE_IMMEDIATE_16 },
		.operation = RL78_OPERATION_MOVW },
	[0x0CA] = { { .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_DE },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x0CB] = { { .type = RL78_OP_TYPE_SFR },
		{ .type = RL78_OP_TYPE_IMMEDIATE_16 },
		.operation = RL78_OPERATION_MOVW },
	[0x0CC] = { { .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x0CD] = { { .type = RL78_OP_TYPE_SADDR },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x0CE] = { { .type = RL78_OP_TYPE_SFR },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },
	[0x0CF] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		{ .type = RL78_OP_TYPE_IMMEDIATE_8 },
		.operation = RL78_OPERATION_MOV },

	// 0X0D0-0x0df
	[0x0D0] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_CMP0 },
	[0x0D1] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_CMP0 },
	[0x0D2] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_CMP0 },
	[0x0D3] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_CMP0 },
	[0x0D4] = { { .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_CMP0 },
	[0x0D5] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_CMP0 },
	[0x0D6] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_MULU },
	[0x0D7] = { .operation = RL78_OPERATION_RET },
	[0x0D8] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_MOV },
	[0x0D9] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_MOV },
	[0x0DA] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_MOVW },
	[0x0DB] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_MOVW },
	[0x0DC] = { { .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BC },
	[0x0DD] = { { .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BZ },
	[0x0DE] = { { .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BNC },
	[0x0DF] = { { .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BNZ },

	// 0X0E0-0x0ef
	[0x0E0] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_ONEB },
	[0x0E1] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ONEB },
	[0x0E2] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_ONEB },
	[0x0E3] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_ONEB },
	[0x0E4] = { { .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_ONEB },
	[0x0E5] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_ONEB },
	[0x0E6] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_ONEW },
	[0x0E7] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		.operation = RL78_OPERATION_ONEW },
	[0x0E8] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_MOV },
	[0x0E9] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_MOV },
	[0x0EA] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_DE },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_MOVW },
	[0x0EB] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_DE },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_MOVW },
	[0x0EC] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_20 },
		.operation = RL78_OPERATION_BR },
	[0x0ED] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_BR },
	[0x0EE] = { { .type = RL78_OP_TYPE_RELATIVE_ADDR_16 },
		.operation = RL78_OPERATION_BR },
	[0x0EF] = { { .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BR },

	// 0X0F0-0x0ff
	[0x0F0] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_CLRB },
	[0x0F1] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_CLRB },
	[0x0F2] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_CLRB },
	[0x0F3] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_CLRB },
	[0x0F4] = { { .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_CLRB },
	[0x0F5] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_CLRB },
	[0x0F6] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_CLRW },
	[0x0F7] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		.operation = RL78_OPERATION_CLRW },
	[0x0F8] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_MOV },
	[0x0F9] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_MOV },
	[0x0FA] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_HL },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_MOVW },
	[0x0FB] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_HL },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_MOVW },
	[0x0FC] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_20 },
		.operation = RL78_OPERATION_CALL },
	[0x0FD] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_CALL },
	[0x0FE] = { { .type = RL78_OP_TYPE_RELATIVE_ADDR_16 },
		.operation = RL78_OPERATION_CALL },

	// map 1
	// 0x100-0x10f
	[0x100] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADD },
	[0x101] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADD },
	[0x102] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADD },
	[0x103] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADD },
	[0x104] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADD },
	[0x105] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADD },
	[0x106] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADD },
	[0x107] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADD },
	[0x108] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_ADD },
	[0x109] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_ADDW },
	[0x10A] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_ADD },
	[0x10B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_ADD },
	[0x10C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		.operation = RL78_OPERATION_ADD },
	[0x10D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		.operation = RL78_OPERATION_ADD },
	[0x10E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		.operation = RL78_OPERATION_ADD },
	[0x10F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		.operation = RL78_OPERATION_ADD },

	// 0x110-0x11f
	[0x110] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADDC },
	[0x111] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADDC },
	[0x112] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADDC },
	[0x113] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADDC },
	[0x114] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADDC },
	[0x115] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADDC },
	[0x116] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADDC },
	[0x117] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_ADDC },
	[0x118] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_ADDC },
	[0x11A] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_ADDC },
	[0x11B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_ADDC },
	[0x11C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		.operation = RL78_OPERATION_ADDC },
	[0x11D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		.operation = RL78_OPERATION_ADDC },
	[0x11E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		.operation = RL78_OPERATION_ADDC },
	[0x11F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		.operation = RL78_OPERATION_ADDC },

	// 0x120-0x12f
	[0x120] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUB },
	[0x121] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUB },
	[0x122] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUB },
	[0x123] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUB },
	[0x124] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUB },
	[0x125] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUB },
	[0x126] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUB },
	[0x127] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUB },
	[0x128] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_SUB },
	[0x129] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_SUBW },
	[0x12A] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_SUB },
	[0x12B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_SUB },
	[0x12C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		.operation = RL78_OPERATION_SUB },
	[0x12D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		.operation = RL78_OPERATION_SUB },
	[0x12E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		.operation = RL78_OPERATION_SUB },
	[0x12F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		.operation = RL78_OPERATION_SUB },

	// 0x130-0x13f
	[0x130] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUBC },
	[0x131] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUBC },
	[0x132] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUBC },
	[0x133] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUBC },
	[0x134] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUBC },
	[0x135] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUBC },
	[0x136] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUBC },
	[0x137] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_SUBC },
	[0x138] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_SUBC },
	[0x13A] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_SUBC },
	[0x13B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_SUBC },
	[0x13C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		.operation = RL78_OPERATION_SUBC },
	[0x13D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		.operation = RL78_OPERATION_SUBC },
	[0x13E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		.operation = RL78_OPERATION_SUBC },
	[0x13F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		.operation = RL78_OPERATION_SUBC },

	// 0x140-0x14f
	[0x140] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_CMP },
	[0x141] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_CMP },
	[0x142] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_CMP },
	[0x143] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_CMP },
	[0x144] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_CMP },
	[0x145] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_CMP },
	[0x146] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_CMP },
	[0x147] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_CMP },
	[0x148] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_CMP },
	[0x149] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_CMPW },
	[0x14A] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_CMP },
	[0x14B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_CMP },
	[0x14C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		.operation = RL78_OPERATION_CMP },
	[0x14D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		.operation = RL78_OPERATION_CMP },
	[0x14E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		.operation = RL78_OPERATION_CMP },
	[0x14F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		.operation = RL78_OPERATION_CMP },

	// 0x150-0x15f
	[0x150] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_AND },
	[0x151] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_AND },
	[0x152] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_AND },
	[0x153] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_AND },
	[0x154] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_AND },
	[0x155] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_AND },
	[0x156] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_AND },
	[0x157] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_AND },
	[0x158] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_AND },
	[0x159] = { { .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_INC },
	[0x15A] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_AND },
	[0x15B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_AND },
	[0x15C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		.operation = RL78_OPERATION_AND },
	[0x15D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		.operation = RL78_OPERATION_AND },
	[0x15E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		.operation = RL78_OPERATION_AND },
	[0x15F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		.operation = RL78_OPERATION_AND },

	// 0x160-0x16f
	[0x160] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_OR },
	[0x161] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_OR },
	[0x162] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_OR },
	[0x163] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_OR },
	[0x164] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_OR },
	[0x165] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_OR },
	[0x166] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_OR },
	[0x167] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_OR },
	[0x168] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_OR },
	[0x169] = { { .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_DEC },
	[0x16A] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_OR },
	[0x16B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_OR },
	[0x16C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		.operation = RL78_OPERATION_OR },
	[0x16D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		.operation = RL78_OPERATION_OR },
	[0x16E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		.operation = RL78_OPERATION_OR },
	[0x16F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		.operation = RL78_OPERATION_OR },

	// 0x170-0x17f
	[0x170] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_XOR },
	[0x171] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_XOR },
	[0x172] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_XOR },
	[0x173] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_XOR },
	[0x174] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_XOR },
	[0x175] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_XOR },
	[0x176] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_XOR },
	[0x177] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_XOR },
	[0x178] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_XOR },
	[0x179] = { { .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_INCW },
	[0x17A] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_XOR },
	[0x17B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_XOR },
	[0x17C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		.operation = RL78_OPERATION_XOR },
	[0x17D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		.operation = RL78_OPERATION_XOR },
	[0x17E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		.operation = RL78_OPERATION_XOR },
	[0x17F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		.operation = RL78_OPERATION_XOR },

	// 0x180-0x18f
	[0x180] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_B },
		.operation = RL78_OPERATION_ADD },
	[0x182] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_C },
		.operation = RL78_OPERATION_ADD },
	[0x184] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x0080 },
		.operation = RL78_OPERATION_CALLT },
	[0x185] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x0090 },
		.operation = RL78_OPERATION_CALLT },
	[0x186] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00A0 },
		.operation = RL78_OPERATION_CALLT },
	[0x187] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00B0 },
		.operation = RL78_OPERATION_CALLT },
	[0x189] = { { .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_DECW },
	[0x18A] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		.operation = RL78_OPERATION_XCH },
	[0x18B] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		.operation = RL78_OPERATION_XCH },
	[0x18C] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_E },
		.operation = RL78_OPERATION_XCH },
	[0x18D] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_D },
		.operation = RL78_OPERATION_XCH },
	[0x18E] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_L },
		.operation = RL78_OPERATION_XCH },
	[0x18F] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_H },
		.operation = RL78_OPERATION_XCH },

	// 0x190-0x19f
	[0x190] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_B },
		.operation = RL78_OPERATION_ADDC },
	[0x192] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_C },
		.operation = RL78_OPERATION_ADDC },
	[0x194] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x0082 },
		.operation = RL78_OPERATION_CALLT },
	[0x195] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x0092 },
		.operation = RL78_OPERATION_CALLT },
	[0x196] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00A2 },
		.operation = RL78_OPERATION_CALLT },
	[0x197] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00B2 },
		.operation = RL78_OPERATION_CALLT },

	// 0x1a0-0x1af
	[0x1a0] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_B },
		.operation = RL78_OPERATION_SUB },
	[0x1a2] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_C },
		.operation = RL78_OPERATION_SUB },
	[0x1a4] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x0084 },
		.operation = RL78_OPERATION_CALLT },
	[0x1a5] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x0094 },
		.operation = RL78_OPERATION_CALLT },
	[0x1a6] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00A4 },
		.operation = RL78_OPERATION_CALLT },
	[0x1a7] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00B4 },
		.operation = RL78_OPERATION_CALLT },
	[0x1a8] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_XCH },
	[0x1a9] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_C },
		.operation = RL78_OPERATION_XCH },
	[0x1aa] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16 },
		.operation = RL78_OPERATION_XCH },
	[0x1ab] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_SFR },
		.operation = RL78_OPERATION_XCH },
	[0x1ac] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_XCH },
	[0x1ad] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_XCH },
	[0x1ae] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_XCH },
	[0x1af] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_XCH },

	// 0x1b0-0x1bf
	[0x1b0] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_B },
		.operation = RL78_OPERATION_SUBC },
	[0x1b2] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_C },
		.operation = RL78_OPERATION_SUBC },
	[0x1b4] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x0086 },
		.operation = RL78_OPERATION_CALLT },
	[0x1b5] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x0096 },
		.operation = RL78_OPERATION_CALLT },
	[0x1b6] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00A6 },
		.operation = RL78_OPERATION_CALLT },
	[0x1b7] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00B6 },
		.operation = RL78_OPERATION_CALLT },
	[0x1b8] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_SFR_ES },
		{ .type = RL78_OP_TYPE_SADDR },
		.operation = RL78_OPERATION_MOV },
	[0x1b9] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_B },
		.operation = RL78_OPERATION_XCH },

	// 0x1c0-0x1cf
	[0x1c0] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_B },
		.operation = RL78_OPERATION_CMP },
	[0x1c2] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_C },
		.operation = RL78_OPERATION_CMP },
	[0x1c3] = { { .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BH },
	[0x1c4] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x0088 },
		.operation = RL78_OPERATION_CALLT },
	[0x1c5] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x0098 },
		.operation = RL78_OPERATION_CALLT },
	[0x1c6] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00A8 },
		.operation = RL78_OPERATION_CALLT },
	[0x1c7] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00B8 },
		.operation = RL78_OPERATION_CALLT },
	[0x1c8] = { .operation = RL78_OPERATION_SKC },
	[0x1c9] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_B },
		.operation = RL78_OPERATION_MOV },
	[0x1ca] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_CALL },
	[0x1cb] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		.operation = RL78_OPERATION_BR },
	[0x1cc] = { .operation = RL78_OPERATION_BRK },
	[0x1cd] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_SFR_PSW },
		.operation = RL78_OPERATION_POP },
	[0x1ce] = { { .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		.operation = RL78_OPERATION_MOVS },
	[0x1cf] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_RB_RB0 },
		.operation = RL78_OPERATION_SEL },

	// 0x1d0-0x1df
	[0x1d0] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_B },
		.operation = RL78_OPERATION_AND },
	[0x1d2] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_C },
		.operation = RL78_OPERATION_AND },
	[0x1d3] = { { .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BNH },
	[0x1d4] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x008A },
		.operation = RL78_OPERATION_CALLT },
	[0x1d5] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x009A },
		.operation = RL78_OPERATION_CALLT },
	[0x1d6] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00AA },
		.operation = RL78_OPERATION_CALLT },
	[0x1d7] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00BA },
		.operation = RL78_OPERATION_CALLT },
	[0x1d8] = { .operation = RL78_OPERATION_SKNC },
	[0x1d9] = { { .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x1da] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		.operation = RL78_OPERATION_CALL },
	[0x1db] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_ROR },
	[0x1dc] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_ROLC },
	[0x1dd] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_SFR_PSW },
		.operation = RL78_OPERATION_PUSH },
	[0x1de] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_X },
		{ .type = RL78_OP_TYPE_BASED_ADDR_8, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_CMPS },
	[0x1df] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_RB_RB1 },
		.operation = RL78_OPERATION_SEL },

	// 0x1e0-0x1ef
	[0x1e0] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_B },
		.operation = RL78_OPERATION_OR },
	[0x1e2] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_C },
		.operation = RL78_OPERATION_OR },
	[0x1e3] = { .operation = RL78_OPERATION_SKH },
	[0x1e4] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x008C },
		.operation = RL78_OPERATION_CALLT },
	[0x1e5] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x009C },
		.operation = RL78_OPERATION_CALLT },
	[0x1e6] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00AC },
		.operation = RL78_OPERATION_CALLT },
	[0x1e7] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00BC },
		.operation = RL78_OPERATION_CALLT },
	[0x1e8] = { .operation = RL78_OPERATION_SKZ },
	[0x1e9] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_C },
		.operation = RL78_OPERATION_MOV },
	[0x1ea] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_DE },
		.operation = RL78_OPERATION_CALL },
	[0x1eb] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_ROL },
	[0x1ec] = { .operation = RL78_OPERATION_RETB },
	[0x1ed] = { .operation = RL78_OPERATION_HALT },
	[0x1ee] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_ROLWC },
	[0x1ef] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_RB_RB2 },
		.operation = RL78_OPERATION_SEL },

	// 0x1f0-0x1ff
	[0x1f0] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_B },
		.operation = RL78_OPERATION_XOR },
	[0x1f2] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_C },
		.operation = RL78_OPERATION_XOR },
	[0x1f3] = { .operation = RL78_OPERATION_SKNH },
	[0x1f4] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x008E },
		.operation = RL78_OPERATION_CALLT },
	[0x1f5] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x009E },
		.operation = RL78_OPERATION_CALLT },
	[0x1f6] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00AE },
		.operation = RL78_OPERATION_CALLT },
	[0x1f7] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .v0 = 0x00BE },
		.operation = RL78_OPERATION_CALLT },
	[0x1f8] = { .operation = RL78_OPERATION_SKNZ },
	[0x1f9] = { { .type = RL78_OP_TYPE_BASED_INDEX_ADDR, .v0 = RL78_GPR_HL, .v1 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		.operation = RL78_OPERATION_MOV },
	[0x1fa] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_HL },
		.operation = RL78_OPERATION_CALL },
	[0x1fb] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_RORC },
	[0x1fc] = { .operation = RL78_OPERATION_RETI },
	[0x1fd] = { .operation = RL78_OPERATION_STOP },
	[0x1fe] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_ROLWC },
	[0x1ff] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_RB_RB3 },
		.operation = RL78_OPERATION_SEL },

	// map 2
	// 0x200-0x20f
	[0x200] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		.operation = RL78_OPERATION_SET1 },
	[0x201] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x202] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		.operation = RL78_OPERATION_SET1 },
	[0x203] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		.operation = RL78_OPERATION_CLR1 },
	[0x204] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		.operation = RL78_OPERATION_MOV1 },
	[0x205] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		.operation = RL78_OPERATION_AND1 },
	[0x206] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		.operation = RL78_OPERATION_OR1 },
	[0x207] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		.operation = RL78_OPERATION_XOR1 },
	[0x208] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		.operation = RL78_OPERATION_CLR1 },
	[0x209] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x20a] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		.operation = RL78_OPERATION_SET1 },
	[0x20b] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		.operation = RL78_OPERATION_CLR1 },
	[0x20c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		.operation = RL78_OPERATION_MOV1 },
	[0x20d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		.operation = RL78_OPERATION_AND1 },
	[0x20e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		.operation = RL78_OPERATION_OR1 },
	[0x20f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x210-0x21f
	[0x210] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		.operation = RL78_OPERATION_SET1 },
	[0x211] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x212] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		.operation = RL78_OPERATION_SET1 },
	[0x213] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		.operation = RL78_OPERATION_CLR1 },
	[0x214] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		.operation = RL78_OPERATION_MOV1 },
	[0x215] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		.operation = RL78_OPERATION_AND1 },
	[0x216] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		.operation = RL78_OPERATION_OR1 },
	[0x217] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		.operation = RL78_OPERATION_XOR1 },
	[0x218] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		.operation = RL78_OPERATION_CLR1 },
	[0x219] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x21a] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		.operation = RL78_OPERATION_SET1 },
	[0x21b] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		.operation = RL78_OPERATION_CLR1 },
	[0x21c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		.operation = RL78_OPERATION_MOV1 },
	[0x21d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		.operation = RL78_OPERATION_AND1 },
	[0x21e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		.operation = RL78_OPERATION_OR1 },
	[0x21f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x220-0x22f
	[0x220] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		.operation = RL78_OPERATION_SET1 },
	[0x221] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x222] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		.operation = RL78_OPERATION_SET1 },
	[0x223] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		.operation = RL78_OPERATION_CLR1 },
	[0x224] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		.operation = RL78_OPERATION_MOV1 },
	[0x225] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		.operation = RL78_OPERATION_AND1 },
	[0x226] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		.operation = RL78_OPERATION_OR1 },
	[0x227] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		.operation = RL78_OPERATION_XOR1 },
	[0x228] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		.operation = RL78_OPERATION_CLR1 },
	[0x229] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x22a] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		.operation = RL78_OPERATION_SET1 },
	[0x22b] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		.operation = RL78_OPERATION_CLR1 },
	[0x22c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		.operation = RL78_OPERATION_MOV1 },
	[0x22d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		.operation = RL78_OPERATION_AND1 },
	[0x22e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		.operation = RL78_OPERATION_OR1 },
	[0x22f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x230-0x23f
	[0x230] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		.operation = RL78_OPERATION_SET1 },
	[0x231] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x232] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		.operation = RL78_OPERATION_SET1 },
	[0x233] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		.operation = RL78_OPERATION_CLR1 },
	[0x234] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		.operation = RL78_OPERATION_MOV1 },
	[0x235] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		.operation = RL78_OPERATION_AND1 },
	[0x236] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		.operation = RL78_OPERATION_OR1 },
	[0x237] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		.operation = RL78_OPERATION_XOR1 },
	[0x238] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		.operation = RL78_OPERATION_CLR1 },
	[0x239] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x23a] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		.operation = RL78_OPERATION_SET1 },
	[0x23b] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		.operation = RL78_OPERATION_CLR1 },
	[0x23c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		.operation = RL78_OPERATION_MOV1 },
	[0x23d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		.operation = RL78_OPERATION_AND1 },
	[0x23e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		.operation = RL78_OPERATION_OR1 },
	[0x23f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x240-0x24f
	[0x240] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		.operation = RL78_OPERATION_SET1 },
	[0x241] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x242] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		.operation = RL78_OPERATION_SET1 },
	[0x243] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		.operation = RL78_OPERATION_CLR1 },
	[0x244] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		.operation = RL78_OPERATION_MOV1 },
	[0x245] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		.operation = RL78_OPERATION_AND1 },
	[0x246] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		.operation = RL78_OPERATION_OR1 },
	[0x247] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		.operation = RL78_OPERATION_XOR1 },
	[0x248] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		.operation = RL78_OPERATION_CLR1 },
	[0x249] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x24a] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		.operation = RL78_OPERATION_SET1 },
	[0x24b] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		.operation = RL78_OPERATION_CLR1 },
	[0x24c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		.operation = RL78_OPERATION_MOV1 },
	[0x24d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		.operation = RL78_OPERATION_AND1 },
	[0x24e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		.operation = RL78_OPERATION_OR1 },
	[0x24f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x250-0x25f
	[0x250] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		.operation = RL78_OPERATION_SET1 },
	[0x251] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x252] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		.operation = RL78_OPERATION_SET1 },
	[0x253] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		.operation = RL78_OPERATION_CLR1 },
	[0x254] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		.operation = RL78_OPERATION_MOV1 },
	[0x255] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		.operation = RL78_OPERATION_AND1 },
	[0x256] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		.operation = RL78_OPERATION_OR1 },
	[0x257] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		.operation = RL78_OPERATION_XOR1 },
	[0x258] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		.operation = RL78_OPERATION_CLR1 },
	[0x259] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x25a] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		.operation = RL78_OPERATION_SET1 },
	[0x25b] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		.operation = RL78_OPERATION_CLR1 },
	[0x25c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		.operation = RL78_OPERATION_MOV1 },
	[0x25d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		.operation = RL78_OPERATION_AND1 },
	[0x25e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		.operation = RL78_OPERATION_OR1 },
	[0x25f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x260-0x26f
	[0x260] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		.operation = RL78_OPERATION_SET1 },
	[0x261] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x262] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		.operation = RL78_OPERATION_SET1 },
	[0x263] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		.operation = RL78_OPERATION_CLR1 },
	[0x264] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		.operation = RL78_OPERATION_MOV1 },
	[0x265] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		.operation = RL78_OPERATION_AND1 },
	[0x266] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		.operation = RL78_OPERATION_OR1 },
	[0x267] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		.operation = RL78_OPERATION_XOR1 },
	[0x268] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		.operation = RL78_OPERATION_CLR1 },
	[0x269] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x26a] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		.operation = RL78_OPERATION_SET1 },
	[0x26b] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		.operation = RL78_OPERATION_CLR1 },
	[0x26c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		.operation = RL78_OPERATION_MOV1 },
	[0x26d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		.operation = RL78_OPERATION_AND1 },
	[0x26e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		.operation = RL78_OPERATION_OR1 },
	[0x26f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x270-0x27f
	[0x270] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		.operation = RL78_OPERATION_SET1 },
	[0x271] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x272] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		.operation = RL78_OPERATION_SET1 },
	[0x273] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		.operation = RL78_OPERATION_CLR1 },
	[0x274] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		.operation = RL78_OPERATION_MOV1 },
	[0x275] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		.operation = RL78_OPERATION_AND1 },
	[0x276] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		.operation = RL78_OPERATION_OR1 },
	[0x277] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		.operation = RL78_OPERATION_XOR1 },
	[0x278] = { { .type = RL78_OP_TYPE_ABSOLUTE_ADDR_16, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		.operation = RL78_OPERATION_CLR1 },
	[0x279] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x27a] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		.operation = RL78_OPERATION_SET1 },
	[0x27b] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		.operation = RL78_OPERATION_CLR1 },
	[0x27c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		.operation = RL78_OPERATION_MOV1 },
	[0x27d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		.operation = RL78_OPERATION_AND1 },
	[0x27e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		.operation = RL78_OPERATION_OR1 },
	[0x27f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x280-0x28f
	[0x280] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_SET1 },
	[0x281] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 0 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x282] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 0 },
		.operation = RL78_OPERATION_SET1 },
	[0x283] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 0 },
		.operation = RL78_OPERATION_CLR1 },
	[0x284] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 0 },
		.operation = RL78_OPERATION_MOV1 },
	[0x285] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 0 },
		.operation = RL78_OPERATION_AND1 },
	[0x286] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 0 },
		.operation = RL78_OPERATION_OR1 },
	[0x287] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 0 },
		.operation = RL78_OPERATION_XOR1 },
	[0x288] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_CLR1 },
	[0x289] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 0 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x28a] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 0 },
		.operation = RL78_OPERATION_SET1 },
	[0x28b] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 0 },
		.operation = RL78_OPERATION_CLR1 },
	[0x28c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 0 },
		.operation = RL78_OPERATION_MOV1 },
	[0x28d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 0 },
		.operation = RL78_OPERATION_AND1 },
	[0x28e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 0 },
		.operation = RL78_OPERATION_OR1 },
	[0x28f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 0 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x280-0x28f
	[0x291] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 1 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x292] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 1 },
		.operation = RL78_OPERATION_SET1 },
	[0x293] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 1 },
		.operation = RL78_OPERATION_CLR1 },
	[0x294] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 1 },
		.operation = RL78_OPERATION_MOV1 },
	[0x295] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 1 },
		.operation = RL78_OPERATION_AND1 },
	[0x296] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 1 },
		.operation = RL78_OPERATION_OR1 },
	[0x297] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 1 },
		.operation = RL78_OPERATION_XOR1 },
	[0x299] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 1 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x29a] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 1 },
		.operation = RL78_OPERATION_SET1 },
	[0x29b] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 1 },
		.operation = RL78_OPERATION_CLR1 },
	[0x29c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 1 },
		.operation = RL78_OPERATION_MOV1 },
	[0x29d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 1 },
		.operation = RL78_OPERATION_AND1 },
	[0x29e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 1 },
		.operation = RL78_OPERATION_OR1 },
	[0x29f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 1 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x280-0x28f
	[0x2a1] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 2 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x2a2] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 2 },
		.operation = RL78_OPERATION_SET1 },
	[0x2a3] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 2 },
		.operation = RL78_OPERATION_CLR1 },
	[0x2a4] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 2 },
		.operation = RL78_OPERATION_MOV1 },
	[0x2a5] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 2 },
		.operation = RL78_OPERATION_AND1 },
	[0x2a6] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 2 },
		.operation = RL78_OPERATION_OR1 },
	[0x2a7] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 2 },
		.operation = RL78_OPERATION_XOR1 },
	[0x2a9] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 2 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x2aa] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 2 },
		.operation = RL78_OPERATION_SET1 },
	[0x2ab] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 2 },
		.operation = RL78_OPERATION_CLR1 },
	[0x2ac] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 2 },
		.operation = RL78_OPERATION_MOV1 },
	[0x2ad] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 2 },
		.operation = RL78_OPERATION_AND1 },
	[0x2ae] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 2 },
		.operation = RL78_OPERATION_OR1 },
	[0x2af] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 2 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x280-0x28f
	[0x2b1] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 3 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x2b2] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 3 },
		.operation = RL78_OPERATION_SET1 },
	[0x2b3] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 3 },
		.operation = RL78_OPERATION_CLR1 },
	[0x2b4] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 3 },
		.operation = RL78_OPERATION_MOV1 },
	[0x2b5] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 3 },
		.operation = RL78_OPERATION_AND1 },
	[0x2b6] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 3 },
		.operation = RL78_OPERATION_OR1 },
	[0x2b7] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 3 },
		.operation = RL78_OPERATION_XOR1 },
	[0x2b9] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 3 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x2ba] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 3 },
		.operation = RL78_OPERATION_SET1 },
	[0x2bb] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 3 },
		.operation = RL78_OPERATION_CLR1 },
	[0x2bc] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 3 },
		.operation = RL78_OPERATION_MOV1 },
	[0x2bd] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 3 },
		.operation = RL78_OPERATION_AND1 },
	[0x2be] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 3 },
		.operation = RL78_OPERATION_OR1 },
	[0x2bf] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 3 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x280-0x28f
	[0x2c0] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_NOT1 },
	[0x2c1] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 4 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x2c2] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 4 },
		.operation = RL78_OPERATION_SET1 },
	[0x2c3] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 4 },
		.operation = RL78_OPERATION_CLR1 },
	[0x2c4] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 4 },
		.operation = RL78_OPERATION_MOV1 },
	[0x2c5] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 4 },
		.operation = RL78_OPERATION_AND1 },
	[0x2c6] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 4 },
		.operation = RL78_OPERATION_OR1 },
	[0x2c7] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 4 },
		.operation = RL78_OPERATION_XOR1 },
	[0x2c9] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 4 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x2ca] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 4 },
		.operation = RL78_OPERATION_SET1 },
	[0x2cb] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 4 },
		.operation = RL78_OPERATION_CLR1 },
	[0x2cc] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 4 },
		.operation = RL78_OPERATION_MOV1 },
	[0x2cd] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 4 },
		.operation = RL78_OPERATION_AND1 },
	[0x2ce] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 4 },
		.operation = RL78_OPERATION_OR1 },
	[0x2cf] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 4 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x280-0x28f
	[0x2d1] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 5 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x2d2] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 5 },
		.operation = RL78_OPERATION_SET1 },
	[0x2d3] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 5 },
		.operation = RL78_OPERATION_CLR1 },
	[0x2d4] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 5 },
		.operation = RL78_OPERATION_MOV1 },
	[0x2d5] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 5 },
		.operation = RL78_OPERATION_AND1 },
	[0x2d6] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 5 },
		.operation = RL78_OPERATION_OR1 },
	[0x2d7] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 5 },
		.operation = RL78_OPERATION_XOR1 },
	[0x2d9] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 5 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x2da] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 5 },
		.operation = RL78_OPERATION_SET1 },
	[0x2db] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 5 },
		.operation = RL78_OPERATION_CLR1 },
	[0x2dc] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 5 },
		.operation = RL78_OPERATION_MOV1 },
	[0x2dd] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 5 },
		.operation = RL78_OPERATION_AND1 },
	[0x2de] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 5 },
		.operation = RL78_OPERATION_OR1 },
	[0x2df] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 5 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x280-0x28f
	[0x2e1] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 6 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x2e2] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 6 },
		.operation = RL78_OPERATION_SET1 },
	[0x2e3] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 6 },
		.operation = RL78_OPERATION_CLR1 },
	[0x2e4] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 6 },
		.operation = RL78_OPERATION_MOV1 },
	[0x2e5] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 6 },
		.operation = RL78_OPERATION_AND1 },
	[0x2e6] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 6 },
		.operation = RL78_OPERATION_OR1 },
	[0x2e7] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 6 },
		.operation = RL78_OPERATION_XOR1 },
	[0x2e9] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 6 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x2ea] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 6 },
		.operation = RL78_OPERATION_SET1 },
	[0x2eb] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 6 },
		.operation = RL78_OPERATION_CLR1 },
	[0x2ec] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 6 },
		.operation = RL78_OPERATION_MOV1 },
	[0x2ed] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 6 },
		.operation = RL78_OPERATION_AND1 },
	[0x2ee] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 6 },
		.operation = RL78_OPERATION_OR1 },
	[0x2ef] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 6 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x280-0x28f
	[0x2f1] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 7 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x2f2] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 7 },
		.operation = RL78_OPERATION_SET1 },
	[0x2f3] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 7 },
		.operation = RL78_OPERATION_CLR1 },
	[0x2f4] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 7 },
		.operation = RL78_OPERATION_MOV1 },
	[0x2f5] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 7 },
		.operation = RL78_OPERATION_AND1 },
	[0x2f6] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 7 },
		.operation = RL78_OPERATION_OR1 },
	[0x2f7] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 7 },
		.operation = RL78_OPERATION_XOR1 },
	[0x2f9] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 7 },
		{ .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		.operation = RL78_OPERATION_MOV1 },
	[0x2fa] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 7 },
		.operation = RL78_OPERATION_SET1 },
	[0x2fb] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 7 },
		.operation = RL78_OPERATION_CLR1 },
	[0x2fc] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 7 },
		.operation = RL78_OPERATION_MOV1 },
	[0x2fd] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 7 },
		.operation = RL78_OPERATION_AND1 },
	[0x2fe] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 7 },
		.operation = RL78_OPERATION_OR1 },
	[0x2ff] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_PSW_CY },
		{ .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 7 },
		.operation = RL78_OPERATION_XOR1 },

	// 0x300-0x30f
	[0x300] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x301] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 0 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x302] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x303] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 0 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x304] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x305] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 0 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },

	// 0x310-0x31f
	[0x310] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x311] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 1 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x312] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x313] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 1 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x314] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x315] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 1 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x317] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_SHL },
	[0x318] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_SHL },
	[0x319] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_SHL },
	[0x31a] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_SHR },
	[0x31b] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_SAR },
	[0x31c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_SHLW },
	[0x31d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_SHLW },
	[0x31e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_SHRW },
	[0x31f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 1 },
		.operation = RL78_OPERATION_SARW },

	// 0x320-0x32f
	[0x320] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x321] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 2 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x322] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x323] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 2 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x324] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x325] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 2 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x327] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 2 },
		.operation = RL78_OPERATION_SHL },
	[0x328] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 2 },
		.operation = RL78_OPERATION_SHL },
	[0x329] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 2 },
		.operation = RL78_OPERATION_SHL },
	[0x32a] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 2 },
		.operation = RL78_OPERATION_SHR },
	[0x32b] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 2 },
		.operation = RL78_OPERATION_SAR },
	[0x32c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 2 },
		.operation = RL78_OPERATION_SHLW },
	[0x32d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 2 },
		.operation = RL78_OPERATION_SHLW },
	[0x32e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 2 },
		.operation = RL78_OPERATION_SHRW },
	[0x32f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 2 },
		.operation = RL78_OPERATION_SARW },

	// 0x330-0x33f
	[0x330] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x331] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 3 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x332] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x333] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 3 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x334] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x335] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 3 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x337] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 3 },
		.operation = RL78_OPERATION_SHL },
	[0x338] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 3 },
		.operation = RL78_OPERATION_SHL },
	[0x339] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 3 },
		.operation = RL78_OPERATION_SHL },
	[0x33a] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 3 },
		.operation = RL78_OPERATION_SHR },
	[0x33b] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 3 },
		.operation = RL78_OPERATION_SAR },
	[0x33c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 3 },
		.operation = RL78_OPERATION_SHLW },
	[0x33d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 3 },
		.operation = RL78_OPERATION_SHLW },
	[0x33e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 3 },
		.operation = RL78_OPERATION_SHRW },
	[0x33f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 3 },
		.operation = RL78_OPERATION_SARW },

	// 0x340-0x34f
	[0x340] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x341] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 4 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x342] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x343] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 4 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x344] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x345] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 4 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x347] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 4 },
		.operation = RL78_OPERATION_SHL },
	[0x348] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 4 },
		.operation = RL78_OPERATION_SHL },
	[0x349] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 4 },
		.operation = RL78_OPERATION_SHL },
	[0x34a] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 4 },
		.operation = RL78_OPERATION_SHR },
	[0x34b] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 4 },
		.operation = RL78_OPERATION_SAR },
	[0x34c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 4 },
		.operation = RL78_OPERATION_SHLW },
	[0x34d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 4 },
		.operation = RL78_OPERATION_SHLW },
	[0x34e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 4 },
		.operation = RL78_OPERATION_SHRW },
	[0x34f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 4 },
		.operation = RL78_OPERATION_SARW },

	// 0x350-0x35f
	[0x350] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x351] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 5 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x352] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x353] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 5 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x354] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x355] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 5 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x357] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 5 },
		.operation = RL78_OPERATION_SHL },
	[0x358] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 5 },
		.operation = RL78_OPERATION_SHL },
	[0x359] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 5 },
		.operation = RL78_OPERATION_SHL },
	[0x35a] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 5 },
		.operation = RL78_OPERATION_SHR },
	[0x35b] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 5 },
		.operation = RL78_OPERATION_SAR },
	[0x35c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 5 },
		.operation = RL78_OPERATION_SHLW },
	[0x35d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 5 },
		.operation = RL78_OPERATION_SHLW },
	[0x35e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 5 },
		.operation = RL78_OPERATION_SHRW },
	[0x35f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 5 },
		.operation = RL78_OPERATION_SARW },

	// 0x360-0x36f
	[0x360] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x361] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 6 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x362] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x363] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 6 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x364] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x365] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 6 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x367] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 6 },
		.operation = RL78_OPERATION_SHL },
	[0x368] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 6 },
		.operation = RL78_OPERATION_SHL },
	[0x369] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 6 },
		.operation = RL78_OPERATION_SHL },
	[0x36a] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 6 },
		.operation = RL78_OPERATION_SHR },
	[0x36b] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 6 },
		.operation = RL78_OPERATION_SAR },
	[0x36c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 6 },
		.operation = RL78_OPERATION_SHLW },
	[0x36d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 6 },
		.operation = RL78_OPERATION_SHLW },
	[0x36e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 6 },
		.operation = RL78_OPERATION_SHRW },
	[0x36f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 6 },
		.operation = RL78_OPERATION_SARW },

	// 0x370-0x37f
	[0x370] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x371] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 7 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x372] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x373] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 7 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x374] = { { .type = RL78_OP_TYPE_SADDR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x375] = { { .type = RL78_OP_TYPE_SYMBOL, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_A, .v1 = 7 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x377] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_C },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 7 },
		.operation = RL78_OPERATION_SHL },
	[0x378] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_B },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 7 },
		.operation = RL78_OPERATION_SHL },
	[0x379] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 7 },
		.operation = RL78_OPERATION_SHL },
	[0x37a] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 7 },
		.operation = RL78_OPERATION_SHR },
	[0x37b] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_A },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 7 },
		.operation = RL78_OPERATION_SAR },
	[0x37c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 7 },
		.operation = RL78_OPERATION_SHLW },
	[0x37d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 7 },
		.operation = RL78_OPERATION_SHLW },
	[0x37e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 7 },
		.operation = RL78_OPERATION_SHRW },
	[0x37f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 7 },
		.operation = RL78_OPERATION_SARW },

	// 0x380-0x38f
	[0x380] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x381] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 0 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x382] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x383] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 0 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x384] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 0 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x385] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 0 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x38c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 8 },
		.operation = RL78_OPERATION_SHLW },
	[0x38d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 8 },
		.operation = RL78_OPERATION_SHLW },
	[0x38e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 8 },
		.operation = RL78_OPERATION_SHRW },
	[0x38f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 8 },
		.operation = RL78_OPERATION_SARW },

	// 0x390-0x39f
	[0x390] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x391] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 1 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x392] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x393] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 1 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x394] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 1 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x395] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 1 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x39c] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 9 },
		.operation = RL78_OPERATION_SHLW },
	[0x39d] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 9 },
		.operation = RL78_OPERATION_SHLW },
	[0x39e] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 9 },
		.operation = RL78_OPERATION_SHRW },
	[0x39f] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 9 },
		.operation = RL78_OPERATION_SARW },

	// 0x3a0-0x3af
	[0x3a0] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x3a1] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 2 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x3a2] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x3a3] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 2 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x3a4] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 2 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x3a5] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 2 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x3ac] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 10 },
		.operation = RL78_OPERATION_SHLW },
	[0x3ad] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 10 },
		.operation = RL78_OPERATION_SHLW },
	[0x3ae] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 10 },
		.operation = RL78_OPERATION_SHRW },
	[0x3af] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 10 },
		.operation = RL78_OPERATION_SARW },

	// 0x3b0-0x3bf
	[0x3b0] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x3b1] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 3 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x3b2] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x3b3] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 3 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x3b4] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 3 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x3b5] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 3 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x3bc] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 11 },
		.operation = RL78_OPERATION_SHLW },
	[0x3bd] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 11 },
		.operation = RL78_OPERATION_SHLW },
	[0x3be] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 11 },
		.operation = RL78_OPERATION_SHRW },
	[0x3bf] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 11 },
		.operation = RL78_OPERATION_SARW },

	// 0x3c0-0x3cf
	[0x3c0] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x3c1] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 4 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x3c2] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x3c3] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 4 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x3c4] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 4 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x3c5] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 4 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x3cc] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 12 },
		.operation = RL78_OPERATION_SHLW },
	[0x3cd] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 12 },
		.operation = RL78_OPERATION_SHLW },
	[0x3ce] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 12 },
		.operation = RL78_OPERATION_SHRW },
	[0x3cf] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 12 },
		.operation = RL78_OPERATION_SARW },

	// 0x3d0-0x3df
	[0x3d0] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x3d1] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 5 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x3d2] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x3d3] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 5 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x3d4] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 5 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x3d5] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 5 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x3dc] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 13 },
		.operation = RL78_OPERATION_SHLW },
	[0x3dd] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 13 },
		.operation = RL78_OPERATION_SHLW },
	[0x3de] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 13 },
		.operation = RL78_OPERATION_SHRW },
	[0x3df] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 13 },
		.operation = RL78_OPERATION_SARW },

	// 0x3e0-0x3ef
	[0x3e0] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x3e1] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 6 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x3e2] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x3e3] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 6 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x3e4] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 6 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x3e5] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 6 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x3ec] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 14 },
		.operation = RL78_OPERATION_SHLW },
	[0x3ed] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 14 },
		.operation = RL78_OPERATION_SHLW },
	[0x3ee] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 14 },
		.operation = RL78_OPERATION_SHRW },
	[0x3ef] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 14 },
		.operation = RL78_OPERATION_SARW },

	// 0x3f0-0x3ff
	[0x3f0] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x3f1] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 7 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BTCLR },
	[0x3f2] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x3f3] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 7 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BT },
	[0x3f4] = { { .type = RL78_OP_TYPE_SFR, .flags = RL78_OP_FLAG_BA, .v1 = 7 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x3f5] = { { .type = RL78_OP_TYPE_INDIRECT_ADDR, .flags = RL78_OP_FLAG_BA, .v0 = RL78_GPR_HL, .v1 = 7 },
		{ .type = RL78_OP_TYPE_RELATIVE_ADDR_8 },
		.operation = RL78_OPERATION_BF },
	[0x3fc] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_BC },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 15 },
		.operation = RL78_OPERATION_SHLW },
	[0x3fd] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 15 },
		.operation = RL78_OPERATION_SHLW },
	[0x3fe] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 15 },
		.operation = RL78_OPERATION_SHRW },
	[0x3ff] = { { .type = RL78_OP_TYPE_SYMBOL, .v0 = RL78_GPR_AX },
		{ .type = RL78_OP_TYPE_DECIMAL, .v0 = 15 },
		.operation = RL78_OPERATION_SARW },
};
