/*
 * YARA rules for x86 PE compiler detection.
 * Copyright (c) 2017 Avast Software, licensed under the MIT license
 */

import "pe"

rule fasm_uv_01 {
	meta:
		tool = "C"
		name = "FASM"
		pattern = "6A00"
	strings:
		$1 = { 6A 00 }
	condition:
		$1 at pe.entry_point
}

rule fasm_uv_02 {
	meta:
		tool = "C"
		name = "FASM"
		pattern = "6A24"
	strings:
		$1 = { 6A 24 }
	condition:
		$1 at pe.entry_point
}

rule fasm_13x {
	meta:
		tool = "C"
		name = "FASM"
		version = "1.3x"
		pattern = "6A??FF15????????A3"
	strings:
		$1 = { 6A ?? FF 15 ?? ?? ?? ?? A3 }
	condition:
		$1 at pe.entry_point
}

rule fasm_15x {
	meta:
		tool = "C"
		name = "FASM"
		version = "1.5x"
		pattern = "6A00FF15????4000A3????4000????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????00"
	strings:
		$1 = { 6A 00 FF 15 ?? ?? 40 00 A3 ?? ?? 40 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 }
	condition:
		$1 at pe.entry_point
}

rule masm_tasm_uv_01 {
	meta:
		tool = "C"
		name = "MASM or TASM"
		pattern = "6A00E8??0?0000A3????4000???????0?0????000000????0??????0?????0?0???????0??0????0?000"
	strings:
		$1 = { 6A 00 E8 ?? 0? 00 00 A3 ?? ?? 40 00 ?? ?? ?? ?0 ?0 ?? ?? 00 00 00 ?? ?? 0? ?? ?? ?0 ?? ?? ?0 ?0 ?? ?? ?? ?0 ?? 0? ?? ?0 ?0 00 }
	condition:
		$1 at pe.entry_point
}

rule masm_tasm_uv_02 {
	meta:
		tool = "C"
		name = "MASM or TASM"
		pattern = "C2??00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00"
	strings:
		$1 = { C2 ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 }
	condition:
		$1 at pe.entry_point
}

rule masm_tasm_uv_03 {
	meta:
		tool = "C"
		name = "MASM or TASM"
		pattern = "CCFF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00"
	strings:
		$1 = { CC FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 }
	condition:
		$1 at pe.entry_point
}

rule masm_tasm_uv_04 {
	meta:
		tool = "C"
		name = "MASM or TASM"
		pattern = "FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00FF25??????00"
	strings:
		$1 = { FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 FF 25 ?? ?? ?? 00 }
	condition:
		$1 at pe.entry_point
}

rule masm32 {
	meta:
		tool = "C"
		name = "MASM32"
		pattern = "6A??680030400068??3040006A00E8070000006A00E806000000FF250820"
	strings:
		$1 = { 6A ?? 68 00 30 40 00 68 ?? 30 40 00 6A 00 E8 07 00 00 00 6A 00 E8 06 00 00 00 FF 25 08 20 }
	condition:
		$1 at pe.entry_point
}

rule aut2exe_264
{
	meta:
		tool = "C"
		name = "Aut2Exe"
		version = "2.64"
		language = "AutoIt"
		bytecode = true
		source = "Made by RetDec Team"
		pattern = "6A606808FD4000E8733D0000BF940000008BC7E84FF6FFFF8965E88BF4893E56FF1558F140008B4E10890D6C3544008B4604A3783544008B560889157C3544008B760C81E6FF7F000089357035440083F902740C81CE00800000893570354400C1E00803"
	strings:
		$1 = { 6A 60 68 08 FD 40 00 E8 73 3D 00 00 BF 94 00 00 00 8B C7 E8 4F F6 FF FF 89 65 E8 8B F4 89 3E 56 FF 15 58 F1 40 00 8B 4E 10 89 0D 6C 35 44 00 8B 46 04 A3 78 35 44 00 8B 56 08 89 15 7C 35 44 00 8B 76 0C 81 E6 FF 7F 00 00 89 35 70 35 44 00 83 F9 02 74 0C 81 CE 00 80 00 00 89 35 70 35 44 00 C1 E0 08 03 }
	condition:
		$1 at pe.entry_point
}

rule aut2exe_3300
{
	meta:
		tool = "C"
		name = "Aut2Exe"
		version = "3.3.0.0"
		language = "AutoIt"
		bytecode = true
		source = "Made by RetDec Team"
		pattern = "E8C4AF0000E979FEFFFF8BFF558BEC8BC18B4D08C70088DA47008B09836008008948045DC208008BFF558BEC538B5D08568BF1C70688DA47008B430889460885C08B430457743185C0742750E8EFD3FFFF8BF84757E810D3FFFF595989460485C07418FF"
	strings:
		$1 = { E8 C4 AF 00 00 E9 79 FE FF FF 8B FF 55 8B EC 8B C1 8B 4D 08 C7 00 88 DA 47 00 8B 09 83 60 08 00 89 48 04 5D C2 08 00 8B FF 55 8B EC 53 8B 5D 08 56 8B F1 C7 06 88 DA 47 00 8B 43 08 89 46 08 85 C0 8B 43 04 57 74 31 85 C0 74 27 50 E8 EF D3 FF FF 8B F8 47 57 E8 10 D3 FF FF 59 59 89 46 04 85 C0 74 18 FF }
	condition:
		$1 at pe.entry_point
}

rule aut2exe_338x {
	meta:
		tool = "C"
		name = "Aut2Exe"
		version = "3.3.8.x"
		language = "AutoIt"
		bytecode = true
		source = "Made by RetDec Team"
		pattern = "E816900000E989FEFFFFCCCCCCCCCC558BEC57568B750C8B4D108B7D088BC18BD103C63BFE76083BF80F82A001000081F980000000721C833D24974A00007413575683E70F83E60F3BFE5E5F7505E9DD030000F7C7030000007514C1E90283E20383F908"
	strings:
		$1 = { E8 16 90 00 00 E9 89 FE FF FF CC CC CC CC CC 55 8B EC 57 56 8B 75 0C 8B 4D 10 8B 7D 08 8B C1 8B D1 03 C6 3B FE 76 08 3B F8 0F 82 A0 01 00 00 81 F9 80 00 00 00 72 1C 83 3D 24 97 4A 00 00 74 13 57 56 83 E7 0F 83 E6 0F 3B FE 5E 5F 75 05 E9 DD 03 00 00 F7 C7 03 00 00 00 75 14 C1 E9 02 83 E2 03 83 F9 08 }
	condition:
		$1 at pe.entry_point
}

rule aut2exe_33100 {
	meta:
		tool = "C"
		name = "Aut2Exe"
		version = "3.3.10.0"
		language = "AutoIt"
		bytecode = true
		source = "Made by RetDec Team"
		pattern = "E88ACF0000E97FFEFFFFCCCC57568B7424108B4C24148B7C240C8BC18BD103C63BFE76083BF80F82680300000FBA2558114C00017307F3A4E91703000081F9800000000F82CE0100008BC733C6A90F000000750E0FBA2570B34B00010F82DA0400000FBA"
	strings:
		$1 = { E8 8A CF 00 00 E9 7F FE FF FF CC CC 57 56 8B 74 24 10 8B 4C 24 14 8B 7C 24 0C 8B C1 8B D1 03 C6 3B FE 76 08 3B F8 0F 82 68 03 00 00 0F BA 25 58 11 4C 00 01 73 07 F3 A4 E9 17 03 00 00 81 F9 80 00 00 00 0F 82 CE 01 00 00 8B C7 33 C6 A9 0F 00 00 00 75 0E 0F BA 25 70 B3 4B 00 01 0F 82 DA 04 00 00 0F BA }
	condition:
		$1 at pe.entry_point
}

rule aut2exe_33102 {
	meta:
		tool = "C"
		name = "Aut2Exe"
		version = "3.3.10.2"
		language = "AutoIt"
		bytecode = true
		source = "Made by RetDec Team"
		pattern = "E897CF0000E97FFEFFFFCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC57568B7424108B4C24148B7C240C8BC18BD103C63BFE76083BF80F82680300000FBA2558014C00017307F3A4E91703000081F9800000000F82CE0100008BC733C6A90F000000750E0FBA25"
	strings:
		$1 = { E8 97 CF 00 00 E9 7F FE FF FF CC CC CC CC CC CC CC CC CC CC CC CC CC CC CC 57 56 8B 74 24 10 8B 4C 24 14 8B 7C 24 0C 8B C1 8B D1 03 C6 3B FE 76 08 3B F8 0F 82 68 03 00 00 0F BA 25 58 01 4C 00 01 73 07 F3 A4 E9 17 03 00 00 81 F9 80 00 00 00 0F 82 CE 01 00 00 8B C7 33 C6 A9 0F 00 00 00 75 0E 0F BA 25 }
	condition:
		$1 at pe.entry_point
}

rule aut2exe_33140 {
	meta:
		tool = "C"
		name = "Aut2Exe"
		version = "3.3.14.0"
		language = "AutoIt"
		bytecode = true
		source = "Made by RetDec Team"
		pattern = "E8B5D00000E97FFEFFFFCCCCCCCCCCCCCCCCCC57568B7424108B4C24148B7C240C8BC18BD103C63BFE76083BF80F82680300000FBA25FC314C00017307F3A4E91703000081F9800000000F82CE0100008BC733C6A90F000000750E0FBA2524E34B00010F"
	strings:
		$1 = { E8 B5 D0 00 00 E9 7F FE FF FF CC CC CC CC CC CC CC CC CC 57 56 8B 74 24 10 8B 4C 24 14 8B 7C 24 0C 8B C1 8B D1 03 C6 3B FE 76 08 3B F8 0F 82 68 03 00 00 0F BA 25 FC 31 4C 00 01 73 07 F3 A4 E9 17 03 00 00 81 F9 80 00 00 00 0F 82 CE 01 00 00 8B C7 33 C6 A9 0F 00 00 00 75 0E 0F BA 25 24 E3 4B 00 01 0F }
	condition:
		$1 at pe.entry_point
}

rule aut2exe_33142 {
	meta:
		tool = "C"
		name = "Aut2Exe"
		version = "3.3.14.2"
		language = "AutoIt"
		bytecode = true
		source = "Made by RetDec Team"
		pattern = "E8B8D00000E97FFEFFFFCCCCCCCCCCCCCCCCCCCCCCCC57568B7424108B4C24148B7C240C8BC18BD103C63BFE76083BF80F82680300000FBA25FC314C00017307F3A4E91703000081F9800000000F82CE0100008BC733C6A90F000000750E0FBA2524E34B"
	strings:
		$1 = { E8 B8 D0 00 00 E9 7F FE FF FF CC CC CC CC CC CC CC CC CC CC CC CC 57 56 8B 74 24 10 8B 4C 24 14 8B 7C 24 0C 8B C1 8B D1 03 C6 3B FE 76 08 3B F8 0F 82 68 03 00 00 0F BA 25 FC 31 4C 00 01 73 07 F3 A4 E9 17 03 00 00 81 F9 80 00 00 00 0F 82 CE 01 00 00 8B C7 33 C6 A9 0F 00 00 00 75 0E 0F BA 25 24 E3 4B }
	condition:
		$1 at pe.entry_point
}

rule aut2exe_33143 {
	meta:
		tool = "C"
		name = "Aut2Exe"
		version = "3.3.14.3"
		language = "AutoIt"
		bytecode = true
		source = "Made by RetDec Team"
		pattern = "E8C8D00000E97FFEFFFFCCCCCCCCCCCCCCCCCCCCCCCC57568B7424108B4C24148B7C240C8BC18BD103C63BFE76083BF80F82680300000FBA25FC414C00017307F3A4E91703000081F9800000000F82CE0100008BC733C6A90F000000750E0FBA2524F34B"
	strings:
		$1 = { E8 C8 D0 00 00 E9 7F FE FF FF CC CC CC CC CC CC CC CC CC CC CC CC 57 56 8B 74 24 10 8B 4C 24 14 8B 7C 24 0C 8B C1 8B D1 03 C6 3B FE 76 08 3B F8 0F 82 68 03 00 00 0F BA 25 FC 41 4C 00 01 73 07 F3 A4 E9 17 03 00 00 81 F9 80 00 00 00 0F 82 CE 01 00 00 8B C7 33 C6 A9 0F 00 00 00 75 0E 0F BA 25 24 F3 4B }
	condition:
		$1 at pe.entry_point
}

rule aut2exe_uv_01 {
	meta:
		tool = "C"
		name = "Aut2Exe"
		language = "AutoIt"
		bytecode = true
	strings:
		$1 = ">AUTOIT SCRIPT<"
		$2 = ">AUTOIT SCRIPT<" wide
		$3 = ">AUTOIT UNICODE SCRIPT<" wide
	condition:
		pe.is_32bit() and
		for 1 of them : (
			@ > pe.sections[pe.section_index(".rdata")].raw_data_offset and
			@ < pe.sections[pe.section_index(".rdata")].raw_data_offset +
			pe.sections[pe.section_index(".rdata")].raw_data_size
		)
}

rule autohotkey_uv_01 {
	meta:
		tool = "C"
		name = "AHK2Exe"
		language = "AutoHotKey"
		bytecode = true
	strings:
		$0 = "Hotkeys/hotstrings are not allowed inside functions." wide ascii
		$1 = "IfWin should be #IfWin." wide ascii
		$2 = "This hotstring is missing its abbreviation." wide ascii
		$3 = "Duplicate hotkey." wide ascii
		$4 = ">AUTOHOTKEY SCRIPT<" wide ascii
    condition:
        pe.is_32bit() 
		and 
		pe.number_of_resources > 0 
		and ((
					(@4 > pe.sections[pe.section_index(".rdata")].raw_data_offset 
					and
					@4 < pe.sections[pe.section_index(".rdata")].raw_data_offset +
					pe.sections[pe.section_index(".rdata")].raw_data_size) 
				or
				(for 1 i in (0 .. pe.number_of_resources) : (
					pe.resources[i].name_string matches />AUTOHOTKEY SCRIPT</))
			)
			or 
			(3 of ($0,$1,$2,$3))
		)
}

rule borland_c {
	meta:
		tool = "C"
		name = "Borland C"
		pattern = "3BCF76052BCFFCF3AA59"
	strings:
		$1 = { 3B CF 76 05 2B CF FC F3 AA 59 }
	condition:
		$1 at pe.entry_point
}

rule borland_cpp_uv_01 {
	meta:
		tool = "C"
		name = "Borland C++"
		pattern = "A1????????C1E002A3"
	strings:
		$1 = { A1 ?? ?? ?? ?? C1 E0 02 A3 }
	condition:
		$1 at pe.entry_point
}

rule borland_cpp_uv_02 {
	meta:
		tool = "C"
		name = "Borland C++"
		extra = "DLL"
		pattern = "EB1066623A432B2B484F4F4B90E9A1C1E002A38B"
	strings:
		$1 = { EB 10 66 62 3A 43 2B 2B 48 4F 4F 4B 90 E9 A1 C1 E0 02 A3 8B }
	condition:
		$1 at pe.entry_point
}

rule borland_cpp_uv_03 {
	meta:
		tool = "C"
		name = "Borland C++"
		pattern = "EB1066623A432B2B484F4F4B90E9????????A1????????C1E002A3????????8B"
	strings:
		$1 = { EB 10 66 62 3A 43 2B 2B 48 4F 4F 4B 90 E9 ?? ?? ?? ?? A1 ?? ?? ?? ?? C1 E0 02 A3 ?? ?? ?? ?? 8B }
	condition:
		$1 at pe.entry_point
}

rule borland_cpp_uv_04 {
	meta:
		tool = "C"
		name = "Borland C++"
		pattern = "575133C0BF????????B9????????3BCF76158BCF81E1????????E30B81E900100000F7D9FCF3AA"
	strings:
		$1 = { 57 51 33 C0 BF ?? ?? ?? ?? B9 ?? ?? ?? ?? 3B CF 76 15 8B CF 81 E1 ?? ?? ?? ?? E3 0B 81 E9 00 10 00 00 F7 D9 FC F3 AA }
	condition:
		for any of them : ( $ in (pe.entry_point .. pe.entry_point + 32) )
}

rule borland_cpp_1991 {
	meta:
		tool = "C"
		name = "Borland C++"
		version = "1991"
		pattern = "2E8C06????2E8C1E????BB????8EDB1EE8????1F"
	strings:
		$1 = { 2E 8C 06 ?? ?? 2E 8C 1E ?? ?? BB ?? ?? 8E DB 1E E8 ?? ?? 1F }
	condition:
		$1 at pe.entry_point
}

rule borland_cpp_1991_1994 {
	meta:
		tool = "C"
		name = "Borland C++"
		version = "1992, 1994"
		pattern = "8CC88ED88C1E????8C06????8C06????8C06"
	strings:
		$1 = { 8C C8 8E D8 8C 1E ?? ?? 8C 06 ?? ?? 8C 06 ?? ?? 8C 06 }
	condition:
		$1 at pe.entry_point
}

rule borland_cpp_1994_01 {
	meta:
		tool = "C"
		name = "Borland C++"
		version = "1994"
		pattern = "8CCA2E89??????B430CD218B2E????8B1E????8EDAA3????8C"
	strings:
		$1 = { 8C CA 2E 89 ?? ?? ?? B4 30 CD 21 8B 2E ?? ?? 8B 1E ?? ?? 8E DA A3 ?? ?? 8C }
	condition:
		$1 at pe.entry_point
}

rule borland_cpp_1994_02 {
	meta:
		tool = "C"
		name = "Borland C++"
		version = "1994"
		pattern = "A1????????C1????A3????????83????????75??575133C0BF"
	strings:
		$1 = { A1 ?? ?? ?? ?? C1 ?? ?? A3 ?? ?? ?? ?? 83 ?? ?? ?? ?? 75 ?? 57 51 33 C0 BF }
	condition:
		$1 at pe.entry_point
}

rule borland_cpp_1995_01 {
	meta:
		tool = "C"
		name = "Borland C++"
		version = "1995"
		pattern = "A1????????C1????A3????????83????????75??80????????????74"
	strings:
		$1 = { A1 ?? ?? ?? ?? C1 ?? ?? A3 ?? ?? ?? ?? 83 ?? ?? ?? ?? 75 ?? 80 ?? ?? ?? ?? ?? ?? 74 }
	condition:
		$1 at pe.entry_point
}

rule borland_cpp_1995_02 {
	meta:
		tool = "C"
		name = "Borland C++"
		version = "1995"
		pattern = "A1????????C1E002A3????????575133C0BF????????B9????????3BCF76052BCFFCF3AA595F"
	strings:
		$1 = { A1 ?? ?? ?? ?? C1 E0 02 A3 ?? ?? ?? ?? 57 51 33 C0 BF ?? ?? ?? ?? B9 ?? ?? ?? ?? 3B CF 76 05 2B CF FC F3 AA 59 5F }
	condition:
		$1 at pe.entry_point
}

rule borland_cpp_1999 {
	meta:
		tool = "C"
		name = "Borland C++"
		version = "1999"
		pattern = "EB1066623A432B2B484F4F4B90E9????????A1????????C1E002A3????????52"
	strings:
		$1 = { EB 10 66 62 3A 43 2B 2B 48 4F 4F 4B 90 E9 ?? ?? ?? ?? A1 ?? ?? ?? ?? C1 E0 02 A3 ?? ?? ?? ?? 52 }
	condition:
		$1 at pe.entry_point
}

rule borland_cpp_551 {
	meta:
		tool = "C"
		name = "Borland C++"
		version = "5.5.1"
		pattern = "EB1066623A432B2B484F4F4B90E9????????A1????????C1E002A3????????526A00E8????00008BD0E8"
	strings:
		$1 = { EB 10 66 62 3A 43 2B 2B 48 4F 4F 4B 90 E9 ?? ?? ?? ?? A1 ?? ?? ?? ?? C1 E0 02 A3 ?? ?? ?? ?? 52 6A 00 E8 ?? ?? 00 00 8B D0 E8 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_uv_01 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		pattern = "558BEC83C4F4"
	strings:
		$1 = { 55 8B EC 83 C4 F4 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_uv_02 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		pattern = "C3E9??????FF8D40"
	strings:
		$1 = { C3 E9 ?? ?? ?? FF 8D 40 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_uv_03 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		pattern = "558BEC83C4??53565733C08945F08945D48945D0E8"
	strings:
		$1 = { 55 8B EC 83 C4 ?? 53 56 57 33 C0 89 45 F0 89 45 D4 89 45 D0 E8 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_uv_04 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		pattern = "558BEC83C4B4B8????????E8????????E8????????8D40"
	strings:
		$1 = { 55 8B EC 83 C4 B4 B8 ?? ?? ?? ?? E8 ?? ?? ?? ?? E8 ?? ?? ?? ?? 8D 40 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_20 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		version = "2.0"
		pattern = "E8????????6A??E8????????8905????????E8????????8905????????C705????????0A??????B8????????C3"
	strings:
		$1 = { E8 ?? ?? ?? ?? 6A ?? E8 ?? ?? ?? ?? 89 05 ?? ?? ?? ?? E8 ?? ?? ?? ?? 89 05 ?? ?? ?? ?? C7 05 ?? ?? ?? ?? 0A ?? ?? ?? B8 ?? ?? ?? ?? C3 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_30_01 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		version = "3.0"
		pattern = "506F7274696F6E7320436F707972696768742028632920313938332C393620426F726C616E6400"
	strings:
		$1 = { 50 6F 72 74 69 6F 6E 73 20 43 6F 70 79 72 69 67 68 74 20 28 63 29 20 31 39 38 33 2C 39 36 20 42 6F 72 6C 61 6E 64 00 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_30_02 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		version = "3.0"
		pattern = "506F7274696F6E7320436F707972696768742028632920313938332C393720426F726C616E6400"
	strings:
		$1 = { 50 6F 72 74 69 6F 6E 73 20 43 6F 70 79 72 69 67 68 74 20 28 63 29 20 31 39 38 33 2C 39 37 20 42 6F 72 6C 61 6E 64 00 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_30_03 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		version = "3.0"
		pattern = "506A??E8????FFFFBA????????528905????????894204E8????????5A58E8????????C3558BEC33C0"
	strings:
		$1 = { 50 6A ?? E8 ?? ?? FF FF BA ?? ?? ?? ?? 52 89 05 ?? ?? ?? ?? 89 42 04 E8 ?? ?? ?? ?? 5A 58 E8 ?? ?? ?? ?? C3 55 8B EC 33 C0 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_30_04 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		version = "3.0"
		pattern = "A1????????C1????A3????????83????????0F??????????80????????????74"
	strings:
		$1 = { A1 ?? ?? ?? ?? C1 ?? ?? A3 ?? ?? ?? ?? 83 ?? ?? ?? ?? 0F ?? ?? ?? ?? ?? 80 ?? ?? ?? ?? ?? ?? 74 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_40_50 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		version = "4.0 - 5.0"
		pattern = "506A??E8????FFFFBA????????528905????????894204C7420800000000C7420C00000000E8????????5A58E8????????C3"
	strings:
		$1 = { 50 6A ?? E8 ?? ?? FF FF BA ?? ?? ?? ?? 52 89 05 ?? ?? ?? ?? 89 42 04 C7 42 08 00 00 00 00 C7 42 0C 00 00 00 00 E8 ?? ?? ?? ?? 5A 58 E8 ?? ?? ?? ?? C3 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_50 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		version = "5.0"
		pattern = "506F7274696F6E7320436F707972696768742028632920313938332C393920426F726C616E6400"
	strings:
		$1 = { 50 6F 72 74 69 6F 6E 73 20 43 6F 70 79 72 69 67 68 74 20 28 63 29 20 31 39 38 33 2C 39 39 20 42 6F 72 6C 61 6E 64 00 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_50_kol {
	meta:
		tool = "C"
		name = "Borland Delphi"
		version = "5.0"
		extra = "with KOL"
		pattern = "558BEC83C4F0B8????4000E8????FFFFE8????FFFFE8????FFFF8BC00000000000000000000000"
	strings:
		$1 = { 55 8B EC 83 C4 F0 B8 ?? ?? 40 00 E8 ?? ?? FF FF E8 ?? ?? FF FF E8 ?? ?? FF FF 8B C0 00 00 00 00 00 00 00 00 00 00 00 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_50_mck {
	meta:
		tool = "C"
		name = "Borland Delphi"
		version = "5.0"
		extra = "with MCK"
		pattern = "558BEC????????????????????????????????????????????????????????????????????????????FF????????????????00????????????????????????000000????????????????00????????????????????????????000000????????????????????????????????????????000000????????????????????????????????????????000000??????????????????????????????????????????????????????00"
	strings:
		$1 = { 55 8B EC ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? FF ?? ?? ?? ?? ?? ?? ?? ?? 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 00 00 ?? ?? ?? ?? ?? ?? ?? ?? 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 00 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 00 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 00 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_60_70_01 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		version = "6.0 - 7.0"
		pattern = "558BEC83C4F0B8????????E8????FBFFA1????????8B??E8????FFFF8B0D????????A1????????8B008B15????????E8????FFFFA1????????8B??E8????FFFFE8????FBFF8D40"
	strings:
		$1 = { 55 8B EC 83 C4 F0 B8 ?? ?? ?? ?? E8 ?? ?? FB FF A1 ?? ?? ?? ?? 8B ?? E8 ?? ?? FF FF 8B 0D ?? ?? ?? ?? A1 ?? ?? ?? ?? 8B 00 8B 15 ?? ?? ?? ?? E8 ?? ?? FF FF A1 ?? ?? ?? ?? 8B ?? E8 ?? ?? FF FF E8 ?? ?? FB FF 8D 40 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_60_70_02 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		version = "6.0 - 7.0"
		pattern = "BA????????837D0C0175??5052C605??????????8B4D08890D????????894A04"
	strings:
		$1 = { BA ?? ?? ?? ?? 83 7D 0C 01 75 ?? 50 52 C6 05 ?? ?? ?? ?? ?? 8B 4D 08 89 0D ?? ?? ?? ?? 89 4A 04 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_60_kol {
	meta:
		tool = "C"
		name = "Borland Delphi"
		version = "6.0"
		extra = "with Key Objects Library"
		pattern = "558BEC83C4F0B8????4000E8????FFFFA1??72400033D2E8????FFFFA1??7240008B0083C014E8????FFFFE8????FFFF"
	strings:
		$1 = { 55 8B EC 83 C4 F0 B8 ?? ?? 40 00 E8 ?? ?? FF FF A1 ?? 72 40 00 33 D2 E8 ?? ?? FF FF A1 ?? 72 40 00 8B 00 83 C0 14 E8 ?? ?? FF FF E8 ?? ?? FF FF }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_60_01 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		version = "6.0"
		pattern = "538BD833C0A3????????6A00E8??????FFA3????????A1????????A3????????33C0A3????????33C0A3????????E8"
	strings:
		$1 = { 53 8B D8 33 C0 A3 ?? ?? ?? ?? 6A 00 E8 ?? ?? ?? FF A3 ?? ?? ?? ?? A1 ?? ?? ?? ?? A3 ?? ?? ?? ?? 33 C0 A3 ?? ?? ?? ?? 33 C0 A3 ?? ?? ?? ?? E8 }
	condition:
		$1 at pe.entry_point
}

rule borland_delphi_60_02 {
	meta:
		tool = "C"
		name = "Borland Delphi"
		version = "6.0"
		pattern = "558BEC83C4F0B8????4500E8??????FFA1????45008B00E8????FFFF8B0D"
	strings:
		$1 = { 55 8B EC 83 C4 F0 B8 ?? ?? 45 00 E8 ?? ?? ?? FF A1 ?? ?? 45 00 8B 00 E8 ?? ?? FF FF 8B 0D }
	condition:
		$1 at pe.entry_point
}

rule borland_dotnet {
	meta:
		tool = "C"
		name = "Borland .NET"
		bytecode = true
		pattern = "480000000200????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001330"
	strings:
		$1 = { 48 00 00 00 02 00 [61] 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 13 30 }
	condition:
		$1 at pe.entry_point
}

rule borland_pascal_70 {
	meta:
		tool = "C"
		name = "Borland Pascal"
		version = "7.0"
		pattern = "B8????8ED88C??????8CD38CC02BD88BC405????C1????03D8B4??CD210E"
	strings:
		$1 = { B8 ?? ?? 8E D8 8C ?? ?? ?? 8C D3 8C C0 2B D8 8B C4 05 ?? ?? C1 ?? ?? 03 D8 B4 ?? CD 21 0E }
	condition:
		$1 at pe.entry_point
}

rule borland_pascal_70_windows {
	meta:
		tool = "C"
		name = "Borland Pascal"
		version = "7.0 for Windows"
		pattern = "9AFFFF00009AFFFF00005589E531C09AFFFF0000"
	strings:
		$1 = { 9A FF FF 00 00 9A FF FF 00 00 55 89 E5 31 C0 9A FF FF 00 00 }
	condition:
		$1 at pe.entry_point
}

rule borland_pascal_70_protected {
	meta:
		tool = "C"
		name = "Borland Pascal"
		version = "7.0 protected mode"
		pattern = "B8????BB????8ED08BE38CD88EC00E1FA1????25????A3????E8????833E??????75"
	strings:
		$1 = { B8 ?? ?? BB ?? ?? 8E D0 8B E3 8C D8 8E C0 0E 1F A1 ?? ?? 25 ?? ?? A3 ?? ?? E8 ?? ?? 83 3E ?? ?? ?? 75 }
	condition:
		$1 at pe.entry_point
}

rule turbo_cpp {
	meta:
		tool = "C"
		name = "Turbo C++"
		version = "3.0 1990"
		pattern = "8CCA2E8916????B430CD218B2E????8B??????8EDAA3????8C06"
	strings:
		$1 = { 8C CA 2E 89 16 ?? ?? B4 30 CD 21 8B 2E ?? ?? 8B ?? ?? ?? 8E DA A3 ?? ?? 8C 06 }
	condition:
		$1 at pe.entry_point
}

rule turbo_borland_pascal_01 {
	meta:
		tool = "C"
		name = "Turbo or Borland Pascal"
		version = "7.0"
		pattern = "9A????????C8??????9A????????09C075??EB??8D??????16576A??9A????????BF????1E5768"
	strings:
		$1 = { 9A ?? ?? ?? ?? C8 ?? ?? ?? 9A ?? ?? ?? ?? 09 C0 75 ?? EB ?? 8D ?? ?? ?? 16 57 6A ?? 9A ?? ?? ?? ?? BF ?? ?? 1E 57 68 }
	condition:
		$1 at pe.entry_point
}

rule turbo_borland_pascal_02 {
	meta:
		tool = "C"
		name = "Turbo or Borland Pascal"
		version = "7.x Unit"
		pattern = "5450555100"
	strings:
		$1 = { 54 50 55 51 00 }
	condition:
		$1 at pe.entry_point
}
import "pe"

rule bero_tiny_pascal {
	meta:
		tool = "C"
		name = "BeRo Tiny Pascal"
		pattern = "E9????????20436F6D70696C65642062793A204265526F54696E7950617363616C202D2028432920436F7079726967687420323030362C2042656E6A616D696E20274265526F2720526F73736561757820"
	strings:
		$1 = { E9 ?? ?? ?? ?? 20 43 6F 6D 70 69 6C 65 64 20 62 79 3A 20 42 65 52 6F 54 69 6E 79 50 61 73 63 61 6C 20 2D 20 28 43 29 20 43 6F 70 79 72 69 67 68 74 20 32 30 30 36 2C 20 42 65 6E 6A 61 6D 69 6E 20 27 42 65 52 6F 27 20 52 6F 73 73 65 61 75 78 20 }
	condition:
		$1 at pe.entry_point
}

rule tmt_pascal_040 {
	meta:
		tool = "P"
		name = "TMT-Pascal"
		version = "0.40"
		pattern = "0E1F068C06????26A1????A3????8EC06633FF6633C9"
	strings:
		$1 = { 0E 1F 06 8C 06 ?? ?? 26 A1 ?? ?? A3 ?? ?? 8E C0 66 33 FF 66 33 C9 }
	condition:
		for any of them : ( $ at pe.entry_point )
}

rule delphi_uv_01 {
	meta:
		tool = "C"
		name = "Delphi"
		pattern = "558BEC83C4F05356B8????????E8????????33C05568????????64FF30648920B8"
	strings:
		$1 = { 55 8B EC 83 C4 F0 53 56 B8 ?? ?? ?? ?? E8 ?? ?? ?? ?? 33 C0 55 68 ?? ?? ?? ?? 64 FF 30 64 89 20 B8 }
	condition:
		$1 at pe.entry_point
}

rule delphi_uv_02 {
	meta:
		tool = "C"
		name = "Delphi"
		pattern = "558BEC83C4F053B8????????E8????????33C05568????????64FF30648920B8????????E8"
	strings:
		$1 = { 55 8B EC 83 C4 F0 53 B8 ?? ?? ?? ?? E8 ?? ?? ?? ?? 33 C0 55 68 ?? ?? ?? ?? 64 FF 30 64 89 20 B8 ?? ?? ?? ?? E8 }
	condition:
		$1 at pe.entry_point
}

rule delphi_20 {
	meta:
		tool = "C"
		name = "Delphi"
		version = "2.0"
		pattern = "44435532"
	strings:
		$1 = { 44 43 55 32 }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_09910_01 {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "0.99.10"
		pattern = "E8006E00005589E58B7D0C8B750889F88B5D1029"
	strings:
		$1 = { E8 00 6E 00 00 55 89 E5 8B 7D 0C 8B 75 08 89 F8 8B 5D 10 29 }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_09910_02 {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "0.99.10"
		pattern = "E8006E00005589E58B7D0C8B750889F88B5D1029"
		start = 19
	strings:
		$1 = { E8 00 6E 00 00 55 89 E5 8B 7D 0C 8B 75 08 89 F8 8B 5D 10 29 }
	condition:
		$1 at pe.entry_point + 19
}

rule free_pascal_104_01 {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "1.0.4"
		pattern = "5589E5C605????????00E8????????5531ED89E0A3????????668CD5892D"
	strings:
		$1 = { 55 89 E5 C6 05 ?? ?? ?? ?? 00 E8 ?? ?? ?? ?? 55 31 ED 89 E0 A3 ?? ?? ?? ?? 66 8C D5 89 2D }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_104_02 {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "1.0.4"
		pattern = "C605????????005589E55356578B7D08893D????????8B7D0C893D????????8B7D10893D"
	strings:
		$1 = { C6 05 ?? ?? ?? ?? 00 55 89 E5 53 56 57 8B 7D 08 89 3D ?? ?? ?? ?? 8B 7D 0C 89 3D ?? ?? ?? ?? 8B 7D 10 89 3D }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_104_03 {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "1.0.4"
		pattern = "C605????????005589E55356578B7D08893D????????8B7D0C893D????????8B7D10893D????????E8????????5F5E5B5DC20C00"
	strings:
		$1 = { C6 05 ?? ?? ?? ?? 00 55 89 E5 53 56 57 8B 7D 08 89 3D ?? ?? ?? ?? 8B 7D 0C 89 3D ?? ?? ?? ?? 8B 7D 10 89 3D ?? ?? ?? ?? E8 ?? ?? ?? ?? 5F 5E 5B 5D C2 0C 00 }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_104_04 {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "1.0.4"
		pattern = "5589E5C605????????00E8????????5531ED89E0A3????????668CD5892D????????DBE3D92D????????31EDE8????????5DE8????????C9C3"
	strings:
		$1 = { 55 89 E5 C6 05 ?? ?? ?? ?? 00 E8 ?? ?? ?? ?? 55 31 ED 89 E0 A3 ?? ?? ?? ?? 66 8C D5 89 2D ?? ?? ?? ?? DB E3 D9 2D ?? ?? ?? ?? 31 ED E8 ?? ?? ?? ?? 5D E8 ?? ?? ?? ?? C9 C3 }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_106 {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "1.0.6"
		pattern = "C605????4000??E8????0000??????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????00"
	strings:
		$1 = { C6 05 ?? ?? 40 00 ?? E8 ?? ?? 00 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_1010_console {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "1.0.10"
		extra = "console version"
		pattern = "C605??????0001E8????0000C605??????0000E8????000050E800000000FF25??????005589E5??EC"
	strings:
		$1 = { C6 05 ?? ?? ?? 00 01 E8 ?? ?? 00 00 C6 05 ?? ?? ?? 00 00 E8 ?? ?? 00 00 50 E8 00 00 00 00 FF 25 ?? ?? ?? 00 55 89 E5 ?? EC }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_1010_gui {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "1.0.10"
		extra = "with GUI"
		pattern = "C605??????0000E8????000050E800000000FF25??????005589E5"
	strings:
		$1 = { C6 05 ?? ?? ?? 00 00 E8 ?? ?? 00 00 50 E8 00 00 00 00 FF 25 ?? ?? ?? 00 55 89 E5 }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_1010_01 {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "1.0.10"
		pattern = "C605????????01E8????0000C605????????00E8????000050E800000000FF25????????55"
	strings:
		$1 = { C6 05 ?? ?? ?? ?? 01 E8 ?? ?? 00 00 C6 05 ?? ?? ?? ?? 00 E8 ?? ?? 00 00 50 E8 00 00 00 00 FF 25 ?? ?? ?? ?? 55 }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_1010_02 {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "1.0.10"
		pattern = "C605????????00E8????000050E800000000FF25????????5589E5"
	strings:
		$1 = { C6 05 ?? ?? ?? ?? 00 E8 ?? ?? 00 00 50 E8 00 00 00 00 FF 25 ?? ?? ?? ?? 55 89 E5 }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_200_01 {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "2.0.0"
		pattern = "5589E5C605????????00E8????????6A0064FF350000000089E0A3"
	strings:
		$1 = { 55 89 E5 C6 05 ?? ?? ?? ?? 00 E8 ?? ?? ?? ?? 6A 00 64 FF 35 00 00 00 00 89 E0 A3 }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_200_02 {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "2.0.0"
		pattern = "C605????????01E874000000C605????????00E86800000050E800000000FF25????????90"
	strings:
		$1 = { C6 05 ?? ?? ?? ?? 01 E8 74 00 00 00 C6 05 ?? ?? ?? ?? 00 E8 68 00 00 00 50 E8 00 00 00 00 FF 25 ?? ?? ?? ?? 90 }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_200_03 {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "2.0.0"
		pattern = "5589E5C605????????00E8????????6A0064FF350000000089E0A3????????5531ED89E0A3????????668CD5892D????????E8????????31EDE8????????5DE8????????C9C3"
	strings:
		$1 = { 55 89 E5 C6 05 ?? ?? ?? ?? 00 E8 ?? ?? ?? ?? 6A 00 64 FF 35 00 00 00 00 89 E0 A3 ?? ?? ?? ?? 55 31 ED 89 E0 A3 ?? ?? ?? ?? 66 8C D5 89 2D ?? ?? ?? ?? E8 ?? ?? ?? ?? 31 ED E8 ?? ?? ?? ?? 5D E8 ?? ?? ?? ?? C9 C3 }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_200_04 {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "2.0.0"
		pattern = "C6050080400001E874000000C6050080400000E86800000050E800000000FF25D8A140009090909090909090909090905589E583EC04895DFCE892000000E8ED00000089C3B9??70400089DAB800000000E80A010000E8C501000089D8E83E020000E8B9010000E8540200008B5DFCC9C38D76000000000000000000000000005589E5C6051080400000E8D10300006A0064FF350000000089E0A3??7040005531ED89E0A32080"
	strings:
		$1 = { C6 05 00 80 40 00 01 E8 74 00 00 00 C6 05 00 80 40 00 00 E8 68 00 00 00 50 E8 00 00 00 00 FF 25 D8 A1 40 00 90 90 90 90 90 90 90 90 90 90 90 90 55 89 E5 83 EC 04 89 5D FC E8 92 00 00 00 E8 ED 00 00 00 89 C3 B9 ?? 70 40 00 89 DA B8 00 00 00 00 E8 0A 01 00 00 E8 C5 01 00 00 89 D8 E8 3E 02 00 00 E8 B9 01 00 00 E8 54 02 00 00 8B 5D FC C9 C3 8D 76 00 00 00 00 00 00 00 00 00 00 00 00 00 55 89 E5 C6 05 10 80 40 00 00 E8 D1 03 00 00 6A 00 64 FF 35 00 00 00 00 89 E0 A3 ?? 70 40 00 55 31 ED 89 E0 A3 20 80 }
	condition:
		$1 at pe.entry_point
}

rule free_pascal_260 {
	meta:
		tool = "C"
		name = "Free Pascal"
		version = "2.6.0"
		pattern = "5589E5C605????????0168????????6AF6E8????????50E8"
	strings:
		$1 = { 55 89 E5 C6 05 ?? ?? ?? ?? 01 68 ?? ?? ?? ?? 6A F6 E8 ?? ?? ?? ?? 50 E8 }
	condition:
		$1 at pe.entry_point
}

rule free_basic_011 {
	meta:
		tool = "C"
		name = "FreeBASIC"
		version = "0.11"
		pattern = "E8????0000E801000000C35589E5"
	strings:
		$1 = { E8 ?? ?? 00 00 E8 01 00 00 00 C3 55 89 E5 }
	condition:
		$1 at pe.entry_point
}

rule free_basic_014 {
	meta:
		tool = "C"
		name = "FreeBASIC"
		version = "0.14"
		pattern = "5589E583EC08C70424??000000FF15??????00E8??FFFFFF89EC31C05DC389F65589E583EC08????????????????????????????????????????????????????5589E583EC088B4508890424FF15??????0089EC5DC3????????????????????5589E5"
	strings:
		$1 = { 55 89 E5 83 EC 08 C7 04 24 ?? 00 00 00 FF 15 ?? ?? ?? 00 E8 ?? FF FF FF 89 EC 31 C0 5D C3 89 F6 55 89 E5 83 EC 08 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 55 89 E5 83 EC 08 8B 45 08 89 04 24 FF 15 ?? ?? ?? 00 89 EC 5D C3 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 55 89 E5 }
	condition:
		$1 at pe.entry_point
}

rule free_basic_016b {
	meta:
		tool = "C"
		name = "FreeBASIC"
		version = "0.16b"
		pattern = "5589E583EC08C7042401000000FF15??????00E888FFFFFF89EC31C05DC389F65589E583EC08C7042402000000FF15??????00E868FFFFFF89EC31C05DC389F65589E583EC088B4508890424FF15??????0089EC5DC38D76008DBC27000000005589E583EC088B4508890424FF15??????0089EC5DC390909090909090909090"
	strings:
		$1 = { 55 89 E5 83 EC 08 C7 04 24 01 00 00 00 FF 15 ?? ?? ?? 00 E8 88 FF FF FF 89 EC 31 C0 5D C3 89 F6 55 89 E5 83 EC 08 C7 04 24 02 00 00 00 FF 15 ?? ?? ?? 00 E8 68 FF FF FF 89 EC 31 C0 5D C3 89 F6 55 89 E5 83 EC 08 8B 45 08 89 04 24 FF 15 ?? ?? ?? 00 89 EC 5D C3 8D 76 00 8D BC 27 00 00 00 00 55 89 E5 83 EC 08 8B 45 08 89 04 24 FF 15 ?? ?? ?? 00 89 EC 5D C3 90 90 90 90 90 90 90 90 90 90 }
	condition:
		$1 at pe.entry_point
}

rule hot_soup_processor_31 {
	meta:
		tool = "C"
		name = "Hot Soup Processor"
		version = "3.1"
		pattern = "6A606830084200E821430000BF940000008BC7E8FDF5FFFF8965E88BF4893E56FF15F00042008B4E10890D884542008B4604A3944542008B56088915984542008B760C81E6FF7F000089358C45420083F902740C81CE0080000089358C454200C1E00803"
	strings:
		$1 = { 6A 60 68 30 08 42 00 E8 21 43 00 00 BF 94 00 00 00 8B C7 E8 FD F5 FF FF 89 65 E8 8B F4 89 3E 56 FF 15 F0 00 42 00 8B 4E 10 89 0D 88 45 42 00 8B 46 04 A3 94 45 42 00 8B 56 08 89 15 98 45 42 00 8B 76 0C 81 E6 FF 7F 00 00 89 35 8C 45 42 00 83 F9 02 74 0C 81 CE 00 80 00 00 89 35 8C 45 42 00 C1 E0 08 03 }
	condition:
		$1 at pe.entry_point
}

rule hot_soup_processor_32 {
	meta:
		tool = "C"
		name = "Hot Soup Processor"
		version = "3.2"
		pattern = "6A6068???84200E8????0000BF940000008BC7E8??F?FFFF8965E88BF4893E56FF15?0??42008B4E10890D????42008B4604A3????42008B56088915????42008B760C81E6FF7F00008935????420083F902740C81CE008000008935????4200C1E00803"
	strings:
		$1 = { 6A 60 68 ?? ?8 42 00 E8 ?? ?? 00 00 BF 94 00 00 00 8B C7 E8 ?? F? FF FF 89 65 E8 8B F4 89 3E 56 FF 15 ?0 ?? 42 00 8B 4E 10 89 0D ?? ?? 42 00 8B 46 04 A3 ?? ?? 42 00 8B 56 08 89 15 ?? ?? 42 00 8B 76 0C 81 E6 FF 7F 00 00 89 35 ?? ?? 42 00 83 F9 02 74 0C 81 CE 00 80 00 00 89 35 ?? ?? 42 00 C1 E0 08 03 }
	condition:
		$1 at pe.entry_point
}

rule intel_xe_13 {
	meta:
		tool = "C"
		name = "Intel XE"
		version = "13"
		pattern = "E8????0000E9A4FEFFFF"
	strings:
		$1 = { E8 ?? ?? 00 00 E9 A4 FE FF FF }
	condition:
		$1 at pe.entry_point
}

rule lcclike_uv_01 {
	meta:
		tool = "C"
		name = "LCC or similar"
		pattern = "5589E553"
	strings:
		$1 = { 55 89 E5 53 }
	condition:
		$1 at pe.entry_point
}

rule lcclike_uv_02 {
	meta:
		tool = "C"
		name = "LCC or similar"
		pattern = "5589E555"
	strings:
		$1 = { 55 89 E5 55 }
	condition:
		$1 at pe.entry_point
}

rule lcc_1x {
	meta:
		tool = "C"
		name = "LCC"
		version = "1.x"
		pattern = "64A1????????5589E56AFF68????????689A1040??50"
	strings:
		$1 = { 64 A1 ?? ?? ?? ?? 55 89 E5 6A FF 68 ?? ?? ?? ?? 68 9A 10 40 ?? 50 }
	condition:
		$1 at pe.entry_point
}

rule lcc_uv_01 {
	meta:
		tool = "C"
		name = "LCC"
		pattern = "5589E5535657837D0C017505E817??????FF7510FF750CFF7508A1"
	strings:
		$1 = { 55 89 E5 53 56 57 83 7D 0C 01 75 05 E8 17 ?? ?? ?? FF 75 10 FF 75 0C FF 75 08 A1 }
	condition:
		$1 at pe.entry_point
}

rule lahey_fortran_2001 {
	meta:
		tool = "C"
		name = "Lahey Fortran 90"
		version = "2001"
		pattern = "558BEC8B45??83E8??72??74??4874??4874??EB??68????????E8????????59E8"
	strings:
		$1 = { 55 8B EC 8B 45 ?? 83 E8 ?? 72 ?? 74 ?? 48 74 ?? 48 74 ?? EB ?? 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? 59 E8 }
	condition:
		$1 at pe.entry_point
}

rule metaware_high_c_uv_01 {
	meta:
		tool = "C"
		name = "MetaWare High C"
		pattern = "B8????8ED8B8????CD21A3????3C037D??B409"
	strings:
		$1 = { B8 ?? ?? 8E D8 B8 ?? ?? CD 21 A3 ?? ?? 3C 03 7D ?? B4 09 }
	condition:
		$1 at pe.entry_point
}

rule metaware_high_c_uv_02 {
	meta:
		tool = "C"
		name = "MetaWare High C"
		pattern = "B8????50B8????50CB"
	strings:
		$1 = { B8 ?? ?? 50 B8 ?? ?? 50 CB }
	condition:
		$1 at pe.entry_point
}

rule metrowerks_codewarrior_20_console {
	meta:
		tool = "C"
		name = "Metrowerks CodeWarrior"
		version = "2.0"
		extra = "console version"
		pattern = "5589E555B8FFFFFFFF505068????????64FF35000000006489250000000068????????E8????????????????????????E8????0000E8????0000E8"
	strings:
		$1 = { 55 89 E5 55 B8 FF FF FF FF 50 50 68 ?? ?? ?? ?? 64 FF 35 00 00 00 00 64 89 25 00 00 00 00 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? E8 ?? ?? 00 00 E8 ?? ?? 00 00 E8 }
	condition:
		$1 at pe.entry_point
}

rule metrowerks_codewarrior_20_gui {
	meta:
		tool = "C"
		name = "Metrowerks CodeWarrior"
		version = "2.0"
		extra = "with GUI"
		pattern = "5589E5535683EC4455B8FFFFFFFF505068????400064FF35000000006489250000000068????????????????????????E8????0000E8????0000E8"
	strings:
		$1 = { 55 89 E5 53 56 83 EC 44 55 B8 FF FF FF FF 50 50 68 ?? ?? 40 00 64 FF 35 00 00 00 00 64 89 25 00 00 00 00 68 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? E8 ?? ?? 00 00 E8 ?? ?? 00 00 E8 }
	condition:
		$1 at pe.entry_point
}

rule metrowerks_codewarrior_20 {
	meta:
		tool = "C"
		name = "Metrowerks CodeWarrior"
		version = "2.0"
		pattern = "5589E55356578B750C8B5D1083FE01740583FE0275125356FF7508E86EFFFFFF09C0750431C0EB215356FF7508E8????????89C709F6740583FE03750A5356FF7508E847FFFFFF89F88D65F45F5E5B5DC20C00C9"
	strings:
		$1 = { 55 89 E5 53 56 57 8B 75 0C 8B 5D 10 83 FE 01 74 05 83 FE 02 75 12 53 56 FF 75 08 E8 6E FF FF FF 09 C0 75 04 31 C0 EB 21 53 56 FF 75 08 E8 ?? ?? ?? ?? 89 C7 09 F6 74 05 83 FE 03 75 0A 53 56 FF 75 08 E8 47 FF FF FF 89 F8 8D 65 F4 5F 5E 5B 5D C2 0C 00 C9 }
	condition:
		$1 at pe.entry_point
}

rule ms_fortran_1989 {
	meta:
		tool = "C"
		name = "Microsoft FORTRAN"
		version = "1989"
		pattern = "B430CD2186E02EA3????3D????73"
	strings:
		$1 = { B4 30 CD 21 86 E0 2E A3 ?? ?? 3D ?? ?? 73 }
	condition:
		$1 at pe.entry_point
}

rule ms_fortran_19xx {
	meta:
		tool = "C"
		name = "Microsoft FORTRAN"
		version = "19xx"
		pattern = "FC1EB8????8ED89A????????81??????8BEC8CDB8EC3BB????9A????????9BDBE39BD92E????33C9"
	strings:
		$1 = { FC 1E B8 ?? ?? 8E D8 9A ?? ?? ?? ?? 81 ?? ?? ?? 8B EC 8C DB 8E C3 BB ?? ?? 9A ?? ?? ?? ?? 9B DB E3 9B D9 2E ?? ?? 33 C9 }
	condition:
		$1 at pe.entry_point
}

rule pelles_c_280_290 {
	meta:
		tool = "C"
		name = "Pelles C"
		version = "2.80 - 2.90"
		pattern = "5589E56AFF68????????68????????64FF35????????648925????????83EC??83EC??5356578965E868000000??E8????????59A3"
	strings:
		$1 = { 55 89 E5 6A FF 68 ?? ?? ?? ?? 68 ?? ?? ?? ?? 64 FF 35 ?? ?? ?? ?? 64 89 25 ?? ?? ?? ?? 83 EC ?? 83 EC ?? 53 56 57 89 65 E8 68 00 00 00 ?? E8 ?? ?? ?? ?? 59 A3 }
	condition:
		$1 at pe.entry_point
}

rule pelles_c_290_300_400 {
	meta:
		tool = "C"
		name = "Pelles C"
		version = "2.90, 3.00, 4.00"
		pattern = "5589E55356578B5D0C8B7510BF0100000085DB7510833D????????00750731C0E9????????83FB01740583FB0275??85FF74"
	strings:
		$1 = { 55 89 E5 53 56 57 8B 5D 0C 8B 75 10 BF 01 00 00 00 85 DB 75 10 83 3D ?? ?? ?? ?? 00 75 07 31 C0 E9 ?? ?? ?? ?? 83 FB 01 74 05 83 FB 02 75 ?? 85 FF 74 }
	condition:
		$1 at pe.entry_point
}

rule pelles_c_300_400_450_01 {
	meta:
		tool = "C"
		name = "Pelles C"
		version = "3.00, 4.00, 4.50"
		pattern = "5589E56AFF68????????68????????64FF35????????648925????????83EC??5356578965E8C745FC????????68????????E8????????59BE????????EB"
	strings:
		$1 = { 55 89 E5 6A FF 68 ?? ?? ?? ?? 68 ?? ?? ?? ?? 64 FF 35 ?? ?? ?? ?? 64 89 25 ?? ?? ?? ?? 83 EC ?? 53 56 57 89 65 E8 C7 45 FC ?? ?? ?? ?? 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? 59 BE ?? ?? ?? ?? EB }
	condition:
		$1 at pe.entry_point
}

rule pelles_c_300_400_450_02 {
	meta:
		tool = "C"
		name = "Pelles C"
		version = "3.00, 4.00, 4.50"
		pattern = "5589E56AFF68????????68????????64FF35????????648925????????83EC??5356578965E86800000002E8????????59A3"
	strings:
		$1 = { 55 89 E5 6A FF 68 ?? ?? ?? ?? 68 ?? ?? ?? ?? 64 FF 35 ?? ?? ?? ?? 64 89 25 ?? ?? ?? ?? 83 EC ?? 53 56 57 89 65 E8 68 00 00 00 02 E8 ?? ?? ?? ?? 59 A3 }
	condition:
		$1 at pe.entry_point
}

rule pelles_c_450 {
	meta:
		tool = "C"
		name = "Pelles C"
		version = "4.50"
		pattern = "5589E55356578B5D0C8B751085DB750D833D????????00750431C0EB5783FB01740583FB0275"
	strings:
		$1 = { 55 89 E5 53 56 57 8B 5D 0C 8B 75 10 85 DB 75 0D 83 3D ?? ?? ?? ?? 00 75 04 31 C0 EB 57 83 FB 01 74 05 83 FB 02 75 }
	condition:
		$1 at pe.entry_point
}

rule perlapp_602_01 {
	meta:
		tool = "C"
		name = "PerlApp"
		version = "6.0.2"
		pattern = "682CEA4000FFD383C40C85C00F85CD0000006A09576820EA4000FFD383C40C85C075128D470950FF151CD1400059A3B8074100EB556A08576814EA4000FFD383C40C85C075118D470850FF151CD140005989442410EB336A09576808EA4000FFD383C40C85C074226A085768FCE94000FFD383C40C85C074116A0B5768F0E94000FFD383C40C85C07555"
	strings:
		$1 = { 68 2C EA 40 00 FF D3 83 C4 0C 85 C0 0F 85 CD 00 00 00 6A 09 57 68 20 EA 40 00 FF D3 83 C4 0C 85 C0 75 12 8D 47 09 50 FF 15 1C D1 40 00 59 A3 B8 07 41 00 EB 55 6A 08 57 68 14 EA 40 00 FF D3 83 C4 0C 85 C0 75 11 8D 47 08 50 FF 15 1C D1 40 00 59 89 44 24 10 EB 33 6A 09 57 68 08 EA 40 00 FF D3 83 C4 0C 85 C0 74 22 6A 08 57 68 FC E9 40 00 FF D3 83 C4 0C 85 C0 74 11 6A 0B 57 68 F0 E9 40 00 FF D3 83 C4 0C 85 C0 75 55 }
	condition:
		$1 at pe.entry_point
}

rule perlapp_602_02 {
	meta:
		tool = "C"
		name = "PerlApp"
		version = "6.0.2"
		pattern = "689CE14000FF15A4D0400085C059740F50FF151CD1400085C0598945FC75626A008D45F8FF750CF6451401508D451450E89B01000083C41085C00F84E90000008B45F883C01450FFD685C0598945FC750EFF7514FF1578D04000E9C9000000688CE14000FF751450"
	strings:
		$1 = { 68 9C E1 40 00 FF 15 A4 D0 40 00 85 C0 59 74 0F 50 FF 15 1C D1 40 00 85 C0 59 89 45 FC 75 62 6A 00 8D 45 F8 FF 75 0C F6 45 14 01 50 8D 45 14 50 E8 9B 01 00 00 83 C4 10 85 C0 0F 84 E9 00 00 00 8B 45 F8 83 C0 14 50 FF D6 85 C0 59 89 45 FC 75 0E FF 75 14 FF 15 78 D0 40 00 E9 C9 00 00 00 68 8C E1 40 00 FF 75 14 50 }
	condition:
		$1 at pe.entry_point
}

rule polybox_c_01 {
	meta:
		tool = "C"
		name = "PolyBox C"
		pattern = "558BEC83C4F05356B8E4410010E83AE1FFFF33C055681144001064FF30648920EB08FCFCFCFCFCFC27546A0A6820440010A11C71001050E8CCE1????????85DB0F847701000053A11C71001050E81EE2FFFF8BF085F60F846101000053A11C71001050E8E0E1FFFF85C00F844D01000050E8DAE1FFFF8BD885DB0F843D01000056B870800010B9010000008B1598410010E89EDEFFFF83C404A1708000108BCE8BD3E8E1E1FFFF6A006A00A170800010B9304400108BD6E8F8FDFFFF"
	strings:
		$1 = { 55 8B EC 83 C4 F0 53 56 B8 E4 41 00 10 E8 3A E1 FF FF 33 C0 55 68 11 44 00 10 64 FF 30 64 89 20 EB 08 FC FC FC FC FC FC 27 54 6A 0A 68 20 44 00 10 A1 1C 71 00 10 50 E8 CC E1 ?? ?? ?? ?? 85 DB 0F 84 77 01 00 00 53 A1 1C 71 00 10 50 E8 1E E2 FF FF 8B F0 85 F6 0F 84 61 01 00 00 53 A1 1C 71 00 10 50 E8 E0 E1 FF FF 85 C0 0F 84 4D 01 00 00 50 E8 DA E1 FF FF 8B D8 85 DB 0F 84 3D 01 00 00 56 B8 70 80 00 10 B9 01 00 00 00 8B 15 98 41 00 10 E8 9E DE FF FF 83 C4 04 A1 70 80 00 10 8B CE 8B D3 E8 E1 E1 FF FF 6A 00 6A 00 A1 70 80 00 10 B9 30 44 00 10 8B D6 E8 F8 FD FF FF }
	condition:
		$1 at pe.entry_point
}

rule polybox_c_02 {
	meta:
		tool = "C"
		name = "PolyBox D"
		pattern = "558BEC33C951515151515333C05568842C400064FF30648920C645FF00B8B8464000BA24000000E88CF3FFFF6A24BAB84640008B0DB0464000A194464000E871FBFFFF84C00F846E0100008B1DD04640008BC383C0240305D84640003B05B44640000F85510100008D45F4BAB8464000B910000000E8A2ECFFFF8B45F4BA9C2C4000E8F1EDFFFF"
	strings:
		$1 = { 55 8B EC 33 C9 51 51 51 51 51 53 33 C0 55 68 84 2C 40 00 64 FF 30 64 89 20 C6 45 FF 00 B8 B8 46 40 00 BA 24 00 00 00 E8 8C F3 FF FF 6A 24 BA B8 46 40 00 8B 0D B0 46 40 00 A1 94 46 40 00 E8 71 FB FF FF 84 C0 0F 84 6E 01 00 00 8B 1D D0 46 40 00 8B C3 83 C0 24 03 05 D8 46 40 00 3B 05 B4 46 40 00 0F 85 51 01 00 00 8D 45 F4 BA B8 46 40 00 B9 10 00 00 00 E8 A2 EC FF FF 8B 45 F4 BA 9C 2C 40 00 E8 F1 ED FF FF }
	condition:
		$1 at pe.entry_point
}

rule powerbasic_uv_01 {
	meta:
		tool = "C"
		name = "PowerBASIC"
		pattern = "558BEC535657BB00????00662EF705????????04000F85??000000"
	strings:
		$1 = { 55 8B EC 53 56 57 BB 00 ?? ?? 00 66 2E F7 05 ?? ?? ?? ?? 04 00 0F 85 ?? 00 00 00 }
	condition:
		$1 at pe.entry_point
}

rule powerbasic_uv_02 {
	meta:
		tool = "C"
		name = "PowerBASIC"
		pattern = "558BEC535657BB00????00662EF705????????04007505E9????????E9"
	strings:
		$1 = { 55 8B EC 53 56 57 BB 00 ?? ?? 00 66 2E F7 05 ?? ?? ?? ?? 04 00 75 05 E9 ?? ?? ?? ?? E9 }
	condition:
		$1 at pe.entry_point
}

rule powerbasic_uv_03 {
	meta:
		tool = "C"
		name = "PowerBASIC"
		pattern = "558BEC535657BB00????0066F705????????04000F85??000000"
	strings:
		$1 = { 55 8B EC 53 56 57 BB 00 ?? ?? 00 66 F7 05 ?? ?? ?? ?? 04 00 0F 85 ?? 00 00 00 }
	condition:
		$1 at pe.entry_point
}

rule powerbasic_uv_04 {
	meta:
		tool = "C"
		name = "PowerBASIC"
		pattern = "558BEC535657BB00????0066F705????????04007505E9????????E9"
	strings:
		$1 = { 55 8B EC 53 56 57 BB 00 ?? ?? 00 66 F7 05 ?? ?? ?? ?? 04 00 75 05 E9 ?? ?? ?? ?? E9 }
	condition:
		$1 at pe.entry_point
}

rule powerbasic_30x {
	meta:
		tool = "C"
		name = "PowerBASIC"
		version = "3.0x"
		pattern = "558BEC535657BB00????00662EF705??????0004000F85"
	strings:
		$1 = { 55 8B EC 53 56 57 BB 00 ?? ?? 00 66 2E F7 05 ?? ?? ?? 00 04 00 0F 85 }
	condition:
		$1 at pe.entry_point
}

rule powerbasic_40x_01 {
	meta:
		tool = "C"
		name = "PowerBASIC"
		version = "4.0x"
		pattern = "558BEC535657BB00??4000662EF705????400004007505E968050000E96E03"
	strings:
		$1 = { 55 8B EC 53 56 57 BB 00 ?? 40 00 66 2E F7 05 ?? ?? 40 00 04 00 75 05 E9 68 05 00 00 E9 6E 03 }
	condition:
		$1 at pe.entry_point
}

rule powerbasic_40x_02 {
	meta:
		tool = "C"
		name = "PureBasic"
		version = "4.0x"
		pattern = "837C240801750E8B442404A3??????10E822000000837C2408027500837C2408007505E8??000000837C2408037500B801000000C20C00680000000068001000006800000000E8??0F0000A3"
	strings:
		$1 = { 83 7C 24 08 01 75 0E 8B 44 24 04 A3 ?? ?? ?? 10 E8 22 00 00 00 83 7C 24 08 02 75 00 83 7C 24 08 00 75 05 E8 ?? 00 00 00 83 7C 24 08 03 75 00 B8 01 00 00 00 C2 0C 00 68 00 00 00 00 68 00 10 00 00 68 00 00 00 00 E8 ?? 0F 00 00 A3 }
	condition:
		$1 at pe.entry_point
}

rule powerbasic_70x {
	meta:
		tool = "C"
		name = "PowerBASIC"
		version = "7.0x"
		pattern = "558BEC535657BB00??4000662EF705????400004000F85DB000000"
	strings:
		$1 = { 55 8B EC 53 56 57 BB 00 ?? 40 00 66 2E F7 05 ?? ?? 40 00 04 00 0F 85 DB 00 00 00 }
	condition:
		$1 at pe.entry_point
}

rule powerbasic_800 {
	meta:
		tool = "C"
		name = "PowerBASIC"
		version = "8.00"
		pattern = "558BEC535657BB00????00662EF705????400004007505E914040000E91902"
	strings:
		$1 = { 55 8B EC 53 56 57 BB 00 ?? ?? 00 66 2E F7 05 ?? ?? 40 00 04 00 75 05 E9 14 04 00 00 E9 19 02 }
	condition:
		$1 at pe.entry_point
}

rule purebasic_4x {
	meta:
		tool = "C"
		name = "PureBasic"
		version = "4.x"
		pattern = "68????0000680000000068????????E8????????83C40C6800000000E8????????A3????????680000000068001000006800000000E8????????A3"
	strings:
		$1 = { 68 ?? ?? 00 00 68 00 00 00 00 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? 83 C4 0C 68 00 00 00 00 E8 ?? ?? ?? ?? A3 ?? ?? ?? ?? 68 00 00 00 00 68 00 10 00 00 68 00 00 00 00 E8 ?? ?? ?? ?? A3 }
	condition:
		$1 at pe.entry_point
}

rule symantec_c_zortech_c_210_400_30r1 {
	meta:
		tool = "C"
		name = "Symantec C or Zortech C"
		version = "2.10, 4.00 or 3.0r1"
		pattern = "FAFCB8????8ED8"
	strings:
		$1 = { FA FC B8 ?? ?? 8E D8 }
	condition:
		$1 at pe.entry_point
}

rule symantec_c_400 {
	meta:
		tool = "C"
		name = "Symantec C"
		version = "4.00"
		pattern = "FAB8????DBE38ED88C06????8BD82B1E????891E????26"
	strings:
		$1 = { FA B8 ?? ?? DB E3 8E D8 8C 06 ?? ?? 8B D8 2B 1E ?? ?? 89 1E ?? ?? 26 }
	condition:
		$1 at pe.entry_point
}

rule symantec_visual_cafe_30 {
	meta:
		tool = "C"
		name = "Symantec Visual Cafe"
		version = "3.0"
		bytecode = true
		pattern = "648B05????????558BEC6AFF68????40??68????40??50648925????????83EC08505356578965E8C745FC"
	strings:
		$1 = { 64 8B 05 ?? ?? ?? ?? 55 8B EC 6A FF 68 ?? ?? 40 ?? 68 ?? ?? 40 ?? 50 64 89 25 ?? ?? ?? ?? 83 EC 08 50 53 56 57 89 65 E8 C7 45 FC }
	condition:
		$1 at pe.entry_point
}

rule open_watcom_19 {
	meta:
		tool = "C"
		name = "Open Watcom"
		version = "1.9"
		pattern = "E9????0000031040004F70656E20576174636F6D20432F432B2B33322052756E2D54696D652073797374656D2E20506F7274696F6E7320436F7079726967687420284329205379626173652C20496E632E20313938382D323030322E"
	strings:
		$1 = { E9 ?? ?? 00 00 03 10 40 00 4F 70 65 6E 20 57 61 74 63 6F 6D 20 43 2F 43 2B 2B 33 32 20 52 75 6E 2D 54 69 6D 65 20 73 79 73 74 65 6D 2E 20 50 6F 72 74 69 6F 6E 73 20 43 6F 70 79 72 69 67 68 74 20 28 43 29 20 53 79 62 61 73 65 2C 20 49 6E 63 2E 20 31 39 38 38 2D 32 30 30 32 2E }
	condition:
		$1 at pe.entry_point
}

rule watcom_c_uv_01 {
	meta:
		tool = "C"
		name = "WATCOM C/C++"
		pattern = "574154434F4D20432F432B2B33322052756E2D54696D652073797374656D2E2028632920436F70797269676874"
		start = 19
	strings:
		$1 = { 57 41 54 43 4F 4D 20 43 2F 43 2B 2B 33 32 20 52 75 6E 2D 54 69 6D 65 20 73 79 73 74 65 6D 2E 20 28 63 29 20 43 6F 70 79 72 69 67 68 74 }
	condition:
		$1 at pe.entry_point + 19
}

rule watcom_c_uv_02 {
	meta:
		tool = "C"
		name = "WATCOM C/C++"
		pattern = "4F70656E20576174636F6D20432F432B2B33322052756E2D54696D652073797374656D2E20506F7274696F6E7320436F70797269676874"
		start = 9
	strings:
		$1 = { 4F 70 65 6E 20 57 61 74 63 6F 6D 20 43 2F 43 2B 2B 33 32 20 52 75 6E 2D 54 69 6D 65 20 73 79 73 74 65 6D 2E 20 50 6F 72 74 69 6F 6E 73 20 43 6F 70 79 72 69 67 68 74 }
	condition:
		$1 at pe.entry_point + 9
}

rule watcom_c_uv_03 {
	meta:
		tool = "C"
		name = "WATCOM C/C++"
		start = 9
		pattern = "574154434F4D20432F432B2B33322052756E2D54696D652073797374656D2E2028632920436F70797269676874"
	strings:
		$1 = { 57 41 54 43 4F 4D 20 43 2F 43 2B 2B 33 32 20 52 75 6E 2D 54 69 6D 65 20 73 79 73 74 65 6D 2E 20 28 63 29 20 43 6F 70 79 72 69 67 68 74 }
	condition:
		$1 at pe.entry_point + 9
}

rule watcom_c_uv_04 {
	meta:
		tool = "C"
		name = "WATCOM C/C++"
		pattern = "535657558B7424148B7C24188B6C241C83FF030F87"
	strings:
		$1 = { 53 56 57 55 8B 74 24 14 8B 7C 24 18 8B 6C 24 1C 83 FF 03 0F 87 }
	condition:
		$1 at pe.entry_point
}

rule watcom_c_uv_05 {
	meta:
		tool = "C"
		name = "WATCOM C/C++"
		pattern = "E9????000003104000574154434F4D20432F432B2B33322052756E2D54696D652073797374656D2E2028632920436F7079726967687420627920574154434F4D20496E7465726E6174696F6E616C20436F72702E20313938382D313939352E20416C6C207269676874732072657365727665642E000000000000"
	strings:
		$1 = { E9 ?? ?? 00 00 03 10 40 00 57 41 54 43 4F 4D 20 43 2F 43 2B 2B 33 32 20 52 75 6E 2D 54 69 6D 65 20 73 79 73 74 65 6D 2E 20 28 63 29 20 43 6F 70 79 72 69 67 68 74 20 62 79 20 57 41 54 43 4F 4D 20 49 6E 74 65 72 6E 61 74 69 6F 6E 61 6C 20 43 6F 72 70 2E 20 31 39 38 38 2D 31 39 39 35 2E 20 41 6C 6C 20 72 69 67 68 74 73 20 72 65 73 65 72 76 65 64 2E 00 00 00 00 00 00 }
	condition:
		$1 at pe.entry_point
}

rule watcom_c_uv_06 {
	meta:
		tool = "C"
		name = "WATCOM C/C++"
		version = "32 Run-Time System"
		pattern = "FB83????89E389??????????89??????????66??????66??????????BB????????29C0B430CD21"
	strings:
		$1 = { FB 83 ?? ?? 89 E3 89 ?? ?? ?? ?? ?? 89 ?? ?? ?? ?? ?? 66 ?? ?? ?? 66 ?? ?? ?? ?? ?? BB ?? ?? ?? ?? 29 C0 B4 30 CD 21 }
	condition:
		$1 at pe.entry_point
}

rule watcom_c_uv_07 {
	meta:
		tool = "C"
		name = "WATCOM C/C++"
		version = "32 Run-Time System"
		pattern = "E9????????????????574154434F4D20432F432B2B33322052756E2D54"
	strings:
		$1 = { E9 ?? ?? ?? ?? ?? ?? ?? ?? 57 41 54 43 4F 4D 20 43 2F 43 2B 2B 33 32 20 52 75 6E 2D 54 }
	condition:
		$1 at pe.entry_point
}

rule watcom_c_uv_08 {
	meta:
		tool = "C"
		name = "WATCOM C/C++"
		version = "32 Run-Time System"
		pattern = "0E1F8CC6B4??50BB????CD2173??58CD2172"
	strings:
		$1 = { 0E 1F 8C C6 B4 ?? 50 BB ?? ?? CD 21 73 ?? 58 CD 21 72 }
	condition:
		$1 at pe.entry_point
}

rule watcom_c_uv_09 {
	meta:
		tool = "C"
		name = "WATCOM C/C++"
		version = "Run-Time system + DOS4GW DOS Extender"
		pattern = "BF????8ED781C4????BE????2BF78BC6B1??D3"
	strings:
		$1 = { BF ?? ?? 8E D7 81 C4 ?? ?? BE ?? ?? 2B F7 8B C6 B1 ?? D3 }
	condition:
		$1 at pe.entry_point
}

rule zortech_c {
	meta:
		tool = "C"
		name = "Zortech C"
		pattern = "E8????2EFF??????FC06"
	strings:
		$1 = { E8 ?? ?? 2E FF ?? ?? ?? FC 06 }
	condition:
		$1 at pe.entry_point
}

rule zortech_c_20 {
	meta:
		tool = "C"
		name = "Zortech C"
		version = "2.0"
		pattern = "FAB8????8ED88C??????268B??????891E????8BD82B1E????891E"
	strings:
		$1 = { FA B8 ?? ?? 8E D8 8C ?? ?? ?? 26 8B ?? ?? ?? 89 1E ?? ?? 8B D8 2B 1E ?? ?? 89 1E }
	condition:
		$1 at pe.entry_point
}

rule zortech_c_30 {
	meta:
		tool = "C"
		name = "Zortech C"
		version = "3.0"
		pattern = "FAFCB8??????8CC88ED8"
	strings:
		$1 = { FA FC B8 ?? ?? ?? 8C C8 8E D8 }
	condition:
		$1 at pe.entry_point
}

rule gc
{
	meta:
		tool = "C"
		name = "gc"
		language = "Go"
		pattern = "??????????240C8D5C241089442404895C2408C70424FFFFFFFFE901000000??E9?BD?FFFF??????????????????????8B5C240464C705340000000000000089E58B4B0489C8C1E00229C489E78B7308FCF3A5FF1389EC8B5C240489430C895310648B0534000000894314C3????????83EC18C70424F4FF"
	strings:
		$1 = { ?? ?? ?? ?? ?? 24 0C 8D 5C 24 10 89 44 24 04 89 5C 24 08 C7 04 24 FF FF FF FF E9 01 00 00 00 ?? E9 ?B D? FF FF ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 8B 5C 24 04 64 C7 05 34 00 00 00 00 00 00 00 89 E5 8B 4B 04 89 C8 C1 E0 02 29 C4 89 E7 8B 73 08 FC F3 A5 FF 13 89 EC 8B 5C 24 04 89 43 0C 89 53 10 64 8B 05 34 00 00 00 89 43 14 C3 ?? ?? ?? ?? 83 EC 18 C7 04 24 F4 FF }
	condition:
		$1 at pe.entry_point
}

rule gcclike_uv_01 {
	meta:
		tool = "C"
		name = "GCC or similar"
		pattern = "5589E556"
	strings:
		$1 = { 55 89 E5 56 }
	condition:
		$1 at pe.entry_point
}

rule gcclike_uv_02 {
	meta:
		tool = "C"
		name = "GCC or similar"
		pattern = "5589E557"
	strings:
		$1 = { 55 89 E5 57 }
	condition:
		$1 at pe.entry_point
}

rule gcclike_uv_03 {
	meta:
		tool = "C"
		name = "GCC or similar"
		pattern = "5589E581EC"
	strings:
		$1 = { 55 89 E5 81 EC }
	condition:
		$1 at pe.entry_point
}

rule gcclike_uv_04 {
	meta:
		tool = "C"
		name = "GCC or similar"
		pattern = "5589E583EC"
	strings:
		$1 = { 55 89 E5 83 EC }
	condition:
		$1 at pe.entry_point
}

rule gcclike_uv_05 {
	meta:
		tool = "C"
		name = "GCC or similar"
		pattern = "5589E58B4508A3????????B801000000"
	strings:
		$1 = { 55 89 E5 8B 45 08 A3 ?? ?? ?? ?? B8 01 00 00 00 }
	condition:
		$1 at pe.entry_point
}

rule gcclike_uv_06 {
	meta:
		tool = "C"
		name = "GCC or similar"
		pattern = "83EC0CC7042402000000FF15????????E8????????8B0D"
	strings:
		$1 = { 83 EC 0C C7 04 24 02 00 00 00 FF 15 ?? ?? ?? ?? E8 ?? ?? ?? ?? 8B 0D }
	condition:
		$1 at pe.entry_point
}

rule gcclike_uv_07 {
	meta:
		tool = "C"
		name = "GCC or similar"
		pattern = "565383EC148B??242483F?0174??8B44242889??2404894424088B442420890424E8"
	strings:
		$1 = { 56 53 83 EC 14 8B ?? 24 24 83 F? 01 74 ?? 8B 44 24 28 89 ?? 24 04 89 44 24 08 8B 44 24 20 89 04 24 E8 }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc {
	meta:
		tool = "C"
		name = "MinGW GCC"
		pattern = "5589E583EC08C70424??000000FF15??????00E8??FEFFFF908DB4260000000055"
	strings:
		$1 = { 55 89 E5 83 EC 08 C7 04 24 ?? 00 00 00 FF 15 ?? ?? ?? 00 E8 ?? FE FF FF 90 8D B4 26 00 00 00 00 55 }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc_2x_01 {
	meta:
		tool = "C"
		name = "MinGW GCC"
		version = "2.x"
		pattern = "5589E5E8????????C9C3????455845"
	strings:
		$1 = { 55 89 E5 E8 ?? ?? ?? ?? C9 C3 ?? ?? 45 58 45 }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc_2x_02 {
	meta:
		tool = "C"
		name = "MinGW GCC"
		version = "2.x"
		pattern = "5589E5??????????????????????000000????????????????????00"
	strings:
		$1 = { 55 89 E5 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 00 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc_3x_01 {
	meta:
		tool = "C"
		name = "MinGW GCC"
		version = "3.x"
		pattern = "5589E583EC08C70424??000000FF15????4000E8"
	strings:
		$1 = { 55 89 E5 83 EC 08 C7 04 24 ?? 00 00 00 FF 15 ?? ?? 40 00 E8 }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc_3x_02 {
	meta:
		tool = "C"
		name = "MinGW GCC"
		version = "3.x"
		pattern = "5589E583EC08C70424??000000FF15????????E8????????????????????????55"
	strings:
		$1 = { 55 89 E5 83 EC 08 C7 04 24 ?? 00 00 00 FF 15 ?? ?? ?? ?? E8 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 55 }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc_32x_01 {
	meta:
		tool = "C"
		name = "MinGW GCC"
		version = "3.2.x"
		pattern = "5589E583EC086A006A006A006A00E80D000000B800000000C9C3909090909090FF253820400090900000000000000000FFFFFFFF00000000FFFFFFFF0000000000"
	strings:
		$1 = { 55 89 E5 83 EC 08 6A 00 6A 00 6A 00 6A 00 E8 0D 00 00 00 B8 00 00 00 00 C9 C3 90 90 90 90 90 90 FF 25 38 20 40 00 90 90 00 00 00 00 00 00 00 00 FF FF FF FF 00 00 00 00 FF FF FF FF 00 00 00 00 00 }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc_32x_02 {
	meta:
		tool = "C"
		name = "MinGW GCC"
		version = "3.2.x"
		pattern = "5589E583EC188975FC8B750C895DF883FE01745C897424048B5510895424088B5508891424E89601000083EC0C83FE0189C3742C85F6750C8B0D0030001085C9751031DB89D88B5DF88B75FC89EC5DC20C00E859000000EBEB8DB4260000000085C075D0E847000000EBC9908D742600C7042480000000E8F4050000A30030001085C0741AC70000000000A310300010E83B020000E8C6010000E975FFFFFFE8BC050000C7000C"
	strings:
		$1 = { 55 89 E5 83 EC 18 89 75 FC 8B 75 0C 89 5D F8 83 FE 01 74 5C 89 74 24 04 8B 55 10 89 54 24 08 8B 55 08 89 14 24 E8 96 01 00 00 83 EC 0C 83 FE 01 89 C3 74 2C 85 F6 75 0C 8B 0D 00 30 00 10 85 C9 75 10 31 DB 89 D8 8B 5D F8 8B 75 FC 89 EC 5D C2 0C 00 E8 59 00 00 00 EB EB 8D B4 26 00 00 00 00 85 C0 75 D0 E8 47 00 00 00 EB C9 90 8D 74 26 00 C7 04 24 80 00 00 00 E8 F4 05 00 00 A3 00 30 00 10 85 C0 74 1A C7 00 00 00 00 00 A3 10 30 00 10 E8 3B 02 00 00 E8 C6 01 00 00 E9 75 FF FF FF E8 BC 05 00 00 C7 00 0C }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc_32x_03 {
	meta:
		tool = "C"
		name = "MinGW GCC"
		version = "3.2.x"
		pattern = "5589E583EC086A006A006A006A00E80D000000B800000000C9C3909090909090FF253820001090900000000000000000FFFFFFFF00000000FFFFFFFF0000000000"
	strings:
		$1 = { 55 89 E5 83 EC 08 6A 00 6A 00 6A 00 6A 00 E8 0D 00 00 00 B8 00 00 00 00 C9 C3 90 90 90 90 90 90 FF 25 38 20 00 10 90 90 00 00 00 00 00 00 00 00 FF FF FF FF 00 00 00 00 FF FF FF FF 00 00 00 00 00 }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc_32x_04 {
	meta:
		tool = "C"
		name = "MinGW GCC"
		version = "3.2.x"
		pattern = "5589E583EC188975FC8B750C895DF883FE01745C897424048B5510895424088B5508891424E87601000083EC0C83FE0189C3742C85F6750C8B0D0030001085C9751031DB89D88B5DF88B75FC89EC5DC20C00E859000000EBEB8DB4260000000085C075D0E847000000EBC9908D742600C7042480000000E8A4050000A30030001085C0741AC70000000000A310300010E81B020000E8A6010000E975FFFFFFE86C050000C7000C"
	strings:
		$1 = { 55 89 E5 83 EC 18 89 75 FC 8B 75 0C 89 5D F8 83 FE 01 74 5C 89 74 24 04 8B 55 10 89 54 24 08 8B 55 08 89 14 24 E8 76 01 00 00 83 EC 0C 83 FE 01 89 C3 74 2C 85 F6 75 0C 8B 0D 00 30 00 10 85 C9 75 10 31 DB 89 D8 8B 5D F8 8B 75 FC 89 EC 5D C2 0C 00 E8 59 00 00 00 EB EB 8D B4 26 00 00 00 00 85 C0 75 D0 E8 47 00 00 00 EB C9 90 8D 74 26 00 C7 04 24 80 00 00 00 E8 A4 05 00 00 A3 00 30 00 10 85 C0 74 1A C7 00 00 00 00 00 A3 10 30 00 10 E8 1B 02 00 00 E8 A6 01 00 00 E9 75 FF FF FF E8 6C 05 00 00 C7 00 0C }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc_32x_05 {
	meta:
		tool = "C"
		name = "MinGW GCC"
		version = "3.2.x"
		pattern = "5589E583EC08C7042401000000FF15E4404000E86800000089EC31C05DC389F65589E583EC08C7042402000000FF15E4404000E84800000089EC31C05DC389F65589E583EC088B5508891424FF150041400089EC5DC38D76008DBC27000000005589E583EC088B5508891424FF15F440400089EC5DC38D76008DBC27000000005589E55383EC24C70424A0114000E88D07000083EC04E885020000C70424002040008B15102040"
	strings:
		$1 = { 55 89 E5 83 EC 08 C7 04 24 01 00 00 00 FF 15 E4 40 40 00 E8 68 00 00 00 89 EC 31 C0 5D C3 89 F6 55 89 E5 83 EC 08 C7 04 24 02 00 00 00 FF 15 E4 40 40 00 E8 48 00 00 00 89 EC 31 C0 5D C3 89 F6 55 89 E5 83 EC 08 8B 55 08 89 14 24 FF 15 00 41 40 00 89 EC 5D C3 8D 76 00 8D BC 27 00 00 00 00 55 89 E5 83 EC 08 8B 55 08 89 14 24 FF 15 F4 40 40 00 89 EC 5D C3 8D 76 00 8D BC 27 00 00 00 00 55 89 E5 53 83 EC 24 C7 04 24 A0 11 40 00 E8 8D 07 00 00 83 EC 04 E8 85 02 00 00 C7 04 24 00 20 40 00 8B 15 10 20 40 }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc_32x_06 {
	meta:
		tool = "C"
		name = "MinGW GCC"
		version = "3.2.x"
		pattern = "5589E583EC08C7042401000000FF15FC404000E86800000089EC31C05DC389F65589E583EC08C7042402000000FF15FC404000E84800000089EC31C05DC389F65589E583EC088B5508891424FF151841400089EC5DC38D76008DBC27000000005589E583EC088B5508891424FF150C41400089EC5DC38D76008DBC27000000005589E55383EC24C70424A0114000E85D08000083EC04E855030000C70424002040008B15102040"
	strings:
		$1 = { 55 89 E5 83 EC 08 C7 04 24 01 00 00 00 FF 15 FC 40 40 00 E8 68 00 00 00 89 EC 31 C0 5D C3 89 F6 55 89 E5 83 EC 08 C7 04 24 02 00 00 00 FF 15 FC 40 40 00 E8 48 00 00 00 89 EC 31 C0 5D C3 89 F6 55 89 E5 83 EC 08 8B 55 08 89 14 24 FF 15 18 41 40 00 89 EC 5D C3 8D 76 00 8D BC 27 00 00 00 00 55 89 E5 83 EC 08 8B 55 08 89 14 24 FF 15 0C 41 40 00 89 EC 5D C3 8D 76 00 8D BC 27 00 00 00 00 55 89 E5 53 83 EC 24 C7 04 24 A0 11 40 00 E8 5D 08 00 00 83 EC 04 E8 55 03 00 00 C7 04 24 00 20 40 00 8B 15 10 20 40 }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc_4x {
	meta:
		tool = "C"
		name = "MinGW G++"
		version = "4.x"
		pattern = "5589E583EC????????????????????????????????????????????????????005589E583EC????????????????????????????????????????????????????0055??????????????????FF????????00"
	strings:
		$1 = { 55 89 E5 83 EC ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 55 89 E5 83 EC ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 55 ?? ?? ?? ?? ?? ?? ?? ?? ?? FF ?? ?? ?? ?? 00 }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc_461 {
	meta:
		tool = "C"
		name = "MinGW GCC"
		version = "4.6.1"
		pattern = "5589E583EC18C7042401000000FF15??????00E87CFDFFFF5589E583EC18C7042402000000FF15??????00E864FDFFFF5589E583EC08A1??????00C9FFE066905589E583EC08A1??????00C9FFE090905589E583EC18C7042400?0??00E8??????005285"
	strings:
		$1 = { 55 89 E5 83 EC 18 C7 04 24 01 00 00 00 FF 15 ?? ?? ?? 00 E8 7C FD FF FF 55 89 E5 83 EC 18 C7 04 24 02 00 00 00 FF 15 ?? ?? ?? 00 E8 64 FD FF FF 55 89 E5 83 EC 08 A1 ?? ?? ?? 00 C9 FF E0 66 90 55 89 E5 83 EC 08 A1 ?? ?? ?? 00 C9 FF E0 90 90 55 89 E5 83 EC 18 C7 04 24 00 ?0 ?? 00 E8 ?? ?? ?? 00 52 85 }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc_473 {
	meta:
		tool = "C"
		name = "MinGW GCC"
		version = "4.7.3"
		pattern = "83EC0CC705??????0000000000E8?E??000083C40CE986FCFFFF909090909090A1??????0085C074435589E583EC18C7042420?0??00FF15???1??00BA0000000083EC0485C07416C74424042E?0??00890424FF15???1??0083EC0889C285D27409C704"
	strings:
		$1 = { 83 EC 0C C7 05 ?? ?? ?? 00 00 00 00 00 E8 ?E ?? 00 00 83 C4 0C E9 86 FC FF FF 90 90 90 90 90 90 A1 ?? ?? ?? 00 85 C0 74 43 55 89 E5 83 EC 18 C7 04 24 20 ?0 ?? 00 FF 15 ?? ?1 ?? 00 BA 00 00 00 00 83 EC 04 85 C0 74 16 C7 44 24 04 2E ?0 ?? 00 89 04 24 FF 15 ?? ?1 ?? 00 83 EC 08 89 C2 85 D2 74 09 C7 04 }
	condition:
		$1 at pe.entry_point
}

rule mingw_gcc_520 {
	meta:
		tool = "C"
		name = "MinGW GCC"
		version = "5.2.0"
		pattern = "83EC0CC705??????0000000000E8?E??000083C40CE976FCFFFF9090909090905589E557565383EC2C8B35???1??00C7042400?0??00FFD683EC0485C00F84BD00000089C3C7042400?0??00FF15???1??008B15???1??0083EC04A3??????00C7442404"
	strings:
		$1 = { 83 EC 0C C7 05 ?? ?? ?? 00 00 00 00 00 E8 ?E ?? 00 00 83 C4 0C E9 76 FC FF FF 90 90 90 90 90 90 55 89 E5 57 56 53 83 EC 2C 8B 35 ?? ?1 ?? 00 C7 04 24 00 ?0 ?? 00 FF D6 83 EC 04 85 C0 0F 84 BD 00 00 00 89 C3 C7 04 24 00 ?0 ?? 00 FF 15 ?? ?1 ?? 00 8B 15 ?? ?1 ?? 00 83 EC 04 A3 ?? ?? ?? 00 C7 44 24 04 }
	condition:
		$1 at pe.entry_point
}

rule dev_cpp_gcc_4 {
	meta:
		tool = "C"
		name = "Dev-C++ GCC"
		version = "4"
		pattern = "5589E583EC0883C4F46A??A1??????00FFD0E8??FFFFFF"
	strings:
		$1 = { 55 89 E5 83 EC 08 83 C4 F4 6A ?? A1 ?? ?? ?? 00 FF D0 E8 ?? FF FF FF }
	condition:
		$1 at pe.entry_point
}

rule dev_cpp_gcc_4992 {
	meta:
		tool = "C"
		name = "Dev-C++ GCC"
		version = "4.9.9.2"
		pattern = "5589E583EC08C7042401000000FF15??????00E8C8FEFFFF908DB426000000005589E583EC08C7042402000000FF15??????00E8A8FEFFFF908DB42600000000558B0D??????0089E55DFFE18D742600558B0D"
	strings:
		$1 = { 55 89 E5 83 EC 08 C7 04 24 01 00 00 00 FF 15 ?? ?? ?? 00 E8 C8 FE FF FF 90 8D B4 26 00 00 00 00 55 89 E5 83 EC 08 C7 04 24 02 00 00 00 FF 15 ?? ?? ?? 00 E8 A8 FE FF FF 90 8D B4 26 00 00 00 00 55 8B 0D ?? ?? ?? 00 89 E5 5D FF E1 8D 74 26 00 55 8B 0D }
	condition:
		$1 at pe.entry_point
}

rule dev_cpp_gcc_5 {
	meta:
		tool = "C"
		name = "Dev-C++ GCC"
		extra = "5"
		pattern = "5589E583EC146A??FF15??????00????????????????????????????00000000"
	strings:
		$1 = { 55 89 E5 83 EC 14 6A ?? FF 15 ?? ?? ?? 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 00 00 00 }
	condition:
		$1 at pe.entry_point
}

rule ms_incremental_linker {
	meta:
		tool = "L"
		name = "Microsoft Incremental Linker"
		version = "5.12.8078"
		extra = "WinASM Studio"
		pattern = "6A006800304000681E3040006A00E80D0000006A00E800000000FF2500204000FF25082040"
	strings:
		$1 = { 6A 00 68 00 30 40 00 68 1E 30 40 00 6A 00 E8 0D 00 00 00 6A 00 E8 00 00 00 00 FF 25 00 20 40 00 FF 25 08 20 40 }
	condition:
		$1 at pe.entry_point
}

rule ms_basic_560 {
	meta:
		tool = "C"
		name = "Microsoft Basic"
		version = "5.60"
		extra = "1982 - 1997"
		pattern = "9A????????9A????????9A????????33DBBA????9A????????C706????????33DB"
	strings:
		$1 = { 9A ?? ?? ?? ?? 9A ?? ?? ?? ?? 9A ?? ?? ?? ?? 33 DB BA ?? ?? 9A ?? ?? ?? ?? C7 06 ?? ?? ?? ?? 33 DB }
	condition:
		$1 at pe.entry_point
}

rule ms_cpp_1990_1992 {
	meta:
		tool = "C"
		name = "Microsoft C++"
		extra = "1990, 1992"
		pattern = "B80030CD213C0373??0E1FBA????B409CD210633C050CB"
	strings:
		$1 = { B8 00 30 CD 21 3C 03 73 ?? 0E 1F BA ?? ?? B4 09 CD 21 06 33 C0 50 CB }
	condition:
		$1 at pe.entry_point
}

rule ms_fortran {
	meta:
		tool = "C"
		name = "Microsoft FORTRAN"
		pattern = "FC1EB8????8ED89A????????81??????8BEC8CDB8EC3BB????B9????9A????????80????????74??E9"
	strings:
		$1 = { FC 1E B8 ?? ?? 8E D8 9A ?? ?? ?? ?? 81 ?? ?? ?? 8B EC 8C DB 8E C3 BB ?? ?? B9 ?? ?? 9A ?? ?? ?? ?? 80 ?? ?? ?? ?? 74 ?? E9 }
	condition:
		$1 at pe.entry_point
}

rule ms_visual_basic_60 {
	meta:
		tool = "C"
		name = "Microsoft Visual Basic"
		version = "6.0"
    condition:
		pe.imports("msvbvm60.dll", 100) or pe.imports("msvbvm60.dll", "ThunRTMain")
}

rule ms_visual_basic_50 {
	meta:
		tool = "C"
		name = "Microsoft Visual Basic"
		version = "5.0"
	condition:
		pe.imports("msvbvm50.dll", 100) or pe.imports("msvbvm50.dll", "ThunRTMain")
}

rule dotnet_01 {
	meta:
		tool = "C"
		name = ".NET"
		bytecode = true
		pattern = "00000000000000005F436F72446C6C4D61696E006D73636F7265652E646C6C0000??0000FF25"
	strings:
		$1 = { 00 00 00 00 00 00 00 00 5F 43 6F 72 44 6C 6C 4D 61 69 6E 00 6D 73 63 6F 72 65 65 2E 64 6C 6C 00 00 ?? 00 00 FF 25 }
	condition:
		$1 at pe.entry_point
}

rule dotnet_02 {
	meta:
		tool = "C"
		name = ".NET"
		bytecode = true
		pattern = "00000000000000005F436F724578654D61696E006D73636F7265652E646C6C0000000000FF25"
	strings:
		$1 = { 00 00 00 00 00 00 00 00 5F 43 6F 72 45 78 65 4D 61 69 6E 00 6D 73 63 6F 72 65 65 2E 64 6C 6C 00 00 00 00 00 FF 25 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_01 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "817C2404"
	strings:
		$1 = { 81 7C 24 04 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_02 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "837C2404"
	strings:
		$1 = { 83 7C 24 04 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_03 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "8B442404"
	strings:
		$1 = { 8B 44 24 04  }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_04 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "8B442408"
	strings:
		$1 = { 8B 44 24 08 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_05 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "8B44240C"
	strings:
		$1 = { 8B 44 24 0C }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_06 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "8B4C2404"
	strings:
		$1 = { 8B 4C 24 04 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_07 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "8B4C2408"
	strings:
		$1 = { 8B 4C 24 08 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_08 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "8B4C240C"
	strings:
		$1 = { 8B 4C 24 0C }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_09 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "8BC0558BEC"
	strings:
		$1 = { 8B C0 55 8B EC }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_10 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "558BEC837D0C"
	strings:
		$1 = { 55 8B EC 83 7D 0C }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_11 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "558D6C249481EC"
	strings:
		$1 = { 55 8D 6C 24 94 81 EC }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_12 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "558B4424??8BEC"
	strings:
		$1 = { 55 8B 44 24 ?? 8B EC }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_13 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "558BEC837C240C0175"
	strings:
		$1 = { 55 8B EC 83 7C 24 0C 01 75 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_14 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "E8????0000E995FEFFFF"
	strings:
		$1 = { E8 ?? ?? 00 00 E9 95 FE FF FF }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_15 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "E8????0000E941FEFFFF"
	strings:
		$1 = { E8 ?? ?? 00 00 E9 41 FE FF FF }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_16 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "E8????0000E987FEFFFF"
	strings:
		$1 = { E8 ?? ?? 00 00 E9 87 FE FF FF }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_17 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "33C0403944240875??8B??2404"
	strings:
		$1 = { 33 C0 40 39 44 24 08 75 ?? 8B ?? 24 04 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_18 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "6A??68????????E8????????33C0"
	strings:
		$1 = { 6A ?? 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? 33 C0 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_19 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "6A??68????????E8????????33DB"
	strings:
		$1 = { 6A ?? 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? 33 DB }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_20 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "6A??68????????E8????????33F6"
	strings:
		$1 = { 6A ?? 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? 33 F6 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_21 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "6A??68????????E8????????33FF"
	strings:
		$1 = { 6A ?? 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? 33 FF }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_22 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "6A??68????????E8????????8365"
	strings:
		$1 = { 6A ?? 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? 83 65 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_23 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "6A??68????????E8????????837D"
	strings:
		$1 = { 6A ?? 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? 83 7D }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_24 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "6A??68????????E8????????8B45"
	strings:
		$1 = { 6A ?? 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? 8B 45 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_25 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "6A??68????????E8????????8D45"
	strings:
		$1 = { 6A ?? 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? 8D 45 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_26 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "5355568B74241485F657B801000000"
	strings:
		$1 = { 53 55 56 8B 74 24 14 85 F6 57 B8 01 00 00 00 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_27 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "53558B6C241056576A015FB3013BEF"
	strings:
		$1 = { 53 55 8B 6C 24 10 56 57 6A 01 5F B3 01 3B EF }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_28 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "535657BB010000008B7C2414553BFB"
	strings:
		$1 = { 53 56 57 BB 01 00 00 00 8B 7C 24 14 55 3B FB }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_29 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "535657BB010000008B7C24145585FF"
	strings:
		$1 = { 53 56 57 BB 01 00 00 00 8B 7C 24 14 55 85 FF }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_30 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "6A??68????????E8????????B?94000000"
	strings:
		$1 = { 6A ?? 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? B? 94 00 00 00 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_31 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "568B74240C83FE017505E8????????8B4424"
	strings:
		$1 = { 56 8B 74 24 0C 83 FE 01 75 05 E8 ?? ?? ?? ?? 8B 44 24 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_32 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "33C040837C2408007505A3D450A764C20C00"
	strings:
		$1 = { 33 C0 40 83 7C 24 08 00 75 05 A3 D4 50 A7 64 C2 0C 00 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_33 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "558BEC51A1E030400053565733FF3BC7740AFF"
	strings:
		$1 = { 55 8B EC 51 A1 E0 30 40 00 53 56 57 33 FF 3B C7 74 0A FF }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_34 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "558BEC51C745FC01000000837D0C007510833D"
	strings:
		$1 = { 55 8B EC 51 C7 45 FC 01 00 00 00 83 7D 0C 00 75 10 83 3D }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_35 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "8BC05355568B74241485F657B8010000000F84"
	strings:
		$1 = { 8B C0 53 55 56 8B 74 24 14 85 F6 57 B8 01 00 00 00 0F 84 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_36 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "6A??68????????E8????????66813D????????4D5A"
	strings:
		$1 = { 6A ?? 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? 66 81 3D ?? ?? ?? ?? 4D 5A }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_37 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "5633F656565656FF150810000156FF15001000015EC3"
	strings:
		$1 = { 56 33 F6 56 56 56 56 FF 15 08 10 00 01 56 FF 15 00 10 00 01 5E C3 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_38 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "33C0837C2408010F94C050E8????????33C05940C20C00"
	strings:
		$1 = { 33 C0 83 7C 24 08 01 0F 94 C0 50 E8 ?? ?? ?? ?? 33 C0 59 40 C2 0C 00 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_39 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "6A??68????????E8????????FF35????????E8????????59A3"
	strings:
		$1 = { 6A ?? 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? FF 35 ?? ?? ?? ?? E8 ?? ?? ?? ?? 59 A3 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_40 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "33C050505050FF15????????50E8????????50FF15????????C3"
	strings:
		$1 = { 33 C0 50 50 50 50 FF 15 ?? ?? ?? ?? 50 E8 ?? ?? ?? ?? 50 FF 15 ?? ?? ?? ?? C3 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_41 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "5589E56AFF68????????68????????64????????????64892500000000"
	strings:
		$1 = { 55 89 E5 6A FF 68 ?? ?? ?? ?? 68 ?? ?? ?? ?? 64 ?? ?? ?? ?? ?? ?? 64 89 25 00 00 00 00 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_42 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "837C2408017505E8????????FF7424048B4C24108B54240CE8????????59C20C00"
	strings:
		$1 = { 83 7C 24 08 01 75 05 E8 ?? ?? ?? ?? FF 74 24 04 8B 4C 24 10 8B 54 24 0C E8 ?? ?? ?? ?? 59 C2 0C 00 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_43 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "558BEC6AFF68?0??400064A100000000506489250000000051B8??7?0100E8?DE?FFFF5356578965F0C?85????F?FF"
	strings:
		$1 = { 55 8B EC 6A FF 68 ?0 ?? 40 00 64 A1 00 00 00 00 50 64 89 25 00 00 00 00 51 B8 ?? 7? 01 00 E8 ?D E? FF FF 53 56 57 89 65 F0 C? 85 ?? ?? F? FF }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_44 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "E9????????E9????????E9????????E9????????E9????????E9????????E9????????E9????????E9????????E9????????E9????????E9????????E9????????E9????????E9????????E9????????E9????????E9????????E9????????E9????????E9"
	strings:
		$1 = { E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 ?? ?? ?? ?? E9 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_45 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "83????6A00FF15F8100BB08D??????516A086A006A0068"
	strings:
		$1 = { 83 ?? ?? 6A 00 FF 15 F8 10 0B B0 8D ?? ?? ?? 51 6A 08 6A 00 6A 00 68 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_46 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "558BEC5657BF010000008B750C"
	strings:
		$1 = { 55 8B EC 56 57 BF 01 00 00 00 8B 75 0C }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_47 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "5355568B??????85F657B8????????75??8B??????????85C975??33C05F5E5D5BC2"
	strings:
		$1 = { 53 55 56 8B ?? ?? ?? 85 F6 57 B8 ?? ?? ?? ?? 75 ?? 8B ?? ?? ?? ?? ?? 85 C9 75 ?? 33 C0 5F 5E 5D 5B C2 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_48 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "8B44240883????74"
	strings:
		$1 = { 8B 44 24 08 83 ?? ?? 74 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_49 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "8B4424085683E8??74??4875"
	strings:
		$1 = { 8B 44 24 08 56 83 E8 ?? 74 ?? 48 75 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_50 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "558BEC5657BF????????8B????3BF70F"
	strings:
		$1 = { 55 8B EC 56 57 BF ?? ?? ?? ?? 8B ?? ?? 3B F7 0F }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_51 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "53565755BB010000008B?C24183B??752AA1"
	strings:
		$1 = { 53 56 57 55 BB 01 00 00 00 8B ?C 24 18 3B ?? 75 2A A1 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_52 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "53565764A1200000008BD88B74241085F60F85"
	strings:
		$1 = { 53 56 57 64 A1 20 00 00 00 8B D8 8B 74 24 10 85 F6 0F 85 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_53 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "558BEC6AFF686864A100000000506489250000000083EC535657"
	strings:
		$1 = { 55 8B EC 6A FF 68 68 64 A1 00 00 00 00 50 64 89 25 00 00 00 00 83 EC 53 56 57 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_54 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "5356578B7C241483FF01"
	strings:
		$1 = { 53 56 57 8B 7C 24 14 83 FF 01 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_55 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "558BEC6AFF68????????68????????64A10000000050??????0000"
	strings:
		$1 = { 55 8B EC 6A FF 68 ?? ?? ?? ?? 68 ?? ?? ?? ?? 64 A1 00 00 00 00 50 ?? ?? ?? 00 00 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_56 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "803D????????007512E81200000084C0B0007409C605????????01B001C20C00"
	strings:
		$1 = { 80 3D ?? ?? ?? ?? 00 75 12 E8 12 00 00 00 84 C0 B0 00 74 09 C6 05 ?? ?? ?? ?? 01 B0 01 C2 0C 00 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_57 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "68????????64A100000000508B442410896C24108D6C24102BE05356578B45F88965E8508B45FCC745FCFFFFFFFF8945F88D45F064A300000000C38B4DF064890D00000000595F5E5BC951C3"
	strings:
		$1 = { 68 ?? ?? ?? ?? 64 A1 00 00 00 00 50 8B 44 24 10 89 6C 24 10 8D 6C 24 10 2B E0 53 56 57 8B 45 F8 89 65 E8 50 8B 45 FC C7 45 FC FF FF FF FF 89 45 F8 8D 45 F0 64 A3 00 00 00 00 C3 8B 4D F0 64 89 0D 00 00 00 00 59 5F 5E 5B C9 51 C3 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_58 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "0FBF44240883E80074254875398B44240450A3????????E8????????FF35????????A3????????FF15????????EB17A1????????85C0740750FF15????????83"
	strings:
		$1 = { 0F BF 44 24 08 83 E8 00 74 25 48 75 39 8B 44 24 04 50 A3 ?? ?? ?? ?? E8 ?? ?? ?? ?? FF 35 ?? ?? ?? ?? A3 ?? ?? ?? ?? FF 15 ?? ?? ?? ?? EB 17 A1 ?? ?? ?? ?? 85 C0 74 07 50 FF 15 ?? ?? ?? ?? 83 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_59 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "FF258??04?00????0??00?000?0000000000???0???00?0?????????0??00?000?0000000000???0???00?0?????????0??00?000?0000000000???0???00?0?????????0??00?000?0000000000???01??0??0?????????0??00?000?0000000000???0"
	strings:
		$1 = { FF 25 8? ?0 4? 00 ?? ?? 0? ?0 0? 00 0? 00 00 00 00 00 ?? ?0 ?? ?0 0? 0? ?? ?? ?? ?? 0? ?0 0? 00 0? 00 00 00 00 00 ?? ?0 ?? ?0 0? 0? ?? ?? ?? ?? 0? ?0 0? 00 0? 00 00 00 00 00 ?? ?0 ?? ?0 0? 0? ?? ?? ?? ?? 0? ?0 0? 00 0? 00 00 00 00 00 ?? ?0 1? ?0 ?? 0? ?? ?? ?? ?? 0? ?0 0? 00 0? 00 00 00 00 00 ?? ?0 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_60 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "FF2500?0400000000?????0?0?0?000000000000?0?00????F??????0?0???????????0?0?0?000000000000?0??0????F??????0?0???????????0?0?0?000000000000?0?0?0???F??????0?0???????????0?0?0?000000000000????????0?0?"
	strings:
		$1 = { FF 25 00 ?0 40 00 00 00 0? ?? ?? 0? 0? 0? 00 00 00 00 00 00 ?0 ?0 0? ?? ?F ?? ?? ?? 0? 0? ?? ?? ?? ?? ?? 0? 0? 0? 00 00 00 00 00 00 ?0 ?? 0? ?? ?F ?? ?? ?? 0? 0? ?? ?? ?? ?? ?? 0? 0? 0? 00 00 00 00 00 00 ?0 ?0 ?0 ?? ?F ?? ?? ?? 0? 0? ?? ?? ?? ?? ?? 0? 0? 0? 00 00 00 00 00 00 ?? ?? ?? ?? 0? 0? }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_61 {
	meta:
		tool = "C"
		name = "MSVC"
		pattern = "FF25???04?00CCCC03300100070000000000000016800C0000042ACC0330010007000000000000001680020000042ACC0330010007000000000000001680070000042ACC03300100070000000000000016800F0000042ACC033001000700000000000000"
	strings:
		$1 = { FF 25 ?? ?0 4? 00 CC CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 0C 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 02 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 07 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 0F 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 }
	condition:
		$1 at pe.entry_point
}

rule msvc_uv_debug {
	meta:
		tool = "C"
		name = "MSVC"
		version = "debug"
		pattern = "E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000E9????0000"
	strings:
		$1 = { E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 E9 ?? ?? 00 00 }
	condition:
		$1 at pe.entry_point
}

rule msvc_20_01 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "2.0"
		pattern = "64A100000000558BEC6AFF68"
	strings:
		$1 = { 64 A1 00 00 00 00 55 8B EC 6A FF 68 }
	condition:
		$1 at pe.entry_point
}

rule msvc_20_02 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "2.0"
		pattern = "535657BB????????8B??????553BFB75"
	strings:
		$1 = { 53 56 57 BB ?? ?? ?? ?? 8B ?? ?? ?? 55 3B FB 75 }
	condition:
		$1 at pe.entry_point
}

rule msvc_30 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "3.0"
		pattern = "64A10000000055????6AFF68????????68????????50??????????000083EC10"
	strings:
		$1 = { 64 A1 00 00 00 00 55 ?? ?? 6A FF 68 ?? ?? ?? ?? 68 ?? ?? ?? ?? 50 ?? ?? ?? ?? ?? 00 00 83 EC 10 }
	condition:
		$1 at pe.entry_point
}

rule msvc_42_01 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "4.2"
		pattern = "64A100000000558BEC6AFF68????????68????????5064????????????83????53565789"
	strings:
		$1 = { 64 A1 00 00 00 00 55 8B EC 6A FF 68 ?? ?? ?? ?? 68 ?? ?? ?? ?? 50 64 ?? ?? ?? ?? ?? ?? 83 ?? ?? 53 56 57 89 }
	condition:
		$1 at pe.entry_point
}

rule msvc_42_02 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "4.2"
		pattern = "53B8010000008B5C240C565785DB5575"
	strings:
		$1 = { 53 B8 01 00 00 00 8B 5C 24 0C 56 57 85 DB 55 75 }
	condition:
		$1 at pe.entry_point
}

rule msvc_4x_01 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "4.x"
		pattern = "64A1000000005589E56AFF68????????68????????506489250000000083EC??535657"
	strings:
		$1 = { 64 A1 00 00 00 00 55 89 E5 6A FF 68 ?? ?? ?? ?? 68 ?? ?? ?? ?? 50 64 89 25 00 00 00 00 83 EC ?? 53 56 57 }
	condition:
		$1 at pe.entry_point
}

rule msvc_4x_02 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "4.x"
		pattern = "64A100000000558BEC6AFF68????????68????????506489250000000083EC??535657"
	strings:
		$1 = { 64 A1 00 00 00 00 55 8B EC 6A FF 68 ?? ?? ?? ?? 68 ?? ?? ?? ?? 50 64 89 25 00 00 00 00 83 EC ?? 53 56 57 }
	condition:
		$1 at pe.entry_point
}

rule msvc_50_01 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "5.0"
		extra = "Visual Studio 97"
		pattern = "64A100000000506489250000000083C4A8535657"
	strings:
		$1 = { 64 A1 00 00 00 00 50 64 89 25 00 00 00 00 83 C4 A8 53 56 57 }
	condition:
		$1 at pe.entry_point
}

rule msvc_50_02 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "5.0"
		extra = "Visual Studio 97"
		pattern = "????24????????????????????????????????????00????????????????8B??240C????????????????????????????????????????????????????????????????????????????83"
	strings:
		$1 = { ?? ?? 24 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 ?? ?? ?? ?? ?? ?? ?? ?? 8B ?? 24 0C ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 83 }
	condition:
		$1 at pe.entry_point
}

rule msvc_60_01 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "6.0"
		extra = "Visual Studio 6.0"
		pattern = "8B44??08"
	strings:
		$1 = { 8B 44 ?? 08 }
	condition:
		$1 at pe.entry_point
}

rule msvc_60_02 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "6.0"
		extra = "Visual Studio 6.0"
		pattern = "558BEC83EC4456FF15????????6A018BF0FF15"
	strings:
		$1 = { 55 8B EC 83 EC 44 56 FF 15 ?? ?? ?? ?? 6A 01 8B F0 FF 15 }
	condition:
		$1 at pe.entry_point
}

rule msvc_60_03 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "6.0"
		extra = "Visual Studio 6.0"
		pattern = "558BEC83EC4456FF15????????8BF08A??3C22"
	strings:
		$1 = { 55 8B EC 83 EC 44 56 FF 15 ?? ?? ?? ?? 8B F0 8A ?? 3C 22 }
	condition:
		$1 at pe.entry_point
}

rule msvc_60_04 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "6.0"
		extra = "Visual Studio 6.0"
		pattern = "837C24080175098B442404A3????0010E88BFFFFFF"
	strings:
		$1 = { 83 7C 24 08 01 75 09 8B 44 24 04 A3 ?? ?? 00 10 E8 8B FF FF FF }
	condition:
		$1 at pe.entry_point
}

rule msvc_60_05 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "6.0"
		extra = "Visual Studio 6.0"
		pattern = "558BEC83EC50535657BE????????8D7DF4A5A566A58B"
	strings:
		$1 = { 55 8B EC 83 EC 50 53 56 57 BE ?? ?? ?? ?? 8D 7D F4 A5 A5 66 A5 8B }
	condition:
		$1 at pe.entry_point
}

rule msvc_60_06 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "6.0"
		extra = "Visual Studio 6.0"
		pattern = "558D6C????81EC????????8B45??83F801560F84????????85C00F84"
	strings:
		$1 = { 55 8D 6C ?? ?? 81 EC ?? ?? ?? ?? 8B 45 ?? 83 F8 01 56 0F 84 ?? ?? ?? ?? 85 C0 0F 84 }
	condition:
		$1 at pe.entry_point
}

rule msvc_60_07 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "6.0"
		extra = "Visual Studio 6.0"
		pattern = "558BEC6AFF68????????68????????64A1????????50648925????????83EC??535657"
	strings:
		$1 = { 55 8B EC 6A FF 68 ?? ?? ?? ?? 68 ?? ?? ?? ?? 64 A1 ?? ?? ?? ?? 50 64 89 25 ?? ?? ?? ?? 83 EC ?? 53 56 57 }
	condition:
		$1 at pe.entry_point
}

rule msvc_60_08 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "6.0"
		extra = "Visual Studio 6.0"
		pattern = "558BEC538B5D08568B750C578B7D1085F67509833D??????????EB2683FE01740583FE027522A1????????85C07409575653FFD085C0740C575653E815FFFFFF85C0750433C0EB4E"
	strings:
		$1 = { 55 8B EC 53 8B 5D 08 56 8B 75 0C 57 8B 7D 10 85 F6 75 09 83 3D ?? ?? ?? ?? ?? EB 26 83 FE 01 74 05 83 FE 02 75 22 A1 ?? ?? ?? ?? 85 C0 74 09 57 56 53 FF D0 85 C0 74 0C 57 56 53 E8 15 FF FF FF 85 C0 75 04 33 C0 EB 4E }
	condition:
		$1 at pe.entry_point
}

rule msvc_60_09 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "6.0"
		extra = "Visual Studio 6.0"
		pattern = "??????????????????00??????????????????????????????????????????????????????????????????????????????????????????????????00??????0D??????00????????????????????00??????????????00????????????????????????????1C????00????????????????????00FF????????????????????????????????????????????????????????00????????00"
	strings:
		$1 = { ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 ?? ?? ?? 0D ?? ?? ?? 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 ?? ?? ?? ?? ?? ?? ?? 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 1C ?? ?? 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 FF ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 ?? ?? ?? ?? 00 }
	condition:
		$1 at pe.entry_point
}

rule msvc_60_debug_01 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "6.0 debug"
		extra = "Visual Studio 6.0"
		pattern = "558BEC538B5D08568B750C578B7D1085F6????83"
	strings:
		$1 = { 55 8B EC 53 8B 5D 08 56 8B 75 0C 57 8B 7D 10 85 F6 ?? ?? 83 }
	condition:
		$1 at pe.entry_point
}

rule msvc_60_debug_02 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "6.0 debug"
		extra = "Visual Studio 6.0"
		pattern = "558BEC51??????01????????????????????????????????????????????00??????????????????????????????????????????????????????????????????????????????????????????????00????????????????????????????????????????????????????????????????????10??????????????????E8"
	strings:
		$1 = { 55 8B EC 51 ?? ?? ?? 01 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 10 ?? ?? ?? ?? ?? ?? ?? ?? ?? E8 }
	condition:
		$1 at pe.entry_point
}

rule msvc_70_01 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "7.0"
		extra = "Visual Studio .NET 2002"
		pattern = "558BEC538B5D08568B750C85F6578B7D10"
	strings:
		$1 = { 55 8B EC 53 8B 5D 08 56 8B 75 0C 85 F6 57 8B 7D 10 }
	condition:
		$1 at pe.entry_point
}

rule msvc_70_02 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "7.0"
		extra = "Visual Studio .NET 2002"
		pattern = "6A0C6888BF0110E8B81C000033C0408945E48B750C33FF3BF7750C393D6C1E12100F84B3000000897DFC3BF0740583FE027531A1983612103BC7740CFF751056"
	strings:
		$1 = { 6A 0C 68 88 BF 01 10 E8 B8 1C 00 00 33 C0 40 89 45 E4 8B 75 0C 33 FF 3B F7 75 0C 39 3D 6C 1E 12 10 0F 84 B3 00 00 00 89 7D FC 3B F0 74 05 83 FE 02 75 31 A1 98 36 12 10 3B C7 74 0C FF 75 10 56 }
	condition:
		$1 at pe.entry_point
}

rule msvc_70_03 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "7.0"
		extra = "Visual Studio .NET 2002"
		pattern = "????4100000000000000630000000000??00??????????00??00??????????????????????????????????00??00??00??????????????????????????????00????20????00??00??????????????00??????????????????????00??00??????00??????????????00??00??00??00??00??00??00??00??00??00??00??00??????00??00??00??00??00??00??00??????00??00??00??00??00??00??00??????00??00"
	strings:
		$1 = { ?? ?? 41 00 00 00 00 00 00 00 63 00 00 00 00 00 ?? 00 ?? ?? ?? ?? ?? 00 ?? 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 ?? 00 ?? 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 ?? ?? 20 ?? ?? 00 ?? 00 ?? ?? ?? ?? ?? ?? ?? 00 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 ?? 00 ?? ?? ?? 00 ?? ?? ?? ?? ?? ?? ?? 00 ?? 00 ?? 00 ?? 00 ?? 00 ?? 00 ?? 00 ?? 00 ?? 00 ?? 00 ?? 00 ?? 00 ?? ?? ?? 00 ?? 00 ?? 00 ?? 00 ?? 00 ?? 00 ?? 00 ?? ?? ?? 00 ?? 00 ?? 00 ?? 00 ?? 00 ?? 00 ?? 00 ?? ?? ?? 00 ?? 00 }
	condition:
		$1 at pe.entry_point
}

rule msvc_71_debug_01 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "7.1 debug"
		extra = "Visual Studio .NET 2003"
		pattern = "558BEC538B5D08568B750C85F6578B7D107509833D????400000EB2683FE01740583FE027522A1"
	strings:
		$1 = { 55 8B EC 53 8B 5D 08 56 8B 75 0C 85 F6 57 8B 7D 10 75 09 83 3D ?? ?? 40 00 00 EB 26 83 FE 01 74 05 83 FE 02 75 22 A1 }
	condition:
		$1 at pe.entry_point
}

rule msvc_71_debug_02 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "7.1 debug"
		extra = "Visual Studio .NET 2003"
		pattern = "558BEC????0C83????????????????????????????????????????????????????????????????????0000008B"
	strings:
		$1 = { 55 8B EC ?? ?? 0C 83 ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? ?? 00 00 00 8B }
	condition:
		$1 at pe.entry_point
}

rule msvc_71_debug_03 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "7.1 debug"
		extra = "Visual Studio .NET 2003"
		pattern = "558BEC83EC0853565755FC8B5D0C8B4508F74004060000000F85AB0000008945F88B45108945FC8D45F88943FC8B730C8B7B0853E8????????83C4040BC0747B83FEFF747D8D0C768B448F040BC074595655"
	strings:
		$1 = { 55 8B EC 83 EC 08 53 56 57 55 FC 8B 5D 0C 8B 45 08 F7 40 04 06 00 00 00 0F 85 AB 00 00 00 89 45 F8 8B 45 10 89 45 FC 8D 45 F8 89 43 FC 8B 73 0C 8B 7B 08 53 E8 ?? ?? ?? ?? 83 C4 04 0B C0 74 7B 83 FE FF 74 7D 8D 0C 76 8B 44 8F 04 0B C0 74 59 56 55 }
	condition:
		$1 at pe.entry_point
}

rule msvc_71_debug_04 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "7.1 debug"
		extra = "Visual Studio .NET 2003"
		pattern = "8BFF558BEC5633F639750C750E3935????????7E2DFF0D????????837D0C01A1????????8B00A3????????753D6880000000FF15????????3BC659A3????????750433C0EB678930A1????????68????????68"
	strings:
		$1 = { 8B FF 55 8B EC 56 33 F6 39 75 0C 75 0E 39 35 ?? ?? ?? ?? 7E 2D FF 0D ?? ?? ?? ?? 83 7D 0C 01 A1 ?? ?? ?? ?? 8B 00 A3 ?? ?? ?? ?? 75 3D 68 80 00 00 00 FF 15 ?? ?? ?? ?? 3B C6 59 A3 ?? ?? ?? ?? 75 04 33 C0 EB 67 89 30 A1 ?? ?? ?? ?? 68 ?? ?? ?? ?? 68 }
	condition:
		$1 at pe.entry_point
}

rule msvc_80_01 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "8.0"
		extra = "Visual Studio 2005"
		pattern = "E8??0?0000E936FDFFFF"
	strings:
		$1 = { E8 ?? 0? 00 00 E9 36 FD FF FF }
	condition:
		$1 at pe.entry_point
}

rule msvc_80_02 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "8.0"
		extra = "Visual Studio 2005"
		pattern = "83EC04837C240C01560F85??00000068"
	strings:
		$1 = { 83 EC 04 83 7C 24 0C 01 56 0F 85 ?? 00 00 00 68 }
	condition:
		$1 at pe.entry_point
}

rule msvc_80_03 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "8.0"
		extra = "Visual Studio 2005"
		pattern = "83EC4456FF15????????8BF08A0?3C2275"
	strings:
		$1 = { 83 EC 44 56 FF 15 ?? ?? ?? ?? 8B F0 8A 0? 3C 22 75 }
	condition:
		$1 at pe.entry_point
}

rule msvc_80_04 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "8.0"
		extra = "Visual Studio 2005"
		pattern = "8BFF558BEC837D0C017505E8??????FF5DE9D6FEFFFFCCCCCCCCCC"
	strings:
		$1 = { 8B FF 55 8B EC 83 7D 0C 01 75 05 E8 ?? ?? ?? FF 5D E9 D6 FE FF FF CC CC CC CC CC }
	condition:
		$1 at pe.entry_point
}

rule msvc_80_05 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "8.0"
		extra = "Visual Studio 2005"
		pattern = "8BFF558BEC538B5D08568B750C85F6578B7D100F84????000083FE01"
	strings:
		$1 = { 8B FF 55 8B EC 53 8B 5D 08 56 8B 75 0C 85 F6 57 8B 7D 10 0F 84 ?? ?? 00 00 83 FE 01 }
	condition:
		$1 at pe.entry_point
}

rule msvc_80_06 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "8.0"
		extra = "Visual Studio 2005"
		pattern = "6A1468????????E8????????BB94000000536A008B??????????FFD750FF??????????8BF085F6750A6A12E8????????59EB18891E56FF??????????5685C0751450FFD750FF??????????B8"
	strings:
		$1 = { 6A 14 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? BB 94 00 00 00 53 6A 00 8B ?? ?? ?? ?? ?? FF D7 50 FF ?? ?? ?? ?? ?? 8B F0 85 F6 75 0A 6A 12 E8 ?? ?? ?? ?? 59 EB 18 89 1E 56 FF ?? ?? ?? ?? ?? 56 85 C0 75 14 50 FF D7 50 FF ?? ?? ?? ?? ?? B8 }
	condition:
		$1 at pe.entry_point
}

rule msvc_80_07 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "8.0"
		extra = "Visual Studio 2005"
		pattern = "E8??030000E99EFDFFFF558BEC81EC28030000A3????4000890D????40008915????4000891D????40008935????4000893D????4000668C15????4000668C0D????4000668C1D????4000668C05????4000668C25????4000668C2D????40009C8F05"
	strings:
		$1 = { E8 ?? 03 00 00 E9 9E FD FF FF 55 8B EC 81 EC 28 03 00 00 A3 ?? ?? 40 00 89 0D ?? ?? 40 00 89 15 ?? ?? 40 00 89 1D ?? ?? 40 00 89 35 ?? ?? 40 00 89 3D ?? ?? 40 00 66 8C 15 ?? ?? 40 00 66 8C 0D ?? ?? 40 00 66 8C 1D ?? ?? 40 00 66 8C 05 ?? ?? 40 00 66 8C 25 ?? ?? 40 00 66 8C 2D ?? ?? 40 00 9C 8F 05 }
	condition:
		$1 at pe.entry_point
}

rule msvc_80_08 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "8.0 debug"
		extra = "Visual Studio 2005"
		pattern = "558BECE8?80?0000E8030000005DC3CC558BEC6AFE68????400068?0??400064A1000000005083C4E4535657A1???040003145F833C5508D45F064A3000000008965E8C745FC00000000C745DC00000000E80A0200008B40048945E0C745E4000000006A"
	strings:
		$1 = { 55 8B EC E8 ?8 0? 00 00 E8 03 00 00 00 5D C3 CC 55 8B EC 6A FE 68 ?? ?? 40 00 68 ?0 ?? 40 00 64 A1 00 00 00 00 50 83 C4 E4 53 56 57 A1 ?? ?0 40 00 31 45 F8 33 C5 50 8D 45 F0 64 A3 00 00 00 00 89 65 E8 C7 45 FC 00 00 00 00 C7 45 DC 00 00 00 00 E8 0A 02 00 00 8B 40 04 89 45 E0 C7 45 E4 00 00 00 00 6A }
	condition:
		$1 at pe.entry_point
}

rule msvc_80_10 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "8.0"
		extra = "Visual Studio 2005"
		pattern = "FF250030400000000?????0?0?0?000000000000????0000000000?0??????0?000???????????0?0?0?000000000000????????000???????????0?0?0?000000000000??????0?000???????????0?0?0?000000000000??????0?000???????????0?"
	strings:
		$1 = { FF 25 00 30 40 00 00 00 0? ?? ?? 0? 0? 0? 00 00 00 00 00 00 ?? ?? 00 00 00 00 00 ?0 ?? ?? ?? 0? 00 0? ?? ?? ?? ?? ?? 0? 0? 0? 00 00 00 00 00 00 ?? ?? ?? ?? 00 0? ?? ?? ?? ?? ?? 0? 0? 0? 00 00 00 00 00 00 ?? ?? ?? 0? 00 0? ?? ?? ?? ?? ?? 0? 0? 0? 00 00 00 00 00 00 ?? ?? ?? 0? 00 0? ?? ?? ?? ?? ?? 0? }
	condition:
		$1 at pe.entry_point
}

rule msvc_80_11 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "8.0"
		extra = "Visual Studio 2005"
		pattern = "E8????????E98DFEFFFFCCCCCCCCCC66813D000000014D5A740433C0EB51A13C00000181B8000000015045000075EB0FB7881800000181F90B010000741B81F90B02000075D483B8840000010E76CB33C93988F8000001EB1183B8740000010E76B833C93988E80000010F95C18BC16A01A3??????01E8????000050FF??????0001830D??????01FF830D??????01FF5959FF15????00018B0D??????018908FF15????00018B"
	strings:
		$1 = { E8 ?? ?? ?? ?? E9 8D FE FF FF CC CC CC CC CC 66 81 3D 00 00 00 01 4D 5A 74 04 33 C0 EB 51 A1 3C 00 00 01 81 B8 00 00 00 01 50 45 00 00 75 EB 0F B7 88 18 00 00 01 81 F9 0B 01 00 00 74 1B 81 F9 0B 02 00 00 75 D4 83 B8 84 00 00 01 0E 76 CB 33 C9 39 88 F8 00 00 01 EB 11 83 B8 74 00 00 01 0E 76 B8 33 C9 39 88 E8 00 00 01 0F 95 C1 8B C1 6A 01 A3 ?? ?? ?? 01 E8 ?? ?? 00 00 50 FF ?? ?? ?? 00 01 83 0D ?? ?? ?? 01 FF 83 0D ?? ?? ?? 01 FF 59 59 FF 15 ?? ?? 00 01 8B 0D ?? ?? ?? 01 89 08 FF 15 ?? ?? 00 01 8B }
	condition:
		$1 at pe.entry_point
}

rule msvc_80_debug_01 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "8.0 debug"
		extra = "Visual Studio 2005"
		pattern = "FF25003040000000033001000F0000000000000023000000000000F07F80070000042ACC0330010007000000000000001680120000042ACC0330010007000000000000001680040000042ACC03300100070000000000000016800C0000042ACC03300100"
	strings:
		$1 = { FF 25 00 30 40 00 00 00 03 30 01 00 0F 00 00 00 00 00 00 00 23 00 00 00 00 00 00 F0 7F 80 07 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 12 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 04 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 0C 00 00 04 2A CC 03 30 01 00 }
	condition:
		$1 at pe.entry_point
}

rule msvc_80_debug_02 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "8.0 debug"
		extra = "Visual Studio 2005"
		pattern = "FF258??04?00CCCCCCCCCCCCCCCCCCCCCCCC03300100070000000000000016800D0000042ACC0330010007000000000000001680030000042ACC0330010007000000000000001680080000042ACC0330010007000000000000001680100000042ACC0330"
	strings:
		$1 = { FF 25 8? ?0 4? 00 CC CC CC CC CC CC CC CC CC CC CC CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 0D 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 03 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 08 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 10 00 00 04 2A CC 03 30 }
	condition:
		$1 at pe.entry_point
}

rule msvc_90_01 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "9.0"
		extra = "Visual Studio 2008"
		pattern = "E8??040000E99FFDFFFF8BFF558BEC81EC28030000A3????4000890D????40008915????4000891D????40008935????4000893D????4000668C15????4000668C0D????4000668C1D????4000668C05????4000668C25????4000668C2D????40009C8F"
	strings:
		$1 = { E8 ?? 04 00 00 E9 9F FD FF FF 8B FF 55 8B EC 81 EC 28 03 00 00 A3 ?? ?? 40 00 89 0D ?? ?? 40 00 89 15 ?? ?? 40 00 89 1D ?? ?? 40 00 89 35 ?? ?? 40 00 89 3D ?? ?? 40 00 66 8C 15 ?? ?? 40 00 66 8C 0D ?? ?? 40 00 66 8C 1D ?? ?? 40 00 66 8C 05 ?? ?? 40 00 66 8C 25 ?? ?? 40 00 66 8C 2D ?? ?? 40 00 9C 8F }
	condition:
		$1 at pe.entry_point
}

rule msvc_90_02 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "9.0"
		extra = "Visual Studio 2008"
		pattern = "FF2500?04000000000033001000B00000000000000200001FF0F80??0000042ACC033001000B00000000000000200100FF0F80??0000042ACC033001000B00000000000000201000FF0F80??0000042ACC0330010007000000000000001680??0000042A"
	strings:
		$1 = { FF 25 00 ?0 40 00 00 00 00 03 30 01 00 0B 00 00 00 00 00 00 00 20 00 01 FF 0F 80 ?? 00 00 04 2A CC 03 30 01 00 0B 00 00 00 00 00 00 00 20 01 00 FF 0F 80 ?? 00 00 04 2A CC 03 30 01 00 0B 00 00 00 00 00 00 00 20 10 00 FF 0F 80 ?? 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 ?? 00 00 04 2A }
	condition:
		$1 at pe.entry_point
}

rule msvc_90_debug_01 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "9.0 debug"
		extra = "Visual Studio 2008"
		pattern = "FF2500?0400000000?????0?0?0?000000000000?0?00????F????0?000???????????0?0?0?000000000000?0??0????F????0?000???????????0?0?0?000000000000?0?0?0???F????0?000???????????0?0?0?000000000000??????0?000?"
	strings:
		$1 = { FF 25 00 ?0 40 00 00 00 0? ?? ?? 0? 0? 0? 00 00 00 00 00 00 ?0 ?0 0? ?? ?F ?? ?? 0? 00 0? ?? ?? ?? ?? ?? 0? 0? 0? 00 00 00 00 00 00 ?0 ?? 0? ?? ?F ?? ?? 0? 00 0? ?? ?? ?? ?? ?? 0? 0? 0? 00 00 00 00 00 00 ?0 ?0 ?0 ?? ?F ?? ?? 0? 00 0? ?? ?? ?? ?? ?? 0? 0? 0? 00 00 00 00 00 00 ?? ?? ?? 0? 00 0? }
	condition:
		$1 at pe.entry_point
}

rule msvc_90_debug_02 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "9.0 debug"
		extra = "Visual Studio 2008"
		pattern = "FF25?????000CCCCCCCC0330010007000000000000001680??0000042ACC0330010007000000000000001680??0000042ACC0330010007000000000000001680??0000042ACC0330010007000000000000001680??0000042ACC03300100070000000000"
	strings:
		$1 = { FF 25 ?? ?? ?0 00 CC CC CC CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 ?? 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 ?? 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 ?? 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 ?? 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 }
	condition:
		$1 at pe.entry_point
}

rule msvc_90_debug_03 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "9.0 debug"
		extra = "Visual Studio 2008"
		pattern = "8BFF558BECE8?60?0000E8110000005DC3CCCCCCCCCCCCCCCCCCCCCCCCCCCCCC8BFF558BEC6AFE68????400068?0??400064A1000000005083C4E4535657A1???040003145F833C5508D45F064A3000000008965E8C745FC00000000C745DC00000000E8"
	strings:
		$1 = { 8B FF 55 8B EC E8 ?6 0? 00 00 E8 11 00 00 00 5D C3 CC CC CC CC CC CC CC CC CC CC CC CC CC CC CC 8B FF 55 8B EC 6A FE 68 ?? ?? 40 00 68 ?0 ?? 40 00 64 A1 00 00 00 00 50 83 C4 E4 53 56 57 A1 ?? ?0 40 00 31 45 F8 33 C5 50 8D 45 F0 64 A3 00 00 00 00 89 65 E8 C7 45 FC 00 00 00 00 C7 45 DC 00 00 00 00 E8 }
	condition:
		$1 at pe.entry_point
}

rule msvc_10_01 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "10.0"
		extra = "Visual Studio 2010"
		pattern = "E8??040000E9B3FDFFFF8BFF558BEC81EC28030000A3????4000890D????40008915????4000891D????40008935????4000893D????4000668C15????4000668C0D????4000668C1D????4000668C05????4000668C25????4000668C2D????40009C8F"
	strings:
		$1 = { E8 ?? 04 00 00 E9 B3 FD FF FF 8B FF 55 8B EC 81 EC 28 03 00 00 A3 ?? ?? 40 00 89 0D ?? ?? 40 00 89 15 ?? ?? 40 00 89 1D ?? ?? 40 00 89 35 ?? ?? 40 00 89 3D ?? ?? 40 00 66 8C 15 ?? ?? 40 00 66 8C 0D ?? ?? 40 00 66 8C 1D ?? ?? 40 00 66 8C 05 ?? ?? 40 00 66 8C 25 ?? ?? 40 00 66 8C 2D ?? ?? 40 00 9C 8F }
	condition:
		$1 at pe.entry_point
}

rule msvc_10_debug_01 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "10.0 debug"
		extra = "Visual Studio 2010"
		pattern = "8BFF558BECE8?60?0000E8110000005DC3CCCCCCCCCCCCCCCCCCCCCCCCCCCCCC8BFF558BEC6AFE68????400068?0??400064A1000000005083C4E4535657A1?0?040003145F833C5508D45F064A3000000008965E8833D????400000750E6A006A006A01"
	strings:
		$1 = { 8B FF 55 8B EC E8 ?6 0? 00 00 E8 11 00 00 00 5D C3 CC CC CC CC CC CC CC CC CC CC CC CC CC CC CC 8B FF 55 8B EC 6A FE 68 ?? ?? 40 00 68 ?0 ?? 40 00 64 A1 00 00 00 00 50 83 C4 E4 53 56 57 A1 ?0 ?0 40 00 31 45 F8 33 C5 50 8D 45 F0 64 A3 00 00 00 00 89 65 E8 83 3D ?? ?? 40 00 00 75 0E 6A 00 6A 00 6A 01 }
	condition:
		$1 at pe.entry_point
}

rule msvc_10_debug_02 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "10.0 debug"
		extra = "Visual Studio 2010"
		pattern = "FF2500?04?000000033001000B00000000000000200001FF0F80??0000042ACC033001000B00000000000000200100FF0F80??0000042ACC033001000B00000000000000201000FF0F80??0000042ACC0330010007000000000000001680??0000042ACC"
	strings:
		$1 = { FF 25 00 ?0 4? 00 00 00 03 30 01 00 0B 00 00 00 00 00 00 00 20 00 01 FF 0F 80 ?? 00 00 04 2A CC 03 30 01 00 0B 00 00 00 00 00 00 00 20 01 00 FF 0F 80 ?? 00 00 04 2A CC 03 30 01 00 0B 00 00 00 00 00 00 00 20 10 00 FF 0F 80 ?? 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 ?? 00 00 04 2A CC }
	condition:
		$1 at pe.entry_point
}

rule msvc_10_debug_03 {
	meta:
		tool = "C"
		name = "MSVC"
		version = "10.0 debug"
		extra = "Visual Studio 2010"
		pattern = "FF258??04?00CCCCCCCCCCCCCCCCCCCC03300100070000000000000016800F0000042ACC0330010007000000000000001680050000042ACC03300100070000000000000016800A0000042ACC0330010007000000000000001680120000042ACC03300100"
	strings:
		$1 = { FF 25 8? ?0 4? 00 CC CC CC CC CC CC CC CC CC CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 0F 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 05 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 0A 00 00 04 2A CC 03 30 01 00 07 00 00 00 00 00 00 00 16 80 12 00 00 04 2A CC 03 30 01 00 }
	condition:
		$1 at pe.entry_point
}

rule metalang_uv {
	meta:
		tool = "C"
		name = "MetaLang"
		pattern = "E9FF000000608B7424248B7C2428FCB28033DBA4B302E86D00000073F633C9E864000000731C33C0E85B0000007323B30241B010E84F00000012C073F7753FAA"
	strings:
		$1 = { E9 FF 00 00 00 60 8B 74 24 24 8B 7C 24 28 FC B2 80 33 DB A4 B3 02 E8 6D 00 00 00 73 F6 33 C9 E8 64 00 00 00 73 1C 33 C0 E8 5B 00 00 00 73 23 B3 02 41 B0 10 E8 4F 00 00 00 12 C0 73 F7 75 3F AA }
	condition:
		for any of them : ( $ at pe.entry_point )
}

rule ocbat2exe {
	meta:
		tool = "C"
		name = "ocBat2Exe"
		version = "1.0"
		pattern = "558BECB9080000006A006A004975F9535657B8583C4000E86CFAFFFF33C055688A3F400064FF306489206A006A006A036A006A0168000000808D55EC33C0E881E9FFFF8B45ECE841F6FFFF50E8F3FAFFFF8BF883FFFF0F84830200006A026A006AEE57E8FCFAFFFF6A006860994F006A12681857400057E8E0FAFFFF833D60994F00120F85560200008D45E4508D45E0BA18574000B940420F00E861F4FFFF8B45E0B912000000BA01000000E83BF6FFFF8B45E48D55E8E804FB????????E8B858994F00E867F3FFFF33C0A360994F008D45DC50B905000000BA01000000A158994F00E804F6FFFF8B45DCBAA43F4000E8E3F4FFFF"
	strings:
		$1 = { 55 8B EC B9 08 00 00 00 6A 00 6A 00 49 75 F9 53 56 57 B8 58 3C 40 00 E8 6C FA FF FF 33 C0 55 68 8A 3F 40 00 64 FF 30 64 89 20 6A 00 6A 00 6A 03 6A 00 6A 01 68 00 00 00 80 8D 55 EC 33 C0 E8 81 E9 FF FF 8B 45 EC E8 41 F6 FF FF 50 E8 F3 FA FF FF 8B F8 83 FF FF 0F 84 83 02 00 00 6A 02 6A 00 6A EE 57 E8 FC FA FF FF 6A 00 68 60 99 4F 00 6A 12 68 18 57 40 00 57 E8 E0 FA FF FF 83 3D 60 99 4F 00 12 0F 85 56 02 00 00 8D 45 E4 50 8D 45 E0 BA 18 57 40 00 B9 40 42 0F 00 E8 61 F4 FF FF 8B 45 E0 B9 12 00 00 00 BA 01 00 00 00 E8 3B F6 FF FF 8B 45 E4 8D 55 E8 E8 04 FB ?? ?? ?? ?? E8 B8 58 99 4F 00 E8 67 F3 FF FF 33 C0 A3 60 99 4F 00 8D 45 DC 50 B9 05 00 00 00 BA 01 00 00 00 A1 58 99 4F 00 E8 04 F6 FF FF 8B 45 DC BA A4 3F 40 00 E8 E3 F4 FF FF }
	condition:
		for any of them : ( $ at pe.entry_point )
}

rule f2ko_bat2exe_uv_01 {
	meta:
		tool = "C"
		name = "F2KO Bat2Exe"
		pattern = "68????0000680000000068????????E8????????83C40C6800000000E8????????A3????????680000000068001000006800000000E8????????A3"
	strings:
		$1 = { 68 ?? ?? 00 00 68 00 00 00 00 68 ?? ?? ?? ?? E8 ?? ?? ?? ?? 83 C4 0C 68 00 00 00 00 E8 ?? ?? ?? ?? A3 ?? ?? ?? ?? 68 00 00 00 00 68 00 10 00 00 68 00 00 00 00 E8 ?? ?? ?? ?? A3 }
	condition:
		$1 at pe.entry_point
}

rule adv_bat_to_exe_uv_01 {
	meta:
		tool = "C"
		name = "Advanced BAT to EXE Converter"
		pattern = "B94FC3000033C08DBD????????F3AB66ABAAA0????????8885????????B96918000033C08DBD????????F3AB66ABAA8A0D????????888D????????B94000000033C08DBD????????F3AB66ABAA8A15????????8895????????B95900000033C08D"
	strings:
		$1 = { B9 4F C3 00 00 33 C0 8D BD ?? ?? ?? ?? F3 AB 66 AB AA A0 ?? ?? ?? ?? 88 85 ?? ?? ?? ?? B9 69 18 00 00 33 C0 8D BD ?? ?? ?? ?? F3 AB 66 AB AA 8A 0D ?? ?? ?? ?? 88 8D ?? ?? ?? ?? B9 40 00 00 00 33 C0 8D BD ?? ?? ?? ?? F3 AB 66 AB AA 8A 15 ?? ?? ?? ?? 88 95 ?? ?? ?? ?? B9 59 00 00 00 33 C0 8D }
	condition:
		$1
}

rule exescript_uv_01 {
	meta:
		tool = "C"
		name = "ExeScript"
		pattern = "558BEC81EC????????566A00FF15????????8985????????C785????????00000000C745????????00FF15????????A3"
	strings:
		$1 = { 55 8B EC 81 EC ?? ?? ?? ?? 56 6A 00 FF 15 ?? ?? ?? ?? 89 85 ?? ?? ?? ?? C7 85 ?? ?? ?? ?? 00 00 00 00 C7 45 ?? ?? ?? ?? 00 FF 15 ?? ?? ?? ?? A3 }
		$2 = "<!-- ----- ExeScript Options Begin -----"
	condition:
		$1 at pe.entry_point and
		@2 > pe.sections[pe.section_index(".rdata")].raw_data_offset and
		@2 < pe.sections[pe.section_index(".rdata")].raw_data_offset +
			pe.sections[pe.section_index(".rdata")].raw_data_size
}

rule plugintoexe_100 {
	meta:
		tool = "P"
		name = "PluginToExe"
		version = "1.00"
		extra = "for PEiD"
		pattern = "E80000000029C05D81EDD140400050FF95B8404000898509404000FF95B440400089851140400050FF95C04040008A0880F922750750FF95C440400089850D4040008B9D09404000606A006A015381C3??????00FFD3616A006844694550FFB50D4040006A0081C3??????00FFD383C410FF95B0404000"
	strings:
		$1 = { E8 00 00 00 00 29 C0 5D 81 ED D1 40 40 00 50 FF 95 B8 40 40 00 89 85 09 40 40 00 FF 95 B4 40 40 00 89 85 11 40 40 00 50 FF 95 C0 40 40 00 8A 08 80 F9 22 75 07 50 FF 95 C4 40 40 00 89 85 0D 40 40 00 8B 9D 09 40 40 00 60 6A 00 6A 01 53 81 C3 ?? ?? ?? 00 FF D3 61 6A 00 68 44 69 45 50 FF B5 0D 40 40 00 6A 00 81 C3 ?? ?? ?? 00 FF D3 83 C4 10 FF 95 B0 40 40 00 }
	condition:
		for any of them : ( $ at pe.entry_point )
}

rule plugintoexe_101 {
	meta:
		tool = "P"
		name = "PluginToExe"
		version = "1.01"
		extra = "for PEiD"
		pattern = "E80000000029C05D81EDC6414000508F857140400050FF95A541400089856D404000FF95A141400050FF95B541400080380074168A0880F922750750FF95B9414000898575404000EB6C6A018F85714040006A586A40FF95A941400089856940400089C768000800006A40FF95A941400089471CC70758000000C7472000080000C7471801000000C74734041088008D8DB9404000894F0C8D8DDB404000894F30FFB569404000FF9595414000FF771C8F85754040008B9D6D404000606A006A015381C3??????00FFD3616A006844694550FFB5754040006A0081C3????0000FFD383C41083BD71404000007410FF771CFF95AD41400057FF95AD4140006A00FF959D414000"
	strings:
		$1 = { E8 00 00 00 00 29 C0 5D 81 ED C6 41 40 00 50 8F 85 71 40 40 00 50 FF 95 A5 41 40 00 89 85 6D 40 40 00 FF 95 A1 41 40 00 50 FF 95 B5 41 40 00 80 38 00 74 16 8A 08 80 F9 22 75 07 50 FF 95 B9 41 40 00 89 85 75 40 40 00 EB 6C 6A 01 8F 85 71 40 40 00 6A 58 6A 40 FF 95 A9 41 40 00 89 85 69 40 40 00 89 C7 68 00 08 00 00 6A 40 FF 95 A9 41 40 00 89 47 1C C7 07 58 00 00 00 C7 47 20 00 08 00 00 C7 47 18 01 00 00 00 C7 47 34 04 10 88 00 8D 8D B9 40 40 00 89 4F 0C 8D 8D DB 40 40 00 89 4F 30 FF B5 69 40 40 00 FF 95 95 41 40 00 FF 77 1C 8F 85 75 40 40 00 8B 9D 6D 40 40 00 60 6A 00 6A 01 53 81 C3 ?? ?? ?? 00 FF D3 61 6A 00 68 44 69 45 50 FF B5 75 40 40 00 6A 00 81 C3 ?? ?? 00 00 FF D3 83 C4 10 83 BD 71 40 40 00 00 74 10 FF 77 1C FF 95 AD 41 40 00 57 FF 95 AD 41 40 00 6A 00 FF 95 9D 41 40 00 }
	condition:
		for any of them : ( $ at pe.entry_point )
}

rule plugintoexe_102 {
	meta:
		tool = "P"
		name = "PluginToExe"
		version = "1.02"
		extra = "for PEiD"
		pattern = "E80000000029C05D81ED32424000508F85DD40400050FF95114240008985D9404000FF950D42400050FF952142400080380074168A0880F922750750FF95254240008985E1404000EB6C6A018F85DD4040006A586A40FF95154240008985D540400089C768000800006A40FF951542400089471CC7075800"
	strings:
		$1 = { E8 00 00 00 00 29 C0 5D 81 ED 32 42 40 00 50 8F 85 DD 40 40 00 50 FF 95 11 42 40 00 89 85 D9 40 40 00 FF 95 0D 42 40 00 50 FF 95 21 42 40 00 80 38 00 74 16 8A 08 80 F9 22 75 07 50 FF 95 25 42 40 00 89 85 E1 40 40 00 EB 6C 6A 01 8F 85 DD 40 40 00 6A 58 6A 40 FF 95 15 42 40 00 89 85 D5 40 40 00 89 C7 68 00 08 00 00 6A 40 FF 95 15 42 40 00 89 47 1C C7 07 58 00 }
	condition:
		for any of them : ( $ at pe.entry_point )
}

rule reg2exe_220_221 {
	meta:
		tool = "P"
		name = "Reg2Exe"
		version = "2.20, 2.21"
		pattern = "6A00E87D120000A3A0444000E8791200006A0A506A00FF35A0444000E80F00000050E869120000CCCCCCCCCCCCCCCCCC682C020000680000000068B0444000E83A12000083C40C8B442404A3B8444000680000000068A00F00006800000000E832120000A3B044400068F401000068BC444000FF35B8444000E81E120000B8BC44400089C18A304080FE5C750289C180FE0075F1C60100E8EC180000E828160000E84A1200006800FA00006808000000FF35B0444000E8E7110000A3B44440008B15D4464000E8650A0000BB00001000B801000000E8720A00007409C7000100000083C004A3D4464000FF35B4444000E8260500008D0DB84640005AE8CF0F0000FF35B4444000FF35B8464000E8EE0600008D0DB44640005AE8"
	strings:
		$1 = { 6A 00 E8 7D 12 00 00 A3 A0 44 40 00 E8 79 12 00 00 6A 0A 50 6A 00 FF 35 A0 44 40 00 E8 0F 00 00 00 50 E8 69 12 00 00 CC CC CC CC CC CC CC CC CC 68 2C 02 00 00 68 00 00 00 00 68 B0 44 40 00 E8 3A 12 00 00 83 C4 0C 8B 44 24 04 A3 B8 44 40 00 68 00 00 00 00 68 A0 0F 00 00 68 00 00 00 00 E8 32 12 00 00 A3 B0 44 40 00 68 F4 01 00 00 68 BC 44 40 00 FF 35 B8 44 40 00 E8 1E 12 00 00 B8 BC 44 40 00 89 C1 8A 30 40 80 FE 5C 75 02 89 C1 80 FE 00 75 F1 C6 01 00 E8 EC 18 00 00 E8 28 16 00 00 E8 4A 12 00 00 68 00 FA 00 00 68 08 00 00 00 FF 35 B0 44 40 00 E8 E7 11 00 00 A3 B4 44 40 00 8B 15 D4 46 40 00 E8 65 0A 00 00 BB 00 00 10 00 B8 01 00 00 00 E8 72 0A 00 00 74 09 C7 00 01 00 00 00 83 C0 04 A3 D4 46 40 00 FF 35 B4 44 40 00 E8 26 05 00 00 8D 0D B8 46 40 00 5A E8 CF 0F 00 00 FF 35 B4 44 40 00 FF 35 B8 46 40 00 E8 EE 06 00 00 8D 0D B4 46 40 00 5A E8 }
	condition:
		for any of them : ( $ at pe.entry_point )
}

rule reg2exe_222_223 {
	meta:
		tool = "P"
		name = "Reg2Exe"
		version = "2.22, 2.23"
		pattern = "6A00E82F1E0000A3C4354000E82B1E00006A0A506A00FF35C4354000E80700000050E81B1E0000CC6848000000680000000068C8354000E87616000083C40C8B442404A3CC354000680000000068A00F00006800000000E8EC1D0000A3C8354000E8621D0000E8921A0000E880160000E8131400006801000000680836400068000000008B1508364000E8713F0000B800001000BB01000000E8823F0000FF3548314000B800010000E80D1300008D0DEC3540005AE8F21300006800010000FF35EC354000E8841D0000A3F4354000FF3548314000FF35F4354000FF35EC354000E8"
	strings:
		$1 = { 6A 00 E8 2F 1E 00 00 A3 C4 35 40 00 E8 2B 1E 00 00 6A 0A 50 6A 00 FF 35 C4 35 40 00 E8 07 00 00 00 50 E8 1B 1E 00 00 CC 68 48 00 00 00 68 00 00 00 00 68 C8 35 40 00 E8 76 16 00 00 83 C4 0C 8B 44 24 04 A3 CC 35 40 00 68 00 00 00 00 68 A0 0F 00 00 68 00 00 00 00 E8 EC 1D 00 00 A3 C8 35 40 00 E8 62 1D 00 00 E8 92 1A 00 00 E8 80 16 00 00 E8 13 14 00 00 68 01 00 00 00 68 08 36 40 00 68 00 00 00 00 8B 15 08 36 40 00 E8 71 3F 00 00 B8 00 00 10 00 BB 01 00 00 00 E8 82 3F 00 00 FF 35 48 31 40 00 B8 00 01 00 00 E8 0D 13 00 00 8D 0D EC 35 40 00 5A E8 F2 13 00 00 68 00 01 00 00 FF 35 EC 35 40 00 E8 84 1D 00 00 A3 F4 35 40 00 FF 35 48 31 40 00 FF 35 F4 35 40 00 FF 35 EC 35 40 00 E8 }
	condition:
		for any of them : ( $ at pe.entry_point )
}

rule reg2exe_224 {
	meta:
		tool = "P"
		name = "Reg2Exe"
		version = "2.24"
		pattern = "6A00E8CF200000A3F4454000E8CB2000006A0A506A00FF35F4454000E80700000050E8BB200000CC6848000000680000000068F8454000E80619000083C40C8B442404A3FC454000680000000068A00F00006800000000E88C200000A3F8454000E802200000E8321D0000E820190000E8A31600006801000000683846400068000000008B1538464000E8714F0000B800001000BB01000000E8824F0000FF3548414000B800010000E89D1500008D0D1C4640005AE8821600006800010000FF351C464000E824200000A324464000FF3548414000FF3524464000FF351C464000E8DC1000008D0D144640005AE84A16"
	strings:
		$1 = { 6A 00 E8 CF 20 00 00 A3 F4 45 40 00 E8 CB 20 00 00 6A 0A 50 6A 00 FF 35 F4 45 40 00 E8 07 00 00 00 50 E8 BB 20 00 00 CC 68 48 00 00 00 68 00 00 00 00 68 F8 45 40 00 E8 06 19 00 00 83 C4 0C 8B 44 24 04 A3 FC 45 40 00 68 00 00 00 00 68 A0 0F 00 00 68 00 00 00 00 E8 8C 20 00 00 A3 F8 45 40 00 E8 02 20 00 00 E8 32 1D 00 00 E8 20 19 00 00 E8 A3 16 00 00 68 01 00 00 00 68 38 46 40 00 68 00 00 00 00 8B 15 38 46 40 00 E8 71 4F 00 00 B8 00 00 10 00 BB 01 00 00 00 E8 82 4F 00 00 FF 35 48 41 40 00 B8 00 01 00 00 E8 9D 15 00 00 8D 0D 1C 46 40 00 5A E8 82 16 00 00 68 00 01 00 00 FF 35 1C 46 40 00 E8 24 20 00 00 A3 24 46 40 00 FF 35 48 41 40 00 FF 35 24 46 40 00 FF 35 1C 46 40 00 E8 DC 10 00 00 8D 0D 14 46 40 00 5A E8 4A 16 }
	condition:
		for any of them : ( $ at pe.entry_point )
}

rule reg2exe_225 {
	meta:
		tool = "P"
		name = "Reg2Exe"
		version = "2.25"
		pattern = "6868000000680000000068707D4000E8AE20000083C40C6800000000E8AF520000A3747D4000680000000068001000006800000000E89C520000A3707D4000E824500000E8E2480000E844340000E854280000E898270000E893200000680100000068D07D400068000000008B15D07D4000E8898F0000B8000010006801000000E89A8F0000FF35A47F40006800010000E83A2300008D0DA87D40005AE85E1F0000FF35A87D40006800010000E82A520000A3B47D4000FF35A47F4000FF35B47D4000FF35A87D4000E85C0C00008D0DA07D40005AE8261F0000FF35"
	strings:
		$1 = { 68 68 00 00 00 68 00 00 00 00 68 70 7D 40 00 E8 AE 20 00 00 83 C4 0C 68 00 00 00 00 E8 AF 52 00 00 A3 74 7D 40 00 68 00 00 00 00 68 00 10 00 00 68 00 00 00 00 E8 9C 52 00 00 A3 70 7D 40 00 E8 24 50 00 00 E8 E2 48 00 00 E8 44 34 00 00 E8 54 28 00 00 E8 98 27 00 00 E8 93 20 00 00 68 01 00 00 00 68 D0 7D 40 00 68 00 00 00 00 8B 15 D0 7D 40 00 E8 89 8F 00 00 B8 00 00 10 00 68 01 00 00 00 E8 9A 8F 00 00 FF 35 A4 7F 40 00 68 00 01 00 00 E8 3A 23 00 00 8D 0D A8 7D 40 00 5A E8 5E 1F 00 00 FF 35 A8 7D 40 00 68 00 01 00 00 E8 2A 52 00 00 A3 B4 7D 40 00 FF 35 A4 7F 40 00 FF 35 B4 7D 40 00 FF 35 A8 7D 40 00 E8 5C 0C 00 00 8D 0D A0 7D 40 00 5A E8 26 1F 00 00 FF 35 }
	condition:
		for any of them : ( $ at pe.entry_point )
}
