/********************************************************************************
*                                                                               *
*                   Julian Date object                                          *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
// Design/implementation taken from jDate by: J. Knight <jimbag@kw.igs.net>
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
using namespace FX;
#include "FXJulianDate.h"
using namespace FXEX;
namespace FXEX {

// some things just dont change...
const FXchar *FXJulianDate::dayNames[]={"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};
const FXchar *FXJulianDate::monthNames[]={"January","February","March","April","May","June","July","August","September","October","November","December"};
static short monthDays[]={0,31,28,31,30,31,30,31,31,30,31,30,31};

// ctor - using today's date
FXJulianDate::FXJulianDate() {
  today();
  }

// ctor - using a specific date
FXJulianDate::FXJulianDate( FXint yr, FXint mo, FXint da ) {
  setDate(yr,mo,da);
  }

// set to specific date
void FXJulianDate::setDate( FXint y, FXint m, FXint d ) {
  julian=greg2jul(y,m,d);
  }

// set to today's date
void FXJulianDate::today() {
#ifndef FX_NATIVE_WIN32
  time_t ltime;
  time(&ltime);
  tm *t=localtime(&ltime);
  julian=greg2jul(t->tm_year+1900,t->tm_mon+1,t->tm_mday);
#else
  SYSTEMTIME t;
  GetLocalTime(&t);
  julian = greg2jul(t.wYear,t.wMonth,t.wDay);
#endif
  }

// return the month name
const FXchar *FXJulianDate::monthName( FXint month ) const {
  return monthNames[month-1];
  }

// return the day name
const FXchar *FXJulianDate::dayName( FXint day ) const {
  return dayNames[day];
  }

// number of days in the months
FXint FXJulianDate::daysInMonth() {
  FXint y,m,d;
  jul2greg(julian,y,m,d);
  if( m == 2 && leapYear(y)) return 29;
  return monthDays[m];
  }

// Sun = 0, ...
FXint FXJulianDate::dayOfWeek() {
  return(((julian+1)%7)+6)%7;
  }

// the day
FXint FXJulianDate::day() {
  FXint d, m, y;
  jul2greg(julian,y,m,d);
  return d;
  }

// Jan = 1, ...
FXint FXJulianDate::month() {
  FXint d, m, y;
  jul2greg(julian,y,m,d);
  return m;
  }

// year since 1900
FXint FXJulianDate::year() {
  FXint d, m, y;
  jul2greg(julian,y,m,d);
  return y;
  }

// day of year - first day of year = 1
FXint FXJulianDate::dayOfYear() {
  FXint y,m,d;
  jul2greg(julian,y,m,d);
  return julian - greg2jul(y,1,1) + 1;
  }

// is value a leap year?
FXbool FXJulianDate::leapYear(FXint y) {
  return y%4 == 0 && y%100 != 0 || y%400 == 0;
  }

// convert date to Julian date
FXuint FXJulianDate::greg2jul(FXint y,FXint m,FXint d) {
  FXuint c, ya;
  if(y <= 99) y += 1900;
  if(m > 2) m -= 3;
  else { m += 9; y--; }
  c = y;
  c /= 100;
  ya = y - 100 * c;
  return 1721119 + d + (146097 * c) / 4 + (1461 * ya ) / 4 + (153 * m + 2 ) / 5;
  }

// convet date to Gregorian date
void FXJulianDate::jul2greg(FXuint jd,FXint &y,FXint &m,FXint &d) {
  FXuint x;
  FXuint j = jd - 1721119;
  y = (j*4 - 1)/146097;
  j = j*4 - 146097*y -1;
  x = j/4;
  j = (x*4+3)/1461;
  y = 100*y+j;
  x = (x*4)+3 - 1461*j;
  x = (x+4)/4;
  m = (5*x-3)/153;
  x = 5*x-3-153*m;
  d = (x+5)/5;
  if( m < 10 )
    m += 3;
  else {
    m -= 9;
    y++;
    }
  }

// save to store
FXStream& operator<< (FXStream& store,const FXJulianDate& d) {
  store << d.julian;
  return store;
  }

// load from store
FXStream& operator>> (FXStream& store,FXJulianDate& d) {
  store >> d.julian;
  return store;
  }

}

