/********************************************************************************
*                                                                               *
*               Interlocking object (a semaphore)                               *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXString.h>
#include <fox/FXStream.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
using namespace FX;
#include "exincs.h"
#include "fxexdefs.h"
#include "FXMutex.h"
#include "FXInterlock.h"
using namespace FXEX;
namespace FXEX {

#ifndef WIN32
// Compare times
static inline int operator<(const struct timeval& a,const struct timeval& b){
  return (a.tv_sec<b.tv_sec) || (a.tv_sec==b.tv_sec && a.tv_usec<b.tv_usec);
  }
#endif


// map
FXIMPLEMENT(FXInterlock,FXWaitable,NULL,0)

// create interlockable object
FXInterlock::FXInterlock(FXObject *tgt,FXSelector sel) : FXWaitable(tgt,sel) {
  wait_=0;
#ifndef WIN32
  FXMALLOC(&semaphore,sem_t,1);
  if(sem_init((sem_t*)semaphore,0,0)!=0){
#else
  if((semaphore=CreateSemaphore(NULL,0,65536,NULL))==NULL){
#endif
    fxerror("%s: Could not create semaphore\n",getClassName());
    }
  }

// destroy resources
FXInterlock::~FXInterlock(){
  mutex->lock();
#ifndef WIN32
  sem_getvalue((sem_t*)semaphore,&wait_);
#endif
  if (wait_>0) fxerror("%s: wait condition is non-zero\n",getClassName());
#ifndef WIN32
  sem_destroy((sem_t*)semaphore);
  FXFREE(&semaphore);
#else
  CloseHandle(semaphore);
#endif
  mutex->unlock();
  }

// wait indefinately
void FXInterlock::wait(){
  mutex->lock();
#ifndef WIN32
  sem_wait((sem_t*)semaphore);
#else
  while (1){
    DWORD result = WaitForSingleObject(semaphore,INFINITE);
    if (result==WAIT_OBJECT_0) break;
    }
  wait_++;
#endif
  FXWaitable::wait();
  mutex->unlock();
  }
  

// wait some period of time for a semaphore
FXbool FXInterlock::trywait(FXuint ms){
#ifndef WIN32
  if (sem_trywait((sem_t*)semaphore)==0) { return FXWaitable::trywait(ms); }
  if (ms==0) return FALSE;
  struct timeval t1,t2;
  gettimeofday(&t1,NULL);
  FXUtils::convert_timeval(&t1,ms);
  gettimeofday(&t2,NULL);
  while(t2<t1) {   // FIXME this is an annoying hack
    mutex->lock();
    if (sem_trywait((sem_t*)semaphore)==0) { mutex->unlock(); return FXWaitable::trywait(ms); }
    mutex->unlock();
    fxsleep(ms/10);
    gettimeofday(&t2,NULL);
    }
  return FALSE;
#else
  DWORD result = WaitForSingleObject(semaphore,ms);
  if (result==WAIT_OBJECT_0) {
    mutex->lock();
    wait_++;
    mutex->unlock();
    return FXWaitable::trywait(ms);
    }
  return FALSE;
#endif
  }

// wake some thread waiting in the wait()
void FXInterlock::activate(){
#ifndef WIN32
  mutex->lock();
  sem_getvalue((sem_t*)semaphore,&wait_);
  if (wait_>0) sem_post((sem_t*)semaphore);
#else
  ReleaseSemaphore(semaphore,1,NULL);
  mutex->lock();
  if (wait>0) wait_--;
#endif
  FXWaitable::activate();
  mutex->unlock();
  }

// wake up all objects waiting in the wait()
void FXInterlock::activateAll(){
  mutex->lock();
#ifndef WIN32
  sem_getvalue((sem_t*)semaphore,&wait_);
  while (wait_>0) {
    sem_post((sem_t*)semaphore);
    mutex->unlock();
    fxsleep(1);  // yield the processor a little, so the wait()ed condition can continue
    mutex->lock();
    sem_getvalue((sem_t*)semaphore,&wait_);
    }
#else
  while (wait_>0) {
    mutex->unlock();
    ReleaseSemaphore(semaphore,1,NULL);
    Sleep(0);  // yield the processor
    mutex->lock();
    wait_--;
    }
#endif
  FXWaitable::activateAll();
  mutex->unlock();
  }

// return the current value of the wait count
FXint FXInterlock::getWaitCount(){
  FXint w;
  mutex->lock();
#ifndef WIN32
  sem_getvalue((sem_t*)semaphore,&wait_);
#endif
  w=wait_;
  mutex->unlock();
  return w;
  }

}

