/********************************************************************************
*                                                                               *
*                       FXVTable                                                *
*                                                                               *
*********************************************************************************
* Copyright (C) 2002 by Nikitinskiy Dmitriy.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#ifndef FXVTABLE_H
#define FXVTABLE_H

#define DEFAULT_MARGIN 2
#define DEFAULT_ROW_HEIGHT 16
#define DEFAULT_COL_WIDTH  80

#ifndef FXCOMPOSITE_H
#include <fox/FXComposite.h>
#endif
namespace FX {
class FXVTable;

/// Cell position
struct FXVTableCellPos {
  FXint row, col, rowSw, colSw;
  FXVTableCellPos() : row(0), col(0), rowSw(0), colSw(0) {};
  FXVTableCellPos(FXint r, FXint c, FXint rs=0, FXint cs=0) : row(r), col(c), rowSw(rs), colSw(cs) {};
  };

/// Table style
class FXVTableStyle {
  friend class FXVTableStyles;

public:
  FXVTableStyle *next;
  FXint   row;
  FXint   col;
  FXint   rowHeight; // If first cell of join. Hold count(cells) of joins. If not first hold relation to fist (<0)
  FXint   colWidth;  // else hold 0
  FXColor background;
  FXColor text;
  FXColor border;
  FXuint  parameters;
  FXuint  cellType; // Registered cell types

public:
  enum{
    DISABLED   = 0x00000001,
    DRAGGABLE  = 0x00000002,
    NOSELECT   = 0x00000004,
    FOCUSED    = 0x00000008,
    SELECTED   = 0x00000010,
    JOIN       = 0x00000020,
    EDITABLE   = 0x00000040,
    RIGHT      = 0x00000100,      /// Align on right
    LEFT       = 0x00000200,      /// Align on left
    TOP        = 0x00000400,      /// Align on top
    BOTTOM     = 0x00000800,      /// Align on bottom
    LBORDER    = 0x00001000,      /// Draw left border
    RBORDER    = 0x00002000,      /// Draw right border
    TBORDER    = 0x00004000,      /// Draw top border
    BBORDER    = 0x00008000,      /// Draw bottom border
    FREE_OPTION= 0x00010000,      /// For use in FXVTableItems...
    LAST_OPTION= 0x80000000
    };

public:
  FXVTableStyle(FXuint p=EDITABLE, FXint r=-1, FXint c=-1, FXint rh=DEFAULT_ROW_HEIGHT, FXint cw=DEFAULT_COL_WIDTH, FXColor back=FXRGB(255,255,255), FXColor t=FXRGB(0,0,0), FXColor bord=FXRGB(127,127,127)) : row(r), col(c), rowHeight(rh), colWidth(cw), background(back), text(t), border(bord), parameters(p), cellType(0) {};
  FXVTableStyle(const FXVTableStyle& s);
  FXVTableStyle& operator=(const FXVTableStyle& f);
  
  FXbool isJoined() const { return parameters&JOIN && (rowHeight<0 || colWidth<0); };
  FXbool isJoining() const { return parameters&JOIN && rowHeight>0 && colWidth>0; };
  FXbool isSelected() const { return parameters&SELECTED; };
  FXbool isFocused() const { return parameters&FOCUSED; };
  FXbool isEditable() const { return parameters&EDITABLE; };
  };


/// table styles
class FXVTableStyles {
  friend class FXVTable;

private:
  FXVTableStyle global;
  FXVTableStyle* rows;
  FXVTableStyle* cols;
  FXVTableStyle* cells;

  FXVTableStyle select;
  FXVTableStyle* selection;

protected:
  void addElem(FXVTableStyle** s,FXVTableStyle* a){
     a->next=*s;
     *s=a;
  };
  FXint delElem(FXVTableStyle** s,FXVTableStyle* a);
  void delElems(FXVTableStyle**,FXint,FXint);
  void delList(FXVTableStyle* l);
  FXVTableStyle* lookup(FXVTableStyle*,FXint,FXint) const;
  void changeAttr(FXint FXVTableStyle::*a, FXint before, FXint d);

public:

  FXVTableStyles() : rows(NULL), cols(NULL), cells(NULL), selection(NULL) {};
  //
  //  All cellPosition must be INTERNAL
  //
  const FXVTableStyle& getStyle(FXint r, FXint c) const { return getStyle(FXVTableCellPos(r,c)); };
  const FXVTableStyle& getStyle(const FXVTableCellPos& cell) const; // First look in selection.
  void setStyle(FXint r, FXint c, const FXVTableStyle& s) { setStyle(FXVTableCellPos(r,c),s); };
  void setStyle(const FXVTableCellPos& cell,const FXVTableStyle& s);
  void delStyle(FXint r, FXint c) { delStyle(FXVTableCellPos(r,c)); };
  void delStyle(const FXVTableCellPos& cell);

  FXbool cellJoined(const FXVTableCellPos& cell);
  
  const FXVTableStyle& getSelectStyle() const { return select; };
  void setSelectStyle(const FXVTableStyle& s) { select=s; } ;
  void addSelection(FXint r, FXint c) { addSelection(FXVTableCellPos(r,c)); };
  void addSelection(const FXVTableCellPos& cell);
  void delSelection(FXint r, FXint c) { delSelection(FXVTableCellPos(r,c)); };
  void delSelection(const FXVTableCellPos& cell);
  void clearSelection();
  FXbool isSelected(const FXVTableCellPos& cell) const { return lookup(selection,cell.row,cell.col)!=NULL; };
  FXVTableStyle* getSelection() const { return selection; };

  void insertColumn(FXint before);
  void insertRow(FXint before);
  void deleteColumn(FXint col);
  void deleteRow(FXint row);

  ~FXVTableStyles();
  };


// a cell event
struct FXVTableCellEvent {
  FXDC* dc;
  FXVTableStyle style;
  FXVTableCellPos pos; // Internal coordinates
  FXint x, y, w,h;
  FXEvent* event;
  FXVTableCellEvent(FXDC* d,const FXVTableStyle& s, const FXVTableCellPos& cell, FXint x_, FXint y_, FXint w_, FXint h_, FXEvent* e=NULL) : dc(d),style(s), pos(cell), x(x_), y(y_), w(w_), h(h_), event(e) {};
  FXVTableCellEvent(FXDC* d,const FXVTableStyle& s, const FXVTableCellPos& cell, const FXRectangle& rect, FXEvent* e=NULL) : dc(d),style(s), pos(cell), x(rect.x), y(rect.y), w(rect.w), h(rect.h), event(e) {};
  };


/// a source of table info
class FXVTableSource {
public:
  virtual ~FXVTableSource() {};
  virtual FXint getRows()=0; // Return total rows= Leading+Normal+Trailing
  virtual FXint getCols()=0; // Return total cols= Leading+Normal+Trailing

  virtual FXbool verifyCell(const FXVTableCellPos& cell)=0;
  virtual FXbool addRequest(const FXVTableCellPos& cell)=0;
  virtual FXbool delRequest(const FXVTableCellPos& cell)=0;

  virtual void add(const FXVTableCellPos& cell)=0;
  virtual void del(const FXVTableCellPos& cell)=0;
  
  virtual FXString getData(const FXVTableCellPos& cell)=0;
  virtual FXbool setData(const FXVTableCellPos& cell, const FXString& d)=0;
  };



class FXVTableItem : public FXObject {
  FXDECLARE(FXVTableItem)

protected:
   FXVTable *table;
    
   FXVTableItem() : table(NULL) {};
   
   void drawFrame(FXVTableCellEvent* ev);
public:
    
    enum {
      ID_LAST=1
    };
    
    FXVTableItem(FXVTable* t): table(t) {};

    virtual ~FXVTableItem() {};
    virtual void scrollContext(FXint x, FXint y, FXint w, FXint h, FXint dx, FXint dy) {};

// Possibly remove this members in future
// cell argument in INTERNAL coordinates
/*virtual*/ FXString getData(const FXVTableCellPos& cell);
/*virtual*/ void setData(const FXVTableCellPos& cell, const FXString& data);
};


/// cell item - as a label
class FXVTableLabel : public FXVTableItem {
  FXDECLARE(FXVTableLabel)

protected:
  FXbool editorChanged;
  FXTextField *editor;
  FXVTableCellPos editorPos; // Internal coordinates
    
  FXVTableLabel() : editor(NULL), editorChanged(FALSE) {};
 
  void drawLabel(FXVTableCellEvent* ev);
  void drawText(FXVTableCellEvent* ev);

public:
  enum {
    ID_EDITOR = FXVTableItem::ID_LAST,
    ID_LAST
    };

public:
  long onPaint(FXObject*,FXSelector,void*);
  long onFocusIn(FXObject*,FXSelector,void*);
  long onFocusOut(FXObject*,FXSelector,void*);
  long onClicked(FXObject*,FXSelector,void*);
  long onKeyPress(FXObject*,FXSelector,void*);
  long onEditorChanged(FXObject*,FXSelector,void*);
  long onEditorCommand(FXObject*,FXSelector,void*);

public:
  FXVTableLabel(FXVTable* t);
  ~FXVTableLabel();
  };


/// cell item - as a button
class FXVTableButton : public FXVTableLabel {
  FXDECLARE(FXVTableButton)

protected:
  FXColor hilite;
  FXColor shadow;
  FXColor base;

protected:  
  /// serialisation
  FXVTableButton() {};

  /// draw the button
  void drawButton(FXVTableCellEvent* ev);

public:
  enum {
    PUSHED = FXVTableStyle::FREE_OPTION
    };

public:
  long onPaint(FXObject*,FXSelector,void*);
  long onFocusIn(FXObject*,FXSelector,void*);
  long onFocusOut(FXObject*,FXSelector,void*);
  long onClicked(FXObject*,FXSelector,void*);
  long onKeyPress(FXObject*,FXSelector,void*);
 
public:
  FXVTableButton(FXVTable* t);
  };


/// a cell based event
struct FXVTableEvent {
  FXVTableCellPos cell;
  FXint event;

  enum {
    CELL_CHANGED,
    CELL_SELECT,    //if col==-1 then row (de)selected
    CELL_DESELECT,  //if row==-1 then col (de)selected
    CELL_FOCUSMOVE, //new focus position
    };

  FXVTableEvent(const FXVTableCellPos c, FXint e) : cell(c), event(e) {}; 
  };


/// table widget
class FXVTable : public FXComposite {
  FXDECLARE(FXVTable)
  friend class FXVTableItem;
  
protected:
  FXint rows;
  FXint cols;
  FXint rowsLeading, rowsTrailing;
  FXint colsLeading, colsTrailing;

  FXVTableStyles styles;

  FXVTableSource* dataSource;

  FXScrollbar *vScroll, *hScroll;

  FXint dataWidth, dataHeight; // Full size of area w/o scrollbars.
  FXRectangle cellsRect; // Rectangle for cells w/o leading/trailing.
  FXint xPos, yPos; //Position upper left corner for firstRow/Col. Usualy 0:0
  FXint firstRow,firstCol; //Visible rows and cols
  FXint lastRow,lastCol;

  FXint cursorRow, cursorCol; // Cursor position
  FXRectangle cursorRect;

  FXint selBeginRow, selBeginCol; //Row/Col for selection begin
  FXint selEndRow,   selEndCol;   //Row/Col for selection end
 
  FXRectangle mouseCell;
  FXint mouseRow, mouseCol;
 
  FXuint mode; // Widget mode 

protected:
  enum{
    NORMAL    = 0x00000000,
    SELECTING = 0x00000001,
    EDITING   = 0x00000002,
    RESIZING_ROW= 0x00000004,
    RESIZING_COL= 0x00000008,
    DRAGGING  = 0x00000010,
    SHIFT_KEY = 0x00000020,
    CTRL_KEY  = 0x00000040,
    KEY_SELECT= 0x00000080
    };

  FXVTableItem** tableItems;
  FXint tableItemsCount;

  FXRectangle noPaint;

protected:
  /// serialisation
  FXVTable() {};

  void drawRange(FXDC& dc,FXRectangle& rect,FXint fr,FXint fc,FXint lr,FXint lc,FXint x, FXint y);
  void drawCell(FXint row, FXint col);
  void drawCells(FXint rb, FXint cb, FXint re, FXint ce);

  
  void recalcView();
  FXint getColsWidth(FXint from,FXint to) const;
  FXint getRowsHeight(FXint from,FXint to) const;
  FXbool cellVisible(FXint row, FXint col) const;
  FXint sendCellEvent(FXuint event, FXVTableCellEvent* ev);

  FXRectangle getCellRect(FXint r,FXint c); // Get cell rectangle by row/col
  FXRectangle getCellRect(FXint cx, FXint cy, FXint& row, FXint& col); // Get cell parameters by point x/y
  FXSize getCellSize(FXint r, FXint c);

void addSelectionRange(FXint rowB, FXint colB, FXint rowE, FXint colE);
  void moveCursorPos(FXint row, FXint col, FXRectangle* r=NULL);
  
  FXVTableCellPos convToInt(const FXVTableCellPos&) const;
  FXVTableCellPos convFromInt(const FXVTableCellPos&) const;

  void scrollContext(FXint x, FXint y, FXint w, FXint h, FXint dx, FXint dy);

  virtual void layout();

public:
  enum{
    ID_STYLE_CHANGED=FXComposite::ID_LAST,
    ID_HSCROLL,
    ID_VSCROLL,
    ID_CHANGE_CELL,
    ID_DELETE_COLUMN,
    ID_DELETE_ROW,
    ID_INSERT_COLUMN,
    ID_INSERT_ROW,
    ID_SELECT_COLUMN,
    ID_SELECT_ROW,
    ID_SELECT_CELL,
    ID_SELECT_ALL,
    ID_DESELECT_ALL,
    ID_MOVE_LEFT,
    ID_MOVE_RIGHT,
    ID_MOVE_UP,
    ID_MOVE_DOWN,
    ID_MOVE_HOME,
    ID_MOVE_END,
    ID_MOVE_TOP,
    ID_MOVE_BOTTOM,
    ID_MOVE_PAGEDOWN,
    ID_MOVE_PAGEUP,
    ID_LAST
    };

public:
  long onPaint(FXObject*,FXSelector,void*);    
  long onFocusIn(FXObject*,FXSelector,void*);
  long onFocusOut(FXObject*,FXSelector,void*);
  long onMotion(FXObject*,FXSelector,void*);
  long onKeyPress(FXObject*,FXSelector,void*);
  long onKeyRelease(FXObject*,FXSelector,void*);
  long onLeftBtnPress(FXObject*,FXSelector,void*);
  long onLeftBtnRelease(FXObject*,FXSelector,void*);
  long onRightBtnPress(FXObject*,FXSelector,void*);
  long onRightBtnRelease(FXObject*,FXSelector,void*);
  long onUngrabbed(FXObject*,FXSelector,void*);
  long onClicked(FXObject*,FXSelector,void*);
  long onDoubleClicked(FXObject*,FXSelector,void*);
  long onCommand(FXObject*,FXSelector,void*);

  long onHScroll(FXObject*,FXSelector,void*);
  long onVScroll(FXObject*,FXSelector,void*);
 
  long onCmdMoveLeft(FXObject*,FXSelector,void*);
  long onCmdMoveRight(FXObject*,FXSelector,void*);
  long onCmdMoveUp(FXObject*,FXSelector,void*);
  long onCmdMoveDown(FXObject*,FXSelector,void*);
  long onCmdMoveHome(FXObject*,FXSelector,void*);
  long onCmdMoveEnd(FXObject*,FXSelector,void*);
  long onCmdMoveTop(FXObject*,FXSelector,void*);
  long onCmdMoveBottom(FXObject*,FXSelector,void*);
  long onCmdMovePageUp(FXObject*,FXSelector,void*);
  long onCmdMovePageDown(FXObject*,FXSelector,void*);

public:
  FXVTable(FXComposite *p,FXVTableSource* src,FXObject* tgt=NULL,FXSelector sel=0,FXuint opts=0,FXint x=0,FXint y=0,FXint w=0,FXint h=0);

  virtual void create();
  virtual void detach();
//  virtual void recalc();

  /// Table widget can receive focus
  virtual FXbool canFocus() const;

  /// Move the focus to this window
  virtual void setFocus();

  /// Remove the focus from this window
  virtual void killFocus();

  /// Getting row/col number.
  FXint getLeadingRow() const { return rowsLeading; };
  FXint getTrailingRow() const { return rowsTrailing; };
  FXint getLeadingCol() const { return colsLeading; };
  FXint getTrailingCol() const { return colsTrailing; };
  //Removing leading/trailing row/col.
  FXint allocLeadingRow() { firstRow++; return ++rowsLeading; };  
  FXint allocTrailingRow() { lastRow--; return ++rowsTrailing; };
  FXint allocLeadingCol() { firstCol++; return ++colsLeading; };  
  FXint allocTrailingCol() { lastCol--; return ++colsTrailing; };
  FXint delLeadingRow() { firstRow--; return --rowsLeading; };  
  FXint delTrailingRow() { lastRow++; return --rowsTrailing; };
  FXint delLeadingCol() { firstCol--; return --colsLeading; };  
  FXint delTrailingCol() { lastCol++; return --colsTrailing; };

  FXbool cellSelected(const FXVTableCellPos& cell) { return styles.isSelected(convToInt(cell)); };
  void clearSelection();
    

  FXint getRowsCount() const { return rows; };
  FXint getColsCount() const { return cols; };

  FXVTableSource* getDataSource() const { return dataSource; };
  void setDataSource(FXVTableSource* src) {};

  const FXVTableStyle getCellStyle(const FXVTableCellPos& cell) const { return styles.getStyle(convToInt(cell)); };
  void setCellStyle(const FXVTableCellPos& cell, const FXVTableStyle& s) { styles.setStyle(convToInt(cell),s); };
  void delCellStyle(const FXVTableCellPos& cell) { styles.delStyle(convToInt(cell)); };
  void setSelectStyle(const FXVTableStyle& s) { styles.setSelectStyle(s); };

  FXbool joinCells(const FXVTableCellPos& cell, FXint right, FXint down); // Makes join cell
  FXbool joinDelete(const FXVTableCellPos& cell); // Unjoin all cells

  void childPosition(FXWindow* o, FXint x, FXint y, FXint w, FXint h);

  virtual ~FXVTable();
  };

}; // namespace FXEX
#endif
