/********************************************************************************
*                                                                               *
*                  XML event generator object                                   *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
using namespace FX;
#include "fxexdefs.h"
#include "FXXmlReader.h"
#include "FXXmlParser.h"
using namespace FXEX;
namespace FXEX {

/* 
 * The reason that the parser and the reader are split into two is that, it will make
 * subsequent development implementing support for SGML (since SGML is a superset
 * of XML) a bit easier - ie the reader could be modified to handle binary data
 */

// map
FXDEFMAP (FXXmlParser) FXXmlParserMap[]={
  FXMAPFUNC(SEL_CHORE,FXXmlParser::ID_XML_PARSER,FXXmlParser::onChore),
  FXMAPFUNC(SEL_TIMEOUT,FXXmlParser::ID_XML_PARSER,FXXmlParser::onTimeout),
  FXMAPFUNC(SEL_TAG,FXXmlParser::ID_XML_PARSER,FXXmlParser::onTag),
  FXMAPFUNC(SEL_CONTENT,FXXmlParser::ID_XML_PARSER,FXXmlParser::onContent),
  FXMAPFUNC(SEL_ERROR,FXXmlParser::ID_XML_PARSER,FXXmlParser::onReadError),
  };
FXIMPLEMENT(FXXmlParser,FXBaseObject,FXXmlParserMap,ARRAYNUMBER(FXXmlParserMap))

// ctor
FXXmlParser::FXXmlParser(FXApp* a,FXStream* s,FXObject* tgt,FXSelector sel) : FXBaseObject(a,tgt,sel),state(FALSE),whitespace(FALSE){
  reader=new FXXmlReader(a,s,this,ID_XML_PARSER);
  }

// dtor
FXXmlParser::~FXXmlParser(){
  getApp()->removeTimeout(this,ID_XML_PARSER);
  getApp()->removeChore(this,ID_XML_PARSER);
  delete reader;
  }

// create resources
void FXXmlParser::create() {
  FXBaseObject::create();
  reader->create();
  }

// set to new stream
void FXXmlParser::setStream(FXStream *s){
  reader->setStream(s);
  }

// return the stream that we are reading from
FXStream* FXXmlParser::getStream(){
  return reader->getStream();
  }

// What we are trying to do is allow the application to keep processing events _while_
// reading in an XML stream, say from a socket. So... it is envisaged that the app
// creates a new socket stream to the URL, then creates a specific parser (ie HTML parser),
// connected to the socket stream.  If the stream takes a while to retrieve data, that's okay,
// we just keep trying to parse the stream, until it is closed or until the 'state' is false.
// 
// an example:
//     - the app creates a HTML layout widget (which probably inherits from some form of
//       freeform layout widget
//     - the HTML layout widget creates a HTML parser object (which inherits from the
//       FXXmlParser object, but understands HTML tag/content events.
//     - the app sends a URL to the layout widget, which forwards it to the HTML parser
//       object, which creates a new FXSocketStream (with the appropriate URL) and attaches
//       that to an FXXmlReader object.
//     - the socket data is read by the SocketStream, into the XmlReader, which creates
//       tag/content events, which the XmlParser/HtmlParser (since the HtmlParser is derived
//       from the XmlParser) unmunges into document-begin/document-end/tag-begin/tag-end/
//       tag/content events.
//     - the HtmlParser then generates appropriate HTML layout events (forwarding them to
//       the HtmlLayout widget) as defined by the  HTML tags that it recognises.
//
// Note: since the SocketStream is waiting for FXApp to forward it data, the HtmlParser/layout
//       engine is the best location to manage the XML/HTML reading vs layout vs socket data
//       reading events.
//
// FIXME: the best way to handle this situation is fox FXStream to inherit from FXObject, so
//       that when new socket data is available, the FXSocketStream can generate an event
//       to tickle this object into reading more data from the stream.
//
FXbool FXXmlParser::parse(){
  FXStream *stream=reader->getStream();
  if (!stream || stream->direction() != FXStreamLoad || stream->status() != FXStreamOK) return FALSE;
  state=TRUE;
  stopPolling=0;
  if (!handle(this,FXSEL(SEL_BEGIN,FXBaseObject::ID_DOCUMENT),NULL) && target)
    target->handle(this,FXSEL(SEL_BEGIN,FXBaseObject::ID_DOCUMENT),NULL);
  while (state && stream->direction() != FXStreamDead) {
    reader->parse();
    getApp()->addChore(this,ID_XML_PARSER);
    getApp()->runUntil(stopPolling);
    stopPolling=0;
    }
  if (!handle(this,FXSEL(SEL_END,FXBaseObject::ID_DOCUMENT),NULL) && target)
    target->handle(this,FXSEL(SEL_END,FXBaseObject::ID_DOCUMENT),NULL);
  return TRUE;
  }

// handle chore for polling for new data (FIXME: this is a bad hack...)
long FXXmlParser::onChore(FXObject*,FXSelector,void*){
  FXStream *s=reader->getStream();
  if (state || s->status() != FXStreamEnd) stopPolling=1;
  else getApp()->addTimeout(this,ID_XML_PARSER,100);
  return 1;
  }

// handle timer for polling for new data (FIXME: this one too...)
long FXXmlParser::onTimeout(FXObject*,FXSelector,void*){
  FXStream *s=reader->getStream();
  if (state || s->status() != FXStreamEnd) stopPolling=1;
  else getApp()->addTimeout(this,ID_XML_PARSER,100);
  return 1;
  }
  
// handle tag
long FXXmlParser::onTag(FXObject*,FXSelector,void* ptr){
  FXString s = (char*) ptr;

  // handle comments
  if (s.left(3) == "!--") {
    s = s.mid(3,s.length()-5).trim();
    if (!handle(this,FXSEL(SEL_TAG,FXBaseObject::ID_COMMENT),(void*)s.text()) && target)
      target->handle(this,FXSEL(SEL_TAG,FXBaseObject::ID_COMMENT),(void*)s.text());
    }

  // handle the special 'xml' tag
  else if (s.left(5).lower() == "?xml "){
    s=s.right(s.length()-5);
    s=s.left(s.length()-1).trim();
    if (!handle(this,FXSEL(SEL_TAG,FXBaseObject::ID_XML),(void*)s.text()) && target)
      target->handle(this,FXSEL(SEL_TAG,FXBaseObject::ID_XML),(void*)s.text());
    }

  // handle the special 'meta' tag
  else if (s.left(4).lower() == "meta "){
    s=s.right(s.length()-5).trim();
    if (!handle(this,FXSEL(SEL_TAG,FXBaseObject::ID_META),(void*)s.text()) && target)
      target->handle(this,FXSEL(SEL_TAG,FXBaseObject::ID_META),(void*)s.text());
    }

  // handle end tag
  else if (s[0] == '/') {
    s = s.right(s.length()-1).trim();
    if (!handle(this,FXSEL(SEL_END,FXBaseObject::ID_TAG),(void*)s.text()) && target)
      target->handle(this,FXSEL(SEL_END,FXBaseObject::ID_TAG),(void*)s.text());
    }

  // handle end tag
  else {
    s.trim();
    if (!handle(this,FXSEL(SEL_BEGIN,FXBaseObject::ID_TAG),(void*)s.text()) && target)
      target->handle(this,FXSEL(SEL_BEGIN,FXBaseObject::ID_TAG),(void*)s.text());
    }

  return 1;
  }

// Handling content...
// - discard content message if the string length is zero, or the string only contains:
//   - carriage-returns
//   - line-feeds
//   - tabs
//   - spaces
// - replace embedded &...; characters
long FXXmlParser::onContent(FXObject*,FXSelector,void* ptr){
  FXString s = (char*) ptr;
  if (!whitespace) removeWhitespace(s);
  replaceEmbedded(s);
  if (s.length()){
    if (!handle(this,FXSEL(SEL_COMMAND,FXBaseObject::ID_CONTENT),(void*)s.text()) && target)
      target->handle(this,FXSEL(SEL_COMMAND,FXBaseObject::ID_CONTENT),(void*)s.text());
    }
  return 1;
  }

// handle read/parse error - not sure what to do here...
long FXXmlParser::onReadError(FXObject*,FXSelector,void* ptr){
  FXString s = (char*) ptr;
  fxmessage("Error reading XML content: '%s'\n",s.text());
  return 1;
  }


// removes carriage returns and line feeds not embedded in quotes
// also strips spaces not in quotes
void FXXmlParser::stripChar(FXString& s,FXchar c) {
  FXbool quote=FALSE;
  FXint i=0;
  while (i < s.length()) {
    if (s[i] == '"') quote= quote?FALSE:TRUE;
    if (!quote && s[i] == c) { s.replace(i,' '); }
    i++;
    }
  }

// removes the &...; embedded characters from an XML stream
//
// here's how we do it:
// - loop through the string until we find the ampersand symbol (&)
// - once we find it, start looking for the semi-colon (;)
//   - any characters between the '&' and ';' we keep in a string buffer
//   - we then lookup the embed'ed character in a lookup table, and add that
//     to the resultant string
//   - if we dont find the ';' before the end of the string, we just ignore
//   - everything after (and including) the '&'
// - otherwise we just add the current character to the resultant string
//
void FXXmlParser::replaceEmbedded(FXString& s){
  FXint i=0,found;
  FXString embed;
  while (i < s.length()) {
    if (s[i] == '&') {
      found=i++;
      while (i < s.length() && s[i] != ';') { embed+=s[i++]; }
      i++;
      if (i >= s.length()) { s.remove(found,i-found); }
      else { s.replace(found,i-found,lookup(embed)); }
      i=found;
      }
    i++;
    }
  }

// look-up table for embedded characters
FXString FXXmlParser::lookup(const FXString& s) {
  if (s == "quot") return FXString("\"");
  if (s == "amp") return FXString("&");
  if (s == "gt") return FXString(">");
  if (s == "lt") return FXString("<");
  return FXString::null;
  }

// remove the whitespace from between tags
// - if we find any characters other than '\n','\r','\t',' ', then we abort the removal
void FXXmlParser::removeWhitespace(FXString& s) {
  FXbool found=FALSE;
  for (FXint i=0; i<s.length(); i++) {
    if (s[i] != '\n' && s[i] != '\r' && s[i] != '\t' && s[i] != ' ') { found=TRUE; break; }
    }
  if (!found) s="";
  }

}

