/********************************************************************************
*                                                                               *
*                  Operating system logging interface                           *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXException.h>
#include <fox/FXWindow.h>
using namespace FX;
#include "exincs.h"
#include "fxexdefs.h"
#include "FXExtendedException.h"
#include "FXSystemLogger.h"
using namespace FXEX;
namespace FXEX {

/*
 * FIXME  add support for logging the message to stderr
 *        - for unix this could be a simple as setting the LOG_PERROR syslog flag
 *        - on win32 we need to use some form of fxwarning()
 *        add support for logging the PID as an optional parameter
 */


FXSystemLogger* FXSystemLogger::thislog = NULL;

#ifndef WIN32
const FXint FXSystemLogger::PRI_ERROR = LOG_ERR;
const FXint FXSystemLogger::PRI_WARNING = LOG_WARNING;
const FXint FXSystemLogger::PRI_NORMAL = LOG_INFO;
const FXint FXSystemLogger::PRI_DEBUG = LOG_DEBUG;
#else
const FXint FXSystemLogger::PRI_ERROR = EVENTLOG_ERROR_TYPE;
const FXint FXSystemLogger::PRI_WARNING = EVENTLOG_WARNING_TYPE;
const FXint FXSystemLogger::PRI_NORMAL = EVENTLOG_INFORMATION_TYPE;
const FXint FXSystemLogger::PRI_DEBUG = EVENTLOG_INFORMATION_TYPE;
static HANDLE logHandle=NULL;
#endif

FXDEFMAP (FXSystemLogger) FXSystemLoggerMap[]={
  FXMAPFUNC(SEL_COMMAND,FXSystemLogger::ID_SYSLOG,FXSystemLogger::onCmdLog),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETVALUE,FXSystemLogger::onCmdSetValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETINTVALUE,FXSystemLogger::onCmdSetIntValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE,FXSystemLogger::onCmdSetStringValue),
  };
FXIMPLEMENT(FXSystemLogger,FXObject,FXSystemLoggerMap,ARRAYNUMBER(FXSystemLoggerMap))

FXSystemLogger::FXSystemLogger() : FXObject() {
  FXApp *app=FXApp::instance();
#ifndef WIN32
  openlog(app->getAppName().text(),LOG_NDELAY,LOG_USER);
#else
  logHandle=RegisterEventSource(NULL,app->getAppName().text());
#endif
  if (!thislog) thislog=this;
  }

FXSystemLogger::~FXSystemLogger(){
#ifndef WIN32
  closelog();
#else
  if (logHandle) DeregisterEventSource(logHandle);
#endif
  thislog=(FXSystemLogger*)-1;
  }

FXSystemLogger& FXSystemLogger::instance(){
  if (!thislog) thislog=new FXSystemLogger();
  return *thislog;
  }

void FXSystemLogger::log(const FXint priority, const FXchar* msg, ...){
  FXString message;
  va_list args;
  va_start(args,msg);
  message.vformat(msg, args);
  va_end(args);
#ifndef WIN32
  syslog(priority,message.text());
#else
  if (logHandle) ReportEvent(h,priority,0,0,NULL,1,0,message.text(),NULL);
#endif
  }

void FXSystemLogger::log(const FXint priority, const FXString& msg){
  log(priority,msg.text());
  }

void FXSystemLogger::log(const FXString& msg){
  log(PRI_NORMAL,"%s",msg.text());
  }

void FXSystemLogger::warning(const FXString& msg){
  log(PRI_WARNING,"%s",msg.text());
  }

void FXSystemLogger::error(const FXString& msg){
  log(PRI_ERROR,"%s",msg.text());
  }

void FXSystemLogger::log(const FXException* e){
  log(PRI_WARNING,"%s",e->what());
  }

void FXSystemLogger::log(const FXExtendedException* e){
  log(PRI_WARNING,"%s (%i)",e->what(),e->error());
  }

// handle SEL_COMMAND
long FXSystemLogger::onCmdLog(FXObject*,FXSelector,void *ptr){
  log(PRI_DEBUG,"%s",(FXchar*)ptr);
  return 1;
  }

// handle SETVALUE
long FXSystemLogger::onCmdSetValue(FXObject*,FXSelector,void *ptr){
  log(PRI_DEBUG,"%i",(FXint)ptr);
  return 1;
  }

// handle SETINTVALUE
long FXSystemLogger::onCmdSetIntValue(FXObject*,FXSelector,void *ptr){
  FXint i=*((FXint*)ptr);
  log(PRI_DEBUG,"%i",i);
  return 1;
  }

// handle SETSTRINGVALUE
long FXSystemLogger::onCmdSetStringValue(FXObject*,FXSelector,void *ptr){
  log(PRI_DEBUG,"%s",(*((FXString*)ptr)).text());
  return 1;
  }

}

