/********************************************************************************
*                                                                               *
*                  Generic socket I/O handling                                  *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXURL.h>
#include <fox/FXFile.h>
using namespace FX;
#include "exincs.h"
#include "fxexdefs.h"
#include "FXResolver.h"
#include "FXSocket.h"
using namespace FXEX;
namespace FXEX {

/* Notes:
 * 1. Child classes can inherit from this class so as to do message decoding, etc.
 * 2. This class responds to message ID's of FXSocket::ID_SOCKET for most events, except
 *    for the ID_READ, ID_SEND and ID_SEND_OOB events.
 * 3. a) If this class is a base class, then the child class can catch the FXSelector of
 *       ID_SOCKET/ID_READ with the appropriate message type (listed below), to handle
 *       the socket event:
 *    b) If the class is being used in an application, then the message sent is that of
 *       the message specified in the ctor.
 *    =>This is similar functionality as that provided with any other FXObject derived class.
 * 4. Whenever a socket fails in non-recoverable way, we generate a SEL_DESTROY (actually its
 *    the base clas FXIOHandle that generates the SEL_DESTROY) with data of FXSocket* which
 *    our child class or the application should handle to know when the socket has been
 *    closed due to some error. It is then up the app/child to delete the instance.
 * 5. When the socket reads zero data from the remote end, it is an indication that the
 *    remote end has closed the connection.  If this object happens to be an object which
 *    was instantiated by a socket server (ie it is an accepted socket), then we also
 *    delete the socket.
 *    Hint: the child/app should handle the SEL_CLOSED message with data of FXSocket*
 *          (generated by FXIOHandle).
 * 6. When a socket server accepts a connection, it does two things:
 *    a) it generates a SEL_CLOSE (not SEL_CLOSED as mentioned above) with data of socket
 *       input handle:
 *       - If the SEL_CLOSE event is handled - the connection is denied
 *       - If the event is unhandled - the connection is accepted
 *    b) If we have accepted the connection, we generate a SEL_OPENED event with data of
 *       FXSocket*.
 * 7. We generate a SEL_OPENED with data of FXSocket*, when a socket becomes connected.
 *    ie when we call connect()
 * 8. When we receive data, we generate:
 *       either: this->handle(this,FXSEL(SEL_DATA,0),(void*)&FXSocketData)
 *           or: target->handle(this,FXSEL(SEL_COMMAND,message),(void*)FXSocketData)
 * 9. To send data, the application/child should generate:
 *    either: this->handle(this,FXSEL(SEL_COMMAND,ID_SEND),(void*)&FXSocketData)
 *        or: socket->handle(this,FXSEL(SEL_COMMAND,ID_SEND),(void*)&FXSocketData)
 * 10.We respond to SEL_COMMAND,FXWindow::ID_GETINTVALUE, which returns the state of the object
 * 11.Since all sockets can be setup in non-blocking mode, when a client connects to a
 *    server, some time after calling create, the socket will geenerate a SEL_OPENED
 *    event.  At this point the socket is fully connected and useable - it is expected
 *    that the application will not send data down the socket until it has received
 *    this event (though, if you are lucky it may work).
 * 12.The getState() flag will indicate the state of the socket at any given time.
 *    Alternatively, the app can generate SEL_COMMAND||ID_GETINTVALUE on the socket
 *    to use the standard FOX message handling mechanism.
 *
 * Win32 Notes:
 * - We need to use winsock2, so we must initialise the winsock2 DLL
 * - FXSocket doesn't support overlapped IO
 *   (I actually think the idea of overlapped IO is good, but it is a hack for the real
 *   problem, which is that the existing socket functionality is bloated, thus a new
 *   API was needed.... linux in particular has pretty efficient function calls... my 2 cents)
 */

#ifndef WIN32
#  define IOHANDLE(x)     x
#  define SOCKFD(x)       x
#  define ERRNO           errno
#  define INVALID_SOCKET  -1
#  define SOCKET_ERROR    -1
#  define ADDRLEN         FXuint
#  define SOCKLEN         socklen_t
#else
#  define IOHANDLE(x)     ((struct win32socket_t *)(x))->handle
#  define SOCKFD(x)       ((struct win32socket_t *)(x))->socket
#  define ERRNO           WSAGetLastError()
#  define ADDRLEN         FXint
#  define EINTR           WSAINTR
#  define ENOTCONN        WSAENOTCONN
#  define EBADR           WSAENETDOWN
#  define ENOTSUP         WSAENOTSUP
#  define EINPROGRESS     WSAEINPROGRESS
#  define SOCKLEN         FXint
#  define FD_MASK         (FD_CONNECT|FD_CLOSE|FD_ACCEPT|FD_OOB|FD_READ|FD_WRITE)
#endif

#ifdef WIN32
// declared static so as to initialise the winsock2 DLL, just once
static FXint winsock2Initialised = 0;

// a special function just for win32 - to initialise the winsock2 DLL
FXbool winsock2init(){
  if (winsock2Initialised == 0) {
    WSADATA wsa;
    if ( WSAStartup(MAKEWORD(2,0),&wsa) != 0i ) {
      fxwarning("Win32 socket initialisation failure\n");
      return FALSE;
      }
    if ( LOBYTE(wsa.wVersion) != 2 || HIBYTE(wsa.wVersion) != 0 ) {
      fxwarning("Wrong Winsock version\n");
      WSACleanup();
      return FALSE;
      }
    }
  winsock2Initialised++;
  return TRUE;
  }

// and another - cleaup winsock2 DLL resources
void winsock2cleanup(){
  winsock2Initialised--;
  if (winsock2Initialised == 0) WSACleanup();
  }
#endif

// map
FXDEFMAP (FXSocket) FXSocketMap[]={
  FXMAPFUNC(SEL_IO_READ,FXSocket::ID_SOCKET,FXSocket::onRead),
  FXMAPFUNC(SEL_IO_EXCEPT,FXSocket::ID_SOCKET,FXSocket::onExcept),
  FXMAPFUNC(SEL_IO_WRITE,FXSocket::ID_CONNECT,FXSocket::onConnect),
  FXMAPFUNC(SEL_IO_READ,0,FXSocket::onIORead),
  FXMAPFUNC(SEL_IO_EXCEPT,0,FXSocket::onIOExcept),
  FXMAPFUNC(SEL_IO_CONNECT,0,FXSocket::onIOConnect),
  FXMAPFUNC(SEL_COMMAND,FXSocket::ID_RESOLVE,FXSocket::onConnectResolved),
  };
FXIMPLEMENT(FXSocket,FXObject,FXSocketMap,ARRAYNUMBER(FXSocketMap))

// for deserialisation
FXSocket::FXSocket() : FXIOHandle() {}

// Port designated socket creation
FXSocket::FXSocket(FXApp* a,FXint port,FXObject *tgt,FXSelector sel,FXSocketFamily family,FXSocketType type) : FXIOHandle(a,tgt,sel) {
  hostname="";
  this->port=port;
  this->family=family;
  this->type=type;
#ifdef WIN32
  if (! winsock2init()) { code=FXIOStatusError; return; }
  struct win32socket_t *sock;
  FXMALLOC(&sock,struct win32socket_t,1);
  sock->handle=CreateHandle();
  sock->socket=INVALID_SOCKET;
  iohandle=(FXInputHandle)sock;
#endif
  if ((family == FXSocketFamilyInet || family == FXSocketFamilyAutomatic) && port < 1)
    fxerror("%s: cannot specify internet-domain or automatic-select socket below port number 1\n",getClassName());
  }

// Service designated socket creation
FXSocket::FXSocket(FXApp* a,const FXString& service,FXObject *tgt,FXSelector sel,FXSocketFamily family,FXSocketType type) : FXIOHandle(a,tgt,sel) {
  hostname="";
  port=0;
  this->family=family;
  this->type=type;
#ifdef WIN32
  if (! winsock2init()) { code=FXIOStatusError; return; }
  struct win32socket_t *sock;
  FXMALLOC(&sock,struct win32socket_t,1);
  sock->handle=CreateHandle();
  sock->socket=INVALID_SOCKET;
  iohandle=(FXInputHandle)sock;
#endif
  struct servent *serv;
  if (type == FXSocketTypeStream) serv = ::getservbyname(service.text(),"tcp");
  else serv = ::getservbyname(service.text(),"udp");
  if (serv != NULL) {
    port = ntohs(serv->s_port);
    if (port < 1) fxerror("%s: failure looking up service: %s\n",getClassName(),service.text());
    }
  }

// Use/encapsulate an already existing socket
// Note: we consider this socket to be an 'accepted' socket - this implies that we dont know
//       how it was created (ie a listener socket could have created it, or it could have
//       dup()ed from somewhere else).
// Win32 note: the FXInputHandle in this case is the special 'struct win32socket_t'...
//       Its the only way I could think of to implement it... (ie it may change one day,
//       if we can think of a better way to pass handles around...)
FXSocket::FXSocket(FXInputHandle s,FXApp *a,FXObject *tgt,FXSelector sel) : FXIOHandle(s,a,tgt,sel) {
  hostname="";
  port=0;
  family=FXSocketFamilyNone;
  type=FXSocketTypeNone;
#ifdef WIN32
  if (! winsock2init()) { code=FXIOStatusError; return; }
#endif
  // make sure the socket is valid
//  if (SOCKFD(s) != INVALID_SOCKET) {
    iohandle=s;
    state=FXIOStateAccepted;
    // get some connection parameters
    port=getRemotePort(iohandle);
    hostname=getRemoteHostname(iohandle);
    family=getSocketFamily(iohandle);
    type=getSocketType(iohandle);
    FXTRACE((150,"lookup of accepted socket port number: %i\n",port));
//    }
//    else code=FXIOStatusError;
  }

// free up all resources
FXSocket::~FXSocket(){
  if(resolver) delete resolver;
  if (SOCKFD(iohandle) != INVALID_SOCKET) close();
#ifdef WIN32
  CloseHandle(IOHANDLE(iohandle));
  FXFREE(&((struct win32socket_t *)iohandle));
  winsock2cleanup();
#endif
  }

// create resources
void FXSocket::create(){
  if (state!=FXIOStateNone && state!=FXIOStateAccepted) return;

  FXTRACE((100,"%s::create %p\n",getClassName(),this));
  FXIOHandle::create();
  if (code!=FXIOStatusOk) { fxerror("%s: create() error\n",getClassName()); return; }

  if (state == FXIOStateAccepted) {
    // Note: We do this here rather than in open() since, by the time an accepted socket
    //       has gotten here, it is already valid; we just need to massage it a little.
    // Also: Open'ing a socket implies we get some form of a brand spanking new file descriptor
    //       which is not what happens to an accepted socket.
    // So: Set some connection parameters (if we can), then add this socket to the list that
    //     FXApp is watching.
    if (!( setSocketOptions() &&
           getApp()->addInput(IOHANDLE(iohandle),INPUT_READ|INPUT_EXCEPT,this,ID_SOCKET) ) ) {
      state=FXIOStateNone;
      code=EBADR;
      errorClose();
      }
    FXTRACE((100,"Socket accepted and is ready\n"));
    handle(this,FXSEL(SEL_OPENED,0),(void*)this);
    }

  // setup any remaining connection properties, for non-accepted sockets
  else {
    // if the user really wants to optimise the connection type, we can determine which
    // socket family to use (at runtime), thus maybe providing some network speed-up
    if (family==FXSocketFamilyAutomatic) {
      family=FXSocketFamilyInet;
      FXString fqlocalname=FXURL::hostname();
      FXint pos=fqlocalname.find('.');
      if (pos<1) pos=fqlocalname.length();
      FXString localname=fqlocalname.left(pos-1);
      if (hostname=="localhost" ||
          hostname=="localhost.localdomain" ||
          hostname=="127.0.0.1" ||
          hostname==fqlocalname ||
          hostname==localname ||
          hostname==""
         ) family=FXSocketFamilyLocal;
      }

    // finally open the socket
    open();
    }
  }

// helper - returns the filesystem path to a local domain socket
FXString FXSocket::getLocalDomainPath(){
  FXString av = getApp()->getAppName() + "." + getApp()->getVendorName();
  FXString file = FXFile::getTempDirectory() + PATHSEP + av + "." + FXStringVal(port);
  return file;
  }

// return the true OS dependant handle to the real object
FXInputHandle FXSocket::getHandle(){
  return IOHANDLE(iohandle);
  }
  
// save object to stream
void FXSocket::save(FXStream& store) const {
  FXIOHandle::save(store);
  store << hostname;
  store << port;
  store << (FXint) family;
  store << (FXint) type;
  }

// FIXME: load object from stream
// Note: the best we can do here is store the data and let the SocketClient/SocketServer
//       objects have a go a restoring the state of the object... I guess the biggest question is:
//
//       "Does it even make sense to try to de-serialise a socket?"
//
//       I'm thinking:
//       yes - for a socket server since the socket would need to be de-serialised into a listener
//             ready to accept any new socket connections
//       yes - for a socket client since the socket would de-serialise, then connect to a server
//       no - for a accepted socket since I'm not sure how to re-setup an accepted socket...
//            I'm not even sure it makes sense to want this since the application state might be
//            in an unknown state
//
void FXSocket::load(FXStream& store){
  FXint dummy;
  FXObject::load(store);
  store >> hostname;
  store >> port;
  store >> dummy; family=(FXSocketFamily)dummy;
  store >> dummy; type=(FXSocketType)dummy;
  }

// set the hostname
void FXSocket::setHostname(const FXString& host){
  if(state==FXIOStateNone) hostname=host;
  }

// set the port
void FXSocket::setPort(FXint port){
  if(state==FXIOStateNone) this->port=port;
  }

// set the socket family
void FXSocket::setSocketFamily(FXSocketFamily family){
  if(state==FXIOStateNone) this->family=family;
  }

// set the socket type
void FXSocket::setSocketType(FXSocketType type){
  if(state==FXIOStateNone) this->type=type;
  }

// Create a socket
FXbool FXSocket::open(){

  // Calling open() on a accepted socket is bad and should always return false
  // since an accepted socket _cannot_ be reopened (ie it makes no sense to)
  if (state!=FXIOStateNone) { code=EISCONN; return FALSE; }

  // see what the parent says about the iohandle (socket)
  // (make sure we dont already have a socket...)
  if (!FXIOHandle::open()) return FALSE;

  // get the socket family
  FXint fam;
  switch (family) {
    case FXSocketFamilyLocal: fam=AF_LOCAL; break;
    case FXSocketFamilyInet: fam=AF_INET; break;
    case FXSocketFamilyInet6: fam=AF_INET6; break;
    default:{ code=ENOTCONN; return FALSE; }
    }

  // get the socket type
  FXint typ=SOCK_STREAM;
  if (type==FXSocketTypeDatagram) typ=SOCK_DGRAM;

  // make the socket
  SOCKFD(iohandle)= ::socket(fam,typ,PF_UNSPEC);
  if (SOCKFD(iohandle) == INVALID_SOCKET) { code=ERRNO; return FALSE; }
#ifndef WIN32
  if (!nonBlocking(TRUE)) return FALSE;
#else
  if (WSAAsyncSelect(SOCKFD(iohandle),IOHANDLE(iohandle),0,FD_MASK) == SOCKET_ERROR) { code=ERRNO; return FALSE; }
#endif
  FXTRACE((100,"%s::open: socket created\n",getClassName()));

  // finish up
  state=FXIOStateUnconnected;
  writeBuffer.iohandle=iohandle;
  return setSocketOptions();
  }

// close the FXInputHandle
void FXSocket::closehandle(){
#ifndef WIN32
  FXIOHandle::closehandle();
#else
// FIXME: is this necessary?
//  ::shutdown(SOCKFD(iohandle), SD_BOTH);
  ::closesocket(SOCKFD(iohandle));
  SOCKFD(iohandle) = -1;
#endif
  }

// Close the socket
void FXSocket::close() {
  if (SOCKFD(iohandle) != INVALID_SOCKET) {
    if (state == FXIOStateListener && family == FXSocketFamilyLocal) {
      FXString file = getLocalDomainPath();
      FXFile::remove(file);
      if (FXFile::exists(file))
        fxwarning("%s::close: could not remove filesystem socket.\n",getClassName());
      }
    FXIOHandle::close();
    }
  }

// send a file down a socket, using the native operating system system call
FXbool FXSocket::sendfile(const FXString& file){
  if (!FXFile::exists(file)) return FALSE;
  FXlong size=(FXlong)FXFile::size(file);
  if (size<1) return FALSE;
  FXInputHandle handle;
  FXlong sent=0;
#ifndef WIN32
  handle=::open(file.text(),O_RDONLY);
  if (handle==INVALID_HANDLE) { code=errno; return FALSE; }
  FXint send=size<INT_MAX?size:INT_MAX;
  FXint n=0;
  while ( sent < size ) {
    n= ::sendfile (SOCKFD(iohandle),handle,(off_t*)&sent,send);
    if (n==-1) return FALSE;
    sent+=n;
    send=(size-sent)<INT_MAX?(size-sent):INT_MAX;
    }
#else
  handle=CreateFile(filename.text(),GENERIC_READ,FILE_SHARE_READ,NULL,OPEN_ALWAYS,NULL);
  if (handle==INVALID_HANDLE) { code=GetLastError(); return FALSE; }
  if (TransmitFile(SOCKFD(iohandle),handle,0,0,NULL,NULL,TF_REUSE_SOCKET|TF_USE_DEFAULT_WORKER)) return TRUE;
#endif
  return FALSE;
  }

// This is a plain old wrapper around the recvfrom() syscall, except that we re-initiate
// the syscall on EINTR since all FOX apps should handle interrupts correctly
FXint FXSocket::read(FXuchar *data,FXint len,FXbool outOfBand){
  FXTRACE((100,"Receiving data on socket: %i\n",SOCKFD(iohandle)));
  struct sockaddr fromaddr;
  ADDRLEN fromlen = sizeof(fromaddr);;
  FXint no,flags=0;
  if (outOfBand) flags=MSG_OOB;
#ifndef WIN32
  flags|=MSG_NOSIGNAL;  // we specifically dont want a SIGPIPE
#endif
  while ( (no= ::recvfrom(SOCKFD(iohandle),(void*)data,len,flags,&fromaddr,&fromlen)) == SOCKET_ERROR ) {
    if (ERRNO != EINTR) break;
    }
  if (no >= 0){
    code=FXIOStatusOk;
    return no;
    }
  code=ERRNO;
  return -1;
  }

// send the data to the destination - a wrapper around the send() syscall - see recv(...)
FXint FXSocket::write(FXuchar *data,FXint n,FXbool outOfBand){
  if(IOHANDLE(iohandle)==INVALID_HANDLE)
    fxerror("%s: cannot write data - socket is not connected",getClassName());
  FXTRACE((100,"Sending data on socket: %i\n",SOCKFD(iohandle)));
  FXint no,flags=0;
  if (outOfBand) flags|=MSG_OOB;
//  struct sockaddr toaddr;         //FIXME support for udp
//  memset((void*)&toaddr,0,sizeof(toaddr));
//  while ( (no= ::sendto(iohandle,(void*)data,n,flags,&toaddr,sizeof(toaddr)) ) < 0 ) {
  while ( (no= ::send(SOCKFD(iohandle),(void*)data,n,flags) ) == SOCKET_ERROR ) {
    if (ERRNO != EINTR) break;
    }
  if (no >= 0){
    code=FXIOStatusOk;
    return no;
    }
  code=ERRNO;
  return -1;
  }

// lets read as much as possible
FXbool FXSocket::readData(FXbool outOfBand){
  FXASSERT(state != FXIOStateListener);
#ifdef WIN32
  WSAResetEvent(IOHANDLE(iohandle));
#endif
  if (!FXIOHandle::readData(outOfBand) && code!=EAGAIN){
    // we 'delete this' on accepted connections, if some non-trival error occurs
    if (state==FXIOStateAccepted) delete this;
    return FALSE;
    }
  return TRUE;
  }

// helper routine to add this socket(iohandle) as an input capable of being watched
// => add this socket to the list that FXApp is watching
FXbool FXSocket::addSocketInput(){
  if (!getApp()->addInput(IOHANDLE(iohandle),INPUT_READ|INPUT_EXCEPT,this,ID_SOCKET) ) {
    code=EBADR;
    errorClose();
    return FALSE;
    }
  code=FXIOStatusOk;
  state=FXIOStateConnected;
  FXTRACE((100,"Socket connected and is ready\n"));
  handle(this,FXSEL(SEL_OPENED,0),(void*)this);
  return TRUE;
  }

// Setup the iohandle to become a connector (ie a client)
// Note: If we call ::connect() in non-blocking mode, it is possible that it will return early with
//       an EINPROGRESS error, which just means that we have to get FXApp to signal us when the
//       socket is ready to be fully connected.
//       (ie we add an INPUT_WRITE event to FXApp for that socket)
FXbool FXSocket::connect(const FXString &host) {
  if (!host.length()) fxerror("%s: must specifiy a hostname to connect to.\n",getClassName());

  // make sure we actually have a valid socket
  if (SOCKFD(iohandle)==INVALID_SOCKET){ code=ENOTCONN; return FALSE; }

  // connect to a unix domain socket server
  // (Local domain sockets use the hostname to encode the filename)
  if (family == FXSocketFamilyLocal) {
    FXTRACE((100,"Connecting to Local domain socket\n"));
    struct sockaddr_un name;
    name.sun_family = AF_LOCAL;
    FXString file = getLocalDomainPath();
    strcpy(name.sun_path,file.text());
    if ( ::connect (SOCKFD(iohandle), (struct sockaddr *) &name, sizeof(name) ) == SOCKET_ERROR ) {
      if (ERRNO!=EINPROGRESS){
        code=ERRNO;
        errorClose();
        return FALSE;
        }
      if (!getApp()->addInput(IOHANDLE(iohandle),INPUT_WRITE,this,ID_CONNECT) ) {
        code=EBADR;
        errorClose();
        }
      return FALSE;
      }
    }

  // connect to an internet domain socket server
  else if (family == FXSocketFamilyInet || family == FXSocketFamilyInet6) {
    FXTRACE((100,"Looking up hostname: %s\n",host.text()));

    // We could block for a long time, while waiting for the resolution of an unknown hostname.
    // As such, we use the FXResolver object to resolve the hostname, in a worker thread
    if(!resolver) resolver = new FXResolver(host,this,ID_RESOLVE);
    else if(resolver->isRunning()) fxerror("%s: trying to call resolver, while it is already running.\n",getClassName());
    else resolver->host(host);
    resolver->setFamily(family);
    resolver->setType(type);
    resolver->resolve();
    return FALSE;

/* FIXME: remove this code, once functional
    // Get the host address information
    struct hostent * hostaddr;
    hostaddr=gethostbyname(host.text());

    // On many windows systems, gethostbyname() fails if called with a valid ip address
    // (i.e. "212.212.212.121") The workaround is trying gethostbyaddr() if the previous fails.
    // Linux systems don't have this problem, as man page for gethostbyname confirms.
    // Since, Linux is not the only unix platform, we use the WIN32 fix under unix as well.
    // Note: we can't use inet_aton instead of the obsolete inet_addr, since windows programmers
    // didn't like it... (sigh)
#ifndef WIN32
    if (hostaddr == NULL) {
      struct in_addr adr;
      FXint result = inet_aton(host.text(),&adr);
      if(result != 0)
        hostaddr = gethostbyaddr((const char *) &(adr.s_addr), sizeof(adr.s_addr), AF_INET);
      }
    if (hostaddr == NULL) {
      code=h_errno;
      errorClose();
      return FALSE;
      }
#else
    if (hostaddr == NULL) {
      unsigned long adr = inet_addr(host.text());
      if(adr!=INADDR_NONE) hostaddr = gethostbyaddr((const char *) &adr, sizeof(adr), AF_INET);
      }
    if (hostaddr == NULL) {
      code=ERRNO;
      errorClose();
      return FALSE;
      }
#endif

    FXTRACE((100,"Connecting to Internet domain socket\n"));
    struct sockaddr_in addr;
    memset((char *) &addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    memcpy( &addr.sin_addr.s_addr, hostaddr->h_addr_list[0], hostaddr->h_length );
    addr.sin_port = htons(port);
    if ( ::connect (SOCKFD(iohandle), (struct sockaddr *) &addr, sizeof(addr) ) == SOCKET_ERROR ) {
      if (ERRNO!=EINPROGRESS){
        code=ERRNO;
        errorClose();
        return FALSE;
        }
      if (!getApp()->addInput(IOHANDLE(iohandle),INPUT_WRITE,this,ID_CONNECT)) {
        code=EBADR;
        errorClose();
        }
      return FALSE;
      }
*/
    }

  // unknown socket family
  else {
    fxerror("%s: Unknown socket family\n",getClassName());
    }

  // if we got here, we must have a connected socket :-)
  // so add the socket to the handles that FXApp is watching
  return addSocketInput();
  }

// handle 'hostname resolved' event
long FXSocket::onConnectResolved(FXObject*,FXSelector,void*){
  FXTRACE((100,"Hostname resolved, connecting to Internet domain socket\n"));
  if ( ::connect (SOCKFD(iohandle), (struct sockaddr *) resolver->socket_address_struct(), sizeof(struct sockaddr) ) == SOCKET_ERROR ) {
    if (ERRNO!=EINPROGRESS){
      code=ERRNO;
      errorClose();
      return 1;
      }
    if (!getApp()->addInput(IOHANDLE(iohandle),INPUT_WRITE,this,ID_CONNECT)) {
      code=EBADR;
      errorClose();
      return 1;
      }
    }
  return 1;
  }

// setup the socket to become a socket listener (ie a server)
FXbool FXSocket::listen(FXint concurrent) {

  // make sure we actually have a valid socket
  if (SOCKFD(iohandle)==INVALID_SOCKET){ code=ENOTCONN; return FALSE; }

  // setup socket as Local domain socket
  if (family == FXSocketFamilyLocal) {
    FXTRACE((100,"Listening as Local domain socket\n"));
    FXString file = getLocalDomainPath();
    if (FXFile::exists(file) && ! FXFile::remove(file)){
      code=EBADR;
      errorClose();
      return FALSE;
      }
    struct sockaddr_un name;
    name.sun_family = AF_LOCAL;
    strcpy(name.sun_path,file.text());
    if ( ::bind (SOCKFD(iohandle), (struct sockaddr *) &name, sizeof(name) ) == SOCKET_ERROR ) {
      code=ERRNO;
      errorClose();
      return FALSE;
      }
    }

  // setup socket as internet domain socket
  else if (family == FXSocketFamilyInet || family == FXSocketFamilyInet6) {
    FXTRACE((100,"Listening as Internet domain socket\n"));
    FXString h= (hostname.length()) ? hostname : "0.0.0.0";
    if(!resolver) resolver = new FXResolver(h,this,ID_RESOLVE);
    else if(resolver->isRunning()) fxerror("%s: trying to call resolver, while it is already running.\n",getClassName());
    else resolver->host(h);
    resolver->setFamily(family);
    resolver->setType(type);
    if (resolver->blockingResolve()) {
      if( ::bind(SOCKFD(iohandle), (struct sockaddr *)resolver->socket_address_struct(), sizeof(struct sockaddr)) == SOCKET_ERROR ) {
        code=ERRNO;
        errorClose();
        return FALSE;
        }
      }
    else {
      code=resolver->status();
      errorClose();
      return FALSE;
      }
/*
    struct sockaddr_in addr;
    memset((char *) &addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    if (family == FXSocketFamilyInet6) addr.sin_family = AF_INET6;
    addr.sin_addr.s_addr = htonl(INADDR_ANY); // FIXME allow certain hosts only
    addr.sin_port = htons(port);
    if( ::bind(SOCKFD(iohandle), (struct sockaddr *) &addr, sizeof(addr)) == SOCKET_ERROR ) {
      code=ERRNO;
      errorClose();
      return FALSE;
      }
*/
    }

  // unknown socket??
  else {
    fxerror("%s: Unknown socket family\n",getClassName());
    }

  // listen to internet/unix domain port
  if ( ::listen(SOCKFD(iohandle),concurrent) == SOCKET_ERROR ) {
    code=ERRNO;
    errorClose();
    return FALSE;
    }

  // Add this socket to the list that FXApp is watching
  if (!getApp()->addInput(IOHANDLE(iohandle),INPUT_READ|INPUT_EXCEPT,this,ID_SOCKET) ) {
    code=EBADR;
    errorClose();
    return FALSE;
    }
  FXTRACE((100,"Socket listening and is ready\n"));

  code=FXIOStatusOk;
  state=FXIOStateListener;
  return TRUE;
  }

// accept incoming socket connections
FXSocket* FXSocket::accept(){
  FXASSERT(state==FXIOStateListener);

  // make sure we actually have a valid socket
  if (SOCKFD(iohandle)==INVALID_SOCKET){ code=ENOTCONN; return FALSE; }

  // accept the new connection
  FXInputHandle h;
#ifdef WIN32
  struct win32socket_t *sock;
  FXMALLOC(&sock,struct win32socket_t,1);
  sock->handle=CreateHandle();
  sock->socket=INVALID_SOCKET;
  h=(FXInputHandle)sock;
#endif
  struct sockaddr addr;
  ADDRLEN len = sizeof(addr);
  // if there was an error, ignore it if it was an interrupt
  while ( (SOCKFD(h)= ::accept(SOCKFD(iohandle),&addr,&len)) == INVALID_SOCKET ) {
    if (ERRNO!=EINTR) {
#ifdef WIN32
      FXFREE(&((struct win32socket_t *)h));
#endif
      return NULL;
      }
    }

  // allow the child or a target a chance to deny the incoming connection
  if ( handle(this,FXSEL(SEL_CLOSE,0),(void*)&h) ){
    FXTRACE((100,"Socket is denied\n"));
#ifndef WIN32
    ::close(h);
#else
    closesocket(SOCKFD(h));
    CloseHandle(IOHANDLE(h));
    FXFREE(&((struct win32socket_t *)h));
#endif
    code=FXIOStatusOk;
    return NULL;
    }

  // Create a new socket object, based on the connection
  // and forward the FXSocket object to the user application
  // Set the socket to non-blocking if the listen socket has been
  // set to non-blocking (or always non-block for win32)
  //
  // FIXME: how do I forward to a derived class?
  FXSocket *s = new FXSocket(h,getApp(),target,message);
  s->create();
  return s;
  }

// FXApp has signaled us to read some data
// - if this object is the main server socket, we must call ::accept() here
// - otherwise we must be:
//   - a client socket
//   - an accepted socket
//   - pre-initialised application socket
//   in any case, we need to read data from the socket
long FXSocket::onRead(FXObject *sender,FXSelector sel,void* ptr) {
  if (!handle(this,FXSEL(SEL_IO_READ,0),ptr)) {
    if (state!=FXIOStateListener) { return FXIOHandle::onRead(sender,sel,ptr); }
    else { accept(); }
    }
  return 1;
  }

// We got signalled that there is exceptional data to handle
// TODO can a listening socket be sent OOB data?
long FXSocket::onExcept(FXObject *sender,FXSelector sel,void* ptr) {
  if (!handle(this,FXSEL(SEL_IO_EXCEPT,0),ptr)) {
    if (state==FXIOStateListener)
      fxwarning("%s: application error: FXIOStateListener received exception data\n",getClassName());
    return FXIOHandle::onExcept(sender,sel,ptr);
    }
    return 1;
  }

// This allows connections to be non-blocking...
// -> the app is signalled when the socket is fully accepted
long FXSocket::onConnect(FXObject*,FXSelector,void* ptr) {
  getApp()->removeInput(IOHANDLE(iohandle),INPUT_WRITE);
  if (!handle(this,FXSEL(SEL_IO_CONNECT,0),ptr)) {
    if (!addSocketInput())
      fxwarning("%s: application error: failure handling incoming connection\n",getClassName());
    }
  return 1;
  }

/*
 * Sockets can have various properties applied to them - the OS can do some work for us
 * We set those values here
 */
FXbool FXSocket::setSocketOptions(){
  FXint setflag;

  // Setup socket so that if we close() a connection, any unsent data is discarded immediately
  // NOTE: I have read on a newsgroup (message written by Stevens) that using SO_LINGER is a bad thing...
  struct linger linger;
  linger.l_onoff=1; linger.l_linger=0;
  if ( setsockopt (SOCKFD(iohandle), SOL_SOCKET, SO_LINGER, &linger, sizeof(linger) ) == SOCKET_ERROR ) {
    code=ERRNO;
    errorClose();
    return FALSE;
    }

  // Set the socket option to send keep alives:
  // - This ensures that we dont wait forever on a socket
  // - app should do an app level ping to ensure that the connection is still functional
  //   since most OS timeouts are quite large
  setflag=1;
  if ( setsockopt (SOCKFD(iohandle), SOL_SOCKET, SO_KEEPALIVE, &setflag, sizeof(setflag) ) == SOCKET_ERROR ) {
    code=ERRNO;
    errorClose();
    return FALSE;
    }

  // Allow sending/receiving on the broadcast addresses
  // Note: multicast addresses are preferred over the broadcast address
  setflag=1;
  if ( setsockopt (SOCKFD(iohandle), SOL_SOCKET, SO_BROADCAST, &setflag, sizeof(setflag) ) == SOCKET_ERROR ) {
    code=ERRNO;
    errorClose();
    return FALSE;
    }

  // setup socket so that we can reuse the address
  setflag=1;
  if ( setsockopt (SOCKFD(iohandle), SOL_SOCKET, SO_REUSEADDR, &setflag, sizeof(setflag) ) == SOCKET_ERROR ) {
    code=ERRNO;
    errorClose();
    return FALSE;
    }

  code=FXIOStatusOk;
  return TRUE;
  }

/*
 * the following functions are declared static so that apps
 * can query a socket to retrieve info
 */

// retrieve hostname of remote machine
FXString FXSocket::getRemoteHostname(FXInputHandle s){
  FXString hostname="unknown";
  struct sockaddr addr;
  ADDRLEN len = sizeof(addr);
  if ( ::getpeername(SOCKFD(s),&addr,&len) != SOCKET_ERROR ){
    if (addr.sa_family == AF_LOCAL){
      hostname="localhost";
      }
    else if (addr.sa_family == AF_INET){
      struct sockaddr_in *ptr = (struct sockaddr_in *)&addr;
      hostname = inet_ntoa(ptr->sin_addr);
      }
    else if (addr.sa_family == AF_INET6){
      fxerror("FXSocket: INET6 not yet supported.\n");
      }
    else {
      fxerror("FXSocket: Unknown socket family: %d\n",addr.sa_family);
      }
    }
  return hostname;
  }

// get port of remote machine
FXint FXSocket::getRemotePort(FXInputHandle s){
  FXint port=0;
  struct sockaddr addr;
  ADDRLEN len = sizeof(addr);
  if ( ::getpeername(SOCKFD(s),&addr,&len) != SOCKET_ERROR ){
    if (addr.sa_family == AF_LOCAL){
      struct sockaddr_un *ptr = (struct sockaddr_un *)&addr;
      port = FXIntVal( FXFile::name(ptr->sun_path) );
      }
    else if (addr.sa_family == AF_INET){
      struct sockaddr_in *ptr = (struct sockaddr_in *)&addr;
      port = ntohs(ptr->sin_port);
      }
    else if (addr.sa_family == AF_INET6){
      fxerror("FXSocket: INET6 not yet supported.\n");
      }
    else {
      fxerror("FXSocket: Unknown socket family: %d\n",addr.sa_family);
      }
    }
  return port;
  }

// get socket family from socket
FXSocketFamily FXSocket::getSocketFamily(FXInputHandle s){
  struct sockaddr addr;
  ADDRLEN len = sizeof(addr);
  if ( ::getpeername(SOCKFD(s),&addr,&len) != SOCKET_ERROR ){
    if (addr.sa_family == AF_LOCAL) return FXSocketFamilyLocal;
    if (addr.sa_family == AF_INET) return FXSocketFamilyInet;
    if (addr.sa_family == AF_INET6) return FXSocketFamilyInet6;
    fxerror("FXSocket: Unknown socket family: %d\n",addr.sa_family);
    }
  return FXSocketFamilyNone;
  }

// returns the socket type
// notes: we could try using getsockname() if getsockopt() doesn't work...
FXSocketType FXSocket::getSocketType(FXInputHandle s){
  FXSocketType type=FXSocketTypeNone;
  SOCKLEN d2;
  char d1[80];
#ifndef WIN32
  if ( (type=(FXSocketType) ::getsockopt(SOCKFD(s),SOL_SOCKET,SO_TYPE,d1,&d2)) == SOCKET_ERROR ) {
    type=FXSocketTypeNone;
    }
#else
  if ( ::getsockopt(SOCKFD(s),SOL_SOCKET,SO_TYPE,d1,&d2) != SOCKET_ERROR ) {
    FXString t(d1,d2);
    type=(FXSocketType)FXIntVal(t);
    }
#endif
  FXSocketType t;
  switch (type) {
    case SOCK_STREAM: t=FXSocketTypeStream; break;
    case SOCK_DGRAM: t=FXSocketTypeDatagram; break;
    default: { fxerror("FXSocket: Unknown socket type: %i\n",type); }
    }
  return t;
  }

/*
 * Create a socket which is effectively a duplicate of this one
 * FIXME add win32 support (using DuplicateHandle, WSADuplicateSocket)
 */
FXSocket* FXSocket::duplicate(FXInputHandle newHandle){
  return dynamic_cast<FXSocket*>(FXIOHandle::duplicate(newHandle));
  }

// implement the instance creation mechanism
FXSocket* FXSocket::newInstance(FXInputHandle h){
  return new FXSocket(h,getApp(),target,message);
  }

}

