/********************************************************************************
*                                                                               *
*                  Generic I/O object                                           *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXWindow.h>
using namespace FX;
#include "exincs.h"
#include "fxexdefs.h"
#include "FXIOHandle.h"
#include "FXDateTime.h"
using namespace FXEX;
namespace FXEX {

#define READ_CHUNK 8192

#ifdef WIN32
#  define errno GetLastError()
#endif


/* FXIOData */

// ctor
FXIOData::FXIOData(FXInputHandle h,FXbool outOfBand) : iohandle(h),data(NULL),no(0),oob(outOfBand){}

// ctor
FXIOData::FXIOData(FXuchar *d,FXuint n) : iohandle(INVALID_HANDLE),data(d),no(n),oob(FALSE){}

//dtor
FXIOData::~FXIOData() { iohandle=INVALID_HANDLE; data=(FXuchar*)-1; }

// save to stream
FXStream& operator<< (FXStream& store,const FXIOData& d){
  store << d.iohandle;
  store << d.no;
  if (d.no) store.save(d.data,d.no);
  store << d.oob;
  return store;
  }

// load from stream
FXStream& operator>> (FXStream& store,FXIOData& d){
  store >> d.iohandle;
  store >> d.no;
  if (d.no) store.load(d.data,d.no);
  store >> d.oob;
  return store;
  }


/* FXIOHandle */

FXDEFMAP (FXIOHandle) FXIOHandleMap[]={
  FXMAPFUNC(SEL_IO_READ,FXIOHandle::ID_IOHANDLE,FXIOHandle::onRead),
  FXMAPFUNC(SEL_IO_WRITE,FXIOHandle::ID_IOHANDLE,FXIOHandle::onWrite),
  FXMAPFUNC(SEL_IO_EXCEPT,FXIOHandle::ID_IOHANDLE,FXIOHandle::onExcept),
  FXMAPFUNC(SEL_TIMEOUT,FXIOHandle::ID_IOHANDLE,FXIOHandle::onActivityTimeout),
  FXMAPFUNC(SEL_IO_READ,0,FXIOHandle::onIORead),
  FXMAPFUNC(SEL_IO_WRITE,0,FXIOHandle::onIOWrite),
  FXMAPFUNC(SEL_IO_EXCEPT,0,FXIOHandle::onIOExcept),
  FXMAPFUNC(SEL_IO_CONNECT,0,FXIOHandle::onIOConnect),
  FXMAPFUNC(SEL_OPENED,0,FXIOHandle::onOpened),
  FXMAPFUNC(SEL_CLOSE,0,FXIOHandle::onClose),
  FXMAPFUNC(SEL_CLOSED,0,FXIOHandle::onClosed),
  FXMAPFUNC(SEL_DESTROY,0,FXIOHandle::onDestroy),
  FXMAPFUNC(SEL_DATA,0,FXIOHandle::onData),
  FXMAPFUNC(SEL_COMMAND,FXIOHandle::ID_WRITE,FXIOHandle::onCmdWrite),
  FXMAPFUNC(SEL_COMMAND,FXIOHandle::ID_WRITE_OOB,FXIOHandle::onCmdWrite),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETINTVALUE,FXIOHandle::onCmdGetIntValue),
  };
FXIMPLEMENT(FXIOHandle,FXBaseObject,FXIOHandleMap,ARRAYNUMBER(FXIOHandleMap))

// for deserialisation
FXIOHandle::FXIOHandle() : FXBaseObject() {}

// derived classes usually use this ctor
FXIOHandle::FXIOHandle(FXApp* a,FXObject *tgt,FXSelector sel) : FXBaseObject(a,tgt,sel) {
  iohandle=INVALID_HANDLE;
  code=EINVAL;
  state=FXIOStateNone;
  readBufSize=READ_CHUNK;
  }

// Use/encapsulate an already existing file-descriptor
FXIOHandle::FXIOHandle(FXInputHandle h,FXApp *a,FXObject *tgt,FXSelector sel) : FXBaseObject(a,tgt,sel) {
  if (h==INVALID_HANDLE) fxerror("%s: handle is invalid.\n",getClassName());
  iohandle=h;
  code=FXIOStatusOk;
  state=FXIOStateConnected;
  readBufSize=READ_CHUNK;
  }

// free up all resources
FXIOHandle::~FXIOHandle(){
  if (iohandle != INVALID_HANDLE) close();
  writeBuffer.data=(FXuchar*)-1;
  writeBuffer.iohandle=INVALID_HANDLE;
  readBuffer=(FXIOData*)-1;
  }

// create resources
void FXIOHandle::create(){
  FXBaseObject::create();
  writeBuffer.iohandle=iohandle;
  if (state==FXIOStateConnected) {
    if (!getApp()->addInput(iohandle,INPUT_READ|INPUT_EXCEPT,this,ID_IOHANDLE)) {
      code=EBADR;
      errorClose();
      }
    }
  }

// detach resources
void FXIOHandle::detach(){
  if (iohandle != INVALID_HANDLE) close();
  FXBaseObject::detach();
  }

// destory resources
void FXIOHandle::destroy(){
  if (iohandle != INVALID_HANDLE) close();
  FXBaseObject::destroy();
  }

// save object to stream
void FXIOHandle::save(FXStream& store) const {
  FXBaseObject::save(store);
  store << iohandle;
  store << code;
  store << state;
  store << readBufSize;
  store << writeBuffer;
  }

// load object from stream
void FXIOHandle::load(FXStream& store){
  FXObject::load(store);
  store >> iohandle;
  store >> code;
  store >> state;
  store >> readBufSize;
  store >> writeBuffer;
  }

// operator ==
FXbool operator==(const FXIOHandle& h1,const FXIOHandle& h2){
  return h1.iohandle == h2.iohandle;
  }

// operator !=
FXbool operator!=(const FXIOHandle& h1,const FXIOHandle& h2){
  return h1.iohandle != h2.iohandle;
  }

// Create a handle - make sure we dont already have a handle...
// Otherwise, we assume that we already are open
FXbool FXIOHandle::open(){
  if (iohandle != INVALID_HANDLE) {
    code=EISCONN;
    return FALSE;
    }
  return TRUE;
  }

// closes the FXInputHandle
void FXIOHandle::closehandle(){
#ifdef WIN32
  CloseHandle(iohandle);
#else
  ::close(iohandle);
#endif
  iohandle=INVALID_HANDLE;
  }

// Close the IO handle
void FXIOHandle::close() {
  if (iohandle != INVALID_HANDLE) {
    FXTRACE((200,"Closing down IO handle\n"));
    getApp()->removeInput(iohandle,INPUT_READ|INPUT_WRITE|INPUT_EXCEPT);
    closehandle();
    if (writeBuffer.data) { FXFREE(&writeBuffer.data); }
    state=FXIOStateNone;
    code=FXIOStatusOk;
    handle(this,FXSEL(SEL_CLOSED,0),(void*)this);
    }
  getApp()->removeTimeout(this,ID_IOHANDLE);
  }

// close the IO handle, letting the app know that an error event happened
void FXIOHandle::errorClose() {
  handle(this,FXSEL(SEL_DESTROY,0),(void*)this);
  close();
  }

// This is a plain old wrapper around the read() syscall, except that we re-initiate
// the syscall on EINTR since all FOX apps should handle interrupts correctly
FXint FXIOHandle::read(FXuchar *data,FXint len,FXbool outOfBand){
  fxmessage("FIXME - rmeove this code\n");
  bzero(data,len);

  FXint n=::read(iohandle,data,len);
  return n;
  if (n<8) return n;

  FXint time;
  memcpy(&time,&data[0],4);
  printf("time: %d\n",FXDateTime::convert(time));
  FXushort code;
  memcpy(&code,&data[4],2);
  printf("code: %u\n",code);
  FXushort value;
  memcpy(&value,&data[6],2);
  printf("value: %u\n",value);
  return n;



  FXTRACE((200,"Reading data on IO handle: %i\n",iohandle));
  FXint no=0;
#ifndef WIN32
  while ( (no= ::read(iohandle,(void*)data,len)) < 0 ) {
    if (errno != EINTR) break;
    }
  if (no >= 0){
#else
#endif
    code=FXIOStatusOk;
    return no;
    }
  code=errno;
  return -1;
  }

// write the data to the destination - a wrapper around the write() syscall
FXint FXIOHandle::write(FXuchar *data,FXint n,FXbool outOfBand){
  if(iohandle==INVALID_HANDLE) fxerror("%s: cannot write data - handle is not connected",getClassName());
  FXTRACE((200,"Writing data on IO handle: %i\n",iohandle));
  FXint no=0;
#ifndef WIN32
  while ( (no= ::write(iohandle,(void*)data,n) ) < 0 ) {
    if (errno != EINTR) break;
    }
  if (no >= 0){
#else
  FXbool result=WriteFile(iohandle,data,n,&no,NULL);
  if (result && no >= 0) {
#endif
    code=FXIOStatusOk;
    return no;
    }
  code=errno;
  return -1;
  }

// lets read as much as possible - the return value indicates whether the handle is still active
FXbool FXIOHandle::readData(FXbool outOfBand){
  FXTRACE((200,"Reading data chunk\n"));
  FXIOData buffer(iohandle,outOfBand);
  if (! FXMALLOC(&buffer.data,FXuchar,readBufSize) ){
    fxwarning("Application error: insufficient memory\n");
    code=ENOMEM;
    return FALSE;
    }
  FXint no=0;

  // Read as much data from the OS as possible - usually this will just result in a single read().
  // This is actually a little complicated - we are trying to read all the data out of the kernel
  // using multiple passes => we pull data quickly into user space
  // Note: we are sacrificing a few extra cycles, for an increased network throughput
  for (;;) {
    no = read(&buffer.data[buffer.no],readBufSize,buffer.oob);
    if (code==FXIOStatusOk) {
      FXTRACE((200,"Received good read (%u bytes)\n",no));
      buffer.no += no;

      // if the read() filled the buffer we just extend the buffer and try to fill it again
      // Once the call stops reading data with a full buffers' worth, then we know the kernel
      // buffers are empty
      if (no == readBufSize) {
        FXRESIZE(&buffer.data,FXuchar,buffer.no+readBufSize);
        continue;
        }

      // We must have had a good read() and the buffer isn't completely full
      break;
      }

    // There was an error on the connection...
    // FIXME I think there is a slight bug here - on error if (buffer.no > 0) then
    //       we should always pass this data on to the application, but close the 
    //       IO handle anyway... time will tell...  I guess the question is whether
    //       any incoming data is still valid even though we are about to close 
    //       connection because of the error...

    // If the buffer.no is > zero, then we had a good read at some point
    // - if the error was EAGAIN, just handle the existing data, if there is any
    if (code == EAGAIN && buffer.no > 0) break;

    // some sort of error occurred - so clean up
    FXTRACE((200,"IO handle read error\n"));
    FXFREE(&buffer.data);

    // if the error wasn't EAGAIN, we _must_ close the connection (thus cleaning up the handle)
    // if it was EAGAIN, then we leave the handle open
    if (code != EAGAIN) {
      errorClose();
// FIXME      fxwarning("Application error: interrupt or lost connection errno=%d\n",errno);
      }
    return FALSE;
    }

  // We now pass the data (if there is any) to the appropriate target,
  // which is either a child class or the application
  if (buffer.no>0){
    readBuffer = &buffer;
    handle(this,FXSEL(SEL_DATA,0),(void*)&buffer);
    readBuffer = NULL;
    }
  FXFREE(&buffer.data);

  // reset the activity timer, if we have one
  if (activityPeriod>0) setActivityTimeout(activityPeriod);

  // if there was no data to read on this IO handle (ie no = 0), then the IO handle was just closed
  // by the other end, so close our end
  if (no == 0){
    close();   // Note: we can close here since the IO handle is no longer active
    return FALSE;
    }
  return TRUE;
  }

// lets write as much as possible
FXbool FXIOHandle::writeData(FXIOData *d){
  FXTRACE((200,"Writing data chunk\n"));

  // if there is already data waiting to be written, just add the data to the queue
  // if the data to be sent is out-of-band data, send it immediately
  if (writeBuffer.no > 0 && !d->oob){
    if(!FXRESIZE(&writeBuffer.data,FXuchar,writeBuffer.no+d->no)){
      fxwarning("Application error: insufficient memory\n");
      code=ENOMEM;
      return FALSE;
      }
    memcpy(&writeBuffer.data[writeBuffer.no],d->data,d->no);
    return TRUE;
    }

  // Send some data
  code=FXIOStatusOk;
  FXint no= write(d->data,d->no,d->oob);
  if (no<0) code=errno;

  // reset the activity timer, if we have one
  if(activityPeriod>0) setActivityTimeout(activityPeriod);

  // all the data was sent - nothing else to do
  if (code == FXIOStatusOk && no==(FXint)d->no) return TRUE;

  // if the write was ok, but not quite all the data was written, get FXApp to signal us 
  // when the kernel can write more data (unless we have just tried to write OOB data)
  // Note: an EAGAIN is equivalent to a valid write of not all the data
  // => save the user data to the internal buffer and add a IO handle write event to FXApp
  if (code==FXIOStatusOk || (code==EAGAIN && !d->oob) ) {
    if (!FXMEMDUP(&writeBuffer.data,FXuchar,&d->data[no],d->no-no)){
      fxwarning("Application error: insufficient memory\n");
      code=ENOMEM;
      return FALSE;
      }
    writeBuffer.no=d->no-no;
    if (!addInput()) return FALSE;
    return TRUE;
    }

  // if we tried to write OOB data and failed, we indicate that we didn't handle the event
  if (d->oob && code==EAGAIN) return FALSE;

  // argh! an error occurred (which wasn't an EAGAIN or EINTR)
  errorClose();
  fxerror("Application terminated: interrupt or lost connection errno=%d\n",errno);
  return FALSE;
  }

// helper for writing data
FXbool FXIOHandle::writeAgain(){
  FXTRACE((200,"Writing remaining data chunk\n"));
  FXASSERT(writeBuffer.data != NULL);

  // write the data :-)
  code=FXIOStatusOk;
  FXint no= write(writeBuffer.data,writeBuffer.no,writeBuffer.oob);
  if (no<0) { code=errno; }

  // make sure any invalid code, is just an EAGAIN
  if (code!=EAGAIN){
    errorClose();
    return FALSE;
    }
  
  // adjust the internal buffer so that we free the memory used by the data just written
  FXuchar *buf = NULL;
  writeBuffer.no -= no;
  if(!FXMEMDUP(&buf,FXuchar,&writeBuffer.data[no],writeBuffer.no)){
    fxwarning("Application error: insufficient memory\n");
    code=ENOMEM;
    return FALSE;
    }
  FXFREE(&writeBuffer.data);
  writeBuffer.data=buf;
  if (writeBuffer.no > 0) {
    if (!addInput()) return FALSE;
    }
  return TRUE;
  }

// helper routine to add this iohandle as an input capable of being watched
// => add this input to the list that FXApp is watching
FXbool FXIOHandle::addInput(){
  if (!getApp()->addInput(iohandle,INPUT_WRITE,this,ID_IOHANDLE) ) {
    code=EBADR;
    errorClose();
    return FALSE;
    }
  return TRUE;
  }

// FXApp has signaled us to read some data
long FXIOHandle::onRead(FXObject*,FXSelector,void* ptr) {
  if (!handle(this,FXSEL(SEL_IO_READ,0),ptr)) readData();
  return 1;
  }

// This gets called when we the IO handle is free to write some data (ie handle will not block)
long FXIOHandle::onWrite(FXObject*,FXSelector,void* ptr) {
  getApp()->removeInput(iohandle,INPUT_WRITE);
  if (!handle(this,FXSEL(SEL_IO_WRITE,0),ptr)) {
    FXASSERT(writeBuffer.no > 0);
    writeAgain();
    }
  return 1;
  }

// We got signalled that there is exceptional data to handle
long FXIOHandle::onExcept(FXObject*,FXSelector,void* ptr) {
  if (!handle(this,FXSEL(SEL_IO_EXCEPT,0),ptr)) readData(TRUE);
  return 1;
  }

// We just received a timeout notification
long FXIOHandle::onActivityTimeout(FXObject*,FXSelector,void*) {
  if (target) target->handle(this,FXSEL(SEL_TIMEOUT,message),NULL);
  return 1;
  }

// pass message to app for unbuffered mode read
long FXIOHandle::onIORead(FXObject*,FXSelector,void* ptr) {
  return target && target->handle(this,FXSEL(SEL_IO_READ,message),ptr);
  }

// pass message to app for unbuffered mode write
long FXIOHandle::onIOWrite(FXObject*,FXSelector,void* ptr) {
  return target && target->handle(this,FXSEL(SEL_IO_WRITE,message),ptr);
  }

// pass message to app for unbuffered mode except
long FXIOHandle::onIOExcept(FXObject*,FXSelector,void* ptr) {
  return target && target->handle(this,FXSEL(SEL_IO_EXCEPT,message),ptr);
  }

// pass message to app for connect event
long FXIOHandle::onIOConnect(FXObject*,FXSelector,void* ptr) {
  return target && target->handle(this,FXSEL(SEL_IO_CONNECT,message),ptr);
  }

// handle opening of connection
long FXIOHandle::onOpened(FXObject*,FXSelector,void *ptr){
  return target && target->handle(this,FXSEL(SEL_CLOSED,message),ptr);
  }

// deny incoming connection
long FXIOHandle::onClose(FXObject*,FXSelector,void *ptr){
  return target && target->handle(this,FXSEL(SEL_CLOSED,message),ptr);
  }

// handle connection closed
long FXIOHandle::onClosed(FXObject*,FXSelector,void *ptr){
  return target && target->handle(this,FXSEL(SEL_CLOSED,message),ptr);
  }

// handle connection destruction
long FXIOHandle::onDestroy(FXObject*,FXSelector,void *ptr){
  return target && target->handle(this,FXSEL(SEL_DESTROY,message),ptr);
  }

// handle data event
long FXIOHandle::onData(FXObject*,FXSelector,void *ptr){
  return target && target->handle(this,FXSEL(SEL_COMMAND,message),ptr);
  }

// The application calls SEL_COMMAND with a pointer to an FXIOData structure
//
// copy the data into our internal buffer only if:
// - there is already data waiting to be written
//   (which also means we are already waiting on an IO handle SEL_IO_WRITE event)
// - the data is larger than the kernel write buffer size
// - the write fails because the kernel write buffer is full (ie EAGAIN is generated)
//
// This regime ensures that small packets dont incurr excessive "scheduling" overhead
//
long FXIOHandle::onCmdWrite(FXObject*,FXSelector sel,void* ptr) {
  FXIOData *output = (FXIOData*) ptr;
  // this oob manipulation allows the app to write OOB data using a different ID to a normal
  // write or by setting the OOB flag in the FXIOData
  output->oob |= (FXbool) (FXSELID(sel) - ID_WRITE);
  writeData(output);
  return 1;
  }

// If the app requests a 'GETINTVALUE' - we send them the state
long FXIOHandle::onCmdGetIntValue(FXObject*,FXSelector,void *ptr){
  *((FXint*)ptr)=(FXint)state;
  return 1;
  }

// simple interface to status
FXbool FXIOHandle::opened(){
  return state >= FXIOStateOk;
  }

// simple interface to status
FXbool FXIOHandle::closed(){
  return state < FXIOStateOk;
  }

/*
 * IO handles can have various properties applied to them - the OS can do some work for us
 * We set those values here
 */

// get the non-blocking status of the IO handle
FXbool FXIOHandle::nonBlocking(){
#ifndef WIN32
  FXint flags;
  if ( (flags= fcntl(iohandle,F_GETFL)) < 0 ) {
    code=errno;
    errorClose();
    return FALSE;
    }
  return flags & O_NONBLOCK;
#else
#endif
  }

// Set the iohandle to non blocking
// Note: we can do this since the SEL_IO_WRITE functionality can handle pushing data around
//       when available, while the select() call in FXApp will do the read availability
FXbool FXIOHandle::nonBlocking(FXbool nonBlock){
#ifndef WIN32
  FXint flags;
  if ( (flags= fcntl(iohandle,F_GETFL)) < 0 ) {
    code=errno;
    errorClose();
    return FALSE;
    }
  if (nonBlock) flags |= O_NONBLOCK;
  else flags = flags & ~O_NONBLOCK;
  if ( fcntl(iohandle, F_SETFL, flags) < 0 ) {
    code=errno;
    errorClose();
    return FALSE;
    }
  return TRUE;
#else
#endif
  }

/*
 * These next two routines allows a simple interface to reading and writing strings,
 * since a lot of IO handle apps just pass strings around
 */

// Get the text just received on an IO handle
// There will only be text in the buffer if we are in the middle of handling a read event
FXString FXIOHandle::getText() const {
  if (readBuffer){
    FXString text((FXchar*)readBuffer->data,readBuffer->no);
    return text;
    }
  return FXString::null;
  }

// send some text over the IO handle
void FXIOHandle::setText(const FXString& message){
  FXIOData output((FXuchar*)message.text(),message.length());
  writeData(&output);
  }

// create a handle activity timer - to ese the implemenatation of appplication-level ping's.
void FXIOHandle::setActivityTimeout(FXint ms){
  activityPeriod=ms;
  getApp()->removeTimeout(this,ID_IOHANDLE);
  if (activityPeriod>0) getApp()->addTimeout(this,ID_IOHANDLE,activityPeriod);
  }

/*
 * Create an IO handle which is effectively a duplicate of this one
 * FIXME: for Win32 use DuplicateHandle
 */
FXIOHandle* FXIOHandle::duplicate(FXInputHandle newHandle){
#ifndef WIN32
  FXInputHandle ioh;
  if (newHandle != INVALID_HANDLE){
    ioh=::dup2(iohandle,newHandle);
    }
  else {
    ioh=::dup(iohandle);
    }
  if (ioh == INVALID_HANDLE) return NULL;
  FXIOHandle *handle=newInstance(ioh);
  handle->create();
  if (nonBlocking()) handle->nonBlocking(TRUE);
  else handle->nonBlocking(FALSE);
  return handle;
#else
#endif
  }

// implement the instance creation mechanism
FXIOHandle* FXIOHandle::newInstance(FXInputHandle h){
  return new FXIOHandle(h,getApp(),target,message);
  }

}

