/*********************************************************************************
*                                                                               *
*                  Log file object                                              *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXLOGGER_H
#define FXLOGGER_H

#ifndef FXFILESTREAM_H
#include <fox/FXStream.h>
using namespace FX;
#endif
namespace FXEX {

/**
 * Simple defines, used to simplify the logging syntax
 * Takes FXString's, FXchar*'s or FXException's
 */
#ifndef NDEBUG
#  define DBG(argument)     FXLogger::instance().debug  (argument)
#  define LOG(argument)     FXLogger::instance().log    (argument)
#  define WRN(argument)     FXLogger::instance().warning(argument)
#  define ERR(argument)     FXLogger::instance().error  (argument)
#  define STATUS(arg1,arg2) FXLogger::instance().status (arg1,arg2)
#else
#  define DBG(argument)
#  define LOG(argument)
#  define WRN(argument)
#  define ERR(argument)
#  define STATUS(arg1,arg2)
#endif

/**
 * A log file manager to allow applications to log entries to a file.
 *
 * The good stuff:
 * - 'stream'ing works (mostly), but it is not automatically new-line terminated (use endl)
 * - Use the #define's (LOG,DBG,WRN,ERR,STATUS) to make application logging simple
 * - Logging of exceptions are supported
 * - No need to add a newline to the end of the string, though you can if you wish.
 *
 * The bad stuff:
 * - The base class for this is currently FXFileStream, it should really be FXFileIO
 * - FXFileStream has no append mode, so each time the app is run it opens a new file
 *   destroying the old one, rather than appending to it - this is rather annoying...
 * - stream operator<< doesn't like FXStrings - will fix one day (not soon)
 */
class FXAPI FXLogger : public FXFileStream {
private:
  static FXLogger* thisLog;   // singleton handle
  FXString         filename;  // the log file
  FXlong           filesize;  // max size of log file
  FXuint           filemode;  // the permisions of the file
  FXbool           enabled;   // indication of whether the logger is enabled
  FXbool           stamp;     // should we add a date stamp to the log entry
  FXbool           nextStamp; // indication of whether the next stream operation will cause a timestamp to be written

protected:
  /// opens the log file
  FXbool open();

  /// closes the log file
  FXbool close();

  /**
   * Ensures that the log file is no bigger than size
   * Returns state whether the log file is open
   */
  FXbool trim();

  /// Add timestamp when using the stream operator '<<'
  void checkStreamStamp(const FXchar& v);

public:
  /// Give me a log file
  FXLogger(const FXString& file,const FXlong size=100000,const FXuint mode=0,const FXbool timestamp=TRUE);

  /// Use the current instance
  static FXLogger& instance();

  /// Set the size of the log file - check on next log entry
  void size(const FXlong size) { filesize=size; }

  /// Get the size of the log file
  FXlong size() { return filesize; }

  /// Change the location of the log file - change is immediate
  FXbool name(const FXString& file);

  /// get the current filename
  FXString name() { return filename; }

  /// indicates whether the log file can/will be written to
  FXbool isEnabled() { return enabled; }

  /// This is for logging output to a file
  /// It is automatically new-line terminated
  void log(const FXchar* msg, ...);
  void log(const FXString& msg);
  void status(const FXString& code, const FXString& msg);
  void warning(const FXString& msg);
  void error(const FXString& msg);
  void debug(const FXString& msg);
  void log(const FXException* e);
  void log(const FXExtendedException* e);

  /// Save to stream
  virtual FXStream& operator<<(const FXchar& v);

  /// load from stream
  virtual FXStream& operator<<(const FXuchar& v);

  /// done
  virtual ~FXLogger();
  };

} // namespace FXEX
#endif // FXLOGGER_H
