/********************************************************************************
*                                                                               *
*                  Generic Database Interface                                   *
*                                                                               *
*********************************************************************************
* Copyright (C) 2002 by Mathew Robertson.       All Rights Reserved.            *
* Copyright (C) 2002 by Giancarlo Formicuccia.  All Rights Reserved.            *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXDATABASEINTERFACE_H
#define FXDATABASEINTERFACE_H

#ifndef FXOBJECTLIST_H
#include <fox/FXObjectList.h>
using namespace FX;
#endif
#include "FXArray.h"
#include "FXBaseObject.h"
#include "FXExtendedException.h"
#include "FXDatabaseQuery.h"
namespace FXEX {

/**
 * FXDatabaseInterface is the base class for connection handling of a real database
 * interface, such as an ODBC interface.  Child classes implement specific handling of
 * connecting to a real database, executing the real query, etc.
 */
class FXAPI FXDatabaseInterface: public FXBaseObject {
  FXDECLARE_ABSTRACT(FXDatabaseInterface)

protected:
  FXString             database;  // name of database
  FXString             username;  // user of database connection
  FXString             password;  // password to database connection
  FXbool              connected;  // connection status
  FXDatabaseQueryList queries;    // allocated queries

  void checkStatus(FXbool should_be);

  /// serialisation
  FXDatabaseInterface();

  /// ctor
  FXDatabaseInterface(FXApp *a,FXObject *tgt=NULL,FXSelector sel=0);
  void notifyAllQueries(FXObject *, FXSelector, void *);

public:
  enum {
    ID_QRYDETACH  = FXBaseObject::ID_LAST,
    ID_LAST
    };

  long onQrydetach(FXObject *, FXSelector, void *);

public:
  static void dbThrow(const FXString &msg, FXint e); /* Exception throw */

public:
  /// indicated whether the object is connected to a real database
  FXbool isConnected() const { return connected; }

  /// set the name of the database we want to be connected to
  void setDatabase(const FXString &name);

  /// get the name of the database we are connected to
  FXString getDatabase() const { return database; }

  /// set the name of the user we want to connect to the database as
  void setUser(const FXString &name);

  /// get the name of the user that we are connected to the database as
  FXString getUser() const { return username; }

  /// set the password for the user
  void setPassword(const FXString &passwd);

  /// this returns a list of tables implemented for this database
  virtual FXStringList getTables() = '\0';

  /*
   * Dataset allocation+append to list
   */
  virtual FXDatabaseQuery *createQuery() = '\0';

  /* Transaction stuff */
  virtual void BeginTrans() = '\0';
  virtual void Commit() = '\0';
  virtual void Rollback() = '\0';

  /**
   * execute a SQL direct query, which doesn't need to return any results,
   * such as an INSERT, APPEND, UPDATE
   */
  virtual void executeDirect(const FXString &command) = '\0';

  /**
   * execute a SQL query and return the resulting data set
   */
  virtual FXDatabaseQuery *execute(const FXString &command);

  /// Stream save and load
  virtual void save(FXStream& store) const;
  virtual void load(FXStream& store);

  /**
   * Connect to database - readOnly is only a hint - the underlying database may not support
   * read-only connections
   */
  virtual void connect(FXbool readOnly = FALSE) = '\0';

  /// Disconnect from database
  virtual void disconnect() = '\0';

  /// dtor
  virtual ~FXDatabaseInterface();
  };

} // namespace FXEX
#endif // FXDATABASEINTERFACE_H
