/********************************************************************************
*                                                                               *
*                  Unix Daemon application object                               *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef WIN32
#ifndef FXDAEMONAPP_H
#define FXDAEMONAPP_H

#ifndef FXOBJECT_H
#include <fox/FXObject.h>
using namespace FX;
#endif
namespace FXEX {
struct FXSysVMessage;

/**
 * Main daemon application object, used for when there is no X display to connect to.  Works
 * in a similar way to FXApp, but with support for System V IPC messages.
 */
class FXAPI FXDaemonApp : public FXObject {
  FXDECLARE(FXDaemonApp)

private:
  // Platform independent private data
  FXRegistry       registry;            // Application setting registry
  FXTimer         *timers;              // List of timers, sorted by time
  FXChore         *chores;              // List of chores
  FXTimer         *timerrecs;           // List of recycled timer records
  FXChore         *chorerecs;           // List of recycled chore records
  FXInvocation    *invocation;          // Modal loop invocation
  FXSignal        *signals;             // Array of signal records
  FXint            nsignals;            // Number of signals
  FXInput         *inputs;              // Input file descriptors being watched
  FXint            ninputs;             // Number of inputs
  FXint            maxinput;            // Maximum input number
  void            *r_fds;               // Set of file descriptors for read
  void            *w_fds;               // Set of file descriptors for write
  void            *e_fds;               // Set of file descriptors for exceptions
  FXbool           initialized;         // Has been initialized
  FXSysVMessage   *sysVmessages;        // List of SysV message queues

private:
  static FXDaemonApp    *app;           // Application pointer

public:
  static const FXuchar version[3];      // Version number
  static const FXuchar copyright[80];   // Copyright notice
  static const FXint   MAXSIGNALS;      // the maximum number of signals this OS supports

private:
  // Internal helper functions
  FXDaemonApp(const FXDaemonApp&);
  FXDaemonApp &operator=(const FXDaemonApp&);
  static void signalhandler(int sig);

protected:
  /// Return TRUE when new raw event is available
  virtual FXbool getNextEvent(FXRawEvent& ev,FXbool blocking=TRUE);

  /// Dispatch raw event
  virtual FXbool dispatchEvent(FXRawEvent& ev);

public:
  // Messages applications understand
  enum {
    ID_QUIT=0,
    ID_DUMP,
    ID_LAST
    };

public:
  // Message handlers
  long onCmdQuit(FXObject*,FXSelector,void*);
  long onCmdDump(FXObject*,FXSelector,void*);

public:
  /**
  * Construct application object; the name and vendor strings are used
  * as keys into the registry database for this application's settings
  */
  FXDaemonApp(const FXString &name="Application",const FXString& vendor="FoxDefault");

  /// Get application name
  FXString getAppName() const { return registry.getAppKey(); }

  /// Get vendor name
  FXString getVendorName() const { return registry.getVendorKey(); }

  /// Add timeout message to be sent to target object in ms milliseconds
  FXTimer* addTimeout(FXint ms,FXObject* tgt,FXSelector sel);

  /// Remove timeout, returns NULL
  FXTimer* removeTimeout(FXTimer *t);

  /**
  * Add a idle processing message to be sent to target object when
  * the system becomes idle, i.e. there are no events to be processed.
  */
  FXChore* addChore(FXObject* tgt,FXSelector sel);

  /// Remove idle processing message
  FXChore* removeChore(FXChore *c);

  /**
  * Add signal processing message to be sent to target object when 
  * the signal sig is raised; flags are to be set as per POSIX definitions.
  * When immediate is TRUE, the message will be sent to the target right away;
  * this should be used with extreme care as the application is interrupted
  * at an unknown point it its execution.
  */
  void addSignal(FXint sig,FXObject* tgt,FXSelector sel,FXuint flags=0);

  /// Remove signal message for signal sig
  void removeSignal(FXint sig);

  /**
  * Add a file descriptor fd to be watched for activity as determined
  * by mode (an OR of INPUT_READ, INPUT_WRITE, and INPUT_EXCEPT).
  * The message will be sent to the target when the specified activity
  * is detected on the file descriptor.
  */
  FXbool addInput(FXInputHandle fd,FXuint mode,FXObject *tgt,FXSelector sel);

  /**
  * Remove input message and target object for the specified file
  * descriptor and mode.
  */
  FXbool removeInput(FXInputHandle fd,FXuint mode);

  /**
  * Add a System V message queue input (receive) to be watched for activity
  * The message will be sent to the target when the specified message queue
  * (and optional data type) is detected on the queue.
  * Note: the message type of zero specifies all message types associated
  *       with that message queue, should be watched
  */
  FXSysVMessage* addMessageQueue(FXint messageid, FXint maxMsgSize, FXObject* tgt, FXSelector sel, FXint messagetype=0);

  /// Remove the System V message queue from the watch list
  FXSysVMessage* removeMessageQueue(FXSysVMessage* que);

  /// Create application's windows
  virtual void create();

  /// Destroy application's windows
  virtual void destroy();

  /// Detach application's windows
  virtual void detach();

  /// Peek to determine if there's an event
  FXbool peekEvent();

  /// Perform one event dispatch
  void runOneEvent();

  /**
  * Run the main application event loop until stop() is called,
  * and return the exit code passed as argument to stop().
  */
  FXint run();

  /// Run an event loop till some flag becomes non-zero
  FXint runUntil(FXuint& condition);

  /// Run event loop while there are events are available in the queue
  FXint runWhileEvents();

  /**
  * Terminate the outermost event loop, and all inner modal loops;
  * All more deeper nested event loops will be terminated with code equal
  * to 0, while the outermost event loop will return code equal to value.
  */
  void stop(FXint value=0);

  /**
  * Initialize application.
  * Parses and removes common command line arguments,
  * opens the display, and reads the registry.
  */
  void init(int& argc,char** argv);

  /**
  * Exit application.
  * Closes the display and writes the registry.
  */
  void exit(FXint code=0);

  /// Get registry
  FXRegistry& reg(){ return registry; }

  /// Beep
  void beep();

  /// Return application instance
  static inline FXDaemonApp* instance(){ return app; }

  /// Save
  virtual void save(FXStream& store) const;

  /// Load
  virtual void load(FXStream& store);

  /// Destroy the application and all reachable resources
  virtual ~FXDaemonApp();
  };

} //  namespace FXEX
#endif // FXDAEMONAPP_H
#endif

